/**
 * 编译着色器程序
 * @param gl {WebGLRenderingContext} The current WebGL context {WebGLProgram}
 * @param vertexSrc  The vertex shader source as an array of strings.顶点着色器
 * @param fragmentSrc  The fragment shader source as an array of strings.片元着色器
 * @param attributeLocations {Object} An attribute location map that lets you manually set the attribute locations
 * @return {WebGLProgram} the shader program 返回着色器程序
 */
export function compileProgram(gl: WebGLRenderingContext, vertexSrc: string, fragmentSrc: string, attributeLocations?: any): WebGLProgram {
	var glVertShader = compileShader(gl, gl.VERTEX_SHADER, vertexSrc);
	var glFragShader = compileShader(gl, gl.FRAGMENT_SHADER, fragmentSrc);

	var program = gl.createProgram();

	gl.attachShader(program, glVertShader);
	gl.attachShader(program, glFragShader);

	// optionally, set the attributes manually for the program rather than letting WebGL decide..
	if (attributeLocations) {
		for (var i in attributeLocations) {
			gl.bindAttribLocation(program, attributeLocations[i], i);
		}
	}


	gl.linkProgram(program);

	// if linking fails, then log and cleanup
	if (!gl.getProgramParameter(program, gl.LINK_STATUS)) {
		console.error('Error: Could not initialize shader.');
		console.error('gl.VALIDATE_STATUS', gl.getProgramParameter(program, gl.VALIDATE_STATUS));
		console.error('gl.getError()', gl.getError());

		// if there is a program info log, log it
		if (gl.getProgramInfoLog(program) !== '') {
			console.warn('Warning: gl.getProgramInfoLog()', gl.getProgramInfoLog(program));
		}

		gl.deleteProgram(program);
		program = null;
	}

	// clean up some shaders
	gl.deleteShader(glVertShader);
	gl.deleteShader(glFragShader);

	return program;
};

/**
 * 创建shader
 * @private
 * @param gl {WebGLRenderingContext} The current WebGL context {WebGLProgram}
 * @param type {Number} the type, can be either VERTEX_SHADER or FRAGMENT_SHADER
 * @param vertexSrc {string|string[]} The vertex shader source as an array of strings.
 * @return {WebGLShader} the shader
 */
var compileShader = function (gl: WebGLRenderingContext, type: number, src: string): WebGLShader {
	var shader = gl.createShader(type);

	gl.shaderSource(shader, src);
	gl.compileShader(shader);

	if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
		console.log(gl.getShaderInfoLog(shader));
		return null;
	}

	return shader;
};
