/**
 * 记录的一些方法和设置
 */

import { BLEND_MODES, DATA_URI, RENDERER_TYPE, URL_FILE_EXTENSION } from "../const"


export * from './twiddle';
export { default as toDisplayDataURL } from "./toDisplayDataURL";
export { default as determineCrossOrigin } from './determineCrossOrigin';
export * from './DrawAllToCanvas'

let nextUid = 0;

/**
 * 获得唯一id
 * @function uid
 * @return {number} 唯一的id
 */
export function uid(): number {
	return ++nextUid;
}

/**
 * 各种使用需要的canvas
 * 比如取canvas上下文创建对象，渐变色等
 * 像素碰撞检测时
 *
 */
export const backupCanvas: HTMLCanvasElement = document.createElement("canvas")

export function getGradientColor(points: any, colors: any): any {
	let colorObj: any;
	let ctx = backupCanvas.getContext("2d");
	if (points.length == 4) {
		colorObj = ctx.createLinearGradient(points[0], points[1], points[2], points[3]);
	} else {
		colorObj = ctx.createRadialGradient(points[0], points[1], 0, points[2], points[3], points[4]);
	}
	for (let i = 0, l = colors.length; i < l; i++) {
		colorObj.addColorStop(colors[i][0], getRGBA(colors[i][1], colors[i][2]));
	}
	return colorObj;
}

export function getCanvasBitmapStyle(image: any): any {
	let ctx = backupCanvas.getContext("2d");
	return ctx.createPattern(image, "repeat");
}

/**
 * Converts a hex color number to an [R, G, B] array
 * @function hex2rgb
 * @param {number} hex - The number to convert
 * @param  {number[]} [out=[]] If supplied, this array will be used rather than returning a new one
 * @return {number[]} An array representing the [R, G, B] of the color.
 */
export function hex2rgb(hex: number, out?: number[] | Float32Array): number[] | Float32Array {
	out = out || [];

	out[0] = ((hex >> 16) & 0xFF) / 255;
	out[1] = ((hex >> 8) & 0xFF) / 255;
	out[2] = (hex & 0xFF) / 255;

	return out;
}

/**
 * 十六进制颜色转字符串
 * 0xffffff转"#ffffff"
 * Converts a hex color number to a string.
 * @function hex2string
 * @param {number} hex - Number in hex
 * @return {string} The string color.
 */
export function hex2string(hex) {
	hex = hex.toString(16);
	hex = '000000'.substr(0, 6 - hex.length) + hex;
	return `#${hex}`;
}

/**
 * 字符串颜色转十六进制
 * "#ffffff"或"0xffffff"转0xffffff
 * @param string
 */
export function string2hex(string: string) {
	if (string.indexOf("#") == 0) {
		string = string.replace("#", "0x")
	}
	return parseInt(string)
}

/**
 * Converts a color as an [R, G, B] array to a hex number
 * rgb范围时0到1，和着色器中的一致
 * @function rgb2hex
 * @param {number[]} rgb - rgb array
 * @return {number} The color number
 */
export function rgb2hex(rgb: number[]): number {
	return (((rgb[0] * 255) << 16) + ((rgb[1] * 255) << 8) + (rgb[2] * 255 | 0));
}

/**
 * 通过24位颜色值和一个透明度值生成RGBA值
 * @method getRGBA
 * @static
 * @public
 * @since 1.0.0
 * @param {string} color 字符串的颜色值,如:#33ffee
 * @param {number} alpha 0-1区间的一个数据 0完全透明 1完全不透明
 * @return {string}
 */
export function getRGBA(color: string, alpha: number): string {
	if (color.indexOf("0x") == 0) {
		color = color.replace("0x", "#");
	}
	//暂没考虑#ff这种
	if (color.length < 7) {
		color = "#000000";
	}
	if (alpha != 1) {
		let r = parseInt("0x" + color.substr(1, 2));
		let g = parseInt("0x" + color.substr(3, 2));
		let b = parseInt("0x" + color.substr(5, 2));
		color = "rgba(" + r + "," + g + "," + b + "," + alpha + ")";
	}
	return color;
}

/**
 * 输入框ios兼容，如果加过输入框，加
 */
export function inputFeildIosEnable() {
	var u = navigator.userAgent, app = navigator.appVersion
	var isIOS = !!u.match(/\(i[^;]+;( U;)? CPU.+Mac OS X/); //ios终端
	if (isIOS) {
		setTimeout(() => {
			window["$"]("input").blur(function () {
				if (isIOS) {
					blurAdjust()
					// alert("1231321233")
				}
			});
		}, 50)
	}

	// 解决苹果不回弹页面
	function blurAdjust() {
		setTimeout(() => {
			// alert("1231321233")
			if (document.activeElement.tagName == 'INPUT' || document.activeElement.tagName == 'TEXTAREA') {
				return
			}
			let result = 'pc';
			if (/(iPhone|iPad|iPod|iOS)/i.test(navigator.userAgent)) { //判断iPhone|iPad|iPod|iOS
				result = 'ios'
			} else if (/(Android)/i.test(navigator.userAgent)) {  //判断Android
				result = 'android'
			}
			if (result = 'ios') {
				document.activeElement["scrollIntoViewIfNeeded"](true);
			}
		}, 100)
	}
}


/**
 * Split a data URI into components. Returns undefined if
 * parameter `dataUri` is not a valid data URI.
 *
 * @memberof utils
 * @function decomposeDataUri
 * @param {string} dataUri - the data URI to check
 * @return {utils~DecomposedDataUri|undefined} The decomposed data uri or undefined
 */
export function decomposeDataUri(dataUri) {
	const dataUriMatch = DATA_URI.exec(dataUri);

	if (dataUriMatch) {
		return {
			mediaType: dataUriMatch[1] ? dataUriMatch[1].toLowerCase() : undefined,
			subType: dataUriMatch[2] ? dataUriMatch[2].toLowerCase() : undefined,
			charset: dataUriMatch[3] ? dataUriMatch[3].toLowerCase() : undefined,
			encoding: dataUriMatch[4] ? dataUriMatch[4].toLowerCase() : undefined,
			data: dataUriMatch[5],
		};
	}

	return undefined;
}

/**
 * Get type of the image by regexp for extension. Returns undefined for unknown extensions.
 *
 * @memberof utils
 * @function getUrlFileExtension
 * @param {string} url - the image path
 * @return {string|undefined} image extension
 */
export function getUrlFileExtension(url) {
	const extension = URL_FILE_EXTENSION.exec(url);

	if (extension) {
		return extension[1].toLowerCase();
	}

	return undefined;
}

/**
 * Returns sign of number
 *
 * @memberof utils
 * @function sign
 * @param {number} n - the number to check the sign of
 * @returns {number} 0 if `n` is 0, -1 if `n` is negative, 1 if `n` is positive
 */
export function sign(n: number): number {
	if (n === 0) return 0;

	return n < 0 ? -1 : 1;
}

export function premultiplyTint(tint: number, alpha: number): number {
	if (alpha === 1.0) {
		return (alpha * 255 << 24) + tint;
	}
	if (alpha === 0.0) {
		return 0;
	}
	var R = ((tint >> 16) & 0xFF);
	var G = ((tint >> 8) & 0xFF);
	var B = (tint & 0xFF);

	R = ((R * alpha) + 0.5) | 0;
	G = ((G * alpha) + 0.5) | 0;
	B = ((B * alpha) + 0.5) | 0;

	return (alpha * 255 << 24) + (R << 16) + (G << 8) + B;
}

/**
 * 贴图缓存
 * 对于链接网络图片，直接以url作为key，自动缓存
 * 对于图集，直接以小图命名作为key（所以避免重复名字，不是同一张图集里的命名也不能一致，否则覆盖）
 * 不自动缓存base64图片
 * @private
 */
export const TextureCache = Object.create(null);

/**
 * 基础贴图缓存
 * 基本不会直接用，不缓存base64
 * @private
 */
export const BaseTextureCache = Object.create(null);

/**
 * 记录TextureSheet，便于帧动画贴图数据查找
 */
export const TextureSheetCache = Object.create(null);

/**
 * Destroys all texture in the cache
 * @function destroyTextureCache
 */
export function destroyTextureCache() {
	let key;

	for (key in TextureCache) {
		TextureCache[key].destroy();
	}
	for (key in BaseTextureCache) {
		BaseTextureCache[key].destroy();
	}
}

/**
 * Removes all textures from cache, but does not destroy them
 * @function clearTextureCache
 */
export function clearTextureCache() {
	let key;

	for (key in TextureCache) {
		delete TextureCache[key];
	}
	for (key in BaseTextureCache) {
		delete BaseTextureCache[key];
	}
}

/**
 * 全局属性记录
 */
export const GlobalPro = {
	/**
	 * 舞台渲染类型，
	 */
	stageRenderType: RENDERER_TYPE.WEBGL,
	/**
	 * 实际渲染分辨率
	 */
	dpi: 1,
	/**
	 * 图集间隙
	 */
	padding: 2,
    /**
     * 时钟起始时间
     */
	startTime: 0

}

/**
 * Helper for checking for webgl support
 * @function isWebGLSupported
 * @return {boolean} is webgl supported
 */
export function isWebGLSupported(): boolean {
	const contextOptions = { stencil: true };//, failIfMajorPerformanceCaveat: true
	try {
		if (!window["WebGLRenderingContext"]) {
			return false;
		}
		const canvas = document.createElement('canvas');
		let gl = canvas.getContext('webgl', contextOptions) || canvas.getContext('experimental-webgl', contextOptions);
		const success = !!(gl && gl["getContextAttributes"]().stencil);
		if (gl) {
			const loseContext = gl["getExtension"]('WEBGL_lose_context');
			if (loseContext) {
				loseContext.loseContext();
			}
		}
		gl = null;
		return success;
	} catch (e) {
		return false;
	}
}

export function removeItems(arr: Array<any>, startIdx: number, removeCount: number) {
	var i, length = arr.length

	if (startIdx >= length || removeCount === 0) {
		return
	}
	removeCount = (startIdx + removeCount > length ? length - startIdx : removeCount)

	var len = length - removeCount

	for (i = startIdx; i < len; ++i) {
		arr[i] = arr[i + removeCount]
	}
	arr.length = len
}

/**
 * Maps gl blend combinations to WebGL.
 * @function mapWebGLBlendModesToPixi
 * @private
 * @param {WebGLRenderingContext} gl - The rendering context.
 * @param {string[]} [array=[]] - The array to output into.
 * @return {string[]} Mapped modes.
 */
export function mapWebGLBlendModesToPixi(gl, array = []) {
	// TODO - premultiply alpha would be different.
	// add a boolean for that!
	array[BLEND_MODES.NORMAL] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.ADD] = [gl.ONE, gl.DST_ALPHA];
	array[BLEND_MODES.MULTIPLY] = [gl.DST_COLOR, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.SCREEN] = [gl.ONE, gl.ONE_MINUS_SRC_COLOR];
	array[BLEND_MODES.OVERLAY] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.DARKEN] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.LIGHTEN] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.COLOR_DODGE] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.COLOR_BURN] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.HARD_LIGHT] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.SOFT_LIGHT] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.DIFFERENCE] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.EXCLUSION] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.HUE] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.SATURATION] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.COLOR] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.LUMINOSITY] = [gl.ONE, gl.ONE_MINUS_SRC_ALPHA];

	// not-premultiplied blend modes
	array[BLEND_MODES.NORMAL_NPM] = [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA, gl.ONE, gl.ONE_MINUS_SRC_ALPHA];
	array[BLEND_MODES.ADD_NPM] = [gl.SRC_ALPHA, gl.DST_ALPHA, gl.ONE, gl.DST_ALPHA];
	array[BLEND_MODES.SCREEN_NPM] = [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_COLOR, gl.ONE, gl.ONE_MINUS_SRC_COLOR];

	return array;
}

