import Container from "../display/Container";
import Graphics from "../graphics/Graphics";
import {MouseEvent} from "../events/MouseEvent";
import {Event} from "../events/Event";
import {SCROLL_DIRECTION} from "../const";

// import  Tween  from "../../tweenSimple/Tween";

/**
 * 滚动视图
 * @class ScrollPage
 * @public
 * @extends Container
 * @since 1.0.0
 */
export class ScrollViewBase extends Container {
	/**
	 * 滚动方向
	 * @property direction
	 * @type {SCROLL_DIRECTION}
	 * @private
	 * @since 1.0.0
	 * @default true
	 */
	protected _direction: SCROLL_DIRECTION;
	/**
	 * 可见区域的宽
	 * @property viewWidth
	 * @type {number}
	 * @private
	 * @since 1.0.0
	 * @default 0
	 */
	//private viewWidth: number = 0;
	/**
	 * 可见区域的高
	 * @property viewHeight
	 * @type {number}
	 * @private
	 * @since 1.0.0
	 * @default 0
	 */
	//private viewHeight: number = 0;
	// private _tweenId: number = 0;
	/**
	 * 整个滚动的最大距离值
	 * @property maxDistance
	 * @type {number}
	 * @public
	 * @since 1.0.0
	 * @default 1040
	 */
	//public maxDistance: number = 1040;
	/**
	 * @property 滚动距离
	 * @type {number}
	 * @protected
	 * @default 0
	 * @since 1.0.0
	 */
	protected distance: number = 0;
	/**
	 * 最小鼠标滑动距离
	 * @type {number}
	 */
	private minDis: number = 0;
	/**
	 * 遮罩对象
	 * @property maskObj
	 * @since 1.0.0
	 * @private
	 * @type {Graphics}
	 */
	private maskObj: Graphics = new Graphics();
	/**
	 * 真正的容器对象，所有滚动的内容都应该是添加到这个容器中
	 * @property view
	 * @public
	 * @since 1.0.0
	 * @type {Container}
	 */
	public viewPort: Container = new Container();
	/**
	 * 最后鼠标经过的坐标值
	 * @property lastValue
	 * @private
	 * @since 1.0.0
	 * @type {number}
	 */
	private lastValue: number = 0;
	/**
	 * 速度
	 * @property speed
	 * @protected
	 * @since 1.0.0
	 * @type {number}
	 */
	protected speed: number = 0;
	/**
	 * 加速度
	 * @property addSpeed
	 * @private
	 * @since 1.0.0
	 * @type {number}
	 */
	private addSpeed: number = 0;
	/**
	 * 是否是停止滚动状态
	 * @property isStop
	 * @public
	 * @since 1.0.0
	 * @type {boolean}
	 * @default true
	 */
	public isStop: boolean = true;
	/**
	 * 滚动的最大速度，直接影响一次滑动之后最长可以滚多远
	 * @property maxSpeed
	 * @public
	 * @since 1.0.0
	 * @default 100
	 * @type {number}
	 */
	public maxSpeed: number = 100;
	/**
	 * 摩擦力,值越大，减速越快
	 * @property fSpeed
	 * @public
	 * @since 1.0.0
	 * @default 20
	 * @type {number}
	 */
	public fSpeed: number = 20;
	protected paramXY: string = "y";
	protected paramSize: string = "height";
	private stopTimes: number = -1;
	private isMouseDownState: number = 0;
	/**
	 * 是否是通过scrollTo方法在滑动中
	 * @property autoScroll
	 * @since 1.0.2
	 * @type {boolean}
	 * @private
	 * @default false
	 */
	private autoScroll: boolean = false;

	public isSpringBack: boolean = true;

	/**
	 * 构造函数
	 * @method  ScrollPage
	 * @param {number} maxDistance 最大滚动的长度
	 * @param isFull
	 * @example
	 *      var sPage=new ScrollPage(640,s.stage.viewRect.height,4943);
	 *      sPage.isSpringBack = false;//是否回弹
	 *      stage.addChild(sPage);
	 *      sPage.view.addChild(view);
	 *      sPage.y=stage.viewRect.y;
	 *
	 */
	constructor(maxDistance?: number, isFull = false) {
		super();
		let s: ScrollViewBase = this;
		s._instanceType = "ScrollContainer";
		super.addChild(s.maskObj);
		super.addChild(s.viewPort);
		if (!isFull) {
			//不全屏才设置mask
			s.viewPort.mask = s.maskObj;
			//为了能接收鼠标事件设置isUsedToMask
			s.maskObj.isUsedToMask = false;
		}
		s.maskObj.alpha = 0;
		if (maxDistance !== undefined) {
			//s.maxDistance = maxDistance;
		}
		s.updateViewRect();
		s.direction = SCROLL_DIRECTION.VERTICAL;
		s.addEventListener(Event.ADDED_TO_STAGE, function (e: Event) {
			s.stage.addEventListener(MouseEvent.MOUSE_UP, s.onMouseEvent, s);
			s.stage.addEventListener(MouseEvent.MOUSE_MOVE, s.onMouseEvent, s);
		});
		s.addEventListener(Event.REMOVED_FROM_STAGE, function (e: Event) {
			s.stage.removeEventListener(MouseEvent.MOUSE_UP, s.onMouseEvent, s);
			s.stage.removeEventListener(MouseEvent.MOUSE_MOVE, s.onMouseEvent, s);
		});
		s.addEventListener(MouseEvent.MOUSE_DOWN, s.onMouseEvent, s, false);
		// s.addEventListener(MouseEvent.MOUSE_UP, s.onMouseEvent, s);
		// s.addEventListener(MouseEvent.MOUSE_OUT, s.onMouseEvent, s);
		s.addEventListener(Event.ENTER_FRAME, function () {
			let view: any = s.viewPort;
			if (s.autoScroll) return;
			if (!s.isSpringBack) {
				if (view[s.paramXY] > 0) {
					s.addSpeed = 0;
					s.speed = 0;
					s.isStop = true;
					view[s.paramXY] = 0;
					return;
				} else if (view[s.paramXY] < s.distance - s.maxDistance) {
					s.addSpeed = 0;
					s.speed = 0;
					s.isStop = true;
					view[s.paramXY] = Math.min(0, s.distance - s.maxDistance);
					return;
				}
			}
			if (!s.isStop) {
				if (Math.abs(s.speed) > 0) {
					view[s.paramXY] += s.speed;
					//是否超过了边界,如果超过了,则加快加速度,让其停止
					if (view[s.paramXY] > 0 || view[s.paramXY] < s.distance - s.maxDistance) {
						s.speed += s.addSpeed * s.fSpeed;
					} else {
						s.speed += s.addSpeed;
					}
					//说明超过了界线,准备回弹
					if (s.speed * s.addSpeed > 0) {
						s.dispatchEvent(Event.ON_SCROLL_STOP);
						s.speed = 0;
					}
				} else {
					//检测是否超出了边界,如果超出了边界则回弹
					if (s.addSpeed != 0) {
						if (view[s.paramXY] > 0 || view[s.paramXY] < s.distance - s.maxDistance) {
							let tarP: number = 0;
							/*if (s.addSpeed > 0) {
								if (s.distance < s.maxDistance) {
									tarP = s.distance - s.maxDistance;
								}
							}*/
							if(view[s.paramXY] < s.distance - s.maxDistance){
								if (s.distance < s.maxDistance) {
									tarP = s.distance - s.maxDistance;
								}
							}
							view[s.paramXY] += 0.4 * (tarP - view[s.paramXY]);
							if (Math.abs(tarP - view[s.paramXY]) < 0.1) {
								s.isStop = true;
								if (s.addSpeed > 0) {
									s.dispatchEvent(Event.ON_SCROLL_TO_END);
								} else {
									s.dispatchEvent(Event.ON_SCROLL_TO_HEAD);
								}
							}
						}
					} else {
						s.isStop = true;
					}
				}
			} else {
				if (s.stopTimes >= 0) {
					//s.stopTimes++;
					if (s.stopTimes >= 15) {
						s.speed = 0;
						if (view[s.paramXY] > 0 || view[s.paramXY] < s.distance - s.maxDistance) {
							s.isStop = false;
							s.stopTimes = -1;
						}
					}
				}
			}
		});
		s.addEventListener(Event.RESIZE, this.updateViewRect, s);
	}

	get maxDistance() {
		return this.calMaxDistance();
	}

	protected calMaxDistance(){
		return this.viewPort[this.paramSize] + this.viewPort.getLocalBounds()[this.paramXY]
	}

	get direction(): SCROLL_DIRECTION {
		return this._direction;
	}

	set direction(value: SCROLL_DIRECTION) {
		this._direction = value;

		this.updateDirection();
	}

	protected updateDirection() {
		let s = this;
		if (this._direction === SCROLL_DIRECTION.VERTICAL) {
			s.distance = s.height;
			s.paramXY = "y";
			s.paramSize = 'height';
		} else {
			s.distance = s.width;
			s.paramXY = "x";
			s.paramSize = 'width';
		}
	}

	/**
	 * 设置可见区域，可见区域的坐标始终在本地坐标中0,0点位置
	 * @method setViewRect
	 * @public
	 * @since 1.1.1
	 */
	public updateViewRect(): void {
		let s: ScrollViewBase = this;
		s.maskObj.clear();
		s.maskObj.beginFill("#000000");
		s.maskObj.drawRect(0, 0, s.width, s.height);
		s.maskObj.endFill();

		this.updateDirection();
	}

	cancelScroll() {
		this.isMouseDownState = 0;
	}

	private onMouseEvent(e: MouseEvent): void {
		let s = this;
		let view: any = s.viewPort;
		// if (s.distance < s.maxDistance) {
		if (e.type == MouseEvent.MOUSE_DOWN) {
			e.stopPropagation();
			if (!s.isStop) {
				s.isStop = true;
			}
			if (s.autoScroll) {
				s.autoScroll = false;
				// Tween.kill(s._tweenId);
			}
			if (s._direction === SCROLL_DIRECTION.VERTICAL) {
				s.lastValue = e.stageY;
			} else {
				s.lastValue = e.stageX;
			}
			s.speed = 0;
			s.isMouseDownState = 1;
		} else if (e.type == MouseEvent.MOUSE_MOVE) {
			if (s.isMouseDownState == 0) return;
			if (s.isMouseDownState == 1) {
				s.dispatchEvent(Event.ON_SCROLL_START);
			}
			s.isMouseDownState = 2;
			let currentValue: number;
			if (s._direction === SCROLL_DIRECTION.VERTICAL) {
				currentValue = e.stageY;
			} else {
				currentValue = e.stageX;
			}
			s.speed = currentValue - s.lastValue;
			if (s.speed > s.minDis) {
				s.addSpeed = -2;
				if (s.speed > s.maxSpeed) {
					s.speed = s.maxSpeed;
				}
			} else if (s.speed < -s.minDis) {
				if (s.speed < -s.maxSpeed) {
					s.speed = -s.maxSpeed;
				}
				s.addSpeed = 2;
			} else {
				s.speed = 0;
			}
			if (s.speed != 0) {
				let speedPer: number = 1;
				if (view[s.paramXY] > 0 || view[s.paramXY] < s.distance - s.maxDistance) {
					speedPer = 0.2;
				}
				view[s.paramXY] += (currentValue - s.lastValue) * speedPer;
			}
			s.lastValue = currentValue;
			s.stopTimes = 0;
		} else {
			/*s.speed = 0;
			if (view[s.paramXY] > 0 || view[s.paramXY] < s.distance - s.maxDistance) {
				s.isStop = false;
				s.stopTimes = -1;
			}*/

			s.isStop = false;
			s.stopTimes = -1;
			if (s.speed == 0 && s.isMouseDownState == 2) {
				s.dispatchEvent(Event.ON_SCROLL_STOP);
			}
			s.isMouseDownState = 0;
		}
		// }
	}

	/**
	 * 滚到指定的坐标位置
	 * @method scrollTo
	 * @param {number} dis 需要去到的位置,初始位置0,最大为maxDistance- s.viewWidth : s.viewHeight
	 * @param {number} time 滚动需要的时间 默认为0 即没有动画效果直接跳到指定页
	 * @since 1.1.1
	 * @public
	 */
	public scrollTo(dis: number, time: number = 0): void {
		let s: ScrollViewBase = this;
		let newDis = s.paramXY == "x" ? s.width : s.height;
		if (dis < 0) {
			dis = 0;
		} else if (dis > s.maxDistance - newDis) {
			dis = s.maxDistance - newDis;
		}
		if (Math.abs(s.viewPort[s.paramXY] + dis) > 2) {
			// s.autoScroll = true;
			// s.isStop = true;
			// s.isMouseDownState = 0;
			// let obj: any = {};
			// obj.onComplete = function () {
			//     s.autoScroll = false;
			// };
			// obj[s.paramXY] = -dis;
			// s._tweenId = Tween.to(s.view, time, obj);
			// if (s.speed == 0) {
			//     s.dispatchEvent(Event.ON_SCROLL_START);
			// }
			s.isStop = true;
			s.isMouseDownState = 0;
			s.viewPort[s.paramXY] = -dis;
		}
	}

	public destroy(): void {
		let s = this;
		s.maskObj.destroy();
		s.viewPort.destroy();
		s.maskObj = null;
		s.viewPort = null;
		super.destroy();
	}
}
