import BaseTexture from './BaseTexture';
import {SCALE_MODES} from '../const';

/**
 * 将显示对象画在上面的贴图
 * A BaseRenderTexture is a special texture that allows any display object to be rendered to it.
 *
 * __Hint__: All DisplayObjects (i.e. Sprites) that render to a BaseRenderTexture should be preloaded
 * otherwise black rectangles will be drawn instead.
 *
 * A BaseRenderTexture takes a snapshot of any Display Object given to its render method. The position
 * and rotation of the given Display Objects is ignored. For example:
 *
 * ```js
 * let renderer = autoDetectRenderer(1024, 1024, { view: canvas, ratio: 1 });
 * let baseRenderTexture = new BaseRenderTexture(800, 600);
 * let sprite = Sprite.fromImage("spinObj_01.png");
 *
 * sprite.position.x = 800/2;
 * sprite.position.y = 600/2;
 * sprite.anchorTexture.x = 0.5;
 * sprite.anchorTexture.y = 0.5;
 *
 * renderer.render(sprite,baseRenderTexture);
 * ```
 *
 * The Sprite in this case will be rendered using its local transform. To render this sprite at 0,0
 * you can clear the transform
 *
 * ```js
 *
 * sprite.setTransform()
 *
 * let baseRenderTexture = new BaseRenderTexture(100, 100);
 * let renderTexture = new RenderTexture(baseRenderTexture);
 *
 * renderer.render(sprite, renderTexture);  // Renders to center of RenderTexture
 * ```
 *
 * @class
 * @extends BaseTexture
 */
export default class BaseRenderTexture extends BaseTexture {
	/**
	 * A map of renderer IDs to webgl renderTargets
	 * 画在他的帧缓存中
	 * @private
	 * @member {object<number, WebGLTexture>}
	 */
	_glRenderTargets: {};
	/**
	 * A reference to the canvas render target (we only need one as this can be shared across renderers)
	 * 画在他的上下文中canvas中
	 * @private
	 * @member {object<number, WebGLTexture>}
	 */
	_canvasRenderTarget: any;
	/**
	 * This will let the renderer know if the texture is valid. If it's not then it cannot be rendered.
	 */
	valid: boolean;

	/**
	 * @param {number} [width=100] - The width of the base render texture
	 * @param {number} [height=100] - The height of the base render texture
	 * @param {number} [scaleMode=SCALE_MODES.LINEAR] - See {@link SCALE_MODES} for possible values
	 */
	constructor(width: number = 100, height: number = 100, scaleMode: number = SCALE_MODES.LINEAR) {
		super(null, scaleMode);

		this.width = Math.ceil(width);
		this.height = Math.ceil(height);

		this.scaleMode = scaleMode;
		this.hasLoaded = true;


		this._glRenderTargets = {};


		this._canvasRenderTarget = null;


		this.valid = false;
	}

	/**
	 * Resizes the BaseRenderTexture.
	 *
	 * @param {number} width - The width to resize to.
	 * @param {number} height - The height to resize to.
	 */
	resize(width: number, height: number) {
		width = Math.ceil(width);
		height = Math.ceil(height);

		if (width === this.width && height === this.height) {
			return;
		}

		this.valid = (width > 0 && height > 0);

		this.width = width;
		this.height = height;

		if (!this.valid) {
			return;
		}

		this.dispatchEvent('update');
	}

	/**
	 * Destroys this texture
	 *
	 */
	destroy() {
		super.destroy();
	}
}
