import {SCALE_MODES, TEXT_ALIGN, TEXT_lINETYPE, VERTICAL_ALIGN} from "../const";
import Texture from "../texture/Texture";
import {getRGBA, hex2string} from "../utils/index";
import {Rectangle} from "../math/index";
import Sprite from "../display/Sprite";
import {Event} from "../events/index";

//文本canvas上xy的偏移量
const padding = 10;

const styleFields = {
	color: 'fillStyle',
	stroke: 'lineWidth',
	strokeColor: 'strokeStyle',
	font: 'font',
};

/**
 *
 * 继承Sprite，暂时发现，只需要切换bitmap和Sprite,TextField永远都是最新的，到时替换
 * 动态文本类,有时需要在canvas里有一个动态文本,能根据我们的显示内容来改变
 * @class TextFieldCon
 * @extends Sprite
 * @since 1.0.0
 * @public
 */
export class TextField extends Sprite {
	canvas: HTMLCanvasElement;
	context: CanvasRenderingContext2D;
	/**
	 * 与其他类不同，用了Boolean，再考虑
	 */
	dirty: boolean;

	offsetX: number;
	offsetY: number;

	constructor() {
		super();
		this._instanceType = "TextFieldNode";
		var canvas = document.createElement('canvas');
		canvas.width = 3;
		canvas.height = 3;
		const texture = Texture.fromCanvas(canvas, SCALE_MODES.LINEAR, 'textCanvas');
		texture.orig = new Rectangle();
		this.texture = texture;

		//baseTexture已自动缓存，把texture也缓存，key textCanvas+num  和baseTexture的一致
		Texture.addToCache(this._texture, this._texture.baseTexture.textureCacheIds[0]);
		this.canvas = canvas;
		this.context = canvas.getContext("2d");
		this.dirty = true;

	}

	/**
	 * @property textAlpha
	 * @since 2.0.0
	 * @public
	 */
	public set textAlpha(value: number) {
		if (this._textAlpha != value) {
			this._textAlpha = value;
			this.dirty = true;
		}
	}

	public get textAlpha(): number {
		return this._textAlpha;
	}

	private _textAlpha: number = 1;

	/**
	 * 文本的水平对齐方式 left center right
	 * 设置过textHeight才有效，如果是多行的，对排版有作用
	 * @property textAlign
	 * @public
	 * @since 1.0.0
	 * @type {string}
	 * @default left
	 */
	public set textAlign(value: TEXT_ALIGN) {
		if (this._textAlign != value) {
			this._textAlign = value;
			this.dirty = true;
		}
	}

	public get textAlign(): TEXT_ALIGN {
		return this._textAlign;
	}

	private _textAlign: TEXT_ALIGN = TEXT_ALIGN.LEFT;

	/**
	 * 垂直对齐方式
	 * 设置过textHeight才有效
	 */
	public set verticalAlign(value: VERTICAL_ALIGN) {
		if (this._verticalAlign != value) {
			this._verticalAlign = value;
			this.dirty = true;
		}
	}

	public get verticalAlign(): VERTICAL_ALIGN {
		return this._verticalAlign;
	}

	private _verticalAlign: VERTICAL_ALIGN = VERTICAL_ALIGN.UP;

	/**
	 * 文本的宽，
	 * @property textWidth
	 * @public
	 * @since 1.0.0
	 * @type {number}
	 * @default 0
	 */
	// public set textWidth(value: number) {
	// 	if (this._textWidth != value) {
	// 		this._textWidth = value;
	// 		this.dirty = true;
	// 	}
	// 	;
	// }

	// public get textWidth(): number {
	// 	if (this._textWidth) {
	// 		//有就这个
	// 		return this._textWidth
	// 	} else {
	// 		//没有就计算出的给width，还是
	// 		this.updateText();
	// 		return this.width - padding * 2;
	// 	}
	// }

	// private _textWidth: number = 0;

	/**
	 * 文本的行高，设置过能进行垂直适配，
	 * 文本超出行高会被裁切
	 * @property textHeight
	 * @public
	 * @since 1.0.0
	 * @type {number}
	 * @default 0
	 */
	// public set textHeight(value: number) {
	// 	if (this._textHeight != value) {
	// 		this._textHeight = value;
	// 		this.dirty = true;
	// 	}
	// }

	// public get textHeight(): number {
	// 	if (this._textHeight) {
	// 		//有就这个
	// 		return this._textHeight
	// 	} else {
	// 		//没有就计算出的给height，还是
	// 		this.updateText();
	// 		return this.height - padding * 2
	// 	}
	// }

	// private _textHeight: number = 0;

	get width(): number {
		if (this._width) return this._width;
		this.updateText();
		return this.scale.x * this.getLocalBounds().width;
	}

	set width(value: number) {
		if (this._width !== value) {
			this._width = value;
			this.dirty = true;
			this.dispatchEvent(Event.RESIZE);
		}
	}

	get height(): number {
		if (this._height) return this._height;
		this.updateText();
		return this.scale.y * this.getLocalBounds().height;
	}

	set height(value: number) {
		if (this._height !== value) {
			this._height = value;
			this.dirty = true;
			this.dispatchEvent(Event.RESIZE);
		}
	}

	/**
	 * 行间距
	 * @property lineSpacing
	 * @public
	 * @since 1.0.0
	 * @param {number} value
	 */
	public set lineSpacing(value: number) {
		if (this._lineSpacing != value) {
			this._lineSpacing = value;
			this.dirty = true;
		}
	}

	public get lineSpacing(): number {
		return this._lineSpacing;
	}

	private _lineSpacing: number = 0;

	/**
	 * 文本类型,单行还是多行 single multi
	 * @property lineType
	 * @public
	 * @since 1.0.0
	 * @type {string} 两种 single和multi
	 * @default single
	 */
	public set lineType(value: TEXT_lINETYPE) {
		if (this._lineType != value) {
			this._lineType = value;
			this.dirty = true;
		}
	}

	public get lineType(): TEXT_lINETYPE {
		return this._lineType;
	}

	private _lineType: TEXT_lINETYPE = TEXT_lINETYPE.SINGLE;

	protected _text: string = "";

	/**
	 * 文本内容
	 * @property text
	 * @type {string}
	 * @public
	 * @default ""
	 * @since 1.0.0
	 */
	public set text(value: string) {
		this._setText(value);
	}

	public get text(): string {
		return this.pureText;
	}

	protected _setText(value) {
		if (this._text != value) {
			this._text = value;
			this.dirty = true;

			this.dispatchEvent(Event.RESIZE);
		}
	}

	protected _textFlow: any;
	protected _pureText = '';

	get textFlow(): any {
		return this._textFlow;
	}

	set textFlow(value: any) {
		this._textFlow = value;
		this.dirty = true;

		this._styleCache.splice(0);
		let text = '';
		for (let item of this._textFlow) {
			text += item.text;
		}
		this._pureText = text;
	}

	get isPureText() {
		return !this._textFlow || this._textFlow.length == 0;
	}

	get pureText() {
		return this.isPureText ? this._text : this._pureText;
	}

	private _styleCache = [];

	protected getStyle(index) {
		if (!this.textFlow) {
			return null;
		}

		if (this._styleCache[index]) {
			return this._styleCache[index];
		}

		let targetItem;
		let count = 0;
		for (let item of this._textFlow) {
			count += item.text.length;
			if (index < count) {
				targetItem = item;
				break;
			}
		}

		if (targetItem && targetItem.style) {
			this._styleCache[index] = targetItem.style;
		}

		return targetItem.style;
	}

	/**
	 * 文本的css字体样式
	 * @property font
	 * @public
	 * @since 1.0.0
	 * @type {string}
	 * @default 12px Arial
	 */
	public set font(value: string) {
		if (this._font != value) {
			this._font = value;
			this.dirty = true;
		}
	}

	public get font(): string {
		return this._font;
	}

	private _font: string = "Arial";

	/**
	 * 文本的size
	 * @property size
	 * @public
	 * @since 1.0.0
	 * @type {number}
	 * @default 12
	 */
	public set size(value: number) {
		if (this._size != value) {
			this._size = value;
			this.dirty = true;
		}
	}

	public get size(): number {
		return this._size;
	}

	private _size: number = 12;

	/**
	 * 文本的填充颜色值
	 * @property fillColor
	 * @type {string}
	 * @public
	 * @since 1.0.0
	 * @default #fff
	 */
	public set fillColor(value: any) {
		if (this._fillColor != value) {
			this._fillColor = value;
			this.dirty = true;
		}
	}

	public get fillColor(): any {
		return this._fillColor;
	}

	private _fillColor: any = "#000";

	/**
	 * 文本的描边颜色值
	 * @property strokeColor
	 * @type {string}
	 * @public
	 * @since 1.0.0
	 * @default #fff
	 */
	public set strokeColor(value: string) {
		if (this._strokeColor != value) {
			this._strokeColor = value;
			this.dirty = true;
		}
	}

	public get strokeColor(): string {
		return this._strokeColor;
	}

	private _strokeColor: string = "#ffffff";

	/**
	 * 文本描边宽度,为0则不描边
	 * @property stroke
	 * @public
	 * @since
	 * @default 0
	 * @type {number}
	 */
	public set stroke(value: number) {
		if (this._stroke != value) {
			this._stroke = value;
			this.dirty = true;
		}
	}

	public get stroke(): number {
		return this._stroke;
	}

	private _stroke: number = 0;

	/**
	 * 文本是否倾斜
	 * @property italic
	 * @public
	 * @since
	 * @default false
	 * @type {boolean}
	 */
	public set italic(value: boolean) {
		if (this._italic != value) {
			this._italic = value;
			this.dirty = true;
		}
	}

	public get italic(): boolean {
		return this._italic;
	}

	private _italic: boolean = false;

	/**
	 * 文本是否加粗
	 * @property bold
	 * @public
	 * @since
	 * @default false
	 * @type {boolean}
	 */
	public set bold(value: boolean) {
		if (this._bold != value) {
			this._bold = value;
			this.dirty = true;
		}
	}

	public get bold(): boolean {
		return this._bold;
	}

	public _bold: boolean = false;

	/**
	 * 设置或获取是否有边框
	 * @property property
	 * @param {boolean} show true或false
	 * @public
	 * @since 1.0.6
	 */
	public set border(value: boolean) {
		if (this._border != value) {
			this._border = value;
			this.dirty = true;
		}
	}

	public get border(): boolean {
		return this._border;
	}

	private _border: boolean = false;

	private _setupFont(font, size, bold, italic) {
		let fontStyle: any = size;
		fontStyle += "px ";
		fontStyle += font;
		if (bold) {
			fontStyle = "bold " + fontStyle;
		}
		if (italic) {
			fontStyle = "italic " + fontStyle;
		}
		return fontStyle;
	}

	/**
	 * 设置文本在canvas里的渲染样式
	 * @method _prepContext
	 * @param ctx
	 * @private
	 * @since 1.0.0
	 */
	private _prepContext(ctx: any): void {
		let s = this;
		ctx.font = this._setupFont(s._font, s._size, s._bold, s._italic);
		ctx.textAlign = s.isPureText ? (s._textAlign || TEXT_ALIGN.LEFT) : TEXT_ALIGN.LEFT;
		//暂时没开放
		ctx.textBaseline = s.isPureText ? 'top' : 'bottom';
		//数字转换
		if (typeof (s._fillColor) == "number") s._fillColor = hex2string(s._fillColor);
		//获取fillStyle
		ctx.fillStyle = s._textAlpha === 1 ? s._fillColor : getRGBA(s._fillColor, s._textAlpha);
	}

	/**
	 * 获取当前文本中单行文字的宽，注意是文字的不是文本框的宽
	 * @method getTextWidth
	 * @param {number} lineIndex 获取的哪一行的高度 默认是第1行
	 * @since 2.0.0
	 * @public
	 * @return {number}
	 */
	public getTextWidth(lineIndex: number = 0): number {
		let s = this;
		s.updateText();
		// let can = s.canvas;
		let ctx = s.context;
		let obj: any = ctx.measureText(s.realLines[lineIndex]);
		return obj.width;
	}

	/**
	 * @property _lines 获取当前文本行数
	 * @type {number}
	 * @public
	 * @readonly
	 * @since 2.0.0
	 */
	get lines(): number {
		return this.realLines.length;
	}

	/**
	 * 获取文本宽
	 * @method _getMeasuredWidth
	 * @param text
	 * @param style
	 * @return {number}
	 * @private
	 * @since 1.0.0
	 */
	private _getMeasuredWidth(text: string, style?): number {
		let ctx = this.context;

		ctx.save();
		if (style && style.font) {
			ctx.font = style.font;
		}

		let w = ctx.measureText(text).width;

		ctx.restore();

		return w;
	}

	private realLines: any = [];

	private _getFontSize(style) {
		if (style && style.font) {
			return parseInt(style.font);
		}
	}

	/**
	 * 更新文本，主要重画canvas
	 */
	public updateText(): void {
		let s: TextField = this;
		let text = s.pureText;
		//如果没有文本
		if (!text) {
			s.canvas.width = 0;
			s.canvas.height = 0;
			if(!this._width && !this._height){
				s._localBoundsSelf.clear();
			}
			this.anchorTexture = {x: 0, y: 0};
			this.updateTexture();
			return
		}
		let measureCache = {};
		if (!s.dirty) return;
		s.dirty = false;

		let {isPureText} = s;

		text += "";
		let can = s.canvas;
		let ctx = s.context;
		let hardLines: any = text.toString().split(/(?:\r\n|\r|\n)/);
		let realLines: any = [];
		let lineWidths = [];
		let lineHeights = [];
		s.realLines = realLines;
		s._prepContext(ctx);

		let textWidth = s._width;
		// let lineH = s._lineSpacing + s.size;
		//单行文本时
		if (isPureText && text.indexOf("\n") < 0 && s.lineType == TEXT_lINETYPE.SINGLE) {
			realLines[realLines.length] = hardLines[0];
			let str = hardLines[0];
			let lineW = s._getMeasuredWidth(str);
			//如果没设置过textWidth就取lineW
			textWidth = textWidth || lineW;
			//如果设置过textWidth的。截取文本
			if (lineW > textWidth) {
				let w = s._getMeasuredWidth(str[0]);
				let lineStr = str[0];
				let wordW = 0;
				let strLen = str.length;
				for (let j = 1; j < strLen; j++) {
					wordW = ctx.measureText(str[j]).width;
					w += wordW;
					if (w > textWidth) {
						realLines[0] = lineStr;
						break;
					} else {
						lineStr += str[j];
					}
				}
			}
		} else {
			//textWidth取每行最大值，如果没设置过textWidth
			const shouldMeasureTextWidth = !textWidth;
			let index = 0;
			for (let i = 0, l = hardLines.length; i < l; i++) {
				let str = hardLines[i];
				if (!str) continue;
				let lineWidth = 0;
				for (let char of str) {
					let charWidth = measureChar(char, index);
					lineWidth += charWidth;
					index++;
				}
				if (shouldMeasureTextWidth) {
					textWidth = Math.max(lineWidth, textWidth);
				}
			}
			index = 0;
			for (let i = 0, l = hardLines.length; i < l; i++) {
				let str = hardLines[i];
				if (!str) continue;
				let w = measureChar(str[0], index++);
				let lineStr = str[0];
				let wordW = 0;
				let strLen = str.length;
				let lineHeight = 0;
				for (let j = 1; j < strLen; j++) {
					wordW = measureChar(str[j], index);
					w += wordW;
					if (w > textWidth) {
						realLines[realLines.length] = lineStr;
						lineWidths.push(w - wordW);
						lineHeights.push(lineHeight || s.size);
						lineStr = str[j];
						w = wordW;
						lineHeight = 0;
					} else {
						lineStr += str[j];
					}
					let style = this.getStyle(index);
					let size = this._getFontSize(style);
					if (size) {
						if (size > lineHeight) {
							lineHeight = size;
						}
					}
					index++;
				}
				realLines[realLines.length] = lineStr;
				lineWidths.push(w);
				lineHeights.push(lineHeight || s.size);
			}
		}
		//console.log(lineHeights);
		// let maxH = lineH * realLines.length;

		let maxH, trueHeight;

		if (isPureText) {
			trueHeight = s.size * realLines.length + s._lineSpacing * (realLines.length - 1);
		} else {
			trueHeight = lineHeights.reduce((a, b) => {
				return a + b;
			}, 0) + s._lineSpacing * (realLines.length - 1);
		}

		if (s._height) {
			maxH = s._height
		} else {
			maxH = trueHeight;
		}
		let maxW = textWidth;
		can.width = maxW + padding * 2;
		can.height = maxH + padding * 2;
		ctx.clearRect(0, 0, can.width, can.height);
		if (s.border) {
			ctx.beginPath();
			ctx.strokeStyle = "#000";
			ctx.lineWidth = 1;
			ctx.strokeRect(padding + 0.5, padding + 0.5, maxW, maxH);
			ctx.closePath();
		}
		let tx = 0;
		if (isPureText) {
			if (s._textAlign == TEXT_ALIGN.CENTER) {
				tx = maxW * 0.5;
			} else if (s._textAlign == TEXT_ALIGN.RIGHT) {
				tx = maxW;
			}
		}
		ctx.setTransform(1, 0, 0, 1, tx + padding, padding);
		s._prepContext(ctx);
		let lineH = s._lineSpacing + s.size;
		//如果有_textHeight,就需要应用竖直对齐
		var upY: number = 0;
		if (s._height) {
			//跟trueHeight比较
			if (s._verticalAlign == VERTICAL_ALIGN.MIDDLE) {
				upY = (s._height - trueHeight) * 0.5;
			} else if (s._verticalAlign == VERTICAL_ALIGN.DOWN) {
				upY = s._height - trueHeight;
			}
		}
		let index = 0;
		let lineY = upY;
		for (let i = 0; i < realLines.length; i++) {
			let line = realLines[i];
			if (isPureText) {
				let y = upY + i * lineH;
				if (s.stroke) {
					ctx.strokeStyle = s.strokeColor;
					ctx.lineWidth = s.stroke * 2;
					ctx.strokeText(line, 0, y, maxW);
				}
				ctx.fillText(line, 0, y, maxW);
			} else {
				let lineWidth = lineWidths[i];
				let lineHeight = lineHeights[i];
				lineY += s._lineSpacing + lineHeight;
				let x = 0;
				if (s._textAlign == TEXT_ALIGN.CENTER) {
					x = (maxW - lineWidth) * 0.5;
				} else if (s._textAlign == TEXT_ALIGN.RIGHT) {
					x = maxW - lineWidth;
				}
				for (let j = 0, lj = line.length; j < lj; j++) {
					const char = line[j];
					let style = s.getStyle(index);
					if (style) {
						ctx.save();
						for (let field in styleFields) {
							if (style.hasOwnProperty(field)) {
								ctx[styleFields[field]] = style[field];
							}
						}
					}
					if (s.stroke) {
						ctx.strokeStyle = s.strokeColor;
						ctx.lineWidth = s.stroke * 2;
						ctx.strokeText(char, x, lineY);
					}
					ctx.fillText(char, x, lineY);
					if (style) {
						ctx.restore();
					}
					x += measureChar(char, index);
					index++;
				}
			}
		}
		//offset用_anchorTexture代替
		s.offsetX = -padding;
		s.offsetY = -padding;

		this.anchorTexture = {x: (padding + 0.5) / can.width, y: padding / can.height};

		//document.body.appendChild(can)
		// s._bounds.height = maxH;
		// s._bounds.width = maxW;
		//x,y都是0
		s._localBoundsSelf.width = maxW;
		s._localBoundsSelf.height = maxH;
		//修改texture及baseTexture属性
		s.updateTexture();

		function measureChar(char, index?) {
			let key = char + ':';

			let style;
			if (!isPureText && index !== undefined) {
				style = s.getStyle(index);
				if (style && style.font) {
					key += style.font;
				}
			}
			let w = measureCache[key];
			if (w === undefined) {
				w = measureCache[char] = s._getMeasuredWidth(char, style);
			}
			return w;
		}
	}

	/**
	 * 更新texture及baseTexture属性
	 * 不考虑trim，
	 */
	updateTexture() {
		const canvas = this.canvas;
		const texture = this._texture;
		const baseTexture = texture.baseTexture;
		baseTexture.hasLoaded = canvas.width && canvas.height ? true : false;
		baseTexture.width = canvas.width;
		baseTexture.height = canvas.height;
		//texture,已绑定过尺寸改变onBaseTextureUpdated
		baseTexture.dispatchEvent('update');
		texture.valid = baseTexture.hasLoaded;
		texture._frame.width = canvas.width;
		texture._frame.height = canvas.height;
		texture.orig.width = texture._frame.width;
		texture.orig.height = texture._frame.height;
	}


	_renderCanvas(renderer) {
		this.updateText();
		super._renderCanvas(renderer);
	}

	_renderWebGL(renderer) {
		this.updateText();
		super._renderWebGL(renderer)
	}

	public destroy(): void {
		super.destroy();
		//todo
	}
}
