import {SHAPES} from '../../const';
import {HashObject} from '../../HashObject';
import CanvasRenderer from '../CanvasRenderer';
import Graphics from '../../graphics/Graphics';
import {holePath, judgeCcw} from '../plugins/CanvasGraphicsRenderer';

/**
 *
 */
export default class CanvasMaskManager extends HashObject {
	renderer

	/**
	 * @param {CanvasRenderer} renderer - The canvas renderer.
	 */
	constructor(renderer: CanvasRenderer) {
		super();
		this._instanceType = "CanvasMaskManager"
		this.renderer = renderer;
	}

	/**
	 * This method adds it to the current stack of masks.
	 *
	 * @param {object} maskData - the maskData that will be pushed
	 */
	pushMask(maskData: Graphics) {
		const renderer = this.renderer;
		renderer.context.save();
		const transform = maskData.transform.worldMatrix;
		renderer.context.setTransform(
			transform.a,
			transform.b,
			transform.c,
			transform.d,
			transform.tx,
			transform.ty
		);
		//兼容下canvas下也能用Shape作为遮罩
		if (maskData.instanceType == "Shape") {
			maskData["_drawShape"](renderer.context)
		} else {
			this.renderGraphicsShape(maskData);
		}
		renderer.context.clip();
	}

	/**
	 * Renders a Graphics shape.
	 *
	 * @param {Graphics} graphics - The object to render.
	 */
	renderGraphicsShape(graphics: Graphics) {
		const context = this.renderer.context;
		const len = graphics.graphicsData.length;

		if (len === 0) return;

		//beginPath 清空子路径列表
		//clip和fill都会把当beginPath后的所有的子路径进行裁剪或填充，且闭合当前子路径，但是不会清空子路径列表
		//stroke不会闭合当前子路径
		//closePath闭合当前子路径，和beginPath并不对应，手动将首尾闭合就没必要调用closePath
		//rect，arc都是顺时针的，rect会首尾闭合，arc全角也会闭合
		//几个都要应用的话，只执行一次beginPath
		context.beginPath();
		for (let i = 0; i < len; i++) {
			const data = graphics.graphicsData[i];
			const shape = data.shape;
			var ccw = true;
			if (data.type === SHAPES.POLY) {
				const points = shape.points;

				context.moveTo(points[0], points[1]);

				for (let j = 1; j < points.length / 2; j++) {
					context.lineTo(points[j * 2], points[(j * 2) + 1]);
				}
				ccw = !judgeCcw(points)
				// if the first and last point are the same close the path - much neater :)
				if (points[0] === points[points.length - 2] && points[1] === points[points.length - 1]) {
					// context.closePath();
				} else {
					context.closePath();
				}
			} else if (data.type === SHAPES.RECT) {
				context.rect(shape.x, shape.y, shape.width, shape.height);
			} else if (data.type === SHAPES.CIRC) {
				// TODO - need to be Undefined!
				context.arc(shape.x, shape.y, shape.radius, 0, 2 * Math.PI);
			} else if (data.type === SHAPES.ELIP) {
				// ellipse code taken from: http://stackoverflow.com/questions/2172798/how-to-draw-an-oval-in-html5-canvas

				const w = shape.width * 2;
				const h = shape.height * 2;

				const x = shape.x - (w / 2);
				const y = shape.y - (h / 2);

				const kappa = 0.5522848;
				const ox = (w / 2) * kappa; // control point offset horizontal
				const oy = (h / 2) * kappa; // control point offset vertical
				const xe = x + w;           // x-end
				const ye = y + h;           // y-end
				const xm = x + (w / 2);       // x-middle
				const ym = y + (h / 2);       // y-middle

				context.moveTo(x, ym);
				context.bezierCurveTo(x, ym - oy, xm - ox, y, xm, y);
				context.bezierCurveTo(xm + ox, y, xe, ym - oy, xe, ym);
				context.bezierCurveTo(xe, ym + oy, xm + ox, ye, xm, ye);
				context.bezierCurveTo(xm - ox, ye, x, ym + oy, x, ym);
			} else if (data.type === SHAPES.RREC) {
				const rx = shape.x;
				const ry = shape.y;
				const width = shape.width;
				const height = shape.height;
				let radius = shape.radius;

				const maxRadius = Math.min(width, height) / 2 | 0;

				radius = radius > maxRadius ? maxRadius : radius;

				context.moveTo(rx, ry + radius);
				context.quadraticCurveTo(rx, ry, rx + radius, ry);
				context.lineTo(rx + width - radius, ry);
				context.quadraticCurveTo(rx + width, ry, rx + width, ry + radius);
				context.lineTo(rx + width, ry + height - radius);
				context.quadraticCurveTo(rx + width, ry + height, rx + width - radius, ry + height);
				context.lineTo(rx + radius, ry + height);
				context.quadraticCurveTo(rx, ry + height, rx, ry + height - radius);
				context.lineTo(rx, ry + radius);
			}
			for (let j = 0; j < data.holes.length; j++) {
				holePath(data.holes[i], context, ccw)
			}
		}
	}

	/**
	 * Restores the current drawing context to the state it was before the mask was applied.
	 *
	 * @param {CanvasRenderer} renderer - The renderer context to use.
	 */
	popMask(renderer: CanvasRenderer) {
		renderer.context.restore();
	}

	/**
	 * Destroys this canvas mask manager.
	 *
	 */
	destroy() {
		/* empty */
	}
}
