import {EventDispatcher} from "../events/EventDispatcher";
import {TextureCache} from "../utils";
import {Texture} from "../texture";
import {httpRequest} from "../net";
import {createTextureSheet} from "../../zeroing/game-warpper/texture-sheet";
import {preloadSound} from "../../zeroing/game-warpper/sound";

export class Loader extends EventDispatcher {

	/**
	 * 记录原始数据，json和image，贴图在建立时会被缓存
	 */
	caches = {};
	crossOrigin: boolean = true;

	//parser: Parser;

	/**
	 *
	 */
	constructor() {
		super();
		this._instanceType = "Loader";
	}

	/**
	 *
	 * @param url
	 */
	loadSheet(url: string) {
		let config;
		return this.loadJson(url)
			.then(json => {
				const {file} = config = json;
				return this.loadTexture(file)
			})
			.then(texture => {
				createTextureSheet(texture.baseTexture, config.frames);
				return config;
			})
	}

	loadRaw(url: string, uuid?: string, type?: 'text' | 'json') {
		let data = this.caches[uuid || url];
		if (data) {
			return Promise.resolve(data);
		} else {
			return httpRequest(url, 'get', {}, type)
				.then(data => {
					this.cache(uuid || url, data);
					return data;
				});
		}
	}

	loadJson(url: string, uuid?: string) {
		return this.loadRaw(url, uuid, 'json');
	}

	loadText(url: string, uuid?: string) {
		return this.loadRaw(url, uuid, 'text');
	}

	loadSound(url: string, uuid?: string) {
		preloadSound(url, uuid);

		return Promise.resolve();
	}

	loadImage(url: string, uuid?: string) {
		return new Promise((resolve, reject) => {
			let img = this.caches[uuid || url];
			if (img) {
				resolve(img);
			} else {
				img = new Image();
				if (this.crossOrigin) {
					img.setAttribute('crossOrigin', 'anonymous');
				}
				img.onload = () => {
					this.cache(uuid || url, img);
					resolve(img);
				};
				img.onerror = function (e) {
					reject(e);
				};
				img.src = url;
			}
		})
	}

	/**
	 * 加载纹理
	 * @param url
	 * @param uuid
	 */
	loadTexture(url: string, uuid?: string) {
		return this.loadImage(url)
			.then(image => {
				return TextureCache[uuid || url] = Texture.from(image);
			})
	}

	/*loadSvga(callback: (suc: boolean, data: VideoEntity) => void, url: string) {
		if (!this.parser) this.parser = new Parser();
		this.parser.load(url, function (videoItem: VideoEntity) {
			callback(true, videoItem)
			//是否缓存
		}, function (error) {
			callback(false, error)
			// alert(error.message);
		})
	}*/

	cache(name: string, data: any) {
		if (this.caches[name]) {
			//console.log("覆盖原先数据：" + name);
		}
		this.caches[name] = data;
	}

	get(name: string) {
		return this.caches[name];
	}
}

async function fetchAsync(url: string) {
	// await response of fetch call
	let response = await fetch(url);
	// only proceed once promise is resolved
	let data = await response.json();
	// only proceed once second promise is resolved
	return data;
}

// fetchAsync("")
//     .then(data => console.log(data))
//     .catch(reason => console.log(reason.message))

export const globalLoader = new Loader();