import {Point, Rectangle} from '../../math';
import {SHAPES} from '../../const';

/**
 * The Ellipse object can be used to specify a hit area for displayObjects
 *
 */
export default class Ellipse {
	/**
	 * @member {number}
	 * @default 0
	 */
	x: number;
	/**
	 * @member {number}
	 * @default 0
	 */
	y: number;
	/**
	 * @member {number}
	 * @default 0
	 */
	width: number;
	/**
	 * @member {number}
	 * @default 0
	 */
	height: number;
	/**
	 * The type of the object, mainly used to avoid `instanceof` checks
	 *
	 * @member {number}
	 * @readOnly
	 * @default SHAPES.ELIP
	 * @see SHAPES
	 */
	type: number;

	/**
	 * @param {number} [x=0] - The X coordinate of the center of this circle
	 * @param {number} [y=0] - The Y coordinate of the center of this circle
	 * @param {number} [width=0] - The half width of this ellipse
	 * @param {number} [height=0] - The half height of this ellipse
	 */
	constructor(x: number = 0, y: number = 0, width: number = 0, height: number = 0) {
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		this.type = SHAPES.ELIP;
	}

	/**
	 * Creates a clone of this Ellipse instance
	 *
	 * @return {Ellipse} a copy of the ellipse
	 */
	clone(): Ellipse {
		return new Ellipse(this.x, this.y, this.width, this.height);
	}

	/**
	 * Checks whether the x and y coordinates given are contained within this ellipse
	 *
	 * @param {Point} point - The  point to test
	 * @return {boolean} Whether the x/y coords are within this ellipse
	 */
	isPointIn(point: Point): boolean {
		if (this.width <= 0 || this.height <= 0) {
			return false;
		}

		// normalize the coords to an ellipse with center 0,0
		let normx = ((point.x - this.x) / this.width);
		let normy = ((point.y - this.y) / this.height);

		normx *= normx;
		normy *= normy;

		return (normx + normy <= 1);
	}

	/**
	 * Returns the framing rectangle of the ellipse as a Rectangle object
	 *
	 * @return {Rectangle} the framing rectangle
	 */
	getBounds(): Rectangle {
		return new Rectangle(this.x - this.width, this.y - this.height, this.width, this.height);
	}
}
