import {Point, Rectangle} from '../../math';
import {SHAPES} from '../../const';

/**
 * 圆形
 */
export default class Circle {
	/**
	 * @member {number}
	 * @default 0
	 */
	x: number;
	/**
	 * @member {number}
	 * @default 0
	 */
	y: number;
	/**
	 * @member {number}
	 * @default 0
	 */
	radius: number;
	/**
	 * 类型
	 * @member {number}
	 * @readOnly
	 * @default SHAPES.CIRC
	 * @see SHAPES
	 */
	type: number;

	/**
	 * @param {number} [x=0] - The X coordinate of the center of this circle
	 * @param {number} [y=0] - The Y coordinate of the center of this circle
	 * @param {number} [radius=0] - The radius of the circle
	 */
	constructor(x: number = 0, y: number = 0, radius: number = 0) {
		this.x = x;
		this.y = y;
		this.radius = radius;
		this.type = SHAPES.CIRC;
	}

	/**
	 * Creates a clone of this Circle instance
	 *
	 * @return {Circle} a copy of the Circle
	 */
	clone(): Circle {
		return new Circle(this.x, this.y, this.radius);
	}

	/**
	 * @param {Point} point - The point to test
	 * @return {boolean} Whether the x/y coordinates are within this Circle
	 */
	isPointIn(point: Point): boolean {
		if (this.radius <= 0) {
			return false;
		}

		const r2 = this.radius * this.radius;
		let dx = (this.x - point.x);
		let dy = (this.y - point.y);
		dx *= dx;
		dy *= dy;

		return (dx + dy <= r2);
	}

	/**
	 * Returns the framing rectangle of the circle as a Rectangle object
	 *
	 * @return {Rectangle} the framing rectangle
	 */
	getBounds(): Rectangle {
		return new Rectangle(this.x - this.radius, this.y - this.radius, this.radius * 2, this.radius * 2);
	}
}
