import {DisplayObject} from './DisplayObject';
import {Rectangle} from "../math/Rectangle";
import {Point} from '../math/index';
import CanvasRenderer from '../renderers/CanvasRenderer';
import {Event} from "../events/Event"
import {WebglRenderer} from '../renderers/WebglRenderer';

import {applyAutoAdjust} from "../../zeroing/decorators/auto-adjust";
import {applyScript} from "../../zeroing/decorators/scripts";
import {applyEvents} from "../../zeroing/decorators/events";
import {afterConstructor} from "../../zeroing/decorators/after-constructor";
import {injectProperties, instantiateScript} from "../../zeroing/utils/index";
import {isUI} from "../../zeroing/game-warpper/nodes/IUIComponent";
import Transform from "../math/Transform";

/**
 * 容器类
 * @class
 * @extends DisplayObject
 */
@afterConstructor
@applyAutoAdjust
@applyScript
@applyEvents
export default class Container extends DisplayObject {
	percentWidth: number;
	percentHeight: number;
	left: number;
	top: number;
	right: number;
	bottom: number;
	horizonCenter: number;
	verticalCenter: number;

	private _transform: Transform;
	private _lastLocalID;

	/**
	 * 为false鼠标事件不再向下传递
	 */
	_mouseChildren: boolean = true;
	/**
	 * 孩子们
	 * @member {DisplayObject[]}
	 * @readonly
	 */
	children: any[];

	/**
	 * 特殊用处
	 * 缓存的Container的updateTransform
	 */
	containerUpdateTransform;

	constructor() {
		super();
		this._instanceType = "Container";
		this.children = [];

		this._transform = this.transform;

		this.afterConstructor();
	}

	get mouseChildren() {
		return this._mouseChildren;
	}

	set mouseChildren(v) {
		this._mouseChildren = v;
	}

	afterConstructor() {

	}

	/**
	 * children改变时
	 * @private
	 */
	onChildrenChange(index) {
		/* empty */
		//子类需要时重写
	}

	attachVisibility() {
		return;
		this.dispatchEvent(this.worldVisible ? Event.VISIBLE : Event.HIDDEN);
		for (let i = 0, li = this.children.length; i < li; i++) {
			const child = <Container>this.children[i];
			child.attachVisibility();
		}
	}

	/**
	 * 添加child
	 * @param {DisplayObject} child
	 * @return {DisplayObject}
	 */
	addChild(child: DisplayObject): DisplayObject {
		//默认添加在最顶层
		this.addChildAt(child, this.children.length);
		return child;
	}

	/**
	 * 在相应index处添加child
	 * @param {DisplayObject} child - The child to add
	 * @param {number} index - The index to place the child in
	 * @return {DisplayObject} The child that was added.
	 */
	addChildAt(child: DisplayObject, index: number): DisplayObject {
		if (!child) return;

		let s = this;
		let sameParent = (s == child.parent);
		let len: number;
		if (child.parent) {
			if (!sameParent) {
				child.parent.removeChild(child);
			} else {
				len = s.children.length;
				for (let i = 0; i < len; i++) {
					if (s.children[i] == child) {
						s.children.splice(i, 1);
						break;
					}
				}
			}
		}
		child.parent = s;

		//保证child的transform会被更新
		child.transform._parentID = -1;

		//确保包围盒重新计算
		this._boundsID++;

		len = s.children.length;
		if (index >= len) {
			s.children[s.children.length] = child;
			index = len;
		} else if (index == 0 || index < 0) {
			s.children.unshift(child);
			index = 0;
		} else {
			s.children.splice(index, 0, child);
		}
		if (s.stage && !sameParent) {
			// child["_cp"] = true;
			child._onDispatchBubbledEvent(Event.ADDED_TO_STAGE);
		}

		this.onChildrenChange(index);

		return child;
	}

	/**
	 * 只用于交换索引
	 * @param {DisplayObject} child - First display object to swap
	 * @param {DisplayObject} child2 - Second display object to swap
	 */
	swapChildren(child1: DisplayObject, child2: DisplayObject) {
		if (child1 === child2) {
			return;
		}
		let s = this;
		let id1 = -1;
		let id2 = -1;
		let childCount = s.children.length;
		if (typeof (child1) == "number") {
			id1 = child1;
		} else {
			id1 = s.getChildIndex(child1);
		}
		if (typeof (child2) == "number") {
			id2 = child2;
		} else {
			id2 = s.getChildIndex(child2);
		}
		if (id1 == id2 || id1 < 0 || id1 >= childCount || id2 < 0 || id2 >= childCount) {
			return false;
		} else {
			let temp: any = s.children[id1];
			s.children[id1] = s.children[id2];
			s.children[id2] = temp;

			this.onChildrenChange(id1 < id2 ? id1 : id2);
			return true;
		}

	}

	/**
	 * 获取child的层级索引index
	 * @param {DisplayObject} child - The DisplayObject instance to identify
	 * @return {number} The index position of the child display object to identify
	 */
	getChildIndex(child: DisplayObject): number {
		const index = this.children.indexOf(child);

		if (index === -1) {
			return null
		}
		return index;
	}

	/**
	 * 是否含有child
	 * @param child
	 */
	contains(child: DisplayObject): boolean {
		return !!this.getChildIndex(child);
	}

	/**
	 * 设置child的层级
	 * @param {DisplayObject} child
	 * @param {number} index
	 */
	setChildIndex(child: DisplayObject, index: number) {
		this.addChildAt(child, index);
	}

	/**
	 * Returns the child at the specified index
	 * @param {number} index - The index to get the child at
	 * @return {DisplayObject} The child at the given index, if any.
	 */
	getChildAt(index: number): DisplayObject {
		if (index < 0 || index >= this.children.length) {
			return null
		}
		return this.children[index];
	}

	/**
	 * 根据路径获取子节点
	 * @param path
	 * @param method
	 */
	getChildByPath(path: any, method: string): DisplayObject {
		if (!path) {
			return null;
		}
		let p = this;
		while (path.length > 0) {
			let segment = path.shift();
			p = p[method](segment);
			if (!p) {
				break;
			}
		}

		return p;
	}

	/**
	 * 根据名称路径获取子节点
	 * @param path
	 */
	getChildByNamePath(path: string): DisplayObject {
		const pathArr = path.split('/');
		return this.getChildByPath(pathArr, 'getChildByName');
	}

	/**
	 * 根据索引路径获取子节点
	 * @param path
	 */
	getChildByIndexPath(path: string): DisplayObject {
		const pathArr = path.split('/').map(seg => parseInt(seg));
		return this.getChildByPath(pathArr, 'getChildAt');
	}

	/**
	 * 根据uuid搜索子节点
	 * @param uuid
	 */
	findChildByUUID(uuid: string) {
		if (this['uuid'] === uuid) {
			return this;
		}

		if (this.children && this.children.length > 0) {
			for (let child of this.children) {
				if (child.findChildByUUID) {
					let target = child.findChildByUUID(uuid);
					if (target) {
						return target;
					}
				}
			}
		}
	}

	/**
	 * 通过名字获取子级
	 * @param name
	 * @param isOnlyOne
	 * @param isRecursive
	 */
	public getChildByName(name: string | RegExp, isOnlyOne: boolean = true, isRecursive: boolean = false): any {
		if (!name) return null;
		let s = this;
		let rex: any;
		if (typeof (name) == "string") {
			rex = new RegExp("^" + name + "$");
		} else {
			rex = name;
		}
		let elements: Array<DisplayObject> = [];
		Container._getElementsByName(rex, s, isOnlyOne, isRecursive, elements);
		let len = elements.length;
		if (len == 0) {
			return null;
		} else if (len == 1) {
			return elements[0];
		} else {
			return elements;
		}
	}

	/**
	 * 移除child
	 * @param {DisplayObject} child
	 * @return {DisplayObject}
	 */
	removeChild(child: DisplayObject): DisplayObject {

		const index = this.children.indexOf(child);

		if (index === -1) return null;

		this.removeChildAt(index);

		return child;
	}

	/**
	 * 在index处移除child
	 * @param {number} index - The index to get the child from
	 * @return {DisplayObject} The child that was removed.
	 */
	removeChildAt(index: number): DisplayObject {
		let s = this;
		let child: any;
		let len = s.children.length - 1;
		if (len < 0) return;
		if (index == len) {
			child = s.children.pop();
		} else if (index == 0) {
			child = s.children.shift();
		} else {
			child = s.children.splice(index, 1)[0];
		}
		child._onDispatchBubbledEvent(Event.REMOVED_FROM_STAGE);

		//保证子级会被更新
		child.parent = null;
		child.transform._parentID = -1;

		//保证包围盒重新计算
		this._boundsID++;

		this.onChildrenChange(index);

		return child;
	}

	/**
	 * 通过索引批量移除child
	 * @param {number} [beginIndex=0]
	 * @param {number} [endIndex=this.children.length]
	 * @returns {DisplayObject[]} List of removed children
	 */
	removeChildren(beginIndex: number = 0, endIndex: number = this.children.length): DisplayObject[] {
		const begin = beginIndex;
		const end = typeof endIndex === 'number' ? endIndex : this.children.length;
		const range = end - begin;
		let removed;

		if (range > 0 && range <= end) {
			removed = this.children.splice(begin, range);

			for (let i = 0; i < removed.length; ++i) {
				removed[i].parent = null;
				if (removed[i].transform) {
					removed[i].transform._parentID = -1;
				}
			}

			this._boundsID++;

			this.onChildrenChange(beginIndex);

			for (let i = 0; i < removed.length; ++i) {
				removed[i]._onDispatchBubbledEvent(Event.REMOVED_FROM_STAGE);
			}

			return removed;
		} else if (range === 0 && this.children.length === 0) {
			return [];
		}

		throw new RangeError('removeChildren: numeric values are outside the acceptable range.');
	}

	/**
	 * 更新矩阵
	 */
	updateTransform() {
		//自己先算
		super.updateTransform();
		//考虑是否要加，
		this._boundsID++;
		//children遍历计算
		for (let i = 0, j = this.children.length; i < j; ++i) {
			const child = this.children[i];
			if (child.visible) {
				child.updateTransform();
			}
		}

		/*if(this._transform === this.transform && isUI(this) && this._lastLocalID !== this.transform.localID){
			this._lastLocalID = this.transform.localID;
			//console.log(this.name, this.instanceId , 'dirty!');
			this.stage.layoutInvalid = true;
		}*/
	}

	/**
	 * 父类重写
	 * 都是全局的
	 */
	calculateBounds() {
		if (this._lastBoundsID == this._boundsID) return
		this._lastBoundsID = this._boundsID
		this._bounds.clear();
		//算自己的
		this._calculateBounds();
		for (let i = 0; i < this.children.length; i++) {
			const child = this.children[i];
			if (!child.visible || !child.renderable) {
				continue;
			}
			child.calculateBounds();
			if (child.$mask) {
				child.$mask.calculateBounds();
				//取交集矩形
				if (child._bounds.x < child.$mask._bounds.x) {
					child._bounds.x = child.$mask._bounds.x;
				}
				if (child._bounds.y < child.$mask._bounds.y) {
					child._bounds.y = child.$mask._bounds.y;
				}
				if (child._bounds.width > child.$mask._bounds.width) {
					child._bounds.width = child.$mask._bounds.width;
				}
				if (child._bounds.height > child.$mask._bounds.height) {
					child._bounds.height = child.$mask._bounds.height;
				}
				Rectangle.createFromRects(this._bounds, child._bounds);
			} else {
				Rectangle.createFromRects(this._bounds, child._bounds);
			}
		}


	}

	/**
	 * 加"_"的方法基本是为了自己特殊处理
	 */
	protected _calculateBounds() {
		//子类自己重写
		//let wp = this.worldMatrix.transformPoint(this.x, this.y);

		let widthSetted = !!this._width && this._width !== 0;
		let heightSetted = !!this._height && this._height !== 0;
		/*if (widthSetted) {
			this._bounds.x = this.x;
			this._bounds.width = this._width;
		}
		if (heightSetted) {
			this._bounds.y = this.y;
			this._bounds.height = this._height;
		}*/

		if(widthSetted || heightSetted){
			const rect = this._localBoundsSelf;
			var matrix = this.transform.worldMatrix;
			matrix.transformPoint(rect.x, rect.y, DisplayObject._p1);
			matrix.transformPoint(rect.x + rect.width, rect.y, DisplayObject._p2);
			matrix.transformPoint(rect.x + rect.width, rect.y + rect.height, DisplayObject._p3);
			matrix.transformPoint(rect.x, rect.y + rect.height, DisplayObject._p4);
			Rectangle.createFromPoints(this._bounds, DisplayObject._p1, DisplayObject._p2, DisplayObject._p3, DisplayObject._p4);
		}
	}

	/**
	 * 检测点是否在任何child上
	 * 重写父类方法
	 */
	hitTestPoint(globalPoint: Point, isMouseEvent: boolean = false) {
		//如果不可见
		if (!this.visible) return null
		//如果禁止子级的鼠标事件
		if (isMouseEvent && !this.mouseChildren){
			return this.hitTestSelf(globalPoint);
		}
		var children = this.children;
		var length = children.length;
		let child, hitDisplayObject;
		//后序遍历，后添加的在上层
		for (var i = length - 1; i >= 0; i--) {
			child = children[i];
			//当作遮罩的不作为检测，跳过
			if (child.isUsedToMask) continue;
			//有遮罩，但是不在遮罩内，跳过
			if (child.mask && !child.mask.hitTestPoint(globalPoint, isMouseEvent)) continue;
			//检测
			hitDisplayObject = child.hitTestPoint(globalPoint, isMouseEvent);
			//存在直接返回
			if (hitDisplayObject) return hitDisplayObject;
		}

		return this.displayObjectHitTestPoint(globalPoint, isMouseEvent);
	}

	hitTestSelf(globalPoint) {
		if (this.mouseEnabled) {
			return this.hitTestSelfBounds(globalPoint);
		}
		return null;
	}

	hitTestSelfBounds(globalPoint) {
		if (this._width && this._height) {
			//let {x: tx, y: ty} = this.getBounds();
			let {tx, ty} = this.worldMatrix;
			const {x, y} = globalPoint;
			if (x > tx &&
				x < tx + this.width &&
				y > ty &&
				y < ty + this.height
			) return this;
		}
		return null
	}

	/**
	 * webgl渲染
	 * @param {WebglRenderer} renderer - The renderer
	 */
	renderWebGL(renderer: WebglRenderer) {
		//不可见，全局透明度为0，或者 不渲染，直接return
		if (!this.visible || this.worldAlpha <= 0 || !this.renderable) {
			return;
		}
		//是否有遮罩。到时如果有滤镜，
		if (this.mask) {
			this.renderAdvancedWebGL(renderer);
		} else {
			//自身先渲染
			this._renderWebGL(renderer);
			//遍历children
			for (let i = 0, j = this.children.length; i < j; ++i) {
				this.children[i].renderWebGL(renderer);
			}
		}
	}

	/**
	 * 高级渲染方法
	 *
	 * @private
	 * @param {WebGLRenderer} renderer - The renderer
	 */
	private renderAdvancedWebGL(renderer: WebglRenderer) {
		//之前的批处理刷掉先
		renderer.batchManager.flush();

		//有滤镜再说

		const mask = this.mask;

		if (mask) {
			//先画遮罩
			renderer.maskManager.pushMask(this, this.mask);
		}

		//渲染自身
		this._renderWebGL(renderer);

		//遍历children
		for (let i = 0, j = this.children.length; i < j; i++) {
			this.children[i].renderWebGL(renderer);
		}

		//刷掉批处理
		renderer.batchManager.flush();

		if (mask) {
			//移除遮罩，支持多重遮罩
			renderer.maskManager.popMask(this, this.mask);
		}
	}

	/**
	 * 自身渲染方式
	 * @private
	 * @param {WebglRenderer} renderer - The renderer
	 */
	protected _renderWebGL(renderer: WebglRenderer) {
		//自身绘制方法
	}

	/**
	 * canvas渲染方式
	 * @param {CanvasRenderer} renderer - The renderer
	 */
	renderCanvas(renderer: CanvasRenderer) {
		if (!this.visible || this.worldAlpha <= 0 || !this.renderable) {
			return;
		}

		if (this.mask) {
			renderer.maskManager.pushMask(this.mask);
		}

		this._renderCanvas(renderer);
		for (let i = 0, j = this.children.length; i < j; ++i) {
			this.children[i].renderCanvas(renderer);
		}

		if (this.mask) {
			renderer.maskManager.popMask(renderer);
		}
	}

	/**
	 * 自身渲染方法
	 *
	 * @private
	 * @param {CanvasRenderer} renderer - The renderer
	 */
	protected _renderCanvas(renderer: CanvasRenderer) {
		//自身绘制方法
	}

	/**
	 * 更新方法
	 */
	update(deltaTime: number) {
		/*if (!this.visible) return;*/
		//更新自己的
		super.update(deltaTime)
		//更新儿子们的
		let len = this.children.length;
		for (let i = len - 1; i >= 0; i--) {
			const child = this.children[i];
			/*if (child.visible)*/
			child.update(deltaTime);
		}
	}


	/**
	 * 调用此方法对自己及其child触发一次指定事件
	 * @method _onDispatchBubbledEvent
	 * @public
	 * @param {string} type
	 * @since 1.0.0
	 */
	public _onDispatchBubbledEvent(type: string): void {
		let s = this;
		let len = s.children.length;
		if (type == Event.REMOVED_FROM_STAGE && !s.stage) return;
		super._onDispatchBubbledEvent(type);
		for (let i = 0; i < len; i++) {
			s.children[i]._onDispatchBubbledEvent(type);
		}

	}

	/**
	 *
	 */
	destroy() {
		let s = this;
		//让子级也destroy
		for (let i = s.children.length - 1; i >= 0; i--) {
			s.children[i].destroy();
		}
		super.destroy();
		this.mouseChildren = false;
	}

	/**
	 * 一般用于获取宽高并设置
	 * 包括子级的,容器的尽量少用，子类可重写
	 * @member {number}
	 */
	get width(): number {
		return this._width || this.scale.x * this.getLocalBounds().width;
	}

	set width(value: number) {
		// const width = this.getLocalBounds().width;

		// if (width !== 0) {
		// 	this.scale.x = value / width;
		// } else {
		// 	this.scale.x = 1;
		// }
		if (this._width !== value) {
			//子类有用，有_width,才需设置scaleX
			this._width = value;
			this._localBoundsSelf.width = value;
			//if (this.stage) this.stage.layoutInvalid = true;
			this.dispatchEvent(Event.RESIZE);
		}
	}

	/**
	 * 高度同width
	 * @member {number}
	 */
	get height(): number {
		return this._height || this.scale.y * this.getLocalBounds().height;
	}

	set height(value: number) {
		// const height = this.getLocalBounds().height;

		// if (height !== 0) {
		// 	this.scale.y = value / height;
		// } else {
		// 	this.scale.y = 1;
		// }
		if (this._height !== value) {
			this._height = value;
			this._localBoundsSelf.height = value;
			//if (this.stage) this.stage.layoutInvalid = true;
			this.dispatchEvent(Event.RESIZE);
		}
	}

	clone(withEvents = false, withScripts = false) {
		let target = this.constructor.apply(Object.create(this.constructor.prototype));

		const originConfig = this['__originConfig'];
		const {name, properties, events, scripts} = originConfig;
		target.name = name;
		target['__originConfig'] = originConfig;
		injectProperties(target, properties);

		if (withScripts) {
			if (scripts && scripts.length > 0) {
				for (let scriptConfig of scripts) {
					instantiateScript(target, scriptConfig);
				}
			}
		}
		if (withEvents) {
			if (events) {
				target.eventsProxy.start(events);
			}
		}

		for (let child of this.children) {
			let childCopy = child.clone(withEvents, withScripts);
			target.addChild(childCopy);
		}

		return target;
	}

	get $store() {
		let p = this;
		do {
			if (p['$isViewRoot']) {
				break;
			}
			p = p.parent;
		}
		while (p.parent);

		if (p) {
			return p['$_store'];
		}
	}

	//全局遍历
	/**
	 * @method _getElementsByName
	 * @param {RegExp} rex
	 * @param {Container} root
	 * @param {boolean} isOnlyOne
	 * @param {boolean} isRecursive
	 * @param {Array<DisplayObject>} resultList
	 * @private
	 * @static
	 */
	private static _getElementsByName(rex: RegExp, root: Container, isOnlyOne: boolean, isRecursive: boolean, resultList: Array<DisplayObject>): void {
		let len = root.children.length;
		if (len > 0) {
			let name: string;
			let child: any;
			for (let i = 0; i < len; i++) {
				child = root.children[i];
				name = child.name;
				if (name && name != "") {
					if (rex.test(name)) {
						resultList[resultList.length] = child;
						if (isOnlyOne) {
							return;
						}
					}
				}
				if (isRecursive) {
					if (child["children"] != null) {
						Container._getElementsByName(rex, child, isOnlyOne, isRecursive, resultList);
					}
				}
			}
		}
	}
}

Container.prototype.containerUpdateTransform = Container.prototype.updateTransform;
