import _ from 'lodash';
import properties from './properties';
import {assetScheme} from "./index";

// 属性的计算方法
const propsComputeRules = {
	x: 'add',
	y: 'add',
	left: 'add',
	top: 'add',
	rotation: 'add',
	scaleX: 'multi',
	scaleY: 'multi',
	alpha: 'multi',
	visible: 'visible'
};

const isClient = typeof window !== 'undefined';

function changeCamle(s) {
	return s ? s.replace(/([A-Z])/g, '-$1').toLowerCase() : '';
}

function invalidAttr(key, value) {
	return !key || typeof value === 'undefined' || value === '';
}

// 属性简称 对照表
const attrShortMapper = {
	x: 'left',
	y: 'top',
	align: 'text-align',
	size: 'font-size',
	font: 'font-family',
	alpha: 'opacity',
	strokeColor: 'border-color',
	strokeWidth: 'border-width',
	fillColor: 'background-color',
	source: 'background-image'
};

// 编辑时想拖拽组件需要生成的css属性
// 只需要位置，不需要来源透明度等等
const operatProps = ['x', 'y', 'left', 'top', 'right', 'bottom', 'width', 'height', 'rotation', 'scaleX', 'scaleY'];
const ignoreProps = ['left', 'right', 'top', 'bottom'];

// 属性单位 对照表, 如果是数值的时候需要添加单位
const attrUnitMapper = {
	x: 'px',
	y: 'px',
	left: 'px',
	top: 'px',
	right: 'px',
	bottom: 'px',
	width: 'px',
	height: 'px',
	fontSize: 'px',
	size: 'px',
	letterSpacing: 'px',
	borderRadius: 'px',
	borderWidth: 'px',
	strokeWidth: 'px',
	minHeight: 'px'
};

/**
 * 根据uuid，从list中找到所有父组件
 * @param {*} uuid
 * @param {*} list
 */
function getParentCmps(uuid, list) {
	let _self = list.find(c => c.uuid === uuid);
	if (!_self) {
		return [];
	}

	let parentLoop = (uuid, list) => {
		let _item = list.find(c => c.uuid === uuid);
		return _item.parent ? [_item].concat(parentLoop(_item.parent, list)) : [_item];
	}

	return _self.parent ? parentLoop(_self.parent, list) : [];
}

export {getParentCmps, completeSelfProps};

export const getCmpByUUID = function (uuid, views) {
	if (!uuid) {
		return null;
	}
	let _node = views.find(v => v.uuid === uuid);
	if (_node) {
		return _node
	} else {
		for (let index = 0; index < views.length; index++) {
			if (views[index].children && views[index].children.length) {
				return getCmpByUUID(uuid, views[index].children)
			}
		}
	}
}

/**
 * 扁平化视图数组
 * @param {*} views
 */
export const flattenViews = function (views) {
	return views.reduce((flat, toFlat) => {
		if (toFlat.children) {
			let _children = toFlat.children.map(c => {
				c.parent = toFlat.uuid;
				return c;
			});
			return flat.concat([toFlat].concat(flattenViews(_children)));
		} else {
			return flat.concat([toFlat]);
		}
	}, []);
}

/**
 * 根据propsComputeRules的计算规则，继承父级的属性
 * @param {*} props
 * @param {*} parent
 */
function inheritProps(props, parent, onlyOpera) {
	_.forIn(parent, (value, key) => {
		let _pValue = parent[key];
		if (_pValue === null || _pValue === undefined) {
			return;
		}
		let rule = propsComputeRules[key];
		// 加法
		// if (rule === 'add') {
		// 	if (props[key] === null || props[key] === undefined) {
		// 		props[key] = 0;
		// 	}
		// 	props[key] += _pValue;
		// }
		// 乘法
		// if (rule === 'multi') {
		// 	if (props[key] === null || props[key] === undefined) {
		// 		props[key] = 1;
		// 	}
		// 	props[key] *= _pValue;
		// }
		// 可见，特殊处理
		// 节点和父节点的visible只要有一个是false，值就是false，否则为true
		// if (rule === 'visible') {
		// 	props[key] = !props[key] || !_pValue ? false : true;
		// }
	});
}

/**
 * 根据组件type 依赖properties.js构建组件完整的属性对象
 * @param {*} component
 */
function completeSelfProps(component) {
	// 根据组件类型，获取默认属性
	let defaultProps = getCmpProps(component.type);

	// 把这些属性格式转换成key: value
	defaultProps = _.mapValues(defaultProps.props, o => (o.default));

	return _.merge(defaultProps, component.properties);
}

export const styles = {
	getStyles(value, key) {
		const attr = attrShortMapper[key] || changeCamle(key);
		let unit = attrUnitMapper[key] || '';

		if (invalidAttr(key, value)) return '';

		// if ((key === 'width' || key === 'height') && (value + '').indexOf('%') > -1) {
		// 	unit = '';
		// }

		switch (attr) {
			case 'rotation':
				return ['transform', [`rotate(${value}deg)`]]; //`transform: rotate(${value}deg);`;
			case 'background-image':
				return ['background-image', [`url(${value})`]]; //`background-image: url(${value});background-position:center;background-size:contain;`;
			case 'scale-x':
				return ['transform', [`scaleX(${value})`]]; //`transform: scaleX(${value});`;
			case 'scale-y':
				return ['transform', [`scaleY(${value})`]]; //`transform: scaleY(${value});`;
			case 'visible':
				return ['display', [value ? 'block' : 'none']]; // `display: ${value ? 'block' : 'none'};`;
			default:
				return [attr, [`${value}${Number.isNaN(value) ? '' : unit}`]]; //`${attr}:${value}${Number.isNaN(value) ? '' : unit};`;
		}
	},
	/**
	 * 根据Object类型的属性对象，生成dom中style属性可用的数据格式
	 * @param {*} obj
	 */
	getStylesFromObj(obj, project) {
		let resultObj = {};

		_.forIn(obj, (value, key) => {
			if (key === 'source' && value.indexOf(assetScheme) === 0) {
				let uuid = value.split('//')[1];
				let asset = project.data.assets.find(a => a.uuid === uuid);
				value = asset ? asset.url : '';
			}
			let _style = styles.getStyles(value, key);
			if (!_style) {
				return;
			}
			if (resultObj[_style[0]]) {
				resultObj[_style[0]] = resultObj[_style[0]].concat(_style[1]);
			} else {
				resultObj[_style[0]] = _style[1];
			}
		});
		return resultObj;
	},
	/**
	 * 根据组件数据，生成完整的style
	 * @param {*} component
	 */
	getComponentStyle(component, project, componentList, onlyOpera) {
		if (!component || !component.uuid) {
			return '';
		}
		// debugger;
		let result = '';
		let cmpSelfProps = completeSelfProps(component);

		// 根据uuid获取节点的所有父节点
		let propsMatrix = getParentCmps(component.uuid, componentList);
		propsMatrix = propsMatrix.map(completeSelfProps);

		if (propsMatrix.length) {
			propsMatrix.forEach(prop => {
				// 继承每个父节点的属性
				inheritProps(cmpSelfProps, prop, onlyOpera);
			});
		}

		if (component.type === 'image') {
			cmpSelfProps.width = cmpSelfProps.width || cmpSelfProps.imageWidth;
			cmpSelfProps.height = cmpSelfProps.height || cmpSelfProps.imageHeight;
		}

		// 获取拖拽组件的样式，只生成operatProps中的属性
		if (onlyOpera) {
			_.forIn(cmpSelfProps, (val, key) => {
				if (operatProps.indexOf(key) === -1) {
					delete cmpSelfProps[key];
				}
			});

			// 如果节点的width/height未定义，则设置拖拽组件的width/height为0
			cmpSelfProps.width = cmpSelfProps.width || 0;
			cmpSelfProps.height = cmpSelfProps.height || 0;

			if (cmpSelfProps.scaleX && cmpSelfProps.width) {
				cmpSelfProps.width *= cmpSelfProps.scaleX;
				delete cmpSelfProps.scaleX;
			}

			if (cmpSelfProps.scaleY && cmpSelfProps.height) {
				cmpSelfProps.height *= cmpSelfProps.scaleY;
				delete cmpSelfProps.scaleY;
			}
		}

		// 把不需要参与样式计算的属性干掉
		_.forEach(ignoreProps, prop => {
			delete cmpSelfProps[prop];
		});

		if (component.type === 'label' || component.type === 'textinput') {
			// 如果是label类型，把fillColor转换为color
			cmpSelfProps.color = cmpSelfProps.fillColor;
			delete cmpSelfProps.fillColor;
			delete cmpSelfProps.stroke;
			delete cmpSelfProps.strokeColor;
			if (cmpSelfProps.italic) {  //斜体
				delete cmpSelfProps.italic;
				cmpSelfProps.fontStyle = 'italic';
			}
			if (!onlyOpera) {
				if (cmpSelfProps.lineType === 'single') {
					// word-break: keep-all;white-space: nowrap;
					cmpSelfProps.wordBreak = 'keep-all';
					cmpSelfProps.whiteSpace = 'nowrap';
				} else {
					cmpSelfProps.wordBreak = 'break-all';
					cmpSelfProps.whiteSpace = 'normal';
				}
			}
		}

		// console.log('cmpSelfProps after inherit ', cmpSelfProps);
		cmpSelfProps = styles.getStylesFromObj(cmpSelfProps, project);
		_.forIn(cmpSelfProps, (value, key) => {
			result += `${key}: ${value.join(' ')};`
			if (key === 'border-color') {
				result += 'border-style: solid;'
			}
		});
		result += `background-position: center;background-size: 100% 100%;transform-origin: left top;`

		if (component.type === 'circle' && !onlyOpera) {
			// 如果是circle类型，加圆角
			result += 'border-radius: 50%;'
		}

		// if (component.type === 'label' && !onlyOpera) {
		// 如果是label类型，禁止自动换行
		// console.log('cmpSelfProps', cmpSelfProps);

		// result += 'word-break: keep-all;white-space: nowrap;'
		// }

		// console.log('getComponentStyle',component.name, result);
		return result;
	}
}

const cmpMetaCache = {};

export function getCmpProps(type) {
	if (!type) {
		return {}
	} else {
		let cmpProps = cmpMetaCache[type];
		if (cmpProps) {
			cmpProps = _.cloneDeep(cmpProps);
		} else {
			let typeMeta = _.cloneDeep(properties[type]);
			let inherits = [typeMeta.props];
			let tempMeta = typeMeta;
			while (tempMeta.base) {
				tempMeta = cmpMetaCache.hasOwnProperty(tempMeta.base) ? cmpMetaCache[tempMeta.base] : properties[tempMeta.base];
				inherits.unshift(_.cloneDeep(tempMeta.props));
			}
			typeMeta.props = Object.assign({}, ...inherits);
			cmpProps = cmpMetaCache[type] = typeMeta;
		}

		return cmpProps;
	}
}
