/**
 * Created by rockyl on 2019-09-19.
 *
 * 项目相关api
 */

import { fetchApi } from "./common"
import { getCookie } from './utils'

/**
 * 获取项目列表
 * @param keyword
 * @param currentPage
 * @param pageSize
 * @param onlyMine
 * @return {Promise<*|any>}
 */
export async function fetchAll(keyword, currentPage, pageSize, onlyMine) {
	return await fetchApi('/api/project/query', {
		params: {name: keyword, currentPage, pageSize, isAll: onlyMine ? 0 : 1 },
		errMessage: 'Failed to fetch projects',
	})
}

/**
 * 创建一个项目
 * @param project
 * @return {Promise<*|any>}
 */
export async function createOne(project) {
	return await fetchApi('/api/project/create', {
		params: project,
		method: 'post',
		errMessage: 'Failed to create project',
	})
}

/**
 * 复制一个项目
 * @param project
 * @return {Promise<*|any>}
 */
export async function duplicateOne(project) {
	return await fetchApi('/api/project/copy', {
		params: project,
		method: 'post',
		errMessage: 'Failed to duplicate project',
	})
}

/**
 * 删除一个项目
 * @param id
 * @return {Promise<*|any>}
 */
export async function deleteOne(id) {
	return await fetchApi('/api/project/delete', {
		params: { id },
		method: 'post',
		errMessage: 'Failed to delete project',
	})
}

/**
 * 获取一个项目
 * @param id
 * @param withData 是否携带data
 * @return {Promise<*|any>}
 */
export async function fetchOne(id, withData = true) {
	return await fetchApi('/api/project/query/data', {
		params: { id, withData },
		method: 'get',
		errMessage: 'Failed to fetch project',
	})
}

/**
 * 从历史版本打开
 * @param dataUrl
 * @return {Promise<*|any>}
 */
export async function fetchOneFromDataUrl(dataUrl) {
	return await fetchApi(dataUrl, {
		auth: false,
		judgeSuccess: false,
		credentials: 'same-origin',
		errMessage: 'Failed to fetch project',
	})
}

/**
 * 获取历史保存列表
 * @param id
 * @param currentPage
 * @param pageSize
 * @return {Promise<*|any>}
 */
export async function fetchHistory(id, currentPage, pageSize) {
	return await fetchApi('/api/project/history', {
		params: { id, currentPage, pageSize },
		method: 'get',
		errMessage: 'Failed to history',
	})
}

/**
 * 保存项目
 * @param project
 * @param remark
 * @return {Promise<*|any>}
 */
export async function saveOne(project, remark) {
	project.remark = remark;
	return await fetchApi('/api/project/update', {
		params: project,
		method: 'post',
		errMessage: 'Failed to save project',
	})
}

/**
 * 打包项目
 * @param id
 * @param debug
 * @param packedAssets
 * @return {Promise<*|any>}
 */
export async function pack(id, debug, packedAssets) {
	return await fetchApi('/api/project/pack', {
		params: { id, debug, packedAssets },
		method: 'post',
		errMessage: 'Failed to pack project',
	})
}

/**
 * 修改开发人员名单
 * @param id
 * @param operators
 * @return {Promise<*|any|>}
 */
export async function updateOperators(id, operators) {
	return await fetchApi('/api/project/updateOperators', {
		params: { id, operators },
		method: 'post',
		errMessage: 'Failed to update operator',
	})
}
