/**
 * Created by rockyl on 2019-01-22.
 */

const ts = require('typescript');

let classDeclareMap;
let _componentsPath;
let _projectPath;
let _assetsPath;
let _filter;

const scillaCorePrefix = 'node_modules/scilla/src';
const componentBaseClassName = '"scilla/core/Component".Component';

const showLog = false;
const showVerboseLog = false;

exports.generateDeclareMap = function(tsconfig, file, componentsPath, projectPath, assetsPath, filter) {
	_componentsPath = componentsPath;
	_projectPath = projectPath;
	_assetsPath = assetsPath;
	_filter = filter;

	const config = ts.parseJsonConfigFileContent(tsconfig, ts.sys, projectPath);

	const files = file ? [file] : config.fileNames;
	const program = ts.createProgram(files, config.options);
	const sourceFiles = program.getSourceFiles();

	classDeclareMap = {};

	if(showLog) console.time('getTypeChecker');
	const checker = program.getTypeChecker();
	if(showLog) console.timeEnd('getTypeChecker');

	if(showLog) console.time('delint');
	for(let sourceFile of sourceFiles){
		if (sourceFile.fileName.indexOf('.d.ts') < 0) {
			if(!_filter || _filter(sourceFile)){
				if(showLog && showVerboseLog) console.time(sourceFile.fileName);
				delint(checker, sourceFile);
				if(showLog && showVerboseLog) console.timeEnd(sourceFile.fileName);
			}
		}
	}
	if(showLog) console.timeEnd('delint');

	/*for(let name in classDeclareMap){
		transFullyDeclare(name);
	}*/

	return classDeclareMap;
};

function delint(checker, sourceFile) {
	delintNode(sourceFile);

	function delintNode(node) {
		switch (node.kind) {
			case ts.SyntaxKind.EnumDeclaration:
				const enumType = checker.getTypeAtLocation(node);
				const {fullClassName: enumName} = getFullyQualifiedNameOfType(checker.getTypeAtLocation(node), checker);

				let members;
				if (enumName in classDeclareMap) {
					members = classDeclareMap[enumName];
				} else {
					classDeclareMap[enumName] = members = {
						__type__: 'enum',
						map: {},
					};
				}

				for (let type of enumType.types) {
					let name = type.symbol.escapedName;
					members.map[name] = {
						value: type.value,
					};
					let s = type.getSymbol();
					getJsDoc(members.map[name], s.valueDeclaration)
				}
				break;
			case ts.SyntaxKind.ClassDeclaration:
				const theType = checker.getTypeAtLocation(node);

				let mf = ts.getCombinedModifierFlags(node);
				if (!(mf & ts.ModifierFlags.Abstract)) {
					const {fullClassName, path, className} = getFullyQualifiedNameOfType(checker.getTypeAtLocation(node), checker);

					let members;
					if (fullClassName in classDeclareMap) {
						members = classDeclareMap[fullClassName];
					} else {
						classDeclareMap[fullClassName] = members = {properties: {}, methods: {}};
					}

					getJsDoc(members, node);

					const analyseResult = analyseBases(theType, checker);
					if(analyseResult.bases.length > 0){
						members["bases"] = analyseResult.bases;
					}
					if(analyseResult.isComponent){
						members['isComponent'] = analyseResult.isComponent;
					}
					members['path'] = path;
					members['className'] = className;

					const props = theType.getSymbol().members;

					props.forEach(function (value) {
						let name = value.name;
						if (name.indexOf("$") === 0 || name.indexOf("_") === 0)
							return;

						const {flags, declarations, valueDeclaration, valueDeclaration: {modifiers}} = value;

						let isAccessibility = true;
						if (modifiers) {
							for (const modifier of modifiers) {
								if (modifier.kind === ts.SyntaxKind.PrivateKeyword || modifier.kind === ts.SyntaxKind.ProtectedKeyword) {
									isAccessibility = false;
									break;
								}
							}
						}

						if (isAccessibility) {
							//console.log(name, flags, valueDeclaration.kind);
							if ((flags & ts.SymbolFlags.Property) || ((flags & ts.SymbolFlags.Accessor) && declarations.length === 2)) {
								addProp(members.properties, name, valueDeclaration, checker);
							} else if (flags & ts.SymbolFlags.Method) {
								const method = members.methods[name] = {};
								getJsDoc(method, valueDeclaration);
								valueDeclaration.locals.forEach(local => {
									const {valueDeclaration, name} = local;
									addProp(method, name, valueDeclaration, checker);
								})
							}
						}
					});
				}
		}
		ts.forEachChild(node, delintNode);
	}
}

function addProp(dataContainer, name, valueDeclaration, checker) {
	if (!valueDeclaration) {
		return;
	}
	const {initializer, type: typeNode} = valueDeclaration;

	let typeName;

	if (typeNode && typeNode.typeName && typeNode.typeName.escapedText && typeNode.typeName.escapedText.charCodeAt(0) >= 97) {
		typeName = typeNode.typeName.escapedText;
	} else {
		let typeDef = checker.getTypeAtLocation(valueDeclaration);
		const {fullClassName} = getFullyQualifiedNameOfType(typeDef, checker);
		typeName = fullClassName;
	}

	let defaultValue;
	if (initializer) {
		if (typeName === 'boolean') {
			defaultValue = initializer.end - initializer.pos === 5;
		} else if (initializer.arguments) {
			defaultValue = [];
			for (let argument of initializer.arguments) {
				let typeDef = checker.getTypeAtLocation(argument);

				let value = argument.text || (argument.name && argument.name.escapedText);
				defaultValue.push(value);
			}
		} else {
			defaultValue = initializer && (initializer.text || (initializer.name && initializer.name.escapedText));
		}
	}

	const prop = {type: typeName};
	getJsDoc(prop, valueDeclaration);
	if (defaultValue !== undefined) {
		prop.defaultValue = defaultValue;
	}
	dataContainer[name] = prop;
}

function getJsDoc(dataContainer, valueDeclaration) {
	const jsDoc = valueDeclaration.jsDoc;
	if (jsDoc && jsDoc.length > 0) {
		dataContainer.__comment__ = jsDoc[jsDoc.length - 1].comment;
	}
}

function getFullyQualifiedNameOfType(type, checker) {
	let symbol = type.getSymbol(), fullClassName, path, className;
	if (symbol) {
		fullClassName = checker.getFullyQualifiedName(symbol);
		if (fullClassName.indexOf('\"') >= 0) {
			path = fullClassName.substring(1, fullClassName.lastIndexOf('.') - 1);
			className = fullClassName.substring(fullClassName.lastIndexOf('.') + 1);
			if (path.indexOf(scillaCorePrefix) >= 0) {
				path = path.substr(path.indexOf(scillaCorePrefix)).replace(scillaCorePrefix, 'scilla');
			} else if (path.indexOf(_assetsPath) >= 0) {
				path = path.replace(_assetsPath, '.')
			} else {
				path = path.replace(_componentsPath, 'components')
			}
			fullClassName = `"${path}".${className}`;
		}
	} else {
		fullClassName = checker.typeToString(type);
	}

	return {
		fullClassName,
		path,
		className,
	}
}

function analyseBases(type, checker) {
	const bases = [];

	let types, isComponent = false;
	while (true) {
		types = checker.getBaseTypes(type);
		if (types.length === 0) {
			break;
		}

		type = types[0];
		const {fullClassName} = getFullyQualifiedNameOfType(type, checker);
		bases.push(fullClassName);

		if (fullClassName === componentBaseClassName) {
			isComponent = true;
		}
	}

	return {
		bases,
		isComponent,
	};
}
