/**
 * Created by renjianfeng on 2020-06-11.
 *
 * 导出spark的视图
 */

import {getTree} from "./psd-tree";
import {walkNode} from "./utils";
import path from 'path'
import Color from 'color'
import generateUUID from 'uuid/v4'
import hash from 'object-hash';
import zlib from 'zlib';

const relativePosPrefixMap = {
	l: {field: 'left',},
	t: {field: 'top',},
	r: {field: 'right',},
	b: {field: 'bottom',},
	h: {field: 'horizonCenter',},
	v: {field: 'verticalCenter',},
	wp: {field: 'width', },
	hp: {field: 'height', },
	lp: {field: 'left', },
	tp: {field: 'top', },
	rp: {field: 'right', },
	bp: {field: 'bottom', },
};

const offsetAll = 176;

export async function execute(psdFile, options) {
	const tree = await getTree(psdFile);
	const {mode = 'none', singleView = true} = options;

	let offset = {x: 0, y: 0};
	let cutSize = {x: 0, y: 0};

	if (mode !== 'none') {
		cutSize.y = offsetAll;
	}
	switch (mode) {
		case 'top':
			offset.y = offsetAll;
			break;
		case 'center':
			offset.y = offsetAll / 2;
			break;
	}

	const isCenter = mode === 'center';

	let viewRoot = {
		name: path.basename(psdFile.name, '.psd'),
		componentName : 'Div',
		uuid: generateUUID(),
	};

	const assets = [];
	const imageHashMap = {};

	let {width: stageWidthOrigin, height: stageHeightOrigin} = tree;
	const stageWidth = stageWidthOrigin - cutSize.x || 0;
	const stageHeight = stageHeightOrigin - cutSize.y || 0;

	await walkNode(tree, async function (node, parent) {
		let {name} = node;
		const {x, y, width, height, opacity, display, origin: {layer, layer: {typeTool, solidColor}}} = node;
		//console.log('walk node:', name);
		let properties = {
			style:{
				width, height, opacity,display ,
			},
			attrs:{

			},
			className:""
		};
		const isSecondLayer = singleView && !parent.origin.parent || !singleView && parent.origin.parent && !parent.origin.parent.parent;
		const shouldVerticalCenter = isSecondLayer && isCenter;

		const {width: parentWidth, height: parentHeight} = parent;

		if (name.includes('|') || shouldVerticalCenter) {
			try {
				let arr = name.split('|');
				name = arr[0];
				let paramsStr = arr[1];
				let relativePos;
				if(paramsStr){
					let params = paramsStr.split(';');
					relativePos = params[0];
				}else if(shouldVerticalCenter){
					relativePos = 'v';
				}

				if (relativePos) {
					let items = relativePos.split(',');
					for (let item of items) {
						let result = item.match(/[a-zA-Z]+/);
						if(!result){
							continue;
						}
						let prefix = result[0];
						let mapItem = relativePosPrefixMap[prefix];
						if (mapItem) {
							let {field,} = mapItem;
							let value = item.substr(prefix.length);
							let hasValue = value.length > 0;
							let fieldChar = prefix[0];
							if(!hasValue){
								switch(fieldChar){
									case 'l':
										value = x - offset.x;
										break;
									case 't':
										value = y - offset.y;
										break;
									case 'r':
										value = stageWidth - (x - offset.x) - width;
										break;
									case 'b':
										value = stageHeight - (y - offset.y) - height;
										break;
									case 'h':
										value = x + width / 2 - stageWidthOrigin / 2;
										break;
									case 'v':
										value = y + height / 2 - stageHeightOrigin / 2;
										break;
								}
							}
							let isPercent = prefix.endsWith('p');
							if (isPercent) {
								if(!hasValue){
									switch(fieldChar){
										case 'l':
										case 'r':
											value /= stageWidth;
											break;
										case 't':
										case 'b':
											value /= stageHeight;
											break;
									}
									value = Math.floor(value * 100);
								}
								value += '%';
							} else {
								value = parseFloat(value);
								if (isNaN(value)) {
									value = 0;
								}
							}
							console.log(properties)
							properties.style[field] = value;
						}
					}
				}
			}catch (e) {
				console.log(e);
			}
		}

		let viewNode = {
			name,
			properties,
			uuid: generateUUID(),
		};
		let dealLater = true;
		if (x !== 0) {
			console.log(properties)
			if(!properties.style.left){
				properties.style.left = x - (isSecondLayer ? offset.x : 0);
			}
			
		}
		if (y !== 0) {
			if(!properties.style.top){
				properties.style.top = y - (isSecondLayer ? offset.y : 0);
			}
		}
		properties.style.position="absolute"
		properties.style.transformOrigin="0px 0px 0px";
		viewNode.rect={
			x: properties.style.left?properties.style.left:0,
			y: properties.style.top?properties.style.top:0,
			width: properties.style.width,
			height: properties.style.height
		}

		if (typeTool) {
			let fontInfo = typeTool();
			properties.attrs.text = fontInfo.textValue;
			const sizes = fontInfo.sizes();
			const colors = fontInfo.colors();
			properties.style.fontSize = sizes ? sizes[0] || 20 : 20;
			let [r, g, b, a] = colors[0];
			console.log("color",[r, g, b, a])
			properties.style.color = `rgba(${r}, ${g}, ${b}, ${a / 255})`;
			viewNode.componentName = 'Label';
			dealLater = false;
		} else if (solidColor && layer.vectorMask) {
			let paths = layer.vectorMask().paths;
			if (paths[2].numPoints === 4) {
				let isRect = true;
				for (let i = 3; i < paths.length; i++) {
					if (paths[i].recordType !== 2) {
						isRect = false;
						break;
					}
				}
				if (isRect) {
					viewNode.componentName = 'Div';
					const {r, g, b} = solidColor();
					properties.style.backgroundColor = `rgba(${r}, ${g}, ${b}, 1)`;
					dealLater = false;
				}
			}
		}

		if (dealLater) {
			if (node.hasOwnProperty('children')) {
				viewNode.componentName = 'Div';
			} else {
				viewNode.componentName = 'Image';

				let uuid = generateUUID();
				const ext = '.png';

				let dataUrl;
				try {
					let img = node.origin.toPng();
					dataUrl = img.src;
				} catch (e) {

				}

				if (dataUrl) {
					let base64Data = dataUrl.replace(/^data:image\/\w+;base64,/, "");
					let buffer = new Buffer(base64Data, 'base64');
					const fileNameHash = hash(buffer);
					if (imageHashMap.hasOwnProperty(fileNameHash)) {
						uuid = imageHashMap[fileNameHash];
					} else {
						imageHashMap[fileNameHash] = uuid;
						assets.push({
							name,
							ext,
							uuid,
							base64Data,
							hash: fileNameHash,
						});
					}

					properties.attrs.source = 'asset://' + uuid;
				}
			}
		}

		let viewParent = parent.view || viewRoot;
		if (!viewParent.hasOwnProperty('children')) {
			viewParent.children = [];
		}
		viewParent.children.push(viewNode);

		node.view = viewNode;
	});
    console.log(psdFile)
	let data = {
		pluginVersion:"0.0.1",
		reference:"psd",
		fileName:psdFile.name,
		assets,
		view: viewRoot,
	};

	let dataString = JSON.stringify(data);

	let buf = new Buffer(dataString);
	return await new Promise((resolve, reject) => {
		zlib.gzip(buf, function (err, res) {
			if (err) {
				reject(err);
			} else {
				console.log(res.length);
				resolve(res);
			}
		});
	})
}
