(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('path'), require('color'), require('uuid/v4'), require('object-hash'), require('zlib')) :
	typeof define === 'function' && define.amd ? define(['exports', 'path', 'color', 'uuid/v4', 'object-hash', 'zlib'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global['psd-parse-web'] = {}, global.path, null, global.generateUUID, global.hash, global.zlib));
}(this, (function (exports, path, color, generateUUID, hash, zlib) { 'use strict';

	function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

	var path__default = /*#__PURE__*/_interopDefaultLegacy(path);
	var generateUUID__default = /*#__PURE__*/_interopDefaultLegacy(generateUUID);
	var hash__default = /*#__PURE__*/_interopDefaultLegacy(hash);
	var zlib__default = /*#__PURE__*/_interopDefaultLegacy(zlib);

	/**
	 * Created by rockyl on 2019-08-09.
	 */

	const PSD = window['require']('psd');

	async function getTree(file) {
		let psd = await PSD.fromDroppedFile(file);

		const root = {};
		walk(psd.tree(), root);

		return root;
	}

	function walk(psNode, dataNode) {
		const {left: pLeft = 0, top: pTop = 0,} = psNode.parent || {};
		const {left, top, width, height, name, layer: {opacity, visible}} = psNode;
		const x = left - pLeft;
		const y = top - pTop;

		Object.assign(dataNode, {x, y, width, height, alpha: opacity / 255, visible, name, origin: psNode, label: `${name} > [${x}, ${y}, ${width}, ${height}]`});
		if (psNode.children() && psNode.children().length > 0){
			dataNode.children = [];
		}

		let children = psNode.children();
		for (let i = children.length - 1; i >= 0; i--) {
			const childPsNode = children[i];

			const childDataNode = {};
			dataNode.children.push(childDataNode);
			walk(childPsNode, childDataNode);
		}
	}

	/**
	 * Created by rockyl on 2019-08-10.
	 */

	async function walkNode(node, callback, includeSelf = false) {
		if (includeSelf) {
			await callback(node, null);
		}
		if (node.children && node.children.length > 0) {
			for (let childNode of node.children) {
				await callback(childNode, node);
				const result = await walkNode(childNode, callback);
				if (result === true) {
					break;
				}
			}
		}
	}

	/**
	 * Created by rockyl on 2019-09-26.
	 *
	 * 导出zeroing的视图
	 */

	const relativePosPrefixMap = {
		l: {field: 'left',},
		t: {field: 'top',},
		r: {field: 'right',},
		b: {field: 'bottom',},
		h: {field: 'horizonCenter',},
		v: {field: 'verticalCenter',},
		wp: {field: 'width',},
		hp: {field: 'height',},
		lp: {field: 'left',},
		tp: {field: 'top',},
		rp: {field: 'right',},
		bp: {field: 'bottom',},
	};

	const offsetAll = 176;

	const enterChar = String.fromCharCode(13);

	async function execute(psdFile, options) {
		const tree = await getTree(psdFile);
		console.log('got tree:', tree);
		const {mode = 'none', singleView = true} = options;
		let offset = {x: 0, y: 0};
		let cutSize = {x: 0, y: 0};
		if (mode !== 'none') {
			cutSize.y = offsetAll;
		}
		switch (mode) {
			case 'top':
				offset.y = offsetAll;
				break;
			case 'center':
				offset.y = offsetAll / 2;
				break;
		}

		const isCenter = mode === 'center';

		let viewRoot = {
			name: path__default['default'].basename(psdFile.name, '.psd'),
			type: 'node',
			uuid: generateUUID__default['default'](),
		};

		const assets = [];
		const imageHashMap = {};

		let {width: stageWidthOrigin, height: stageHeightOrigin} = tree;
		const stageWidth = stageWidthOrigin - cutSize.x || 0;
		const stageHeight = stageHeightOrigin - cutSize.y || 0;

		await walkNode(tree, async function (node, parent) {
			let {name} = node;
			const {x, y, width, height, alpha, visible, origin: {layer, layer: {typeTool, solidColor}}} = node;
			console.log('walk node:', name);
			let properties = {
				width, height, alpha, visible,
			};
			const isSecondLayer = singleView && !parent.origin.parent || !singleView && parent.origin.parent && !parent.origin.parent.parent;
			const shouldVerticalCenter = isSecondLayer && isCenter;

			if (name.includes('|') || shouldVerticalCenter) {
				try {
					let arr = name.split('|');
					name = arr[0];
					let paramsStr = arr[1];
					let relativePos;
					if(paramsStr){
						let params = paramsStr.split(';');
						relativePos = params[0];
					}else if(shouldVerticalCenter){
						relativePos = 'v';
					}
					if (relativePos) {
						let items = relativePos.split(',');
						for (let item of items) {
							let result = item.match(/[a-zA-Z]+/);
							if (!result) {
								continue;
							}
							let prefix = result[0];
							let mapItem = relativePosPrefixMap[prefix];
							if (mapItem) {
								let {field,} = mapItem;
								let value = item.substr(prefix.length);
								let hasValue = value.length > 0;
								let fieldChar = prefix[0];
								if (!hasValue) {
									switch (fieldChar) {
										case 'l':
											value = x - offset.x;
											break;
										case 't':
											value = y - offset.y;
											break;
										case 'r':
											value = stageWidth - (x - offset.x) - width;
											break;
										case 'b':
											value = stageHeight - (y - offset.y) - height;
											break;
										case 'h':
											value = x + width / 2 - stageWidthOrigin / 2;
											break;
										case 'v':
											value = y + height / 2 - stageHeightOrigin / 2;
											break;
									}
								}
								let isPercent = prefix.endsWith('p');
								if (isPercent) {
									if (!hasValue) {
										switch (fieldChar) {
											case 'l':
											case 'r':
												value /= stageWidth;
												break;
											case 't':
											case 'b':
												value /= stageHeight;
												break;
										}
										value = Math.floor(value * 100);
									}
									value += '%';
								} else {
									value = parseFloat(value);
									if (isNaN(value)) {
										value = 0;
									}
								}
								properties[field] = value;
							}
						}
					}
				} catch (e) {
					console.log(e);
				}
			}

			let viewNode = {
				name,
				properties,
				uuid: generateUUID__default['default'](),
			};
			let dealLater = true;
			let px = x;// - (!singleView && isSecondLayer ? offset.x : 0);
			let py = y;// - (!singleView && isSecondLayer ? offset.y : 0);
			if (px !== 0) {
				properties.x = px;
			}
			if (py !== 0) {
				properties.y = py;
			}

			if (typeTool) {
				let fontInfo = typeTool();
				let text = fontInfo.textValue;
				if (text.indexOf(enterChar) < 0) {  //如果未包含回车说明是多行，去掉宽度
					delete properties['width'];
				}
				properties.text = text;
				const sizes = fontInfo.sizes();
				const colors = fontInfo.colors();
				properties.size = sizes ? sizes[0] || 20 : 20;
				let [r, g, b, a] = colors[0];
				properties.fillColor = `rgba(${r}, ${g}, ${b}, ${a / 255})`;
				viewNode.type = 'label';
				dealLater = false;
			} else if (solidColor && layer.vectorMask) {

				let paths = layer.vectorMask().paths;
				if (paths[2].numPoints === 4) {
					let isRect = true;
					for (let i = 3; i < paths.length; i++) {
						if (paths[i].recordType !== 2) {
							isRect = false;
							break;
						}
					}
					if (isRect) {
						viewNode.type = 'rect';
						const {r, g, b} = solidColor();
						properties.fillColor = `rgba(${r}, ${g}, ${b}, 1)`;
						dealLater = false;
					}
				}
			}

			if (dealLater) {
				if (node.hasOwnProperty('children')) {
					viewNode.type = 'node';
				} else {
					viewNode.type = 'image';

					let uuid = generateUUID__default['default']();
					const ext = '.png';

					let dataUrl;
					try {
						let img = node.origin.toPng();
						dataUrl = img.src;
					} catch (e) {

					}

					if (dataUrl) {
						let base64Data = dataUrl.replace(/^data:image\/\w+;base64,/, "");
						let buffer = new Buffer(base64Data, 'base64');
						const fileNameHash = hash__default['default'](buffer);
						if (imageHashMap.hasOwnProperty(fileNameHash)) {
							uuid = imageHashMap[fileNameHash];
						} else {
							imageHashMap[fileNameHash] = uuid;
							assets.push({
								name,
								ext,
								uuid,
								base64Data,
								hash: fileNameHash,
							});
						}

						properties.source = 'asset://' + uuid;
					}
				}
			}

			let viewParent = parent.view || viewRoot;
			if (!viewParent.hasOwnProperty('children')) {
				viewParent.children = [];
			}
			viewParent.children.push(viewNode);

			node.view = viewNode;
		});

		let data = {
			view: viewRoot,
			assets,
		};

		let dataString = JSON.stringify(data);

		let buf = new Buffer(dataString);
		return await new Promise((resolve, reject) => {
			zlib__default['default'].gzip(buf, function (err, res) {
				if (err) {
					reject(err);
				} else {
					console.log(res.length);
					resolve(res);
				}
			});
		})
	}

	exports.getTree = getTree;
	exports.toZeroing = execute;

	Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=index.umd.js.map
