/**
 * 转换接口数据
 */
interface ICData {
	width: number,
	height: number,
	nodes: INodeData[]
}

/**
 * 节点类型
 */
enum NodeType {
	ANY = 0,
	TEXT,
	IMAGE,
	CANVAS,
}

/**
 * 节点数据
 */
interface INodeData {
	type: NodeType,
	x: number,
	y: number,
	width: number,
	height: number,
	//图片需要
	src?: string,
	borderRadius?: string,
	img?: HTMLImageElement,//加载后挂上的
	imgBg?: HTMLImageElement,//背景图加载后挂上的
	//文本需要
	text?: string
	color?: string,
	fontFamily?: string,
	fontSize?: string,//文本字号
	fontWeight?: string,//文本粗细
	wordWrap?: "break-word" | null
	textAlign?: "center" | "left" | "right"
	fontStyle?: "normal" | "italic" | "oblique"

	//后面加的属性
	'backgroundColor'?,
	'backgroundImage'?,
	'borderColor'?,
	'borderWidth'?,
	'host'?,

	maskBegin?: boolean,
	maskEnd?: boolean,
}

export interface RenderOptions {
	scale?: number,
	type?: 'canvas' | 'jpeg' | 'png',
	quality?: number,
}

/**
 * 渲染
 * @param data
 * @param options
 * @param callback
 * @return Promise<HTMLCanvasElement | string>
 */
export async function toCanvas(data: ICData, options: RenderOptions = {}, callback?: (canvas: HTMLCanvasElement) => void): Promise<HTMLCanvasElement | string> {
	const {type: exportType = 'png', quality = 0.7} = options
	let {scale: dpiScale = 1} = options

	const scale = (window['devicePixelRatio'] || 1) * dpiScale

	let {nodes, width, height} = data
	width *= scale
	height *= scale

	let canvas = document.createElement("canvas")
	canvas.width = width
	canvas.height = height
	let ctx = canvas.getContext("2d")
	//先加载完所有图片
	let p: Promise<void>[] = []
	nodes.forEach((n) => {
		if (n.type == NodeType.IMAGE) {//图片标签
			p.push(
				new Promise((resolve, reject) => {
					let img = new Image()
					img.crossOrigin = 'anonymous'
					img.onload = () => {
						n.img = img
						resolve()
					}
					img.onerror = () => {
						reject(`html shot error: can't fetch image[${img.src}]`)
					}
					img.src = n.src
				})
			)
		}
		//背景图片标签
		if (n.backgroundImage) {
			p.push(
				new Promise((resolve, reject) => {
					let img = new Image()
					img.crossOrigin = 'anonymous'
					img.onload = () => {
						n.imgBg = img
						resolve()
					}
					img.onerror = () => {
						reject(`html shot error: can't fetch image[${img.src}]`)
					}
					let result = n.backgroundImage.match(/url\((.*)*\)/)
					if (result) {
						img.src = result[1].replace(/['"]/g, '')
					} else {
						reject(`html shot error: can't parse ${n.backgroundImage}`)
					}
				})
			)
		}
	})
	if (p.length) await Promise.all(p)
	nodes.forEach((n) => {
		if (n.maskBegin) {
			ctx.save();
			ctx.globalAlpha = 0;
			drawBackgroundColor(n, ctx, scale)
			ctx.globalAlpha = 1;//可能不需要
			ctx.clip()
		}
		//通用属性先绘制,背景颜色,边框等等
		//背景颜色
		if (n.backgroundColor) drawBackgroundColor(n, ctx, scale)
		//背景图片
		if (n.backgroundImage) ctx.drawImage(n.imgBg, n.x * scale, n.y * scale, n.width * scale, n.height * scale)

		switch (n.type) {
			case NodeType.IMAGE:
			case NodeType.CANVAS:
				drawImage(n, ctx, scale)
				break
			case NodeType.TEXT:
				drawText(n, ctx, scale)
				break
		}

		if (!n.maskBegin) {
			let node = n
			if(n.host){
				node = n.host
			}
			//边框
			if (parseInt(node.borderWidth)) {
				drawBorder(node, ctx, scale)
			}
		}

		if (n.maskEnd) {
			ctx.restore();
		}
	})

	let result: any = canvas
	switch (exportType) {
		case 'jpeg':
		case 'png':
			result = canvas.toDataURL('image/' + exportType, quality)
			break
	}

	//document.body.appendChild(canvas)

	callback && callback(result)
	return result
}

/**
 * 绘制图片
 * @param data
 * @param ctx
 * @param scale
 */
function drawImage(data: INodeData, ctx: CanvasRenderingContext2D, scale: number) {
	let {x, y, img, width, height, borderRadius: borderRadiusTxt} = data
	x *= scale
	y *= scale
	width *= scale
	height *= scale
	let borderRadius = parseFloat(borderRadiusTxt) * scale

	if (borderRadius) {//有圆角,画遮罩,暂时只管一个
		ctx.save()
		borderRadiusPath(data, ctx, scale)
		ctx.clip()
	}
	ctx.drawImage(img, x, y, width, height)
	if (borderRadius) ctx.restore()
}

/**
 * 绘制文本,暂时没那么复杂,不用离屏玩
 * @param data
 * @param ctx
 * @param scale
 */
function drawText(data: INodeData, ctx: CanvasRenderingContext2D, scale: number) {
	let {
		x, y, width, height,
		text,
		color,
		fontFamily,
		fontSize: fontSizeTxt,
		fontWeight,
		fontStyle,
		wordWrap,
		textAlign
	} = data

	x *= scale
	y *= scale
	width *= scale
	height *= scale
	let fontSize = parseFloat(fontSizeTxt) * scale

	let font = fontSize + 'px'
	font += ' ' + (fontFamily || "Arial")
	//font-weight:bold;font-style:italic
	if (fontWeight) font = fontWeight + " " + font
	if (fontStyle) font = fontStyle + " " + font

	ctx.font = font
	ctx.textBaseline = "bottom"
	ctx.fillStyle = color || "black"
	var textMetrics = ctx.measureText(text)
	let widthAll = textMetrics.width

	// var actualBoundingBoxAscent = textMetrics.actualBoundingBoxAscent
	// console.log(actualBoundingBoxAscent)
	// console.log(ctx.measureText(text))
	//超过宽度需要换行,且需要用到居中方式
	if (wordWrap == "break-word" && widthAll > width) {
		let realLines = []
		let w = ctx.measureText(text[0]).width
		let lineStr = text[0]
		let wordW = 0
		let strLen = text.length
		for (let j = 1; j < strLen; j++) {
			wordW = ctx.measureText(text[j]).width
			w += wordW
			if (w > width) {
				realLines[realLines.length] = lineStr
				lineStr = text[j]
				w = wordW
			} else {
				lineStr += text[j]
			}
		}
		//最后一行
		realLines[realLines.length] = lineStr
		ctx.textAlign = textAlign || "left"
		let tx = 0
		if (ctx.textAlign == "center") {
			tx = width * 0.5
		} else if (ctx.textAlign == "right") {
			tx = width
		}
		//有待考虑.现在直接拿高度取平均算每行高度
		let lineH = height / realLines.length
		realLines.forEach((r, i) => {
			ctx.fillText(r, x + tx, y + i * lineH + lineH)
		})
	} else {
		ctx.textAlign = "left"
		ctx.fillText(text, x, y + height)
	}
}


function drawBackgroundColor(data: INodeData, ctx: CanvasRenderingContext2D, scale: number) {
	let {
		x, y, width, height,
		backgroundColor,
		borderRadius: borderRadiusTxt
	} = data
	x *= scale
	y *= scale
	width *= scale
	height *= scale

	ctx.fillStyle = backgroundColor
	let borderRadius = parseFloat(borderRadiusTxt) * scale
	ctx.beginPath()
	if (borderRadius) {//有圆角,画遮罩,暂时只管一个
		borderRadiusPath(data, ctx, scale)
		ctx.fill()
	} else {
		// ctx.fillRect(x, y, width, height)
		ctx.moveTo(x, y)
		ctx.lineTo(x + width, y)
		ctx.lineTo(x + width, y + height)
		ctx.lineTo(x, y + height)
		ctx.lineTo(x, y)
		ctx.fill()
	}
}

function drawBorder(data: INodeData, ctx: CanvasRenderingContext2D, scale: number) {
	let {
		x, y, width, height,
		borderRadius: borderRadiusTxt,
		borderWidth: borderWidthTxt,
		borderColor
	} = data
	let borderRadius = parseFloat(borderRadiusTxt) * scale
	let borderWidth = parseInt(borderWidthTxt) * scale

	//ctx.lineCap = "round"
	//ctx.lineJoin = "round"
	ctx.miterLimit = 0
	ctx.strokeStyle = borderColor
	if (borderRadius) {
		ctx.lineWidth = borderWidth * 2
		borderRadiusPath(data, ctx, scale)
		ctx.stroke()
	} else {
		ctx.lineWidth = borderWidth
		x *= scale
		y *= scale
		width *= scale
		height *= scale
		x += borderWidth / 2
		y += borderWidth / 2
		width -= borderWidth
		height -= borderWidth

		ctx.strokeRect(x, y, width, height)
	}
}

function borderRadiusPath(data: INodeData, ctx: CanvasRenderingContext2D, scale: number) {
	let {
		x, y, width, height,
		borderRadius: borderRadiusTxt,
		borderWidth: borderWidthTxt,
	} = data

	let borderRadius = parseFloat(borderRadiusTxt) * scale

	x *= scale
	y *= scale
	width *= scale
	height *= scale

	if (borderRadius) {//有圆角,画遮罩,暂时只管一个
		let max = (width < height ? width : height) / 2
		let radius = Math.min(borderRadius, max)
		// ctx.beginPath()
		// ctx.moveTo(x, y + radius)
		// ctx.quadraticCurveTo(x, y, x + radius, y)
		// ctx.lineTo(x + width - radius, y)
		// ctx.quadraticCurveTo(x + width, y, x + width, y + radius)
		// ctx.lineTo(x + width, y + height - radius)
		// ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height)
		// ctx.lineTo(x + radius, y + height)
		// ctx.quadraticCurveTo(x, y + height, x, y + height - radius)
		// ctx.lineTo(x, y + radius)
		// ctx.clip()
		ctx.beginPath()
		ctx.moveTo(x + width - radius, y)
		ctx.arcTo(x + width, y, x + width, y + radius, radius)
		ctx.lineTo(x + width, y + height - radius)
		ctx.arcTo(x + width, y + height, x + width - radius, y + height, radius)
		ctx.lineTo(x + radius, y + height)
		ctx.arcTo(x, y + height, x, y + height - radius, radius)
		ctx.lineTo(x, y + radius)
		ctx.arcTo(x, y, x + radius, y, radius)
		ctx.closePath()
	}
}
