/**
 * Created by rockyl on 2019-09-19.
 *
 * 封装fetch请求
 */

import './object-assign-polyfill'
import {stringify} from 'querystringify';
import {codeOnlyMode} from './builtin-successfully-judges'

export * from './builtin-successfully-judges';

/**
 * api错误
 */
class ApiError extends Error {
	code: number = 0;
	details: string = '';

	constructor(name: string, code?: number, message?: string) {
		super();

		if (name) {
			this.name = name;
		}
		if (code) {
			this.code = code;
		}
		if (message) {
			this.message = message;
		}
	}
}

/**
 * 总配置
 */
interface SetupOptions {
	apiHost?: string;
	fetch?: Function;
	successfullyJudge?: (resp) => boolean;
	messageField?: string;
}

/**
 * 请求参数
 */
interface FetchApiOptions {
	/**
	 * 域名
	 * @default ''
	 */
	host?: string;
	/**
	 * 请求参数
	 * @default null
	 */
	params?: any;
	/**
	 * 头部参数
	 * @default null
	 */
	headers?: any;
	/**
	 * 请求方法
	 * @default 'get'
	 */
	method?: 'get' | 'post';
	/**
	 * 携带凭证类型
	 * @default 'include'
	 */
	credentials?: 'omit' | 'same-origin' | 'include';
	/**
	 * 返回类型
	 * @default 'json'
	 */
	contentType?: string;
	/**
	 * 附加错误信息
	 * @default null
	 */
	errMessage?: string;
}

let _options: SetupOptions = {
	apiHost: '',
	successfullyJudge: codeOnlyMode,
	messageField: 'msg',
};

/**
 * 配置
 * @param options
 */
export function setup(options: SetupOptions) {
	Object.assign(_options, options);
}

/**
 * 请求接口
 * @param uri string
 * @param options FetchApiOptions
 */
export async function fetchApi(
	uri: string,
	{
		host = '',
		params,
		headers = {},
		method = 'get',
		credentials = 'include',
		contentType = 'json',
		errMessage,
	}: FetchApiOptions = {},
) {
	let url =
		host +
		(uri.startsWith('http') || uri.startsWith('//') ? uri : _options.apiHost + uri);

	const options: any = {
		method,
		headers: {...headers},
		credentials,
	};
	if (params) {
		if (method.toLowerCase() === 'post') {
			switch (contentType) {
				case 'form-data':
					let formData = new FormData();
					for (let key in params) {
						let value = params[key];
						if (value instanceof File) {
							formData.append(key, value, value.name);
						} else {
							formData.append(key, value);
						}
					}
					options.body = formData;
					break;
				case 'form':
					options.headers['Content-Type'] = 'application/x-www-form-urlencoded';
					options.body = stringify(params);
					break;
				case 'json':
					options.headers['Content-Type'] = 'application/json';
					options.body = JSON.stringify(params);
					break;
			}
		} else {
			url += url.indexOf('?') < 0 ? '?' : '';
			url += (url.endsWith('?') ? '' : '&') + stringify(params);
		}
	}

	try {
		let fetchMethod = _options.fetch;
		const response = await fetchMethod(url, options);
		const respText = await response.text();

		if (response.status === 310) {
			//客户端重定向,用于跨域重定向
			console.log('重定向到:' + respText)
			try {
				location.href = respText;
			} catch (e) {

			}
		}

		let jsonObj = JSON.parse(respText);
		if (_options.successfullyJudge(jsonObj)) {
			return jsonObj.data;
		}

		console.log(jsonObj)

		return Promise.reject(
			new ApiError('call api failed', jsonObj.code, errMessage || jsonObj[_options.messageField]),
		);
	} catch (e) {
		return Promise.reject(e);
	}
}
