/**
 * Created by rockyl on 2020-02-12.
 */
import {getConfigs, prepare as prepareRemoteConfigs} from "./remote-configs";

const restify = require('restify')
import {getEurekaClient} from "./eureka"
import logger from "./logger"
import * as config from "config"
import applyApi from "./api"
import {setup as setupOSS} from 'ali-oss-duiba'
import cors from "./cors";
import {prepareRedis} from "./redis";

let eureka

const server = restify.createServer({
	name: 'tiny-image',
	version: '1.0.0'
})

const ignoreUrlLogger = ['/monitor/check']

server.use(restify.plugins.acceptParser(server.acceptable))
server.use(restify.plugins.queryParser())
server.use(restify.plugins.bodyParser())
server.use(cors())
server.use(function (req, res, next) {
	if (!ignoreUrlLogger.includes(req.url as string)) {
		logger.info(req.url)
	}
	return next()
})

applyApi(server)

process.on('uncaughtException', (err) => {
	logger.error(err)
})

process.on("SIGTERM", () => {
	eureka.stop()
	setTimeout(() => {
		process.exit(1)
	}, 6000)
})

async function start() {
	await prepareRemoteConfigs()

	const ossConfigs = getConfigs(
		{name: `apollo.oss.duiba.accessId`, alias: 'accessId'},
		{name: `apollo.oss.duiba.accessKey`, alias: 'accessKey'},
		{name: `apollo.oss.duiba.bucketName`, alias: 'bucketName'},
	)
	logger.info('got oss configs:\n', ossConfigs)

	const redisConfigs = getConfigs(
		{name: `apollo.redis.duiba-tiny-image-node.host`, alias: 'host'},
		{name: `apollo.redis.duiba-tiny-image-node.port`, alias: 'port'},
		{name: `apollo.redis.duiba-tiny-image-node.password`, alias: 'password'},
	)
	logger.info('got redis configs:\n', redisConfigs)

	setupOSS({
		accessKeyId: ossConfigs.accessId,
		accessKeySecret: ossConfigs.accessKey,
		bucket: ossConfigs.bucketName,
		region: 'oss-cn-hangzhou',
	})

	await prepareRedis(redisConfigs)

	const port = config.get('PORT')
	server.listen(port, async function () {
		if (process.env.NODE_ENV) {
			try {
				eureka = await getEurekaClient(port)
				logger.info('eureka register success')
			} catch (e) {
				logger.error(e.name)
			}
		}
		logger.info('%s listening at %s', server.name, server.url)
	})
}

start().catch(e => {
	console.log(e)
	process.exit(2)
})
