/**
 * Created by rockyl on 2020-02-13.
 */

import logger from "./logger"
import {compressImage} from "./compress"
import {uploadFile} from 'ali-oss-duiba'
import * as objectHash from 'object-hash'
import * as path from 'path'
import * as fs from 'fs-extra'
import {getUrlFromCache, setUrlToCache} from "./redis";

const TYPE_DATA_URL = 'text/dataurl'
const acceptTypes = ['image/png', 'image/jpeg', TYPE_DATA_URL]

export default function applyApi(server) {
	server.get('/info', function (req, res, next) {
		res.sendRaw('Tiny image serve is running.')
		return next()
	})

	server.get('/monitor/check', function (req, res, next) {
		res.sendRaw('ok')
		return next()
	})

	server.post('/tinify', tinify)
}

//一个文件可以返回blob，多个文件必须传cdnRoot
async function tinify(req, res, next) {
	const {type = 'builtin', cdnRoot, ignoreCache} = req.query

	const inputs = await prepareInputs(req, ignoreCache)
	const fileCount = inputs.length
	if (fileCount === 0) {
		res.send({
			code: 1,
			msg: 'file not exists'
		})
	} else {
		const needUpdateHashMap = {}
		if (fileCount === 1) {
			const [input] = inputs
			if (cdnRoot && input.url) {
				res.send({
					code: 0,
					data: input.url
				})
			} else {
				const result = await tinifyOnce(input, type, cdnRoot)
				if (typeof result === 'string') {
					needUpdateHashMap[input.hash] = result
					res.send({
						code: 0,
						data: result
					})
				} else if (Buffer.isBuffer(result)) {
					res.sendRaw(result, {
						'Content-Type': input.mimeType,
					})
				} else {
					logger.error('/tinify', 'compress failed', result.message)
					res.send({
						code: 3,
						msg: 'compress failed'
					})
				}
			}
		} else if (cdnRoot) {
			const result = await Promise.all<any>(inputs.map(input => tinifyOnce(input, type, cdnRoot)
				.then(result => {
					if (result) {
						needUpdateHashMap[input.hash] = result
					} else {
						result = input.url
					}
					if (typeof result === 'string') {
						return {
							key: input.key,
							url: result,
						}
					} else {
						return result
					}
				})
			))
			const data = {}
			for (let item of result) {
				if (item.key) {
					data[item.key] = item.url
				}
			}
			res.send({
				code: 0,
				data,
			})
		} else {
			res.send({
				code: 2,
				msg: 'specify cdnRoot when multi files'
			})
		}
		await setUrlToCache(needUpdateHashMap)
	}
	return next()
}

async function prepareInputs(req, ignoreCache) {
	const inputs: any[] = []
	for (let key in req.files) {
		const file = req.files[key]
		if (!acceptTypes.includes(file.type)) {
			continue
		}

		let extname, buffer, mimeType
		if (file.type === TYPE_DATA_URL) {
			const dataUrl = await fs.readFile(file.path, 'utf-8')
			const result = dataUrl.match(/data:(image\/\w+);/)
			if (result) {
				mimeType = result[1]
				if (!acceptTypes.includes(mimeType)) {
					continue
				}
				extname = '.' + mimeType.substr(mimeType.indexOf('/') + 1)
				let base64Data = dataUrl.replace(/^data:image\/\w+;base64,/, "")
				buffer = new Buffer(base64Data, 'base64')
			}
		} else {
			mimeType = file.type
			if (!acceptTypes.includes(mimeType)) {
				continue
			}
			extname = '.' + mimeType.substr(mimeType.indexOf('/') + 1)
			buffer = await fs.readFile(file.path)
		}

		const input: any = {
			mimeType,
			key,
			extname,
			buffer,
			hash: objectHash(buffer),
		}
		inputs.push(input)
	}

	if (!ignoreCache) {
		const hashs = inputs.map(input => input.hash)
		const urls = await getUrlFromCache(hashs)
		for (let i = 0; i < inputs.length; i++) {
			const input = inputs[i]
			if (urls[i]) {
				input.url = urls[i]
			}
		}
	}

	return inputs
}

async function tinifyOnce(input, type, cdnRoot) {
	try {
		if (input.url) {
			return
		}
		const buffer = await compressImage(input.buffer, type)
		console.log(`compressImage size: ${input.buffer.length} => ${buffer.length}`)
		if (cdnRoot) {
			const cdnPath = path.join(cdnRoot, objectHash(buffer) + input.extname)
			return uploadFile(buffer, cdnPath)
		} else {
			return buffer
		}
	} catch (e) {
		logger.error('tinifyOnce error:', e)
		return e
	}
}
