package com.oto.utils;
import static com.oto.config.BasicConfig.*;
import static io.restassured.RestAssured.*;

import com.alibaba.fastjson.JSONObject;
import io.restassured.response.Response;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;


public class NetworkUtils {

    // 代理人的 cookies和headers
    public Map<String, Object> agentCookies = new HashMap<String, Object>();
    public Map<String, Object> agentHeaders = new HashMap<String, Object>();
    public Map<String, Object> fileHeaders = new HashMap<String, Object>();

    // 钉钉的 headers
    public Map<String, Object> dingTalkHeaders = new HashMap<String, Object>();
    private volatile static NetworkUtils network;

    // 构造函数 初始化tku、content-type
    private NetworkUtils(){
        //cookies中的tku
        agentCookies.put("tku",AGENT_TKU);
        //多场景测试标志
//        agentCookies.put("_duibaServiceGroupKey","miria-2652");
        //json类型的headers
        agentHeaders.put("Content-Type","application/json");
        //二进制文件的headers
        fileHeaders.put("Content-Type","multipart/form-data");
        //钉钉的headers
        dingTalkHeaders.put("Content-Type","application/json;charset=utf-8");
    }

    //单例模式
    //通过设置同步代码块，使用DCL双检查锁机制
    //使用双检查锁机制成功的解决了单例模式的懒汉实现的线程不安全问题和效率问题
    //DCL 也是大多数多线程结合单例模式使用的解决方案
    public static NetworkUtils getInstance() {
        if (network == null) {
            synchronized (NetworkUtils.class) {
                if (network == null) {
                    network = new NetworkUtils();
                }
            }
        }
        return network;
    }


    // get 不带参请求
    public Response getResponse(String path) {
        Response response = given()
                .cookies(agentCookies)
                .headers(agentHeaders)
                .get(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        System.out.println(response.body().asString());
        return response;
    }

    // get 带参请求
    public Response getResponse(Map<String, Object> params, String path) {
        Response response = given()
                .cookies(agentCookies)
                .headers(agentHeaders)
                .params(params)
                .get(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        System.out.println(response.body().asString());
        return response;
    }

    // post 不帶参请求
    public Response postResponse(String path) {
        Response response = given()
                .cookies(agentCookies)
                .headers(agentHeaders)
                .post(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        System.out.println(response.body().asString());
        return response;
    }

    // post 带参请求
    public Response postResponse(Map<String, Object> params, String path) {
        String jsonStr = JSONObject.toJSONString(params);
        Response response = given()
                .cookies(agentCookies)
                .headers(agentHeaders)
                .body(jsonStr)
                .post(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        System.out.println(response.body().asString());
        return response;
    }

    // post 头部带参请求
    public Response postHeaderResponse(Map<String, Object> params, String path) {
        Response response = given()
                .cookies(agentCookies)
                .params(params)
                .post(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        System.out.println(response.body().asString());
        return response;
    }

    // post 文件上传
    public Response postFileResponse(String filePath, String path){
        Response response = given()
                .cookies(agentCookies)
                .headers(fileHeaders)
                .multiPart("file",new File(filePath))
                .post(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        return response;
    }

    // post xml请求方式
    public Response postXMLResponse(String str, String path) {
        Response response = given()
                .cookies(agentCookies)
                .headers(agentHeaders)
                .body(str)
                .post(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        return response;
    }

    // 管理后台
    public Response getManagerCookies(String path){
        Response response = given().
                cookies(agentCookies).
                redirects().
                follow(false).
                get(path);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，path=【"+path+"】 code=【"+response.statusCode()+"】");
        }
        return response;
    }

    // post 钉钉机器人
    public Response dingTalkRobot(Map<String, Object> params) {
        String jsonStr = JSONObject.toJSONString(params);
        Response response = given()
                .headers(dingTalkHeaders)
                .body(jsonStr)
                .post(dingTalkPath);
        if (!Objects.equals(response.statusCode(),200)){
            System.out.println("请求状态码错误，code=【"+response.statusCode()+"】");
        }
        return response;
    }

    /**
     *  带参的接口完整信息
     * @param params 接口参数
     * @param path   接口路径
     * @param errorMessage  自己打印的错误信息
     * @param responseBody  返回的response
     * @return  完整的接口信息
     */
    public String message(Map<String, Object> params, String path, String errorMessage, String responseBody){
        return  "\n" + "path: " + path +
                "\n" + "params: " + params +
                "\n" + "message: " + errorMessage +
                "\n" + "response：" + responseBody + "\n";
    }

    /**
     * 不带参的接口完整信息
     * @param path 接口路径
     * @param errorMessage  自己打印的错误信息
     * @param responseBody  返回的response
     * @return  完整的接口信息
     */
    public String message(String path, String errorMessage, String responseBody){
        return  "\n" + "path: " + path +
                "\n" + "message: " + errorMessage +
                "\n" + "response：" + responseBody + "\n";
    }



}
