import { defineConfig } from 'vite'
import react from '@vitejs/plugin-react'
import legacy from '@vitejs/plugin-legacy'
import autoprefixer from "autoprefixer"
import postcsspxtorem from "postcss-pxtorem"
import { viteMockServe } from "vite-plugin-mock";
import DuibaPublish from "./config/DuibaPublish/DuibaPublish.ts";
import dotenv from 'dotenv';

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => {

        dotenv.config({ path: [`./config/.env.global`, `./config/.env.${mode}`] });

        const {
            NODE_ENV,
            UPLOAD_DIR, CDN_DOMAIN,
            OSS_REGION, OSS_BUCKET,
            OSS_ACCESS_KEY_ID, OSS_ACCESS_SECRET,
        } = process.env;

        // console.log(UPLOAD_DIR, NODE_ENV, CDN_DOMAIN, OSS_REGION, OSS_BUCKET, OSS_ACCESS_KEY_ID, OSS_ACCESS_SECRET)

        const isDev = NODE_ENV == "development";

        const versionStamp = Date.now();

        const prodBase = `${CDN_DOMAIN}/${UPLOAD_DIR}/${versionStamp}/`;

        return {
            base: !isDev ? prodBase : "",
            server: {
                // port: 3001
                open: true,
            },
            assetsInclude: [/\.(svga)$/],
            plugins: [
                react({
                    babel: {
                        plugins: [
                            ["@babel/plugin-proposal-decorators", { legacy: true }],
                            ["@babel/plugin-proposal-class-properties", { loose: true }],
                        ],
                    },
                }),
                legacy({
                    targets: ['defaults', 'not IE 11'],
                }),
                viteMockServe({
                    // default
                    mockPath: 'mock',
                    enable: true,
                }),
                !isDev && DuibaPublish({
                    buildVersion: versionStamp,
                    uploadDir: UPLOAD_DIR,
                    region: OSS_REGION,
                    bucket: OSS_BUCKET,
                    accessKeyId: OSS_ACCESS_KEY_ID,
                    accessKeySecret: OSS_ACCESS_SECRET,
                }),
            ],
            css: {
                postcss: {
                    plugins: [
                        autoprefixer({
                            overrideBrowserslist: [
                                "Android 4.1",
                                "iOS 7.1",
                                "Chrome > 31",
                                "ff > 31",
                                "ie >= 8",
                                "last 10 versions", // 所有主流浏览器最近10版本用
                            ],
                            grid: true
                        }),
                        postcsspxtorem({
                            rootValue: 100,
                            propList: ["*", "!border"], // 除 border 外所有px 转 rem
                            selectorBlackList: [".noRem-"], // 过滤掉.noRem-开头的class，不进行rem转换
                        }),
                    ],
                },
                preprocessorOptions: {
                    less: {
                        javascriptEnabled: true,
                    },
                },
                modules: {
                    localsConvention: 'camelCase'
                }
            }
        }
    }
)
