import React, { forwardRef, useEffect, useImperativeHandle, useRef } from "react";

import { Player } from 'svga';
import { loadSvga } from "../utils/loader.ts";

export interface BaseSvgaPlayerProps {
    className?: string;
    styles?: { icon: React.CSSProperties };
    src: string;

    autoplay?: boolean;
    loop?: boolean;

    startFrame?: number,
    endFrame?: number,

    onStart?: () => void;
    onEnd?: () => void;

}

type MergedHTMLAttributes = Omit<
    React.HTMLAttributes<HTMLElement> &
    React.CanvasHTMLAttributes<HTMLElement>,
    'type' | 'color'
>;

export interface SvgaPlayerRef {
    nativeElement: HTMLCanvasElement;
    player: Player;
    start: () => Player;
    pause: () => Player;
    resume: () => Player;
    stop: () => Player;

}


export interface SvgaPlayerProps extends BaseSvgaPlayerProps, MergedHTMLAttributes {
}

export const SvgaPlayer = forwardRef<
    SvgaPlayerRef,
    SvgaPlayerProps
>((props, ref) => {
    const {
        className, style,
        onClick,
        src, autoplay = true, loop = true,
        startFrame = 0,
        endFrame = 0,
        onStart, onEnd,
    } = props;
    const canvasRef = useRef<HTMLCanvasElement>(null);

    const playInfo = useRef<{
        playing: boolean;
        player: Player;
    }>({
        playing: false,
        player: null,
    });

    useImperativeHandle(ref, () => ({
        nativeElement: canvasRef.current,
        player: playInfo.current.player,
        start: () => {
            playInfo.current.player.start();
            return playInfo.current.player;
        },
        pause: () => {
            playInfo.current.player.pause();
            return playInfo.current.player;
        },
        resume: () => {
            playInfo.current.player.resume();
            return playInfo.current.player;
        },
        stop: () => {
            playInfo.current.player.stop();
            return playInfo.current.player;
        }
    }));

    useEffect(() => {
        const player = new Player({ container: canvasRef.current });
        playInfo.current.player = player;

        player.onStart = () => {
            onStart && onStart();
        }
        player.onResume = () => {
            console.log('onResume')
        }
        player.onPause = () => {
            console.log('onPause')
        }
        player.onStop = () => {
            console.log('onStop')
        }
        // player.onProcess = () => console.log('onProcess', player);
        player.onEnd = () => {
            console.log('onEnd')
            onEnd && onEnd();
        }

        return () => {
            player.destroy();
        }
    }, []);

    useEffect(() => {
        updateSrc();
    }, [src]);

    const updateSrc = async () => {

        if (!src) return;

        const svga = await loadSvga(src);

        const { player } = playInfo.current;

        player.setConfig({
            loop,
            startFrame,
            endFrame,
        })

        player.clear();
        if (svga) {
            await player.mount(svga);
            if (autoplay) {
                player.start();
            } else {
                console.log(player);
                // @ts-ignore
                player.drawFrame(startFrame);
            }
        }

    }

    return <canvas className={className} style={style} onClick={onClick} ref={canvasRef} />
});
