import requestModule from './request.js';
const {
    api
} = requestModule;



/**
 * 保存地址
 * @param {Object} data - 请求参数
 * @param {string} [data.id] - 地址ID (可选，编辑时必传)
 * @param {boolean} [data.current=true] - 是否是默认地址 (可选，默认true)
 * @param {string} data.name - 联系人 (必须)
 * @param {string} data.mobile - 联系电话 (必须)
 * @param {string} data.email - 邮箱 (必须)
 * @param {string} [data.province] - 省 (可选)
 * @param {integer} [data.provinceCode] - 省代码 (可选)
 * @param {string} [data.city] - 市 (可选)
 * @param {integer} [data.cityCode] - 市代码 (可选)
 * @param {string} [data.area] - 区/县 (可选)
 * @param {integer} [data.areaCode] - 区/县代码 (可选)
 * @param {string} [data.town] - 街道地址 (可选)
 * @param {integer} [data.townCode] - 街道代码 (可选)
 * @param {string} data.detail - 详细地址 (必须)
 * @param {string} [data.postalCode] - 邮政编码 (可选)
 * @param {string} [data.floorNo] - 门牌号 (可选)
 * @returns {Promise} API响应
 * @description 保存或更新地址信息
 * @response {Object} 响应数据
 * @response {boolean} response.ok - 请求是否成功
 * @response {boolean} response.success - 操作是否成功
 * @response {string} response.msg - 响应消息
 * @response {string} response.code - 响应代码
 * @response {Object} response.data - 响应数据
 */
export const saveAddress = (data) => api.post('/c/user/address/save', data);


/**
 * 获取地址列表
 * @returns {Promise} API响应
 * @description 返回地址列表，每个地址对象包含以下字段：
 * - id: 地址ID (string)
 * - current: 是否是默认地址 (boolean)
 * - name: 联系人 (string)
 * - mobile: 联系电话 (string)
 * - email: 邮箱 (string)
 * - province: 省 (string)
 * - provinceCode: 省代码 (integer)
 * - city: 市 (string)
 * - cityCode: 市代码 (integer)
 * - area: 区/县 (string)
 * - areaCode: 区/县代码 (integer)
 * - town: 乡镇 (string)
 * - townCode: 乡镇代码 (integer)
 * - detail: 详细地址 (string)
 * - postalCode: 邮政编码 (string)
 * - floorNo: 门牌号 (string)
 */
export const getAddressList = () => api.get('/c/user/address');


/**
 * 更新地址
 * @param {Object} data - 请求参数
 * @param {string} [data.id] - 地址ID (可选，编辑时必传)
 * @param {boolean} [data.current=true] - 是否是默认地址 (可选，默认true)
 * @param {string} data.name - 联系人 (必须)
 * @param {string} data.mobile - 联系电话 (必须)
 * @param {string} data.email - 邮箱 (必须)
 * @param {string} [data.province] - 省 (可选)
 * @param {integer} [data.provinceCode] - 省代码 (可选)
 * @param {string} [data.city] - 市 (可选)
 * @param {integer} [data.cityCode] - 市代码 (可选)
 * @param {string} [data.area] - 区/县 (可选)
 * @param {integer} [data.areaCode] - 区/县代码 (可选)
 * @param {string} [data.town] - 街道地址 (可选)
 * @param {integer} [data.townCode] - 街道代码 (可选)
 * @param {string} data.detail - 详细地址 (必须)
 * @param {string} [data.postalCode] - 邮政编码 (可选)
 * @param {string} [data.floorNo] - 门牌号 (可选)
 * @returns {Promise} API响应
 * @description 更新地址信息
 * @response {Object} 响应数据
 * @response {boolean} response.ok - 请求是否成功
 * @response {boolean} response.success - 操作是否成功
 * @response {string} response.msg - 响应消息
 * @response {string} response.code - 响应代码
 * @response {Object} response.data - 响应数据
 */
export const updateAddress = (data) => api.post('/c/user/address/update', data);


/**
 * 删除地址
 * @param {Object} data - 请求参数
 * @param {string} data.id - 地址ID (必须)
 * @returns {Promise} API响应
 * @description 删除指定ID的地址
 * @response {Object} 响应数据
 * @response {boolean} response.ok - 请求是否成功
 * @response {boolean} response.success - 操作是否成功
 * @response {string} response.msg - 响应消息
 * @response {string} response.code - 响应代码
 * @response {boolean} response.data - 响应数据
 */
export const deleteAddress = (data) => api.post('/c/user/address/del', data);



/**
 * 设置默认地址
 * @param {*} data 
 * @returns 
 */
export const setDefaultAddress = (data) => api.post('/c/user/address/current', data);


/**
 * 获取地址详情
 * @param {Object} data - 请求参数
 * @param {string} data.id - 地址ID (必须)
 * @returns {Promise} API响应
 * @description 根据地址ID获取地址详细信息
 * @response {Object} 响应数据
 * @response {boolean} response.ok - 请求是否成功
 * @response {boolean} response.success - 操作是否成功
 * @response {string} response.msg - 响应消息
 * @response {string} response.code - 响应代码
 * @response {Object} response.data - 地址详情数据
 * @response {string} response.data.id - 地址ID
 * @response {boolean} response.data.current - 是否是默认地址
 * @response {string} response.data.name - 联系人
 * @response {string} response.data.mobile - 联系电话
 * @response {string} response.data.email - 邮箱
 * @response {string} response.data.province - 省
 * @response {integer} response.data.provinceCode - 省代码
 * @response {string} response.data.city - 市
 * @response {integer} response.data.cityCode - 市代码
 * @response {string} response.data.area - 区/县
 * @response {integer} response.data.areaCode - 区/县代码
 * @response {string} response.data.town - 乡镇
 * @response {integer} response.data.townCode - 乡镇代码
 * @response {string} response.data.detail - 详细地址
 * @response {string} response.data.postalCode - 邮政编码
 * @response {string} response.data.floorNo - 门牌号
 */
export const getAddressDetail = (data) => api.get('/c/user/address/detail', data);