// 导入全局store
import {useGlobalStore} from "../stores/global";

// 跳转类型枚举
export const JumpType = {
	INNER: 1, // 内部小程序页面跳转
	MINI: 2, // 其他小程序页面跳转
	H5: 3, // https 网络连接跳转
};

const OPEN_EMBEDDED_APPS = [
	"wx4205ec55b793245e", // 星妈优选
	"wxab14ac8f684ad962",
]

/**
 * 通用跳转方法
 * @param {Object} options
 * @param {number} options.type 跳转类型（1-内部，2-小程序，3-H5）
 * @param {string} options.url 跳转地址（页面路径/小程序路径/https链接）
 * @param {Object} [options.extra] 额外参数（如 appId、extraData 等）
 */
export function jump({type, url, extra = {}}) {
	switch (type) {
		case JumpType.INNER:
			try {

				const pages = getCurrentPages();
				const currentPage = pages[pages.length - 1];
				const currentPath = currentPage.route;

				if (url.indexOf('pages/index/index') >= 0 && currentPath.indexOf('pages/index/index') >= 0) {
					// 获取全局store
					const globalStore = useGlobalStore();
					if (url.indexOf('home') >= 0) {
						globalStore.setCurTabIndex(0);
					} else if (url.indexOf('brand') >= 0) {
						globalStore.setCurTabIndex(1);
					} else if (url.indexOf('integral') >= 0) {
						globalStore.setCurTabIndex(2);
					} else if (url.indexOf('my') >= 0) {
						globalStore.setCurTabIndex(3);
					}
				} else {
					// 内部小程序页面跳转
					uni.navigateTo({
						url,
						...extra,
					});
				}
			} catch (error) {
				console.error('跳转处理错误:', error);
				// 出错时默认执行内部跳转
				uni.navigateTo({
					url,
					...extra,
				});
			}

			break;
		case JumpType.MINI:
			// 跳转到其他小程序页面
			console.log("extra:", url, extra);

			const jumpParams = {
				appId: extra.appId || "",
				path: url,
				extraData: extra.extraData || {},
				envVersion: extra.envVersion || "release",
				success: extra.success,
				fail: extra.fail,
			};

			console.log("jumpParams:", jumpParams);

			if (extra.embedded || OPEN_EMBEDDED_APPS.includes(jumpParams.appId)) {
				console.log('半屏拉起', jumpParams)
				uni.openEmbeddedMiniProgram(jumpParams);
			} else {
				console.log('直接跳转', jumpParams)
				uni.navigateToMiniProgram(jumpParams);
			}
			break;
		case JumpType.H5:
			// 跳转到 https 网络链接
			if (/^https?:\/\//.test(url)) {
				// uni-app 推荐用 webview 页面承载 H5
				uni.navigateTo({
					url: `/pages/webview/webview?url=${encodeURIComponent(url)}`,
				});
			} else {
				console.error("H5 跳转地址必须为 https 链接");
			}
			break;
		default:
			console.error("不支持的跳转类型");
	}
}

export function formatDate(timestamp) {
	const date = new Date(timestamp);
	if (isNaN(date.getTime())) {
		console.error('无效的时间戳:', timestamp);
		return '';
	}

	const year = date.getFullYear()
	const month = String(date.getMonth() + 1).padStart(2, '0')
	const day = String(date.getDate()).padStart(2, '0')
	return `${year}-${month}-${day}`
}

/**
 * 防连点函数
 * @param {Function} fn 需要防连点的函数
 * @param {number} [delay=1000] 防连点时间间隔(ms)
 * @returns {Function} 包装后的防连点函数
 */
export function throttleTap(fn, delay = 1000) {
	let timer = null;
	let lastTime = 0;

	return function (...args) {
		const now = Date.now();
		if (now - lastTime < delay) {
			// 小于间隔时间,阻止执行
			console.log("防连点");
			return;
		}

		lastTime = now;
		clearTimeout(timer);

		// 立即执行一次
		fn.apply(this, args);

		timer = setTimeout(() => {
			lastTime = 0; // 重置时间
		}, delay);
	};
}

let loadingCount = 0;
let loadingTimer = null;

export function showLoading(title = "加载中") {
	if (loadingCount == 0) {
		uni.showLoading({
			title,
			mask: true,
		});

		loadingTimer = setTimeout(() => {
			uni.hideLoading();
		}, 10000);
	}
	loadingCount++;
}

export function hideLoading() {
	if (loadingCount <= 0) return;

	loadingCount--;
	if (loadingCount == 0) {
		loadingTimer && clearTimeout(loadingTimer);
		uni.hideLoading();
	}
}

/**
 * 日期格式化
 * @param date    接收可以被new Date()方法转换的内容
 * @param format  字符串，需要的格式例如：'yyyy/MM/dd hh:mm:ss'
 * @returns {String}
 */
const dateFormatter = (date, format = "yyyy/MM/dd") => {
	if (!date) return "-";
	date = new Date(
		typeof date === "string" && isNaN(date)
			? date.replace(/-/g, "/")
			: Number(date)
	);
	const o = {
		"M+": date.getMonth() + 1,
		"d+": date.getDate(),
		"h+": date.getHours(),
		"m+": date.getMinutes(),
		"s+": date.getSeconds(),
		"q+": Math.floor((date.getMonth() + 3) / 3),
		S: date.getMilliseconds(),
	};
	if (/(y+)/.test(format)) {
		format = format.replace(
			RegExp.$1,
			(date.getFullYear() + "").substr(4 - RegExp.$1.length)
		);
	}
	for (const k in o) {
		if (new RegExp("(" + k + ")").test(format)) {
			format = format.replace(
				RegExp.$1,
				RegExp.$1.length === 1 ? o[k] : ("00" + o[k]).substr(("" + o[k]).length)
			);
		}
	}
	return format;
};
export {
	dateFormatter
}