# MonthGift 月礼包权益组件

## 组件概述

`MonthGift` 是一个 Vue 3 组件，用于展示不同用户等级的月礼包权益，包括精选好券和限时1分购功能。组件根据传入的用户等级参数动态显示对应的权益页面。

## 功能特性

### 1. 精选好券 (Selected Good Coupons)
- **领取条件**: 每月可领取1次，黄金及以上等级都可领取
- **更新时间**: 每月28日10点更新
- **按钮状态**:
  - 初始状态: "立即领取"
  - 领取成功后: "去使用" (跳转到星妈优选优惠券页面)
  - 下月更新后: 重新变为 "立即领取"
- **等级切换**: 若用户在高等级已领取权益，切换其他等级时均变为 "去使用"

### 2. 限时1分购 (Limited-time 1-cent Purchase)
- **活动时间**: 每月28号限量发放
- **等级要求**: 需钻石以上等级
- **时间规则**:
  - 28号10点之前: 按钮显示 "待开启"，点击提示 "活动暂未开启"
  - 28号10点-24点: 
    - 钻石以下等级: 锁住状态，按钮显示 "去升级"，点击跳转升级攻略页面
    - 钻石以上等级: 按钮显示 "立即领取"，一键完成领取
- **会员中心对接**: 和会员中心对接券发放相关

### 3. 时间段切换功能 ⭐
- **4个时间段**: 10:00、12:00、14:00、16:00
- **动态背景**: 切换时间段时，对应的背景图片 (SEL1-SEL4) 会同步切换
- **状态显示**: 每个时间段显示对应的状态文本和状态图片
- **交互反馈**: 点击时间段可切换，选中的时间段会高亮显示
- **全等级支持**: 所有等级（diamond、starshine、starlight、gold、platinum）都支持时间段切换
- **背景同步**: 只有当前选中的时间段背景图片会显示，其他时间段背景图片会隐藏
- **数据统一**: 时间段状态与 `stockBatchList` 参数完全统一，包含：
  - `stockBatchId`: 库存批次ID
  - `startTime`: 开始时间
  - `endTime`: 结束时间
  - `stockUsage`: 已使用库存
  - `stockLimit`: 库存限制
  - `received`: 是否已领取
  - `status`: 状态（locked/unlocked/partial/purchased）

## API 接口

组件使用 `monthGift.js` 中定义的接口函数，通过 `request.js` 模块进行 HTTP 请求。

### 1. monthlyGiftIndex - 获取月礼包首页数据
```javascript
// 引入
import { monthlyGiftIndex } from '../../api/monthGift.js'

// 调用
const result = await monthlyGiftIndex({ level: 'diamond' })

// 响应
{
  "success": true,
  "data": {
    "batchCoupon": [
      {
        "equityCouponInfoId": 1,        // 权益优惠券信息ID
        "couponId": 1,                  // 优惠券ID
        "couponName": "限时1分购券",     // 优惠券名称
        "equityIcon": "icon_url",       // 权益图片
        "gradeAllow": true,             // 等级是否允许 true允许
        "startTime": "2024-10-23 00:00:00", // 起始时间
        "endTime": "2024-10-23 23:59:59",   // 结束时间
        "stockUsage": 50,               // 已使用库存数量
        "stockLimit": 100,              // 可使用库存总数量
        "received": false,              // 是否已领取
        "stockBatchList": [
          {
            "stockBatchId": 1,          // 批次ID
            "startTime": "2024-10-28 10:00:00", // 生效时间
            "status": 1                 // 状态 0-待开始 1-抢券中 2-已结束
          }
        ]
      }
    ],
    "equityCouponList": [
      {
        "equityCouponInfoId": 1,        // 权益优惠券信息ID
        "couponId": 1,                  // 优惠券ID
        "couponName": "精选好券",       // 优惠券名称
        "couponSystem": "BMP",          // 优惠券系统
        "equityIcon": "icon_url",       // 权益图片
        "receiveStatus": 0,             // 优惠券状态 0-未领取 1-已领取
        "receiveTime": null             // 领取时间
      }
    ]
  }
}
```

### 2. monthlyGiftDrawOne - 领取精选好券
```javascript
// 引入
import { monthlyGiftDrawOne } from '../../api/monthGift.js'

// 调用
const result = await monthlyGiftDrawOne({ 
  stockBatchId: 'batch_123',
  equityCouponInfoId: 'coupon_123'
})

// 响应
{
  "success": true,
  "message": "领取成功",
  "data": {
    "couponId": "123",
    "couponName": "精选好券",
    "resourceId": "resource_123",
    "receiveResult": 1  // 领取结果 0-失败 1-成功
  }
}
```

### 3. monthlyGiftDrawAll - 领取限时1分购
```javascript
// 引入
import { monthlyGiftDrawAll } from '../../api/monthGift.js'

// 调用
const result = await monthlyGiftDrawAll()

// 响应
{
  "success": true,
  "message": "领取成功",
  "data": {
    "status": 1,
    "receivedNormalCouponList": [
      {
        "couponId": "123",
        "equityCouponInfoId": "456",
        "couponName": "限时1分购券",
        "equityIcon": "icon_url",
        "resourceId": "resource_789"
      }
    ]
  }
}
```

### 4. 数据结构说明
时间段状态与 `stockBatchList` 完全统一，根据新的接口返回结构：

```javascript
// 组件状态结构
state: {
  // 精选好券状态 - 根据 equityCouponList 动态生成
  couponStatus: {
    diamond: 'unlocked',    // 'locked', 'unlocked', 'used'
    starshine: 'unlocked',
    starlight: 'unlocked',
    gold: 'unlocked',
    platinum: 'unlocked'
  },
  
  // 限时1分购状态 - 根据 batchCoupon 动态生成
  buyStatus: {
    diamond: 'locked',      // 'locked', 'unlocked', 'purchased'
    starshine: 'locked',
    starlight: 'locked',
    gold: 'locked',
    platinum: 'locked'
  },
  
  // 时间段状态 - 根据 batchCoupon.stockBatchList 动态生成
  timeSlots: [
    {
      time: '10:00',                    // 时间段显示
      status: 'unlocked',               // 状态：locked/unlocked/ended/purchased
      stockBatchId: 'batch_1',          // 库存批次ID
      startTime: '2024-10-28 10:00:00', // 生效时间
      endTime: '2024-10-28 11:59:59',   // 结束时间
      stockUsage: 50,                   // 已使用库存
      stockLimit: 100,                  // 库存限制
      received: false                   // 是否已领取
    }
    // ... 其他3个时间段
  ],
  
  // 原始数据
  batchCoupon: null,        // batchCoupon 数据
  equityCouponList: []      // equityCouponList 数据
}

// 状态映射关系
// stockBatchList.status -> 时间段status
// 0 (待开始) -> 'locked'      // 锁定状态
// 1 (抢券中) -> 'unlocked'    // 可领取
// 2 (已结束) -> 'ended'       // 结束状态
```

## 使用方法

### 基础使用
```vue
<template>
  <MonthGift current-level="diamond" />
</template>

<script setup>
import MonthGift from '@/components/pointRightsCom/monthGift.vue'
</script>
```

### 动态切换等级
```vue
<template>
  <MonthGift :current-level="currentUserLevel" />
</template>

<script setup>
import { ref } from 'vue'
import MonthGift from '@/components/pointRightsCom/monthGift.vue'

const currentUserLevel = ref('diamond')

const setUserLevel = (level) => {
  currentUserLevel.value = level
}
</script>
```

### 接口引入
组件内部已经引入了所需的接口函数：

```javascript
import { monthlyGiftIndex, monthlyGiftDrawOne, monthlyGiftDrawAll } from '../../api/monthGift.js'
```

这些接口函数通过 `request.js` 模块进行 HTTP 请求，确保网络请求的一致性和可维护性。

## 组件状态

### 精选好券状态
- `locked`: 锁定状态（等级不够）
- `unlocked`: 可领取状态
- `used`: 已使用状态

### 限时1分购状态
- `locked`: 锁定状态
- `unlocked`: 可领取状态
- `purchased`: 已领取状态

### 时间段状态
- `locked`: 锁定状态
- `unlocked`: 解锁状态
- `active`: 进行中状态

## 样式类名

### 精选好券按钮
- `coupon-btn-unlocked`: 可领取状态
- `coupon-btn-used`: 已使用状态
- `coupon-btn-locked`: 锁定状态

### 限时1分购按钮
- `buy-btn-unlocked`: 可领取状态
- `buy-btn-purchased`: 已领取状态
- `buy-btn-disabled`: 禁用状态
- `buy-btn-upgrade`: 升级状态

### 时间段
- `time-slot-active`: 当前选中时间段
- `time-slot-inactive`: 未选中时间段

## 业务逻辑

### 时间判断逻辑
```javascript
// 每月28号10点-24点
const isActivityTime = computed(() => {
  const now = new Date()
  const currentDate = now.getDate()
  const currentHour = now.getHours()
  
  return currentDate === 28 && currentHour >= 10 && currentHour < 24
})

// 每月28号10点之前
const isBeforeActivityTime = computed(() => {
  const now = new Date()
  const currentDate = now.getDate()
  const currentHour = now.getHours()
  
  return currentDate === 28 && currentHour < 10
})
```

### 等级权限判断
```javascript
// 钻石以上等级可以访问限时1分购
const canAccessBuy = computed(() => {
  const levelOrder = ['platinum', 'gold', 'starlight', 'starshine', 'diamond']
  const userLevelIndex = levelOrder.indexOf(props.currentLevel)
  const diamondIndex = levelOrder.indexOf('diamond')
  
  return userLevelIndex <= diamondIndex
})
```

## 注意事项

1. **图片资源**: 组件依赖 `monthGiftImages.js` 文件中的图片路径配置
2. **API 接口**: 需要后端提供对应的接口支持
3. **用户数据**: 用户等级和权益状态需要从后端获取
4. **时间同步**: 活动时间判断基于客户端时间，建议与服务器时间同步
5. **状态持久化**: 组件状态在页面刷新后会重置，需要与后端数据同步

## 测试功能

在 `testPage` 中提供了完整的测试功能：
- 用户等级切换
- 测试时间设置
- 状态重置
- 实时预览组件效果

## 更新日志

### v1.0.0
- 实现基础组件结构
- 添加精选好券功能
- 添加限时1分购功能
- 实现时间段切换功能
- 集成 API 接口调用
- 添加完整的业务逻辑
- 实现响应式状态管理
