# H5分享方法调用文档

## 概述

本文档说明H5页面如何通过微信小程序的消息机制调用小程序的分享功能，实现自定义分享内容。

## 适用场景

当H5页面需要在小程序中自定义分享内容（标题、图片、路径）时，可以通过 `postMessage` API向小程序发送分享数据。

## 前置条件

1. H5页面必须在微信小程序的 `<web-view>` 组件中加载
2. 小程序页面路径：`/pages/gameWebview/webview`
3. 需要在微信小程序环境中运行（微信开发者工具或真实微信环境）

## 调用方法

### 基本语法

```javascript
wx.miniProgram.postMessage({
  data: {
    // 分享数据对象
    shareUrl: '/pages/index/index?param=value',
    shareTitle: '分享标题',
    shareImageUrl: 'https://example.com/share-image.jpg'
  }
})
```

### 完整示例

```javascript
// H5页面代码
function shareToWechat() {
  // 检查是否在微信小程序环境中
  if (typeof wx !== 'undefined' && wx.miniProgram) {
    wx.miniProgram.postMessage({
      data: [{
        shareUrl: '/pages/index/index?activityId=123&type=game',
        shareTitle: '快来参与游戏，赢取丰厚奖品！',
        shareImageUrl: 'https://example.com/images/share.jpg'
      }]
    })
    
    // 触发分享（需要用户手动触发，如点击按钮）
    wx.miniProgram.navigateBack()
  } else {
    console.warn('当前不在微信小程序环境中')
  }
}
```

## 分享数据格式

### 数据结构

分享数据必须是一个**数组**，数组中的每个元素是一个对象，包含以下字段：

```javascript
[
  {
    shareUrl: string,        // 分享路径（小程序页面路径）
    shareTitle: string,       // 分享标题
    shareImageUrl: string     // 分享图片URL（可选）
  },
  // ... 可以包含多个分享数据对象
]
```

**重要**：小程序会使用数组的**最后一个元素**作为分享数据。

### 字段说明

| 字段名 | 类型 | 必填 | 说明 | 示例 |
|--------|------|------|------|------|
| `shareUrl` | String | 是 | 分享后跳转的小程序页面路径（支持参数） | `/pages/index/index?activityId=123` |
| `shareTitle` | String | 是 | 分享卡片显示的标题 | `快来参与活动！` |
| `shareImageUrl` | String | 否 | 分享卡片显示的图片URL（网络图片） | `https://example.com/share.jpg` |

### 字段详细说明

#### shareUrl

- **格式**：小程序页面路径，以 `/` 开头
- **支持参数**：可以在路径后添加查询参数
- **默认值**：如果未提供或为空，使用 `/pages/index/index`
- **示例**：
  - `/pages/index/index`
  - `/pages/index/index?pageType=home`
  - `/pages/gameWebview/webview?url=https://example.com/h5/page&activityId=123`

#### shareTitle

- **格式**：纯文本字符串
- **长度限制**：建议不超过20个字符（微信分享卡片显示限制）
- **示例**：`快来参与活动，赢取丰厚奖品！`

#### shareImageUrl

- **格式**：完整的网络图片URL（必须以 `http://` 或 `https://` 开头）
- **尺寸要求**：建议 500x500 像素或更高（微信会自动压缩）
- **格式要求**：支持 JPG、PNG 格式
- **大小限制**：建议不超过 200KB
- **默认值**：如果未提供或为空，使用小程序默认分享图

## 调用时机

### 推荐调用时机

1. **用户点击分享按钮时**
   ```javascript
   document.getElementById('share-btn').addEventListener('click', function() {
     wx.miniProgram.postMessage({
       data: [{
         shareUrl: '/pages/index/index?activityId=123',
         shareTitle: '分享标题',
         shareImageUrl: 'https://example.com/share.jpg'
       }]
     })
   })
   ```

2. **页面加载完成后**
   ```javascript
   window.addEventListener('load', function() {
     // 设置默认分享数据
     wx.miniProgram.postMessage({
       data: [{
         shareUrl: window.location.href,
         shareTitle: document.title,
         shareImageUrl: 'https://example.com/default-share.jpg'
       }]
     })
   })
   ```

3. **动态更新分享内容**
   ```javascript
   function updateShareContent(activityId, activityName) {
     wx.miniProgram.postMessage({
       data: [{
         shareUrl: `/pages/index/index?activityId=${activityId}`,
         shareTitle: `快来参与${activityName}！`,
         shareImageUrl: `https://example.com/activity/${activityId}/share.jpg`
       }]
     })
   }
   ```

## 完整示例代码

### 示例1：基础分享功能

```html
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>H5分享示例</title>
</head>
<body>
  <button id="shareBtn">分享给好友</button>
  
  <script>
    // 检查是否在微信小程序环境中
    function isInMiniProgram() {
      return typeof wx !== 'undefined' && wx.miniProgram
    }
    
    // 设置分享内容
    function setShareContent() {
      if (!isInMiniProgram()) {
        console.warn('当前不在微信小程序环境中')
        return
      }
      
      wx.miniProgram.postMessage({
        data: [{
          shareUrl: '/pages/index/index?source=h5&activityId=123',
          shareTitle: '快来参与活动，赢取丰厚奖品！',
          shareImageUrl: 'https://example.com/images/share-activity.jpg'
        }]
      })
      
      console.log('分享内容已设置')
    }
    
    // 页面加载时设置默认分享内容
    window.addEventListener('load', function() {
      setShareContent()
    })
    
    // 点击分享按钮
    document.getElementById('shareBtn').addEventListener('click', function() {
      setShareContent()
      alert('分享内容已更新，请点击右上角分享按钮')
    })
  </script>
</body>
</html>
```

### 示例2：动态更新分享内容

```javascript
// 根据活动状态动态更新分享内容
function updateShareByActivity(activity) {
  if (!isInMiniProgram()) {
    return
  }
  
  const shareData = {
    shareUrl: `/pages/index/index?activityId=${activity.id}&type=${activity.type}`,
    shareTitle: activity.shareTitle || `快来参与${activity.name}！`,
    shareImageUrl: activity.shareImage || 'https://example.com/default-share.jpg'
  }
  
  wx.miniProgram.postMessage({
    data: [shareData]
  })
}

// 获取活动信息后更新分享
fetch('/api/activity/123')
  .then(res => res.json())
  .then(activity => {
    updateShareByActivity(activity)
  })
```

### 示例3：多个分享数据（使用最后一个）

```javascript
// 小程序会使用数组最后一个元素
wx.miniProgram.postMessage({
  data: [
    {
      shareUrl: '/pages/index/index',
      shareTitle: '第一个分享（不会被使用）',
      shareImageUrl: 'https://example.com/image1.jpg'
    },
    {
      shareUrl: '/pages/index/index?activityId=123',
      shareTitle: '第二个分享（会被使用）',
      shareImageUrl: 'https://example.com/image2.jpg'
    }
  ]
})
```

## 小程序端处理

小程序端会自动接收H5页面发送的分享数据，并在用户触发分享时使用。

### 分享类型

小程序支持两种分享方式：

1. **分享给好友**（`onShareAppMessage`）
2. **分享到朋友圈**（`onShareTimeline`）

两种分享方式都会使用相同的分享数据。

## 注意事项

### 1. 数据格式要求

- 分享数据必须是**数组格式**
- 数组最后一个元素会被使用
- 如果数组为空或格式错误，会使用默认分享内容

### 2. 调用时机限制

- `postMessage` 必须在用户交互（如点击）后调用才能生效
- 不能在页面加载时立即调用（可能被微信拦截）

### 3. 环境检测

建议在调用前检测是否在微信小程序环境中：

```javascript
function isInMiniProgram() {
  return typeof wx !== 'undefined' && wx.miniProgram
}

if (isInMiniProgram()) {
  // 调用 postMessage
} else {
  console.warn('当前不在微信小程序环境中')
}
```

### 4. 分享图片要求

- 必须是网络图片（不能是本地路径）
- 建议尺寸：500x500 像素或更高
- 建议格式：JPG、PNG
- 建议大小：不超过 200KB

### 5. 分享路径

- 必须以 `/` 开头
- 可以是小程序内任意页面路径
- 支持查询参数
- 路径必须在 `pages.json` 中配置

### 6. 调试方法

在微信开发者工具中：
1. 打开调试器
2. 查看 Console 输出
3. 检查 `postMessage` 是否成功调用
4. 点击右上角分享按钮测试分享功能

## 常见问题

### Q1: 分享数据发送后没有生效？

A: 检查以下几点：
- 数据格式是否正确（必须是数组）
- 是否在用户交互后调用（不能直接调用）
- 是否在微信小程序环境中
- 分享数据字段是否完整

### Q2: 如何动态更新分享内容？

A: 可以在需要的时候多次调用 `postMessage`，每次调用都会更新分享内容。

### Q3: 分享图片显示不出来？

A: 检查：
- 图片URL是否可访问（必须是网络图片）
- 图片格式是否正确（JPG、PNG）
- 图片大小是否合适（建议不超过200KB）

### Q4: 可以同时设置多个分享数据吗？

A: 可以，但小程序只会使用数组的最后一个元素。

### Q5: 分享路径可以跳转到其他小程序吗？

A: 不可以，`shareUrl` 只能是小程序内的页面路径，不能跳转到其他小程序或H5页面。

## 最佳实践

1. **及时更新分享内容**：根据页面状态动态更新分享内容
2. **提供默认分享**：页面加载时设置默认分享内容
3. **优化分享图片**：使用高质量、合适尺寸的分享图片
4. **测试分享功能**：在不同场景下测试分享功能是否正常
5. **错误处理**：添加环境检测和错误处理逻辑

## 更新日志

- 2025-01-XX: 初始版本，支持基础分享功能
- 2025-01-XX: 添加分享到朋友圈支持

