// 跳转类型枚举
export const JumpType = {
  INNER: 1,   // 内部小程序页面跳转
  MINI: 2,    // 其他小程序页面跳转
  H5: 3       // https 网络连接跳转
}

/**
 * 通用跳转方法
 * @param {Object} options
 * @param {number} options.type 跳转类型（1-内部，2-小程序，3-H5）
 * @param {string} options.url 跳转地址（页面路径/小程序路径/https链接）
 * @param {Object} [options.extra] 额外参数（如 appId、extraData 等）
 */
export function jump({ type, url, extra = {} }) {
  switch (type) {
    case JumpType.INNER:
      // 内部小程序页面跳转
      uni.navigateTo({
        url,
        ...extra
      })
      break;
    case JumpType.MINI:
      // 跳转到其他小程序页面
      uni.navigateToMiniProgram({
        appId: extra.appId || '',
        path: url,
        extraData: extra.extraData || {},
        envVersion: extra.envVersion || 'release',
        success: extra.success,
        fail: extra.fail
      })
      break;
    case JumpType.H5:
      // 跳转到 https 网络链接
      if (/^https?:\/\//.test(url)) {
        // uni-app 推荐用 webview 页面承载 H5
        uni.navigateTo({
          url: `/pages/webview/webview?url=${encodeURIComponent(url)}`
        })
      } else {
        console.error('H5 跳转地址必须为 https 链接')
      }
      break;
    default:
      console.error('不支持的跳转类型')
  }
}
