export namespace MUtils {
    export function setButtonImage(button: eui.Button, texture: egret.Texture) {
        for (let i of button.$children) {
            if (i instanceof eui.Image) {
                i.source = texture;
                return;
            }
        }
        console.error("can not get image in the button");
    }

    const imageCache: { [key: string]: egret.Texture } = {}
    const loadImageCache: { [key: string]: egret.ImageLoader } = {}
    export function loadImage(url: string) {
        if (!url) {
            console.error("empty url");
            return;
        }

        return new Promise<egret.Texture>((resolve) => {
            if (imageCache[url]) {
                resolve(imageCache[url]);
                return;
            }

            if (loadImageCache[url]) {
                loadImageCache[url].once(egret.Event.COMPLETE, (event: egret.Event) => {
                    if (imageCache[url]) {
                        resolve(imageCache[url]);
                    } else {
                        console.error("get image cache fail");
                    }
                }, this);
            } else {
                let imgLoader: egret.ImageLoader = new egret.ImageLoader();
                imgLoader.once(egret.Event.COMPLETE, (event: egret.Event) => {
                    let loader: egret.ImageLoader = event.currentTarget;
                    let texture = new egret.Texture();
                    texture.bitmapData = loader.data;
                    imageCache[url] = texture;
                    resolve(texture);
                }, this);
                imgLoader.load(url);

                loadImageCache[url] = imgLoader;
            }
        });
    }

    /**左闭右开 */
    export function random(min: number, max: number) {
        return Math.random() * (max - min) + min;
    }

    /**左闭右开 */
    export function randomInt(min: number, max: number) {
        return Math.floor(Math.random() * (max - min) + min);
    }

    export function toRadians(degrees: number) {
        return degrees / 180 * Math.PI;
    }

    export function toDegress(radians: number) {
        return radians / Math.PI * 180;
    }

    export function degreesAngelAdd(a: number, b: number) {
        return (a + b) % 360;
    }

    export function dot(vec1: egret.Point, vec2: egret.Point): number {
        return vec1.x * vec2.x + vec1.y * vec2.y;
    }

    export function getAngle(point: egret.Point) {
        let radians = Math.acos(point.x / point.length);
        radians = point.y > 0 ? radians : -radians;
        return toDegress(radians);
    }

    /**
     * 根据角度创建单位向量
     * @param angle 角度制的角度
     */
    export function getVector(angle: number) {
        angle = toRadians(angle);
        return new egret.Point(Math.cos(angle), Math.sin(angle));
    }

    export function stringOverEllipsis(str: string, limit: number) {
        if (str.length > limit) {
            return str.substring(0, limit) + "...";
        } else {
            return str;
        }
    }

    export function getClassName(cls: any) {
        return cls["prototype"]["constructor"]["name"];
    }

    export function getInstanceClassName(instance: any) {
        return getClassName(instance["constructor"]);
    }

    export function setColorFilter(image: egret.DisplayObject, color: number) {
        // 将16进制颜色分割成rgb值
        let spliceColor = (color: number) => {
            let result = { r: -1, g: -1, b: -1 };
            result.b = color % 256;
            result.g = Math.floor((color / 256)) % 256;
            result.r = Math.floor((color / 256) / 256);
            return result;
        }
        let result = spliceColor(color);
        let colorMatrix = [
            1, 0, 0, 0, 0,
            0, 1, 0, 0, 0,
            0, 0, 1, 0, 0,
            0, 0, 0, 1, 0
        ];
        colorMatrix[0] = result.r / 255;
        colorMatrix[6] = result.g / 255;
        colorMatrix[12] = result.b / 255;
        let colorFilter = new egret.ColorMatrixFilter(colorMatrix);

        image.filters = [colorFilter];
    }
}