import { Data } from './../data/Data';
// import { ABNetManager, GTime, INetData, GDispatcher } from "duiba-tc";
import { TwLang } from "../util/TwLang";
import { DataManager } from "./DataManager";
import { NetName } from '../enum/NetName';
import { ABNetManager } from '../../tc/manager/ABNetManager';
import { INetData } from '../../tc/interface/INetData';
import { GTime } from '../../tc/util/GTime';
import { IExposureData } from '../data/common/IExposureData';
import { GDispatcher } from '../../tc/util/GDispatcher';
// import { IExposureData } from '..';

export class NetManager extends ABNetManager {
    private static instance: NetManager;

    public static get ins(): NetManager {
        if (!this.instance) {
            this.instance = new NetManager();
        }
        return this.instance;
    }

    private isInit: boolean;
    constructor() {
        super();
        if (this.isInit) {
            throw Error(TwLang.lang_001);
        }
        this.isInit = true;
    }

    /**
     * 获取用户角色信息
     * @param {number} roleActivityId 角色id
     * @param {number} type 类型
     */
    public getRole(callback: Function, roleActivityId: string, type = 1): void {
        const net: INetData = {
            name: NetName.GET_ROLE,
            uri: '/activityCommon/getRole',
            type: 'get',
            dataType: 'json',
            param: {
                id: roleActivityId,
                type: type
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 活动设置用户角色信息
     * @param {number} actId  活动id
     * @param {number} type 活动类型 0:插件活动; 1:入库活动
     * @param {string} role 角色信息，不能超过32个字符
     */
    public setRole(callback: Function, actId: number, type: number, role: string): void {
        const net: INetData = {
            name: NetName.SET_ROLE,
            uri: '/activityCommon/setRole',
            type: 'post',
            dataType: 'json',
            param: {
                id: actId,
                type: type,
                role: role
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询用户现在的剩余积分
     */
    public getCredits(callback: Function): void {
        const net: INetData = {
            name: NetName.GET_CREDITS,
            uri: '/ctool/getCredits',
            type: 'get',
            dataType: 'json',
            param: null,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 增加活动免费次数
     * @param {number} type 1游戏、2活动工具
     * @param {number} count 次数
     * @param {number} validType 如果不传该参数，代表增加的次数是该用户可以永久使用的次数；如果传该参数，代表增加的次数是仅当天可用的，该参数值只能是1
     */
    public addTimesForActivity(callback: Function, type: number, count: number, validType: number): void {
        let activityId: number;
        if (type == 1) {
            activityId = DataManager.ins.gameCfgData.gameInfo.oaId;
        } else {
            activityId = DataManager.ins.customCfgData.oaId;
        }
        const param: any = {
            timestamp: GTime.getTimestamp(),
            addCount: count,
            activityId: activityId
        };
        if (validType) {
            param.validType = validType;
        }

        const net: INetData = {
            name: NetName.ADD_TIMES,
            uri: '/activityVist/addTimesForActivity',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取集卡数据
     * @param {number} collectRuleId 集卡规则ID
     */
    public getCollectRule(callback: Function, collectRuleId: number): void {
        const net: INetData = {
            name: NetName.COLLECT_RULE,
            uri: '/plugin/collectRule/getCollectRule',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取用户集卡信息接口（返回用户获得最高奖项）
     * @param {number} collectRuleId 集卡规则ID
     */
    public getCollectGoodsInfo(callback: Function, collectRuleId: number, needExtra: boolean = true): void {
        const net: INetData = {
            name: NetName.COLLECT_GOODS_INFO,
            uri: '/plugin/collectRule/getCollectGoodsInfo',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                needExtra: needExtra
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 集卡开奖
     * @param {number} collectRuleId 集卡规则ID
     * @param {number} type 1游戏、2活动工具、其他类型可以不传，发接口actid也不用传
     */
    public openCollectGoodsPrize(callback: Function, collectRuleId: number, type?: number): void {
        let actId: number;
        let param: any;
        if (type == 1) {
            actId = DataManager.ins.gameCfgData.gameInfo.oaId;
        } else if (type == 2) {
            actId = DataManager.ins.customCfgData.oaId;
        }
        if (type) {
            param = { collectRuleId: collectRuleId, actId: actId }
        } else {
            param = { collectRuleId: collectRuleId };
        }
        const net: INetData = {
            name: NetName.OPEN_COLLECT,
            uri: '/plugin/collectRule/openCollectGoodsPrize',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------游戏-----------------------------------------

    /**
     * 游戏基础信息
     */
    public getInfo(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_INFO,
            uri: '/ngame/new/getInfo',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 夏日活动游戏基础信息
     */
    public getSummerInfo(callback: Function, rankBaseConfigId: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMERINFO,
            uri: '/summer/getUserActivityInfo',
            type: 'get',
            dataType: 'json',
            param: {
                rankBaseConfigId: rankBaseConfigId,
                isMainActivity: false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  购买鱼钩鱼线道具
     * @param callback 
     * @param operatingActivityId 活动oaid
     * @param type 道具类型 0：鱼线 1：鱼钩
     */
    public summerBuyProp(callback: Function, operatingActivityId: number, type: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_BUYPROP,
            uri: '/summer/buyProp',
            type: 'post',
            dataType: 'json',
            param: {
                operatingActivityId: operatingActivityId,
                type: type
            },
            callback: callback, hideMsg: true
        };
        this.send(net);
    }

    /**
     * 获取鱼钩鱼线数量
     * @param callback 
     */
    public summerGetToyInfo(operatingActivityId, callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_TOY_INFO,
            uri: '/summer/getToyInfo',
            type: 'get',
            dataType: 'json',
            param: { operatingActivityId: operatingActivityId },
            callback: callback
        };
        this.send(net);
    }



    /**
     * 开始游戏
     * @param {string} isAgain 是否是再来一次
     * @param {number} credits 主动要求花费多少积分玩游戏
     * @param {number} customizedType xx类型
     */
    public doStart(callback: Function, isAgain: boolean, credits?: number, customizedType?: number): void {

        let addUrl = '';
        if (isAgain) {
            addUrl += '?dpm=' + DataManager.ins.gameGetSubmitResultData.againExposure.dpm;
        }

        const param: any = {
            id: DataManager.ins.gameCfgData.gameInfo.gameId,
            oaId: DataManager.ins.gameCfgData.gameInfo.oaId
        };

        if (credits) {
            param.credits = credits;
        }

        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_START,
            uri: '/ngapi/dostart',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            addUrl: addUrl
        };
        this.getToken(net);
    }

    /**
     * 查询开始游戏状态
     * @param {number} ticketId 订单ID
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} pollingCount 最大轮询次数
     * @param {number} customizedType
     */
    public getStartStatus(callback: Function, ticketId: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            ticketId: ticketId
        };
        if (customizedType) {
            param.customizedType = customizedType
        }

        const net: INetData = {
            name: NetName.GAME_START_STATUS,
            uri: '/ngapi/getStartStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    public getSummerOrderStatus(callback: Function, orderNum: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderNum: orderNum
        };

        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_ORDER_STATUS,
            uri: '/summer/getOrderStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }
    /**
     * 复活
     */
    public doRevive(callback: Function, collectRuleId: number, itemId: number): void {

        const net: INetData = {
            name: NetName.GAME_REVIVE,
            uri: '/summer/doRevive',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };
        this.getToken(net);
    }
    /**
     * 获得复活卡数量
     */
    public getReviveCardNum(callback: Function, collectRuleId: number, itemId: number): void {
        const net: INetData = {
            name: NetName.GAME_REVIVE_STATUS,
            uri: '/summer/getReviveCardNum',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };

        this.getToken(net);
    }
    /**
     * 猜扑克
     * @param {number} pluginId 插件ID
     * @param {number} ticketId 订单ID
     * @param {number} prizeId 集卡规则ID
     * @param {number} betting 下注方式 0:大 1:小 2:红 3:黑
     */
    public guessPoker(callback: Function, plugId: number, ticketId: number, prizeId: number, betting: number): void {
        const net: INetData = {
            name: NetName.GAME_GUESS_POKER,
            uri: '/ngapi/guessPoker',
            type: 'post',
            dataType: 'json',
            param: {
                pluginId: plugId,
                ticketId: ticketId,
                prizeId: prizeId,
                betting: betting
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 游戏阶段性数据提交
     * @param ticketId 
     * @param dynamicData 阶段性交互数据
     */
    public datapash(callback: Function, ticketId: number, dynamicData: string): void {
        const net: INetData = {
            name: NetName.GAME_DATA_PASH,
            uri: '/ngame/new/datapash',
            type: 'post',
            dataType: 'json',
            param: {
                ticketId: ticketId,
                gameId: DataManager.ins.gameCfgData.gameInfo.id,
                dynamicData: dynamicData
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 提交游戏数据
     *
     * @param {number} ticketId 订单ID
     * @param {number} score 分数
     * @param {any} gameData 防作弊信息
     * @param {string} submitToken
     * @param {string} dynamicData 行为数据
     * @param {boolean} checkScore 是否校验分数
     * @param {number} customizedType
     */
    public gameSubmit(callback: Function,
        ticketId: number,
        score: number,
        gameData: any,
        submitToken: string,
        dynamicData: string,
        checkScore?: boolean,
        customizedType?: number): void {
        const sign = this.createSgin(ticketId, score, gameData, submitToken);
        const param: any = {
            ticketId: ticketId,
            score: score,
            gameData: gameData,
            sgin: sign,
            dynamicData: dynamicData
        };
        if (checkScore) {
            param.checkScore = checkScore;
        }
        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_SUBMIT,
            uri: '/ngame/new/submit',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 生成签名
     * @param {number} ticketId
     * @param {number} score
     * @param {any} gameData
     * @param {string} submitToken
     * @returns {string} 签名
     */
    private createSgin(ticketId: number, score: number, gameData: any, submitToken: string): string {
        return window['duiba_md5'](ticketId + '' + score + '' + gameData + '' + submitToken);
    }

    /**
     * 游戏结束获取奖品数据
     * @param {number} orderId 订单ID 
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} 最大轮询次数
     */
    public getSubmitResult(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.GAME_SUBMIT_STATUS,
            uri: '/ngame/new/getSubmitResult',
            type: 'get',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 查看中奖名单
     * @param {boolean} showName 是否返回昵称
     */
    public winRanks(callback: Function, showName = false): void {
        const net: INetData = {
            name: NetName.GAME_RANKS,
            uri: '/ngapi/winranks',
            type: 'post',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                showName: showName
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  获取奖项
     */
    public getOptions(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_OPTIONS,
            uri: '/ngapi/getOptions',
            type: 'post',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  获取规则
     */
    public getRule(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_RULE,
            uri: '/ngapi/getRule',
            type: 'post',
            dataType: 'html',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 实时排行榜
     * @param {number} type 0总排行榜 1今日排行榜 2 多游戏总排行榜 3 昨日排行榜
     * @param {number} count 返回榜单长度 最大50
     */
    public realtimerank(callback: Function, type: number, count = 50): void {
        const net: INetData = {
            name: NetName.GAME_REAL_TIME_RANK,
            uri: '/ngapi/realtimerank',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                count: count,
            },
            callback: callback,
            addUrl: '/' + type
        };
        this.send(net);
    }

    /**
     * 获取成长值
     * @param {number} gametotalid 游戏中心榜单ID
     */
    public getUserTotalScore(callback: Function, gametotalid: number): void {
        const net: INetData = {
            name: NetName.GAME_TOTAL_SCORE,
            uri: '/ngapi/getUserTotalScore',
            type: 'get',
            dataType: 'json',
            param: {
                id: gametotalid
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------插件-----------------------------------------

    /**
     * 插件抽奖
     * @param {number} activityId 插件ID
     */
    public doJoinPlugDraw(callback: Function, activityId: number, deductCredits?: boolean): void {
        const net: INetData = {
            name: NetName.PLUG_DO_JOIN,
            uri: '/activityPlugDrawInfo/doJoinPlugdraw',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: activityId,
                deductCredits: deductCredits ? deductCredits : false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询插件抽奖订单
     * @param {number} orderId 订单ID
     * @param {number} prizeLevel 开奖等级，暂未开放
     */
    public getPlugOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5, prizeLevel?: number): void {
        const net: INetData = {
            name: NetName.PLUG_ORDER_STATUS,
            uri: '/plugin/getOrderStatus',
            type: 'get',
            dataType: 'json',
            hideMsg: true,
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 查询插件信息
     * @param {string} orderId 订单ID
     */
    public getPrizeInfo(callback: Function, plugId: number): void {
        const net: INetData = {
            name: NetName.PLUG_PRIZE_INFO,
            uri: '/activityPlugDrawInfo/getPrizeInfo',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: plugId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件-批量查询奖项信息
     * @param {string} ids 插件ID 用，间隔
     */
    public optionInfo(callback: Function, ids: string): void {
        const net: INetData = {
            name: NetName.PLUG_OPTION_INFO,
            uri: '/activityPlugin/optionInfo',
            type: 'get',
            dataType: 'json',
            param: {
                ids: ids
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件校验解锁纪录
     * @param {number} id 插件ID
     */
    public unblocking(callback: Function, id: number): void {
        const net: INetData = {
            name: NetName.PLUG_UNBLOCKING,
            uri: 'activityPlugin/unblocking',
            type: 'post',
            dataType: 'json',
            param: {
                id: id
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------自定义活动工具-----------------------------------------

    /**
     * 获取基础信息
     */
    public ajaxElement(callback: Function): void {
        const net: INetData = {
            name: NetName.CUSTOM_ELEMENT,
            uri: window['CFG'].getElement,
            type: 'post',
            dataType: 'json',
            param: {
                hdType: DataManager.ins.customCfgData.hdType,
                hdToolId: DataManager.ins.customCfgData.hdToolId,
                actId: DataManager.ins.customCfgData.actId,
                preview: DataManager.ins.customCfgData.preview
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询闯关游戏 配置信息
     * @param {number} throughId
     */
    public ajaxThroughInfo(callback: Function, throughId = 1): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_INFO,
            uri: window['CFG'].ajaxThroughInfo,
            type: 'post',
            dataType: 'json',
            param: {
                hdtoolId: DataManager.ins.customCfgData.hdToolId,
                throughId: 1
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 活动工具抽奖
     * @param {number} collectRuleId 集卡规则ID，活动与集卡规则关联才能获得发卡资格
     */
    public doJoin(callback: Function, collectRuleId?: number): void {
        const param: any = {
            actId: DataManager.ins.customCfgData.actId,
            oaId: DataManager.ins.customCfgData.oaId
        };
        if (collectRuleId) {
            param.collectRuleId = collectRuleId;
        }
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_INFO,
            uri: window['CFG'].doJoin,
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.getToken(net);
    }

    /**
     * 活动工具查询订单结果
     * @param orderId
     */
    public getCustomOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_STATUS,
            uri: window['CFG'].quireOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 获取抽奖前置信息
     * @param {string} orderId 订单ID
     */
    public getOrderInfo(callback: Function, orderId: number, showMsg = true): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_INFO,
            uri: window['CFG'].gameGetOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 前置开奖提交
     * @param {string} orderId 订单ID
     * @param {string} facePrice 前置开奖所需分值
     */
    public beforSubmit(callback: Function, orderId: number, facePrice: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_BEFOR_SUBMIT,
            uri: window['CFG'].gameSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId,
                facePrice: facePrice
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 答题提交
     */
    public questionSubmit(): void {

    }

    /**
     * 闯关游戏提交
     */
    public throughSubmit(callback: Function, orderId: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_SUBMIT,
            uri: window['CFG'].throughSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------养成-----------------------------------------

    /**
     * 宠物领养
     * @param {number} activityId 宠物养成活动ID
     * @param {string} petName 宠物名称
     */
    public petAdopte(callback: Function, activityId: number, petName: string): void {
        const net: INetData = {
            name: NetName.PET_ADOPTE,
            uri: '/signpet/adopte',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId,
                petName: petName
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 宠物状态刷新
     * @param {number} petId 宠物ID
     */
    public getPetStatus(callback: Function, petId: number, activityId: number): void {
        const net: INetData = {
            name: NetName.PET_STATUS,
            uri: '/signpet/status',
            type: 'get',
            dataType: 'json',
            param: {
                petId: petId,
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 宠物喂食
     * @param {number} petId 宠物ID
     * @param {number} feedNum 喂食数量
     */
    public petFeed(callback: Function, petId: number, feedNum: number, activityId: number): void {
        const net: INetData = {
            name: NetName.PET_FEED,
            uri: '/signpet/feed',
            type: 'post',
            dataType: 'json',
            param: {
                petId: petId,
                feedNum: feedNum,
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取宠物信息
     * @param callback 
     * @param activityId 
     */
    public getPetInfo(callback: Function, activityId: number): void {
        const net: INetData = {
            name: NetName.PET_INFO,
            uri: '/signpet/getPetInfo',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到信息查询
     * @param {Function} callback
     * @param {number} signActivityId 签到活动ID
     */
    public getSignInfo(callback: Function, signActivityId: number): void {
        const net: INetData = {
            name: NetName.SIGN_INFO,
            uri: '/signactivity/getSignInfo',
            type: 'post',
            dataType: 'json',
            param: {
                signActivityId: signActivityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到接口
     * @param {Function} callback
     * @param {number} signActivityId 签到活动ID
     * @param {number} activityId 插件活动ID，用于加抽奖次数，不传则使用签到身上配置的插件活动ID，否则不加抽奖次数
     */
    public doSign(callback: Function, signActivityId: number, activityId?: number): void {
        var data: any;
        if (activityId) {
            data = {
                id: signActivityId,
                activityId: activityId
            };
        } else {
            data = {
                id: signActivityId
            };
        }
        const net: INetData = {
            name: NetName.SIGN_DO_SIGN,
            uri: '/signactivity/doSign',
            type: 'post',
            dataType: 'json',
            param: data,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到结果查询
     * @param callback 
     * @param logId 	签到流水ID
     */
    public getSignResult(callback: Function, logId: number, pollingCheck: Function, pollingCount = 10): void {
        var data: any = {
            logId: logId
        };

        const net: INetData = {
            name: NetName.GET_SIGN_RESULT,
            uri: '/signactivity/getSignResult',
            type: 'post',
            dataType: 'json',
            param: data,
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 道具展示接口
     * @param callback 
     * @param {number} actId 宠物活动ID
     */
    public getToys(callback: Function, actId: number): void {
        const net: INetData = {
            name: NetName.PET_TOYS,
            uri: '/signpet/addition/getToys',
            type: 'get',
            dataType: 'json',
            param: {
                actId: actId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 道具兑换接口
     * @param callback 
     * @param {number} toyId 道具id
     * @param {number} credits 兑换所需积分
     */
    public toyExchange(callback: Function, toyId: number, credits: number): void {
        const net: INetData = {
            name: NetName.PET_TOY_EXCHANGE,
            uri: '/signpet/addition/toyExchange',
            type: 'post',
            dataType: 'json',
            param: {
                toyId: toyId,
                credits: credits
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 道具使用接口
     * @param callback 
     * @param {string} identifier 道具唯一标识
     * @param {number} petId 宠物id
     */
    public toyUse(callback: Function, identifier: string, petId: number): void {
        const net: INetData = {
            name: NetName.PET_TOY_USE,
            uri: '/signpet/addition/toyUse',
            type: 'post',
            dataType: 'json',
            param: {
                identifier: identifier,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 收取礼物接口
     * @param callback 
     * @param {number} giftId 礼物id
     * @param {string} giftName 礼物名称
     * @param {string} giftDesc 礼物描述
     * @param {number} giftType 礼物类型 1：用户自定义， 2：食物
     * @param {string} giftLink 礼物链接
     * @param {number} petId 宠物id
     */
    public collect(callback: Function,
        giftId: number,
        giftNum: number,
        giftName: string,
        giftDesc: string,
        giftType: number,
        giftLink: string,
        petId: number): void {
        const net: INetData = {
            name: NetName.PET_COLLECT,
            uri: '/signpet/addition/collect',
            type: 'post',
            dataType: 'json',
            param: {
                giftId: giftId,
                giftNum: giftNum,
                giftName: giftName,
                giftDesc: giftDesc,
                giftType: giftType,
                giftLink: giftLink,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 群内喂食排行榜
     * @param callback 
     * @param {number} petId 宠物id
     * @param {number} topNum 排行榜显示top个数
     */
    public getRankList(callback: Function, petId: number, topNum: number, pollingCheck: Function, pollingCount?: number): void {
        const net: INetData = {
            name: NetName.PET_GET_RANK_LIST,
            uri: '/signpet/getRankList',
            type: 'get',
            dataType: 'json',
            param: {
                petId: petId,
                topNum: topNum,
                pollingCheck: pollingCheck,
                pollingCount: pollingCount
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询待领取粮食
     * @param callback 
     */
    public getFoodPiles(callback: Function): void {
        const net: INetData = {
            name: NetName.PET_GET_FOOD_PILES,
            uri: '/signpet/getFoodPiles',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 待领取粮食收取
     * @param callback 
     * @param {number} id 待领取粮食主键ID
     * @param {number} activityId 活动id
     */
    public collectFood(callback: Function, id: number, activityId: number, petId: number): void {
        const net: INetData = {
            name: NetName.PET_COLLECT_FOOD,
            uri: '/signpet/collectFood',
            type: 'post',
            dataType: 'json',
            param: {
                id: id,
                activityId: activityId,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------------------------------外来宠物养成--------------------------------------------------------------------------
    /**
     * 活动主信息接口
     * @param callback 
     * @param activityId 
     */
    public getHomeInfo(callback: Function, activityId: number): void {
        const net: INetData = {
            name: NetName.PETS_GET_HOME_INFO,
            uri: '/sign/wander/getHomeInfo',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 商店道具接口
     * @param callback 
     * @param actId 
     * @param actType 	2-日历 4-养成 5-契约 6-外来宠物
     * @param toyType   1-状态 2-装饰 3-功能 4-食物 5-玩具
     */
    public getActToys(callback: Function, actId: number, actType: number, toyType: number, petId: number): void {
        const net: INetData = {
            name: NetName.PETS_GET_ACT_TOYS,
            uri: '/sign/addition/getActToys',
            type: 'get',
            dataType: 'json',
            param: {
                actId: actId,
                actType: actType,
                toyType: toyType,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 外来宠物养成-道具兑换接口
     * @param callback 
     * @param {number} toyId 道具id
     * @param {number} count 购买数量
     */
    public toyExchanges(callback: Function, toyId: number, count?: number): void {
        var param: any = {}
        if (count) {
            param = {
                toyId: toyId,
                count: count
            };
        } else {
            param = {
                toyId: toyId
            };
        }

        const net: INetData = {
            name: NetName.PETS_TOY_EXCHANGE,
            uri: '/sign/addition/toyExchange',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 道具使用接口
     * @param callback 
     * @param {number} actId 活动id
     * @param {any[]} toyList 使用的道具列表 [{identifier:'道具标识', position:'投放位置'}, {identifier:'道具标识', position:'投放位置'}]
     */
    public useToy(callback: Function, actId: number, toyList: any[]): void {
        const net: INetData = {
            name: NetName.PETS_USE_TOYS,
            uri: '/sign/addition/useToy',
            type: 'post',
            dataType: 'json',
            param: {
                actId: actId,
                toyList: JSON.stringify(toyList)
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 道具使用携程
     * @param callback 
     * @param activityId 
     * @param identifier 
     * @param petId 
     */
    public toyUse_xiecheng(callback: Function, activityId: number, identifier: string, petId: number): void {
        const net: INetData = {
            name: NetName.PETS_USE_TOYS_XCH,
            uri: '/sign/addition/toyUse',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId,
                identifier: identifier,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 已购买的道具接口
     * @param callback 
     * @param petId     宠物id 没有传0
     * @param toyType   1-状态 2-装饰 3-功能 4-食物 5-玩具
     */
    public getUserToys(callback: Function, petId: number, toyType: number): void {
        const net: INetData = {
            name: NetName.PETS_GET_USER_TOYS,
            uri: '/sign/addition/getUserToys',
            type: 'get',
            dataType: 'json',
            param: {
                petId: petId,
                toyType: toyType
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 粮食收取
     * @param callback 
     * @param actId     活动id
     * @param ids       收取粮食堆的主键id列表[1,2,3,4]
     */
    public batchollectFood(callback: Function, actId: number, ids: number[]): void {
        const net: INetData = {
            name: NetName.PETS_BATCHOLLECT_FOOD,
            uri: '/sign/addition/batchollectFood',
            type: 'post',
            dataType: 'json',
            param: {
                ids: JSON.stringify(ids),
                actId: actId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 来访统计
     * @param callback 
     * @param actId 
     */
    public visitStatistics(callback: Function, actId: number): void {
        const net: INetData = {
            name: NetName.PETS_VISI_STATISTICS,
            uri: '/sign/wander/visitStatistics',
            type: 'get',
            dataType: 'json',
            param: {
                actId: actId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 动态信息列表
     * @param callback 
     * @param actId         活动id
     * @param showNum       显示条数
     */
    public getVisitInfo(callback: Function, actId: number, showNum: number): void {
        const net: INetData = {
            name: NetName.PETS_GET_VISIT_INFO,
            uri: '/sign/wander/getVisitInfo',
            type: 'get',
            dataType: 'json',
            param: {
                showNum: showNum,
                actId: actId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 到访记录详情
     * @param callback 
     * @param actId ：活动id
     * @param identifier ：宠物唯一标识
     */
    public visitDetail(callback: Function, actId: number, identifier: string): void {
        const net: INetData = {
            name: NetName.PETS_VISIT_DETAIL,
            uri: '/sign/wander/visitDetail',
            type: 'get',
            dataType: 'json',
            param: {
                actId: actId,
                identifier: identifier
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 礼物收取
     * @param callback 
     * @param recordId 礼物记录ID
     */
    public collectGiftRecord(recordId: number, callback: Function): void {
        const net: INetData = {
            name: NetName.PETS_COLLECT_GIFT_RECORD,
            uri: '/sign/wander/collectGiftRecord',
            type: 'post',
            dataType: 'json',
            param: {
                recordId: recordId
            },
            callback: callback
        };
        this.send(net);
    }

    //-----------------------------------------------------------------订制接口-----------------------------------------------------------------

    // /**
    //  * 收入明细
    //  * @param callback 
    //  * @param activityId 活动ID
    //  */
    // public getProfitDetail(callback: Function, activityId: number): void {
    //     const net: INetData = {
    //         name: NetName.GET_PROFIT_DETAIL,
    //         uri: '/sign/custom/getProfitDetail',
    //         type: 'get',
    //         dataType: 'json',
    //         param: {
    //             activityId: activityId
    //         },
    //         callback: callback
    //     };
    //     this.send(net);
    // }

    /**
     * 宠物出行
     * @param callback 
     * @param activityId 活动ID
     * @param petId 宠物ID
     * @param mapType 地图类型，1：中国，2：亚洲，3：欧洲
     */
    public petGoout(callback: Function, activityId: number, petId: number, mapType: number): void {
        const net: INetData = {
            name: NetName.PET_GOOUT,
            uri: '/signpet/ctrip/goout',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId,
                petId: petId,
                mapType: mapType
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 宠物出行结果查询
     * @param callback 
     * @param activityId 活动ID
     * @param petId 宠物ID
     * @param recordId 出行ID
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} pollingCount 最大轮询次数
     */
    public petGooutResult(callback: Function, activityId: number, petId: number, recordId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.PET_GOOUT_RESULT,
            uri: '/signpet/ctrip/goout/result',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId,
                petId: petId,
                recordId: recordId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 携程宠物出行奖励查询
     * @param callback 
     * @param activityId 活动ID
     * @param petId 宠物ID
     */
    public getPetGooutReward(callback: Function, activityId: number, petId: number): void {
        const net: INetData = {
            name: NetName.PET_GOOUT_REWARD,
            uri: '/signpet/ctrip/goout/reward',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 道具兑换接口---携程
     * @param callback 
     * @param activityId 活动id
     * @param petId 宠物id
     * @param toyId 道具id
     * @param count 道具数量
     */
    public toyExchange_xc(callback: Function, activityId: number, petId: number, toyId: number, count: number): void {
        const net: INetData = {
            name: NetName.TOY_EXCHANGES_DATA_XC,
            uri: '/sign/addition/toy/exchange',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId,
                petId: petId,
                toyId: toyId,
                count: count
            },
            callback: callback
        };
        this.send(net);
    }

   
    /**
     * 主页-书旗小七
     */

    public getIndex(callback: Function,activityId,refreshReading,shareCode?){
        let param
        if(shareCode){
            param={
                activityId:activityId,
                refreshReading:refreshReading,
                shareCode:shareCode,
            }
        }else{
            param={
                activityId:activityId,
                refreshReading:refreshReading
            }
        }
        const net: INetData = {
            name: NetName.GET_INDEX,
            uri: '/customActivity/shuqi/pet/index',
            type: 'get',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

     /**
     * 主页-书旗小七-02、游戏区数据接口
     */

    public getGameZone(callback: Function,activityId){
        let param
        param={
            activityId:activityId
        }
        const net: INetData = {
            name: NetName.GET_HOME,
            uri: '/customActivity/shuqi/pet/gameZone',
            type: 'get',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 召唤狗狗-云集-汪星球游戏
     * @param callback 
     * @param level 
     * @param activityId 
     * @param location 
     */

    public getSummon(callback: Function, level: number,activityId,location){
        const net: INetData = {
            name: NetName.GET_SUMMON,
            uri: '/customActivity/yunji/dog/summon',
            type: 'post',
            dataType: 'json',
            param: {
                level:level,
                activityId:activityId,
                location:location
            },
            callback: callback
        };
        this.send(net);
    }

   
    /**
     * 变更位置-云集-汪星球游戏
     * @param callback 
     * @param form 
     * @param to 
     */

    public getLocation(callback: Function, form:number,to:number,activityId){
        const net: INetData = {
            name: NetName.GET_LOCATION,
            uri: '/customActivity/yunji/dog/location',
            type: 'post',
            dataType: 'json',
            param: {
                from:form,
                to:to,
                activityId:activityId
            },
            callback: callback
        };
        this.send(net);
    }

   
    /**
     * 变更位置-云集-汪星球游
     * @param callback 
     * @param form 
     * @param to 
     * @param currentLevel 
     * @param activityId 
     */

    public getMerge(callback: Function, form:number,to:number,activityId){
        const net: INetData = {
            name: NetName.GET_MERGE,
            uri: '/customActivity/yunji/dog/merge',
            type: 'post',
            dataType: 'json',
            param: {
                activityId:activityId,
                from:form,
                to:to
            },
            callback: callback
        };
        this.send(net);
    }

    
    /**
     * 回收狗狗-云集-汪星球游戏
     * @param callback 
     * @param location 
     * @param activityId 
     */

    public getRecycle(callback: Function,location,activityId){
        const net: INetData = {
            name: NetName.GET_RECYCLE,
            uri: '/customActivity/yunji/dog/recycle',
            type: 'post',
            dataType: 'json',
            param: {
                location:location,
                activityId:activityId
            },
            callback: callback
        };
        this.send(net);
    }


    /**
     * 获取队伍信息-云集-汪星球游戏
     */
    public getTeamInfo(callback: Function,activityId){
        const net: INetData = {
            name: NetName.GET_TEAMINFO,
            uri: '/customActivity/yunji/dog/teamInfo',
            type: 'get',
            dataType: 'json',
            param: {
                activityId:activityId
            },
            callback: callback
        };
        this.send(net);
    }

      /**
     * 获取队伍信息-云集-汪星球游戏
     */
    public getShareCode(callback: Function,activityId,activityType){
        const net: INetData = {
            name: NetName.GET_SHARECODE,
            uri: '/activtyShareCode/getShareCode',
            type: 'get',
            dataType: 'json',
            param: {
                activityId:activityId,
                activityType:activityType,
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取队伍信息-云集-汪星球游戏
     */
    public joinTeam(callback: Function,shareCode,activityId){
        const net: INetData = {
            name: NetName.JOINTEAM,
            uri: '/customActivity/yunji/dog/join',
            type: 'post',
            dataType: 'json',
            param: {
                shareCode:shareCode	,
                activityId:activityId,
            },
            callback: callback
        };
        this.send(net);
    }


    

    /**
     * 获取商店信息-云集养狗
     * @param callback 
     */
    public shopYuji(callback: Function, activityId:string):void{
        const net: INetData = {
            name: NetName.SHOP_YUNJI,
            uri: '/customActivity/yunji/dog/shop',
            type: 'get',
            dataType: 'json',
            param: {
                activityId:activityId
            },
            callback: callback
        };
        this.send(net);
    }

    

    


    

    /**
     * 查询购买道具接口（携程）
     * @param callback 
     * @param activityId 活动id
     * @param petId 宠物id
     * @param orderId 订单id
     * @param pollingCheck 
     * @param pollingCount 
     */
    public exchangeResult(callback: Function, activityId: number, petId: number, orderId: number, pollingCheck: Function, pollingCount = 10): void {
        const net: INetData = {
            name: NetName.EXCHANGE_RECSULT,
            uri: '/sign/addition/toy/exchange/result',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId,
                petId: petId,
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 获取用户积分流水明细
     * @param callback 
     * @param appId appId
     * @param uid 用户Id
     * @param orderStatus 订单状态：待确认／已完成（process／finish）
     * @param rowId 当前页
     * @param max 每页条数
     */
    public weOrder(callback: Function, appId: number, uid: string, rowId: number, max: number, orderStatus: string = "success"): void {
        const net: INetData = {
            name: NetName.WE_ORDER,
            uri: (window['weUrl'] ? window['weUrl'] : '') + '/duiba/weOrder',
            type: 'get',
            dataType: 'json',
            param: {
                appId: appId,
                uid: uid,
                rowId: rowId,
                max: max,
                orderStatus: orderStatus
            },
            callback: callback
        };
        let gTime: string = '?_=' + GTime.getTimestamp();
        let realUrl: string = net.uri;
        if (realUrl.indexOf('?') != -1) {
            gTime = '&_=' + GTime.getTimestamp();
        }

        //url加参数等特殊需求（例如再玩一次需要在dostart接口的url上加埋点）
        if (net.addUrl) {
            realUrl += net.addUrl;
        }

        window['$'].ajax({
            type: net.type,
            // url: realUrl + gTime,
            url: realUrl,
            dataType: net.dataType,
            data: net.param,
            async: true,
            xhrFields: {
                withCredentials: false
            },
            crossDomain: true,
            cache: false,
            success: (result) => {
                this.onResponse(net, result);
            },
            error: (message) => {
                this.onError(net);
            }
        });
    }



    /**
     * 获取token
     * @param net 
     */
    public getToken(net: INetData): void {
        if (window['getDuibaToken']) {
            window['getDuibaToken']((tokenObj: any) => {
                net.param.token = tokenObj.token;
                this.send(net);
            }, (key: string, messageObj: any) => {
                this.onError(net);
            });
        } else {
            this.send(net);
        }
    }

    /**
     * 查询我的奖品记录
     * @param callback 
     * @param {number} page 当前页数
     */
    public getRecord(callback: Function, page: number): void {
        const net: INetData = {
            name: NetName.GET_RECORD,
            uri: (window['recordUrl'] ? window['recordUrl'] : '') + '/Crecord/getRecord',
            type: 'get',
            dataType: 'json',
            param: {
                page: page
            },
            callback: callback
        };

        let gTime: string = '?_=' + GTime.getTimestamp();
        let realUrl: string = net.uri;
        if (realUrl.indexOf('?') != -1) {
            gTime = '&_=' + GTime.getTimestamp();
        }

        //url加参数等特殊需求（例如再玩一次需要在dostart接口的url上加埋点）
        if (net.addUrl) {
            realUrl += net.addUrl;
        }

        window['$'].ajax({
            type: net.type,
            // url: realUrl + gTime,
            url: realUrl,
            dataType: net.dataType,
            data: net.param,
            async: true,
            xhrFields: {
                withCredentials: true
            },
            crossDomain: true,
            cache: false,
            success: (result) => {
                this.onResponse(net, result);
            },
            error: (message) => {
                this.onError(net);
            }
        });
    }

    /**
     * 推啊曝光埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmshow(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmshow',
            uri: domain + 'spmshow',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 推啊点击埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmclick(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmclick',
            uri: domain + 'spmclick',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧点击埋点
     * @param {IExposureData} exposure
     */
    public clickLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'clickLog',
            uri: '/log/click',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧曝光埋点
     * @param {IExposureData} exposure
     */
    public showLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'showLog',
            uri: exposure.domain + '/exposure/standard',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
      * 消息响应
      * @param net 
      * @param result 结果
      */
    protected onResponse(net: INetData, result: any): void {
        //数据处理
        const data: Data = DataManager.ins.updateData(net.name, result, net.param);
        if (net.pollingCheck) {
            console.log('result=====', result);
            console.log('data=====', data);
        }
        //接口成功
        if (net.pollingCount && net.pollingCheck(data)) {
            net.pollingCount -= 1;
            //轮询接口特殊处理
            setTimeout(() => {
                this.send(net);
            }, 500);
        } else if (net.callback) {
            if (net.pollingCheck && !net.pollingCount && net.pollingCheck(data)) {
                net.callback(false, 'pollingFaild');
            } else {
                net.callback(data.success, data || result);
            }

        }
        if (!data.success && !net.hideMsg) {
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net, result.message || result.desc || result.msg,result);
        }
    }

    /**
     * 通讯底层错误
     * @param net 
     * @param message 
     */
    protected onError(net: INetData): void {
        if (net.callback) {
            net.callback(false);
        }

        if (!net.hideMsg) {
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net);
        }
    }
    // tslint:disable-next-line:max-file-line-count
}