import { GDispatcher } from "../../../libs/tc/util/GDispatcher";

enum STATE {
    NONE = 1,     // 无状态
    ADDSPEED, // 加速状态
    MIDSTATE, // 中间状态
    SUBSPEED, // 减速状态
}

/**
 * 大转盘
 */
export default class RotaryTable1 {
    private panel: eui.Image;

    private _endRotation: number;  // 结束时的旋转角度   
    private _maxSpeed: number;    // 旋转最大速度
    private _addSpeed: number;  // 旋转加速度
    private _speed: number; // 当前旋转速度
    private _midTime: number; // 中间状态的时间，毫秒

    private rotation: number;   // 累计旋转角度
    private _end: number;   // 结束时的累计旋转角度

    private _state: STATE = STATE.NONE;    // 状态

    // 计算帧时间
    private lastTime: number = egret.getTimer();	// 上一帧结束的时间

    /**
     * @constructor
     * @param panel 转盘图片
     */
    constructor(panel) {
        this.panel = panel;

        // 声明周期update
        egret.lifecycle.addLifecycleListener((context) => {
            context.onUpdate = () => {
                this.update();
            }
        })
    }

    private update() {
        let currentTime: number = egret.getTimer();		// 当前时间
        let dt: number = (currentTime - this.lastTime) / 1000;	// 帧时间

        if (this._state !== STATE.NONE) {

            switch (this._state) {
                case STATE.ADDSPEED:
                    this._speed += this._addSpeed * dt;
                    if (this._speed >= this._maxSpeed) {    // 如果加速到最大值则切换状态
                        this.changeState(STATE.MIDSTATE);
                    }
                    break;

                case STATE.MIDSTATE:
                    break;

                case STATE.SUBSPEED:
                    this._speed += this._addSpeed * dt;
                    if ((this.rotation + this._speed * dt) >= this._end){
                        this._speed = 0;
                    } 
                    if (this._speed <= 0) {
                        this._speed = 0;
                        this.panel.rotation = this._endRotation;
                        this.changeState(STATE.NONE);
                    }
                    break;

                default:
                    break;
            }
            console.log(this._speed)
            this.panel.rotation += this._speed * dt;
            this.rotation += this._speed * dt;
        }

        this.lastTime = egret.getTimer();	// 上一帧结束的时间
    }

    /**
     * 开始一次旋转
     * @param addSpeed - 加速度 度/秒
     * @param maxSpeed - 最大速度 度/秒
     * @param endRotation - 结束时的角度
     * @param midTime - 最大速度旋转时间 毫秒  默认:2500毫秒
     */
    public rotate(addSpeed: number, maxSpeed: number, endRotation: number, midTime: number = 1000) {
        this.rotation = this.panel.rotation;
        this._speed = 0; // 当前旋转速度
        this._addSpeed = addSpeed;  // 旋转加速度
        this._maxSpeed = maxSpeed;    // 旋转最大速度

        this._midTime = midTime; // 中间状态的时间，毫秒

        this._endRotation = endRotation;  // 结束时的旋转角度
        this.changeState(STATE.ADDSPEED);  // 切换到加速状态
    }


    /**
     * 切换状态
     * @param state 状态
     */
    private changeState(state: STATE) {
        if (state === this._state) {
            return;
        }
        this._state = state;
        switch (this._state) {
            case STATE.NONE:
                this.panel.rotation %= 360;
                this.rotation = this.panel.rotation;
                GDispatcher.dispatchEvent('rotaryComplete');
                break;

            case STATE.ADDSPEED:

                break;

            case STATE.MIDSTATE:
                setTimeout(() => {
                    this.changeState(STATE.SUBSPEED);
                }, this._midTime);
                break;

            case STATE.SUBSPEED:
                let t = Math.random() * 6 + 2;
                if(t > 3 && t < 4){
                    this._endRotation += 360 * 2;
                } else if( t >= 5 && t < 6){
                    this._endRotation += 360 * 3;
                } else if (t >= 7 && t < 8) {
                    this._endRotation += 360 * 4;
                }
                this._end = this.rotation + this._endRotation + (this.panel.rotation > 0 ? this.panel.rotation : 360 + this.panel.rotation);
                this._addSpeed = 2 * (this._end - this.rotation - this._speed * t) / (t * t);
                console.log(this._addSpeed);
                
                
                break;

            default:
                break;
        }
    }
}