import { Data } from './../data/Data';
// import { ABNetManager, GTime, INetData, GDispatcher } from "duiba-tc";
import { TwLang } from "../util/TwLang";
import { DataManager } from "./DataManager";
import { NetName } from '../enum/NetName';
import { ABNetManager } from '../../tc/manager/ABNetManager';
import { INetData } from '../../tc/interface/INetData';
import { GTime } from '../../tc/util/GTime';
import { IExposureData } from '../data/common/IExposureData';
import { GDispatcher } from '../../tc/util/GDispatcher';
// import { IExposureData } from '..';

export class NetManager extends ABNetManager {
    private static instance: NetManager;

    public static get ins(): NetManager {
        if (!this.instance) {
            this.instance = new NetManager();
        }
        return this.instance;
    }

    private isInit: boolean;
    constructor() {
        super();
        if (this.isInit) {
            throw Error(TwLang.lang_001);
        }
        this.isInit = true;
    }

    /**
     * 获取用户角色信息
     * @param {number} roleActivityId 角色id
     * @param {number} type 类型
     */
    public getRole(callback: Function, roleActivityId: string, type = 1): void {
        const net: INetData = {
            name: NetName.GET_ROLE,
            uri: '/activityCommon/getRole',
            type: 'get',
            dataType: 'json',
            param: {
                id: roleActivityId,
                type: type
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 活动设置用户角色信息
     * @param {number} actId  活动id
     * @param {number} type 活动类型 0:插件活动; 1:入库活动
     * @param {string} role 角色信息，不能超过32个字符
     */
    public setRole(callback: Function, actId: number, type: number, role: string): void {
        const net: INetData = {
            name: NetName.SET_ROLE,
            uri: '/activityCommon/setRole',
            type: 'post',
            dataType: 'json',
            param: {
                id: actId,
                type: type,
                role: role
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询用户现在的剩余积分
     */
    public getCredits(callback: Function): void {
        const net: INetData = {
            name: NetName.GET_CREDITS,
            uri: '/ctool/getCredits',
            type: 'get',
            dataType: 'json',
            param: null,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 增加活动免费次数
     * @param {number} type 1游戏、2活动工具
     * @param {number} count 次数
     * @param {number} validType 如果不传该参数，代表增加的次数是该用户可以永久使用的次数；如果传该参数，代表增加的次数是仅当天可用的，该参数值只能是1
     */
    public addTimesForActivity(callback: Function, type: number, count: number, validType: number): void {
        let activityId: number;
        if (type == 1) {
            activityId = DataManager.ins.gameCfgData.gameInfo.oaId;
        } else {
            activityId = DataManager.ins.customCfgData.oaId;
        }
        const param: any = {
            timestamp: GTime.getTimestamp(),
            addCount: count,
            activityId: activityId
        };
        if (validType) {
            param.validType = validType;
        }

        const net: INetData = {
            name: NetName.ADD_TIMES,
            uri: '/activityVist/addTimesForActivity',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取集卡数据
     * @param {number} collectRuleId 集卡规则ID
     */
    public getCollectRule(callback: Function, collectRuleId: number): void {
        const net: INetData = {
            name: NetName.COLLECT_RULE,
            uri: '/plugin/collectRule/getCollectRule',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取用户集卡信息接口（返回用户获得最高奖项）
     * @param {number} collectRuleId 集卡规则ID
     */
    public getCollectGoodsInfo(callback: Function, collectRuleId: number, needExtra: boolean = true): void {
        const net: INetData = {
            name: NetName.COLLECT_GOODS_INFO,
            uri: '/plugin/collectRule/getCollectGoodsInfo',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                needExtra: needExtra
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 集卡开奖
     * @param {number} collectRuleId 集卡规则ID
     * @param {number} type 1游戏、2活动工具、其他类型可以不传，发接口actid也不用传
     */
    public openCollectGoodsPrize(callback: Function, collectRuleId: number, type?: number): void {
        let actId: number;
        let param: any;
        if (type == 1) {
            actId = DataManager.ins.gameCfgData.gameInfo.oaId;
        } else if (type == 2) {
            actId = DataManager.ins.customCfgData.oaId;
        }
        if (type) {
            param = { collectRuleId: collectRuleId, actId: actId }
        } else {
            param = { collectRuleId: collectRuleId };
        }
        const net: INetData = {
            name: NetName.OPEN_COLLECT,
            uri: '/plugin/collectRule/openCollectGoodsPrize',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------游戏-----------------------------------------

    /**
     * 游戏基础信息
     */
    public getInfo(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_INFO,
            uri: '/ngame/new/getInfo',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 夏日活动游戏基础信息
     */
    public getSummerInfo(callback: Function, rankBaseConfigId: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMERINFO,
            uri: '/summer/getUserActivityInfo',
            type: 'get',
            dataType: 'json',
            param: {
                rankBaseConfigId: rankBaseConfigId,
                isMainActivity: false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  购买鱼钩鱼线道具
     * @param callback 
     * @param operatingActivityId 活动oaid
     * @param type 道具类型 0：鱼线 1：鱼钩
     */
    public summerBuyProp(callback: Function, operatingActivityId: number, type: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_BUYPROP,
            uri: '/summer/buyProp',
            type: 'post',
            dataType: 'json',
            param: {
                operatingActivityId: operatingActivityId,
                type: type
            },
            callback: callback, hideMsg: true
        };
        this.send(net);
    }

    /**
     * 获取鱼钩鱼线数量
     * @param callback 
     */
    public summerGetToyInfo(operatingActivityId, callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_TOY_INFO,
            uri: '/summer/getToyInfo',
            type: 'get',
            dataType: 'json',
            param: { operatingActivityId: operatingActivityId },
            callback: callback
        };
        this.send(net);
    }



    /**
     * 开始游戏
     * @param {string} isAgain 是否是再来一次
     * @param {number} credits 主动要求花费多少积分玩游戏
     * @param {number} customizedType xx类型
     */
    public doStart(callback: Function, isAgain: boolean, credits?: number, customizedType?: number): void {

        let addUrl = '';
        if (isAgain) {
            addUrl += '?dpm=' + DataManager.ins.gameGetSubmitResultData.againExposure.dpm;
        }

        const param: any = {
            id: DataManager.ins.gameCfgData.gameInfo.gameId,
            oaId: DataManager.ins.gameCfgData.gameInfo.oaId
        };

        if (credits) {
            param.credits = credits;
        }

        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_START,
            uri: '/ngapi/dostart',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            addUrl: addUrl
        };
        this.getToken(net);
    }

    /**
     * 查询开始游戏状态
     * @param {number} orderId 订单ID
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} pollingCount 最大轮询次数
     * @param {number} customizedType
     */
    public getStartStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderId: orderId
        };
        if (customizedType) {
            param.customizedType = customizedType
        }

        const net: INetData = {
            name: NetName.GAME_START_STATUS_NEW,
            uri: '/hdtool/recon/ngame/getNgameStartStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    public getSummerOrderStatus(callback: Function, orderNum: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderNum: orderNum
        };

        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_ORDER_STATUS,
            uri: '/summer/getOrderStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }
    /**
     * 复活
     */
    public doRevive(callback: Function, collectRuleId: number, itemId: number): void {

        const net: INetData = {
            name: NetName.GAME_REVIVE,
            uri: '/summer/doRevive',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };
        this.getToken(net);
    }
    /**
     * 获得复活卡数量
     */
    public getReviveCardNum(callback: Function, collectRuleId: number, itemId: number): void {
        const net: INetData = {
            name: NetName.GAME_REVIVE_STATUS,
            uri: '/summer/getReviveCardNum',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };

        this.getToken(net);
    }
    /**
     * 猜扑克
     * @param {number} pluginId 插件ID
     * @param {number} ticketId 订单ID
     * @param {number} prizeId 集卡规则ID
     * @param {number} betting 下注方式 0:大 1:小 2:红 3:黑
     */
    public guessPoker(callback: Function, plugId: number, ticketId: number, prizeId: number, betting: number): void {
        const net: INetData = {
            name: NetName.GAME_GUESS_POKER,
            uri: '/ngapi/guessPoker',
            type: 'post',
            dataType: 'json',
            param: {
                pluginId: plugId,
                ticketId: ticketId,
                prizeId: prizeId,
                betting: betting
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 游戏阶段性数据提交
     * @param ticketId 
     * @param dynamicData 阶段性交互数据
     */
    public datapash(callback: Function, ticketId: number, dynamicData: string): void {
        const net: INetData = {
            name: NetName.GAME_DATA_PASH,
            uri: '/ngame/new/datapash',
            type: 'post',
            dataType: 'json',
            param: {
                ticketId: ticketId,
                gameId: DataManager.ins.gameCfgData.gameInfo.id,
                dynamicData: dynamicData
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 提交游戏数据
     *
     * @param {number} ticketId 订单ID
     * @param {number} score 分数
     * @param {any} gameData 防作弊信息
     * @param {string} submitToken
     * @param {string} dynamicData 行为数据
     * @param {boolean} checkScore 是否校验分数
     * @param {number} customizedType
     */
    public gameSubmit(callback: Function,
        ticketId: number,
        score: number,
        gameData: any,
        submitToken: string,
        dynamicData: string,
        checkScore?: boolean,
        customizedType?: number): void {
        const sign = this.createSgin(ticketId, score, gameData, submitToken);
        const param: any = {
            ticketId: ticketId,
            score: score,
            gameData: gameData,
            sgin: sign,
            dynamicData: dynamicData
        };
        if (checkScore) {
            param.checkScore = checkScore;
        }
        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_SUBMIT,
            uri: '/ngame/new/submit',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 生成签名
     * @param {number} ticketId
     * @param {number} score
     * @param {any} gameData
     * @param {string} submitToken
     * @returns {string} 签名
     */
    private createSgin(ticketId: number, score: number, gameData: any, submitToken: string): string {
        return window['duiba_md5'](ticketId + '' + score + '' + gameData + '' + submitToken);
    }

    /**
     * 游戏结束获取奖品数据
     * @param {number} orderId 订单ID 
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} 最大轮询次数
     */
    public getSubmitResult(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.GAME_SUBMIT_STATUS,
            uri: '/ngame/new/getSubmitResult',
            type: 'get',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 查看中奖名单
     * @param {boolean} showName 是否返回昵称
     */
    public winRanks(callback: Function, showName = false): void {
        const net: INetData = {
            name: NetName.GAME_RANKS,
            uri: '/ngapi/winranks',
            type: 'post',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                showName: showName
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  获取规则
     */
    public getRule(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_RULE,
            uri: '/ngapi/getRule',
            type: 'post',
            dataType: 'html',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 实时排行榜
     * @param {number} type 0总排行榜 1今日排行榜 2 多游戏总排行榜 3 昨日排行榜
     * @param {number} count 返回榜单长度 最大50
     */
    public realtimerank(callback: Function, type: number, count = 50): void {
        const net: INetData = {
            name: NetName.GAME_REAL_TIME_RANK,
            uri: '/ngapi/realtimerank',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                count: count,
            },
            callback: callback,
            addUrl: '/' + type
        };
        this.send(net);
    }

    /**
     * 获取成长值
     * @param {number} gametotalid 游戏中心榜单ID
     */
    public getUserTotalScore(callback: Function, gametotalid: number): void {
        const net: INetData = {
            name: NetName.GAME_TOTAL_SCORE,
            uri: '/ngapi/getUserTotalScore',
            type: 'get',
            dataType: 'json',
            param: {
                id: gametotalid
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------插件-----------------------------------------

    /**
     * 插件抽奖
     * @param {number} activityId 插件ID
     */
    public doJoinPlugDraw(callback: Function, activityId: number, deductCredits?: boolean): void {
        const net: INetData = {
            name: NetName.PLUG_DO_JOIN,
            uri: '/activityPlugDrawInfo/doJoinPlugdraw',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: activityId,
                deductCredits: deductCredits ? deductCredits : false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询插件抽奖订单
     * @param {number} orderId 订单ID
     * @param {number} prizeLevel 开奖等级，暂未开放
     */
    public getPlugOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5, prizeLevel?: number): void {
        const net: INetData = {
            name: NetName.PLUG_ORDER_STATUS,
            uri: '/plugin/getOrderStatus',
            type: 'get',
            dataType: 'json',
            hideMsg: true,
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 查询插件信息
     * @param {string} orderId 订单ID
     */
    public getPrizeInfo(callback: Function, plugId: number): void {
        const net: INetData = {
            name: NetName.PLUG_PRIZE_INFO,
            uri: '/activityPlugDrawInfo/getPrizeInfo',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: plugId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件-批量查询奖项信息
     * @param {string} ids 插件ID 用，间隔
     */
    public optionInfo(callback: Function, ids: string): void {
        const net: INetData = {
            name: NetName.PLUG_OPTION_INFO,
            uri: '/activityPlugin/optionInfo',
            type: 'get',
            dataType: 'json',
            param: {
                ids: ids
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件校验解锁纪录
     * @param {number} id 插件ID
     */
    public unblocking(callback: Function, id: number): void {
        const net: INetData = {
            name: NetName.PLUG_UNBLOCKING,
            uri: 'activityPlugin/unblocking',
            type: 'post',
            dataType: 'json',
            param: {
                id: id
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------自定义活动工具-----------------------------------------

    /**
     * 获取基础信息
     */
    public ajaxElement(callback: Function): void {
        const net: INetData = {
            name: NetName.CUSTOM_ELEMENT,
            uri: window['CFG'].getElement,
            type: 'post',
            dataType: 'json',
            param: {
                hdType: DataManager.ins.customCfgData.hdType,
                hdToolId: DataManager.ins.customCfgData.hdToolId,
                actId: DataManager.ins.customCfgData.actId,
                preview: DataManager.ins.customCfgData.preview
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询闯关游戏 配置信息
     * @param {number} throughId
     */
    public ajaxThroughInfo(callback: Function, throughId = 1): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_INFO,
            uri: window['CFG'].ajaxThroughInfo,
            type: 'post',
            dataType: 'json',
            param: {
                hdtoolId: DataManager.ins.customCfgData.hdToolId,
                throughId: 1
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 活动工具抽奖
     * @param {number} collectRuleId 集卡规则ID，活动与集卡规则关联才能获得发卡资格
     */
    public doJoin(callback: Function, collectRuleId?: number): void {
        const param: any = {
            activityId: DataManager.ins.customCfgData.actId,
            oaId: DataManager.ins.customCfgData.oaId
        };
        if (collectRuleId) {
            param.collectRuleId = collectRuleId;
        }
        const net: INetData = {
            name: NetName.DO_JOIN_DATA,
            uri: "/hdtool/recon/doJoin",
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.getToken(net);
    }

    /**
     * 活动工具查询订单结果
     * @param orderId
     */
    public getCustomOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_STATUS,
            uri: window['CFG'].quireOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 获取抽奖前置信息
     * @param {string} orderId 订单ID
     */
    public getOrderInfo(callback: Function, orderId: number, showMsg = true): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_INFO,
            uri: window['CFG'].gameGetOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 前置开奖提交
     * @param {string} orderId 订单ID
     * @param {string} facePrice 前置开奖所需分值
     */
    public beforSubmit(callback: Function, orderId: number, facePrice: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_BEFOR_SUBMIT,
            uri: window['CFG'].gameSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId,
                facePrice: facePrice
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 答题提交
     */
    public questionSubmit(): void {

    }

    /**
     * 闯关游戏提交
     */
    public throughSubmit(callback: Function, orderId: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_SUBMIT,
            uri: window['CFG'].throughSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到信息查询
     * @param {Function} callback
     * @param {number} signActivityId 签到活动ID
     */
    public getSignInfo(callback: Function, signActivityId: number): void {
        const net: INetData = {
            name: NetName.SIGN_INFO,
            uri: '/signactivity/getSignInfo',
            type: 'post',
            dataType: 'json',
            param: {
                signActivityId: signActivityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到接口
     * @param {Function} callback
     * @param {number} signActivityId 签到活动ID
     * @param {number} activityId 插件活动ID，用于加抽奖次数，不传则使用签到身上配置的插件活动ID，否则不加抽奖次数
     */
    public doSign(callback: Function, signActivityId: number, activityId?: number): void {
        var data: any;
        if (activityId) {
            data = {
                id: signActivityId,
                activityId: activityId
            };
        } else {
            data = {
                id: signActivityId
            };
        }
        const net: INetData = {
            name: NetName.SIGN_DO_SIGN,
            uri: '/signactivity/doSign',
            type: 'post',
            dataType: 'json',
            param: data,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 签到结果查询
     * @param callback 
     * @param logId 	签到流水ID
     */
    public getSignResult(callback: Function, logId: number, pollingCheck: Function, pollingCount = 10): void {
        var data: any = {
            logId: logId
        };

        const net: INetData = {
            name: NetName.GET_SIGN_RESULT,
            uri: '/signactivity/getSignResult',
            type: 'post',
            dataType: 'json',
            param: data,
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    //////////////////////////////////////// 屈臣氏 ////////////////////////////////////////

    /**
     * 获取域名
     * @param callback 
     */
    public getDomain(callback: Function) {
        const net: INetData = {
            name: NetName.GET_DOMAIN,
            uri: '/projectx/getDomain.query',
            type: 'get',
            dataType: 'json',
            param: {
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取获取抽奖次数
     * @param callback 
     */
    public getCount(callback: Function) {
        const net: INetData = {
            name: NetName.GET_COUNT,
            uri: '/projectx/' + window['peojextId'] + '/w7d23fd8b/a03828e7a.do',
            type: 'get',
            dataType: 'json',
            param: {
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询发奖规则的奖项
     * @param callback 
     */
    public queryPrizeList(callback: Function, optionId) {
        const net: INetData = {
            name: NetName.QUERY_PRIZE_LIST,
            uri: '/projectx/' + window['peojextId'] + '/' + window['strategyId'] + '.query',
            type: 'get',
            dataType: 'json',
            param: {
                optionId: optionId
            },
            callback: callback
        };
        this.send(net);
    }


    /**
     * 预扣积分
     * @param callback 
     */
    public creditsCost(callback: Function) {
        const net: INetData = {
            name: NetName.CREDITS_COST,
            uri: '/projectx/' + window['peojextId'] + '/credits/creditsCost.do',
            type: 'post',
            dataType: 'json',
            param: {
                toPlaywayId: window['toPlaywayId'],
                toActionId: window['toActionId']
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询扣积分状态
     * @param callback 
     */
    public queryStatus(callback: Function, ticketNum, pollingCheck: Function, pollingCount: number = 10) {
        const net: INetData = {
            name: NetName.QUERY_STATUS,
            uri: '/projectx/' + window['peojextId'] + '/credits/queryStatus.do',
            type: 'get',
            dataType: 'json',
            param: {
                ticketNum: ticketNum,
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount,
        };
        this.send(net);
    }

    /**
     * 查询奖品
     * @param callback 
     */
    public getPrize(callback: Function, ticketNum?) {
        const net: INetData = {
            name: NetName.GET_PRIZE,
            uri: '/projectx/' + window['peojextId'] + '/w7d23fd8b/doJoin.do',
            type: 'get',
            dataType: 'json',
            param: {
                ticketNum: ticketNum,
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询奖品信息
     * @param callback 
     */
    public queryPrizeInfo(callback: Function, ids) {
        const net: INetData = {
            name: NetName.QUERY_PRIZE_INFO,
            uri: '/projectx/' + window['peojextId'] + '/prize.query',
            type: 'get',
            dataType: 'json',
            param: {
                ids: ids,
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询优惠卷信息
     * @param callback 
     */
    public getCouponInfo(callback: Function, projectOrderNo, pollingCheck: Function, pollingCount: number = 10) {
        const net: INetData = {
            name: NetName.GET_COUPON_INFO,
            uri: '/aaw/projectx/getCouponInfo',
            type: 'get',
            dataType: 'json',
            param: {
                projectOrderNo: projectOrderNo,
            },
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 查询我的奖品列表
     * @param callback - 回调函数
     * @param credits - 是否包含加积分，传参数代表包含加积分
     */
    public queryRecords(callback: Function, credits?) {
        const net: INetData = {
            name: NetName.QUERY_RECORDS,
            uri: '/projectx/' + window['peojextId'] + '/records.query',
            type: 'get',
            dataType: 'json',
            param: {
                credits: credits,
            },
            callback: callback
        };
        this.send(net);
    }

    public getProjectUserInfo(callback: Function) {
        const net: INetData = {
            name: NetName.GET_PROJECTUSERINFO,
            uri: '/ctool/getProjectUserInfo',
            type: 'get',
            dataType: 'json',
            param: {
            },
            callback: callback
        };
        this.send(net);
    }

    //////////////////////////////////////////////////////////////////

    /**
     * 获取奖项
     */
    public getOptions(callback: Function) {
        // const net: INetData = {
        //     name: NetName.GET_DOMAIN,
        //     uri: '/projectx/getDomain.query',
        //     type: 'get',
        //     dataType: 'json',
        //     param: {
        //     },
        //     callback: callback
        // };
        // this.send(net);
    }


    //阅读任务
    public reportToRead(callback: Function, activityId) {
        const net: INetData = {
            name: NetName.PEPORT_TO_READ,
            uri: '/customActivity/shuqi/pet/reportToRead',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    //阅读任务
    public uploadNickName(callback: Function, activityId, nickName) {
        const net: INetData = {
            name: NetName.UPLOAD_NICKNAME,
            uri: '/customActivity/shuqi/pet/uploadNickName',
            type: 'post',
            dataType: 'json',
            param: {
                activityId: activityId,
                nickName: nickName
            },
            callback: callback
        };
        this.send(net);
    }





    /**
     * 获取队伍信息-云集-汪星球游戏
     */
    public getTeamInfo(callback: Function, activityId) {
        const net: INetData = {
            name: NetName.GET_TEAMINFO,
            uri: '/customActivity/yunji/dog/teamInfo',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
   * 获取队伍信息-云集-汪星球游戏
   */
    public getShareCode(callback: Function, activityId, activityType) {
        const net: INetData = {
            name: NetName.GET_SHARECODE,
            uri: '/activtyShareCode/getShareCode',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId,
                activityType: activityType,
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取队伍信息-云集-汪星球游戏
     */
    public joinTeam(callback: Function, shareCode, activityId) {
        const net: INetData = {
            name: NetName.JOINTEAM,
            uri: '/customActivity/yunji/dog/join',
            type: 'post',
            dataType: 'json',
            param: {
                shareCode: shareCode,
                activityId: activityId,
            },
            callback: callback
        };
        this.send(net);
    }




    /**
     * 获取商店信息-云集养狗
     * @param callback 
     */
    public shopYuji(callback: Function, activityId: string): void {
        const net: INetData = {
            name: NetName.SHOP_YUNJI,
            uri: '/customActivity/yunji/dog/shop',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: activityId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取用户积分流水明细
     * @param callback 
     * @param appId appId
     * @param uid 用户Id
     * @param orderStatus 订单状态：待确认／已完成（process／finish）
     * @param rowId 当前页
     * @param max 每页条数
     */
    public weOrder(callback: Function, appId: number, uid: string, rowId: number, max: number, orderStatus: string = "success"): void {
        const net: INetData = {
            name: NetName.WE_ORDER,
            uri: (window['weUrl'] ? window['weUrl'] : '') + '/duiba/weOrder',
            type: 'get',
            dataType: 'json',
            param: {
                appId: appId,
                uid: uid,
                rowId: rowId,
                max: max,
                orderStatus: orderStatus
            },
            callback: callback
        };
        let gTime: string = '?_=' + GTime.getTimestamp();
        let realUrl: string = net.uri;
        if (realUrl.indexOf('?') != -1) {
            gTime = '&_=' + GTime.getTimestamp();
        }

        //url加参数等特殊需求（例如再玩一次需要在dostart接口的url上加埋点）
        if (net.addUrl) {
            realUrl += net.addUrl;
        }

        window['$'].ajax({
            type: net.type,
            // url: realUrl + gTime,
            url: realUrl,
            dataType: net.dataType,
            data: net.param,
            async: true,
            xhrFields: {
                withCredentials: false
            },
            crossDomain: true,
            cache: false,
            success: (result) => {
                this.onResponse(net, result);
            },
            error: (message) => {
                this.onError(net);
            }
        });
    }



    /**
     * 获取token
     * @param net 
     */
    public getToken(net: INetData): void {
        if (window['getDuibaToken']) {
            window['getDuibaToken']((tokenObj: any) => {
                net.param.token = tokenObj.token;
                this.send(net);
            }, (key: string, messageObj: any) => {
                this.onError(net);
            });
        } else {
            this.send(net);
        }
    }

    /**
     * 查询我的奖品记录
     * @param callback 
     * @param {number} page 当前页数
     */
    public getRecord(callback: Function, page: number): void {
        const net: INetData = {
            name: NetName.GET_RECORD,
            uri: (window['recordUrl'] ? window['recordUrl'] : '') + '/Crecord/getRecord',
            type: 'get',
            dataType: 'json',
            param: {
                page: page
            },
            callback: callback
        };

        let gTime: string = '?_=' + GTime.getTimestamp();
        let realUrl: string = net.uri;
        if (realUrl.indexOf('?') != -1) {
            gTime = '&_=' + GTime.getTimestamp();
        }

        //url加参数等特殊需求（例如再玩一次需要在dostart接口的url上加埋点）
        if (net.addUrl) {
            realUrl += net.addUrl;
        }

        window['$'].ajax({
            type: net.type,
            // url: realUrl + gTime,
            url: realUrl,
            dataType: net.dataType,
            data: net.param,
            async: true,
            xhrFields: {
                withCredentials: true
            },
            crossDomain: true,
            cache: false,
            success: (result) => {
                this.onResponse(net, result);
            },
            error: (message) => {
                this.onError(net);
            }
        });
    }

    /**
     * 推啊曝光埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmshow(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmshow',
            uri: domain + 'spmshow',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 推啊点击埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmclick(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmclick',
            uri: domain + 'spmclick',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧点击埋点
     * @param {IExposureData} exposure
     */
    public clickLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'clickLog',
            uri: '/log/click',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧曝光埋点
     * @param {IExposureData} exposure
     */
    public showLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'showLog',
            uri: exposure.domain + '/exposure/standard',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
      * 消息响应
      * @param net 
      * @param result 结果
      */
    protected onResponse(net: INetData, result: any): void {
        //数据处理
        const data: Data = DataManager.ins.updateData(net.name, result, net.param);
        if (net.pollingCheck) {
            console.log('result=====', result);
            console.log('data=====', data);
        }
        //接口成功
        if (net.pollingCount && net.pollingCheck(data)) {
            net.pollingCount -= 1;
            //轮询接口特殊处理
            setTimeout(() => {
                this.send(net);
            }, 500);
        } else if (net.callback) {
            if (net.pollingCheck && !net.pollingCount && net.pollingCheck(data)) {
                net.callback(false, 'pollingFaild');
            } else {
                net.callback(data.success, data || result);
            }

        }
        if (!data.success && !net.hideMsg) {
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net, result.message || result.desc || result.msg, result);
        }
    }

    /**
     * 通讯底层错误
     * @param net 
     * @param message 
     */
    protected onError(net: INetData): void {
        if (net.callback) {
            net.callback(false);
        }

        if (!net.hideMsg) {
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net);
        }
    }
    // tslint:disable-next-line:max-file-line-count
}