

declare module 'babylonjs' { 
    export = BABYLON; 
}
declare module BABYLON {
    /**
     * Set of assets to keep when moving a scene into an asset container.
     */
    class KeepAssets {
        /**
         * Cameras to keep.
         */
        cameras: Camera[];
        /**
         * Lights to keep.
         */
        lights: Light[];
        /**
         * Meshes to keep.
         */
        meshes: AbstractMesh[];
        /**
         * Skeletons to keep.
         */
        skeletons: Skeleton[];
        /**
         * ParticleSystems to keep.
         */
        particleSystems: ParticleSystem[];
        /**
         * Animations to keep.
         */
        animations: Animation[];
        /**
         * MultiMaterials to keep.
         */
        multiMaterials: MultiMaterial[];
        /**
         * Materials to keep.
         */
        materials: Material[];
        /**
         * MorphTargetManagers to keep.
         */
        morphTargetManagers: MorphTargetManager[];
        /**
         * Geometries to keep.
         */
        geometries: Geometry[];
        /**
         * TransformNodes to keep.
         */
        transformNodes: TransformNode[];
        /**
         * LensFlareSystems to keep.
         */
        lensFlareSystems: LensFlareSystem[];
        /**
         * ShadowGenerators to keep.
         */
        shadowGenerators: ShadowGenerator[];
        /**
         * ActionManagers to keep.
         */
        actionManagers: ActionManager[];
        /**
         * Sounds to keep.
         */
        sounds: Sound[];
    }
    /**
     * Container with a set of assets that can be added or removed from a scene.
     */
    class AssetContainer {
        /**
         * The scene the AssetContainer belongs to.
         */
        scene: Scene;
        /**
         * Cameras populated in the container.
         */
        cameras: Camera[];
        /**
         * Lights populated in the container.
         */
        lights: Light[];
        /**
         * Meshes populated in the container.
         */
        meshes: AbstractMesh[];
        /**
         * Skeletons populated in the container.
         */
        skeletons: Skeleton[];
        /**
         * ParticleSystems populated in the container.
         */
        particleSystems: IParticleSystem[];
        /**
         * Animations populated in the container.
         */
        animations: Animation[];
        /**
         * MultiMaterials populated in the container.
         */
        multiMaterials: MultiMaterial[];
        /**
         * Materials populated in the container.
         */
        materials: Material[];
        /**
         * MorphTargetManagers populated in the container.
         */
        morphTargetManagers: MorphTargetManager[];
        /**
         * Geometries populated in the container.
         */
        geometries: Geometry[];
        /**
         * TransformNodes populated in the container.
         */
        transformNodes: TransformNode[];
        /**
         * LensFlareSystems populated in the container.
         */
        lensFlareSystems: LensFlareSystem[];
        /**
         * ShadowGenerators populated in the container.
         */
        shadowGenerators: ShadowGenerator[];
        /**
         * ActionManagers populated in the container.
         */
        actionManagers: ActionManager[];
        /**
         * Sounds populated in the container.
         */
        sounds: Sound[];
        /**
         * Instantiates an AssetContainer.
         * @param scene The scene the AssetContainer belongs to.
         */
        constructor(scene: Scene);
        /**
         * Adds all the assets from the container to the scene.
         */
        addAllToScene(): void;
        /**
         * Removes all the assets in the container from the scene
         */
        removeAllFromScene(): void;
        private _moveAssets<T>(sourceAssets, targetAssets, keepAssets);
        /**
         * Removes all the assets contained in the scene and adds them to the container.
         * @param keepAssets Set of assets to keep in the scene. (default: empty)
         */
        moveAllFromScene(keepAssets?: KeepAssets): void;
    }
}

interface Window {
    mozIndexedDB: IDBFactory;
    webkitIndexedDB: IDBFactory;
    msIndexedDB: IDBFactory;
    webkitURL: typeof URL;
    mozRequestAnimationFrame(callback: FrameRequestCallback): number;
    oRequestAnimationFrame(callback: FrameRequestCallback): number;
    WebGLRenderingContext: WebGLRenderingContext;
    MSGesture: MSGesture;
    CANNON: any;
    AudioContext: AudioContext;
    webkitAudioContext: AudioContext;
    PointerEvent: any;
    Math: Math;
    Uint8Array: Uint8ArrayConstructor;
    Float32Array: Float32ArrayConstructor;
    mozURL: typeof URL;
    msURL: typeof URL;
    VRFrameData: any;
    DracoDecoderModule: any;
}
interface WebGLRenderingContext {
    drawArraysInstanced(mode: number, first: number, count: number, primcount: number): void;
    drawElementsInstanced(mode: number, count: number, type: number, offset: number, primcount: number): void;
    vertexAttribDivisor(index: number, divisor: number): void;
    createVertexArray(): any;
    bindVertexArray(vao?: WebGLVertexArrayObject | null): void;
    deleteVertexArray(vao: WebGLVertexArrayObject): void;
    blitFramebuffer(srcX0: number, srcY0: number, srcX1: number, srcY1: number, dstX0: number, dstY0: number, dstX1: number, dstY1: number, mask: number, filter: number): void;
    renderbufferStorageMultisample(target: number, samples: number, internalformat: number, width: number, height: number): void;
    bindBufferBase(target: number, index: number, buffer: WebGLBuffer | null): void;
    getUniformBlockIndex(program: WebGLProgram, uniformBlockName: string): number;
    uniformBlockBinding(program: WebGLProgram, uniformBlockIndex: number, uniformBlockBinding: number): void;
    createQuery(): WebGLQuery;
    deleteQuery(query: WebGLQuery): void;
    beginQuery(target: number, query: WebGLQuery): void;
    endQuery(target: number): void;
    getQueryParameter(query: WebGLQuery, pname: number): any;
    getQuery(target: number, pname: number): any;
    MAX_SAMPLES: number;
    RGBA8: number;
    READ_FRAMEBUFFER: number;
    DRAW_FRAMEBUFFER: number;
    UNIFORM_BUFFER: number;
    HALF_FLOAT_OES: number;
    RGBA16F: number;
    RGBA32F: number;
    R32F: number;
    RG32F: number;
    RGB32F: number;
    RED: number;
    RG: number;
    UNSIGNED_INT_24_8: number;
    DEPTH24_STENCIL8: number;
    drawBuffers(buffers: number[]): void;
    readBuffer(src: number): void;
    readonly COLOR_ATTACHMENT0: number;
    readonly COLOR_ATTACHMENT1: number;
    readonly COLOR_ATTACHMENT2: number;
    readonly COLOR_ATTACHMENT3: number;
    ANY_SAMPLES_PASSED_CONSERVATIVE: number;
    ANY_SAMPLES_PASSED: number;
    QUERY_RESULT_AVAILABLE: number;
    QUERY_RESULT: number;
}
interface Document {
    mozCancelFullScreen(): void;
    msCancelFullScreen(): void;
    mozFullScreen: boolean;
    msIsFullScreen: boolean;
    fullscreen: boolean;
    mozPointerLockElement: HTMLElement;
    msPointerLockElement: HTMLElement;
    webkitPointerLockElement: HTMLElement;
}
interface HTMLCanvasElement {
    msRequestPointerLock?(): void;
    mozRequestPointerLock?(): void;
    webkitRequestPointerLock?(): void;
}
interface CanvasRenderingContext2D {
    msImageSmoothingEnabled: boolean;
}
interface WebGLBuffer {
    references: number;
    capacity: number;
    is32Bits: boolean;
}
interface WebGLProgram {
    transformFeedback?: WebGLTransformFeedback | null;
    __SPECTOR_rebuildProgram?: ((vertexSourceCode: string, fragmentSourceCode: string, onCompiled: (program: WebGLProgram) => void, onError: (message: string) => void) => void) | null;
}
interface MouseEvent {
    mozMovementX: number;
    mozMovementY: number;
    webkitMovementX: number;
    webkitMovementY: number;
    msMovementX: number;
    msMovementY: number;
}
interface Navigator {
    mozGetVRDevices: (any: any) => any;
    webkitGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    mozGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    msGetUserMedia(constraints: MediaStreamConstraints, successCallback: NavigatorUserMediaSuccessCallback, errorCallback: NavigatorUserMediaErrorCallback): void;
    webkitGetGamepads(): Gamepad[];
    msGetGamepads(): Gamepad[];
    webkitGamepads(): Gamepad[];
}
interface HTMLVideoElement {
    mozSrcObject: any;
}
interface Screen {
    orientation: string;
    mozOrientation: string;
}
interface Math {
    fround(x: number): number;
    imul(a: number, b: number): number;
}
interface GamepadPose {
    hasOrientation: boolean;
    hasPosition: boolean;
    position?: Float32Array;
    linearVelocity?: Float32Array;
    linearAcceleration?: Float32Array;
    orientation?: Float32Array;
    angularVelocity?: Float32Array;
    angularAcceleration?: Float32Array;
}
interface EXT_disjoint_timer_query {
    QUERY_COUNTER_BITS_EXT: number;
    TIME_ELAPSED_EXT: number;
    TIMESTAMP_EXT: number;
    GPU_DISJOINT_EXT: number;
    QUERY_RESULT_EXT: number;
    QUERY_RESULT_AVAILABLE_EXT: number;
    queryCounterEXT(query: WebGLQuery, target: number): void;
    createQueryEXT(): WebGLQuery;
    beginQueryEXT(target: number, query: WebGLQuery): void;
    endQueryEXT(target: number): void;
    getQueryObjectEXT(query: WebGLQuery, target: number): any;
    deleteQueryEXT(query: WebGLQuery): void;
}
interface WebGLUniformLocation {
    _currentState: any;
}

declare module BABYLON {
    /**
     * Node is the basic class for all scene objects (Mesh, Light Camera).
     */
    class Node {
        /**
         * Gets or sets the name of the node
         */
        name: string;
        /**
         * Gets or sets the id of the node
         */
        id: string;
        /**
         * Gets or sets the unique id of the node
         */
        uniqueId: number;
        /**
         * Gets or sets a string used to store user defined state for the node
         */
        state: string;
        /**
         * Gets or sets an object used to store user defined information for the node
         */
        metadata: any;
        /**
         * Gets or sets a boolean used to define if the node must be serialized
         */
        doNotSerialize: boolean;
        /** @ignore */
        _isDisposed: boolean;
        /**
         * Gets a list of {BABYLON.Animation} associated with the node
         */
        animations: Animation[];
        private _ranges;
        /**
         * Callback raised when the node is ready to be used
         */
        onReady: (node: Node) => void;
        private _isEnabled;
        private _isReady;
        /** @ignore */
        _currentRenderId: number;
        private _parentRenderId;
        protected _childRenderId: number;
        /** @ignore */
        _waitingParentId: Nullable<string>;
        private _scene;
        /** @ignore */
        _cache: any;
        private _parentNode;
        private _children;
        /**
         * Gets a boolean indicating if the node has been disposed
         * @returns true if the node was disposed
         */
        isDisposed(): boolean;
        /**
         * Gets or sets the parent of the node
         */
        parent: Nullable<Node>;
        private _animationPropertiesOverride;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: Nullable<AnimationPropertiesOverride>;
        /**
         * Gets a string idenfifying the name of the class
         * @returns "Node" string
         */
        getClassName(): string;
        /**
        * An event triggered when the mesh is disposed
        */
        onDisposeObservable: Observable<Node>;
        private _onDisposeObserver;
        /**
         * Sets a callback that will be raised when the node will be disposed
         */
        onDispose: () => void;
        /**
         * Creates a new Node
         * @param {string} name - the name and id to be given to this node
         * @param {BABYLON.Scene} the scene this node will be added to
         */
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Gets the scene of the node
         * @returns a {BABYLON.Scene}
         */
        getScene(): Scene;
        /**
         * Gets the engine of the node
         * @returns a {BABYLON.Engine}
         */
        getEngine(): Engine;
        private _behaviors;
        /**
         * Attach a behavior to the node
         * @see http://doc.babylonjs.com/features/behaviour
         * @param behavior defines the behavior to attach
         * @returns the current Node
         */
        addBehavior(behavior: Behavior<Node>): Node;
        /**
         * Remove an attached behavior
         * @see http://doc.babylonjs.com/features/behaviour
         * @param behavior defines the behavior to attach
         * @returns the current Node
         */
        removeBehavior(behavior: Behavior<Node>): Node;
        /**
         * Gets the list of attached behaviors
         * @see http://doc.babylonjs.com/features/behaviour
         */
        readonly behaviors: Behavior<Node>[];
        /**
         * Gets an attached behavior by name
         * @param name defines the name of the behavior to look for
         * @see http://doc.babylonjs.com/features/behaviour
         * @returns null if behavior was not found else the requested behavior
         */
        getBehaviorByName(name: string): Nullable<Behavior<Node>>;
        /**
         * Returns the world matrix of the node
         * @returns a matrix containing the node's world matrix
         */
        getWorldMatrix(): Matrix;
        /** @ignore */
        _initCache(): void;
        /** @ignore */
        updateCache(force?: boolean): void;
        /** @ignore */
        _updateCache(ignoreParentClass?: boolean): void;
        /** @ignore */
        _isSynchronized(): boolean;
        /** @ignore */
        _markSyncedWithParent(): void;
        /** @ignore */
        isSynchronizedWithParent(): boolean;
        /** @ignore */
        isSynchronized(updateCache?: boolean): boolean;
        /** @ignore */
        hasNewParent(update?: boolean): boolean;
        /**
         * Is this node ready to be used/rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @return true if the node is ready
         */
        isReady(completeCheck?: boolean): boolean;
        /**
         * Is this node enabled?
         * If the node has a parent, all ancestors will be checked and false will be returned if any are false (not enabled), otherwise will return true
         * @param checkAncestors indicates if this method should check the ancestors. The default is to check the ancestors. If set to false, the method will return the value of this node without checking ancestors
         * @return whether this node (and its parent) is enabled
         */
        isEnabled(checkAncestors?: boolean): boolean;
        /**
         * Set the enabled state of this node
         * @param value defines the new enabled state
         */
        setEnabled(value: boolean): void;
        /**
         * Is this node a descendant of the given node?
         * The function will iterate up the hierarchy until the ancestor was found or no more parents defined
         * @param ancestor defines the parent node to inspect
         * @returns a boolean indicating if this node is a descendant of the given node
         */
        isDescendantOf(ancestor: Node): boolean;
        /** @ignore */
        _getDescendants(results: Node[], directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): void;
        /**
         * Will return all nodes that have this node as ascendant
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @return all children nodes of all types
         */
        getDescendants(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): Node[];
        /**
         * Get all child-meshes of this node
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.AbstractMesh}
         */
        getChildMeshes(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): AbstractMesh[];
        /**
         * Get all child-transformNodes of this node
         * @param directDescendantsOnly defines if true only direct descendants of 'this' will be considered, if false direct and also indirect (children of children, an so on in a recursive manner) descendants of 'this' will be considered
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.TransformNode}
         */
        getChildTransformNodes(directDescendantsOnly?: boolean, predicate?: (node: Node) => boolean): TransformNode[];
        /**
         * Get all direct children of this node
         * @param predicate defines an optional predicate that will be called on every evaluated child, the predicate must return true for a given child to be part of the result, otherwise it will be ignored
         * @returns an array of {BABYLON.Node}
         */
        getChildren(predicate?: (node: Node) => boolean): Node[];
        /** @ignore */
        _setReady(state: boolean): void;
        /**
         * Get an animation by name
         * @param name defines the name of the animation to look for
         * @returns null if not found else the requested animation
         */
        getAnimationByName(name: string): Nullable<Animation>;
        /**
         * Creates an animation range for this node
         * @param name defines the name of the range
         * @param from defines the starting key
         * @param to defines the end key
         */
        createAnimationRange(name: string, from: number, to: number): void;
        /**
         * Delete a specific animation range
         * @param name defines the name of the range to delete
         * @param deleteFrames defines if animation frames from the range must be deleted as well
         */
        deleteAnimationRange(name: string, deleteFrames?: boolean): void;
        /**
         * Get an animation range by name
         * @param name defines the name of the animation range to look for
         * @returns null if not found else the requested animation range
         */
        getAnimationRange(name: string): Nullable<AnimationRange>;
        /**
         * Will start the animation sequence
         * @param name defines the range frames for animation sequence
         * @param loop defines if the animation should loop (false by default)
         * @param speedRatio defines the speed factor in which to run the animation (1 by default)
         * @param onAnimationEnd defines a function to be executed when the animation ended (undefined by default)
         * @returns the object created for this animation. If range does not exist, it will return null
         */
        beginAnimation(name: string, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Serialize animation ranges into a JSON compatible object
         * @returns serialization object
         */
        serializeAnimationRanges(): any;
        /**
         * Computes the world matrix of the node
         * @param force defines if the cache version should be invalidated forcing the world matrix to be created from scratch
         * @returns the world matrix
         */
        computeWorldMatrix(force?: boolean): Matrix;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Parse animation range data from a serialization object and store them into a given node
         * @param node defines where to store the animation ranges
         * @param parsedNode defines the serialization object to read data from
         * @param scene defines the hosting scene
         */
        static ParseAnimationRanges(node: Node, parsedNode: any, scene: Scene): void;
    }
}

declare module BABYLON {
    interface IDisposable {
        dispose(): void;
    }
    interface IActiveMeshCandidateProvider {
        getMeshes(scene: Scene): AbstractMesh[];
        readonly checksIsEnabled: boolean;
    }
    /**
     * This class is used by the onRenderingGroupObservable
     */
    class RenderingGroupInfo {
        /**
         * The Scene that being rendered
         */
        scene: Scene;
        /**
         * The camera currently used for the rendering pass
         */
        camera: Nullable<Camera>;
        /**
         * The ID of the renderingGroup being processed
         */
        renderingGroupId: number;
        /**
         * The rendering stage, can be either STAGE_PRECLEAR, STAGE_PREOPAQUE, STAGE_PRETRANSPARENT, STAGE_POSTTRANSPARENT
         */
        renderStage: number;
        /**
         * Stage corresponding to the very first hook in the renderingGroup phase: before the render buffer may be cleared
         * This stage will be fired no matter what
         */
        static STAGE_PRECLEAR: number;
        /**
         * Called before opaque object are rendered.
         * This stage will be fired only if there's 3D Opaque content to render
         */
        static STAGE_PREOPAQUE: number;
        /**
         * Called after the opaque objects are rendered and before the transparent ones
         * This stage will be fired only if there's 3D transparent content to render
         */
        static STAGE_PRETRANSPARENT: number;
        /**
         * Called after the transparent object are rendered, last hook of the renderingGroup phase
         * This stage will be fired no matter what
         */
        static STAGE_POSTTRANSPARENT: number;
    }
    /**
     * Represents a scene to be rendered by the engine.
     * @see http://doc.babylonjs.com/features/scene
     */
    class Scene implements IAnimatable {
        private static _FOGMODE_NONE;
        private static _FOGMODE_EXP;
        private static _FOGMODE_EXP2;
        private static _FOGMODE_LINEAR;
        private static _uniqueIdCounter;
        static MinDeltaTime: number;
        static MaxDeltaTime: number;
        /** The fog is deactivated */
        static readonly FOGMODE_NONE: number;
        /** The fog density is following an exponential function */
        static readonly FOGMODE_EXP: number;
        /** The fog density is following an exponential function faster than FOGMODE_EXP */
        static readonly FOGMODE_EXP2: number;
        /** The fog density is following a linear function. */
        static readonly FOGMODE_LINEAR: number;
        autoClear: boolean;
        autoClearDepthAndStencil: boolean;
        clearColor: Color4;
        ambientColor: Color3;
        _environmentBRDFTexture: BaseTexture;
        protected _environmentTexture: BaseTexture;
        /**
         * Texture used in all pbr material as the reflection texture.
         * As in the majority of the scene they are the same (exception for multi room and so on),
         * this is easier to reference from here than from all the materials.
         */
        /**
         * Texture used in all pbr material as the reflection texture.
         * As in the majority of the scene they are the same (exception for multi room and so on),
         * this is easier to set here than in all the materials.
         */
        environmentTexture: BaseTexture;
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Default image processing configuration used either in the rendering
         * Forward main pass or through the imageProcessingPostProcess if present.
         * As in the majority of the scene they are the same (exception for multi camera),
         * this is easier to reference from here than from all the materials and post process.
         *
         * No setter as we it is a shared configuration, you can set the values instead.
         */
        readonly imageProcessingConfiguration: ImageProcessingConfiguration;
        private _forceWireframe;
        forceWireframe: boolean;
        private _forcePointsCloud;
        forcePointsCloud: boolean;
        forceShowBoundingBoxes: boolean;
        clipPlane: Nullable<Plane>;
        animationsEnabled: boolean;
        useConstantAnimationDeltaTime: boolean;
        constantlyUpdateMeshUnderPointer: boolean;
        hoverCursor: string;
        defaultCursor: string;
        /**
         * This is used to call preventDefault() on pointer down
         * in order to block unwanted artifacts like system double clicks
         */
        preventDefaultOnPointerDown: boolean;
        metadata: any;
        loadingPluginName: string;
        /**
         * Use this array to add regular expressions used to disable offline support for specific urls
         */
        disableOfflineSupportExceptionRules: RegExp[];
        private _spritePredicate;
        /**
        * An event triggered when the scene is disposed.
        */
        onDisposeObservable: Observable<Scene>;
        private _onDisposeObserver;
        /** A function to be executed when this scene is disposed. */
        onDispose: () => void;
        /**
        * An event triggered before rendering the scene (right after animations and physics)
        */
        onBeforeRenderObservable: Observable<Scene>;
        private _onBeforeRenderObserver;
        /** A function to be executed before rendering this scene */
        beforeRender: Nullable<() => void>;
        /**
        * An event triggered after rendering the scene
        */
        onAfterRenderObservable: Observable<Scene>;
        private _onAfterRenderObserver;
        /** A function to be executed after rendering this scene */
        afterRender: Nullable<() => void>;
        /**
        * An event triggered before animating the scene
        */
        onBeforeAnimationsObservable: Observable<Scene>;
        /**
        * An event triggered after animations processing
        */
        onAfterAnimationsObservable: Observable<Scene>;
        /**
        * An event triggered before draw calls are ready to be sent
        */
        onBeforeDrawPhaseObservable: Observable<Scene>;
        /**
        * An event triggered after draw calls have been sent
        */
        onAfterDrawPhaseObservable: Observable<Scene>;
        /**
        * An event triggered when physic simulation is about to be run
        */
        onBeforePhysicsObservable: Observable<Scene>;
        /**
        * An event triggered when physic simulation has been done
        */
        onAfterPhysicsObservable: Observable<Scene>;
        /**
        * An event triggered when the scene is ready
        */
        onReadyObservable: Observable<Scene>;
        /**
        * An event triggered before rendering a camera
        */
        onBeforeCameraRenderObservable: Observable<Camera>;
        private _onBeforeCameraRenderObserver;
        beforeCameraRender: () => void;
        /**
        * An event triggered after rendering a camera
        */
        onAfterCameraRenderObservable: Observable<Camera>;
        private _onAfterCameraRenderObserver;
        afterCameraRender: () => void;
        /**
        * An event triggered when active meshes evaluation is about to start
        */
        onBeforeActiveMeshesEvaluationObservable: Observable<Scene>;
        /**
        * An event triggered when active meshes evaluation is done
        */
        onAfterActiveMeshesEvaluationObservable: Observable<Scene>;
        /**
        * An event triggered when particles rendering is about to start
        * Note: This event can be trigger more than once per frame (because particles can be rendered by render target textures as well)
        */
        onBeforeParticlesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when particles rendering is done
        * Note: This event can be trigger more than once per frame (because particles can be rendered by render target textures as well)
        */
        onAfterParticlesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when sprites rendering is about to start
        * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
        */
        onBeforeSpritesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when sprites rendering is done
        * Note: This event can be trigger more than once per frame (because sprites can be rendered by render target textures as well)
        */
        onAfterSpritesRenderingObservable: Observable<Scene>;
        /**
        * An event triggered when SceneLoader.Append or SceneLoader.Load or SceneLoader.ImportMesh were successfully executed
        */
        onDataLoadedObservable: Observable<Scene>;
        /**
        * An event triggered when a camera is created
        */
        onNewCameraAddedObservable: Observable<Camera>;
        /**
        * An event triggered when a camera is removed
        */
        onCameraRemovedObservable: Observable<Camera>;
        /**
        * An event triggered when a light is created
        */
        onNewLightAddedObservable: Observable<Light>;
        /**
        * An event triggered when a light is removed
        */
        onLightRemovedObservable: Observable<Light>;
        /**
        * An event triggered when a geometry is created
        */
        onNewGeometryAddedObservable: Observable<Geometry>;
        /**
        * An event triggered when a geometry is removed
        */
        onGeometryRemovedObservable: Observable<Geometry>;
        /**
        * An event triggered when a transform node is created
        */
        onNewTransformNodeAddedObservable: Observable<TransformNode>;
        /**
        * An event triggered when a transform node is removed
        */
        onTransformNodeRemovedObservable: Observable<TransformNode>;
        /**
        * An event triggered when a mesh is created
        */
        onNewMeshAddedObservable: Observable<AbstractMesh>;
        /**
        * An event triggered when a mesh is removed
        */
        onMeshRemovedObservable: Observable<AbstractMesh>;
        /**
        * An event triggered when render targets are about to be rendered
        * Can happen multiple times per frame.
        */
        OnBeforeRenderTargetsRenderObservable: Observable<Scene>;
        /**
        * An event triggered when render targets were rendered.
        * Can happen multiple times per frame.
        */
        OnAfterRenderTargetsRenderObservable: Observable<Scene>;
        /**
        * An event triggered before calculating deterministic simulation step
        */
        onBeforeStepObservable: Observable<Scene>;
        /**
        * An event triggered after calculating deterministic simulation step
        */
        onAfterStepObservable: Observable<Scene>;
        /**
         * This Observable will be triggered for each stage of each renderingGroup of each rendered camera.
         * The RenderinGroupInfo class contains all the information about the context in which the observable is called
         * If you wish to register an Observer only for a given set of renderingGroup, use the mask with a combination of the renderingGroup index elevated to the power of two (1 for renderingGroup 0, 2 for renderingrOup1, 4 for 2 and 8 for 3)
         */
        onRenderingGroupObservable: Observable<RenderingGroupInfo>;
        animations: Animation[];
        private _registeredForLateAnimationBindings;
        pointerDownPredicate: (Mesh: AbstractMesh) => boolean;
        pointerUpPredicate: (Mesh: AbstractMesh) => boolean;
        pointerMovePredicate: (Mesh: AbstractMesh) => boolean;
        public _onPointerMove;
        public _onPointerDown;
        public _onPointerUp;
        /** Deprecated. Use onPointerObservable instead */
        onPointerMove: (evt: PointerEvent, pickInfo: PickingInfo, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerDown: (evt: PointerEvent, pickInfo: PickingInfo, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerUp: (evt: PointerEvent, pickInfo: Nullable<PickingInfo>, type: PointerEventTypes) => void;
        /** Deprecated. Use onPointerObservable instead */
        onPointerPick: (evt: PointerEvent, pickInfo: PickingInfo) => void;
        private _gamepadManager;
        readonly gamepadManager: GamepadManager;
        /**
         * This observable event is triggered when any ponter event is triggered. It is registered during Scene.attachControl() and it is called BEFORE the 3D engine process anything (mesh/sprite picking for instance).
         * You have the possibility to skip the process and the call to onPointerObservable by setting PointerInfoPre.skipOnPointerObservable to true
         */
        onPrePointerObservable: Observable<PointerInfoPre>;
        /**
         * Observable event triggered each time an input event is received from the rendering canvas
         */
        onPointerObservable: Observable<PointerInfo>;
        readonly unTranslatedPointer: Vector2;
        /** The distance in pixel that you have to move to prevent some events */
        static DragMovementThreshold: number;
        /** Time in milliseconds to wait to raise long press events if button is still pressed */
        static LongPressDelay: number;
        /** Time in milliseconds with two consecutive clicks will be considered as a double click */
        static DoubleClickDelay: number;
        /** If you need to check double click without raising a single click at first click, enable this flag */
        static ExclusiveDoubleClickMode: boolean;
        private _initClickEvent;
        private _initActionManager;
        private _delayedSimpleClick;
        private _delayedSimpleClickTimeout;
        private _previousDelayedSimpleClickTimeout;
        private _meshPickProceed;
        private _previousButtonPressed;
        private _currentPickResult;
        private _previousPickResult;
        private _totalPointersPressed;
        private _doubleClickOccured;
        /** Define this parameter if you are using multiple cameras and you want to specify which one should be used for pointer position */
        cameraToUseForPointers: Nullable<Camera>;
        private _pointerX;
        private _pointerY;
        private _unTranslatedPointerX;
        private _unTranslatedPointerY;
        private _startingPointerPosition;
        private _previousStartingPointerPosition;
        private _startingPointerTime;
        private _previousStartingPointerTime;
        private _timeAccumulator;
        private _currentStepId;
        private _currentInternalStep;
        _mirroredCameraPosition: Nullable<Vector3>;
        /**
         * This observable event is triggered when any keyboard event si raised and registered during Scene.attachControl()
         * You have the possibility to skip the process and the call to onKeyboardObservable by setting KeyboardInfoPre.skipOnPointerObservable to true
         */
        onPreKeyboardObservable: Observable<KeyboardInfoPre>;
        /**
         * Observable event triggered each time an keyboard event is received from the hosting window
         */
        onKeyboardObservable: Observable<KeyboardInfo>;
        private _onKeyDown;
        private _onKeyUp;
        private _onCanvasFocusObserver;
        private _onCanvasBlurObserver;
        /**
        * use right-handed coordinate system on this scene.
        */
        private _useRightHandedSystem;
        useRightHandedSystem: boolean;
        setStepId(newStepId: number): void;
        getStepId(): number;
        getInternalStep(): number;
        private _fogEnabled;
        /**
        * is fog enabled on this scene.
        */
        fogEnabled: boolean;
        private _fogMode;
        fogMode: number;
        fogColor: Color3;
        fogDensity: number;
        fogStart: number;
        fogEnd: number;
        /**
        * is shadow enabled on this scene.
        */
        private _shadowsEnabled;
        shadowsEnabled: boolean;
        /**
        * is light enabled on this scene.
        */
        private _lightsEnabled;
        lightsEnabled: boolean;
        /**
        * All of the lights added to this scene.
        */
        lights: Light[];
        /** All of the cameras added to this scene. */
        cameras: Camera[];
        /** All of the active cameras added to this scene. */
        activeCameras: Camera[];
        /** The current active camera */
        activeCamera: Nullable<Camera>;
        /**
        * All of the tranform nodes added to this scene.
        */
        transformNodes: TransformNode[];
        /**
        * All of the (abstract) meshes added to this scene.
        */
        meshes: AbstractMesh[];
        /**
        * All of the animation groups added to this scene.
        */
        animationGroups: AnimationGroup[];
        private _geometries;
        materials: Material[];
        multiMaterials: MultiMaterial[];
        private _defaultMaterial;
        /** The default material used on meshes when no material is affected */
        /** The default material used on meshes when no material is affected */
        defaultMaterial: Material;
        private _texturesEnabled;
        texturesEnabled: boolean;
        textures: BaseTexture[];
        particlesEnabled: boolean;
        particleSystems: IParticleSystem[];
        spritesEnabled: boolean;
        spriteManagers: SpriteManager[];
        /**
         * The list of layers (background and foreground) of the scene.
         */
        layers: Layer[];
        /**
         * The list of effect layers (highlights/glow) contained in the scene.
         */
        effectLayers: EffectLayer[];
        private _skeletonsEnabled;
        skeletonsEnabled: boolean;
        skeletons: Skeleton[];
        morphTargetManagers: MorphTargetManager[];
        lensFlaresEnabled: boolean;
        lensFlareSystems: LensFlareSystem[];
        collisionsEnabled: boolean;
        private _workerCollisions;
        collisionCoordinator: ICollisionCoordinator;
        /** Defines the gravity applied to this scene */
        gravity: Vector3;
        postProcesses: PostProcess[];
        postProcessesEnabled: boolean;
        postProcessManager: PostProcessManager;
        private _postProcessRenderPipelineManager;
        readonly postProcessRenderPipelineManager: PostProcessRenderPipelineManager;
        renderTargetsEnabled: boolean;
        dumpNextRenderTargets: boolean;
        customRenderTargets: RenderTargetTexture[];
        useDelayedTextureLoading: boolean;
        importedMeshesFiles: String[];
        probesEnabled: boolean;
        reflectionProbes: ReflectionProbe[];
        database: Database;
        /**
         * This scene's action manager
        */
        actionManager: ActionManager;
        _actionManagers: ActionManager[];
        private _meshesForIntersections;
        proceduralTexturesEnabled: boolean;
        _proceduralTextures: ProceduralTexture[];
        private _mainSoundTrack;
        soundTracks: SoundTrack[];
        private _audioEnabled;
        private _headphone;
        readonly mainSoundTrack: SoundTrack;
        VRHelper: VRExperienceHelper;
        simplificationQueue: SimplificationQueue;
        private _engine;
        private _totalVertices;
        _activeIndices: PerfCounter;
        _activeParticles: PerfCounter;
        _activeBones: PerfCounter;
        private _animationRatio;
        private _animationTimeLast;
        private _animationTime;
        animationTimeScale: number;
        _cachedMaterial: Nullable<Material>;
        _cachedEffect: Nullable<Effect>;
        _cachedVisibility: Nullable<number>;
        private _renderId;
        private _executeWhenReadyTimeoutId;
        private _intermediateRendering;
        private _viewUpdateFlag;
        private _projectionUpdateFlag;
        private _alternateViewUpdateFlag;
        private _alternateProjectionUpdateFlag;
        _toBeDisposed: SmartArray<Nullable<IDisposable>>;
        private _activeRequests;
        private _pendingData;
        private _isDisposed;
        dispatchAllSubMeshesOfActiveMeshes: boolean;
        private _activeMeshes;
        private _processedMaterials;
        private _renderTargets;
        _activeParticleSystems: SmartArray<IParticleSystem>;
        private _activeSkeletons;
        private _softwareSkinnedMeshes;
        private _renderingManager;
        private _physicsEngine;
        _activeAnimatables: Animatable[];
        private _transformMatrix;
        private _sceneUbo;
        private _alternateSceneUbo;
        private _pickWithRayInverseMatrix;
        private _boundingBoxRenderer;
        private _outlineRenderer;
        private _viewMatrix;
        private _projectionMatrix;
        private _alternateViewMatrix;
        private _alternateProjectionMatrix;
        private _alternateTransformMatrix;
        private _useAlternateCameraConfiguration;
        private _alternateRendering;
        _forcedViewPosition: Nullable<Vector3>;
        readonly _isAlternateRenderingEnabled: boolean;
        private _frustumPlanes;
        readonly frustumPlanes: Plane[];
        requireLightSorting: boolean;
        private _selectionOctree;
        private _pointerOverMesh;
        private _pointerOverSprite;
        private _debugLayer;
        private _depthRenderer;
        private _geometryBufferRenderer;
        /**
         * Gets the current geometry buffer associated to the scene.
         */
        /**
         * Sets the current geometry buffer for the scene.
         */
        geometryBufferRenderer: Nullable<GeometryBufferRenderer>;
        private _pickedDownMesh;
        private _pickedUpMesh;
        private _pickedDownSprite;
        private _externalData;
        private _uid;
        /**
         * @constructor
         * @param {BABYLON.Engine} engine - the engine to be used to render this scene.
         */
        constructor(engine: Engine);
        readonly debugLayer: DebugLayer;
        workerCollisions: boolean;
        readonly selectionOctree: Octree<AbstractMesh>;
        /**
         * The mesh that is currently under the pointer.
         * @return {BABYLON.AbstractMesh} mesh under the pointer/mouse cursor or null if none.
         */
        readonly meshUnderPointer: Nullable<AbstractMesh>;
        /**
         * Current on-screen X position of the pointer
         * @return {number} X position of the pointer
         */
        readonly pointerX: number;
        /**
         * Current on-screen Y position of the pointer
         * @return {number} Y position of the pointer
         */
        readonly pointerY: number;
        getCachedMaterial(): Nullable<Material>;
        getCachedEffect(): Nullable<Effect>;
        getCachedVisibility(): Nullable<number>;
        isCachedMaterialInvalid(material: Material, effect: Effect, visibility?: number): boolean;
        getBoundingBoxRenderer(): BoundingBoxRenderer;
        getOutlineRenderer(): OutlineRenderer;
        getEngine(): Engine;
        getTotalVertices(): number;
        readonly totalVerticesPerfCounter: PerfCounter;
        getActiveIndices(): number;
        readonly totalActiveIndicesPerfCounter: PerfCounter;
        getActiveParticles(): number;
        readonly activeParticlesPerfCounter: PerfCounter;
        getActiveBones(): number;
        readonly activeBonesPerfCounter: PerfCounter;
        getInterFramePerfCounter(): number;
        readonly interFramePerfCounter: Nullable<PerfCounter>;
        getLastFrameDuration(): number;
        readonly lastFramePerfCounter: Nullable<PerfCounter>;
        getEvaluateActiveMeshesDuration(): number;
        readonly evaluateActiveMeshesDurationPerfCounter: Nullable<PerfCounter>;
        getActiveMeshes(): SmartArray<AbstractMesh>;
        getRenderTargetsDuration(): number;
        getRenderDuration(): number;
        readonly renderDurationPerfCounter: Nullable<PerfCounter>;
        getParticlesDuration(): number;
        readonly particlesDurationPerfCounter: Nullable<PerfCounter>;
        getSpritesDuration(): number;
        readonly spriteDuractionPerfCounter: Nullable<PerfCounter>;
        getAnimationRatio(): number;
        getRenderId(): number;
        incrementRenderId(): void;
        private _updatePointerPosition(evt);
        private _createUbo();
        private _createAlternateUbo();
        /**
         * Use this method to simulate a pointer move on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         */
        simulatePointerMove(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerMove(pickResult, evt);
        /**
         * Use this method to simulate a pointer down on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         */
        simulatePointerDown(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerDown(pickResult, evt);
        /**
         * Use this method to simulate a pointer up on a mesh
         * The pickResult parameter can be obtained from a scene.pick or scene.pickWithRay
         * @param pickResult pickingInfo of the object wished to simulate pointer event on
         * @param pointerEventInit pointer event state to be used when simulating the pointer event (eg. pointer id for multitouch)
         */
        simulatePointerUp(pickResult: PickingInfo, pointerEventInit?: PointerEventInit): Scene;
        private _processPointerUp(pickResult, evt, clickInfo);
        /**
        * Attach events to the canvas (To handle actionManagers triggers and raise onPointerMove, onPointerDown and onPointerUp
        * @param attachUp defines if you want to attach events to pointerup
        * @param attachDown defines if you want to attach events to pointerdown
        * @param attachMove defines if you want to attach events to pointermove
        */
        attachControl(attachUp?: boolean, attachDown?: boolean, attachMove?: boolean): void;
        detachControl(): void;
        /**
         * This function will check if the scene can be rendered (textures are loaded, shaders are compiled)
         * Delay loaded resources are not taking in account
         * @return true if all required resources are ready
         */
        isReady(): boolean;
        resetCachedMaterial(): void;
        registerBeforeRender(func: () => void): void;
        unregisterBeforeRender(func: () => void): void;
        registerAfterRender(func: () => void): void;
        unregisterAfterRender(func: () => void): void;
        private _executeOnceBeforeRender(func);
        /**
         * The provided function will run before render once and will be disposed afterwards.
         * A timeout delay can be provided so that the function will be executed in N ms.
         * The timeout is using the browser's native setTimeout so time percision cannot be guaranteed.
         * @param func The function to be executed.
         * @param timeout optional delay in ms
         */
        executeOnceBeforeRender(func: () => void, timeout?: number): void;
        _addPendingData(data: any): void;
        _removePendingData(data: any): void;
        getWaitingItemsCount(): number;
        readonly isLoading: boolean;
        /**
         * Registers a function to be executed when the scene is ready.
         * @param {Function} func - the function to be executed.
         */
        executeWhenReady(func: () => void): void;
        /**
         * Returns a promise that resolves when the scene is ready.
         * @returns A promise that resolves when the scene is ready.
         */
        whenReadyAsync(): Promise<void>;
        _checkIsReady(): void;
        /**
         * Will start the animation sequence of a given target
         * @param target defines the target
         * @param from defines from which frame should animation start
         * @param to defines until which frame should animation run.
         * @param weight defines the weight to apply to the animation (1.0 by default)
         * @param loop defines if the animation loops
         * @param speedRatio defines the speed in which to run the animation (1.0 by default)
         * @param onAnimationEnd defines the function to be executed when the animation ends
         * @param animatable defines an animatable object. If not provided a new one will be created from the given params
         * @returns the animatable object created for this animation
         */
        beginWeightedAnimation(target: any, from: number, to: number, weight?: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable): Animatable;
        /**
         * Will start the animation sequence of a given target
         * @param target defines the target
         * @param from defines from which frame should animation start
         * @param to defines until which frame should animation run.
         * @param loop defines if the animation loops
         * @param speedRatio defines the speed in which to run the animation (1.0 by default)
         * @param onAnimationEnd defines the function to be executed when the animation ends
         * @param animatable defines an animatable object. If not provided a new one will be created from the given params
         * @param stopCurrent defines if the current animations must be stopped first (true by default)
         * @returns the animatable object created for this animation
         */
        beginAnimation(target: any, from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void, animatable?: Animatable, stopCurrent?: boolean): Animatable;
        /**
         * Begin a new animation on a given node
         * @param {BABYLON.Node} node defines the root node where the animation will take place
         * @param {BABYLON.Animation[]} defines the list of animations to start
         * @param {number} from defines the initial value
         * @param {number} to defines the final value
         * @param {boolean} loop defines if you want animation to loop (off by default)
         * @param {number} speedRatio defines the speed ratio to apply to all animations
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of created animatables
         */
        beginDirectAnimation(target: any, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Animatable;
        /**
         * Begin a new animation on a given node and its hierarchy
         * @param {BABYLON.Node} node defines the root node where the animation will take place
         * @param {boolean} directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used.
         * @param {BABYLON.Animation[]} defines the list of animations to start
         * @param {number} from defines the initial value
         * @param {number} to defines the final value
         * @param {boolean} loop defines if you want animation to loop (off by default)
         * @param {number} speedRatio defines the speed ratio to apply to all animations
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of animatables created for all nodes
         */
        beginDirectHierarchyAnimation(target: Node, directDescendantsOnly: boolean, animations: Animation[], from: number, to: number, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Animatable[];
        getAnimatableByTarget(target: any): Nullable<Animatable>;
        /**
         * Gets all animatables associated with a given target
         * @param target defines the target to look animatables for
         * @returns an array of Animatables
         */
        getAllAnimatablesByTarget(target: any): Array<Animatable>;
        readonly animatables: Animatable[];
        /**
         * Will stop the animation of the given target
         * @param target - the target
         * @param animationName - the name of the animation to stop (all animations will be stopped if empty)
         */
        stopAnimation(target: any, animationName?: string): void;
        /**
         * Stops and removes all animations that have been applied to the scene
         */
        stopAllAnimations(): void;
        private _animate();
        /** @ignore */
        _registerTargetForLateAnimationBinding(runtimeAnimation: RuntimeAnimation): void;
        private _processLateAnimationBindings();
        _switchToAlternateCameraConfiguration(active: boolean): void;
        getViewMatrix(): Matrix;
        getProjectionMatrix(): Matrix;
        getTransformMatrix(): Matrix;
        setTransformMatrix(view: Matrix, projection: Matrix): void;
        _setAlternateTransformMatrix(view: Matrix, projection: Matrix): void;
        getSceneUniformBuffer(): UniformBuffer;
        getUniqueId(): number;
        addMesh(newMesh: AbstractMesh): void;
        removeMesh(toRemove: AbstractMesh): number;
        addTransformNode(newTransformNode: TransformNode): void;
        removeTransformNode(toRemove: TransformNode): number;
        removeSkeleton(toRemove: Skeleton): number;
        removeMorphTargetManager(toRemove: MorphTargetManager): number;
        removeLight(toRemove: Light): number;
        removeCamera(toRemove: Camera): number;
        removeParticleSystem(toRemove: IParticleSystem): number;
        removeAnimation(toRemove: Animation): number;
        removeMultiMaterial(toRemove: MultiMaterial): number;
        removeMaterial(toRemove: Material): number;
        removeLensFlareSystem(toRemove: LensFlareSystem): number;
        removeActionManager(toRemove: ActionManager): number;
        addLight(newLight: Light): void;
        sortLightsByPriority(): void;
        addCamera(newCamera: Camera): void;
        addSkeleton(newSkeleton: Skeleton): void;
        addParticleSystem(newParticleSystem: IParticleSystem): void;
        addAnimation(newAnimation: Animation): void;
        addMultiMaterial(newMultiMaterial: MultiMaterial): void;
        addMaterial(newMaterial: Material): void;
        addMorphTargetManager(newMorphTargetManager: MorphTargetManager): void;
        addGeometry(newGeometrie: Geometry): void;
        addLensFlareSystem(newLensFlareSystem: LensFlareSystem): void;
        addActionManager(newActionManager: ActionManager): void;
        /**
         * Switch active camera
         * @param {Camera} newCamera - new active camera
         * @param {boolean} attachControl - call attachControl for the new active camera (default: true)
         */
        switchActiveCamera(newCamera: Camera, attachControl?: boolean): void;
        /**
         * sets the active camera of the scene using its ID
         * @param {string} id - the camera's ID
         * @return {BABYLON.Camera|null} the new active camera or null if none found.
         */
        setActiveCameraByID(id: string): Nullable<Camera>;
        /**
         * sets the active camera of the scene using its name
         * @param {string} name - the camera's name
         * @return {BABYLON.Camera|null} the new active camera or null if none found.
         */
        setActiveCameraByName(name: string): Nullable<Camera>;
        /**
         * get an animation group using its name
         * @param {string} the material's name
         * @return {BABYLON.AnimationGroup|null} the animation group or null if none found.
         */
        getAnimationGroupByName(name: string): Nullable<AnimationGroup>;
        /**
         * get a material using its id
         * @param {string} the material's ID
         * @return {BABYLON.Material|null} the material or null if none found.
         */
        getMaterialByID(id: string): Nullable<Material>;
        /**
         * get a material using its name
         * @param {string} the material's name
         * @return {BABYLON.Material|null} the material or null if none found.
         */
        getMaterialByName(name: string): Nullable<Material>;
        getLensFlareSystemByName(name: string): Nullable<LensFlareSystem>;
        getLensFlareSystemByID(id: string): Nullable<LensFlareSystem>;
        getCameraByID(id: string): Nullable<Camera>;
        getCameraByUniqueID(uniqueId: number): Nullable<Camera>;
        /**
         * get a camera using its name
         * @param {string} the camera's name
         * @return {BABYLON.Camera|null} the camera or null if none found.
         */
        getCameraByName(name: string): Nullable<Camera>;
        /**
         * get a bone using its id
         * @param {string} the bone's id
         * @return {BABYLON.Bone|null} the bone or null if not found
         */
        getBoneByID(id: string): Nullable<Bone>;
        /**
        * get a bone using its id
        * @param {string} the bone's name
        * @return {BABYLON.Bone|null} the bone or null if not found
        */
        getBoneByName(name: string): Nullable<Bone>;
        /**
         * get a light node using its name
         * @param {string} the light's name
         * @return {BABYLON.Light|null} the light or null if none found.
         */
        getLightByName(name: string): Nullable<Light>;
        /**
         * get a light node using its ID
         * @param {string} the light's id
         * @return {BABYLON.Light|null} the light or null if none found.
         */
        getLightByID(id: string): Nullable<Light>;
        /**
         * get a light node using its scene-generated unique ID
         * @param {number} the light's unique id
         * @return {BABYLON.Light|null} the light or null if none found.
         */
        getLightByUniqueID(uniqueId: number): Nullable<Light>;
        /**
         * get a particle system by id
         * @param id {number} the particle system id
         * @return {BABYLON.IParticleSystem|null} the corresponding system or null if none found.
         */
        getParticleSystemByID(id: string): Nullable<IParticleSystem>;
        /**
         * get a geometry using its ID
         * @param {string} the geometry's id
         * @return {BABYLON.Geometry|null} the geometry or null if none found.
         */
        getGeometryByID(id: string): Nullable<Geometry>;
        /**
         * add a new geometry to this scene.
         * @param {BABYLON.Geometry} geometry - the geometry to be added to the scene.
         * @param {boolean} [force] - force addition, even if a geometry with this ID already exists
         * @return {boolean} was the geometry added or not
         */
        pushGeometry(geometry: Geometry, force?: boolean): boolean;
        /**
         * Removes an existing geometry
         * @param {BABYLON.Geometry} geometry - the geometry to be removed from the scene.
         * @return {boolean} was the geometry removed or not
         */
        removeGeometry(geometry: Geometry): boolean;
        getGeometries(): Geometry[];
        /**
         * Get the first added mesh found of a given ID
         * @param {string} id - the id to search for
         * @return {BABYLON.AbstractMesh|null} the mesh found or null if not found at all.
         */
        getMeshByID(id: string): Nullable<AbstractMesh>;
        getMeshesByID(id: string): Array<AbstractMesh>;
        /**
         * Get the first added transform node found of a given ID
         * @param {string} id - the id to search for
         * @return {BABYLON.TransformNode|null} the transform node found or null if not found at all.
         */
        getTransformNodeByID(id: string): Nullable<TransformNode>;
        getTransformNodesByID(id: string): Array<TransformNode>;
        /**
         * Get a mesh with its auto-generated unique id
         * @param {number} uniqueId - the unique id to search for
         * @return {BABYLON.AbstractMesh|null} the mesh found or null if not found at all.
         */
        getMeshByUniqueID(uniqueId: number): Nullable<AbstractMesh>;
        /**
         * Get a the last added mesh found of a given ID
         * @param {string} id - the id to search for
         * @return {BABYLON.AbstractMesh|null} the mesh found or null if not found at all.
         */
        getLastMeshByID(id: string): Nullable<AbstractMesh>;
        /**
         * Get a the last added node (Mesh, Camera, Light) found of a given ID
         * @param {string} id - the id to search for
         * @return {BABYLON.Node|null} the node found or null if not found at all.
         */
        getLastEntryByID(id: string): Nullable<Node>;
        getNodeByID(id: string): Nullable<Node>;
        getNodeByName(name: string): Nullable<Node>;
        getMeshByName(name: string): Nullable<AbstractMesh>;
        getTransformNodeByName(name: string): Nullable<TransformNode>;
        getSoundByName(name: string): Nullable<Sound>;
        getLastSkeletonByID(id: string): Nullable<Skeleton>;
        getSkeletonById(id: string): Nullable<Skeleton>;
        getSkeletonByName(name: string): Nullable<Skeleton>;
        getMorphTargetManagerById(id: number): Nullable<MorphTargetManager>;
        isActiveMesh(mesh: AbstractMesh): boolean;
        /**
         * Return a the first highlight layer of the scene with a given name.
         * @param name The name of the highlight layer to look for.
         * @return The highlight layer if found otherwise null.
         */
        getHighlightLayerByName(name: string): Nullable<HighlightLayer>;
        /**
         * Return a the first highlight layer of the scene with a given name.
         * @param name The name of the highlight layer to look for.
         * @return The highlight layer if found otherwise null.
         */
        getGlowLayerByName(name: string): Nullable<GlowLayer>;
        /**
         * Return a unique id as a string which can serve as an identifier for the scene
         */
        readonly uid: string;
        /**
         * Add an externaly attached data from its key.
         * This method call will fail and return false, if such key already exists.
         * If you don't care and just want to get the data no matter what, use the more convenient getOrAddExternalDataWithFactory() method.
         * @param key the unique key that identifies the data
         * @param data the data object to associate to the key for this Engine instance
         * @return true if no such key were already present and the data was added successfully, false otherwise
         */
        addExternalData<T>(key: string, data: T): boolean;
        /**
         * Get an externaly attached data from its key
         * @param key the unique key that identifies the data
         * @return the associated data, if present (can be null), or undefined if not present
         */
        getExternalData<T>(key: string): Nullable<T>;
        /**
         * Get an externaly attached data from its key, create it using a factory if it's not already present
         * @param key the unique key that identifies the data
         * @param factory the factory that will be called to create the instance if and only if it doesn't exists
         * @return the associated data, can be null if the factory returned null.
         */
        getOrAddExternalDataWithFactory<T>(key: string, factory: (k: string) => T): T;
        /**
         * Remove an externaly attached data from the Engine instance
         * @param key the unique key that identifies the data
         * @return true if the data was successfully removed, false if it doesn't exist
         */
        removeExternalData(key: string): boolean;
        private _evaluateSubMesh(subMesh, mesh);
        /**
         * Clear the processed materials smart array preventing retention point in material dispose.
         */
        freeProcessedMaterials(): void;
        /**
         * Clear the active meshes smart array preventing retention point in mesh dispose.
         */
        freeActiveMeshes(): void;
        /**
         * Clear the info related to rendering groups preventing retention points during dispose.
         */
        freeRenderingGroups(): void;
        _isInIntermediateRendering(): boolean;
        private _activeMeshCandidateProvider;
        setActiveMeshCandidateProvider(provider: IActiveMeshCandidateProvider): void;
        getActiveMeshCandidateProvider(): IActiveMeshCandidateProvider;
        private _activeMeshesFrozen;
        /**
         * Use this function to stop evaluating active meshes. The current list will be keep alive between frames
         */
        freezeActiveMeshes(): Scene;
        /**
         * Use this function to restart evaluating active meshes on every frame
         */
        unfreezeActiveMeshes(): this;
        private _evaluateActiveMeshes();
        private _activeMesh(sourceMesh, mesh);
        updateTransformMatrix(force?: boolean): void;
        updateAlternateTransformMatrix(alternateCamera: Camera): void;
        private _renderForCamera(camera, rigParent?);
        private _processSubCameras(camera);
        private _checkIntersections();
        render(): void;
        private _updateAudioParameters();
        audioEnabled: boolean;
        private _disableAudio();
        private _enableAudio();
        headphone: boolean;
        private _switchAudioModeForHeadphones();
        private _switchAudioModeForNormalSpeakers();
        /**
         * Creates a depth renderer a given camera which contains a depth map which can be used for post processing.
         * @param camera The camera to create the depth renderer on (default: scene's active camera)
         * @returns the created depth renderer
         */
        enableDepthRenderer(camera?: Nullable<Camera>): DepthRenderer;
        /**
         * Disables a depth renderer for a given camera
         * @param camera The camera to disable the depth renderer on (default: scene's active camera)
         */
        disableDepthRenderer(camera?: Nullable<Camera>): void;
        enableGeometryBufferRenderer(ratio?: number): Nullable<GeometryBufferRenderer>;
        disableGeometryBufferRenderer(): void;
        freezeMaterials(): void;
        unfreezeMaterials(): void;
        dispose(): void;
        readonly isDisposed: boolean;
        disposeSounds(): void;
        /**
         * Get the world extend vectors with an optional filter
         *
         * @param filterPredicate the predicate - which meshes should be included when calculating the world size
         * @returns {{ min: Vector3; max: Vector3 }} min and max vectors
         */
        getWorldExtends(filterPredicate?: (mesh: AbstractMesh) => boolean): {
            min: Vector3;
            max: Vector3;
        };
        createOrUpdateSelectionOctree(maxCapacity?: number, maxDepth?: number): Octree<AbstractMesh>;
        createPickingRay(x: number, y: number, world: Matrix, camera: Nullable<Camera>, cameraViewSpace?: boolean): Ray;
        createPickingRayToRef(x: number, y: number, world: Matrix, result: Ray, camera: Nullable<Camera>, cameraViewSpace?: boolean): Scene;
        createPickingRayInCameraSpace(x: number, y: number, camera?: Camera): Ray;
        createPickingRayInCameraSpaceToRef(x: number, y: number, result: Ray, camera?: Camera): Scene;
        private _internalPick(rayFunction, predicate?, fastCheck?);
        private _internalMultiPick(rayFunction, predicate?);
        private _internalPickSprites(ray, predicate?, fastCheck?, camera?);
        private _tempPickingRay;
        /** Launch a ray to try to pick a mesh in the scene
         * @param x position on screen
         * @param y position on screen
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null.
         * @param camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         */
        pick(x: number, y: number, predicate?: (mesh: AbstractMesh) => boolean, fastCheck?: boolean, camera?: Nullable<Camera>): Nullable<PickingInfo>;
        /** Launch a ray to try to pick a sprite in the scene
         * @param x position on screen
         * @param y position on screen
         * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null.
         * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         */
        pickSprite(x: number, y: number, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean, camera?: Camera): Nullable<PickingInfo>;
        private _cachedRayForTransform;
        /** Use the given ray to pick a mesh in the scene
         * @param ray The ray to use to pick meshes
         * @param predicate Predicate function used to determine eligible sprites. Can be set to null. In this case, a sprite must have isPickable set to true
         * @param fastCheck Launch a fast check only using the bounding boxes. Can be set to null.
         */
        pickWithRay(ray: Ray, predicate: (mesh: AbstractMesh) => boolean, fastCheck?: boolean): Nullable<PickingInfo>;
        /**
         * Launch a ray to try to pick a mesh in the scene
         * @param x X position on screen
         * @param y Y position on screen
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         * @param camera camera to use for computing the picking ray. Can be set to null. In this case, the scene.activeCamera will be used
         */
        multiPick(x: number, y: number, predicate?: (mesh: AbstractMesh) => boolean, camera?: Camera): Nullable<PickingInfo[]>;
        /**
         * Launch a ray to try to pick a mesh in the scene
         * @param ray Ray to use
         * @param predicate Predicate function used to determine eligible meshes. Can be set to null. In this case, a mesh must be enabled, visible and with isPickable set to true
         */
        multiPickWithRay(ray: Ray, predicate: (mesh: AbstractMesh) => boolean): Nullable<PickingInfo[]>;
        setPointerOverMesh(mesh: Nullable<AbstractMesh>): void;
        getPointerOverMesh(): Nullable<AbstractMesh>;
        setPointerOverSprite(sprite: Nullable<Sprite>): void;
        getPointerOverSprite(): Nullable<Sprite>;
        getPhysicsEngine(): Nullable<PhysicsEngine>;
        /**
         * Enables physics to the current scene
         * @param {BABYLON.Vector3} [gravity] - the scene's gravity for the physics engine
         * @param {BABYLON.IPhysicsEnginePlugin} [plugin] - The physics engine to be used. defaults to OimoJS.
         * @return {boolean} was the physics engine initialized
         */
        enablePhysics(gravity?: Nullable<Vector3>, plugin?: IPhysicsEnginePlugin): boolean;
        disablePhysicsEngine(): void;
        isPhysicsEnabled(): boolean;
        deleteCompoundImpostor(compound: any): void;
        _rebuildGeometries(): void;
        _rebuildTextures(): void;
        /**
         * Creates a default light for the scene.
         * @param replace Whether to replace the existing lights in the scene.
         */
        createDefaultLight(replace?: boolean): void;
        /**
         * Creates a default camera for the scene.
         * @param createArcRotateCamera Whether to create an arc rotate or a free camera.
         * @param replace Whether to replace the existing active camera in the scene.
         * @param attachCameraControls Whether to attach camera controls to the canvas.
         */
        createDefaultCamera(createArcRotateCamera?: boolean, replace?: boolean, attachCameraControls?: boolean): void;
        createDefaultCameraOrLight(createArcRotateCamera?: boolean, replace?: boolean, attachCameraControls?: boolean): void;
        createDefaultSkybox(environmentTexture?: BaseTexture, pbr?: boolean, scale?: number, blur?: number): Nullable<Mesh>;
        createDefaultEnvironment(options: Partial<IEnvironmentHelperOptions>): Nullable<EnvironmentHelper>;
        createDefaultVRExperience(webVROptions?: VRExperienceHelperOptions): VRExperienceHelper;
        private _getByTags(list, tagsQuery, forEach?);
        getMeshesByTags(tagsQuery: string, forEach?: (mesh: AbstractMesh) => void): Mesh[];
        getCamerasByTags(tagsQuery: string, forEach?: (camera: Camera) => void): Camera[];
        getLightsByTags(tagsQuery: string, forEach?: (light: Light) => void): Light[];
        getMaterialByTags(tagsQuery: string, forEach?: (material: Material) => void): Material[];
        /**
         * Overrides the default sort function applied in the renderging group to prepare the meshes.
         * This allowed control for front to back rendering or reversly depending of the special needs.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param opaqueSortCompareFn The opaque queue comparison function use to sort.
         * @param alphaTestSortCompareFn The alpha test queue comparison function use to sort.
         * @param transparentSortCompareFn The transparent queue comparison function use to sort.
         */
        setRenderingOrder(renderingGroupId: number, opaqueSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, alphaTestSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, transparentSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>): void;
        /**
         * Specifies whether or not the stencil and depth buffer are cleared between two rendering groups.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param autoClearDepthStencil Automatically clears depth and stencil between groups if true.
         * @param depth Automatically clears depth between groups if true and autoClear is true.
         * @param stencil Automatically clears stencil between groups if true and autoClear is true.
         */
        setRenderingAutoClearDepthStencil(renderingGroupId: number, autoClearDepthStencil: boolean, depth?: boolean, stencil?: boolean): void;
        /**
         * Will flag all materials as dirty to trigger new shader compilation
         * @param predicate If not null, it will be used to specifiy if a material has to be marked as dirty
         */
        markAllMaterialsAsDirty(flag: number, predicate?: (mat: Material) => boolean): void;
        _loadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (data: any) => void, useDatabase?: boolean, useArrayBuffer?: boolean, onError?: (request?: XMLHttpRequest, exception?: any) => void): IFileRequest;
        /** @ignore */
        _loadFileAsync(url: string, useDatabase?: boolean, useArrayBuffer?: boolean): Promise<string | ArrayBuffer>;
    }
}

declare module BABYLON {
    type Nullable<T> = T | null;
    type float = number;
    type double = number;
    type int = number;
    type FloatArray = number[] | Float32Array;
    type IndicesArray = number[] | Int32Array | Uint32Array | Uint16Array;
}

declare module BABYLON {
    /**
     * The action to be carried out following a trigger
     * @see http://doc.babylonjs.com/how_to/how_to_use_actions#available-actions
     */
    class Action {
        triggerOptions: any;
        /**
         * Trigger for the action
         */
        trigger: number;
        /**
         * Internal only - manager for action
         * @ignore
         */
        _actionManager: ActionManager;
        private _nextActiveAction;
        private _child;
        private _condition?;
        private _triggerParameter;
        /**
        * An event triggered prior to action being executed.
        */
        onBeforeExecuteObservable: Observable<Action>;
        /**
         * Creates a new Action
         * @param triggerOptions the trigger, with or without parameters, for the action
         * @param condition an optional determinant of action
         */
        constructor(triggerOptions: any, condition?: Condition);
        /**
         * Internal only
         * @ignore
         */
        _prepare(): void;
        /**
         * Gets the trigger parameters
         * @returns the trigger parameters
         */
        getTriggerParameter(): any;
        /**
         * Internal only - executes current action event
         * @ignore
         */
        _executeCurrent(evt?: ActionEvent): void;
        /**
         * Execute placeholder for child classes
         * @param evt optional action event
         */
        execute(evt?: ActionEvent): void;
        /**
         * Skips to next active action
         */
        skipToNextActiveAction(): void;
        /**
         * Adds action to chain of actions, may be a DoNothingAction
         * @param index The index of the attribute.
         * @returns The action passed in
         * @see https://www.babylonjs-playground.com/#1T30HR#0
         */
        then(action: Action): Action;
        /**
         * Internal only
         * @ignore
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @ignore
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @param parent of child
         * @returns the serialized object
         */
        serialize(parent: any): any;
        /**
         * Internal only called by serialize
         * @ignore
         */
        protected _serialize(serializedAction: any, parent?: any): any;
        /**
         * Internal only
         * @ignore
         */
        static _SerializeValueAsString: (value: any) => string;
        /**
         * Internal only
         * @ignore
         */
        static _GetTargetProperty: (target: Scene | Node) => {
            name: string;
            targetType: string;
            value: string;
        };
    }
}

declare module BABYLON {
    /**
     * ActionEvent is the event being sent when an action is triggered.
     */
    class ActionEvent {
        source: any;
        pointerX: number;
        pointerY: number;
        meshUnderPointer: Nullable<AbstractMesh>;
        sourceEvent: any;
        additionalData: any;
        /**
         * @param source The mesh or sprite that triggered the action.
         * @param pointerX The X mouse cursor position at the time of the event
         * @param pointerY The Y mouse cursor position at the time of the event
         * @param meshUnderPointer The mesh that is currently pointed at (can be null)
         * @param sourceEvent the original (browser) event that triggered the ActionEvent
         */
        constructor(source: any, pointerX: number, pointerY: number, meshUnderPointer: Nullable<AbstractMesh>, sourceEvent?: any, additionalData?: any);
        /**
         * Helper function to auto-create an ActionEvent from a source mesh.
         * @param source The source mesh that triggered the event
         * @param evt {Event} The original (browser) event
         */
        static CreateNew(source: AbstractMesh, evt?: Event, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a source mesh.
         * @param source The source sprite that triggered the event
         * @param scene Scene associated with the sprite
         * @param evt {Event} The original (browser) event
         */
        static CreateNewFromSprite(source: Sprite, scene: Scene, evt?: Event, additionalData?: any): ActionEvent;
        /**
         * Helper function to auto-create an ActionEvent from a scene. If triggered by a mesh use ActionEvent.CreateNew
         * @param scene the scene where the event occurred
         * @param evt {Event} The original (browser) event
         */
        static CreateNewFromScene(scene: Scene, evt: Event): ActionEvent;
        static CreateNewFromPrimitive(prim: any, pointerPos: Vector2, evt?: Event, additionalData?: any): ActionEvent;
    }
    /**
     * Action Manager manages all events to be triggered on a given mesh or the global scene.
     * A single scene can have many Action Managers to handle predefined actions on specific meshes.
     */
    class ActionManager {
        private static _NothingTrigger;
        private static _OnPickTrigger;
        private static _OnLeftPickTrigger;
        private static _OnRightPickTrigger;
        private static _OnCenterPickTrigger;
        private static _OnPickDownTrigger;
        private static _OnDoublePickTrigger;
        private static _OnPickUpTrigger;
        private static _OnLongPressTrigger;
        private static _OnPointerOverTrigger;
        private static _OnPointerOutTrigger;
        private static _OnEveryFrameTrigger;
        private static _OnIntersectionEnterTrigger;
        private static _OnIntersectionExitTrigger;
        private static _OnKeyDownTrigger;
        private static _OnKeyUpTrigger;
        private static _OnPickOutTrigger;
        static readonly NothingTrigger: number;
        static readonly OnPickTrigger: number;
        static readonly OnLeftPickTrigger: number;
        static readonly OnRightPickTrigger: number;
        static readonly OnCenterPickTrigger: number;
        static readonly OnPickDownTrigger: number;
        static readonly OnDoublePickTrigger: number;
        static readonly OnPickUpTrigger: number;
        static readonly OnPickOutTrigger: number;
        static readonly OnLongPressTrigger: number;
        static readonly OnPointerOverTrigger: number;
        static readonly OnPointerOutTrigger: number;
        static readonly OnEveryFrameTrigger: number;
        static readonly OnIntersectionEnterTrigger: number;
        static readonly OnIntersectionExitTrigger: number;
        static readonly OnKeyDownTrigger: number;
        static readonly OnKeyUpTrigger: number;
        static Triggers: {
            [key: string]: number;
        };
        actions: Action[];
        hoverCursor: string;
        private _scene;
        constructor(scene: Scene);
        dispose(): void;
        getScene(): Scene;
        /**
         * Does this action manager handles actions of any of the given triggers
         * @param {number[]} triggers - the triggers to be tested
         * @return {boolean} whether one (or more) of the triggers is handeled
         */
        hasSpecificTriggers(triggers: number[]): boolean;
        /**
         * Does this action manager handles actions of a given trigger
         * @param {number} trigger - the trigger to be tested
         * @return {boolean} whether the trigger is handeled
         */
        hasSpecificTrigger(trigger: number): boolean;
        /**
         * Does this action manager has pointer triggers
         * @return {boolean} whether or not it has pointer triggers
         */
        readonly hasPointerTriggers: boolean;
        /**
         * Does this action manager has pick triggers
         * @return {boolean} whether or not it has pick triggers
         */
        readonly hasPickTriggers: boolean;
        /**
         * Does exist one action manager with at least one trigger
         * @return {boolean} whether or not it exists one action manager with one trigger
        **/
        static readonly HasTriggers: boolean;
        /**
         * Does exist one action manager with at least one pick trigger
         * @return {boolean} whether or not it exists one action manager with one pick trigger
        **/
        static readonly HasPickTriggers: boolean;
        /**
         * Does exist one action manager that handles actions of a given trigger
         * @param {number} trigger - the trigger to be tested
         * @return {boolean} whether the trigger is handeled by at least one action manager
        **/
        static HasSpecificTrigger(trigger: number): boolean;
        /**
         * Registers an action to this action manager
         * @param {BABYLON.Action} action - the action to be registered
         * @return {BABYLON.Action} the action amended (prepared) after registration
         */
        registerAction(action: Action): Nullable<Action>;
        /**
         * Unregisters an action to this action manager
         * @param action The action to be unregistered
         * @return whether the action has been unregistered
         */
        unregisterAction(action: Action): Boolean;
        /**
         * Process a specific trigger
         * @param {number} trigger - the trigger to process
         * @param evt {BABYLON.ActionEvent} the event details to be processed
         */
        processTrigger(trigger: number, evt?: ActionEvent): void;
        _getEffectiveTarget(target: any, propertyPath: string): any;
        _getProperty(propertyPath: string): string;
        serialize(name: string): any;
        static Parse(parsedActions: any, object: Nullable<AbstractMesh>, scene: Scene): void;
        static GetTriggerName(trigger: number): string;
    }
}

declare module BABYLON {
    /**
     * A Condition applied to an Action
     */
    class Condition {
        /**
         * Internal only - manager for action
         * @ignore
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @ignore
         */
        _evaluationId: number;
        /**
         * Internal only
         * @ignore
         */
        _currentResult: boolean;
        /**
         * Creates a new Condition
         * @param actionManager the manager of the action the condition is applied to
         */
        constructor(actionManager: ActionManager);
        /**
         * Check if the current condition is valid
         * @returns a boolean
         */
        isValid(): boolean;
        /**
         * Internal only
         * @ignore
         */
        _getProperty(propertyPath: string): string;
        /**
         * Internal only
         * @ignore
         */
        _getEffectiveTarget(target: any, propertyPath: string): any;
        /**
         * Serialize placeholder for child classes
         * @returns the serialized object
         */
        serialize(): any;
        /**
         * Internal only
         * @ignore
         */
        protected _serialize(serializedCondition: any): any;
    }
    /**
     * Defines specific conditional operators as extensions of Condition
     */
    class ValueCondition extends Condition {
        propertyPath: string;
        value: any;
        operator: number;
        /**
         * Internal only
         * @ignore
         */
        private static _IsEqual;
        /**
         * Internal only
         * @ignore
         */
        private static _IsDifferent;
        /**
         * Internal only
         * @ignore
         */
        private static _IsGreater;
        /**
         * Internal only
         * @ignore
         */
        private static _IsLesser;
        /**
         * returns the number for IsEqual
         */
        static readonly IsEqual: number;
        /**
         * Returns the number for IsDifferent
         */
        static readonly IsDifferent: number;
        /**
         * Returns the number for IsGreater
         */
        static readonly IsGreater: number;
        /**
         * Returns the number for IsLesser
         */
        static readonly IsLesser: number;
        /**
         * Internal only The action manager for the condition
         * @ignore
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @ignore
         */
        private _target;
        /**
         * Internal only
         * @ignore
         */
        private _effectiveTarget;
        /**
         * Internal only
         * @ignore
         */
        private _property;
        /**
         * Creates a new ValueCondition
         * @param actionManager manager for the action the condition applies to
         * @param target for the action
         * @param propertyPath path to specify the property of the target the conditional operator uses
         * @param value the vale compared by the conditional operator against the current value of the property
         * @param operator the conditional operator, default {BABYLON.ValueCondition.IsEqual}
         */
        constructor(actionManager: ActionManager, target: any, propertyPath: string, value: any, operator?: number);
        /**
         * Compares the given value with the property value for the specified conditional operator
         * @returns the result of the comparison
         */
        isValid(): boolean;
        /**
         * Serialize the ValueCondition into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
        /**
         * Gets the name of the conditional operator for the ValueCondition
         * @param operator the conditional operator
         * @returns the name
         */
        static GetOperatorName(operator: number): string;
    }
    /**
     * Defines a predicate condition as an extension of Condition
     */
    class PredicateCondition extends Condition {
        predicate: () => boolean;
        /**
         * Internal only - manager for action
         * @ignore
         */
        _actionManager: ActionManager;
        /**
         * Creates a new {BABYLON.PredicateCondition}
         * @param actionManager manager for the action the condition applies to
         * @param predicate
         */
        constructor(actionManager: ActionManager, predicate: () => boolean);
        /**
         * @returns the validity of the predicate condition
         */
        isValid(): boolean;
    }
    /**
     * Defines a state condition as an extension of {BABYLON.Condition}
     */
    class StateCondition extends Condition {
        value: string;
        /**
         * Internal only - manager for action
         * @ignore
         */
        _actionManager: ActionManager;
        /**
         * Internal only
         * @ignore
         */
        private _target;
        /**
         * Creates a new {BABYLON.StateCondition}
         * @param actionManager manager for the action the condition applies to
         * @param target of the condition
         * @param value to compare with target state
         */
        constructor(actionManager: ActionManager, target: any, value: string);
        /**
         * @returns the validity of the state
         */
        isValid(): boolean;
        /**
         * Serialize the {BABYLON.StateCondition} into a JSON compatible object
         * @returns serialization object
         */
        serialize(): any;
    }
}

declare module BABYLON {
    class SwitchBooleanAction extends Action {
        propertyPath: string;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class SetStateAction extends Action {
        value: string;
        private _target;
        constructor(triggerOptions: any, target: any, value: string, condition?: Condition);
        execute(): void;
        serialize(parent: any): any;
    }
    class SetValueAction extends Action {
        propertyPath: string;
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class IncrementValueAction extends Action {
        propertyPath: string;
        value: any;
        private _target;
        private _effectiveTarget;
        private _property;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class PlayAnimationAction extends Action {
        from: number;
        to: number;
        loop: boolean | undefined;
        private _target;
        constructor(triggerOptions: any, target: any, from: number, to: number, loop?: boolean | undefined, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class StopAnimationAction extends Action {
        private _target;
        constructor(triggerOptions: any, target: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class DoNothingAction extends Action {
        constructor(triggerOptions?: any, condition?: Condition);
        execute(): void;
        serialize(parent: any): any;
    }
    class CombineAction extends Action {
        children: Action[];
        constructor(triggerOptions: any, children: Action[], condition?: Condition);
        _prepare(): void;
        execute(evt: ActionEvent): void;
        serialize(parent: any): any;
    }
    class ExecuteCodeAction extends Action {
        func: (evt: ActionEvent) => void;
        constructor(triggerOptions: any, func: (evt: ActionEvent) => void, condition?: Condition);
        execute(evt: ActionEvent): void;
    }
    class SetParentAction extends Action {
        private _parent;
        private _target;
        constructor(triggerOptions: any, target: any, parent: any, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class PlaySoundAction extends Action {
        private _sound;
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
    class StopSoundAction extends Action {
        private _sound;
        constructor(triggerOptions: any, sound: Sound, condition?: Condition);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
}

declare module BABYLON {
    class InterpolateValueAction extends Action {
        propertyPath: string;
        value: any;
        duration: number;
        stopOtherAnimations: boolean | undefined;
        onInterpolationDone: (() => void) | undefined;
        private _target;
        private _effectiveTarget;
        private _property;
        onInterpolationDoneObservable: Observable<InterpolateValueAction>;
        constructor(triggerOptions: any, target: any, propertyPath: string, value: any, duration?: number, condition?: Condition, stopOtherAnimations?: boolean | undefined, onInterpolationDone?: (() => void) | undefined);
        _prepare(): void;
        execute(): void;
        serialize(parent: any): any;
    }
}

declare module BABYLON {
    class Animatable {
        target: any;
        fromFrame: number;
        toFrame: number;
        loopAnimation: boolean;
        onAnimationEnd: (() => void) | null | undefined;
        private _localDelayOffset;
        private _pausedDelay;
        private _runtimeAnimations;
        private _paused;
        private _scene;
        private _speedRatio;
        private _weight;
        private _syncRoot;
        animationStarted: boolean;
        /**
         * Gets the root Animatable used to synchronize and normalize animations
         */
        readonly syncRoot: Animatable;
        /**
         * Gets the current frame of the first RuntimeAnimation
         * Used to synchronize Animatables
         */
        readonly masterFrame: number;
        /**
         * Gets or sets the animatable weight (-1.0 by default meaning not weighted)
         */
        weight: number;
        /**
         * Gets or sets the speed ratio to apply to the animatable (1.0 by default)
         */
        speedRatio: number;
        constructor(scene: Scene, target: any, fromFrame?: number, toFrame?: number, loopAnimation?: boolean, speedRatio?: number, onAnimationEnd?: (() => void) | null | undefined, animations?: any);
        /**
         * Synchronize and normalize current Animatable with a source Animatable.
         * This is useful when using animation weights and when animations are not of the same length
         * @param root defines the root Animatable to synchronize with
         * @returns the current Animatable
         */
        syncWith(root: Animatable): Animatable;
        getAnimations(): RuntimeAnimation[];
        appendAnimations(target: any, animations: Animation[]): void;
        getAnimationByTargetProperty(property: string): Nullable<Animation>;
        getRuntimeAnimationByTargetProperty(property: string): Nullable<RuntimeAnimation>;
        reset(): void;
        enableBlending(blendingSpeed: number): void;
        disableBlending(): void;
        goToFrame(frame: number): void;
        pause(): void;
        restart(): void;
        stop(animationName?: string): void;
        _animate(delay: number): boolean;
    }
}

declare module BABYLON {
    class AnimationRange {
        name: string;
        from: number;
        to: number;
        constructor(name: string, from: number, to: number);
        clone(): AnimationRange;
    }
    /**
     * Composed of a frame, and an action function
     */
    class AnimationEvent {
        frame: number;
        action: () => void;
        onlyOnce: boolean | undefined;
        isDone: boolean;
        constructor(frame: number, action: () => void, onlyOnce?: boolean | undefined);
    }
    class PathCursor {
        private path;
        private _onchange;
        value: number;
        animations: Animation[];
        constructor(path: Path2);
        getPoint(): Vector3;
        moveAhead(step?: number): PathCursor;
        moveBack(step?: number): PathCursor;
        move(step: number): PathCursor;
        private ensureLimits();
        private raiseOnChange();
        onchange(f: (cursor: PathCursor) => void): PathCursor;
    }
    interface IAnimationKey {
        frame: number;
        value: any;
        inTangent?: any;
        outTangent?: any;
        interpolation?: AnimationKeyInterpolation;
    }
    enum AnimationKeyInterpolation {
        /**
         * Do not interpolate between keys and use the start key value only. Tangents are ignored.
         */
        STEP = 1,
    }
    class Animation {
        name: string;
        targetProperty: string;
        framePerSecond: number;
        dataType: number;
        loopMode: number | undefined;
        enableBlending: boolean | undefined;
        static AllowMatricesInterpolation: boolean;
        private _keys;
        private _easingFunction;
        _runtimeAnimations: RuntimeAnimation[];
        private _events;
        targetPropertyPath: string[];
        blendingSpeed: number;
        private _ranges;
        static _PrepareAnimation(name: string, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction): Nullable<Animation>;
        /**
         * Sets up an animation.
         * @param property the property to animate
         * @param animationType the animation type to apply
         * @param easingFunction the easing function used in the animation
         * @returns The created animation
         */
        static CreateAnimation(property: string, animationType: number, framePerSecond: number, easingFunction: EasingFunction): Animation;
        /**
         * Create and start an animation on a node
         * @param {string} name defines the name of the global animation that will be run on all nodes
         * @param {BABYLON.Node} node defines the root node where the animation will take place
         * @param {string} targetProperty defines property to animate
         * @param {number} framePerSecond defines the number of frame per second yo use
         * @param {number} totalFrame defines the number of frames in total
         * @param {any} from defines the initial value
         * @param {any} to defines the final value
         * @param {number} loopMode defines which loop mode you want to use (off by default)
         * @param {BABYLON.EasingFunction} easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when animation end
         * @returns the animatable created for this animation
         */
        static CreateAndStartAnimation(name: string, node: Node, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Create and start an animation on a node and its descendants
         * @param {string} name defines the name of the global animation that will be run on all nodes
         * @param {BABYLON.Node} node defines the root node where the animation will take place
         * @param {boolean} directDescendantsOnly if true only direct descendants will be used, if false direct and also indirect (children of children, an so on in a recursive manner) descendants will be used.
         * @param {string} targetProperty defines property to animate
         * @param {number} framePerSecond defines the number of frame per second yo use
         * @param {number} totalFrame defines the number of frames in total
         * @param {any} from defines the initial value
         * @param {any} to defines the final value
         * @param {number} loopMode defines which loop mode you want to use (off by default)
         * @param {BABYLON.EasingFunction} easingFunction defines the easing function to use (linear by default)
         * @param onAnimationEnd defines the callback to call when an animation ends (will be called once per node)
         * @returns the list of animatables created for all nodes
         * @example https://www.babylonjs-playground.com/#MH0VLI
         */
        static CreateAndStartHierarchyAnimation(name: string, node: Node, directDescendantsOnly: boolean, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable[]>;
        static CreateMergeAndStartAnimation(name: string, node: Node, targetProperty: string, framePerSecond: number, totalFrame: number, from: any, to: any, loopMode?: number, easingFunction?: EasingFunction, onAnimationEnd?: () => void): Nullable<Animatable>;
        /**
         * Transition property of the Camera to the target Value.
         * @param property The property to transition
         * @param targetValue The target Value of the property
         * @param host The object where the property to animate belongs
         * @param scene Scene used to run the animation
         * @param frameRate Framerate (in frame/s) to use
         * @param transition The transition type we want to use
         * @param duration The duration of the animation, in milliseconds
         * @param onAnimationEnd Call back trigger at the end of the animation.
         */
        static TransitionTo(property: string, targetValue: any, host: any, scene: Scene, frameRate: number, transition: Animation, duration: number, onAnimationEnd?: Nullable<() => void>): Nullable<Animatable>;
        /**
         * Return the array of runtime animations currently using this animation
         */
        readonly runtimeAnimations: RuntimeAnimation[];
        readonly hasRunningRuntimeAnimations: boolean;
        constructor(name: string, targetProperty: string, framePerSecond: number, dataType: number, loopMode?: number | undefined, enableBlending?: boolean | undefined);
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        /**
         * Add an event to this animation.
         */
        addEvent(event: AnimationEvent): void;
        /**
         * Remove all events found at the given frame
         * @param frame
         */
        removeEvents(frame: number): void;
        getEvents(): AnimationEvent[];
        createRange(name: string, from: number, to: number): void;
        deleteRange(name: string, deleteFrames?: boolean): void;
        getRange(name: string): Nullable<AnimationRange>;
        getKeys(): Array<IAnimationKey>;
        getHighestFrame(): number;
        getEasingFunction(): IEasingFunction;
        setEasingFunction(easingFunction: EasingFunction): void;
        floatInterpolateFunction(startValue: number, endValue: number, gradient: number): number;
        floatInterpolateFunctionWithTangents(startValue: number, outTangent: number, endValue: number, inTangent: number, gradient: number): number;
        quaternionInterpolateFunction(startValue: Quaternion, endValue: Quaternion, gradient: number): Quaternion;
        quaternionInterpolateFunctionWithTangents(startValue: Quaternion, outTangent: Quaternion, endValue: Quaternion, inTangent: Quaternion, gradient: number): Quaternion;
        vector3InterpolateFunction(startValue: Vector3, endValue: Vector3, gradient: number): Vector3;
        vector3InterpolateFunctionWithTangents(startValue: Vector3, outTangent: Vector3, endValue: Vector3, inTangent: Vector3, gradient: number): Vector3;
        vector2InterpolateFunction(startValue: Vector2, endValue: Vector2, gradient: number): Vector2;
        vector2InterpolateFunctionWithTangents(startValue: Vector2, outTangent: Vector2, endValue: Vector2, inTangent: Vector2, gradient: number): Vector2;
        sizeInterpolateFunction(startValue: Size, endValue: Size, gradient: number): Size;
        color3InterpolateFunction(startValue: Color3, endValue: Color3, gradient: number): Color3;
        matrixInterpolateFunction(startValue: Matrix, endValue: Matrix, gradient: number): Matrix;
        clone(): Animation;
        setKeys(values: Array<IAnimationKey>): void;
        serialize(): any;
        private static _ANIMATIONTYPE_FLOAT;
        private static _ANIMATIONTYPE_VECTOR3;
        private static _ANIMATIONTYPE_QUATERNION;
        private static _ANIMATIONTYPE_MATRIX;
        private static _ANIMATIONTYPE_COLOR3;
        private static _ANIMATIONTYPE_VECTOR2;
        private static _ANIMATIONTYPE_SIZE;
        private static _ANIMATIONLOOPMODE_RELATIVE;
        private static _ANIMATIONLOOPMODE_CYCLE;
        private static _ANIMATIONLOOPMODE_CONSTANT;
        static readonly ANIMATIONTYPE_FLOAT: number;
        static readonly ANIMATIONTYPE_VECTOR3: number;
        static readonly ANIMATIONTYPE_VECTOR2: number;
        static readonly ANIMATIONTYPE_SIZE: number;
        static readonly ANIMATIONTYPE_QUATERNION: number;
        static readonly ANIMATIONTYPE_MATRIX: number;
        static readonly ANIMATIONTYPE_COLOR3: number;
        static readonly ANIMATIONLOOPMODE_RELATIVE: number;
        static readonly ANIMATIONLOOPMODE_CYCLE: number;
        static readonly ANIMATIONLOOPMODE_CONSTANT: number;
        static Parse(parsedAnimation: any): Animation;
        static AppendSerializedAnimations(source: IAnimatable, destination: any): any;
    }
}

declare module BABYLON {
    /**
     * This class defines the direct association between an animation and a target
     */
    class TargetedAnimation {
        animation: Animation;
        target: any;
    }
    /**
     * Use this class to create coordinated animations on multiple targets
     */
    class AnimationGroup implements IDisposable {
        name: string;
        private _scene;
        private _targetedAnimations;
        private _animatables;
        private _from;
        private _to;
        private _isStarted;
        private _speedRatio;
        onAnimationEndObservable: Observable<TargetedAnimation>;
        /**
         * Define if the animations are started
         */
        readonly isStarted: boolean;
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        /**
         * Gets or sets the speed ratio to use for all animations
         */
        speedRatio: number;
        /**
         * Gets the targeted animations for this animation group
         */
        readonly targetedAnimations: Array<TargetedAnimation>;
        /**
         * returning the list of animatables controlled by this animation group.
         */
        readonly animatables: Array<Animatable>;
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Add an animation (with its target) in the group
         * @param animation defines the animation we want to add
         * @param target defines the target of the animation
         * @returns the {BABYLON.TargetedAnimation} object
         */
        addTargetedAnimation(animation: Animation, target: any): TargetedAnimation;
        /**
         * This function will normalize every animation in the group to make sure they all go from beginFrame to endFrame
         * It can add constant keys at begin or end
         * @param beginFrame defines the new begin frame for all animations. It can't be bigger than the smallest begin frame of all animations
         * @param endFrame defines the new end frame for all animations. It can't be smaller than the largest end frame of all animations
         */
        normalize(beginFrame?: number, endFrame?: number): AnimationGroup;
        /**
         * Start all animations on given targets
         * @param loop defines if animations must loop
         * @param speedRatio defines the ratio to apply to animation speed (1 by default)
         * @param from defines the from key (optional)
         * @param to defines the to key (optional)
         * @returns the current animation group
         */
        start(loop?: boolean, speedRatio?: number, from?: number, to?: number): AnimationGroup;
        /**
         * Pause all animations
         */
        pause(): AnimationGroup;
        /**
         * Play all animations to initial state
         * This function will start() the animations if they were not started or will restart() them if they were paused
         * @param loop defines if animations must loop
         */
        play(loop?: boolean): AnimationGroup;
        /**
         * Reset all animations to initial state
         */
        reset(): AnimationGroup;
        /**
         * Restart animations from key 0
         */
        restart(): AnimationGroup;
        /**
         * Stop all animations
         */
        stop(): AnimationGroup;
        /**
         * Goes to a specific frame in this animation group
         *
         * @param frame the frame number to go to
         * @return the animationGroup
         */
        goToFrame(frame: number): AnimationGroup;
        /**
         * Dispose all associated resources
         */
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Class used to override all child animations of a given target
     */
    class AnimationPropertiesOverride {
        /**
         * Gets or sets a value indicating if animation blending must be used
         */
        enableBlending: boolean;
        /**
         * Gets or sets the blending speed to use when enableBlending is true
         */
        blendingSpeed: number;
        /**
         * Gets or sets the default loop mode to use
         */
        loopMode: number;
    }
}

declare module BABYLON {
    interface IEasingFunction {
        ease(gradient: number): number;
    }
    class EasingFunction implements IEasingFunction {
        private static _EASINGMODE_EASEIN;
        private static _EASINGMODE_EASEOUT;
        private static _EASINGMODE_EASEINOUT;
        static readonly EASINGMODE_EASEIN: number;
        static readonly EASINGMODE_EASEOUT: number;
        static readonly EASINGMODE_EASEINOUT: number;
        private _easingMode;
        setEasingMode(easingMode: number): void;
        getEasingMode(): number;
        easeInCore(gradient: number): number;
        ease(gradient: number): number;
    }
    class CircleEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class BackEase extends EasingFunction implements IEasingFunction {
        amplitude: number;
        constructor(amplitude?: number);
        easeInCore(gradient: number): number;
    }
    class BounceEase extends EasingFunction implements IEasingFunction {
        bounces: number;
        bounciness: number;
        constructor(bounces?: number, bounciness?: number);
        easeInCore(gradient: number): number;
    }
    class CubicEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class ElasticEase extends EasingFunction implements IEasingFunction {
        oscillations: number;
        springiness: number;
        constructor(oscillations?: number, springiness?: number);
        easeInCore(gradient: number): number;
    }
    class ExponentialEase extends EasingFunction implements IEasingFunction {
        exponent: number;
        constructor(exponent?: number);
        easeInCore(gradient: number): number;
    }
    class PowerEase extends EasingFunction implements IEasingFunction {
        power: number;
        constructor(power?: number);
        easeInCore(gradient: number): number;
    }
    class QuadraticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class QuarticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class QuinticEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class SineEase extends EasingFunction implements IEasingFunction {
        easeInCore(gradient: number): number;
    }
    class BezierCurveEase extends EasingFunction implements IEasingFunction {
        x1: number;
        y1: number;
        x2: number;
        y2: number;
        constructor(x1?: number, y1?: number, x2?: number, y2?: number);
        easeInCore(gradient: number): number;
    }
}

declare module BABYLON {
    class RuntimeAnimation {
        private _currentFrame;
        private _animation;
        private _target;
        private _host;
        private _originalValue;
        private _originalBlendValue;
        private _offsetsCache;
        private _highLimitsCache;
        private _stopped;
        private _blendingFactor;
        private _scene;
        private _currentValue;
        private _activeTarget;
        private _targetPath;
        private _weight;
        /**
         * Gets the current frame
         */
        readonly currentFrame: number;
        /**
         * Gets the weight of the runtime animation
         */
        readonly weight: number;
        /**
         * Gets the original value of the runtime animation
         */
        readonly originalValue: any;
        /**
         * Gets the current value of the runtime animation
         */
        readonly currentValue: any;
        /**
         * Gets the path where to store the animated value in the target
         */
        readonly targetPath: string;
        /**
         * Gets the actual target of the runtime animation
         */
        readonly target: any;
        /**
         * Create a new RuntimeAnimation object
         * @param target defines the target of the animation
         * @param animation defines the source {BABYLON.Animation} object
         * @param scene defines the hosting scene
         * @param host defines the initiating Animatable
         */
        constructor(target: any, animation: Animation, scene: Scene, host: Animatable);
        readonly animation: Animation;
        reset(): void;
        isStopped(): boolean;
        dispose(): void;
        private _getKeyValue(value);
        private _interpolate(currentFrame, repeatCount, loopMode?, offsetValue?, highLimitValue?);
        /**
         * Affect the interpolated value to the target
         * @param currentValue defines the value computed by the animation
         * @param weight defines the weight to apply to this value
         */
        setValue(currentValue: any, weight?: number): void;
        private _getCorrectLoopMode();
        /**
         * Move the current animation to a given frame
         * @param frame defines the frame to move to
         */
        goToFrame(frame: number): void;
        _prepareForSpeedRatioChange(newSpeedRatio: number): void;
        private _ratioOffset;
        private _previousDelay;
        private _previousRatio;
        /**
         * Execute the current animation
         * @param delay defines the delay to add to the current frame
         * @param from defines the lower bound of the animation range
         * @param to defines the upper bound of the animation range
         * @param loop defines if the current animation must loop
         * @param speedRatio defines the current speed ratio
         * @param weight defines the weight of the animation (default is -1 so no weight)
         * @returns a boolean indicating if the animation has ended
         */
        animate(delay: number, from: number, to: number, loop: boolean, speedRatio: number, weight?: number): boolean;
    }
}

declare module BABYLON {
    class Analyser {
        SMOOTHING: number;
        FFT_SIZE: number;
        BARGRAPHAMPLITUDE: number;
        DEBUGCANVASPOS: {
            x: number;
            y: number;
        };
        DEBUGCANVASSIZE: {
            width: number;
            height: number;
        };
        private _byteFreqs;
        private _byteTime;
        private _floatFreqs;
        private _webAudioAnalyser;
        private _debugCanvas;
        private _debugCanvasContext;
        private _scene;
        private _registerFunc;
        private _audioEngine;
        constructor(scene: Scene);
        getFrequencyBinCount(): number;
        getByteFrequencyData(): Uint8Array;
        getByteTimeDomainData(): Uint8Array;
        getFloatFrequencyData(): Uint8Array;
        drawDebugCanvas(): void;
        stopDebugCanvas(): void;
        connectAudioNodes(inputAudioNode: AudioNode, outputAudioNode: AudioNode): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class AudioEngine {
        private _audioContext;
        private _audioContextInitialized;
        canUseWebAudio: boolean;
        masterGain: GainNode;
        private _connectedAnalyser;
        WarnedWebAudioUnsupported: boolean;
        unlocked: boolean;
        onAudioUnlocked: () => any;
        isMP3supported: boolean;
        isOGGsupported: boolean;
        readonly audioContext: Nullable<AudioContext>;
        constructor();
        private _unlockiOSaudio();
        private _initializeAudioContext();
        dispose(): void;
        getGlobalVolume(): number;
        setGlobalVolume(newVolume: number): void;
        connectToAnalyser(analyser: Analyser): void;
    }
}

declare module BABYLON {
    class Sound {
        name: string;
        autoplay: boolean;
        loop: boolean;
        useCustomAttenuation: boolean;
        soundTrackId: number;
        spatialSound: boolean;
        refDistance: number;
        rolloffFactor: number;
        maxDistance: number;
        distanceModel: string;
        private _panningModel;
        onended: () => any;
        private _playbackRate;
        private _streaming;
        private _startTime;
        private _startOffset;
        private _position;
        private _localDirection;
        private _volume;
        private _isReadyToPlay;
        isPlaying: boolean;
        isPaused: boolean;
        private _isDirectional;
        private _readyToPlayCallback;
        private _audioBuffer;
        private _soundSource;
        private _streamingSource;
        private _soundPanner;
        private _soundGain;
        private _inputAudioNode;
        private _ouputAudioNode;
        private _coneInnerAngle;
        private _coneOuterAngle;
        private _coneOuterGain;
        private _scene;
        private _connectedMesh;
        private _customAttenuationFunction;
        private _registerFunc;
        private _isOutputConnected;
        private _htmlAudioElement;
        private _urlType;
        /**
        * Create a sound and attach it to a scene
        * @param name Name of your sound
        * @param urlOrArrayBuffer Url to the sound to load async or ArrayBuffer
        * @param readyToPlayCallback Provide a callback function if you'd like to load your code once the sound is ready to be played
        * @param options Objects to provide with the current available options: autoplay, loop, volume, spatialSound, maxDistance, rolloffFactor, refDistance, distanceModel, panningModel, streaming
        */
        constructor(name: string, urlOrArrayBuffer: any, scene: Scene, readyToPlayCallback?: Nullable<() => void>, options?: any);
        dispose(): void;
        isReady(): boolean;
        private _soundLoaded(audioData);
        setAudioBuffer(audioBuffer: AudioBuffer): void;
        updateOptions(options: any): void;
        private _createSpatialParameters();
        private _updateSpatialParameters();
        switchPanningModelToHRTF(): void;
        switchPanningModelToEqualPower(): void;
        private _switchPanningModel();
        connectToSoundTrackAudioNode(soundTrackAudioNode: AudioNode): void;
        /**
        * Transform this sound into a directional source
        * @param coneInnerAngle Size of the inner cone in degree
        * @param coneOuterAngle Size of the outer cone in degree
        * @param coneOuterGain Volume of the sound outside the outer cone (between 0.0 and 1.0)
        */
        setDirectionalCone(coneInnerAngle: number, coneOuterAngle: number, coneOuterGain: number): void;
        setPosition(newPosition: Vector3): void;
        setLocalDirectionToMesh(newLocalDirection: Vector3): void;
        private _updateDirection();
        updateDistanceFromListener(): void;
        setAttenuationFunction(callback: (currentVolume: number, currentDistance: number, maxDistance: number, refDistance: number, rolloffFactor: number) => number): void;
        /**
        * Play the sound
        * @param time (optional) Start the sound after X seconds. Start immediately (0) by default.
        * @param offset (optional) Start the sound setting it at a specific time
        */
        play(time?: number, offset?: number): void;
        private _onended();
        /**
        * Stop the sound
        * @param time (optional) Stop the sound after X seconds. Stop immediately (0) by default.
        */
        stop(time?: number): void;
        pause(): void;
        setVolume(newVolume: number, time?: number): void;
        setPlaybackRate(newPlaybackRate: number): void;
        getVolume(): number;
        attachToMesh(meshToConnectTo: AbstractMesh): void;
        detachFromMesh(): void;
        private _onRegisterAfterWorldMatrixUpdate(node);
        clone(): Nullable<Sound>;
        getAudioBuffer(): AudioBuffer | null;
        serialize(): any;
        static Parse(parsedSound: any, scene: Scene, rootUrl: string, sourceSound?: Sound): Sound;
    }
}

declare module BABYLON {
    class SoundTrack {
        private _outputAudioNode;
        private _scene;
        id: number;
        soundCollection: Array<Sound>;
        private _isMainTrack;
        private _connectedAnalyser;
        private _options;
        private _isInitialized;
        constructor(scene: Scene, options?: any);
        private _initializeSoundTrackAudioGraph();
        dispose(): void;
        AddSound(sound: Sound): void;
        RemoveSound(sound: Sound): void;
        setVolume(newVolume: number): void;
        switchPanningModelToHRTF(): void;
        switchPanningModelToEqualPower(): void;
        connectToAnalyser(analyser: Analyser): void;
    }
}

declare module BABYLON {
    interface Behavior<T extends Node> {
        name: string;
        init(): void;
        attach(node: T): void;
        detach(): void;
    }
}

declare module BABYLON {
    class Bone extends Node {
        name: string;
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        children: Bone[];
        animations: Animation[];
        length: number;
        _index: Nullable<number>;
        private _skeleton;
        private _localMatrix;
        private _restPose;
        private _baseMatrix;
        private _worldTransform;
        private _absoluteTransform;
        private _invertedAbsoluteTransform;
        private _parent;
        private _scaleMatrix;
        private _scaleVector;
        private _negateScaleChildren;
        private _scalingDeterminant;
        _matrix: Matrix;
        constructor(name: string, skeleton: Skeleton, parentBone?: Nullable<Bone>, localMatrix?: Nullable<Matrix>, restPose?: Nullable<Matrix>, baseMatrix?: Nullable<Matrix>, index?: Nullable<number>);
        getSkeleton(): Skeleton;
        getParent(): Nullable<Bone>;
        setParent(parent: Nullable<Bone>, updateDifferenceMatrix?: boolean): void;
        getLocalMatrix(): Matrix;
        getBaseMatrix(): Matrix;
        getRestPose(): Matrix;
        returnToRest(): void;
        getWorldMatrix(): Matrix;
        getInvertedAbsoluteTransform(): Matrix;
        getAbsoluteTransform(): Matrix;
        position: Vector3;
        rotation: Vector3;
        rotationQuaternion: Quaternion;
        scaling: Vector3;
        /**
         * Gets the animation properties override
         */
        readonly animationPropertiesOverride: AnimationPropertiesOverride;
        updateMatrix(matrix: Matrix, updateDifferenceMatrix?: boolean): void;
        _updateDifferenceMatrix(rootMatrix?: Matrix): void;
        markAsDirty(): void;
        copyAnimationRange(source: Bone, rangeName: string, frameOffset: number, rescaleAsRequired?: boolean, skelDimensionsRatio?: Nullable<Vector3>): boolean;
        /**
         * Translate the bone in local or world space.
         * @param vec The amount to translate the bone.
         * @param space The space that the translation is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        translate(vec: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the postion of the bone in local or world space.
         * @param position The position to set the bone.
         * @param space The space that the position is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setPosition(position: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the absolute postion of the bone (world space).
         * @param position The position to set the bone.
         * @param mesh The mesh that this bone is attached to.
         */
        setAbsolutePosition(position: Vector3, mesh?: AbstractMesh): void;
        /**
         * Set the scale of the bone on the x, y and z axes.
         * @param x The scale of the bone on the x axis.
         * @param x The scale of the bone on the y axis.
         * @param z The scale of the bone on the z axis.
         * @param scaleChildren Set this to true if children of the bone should be scaled.
         */
        setScale(x: number, y: number, z: number, scaleChildren?: boolean): void;
        /**
         * Scale the bone on the x, y and z axes.
         * @param x The amount to scale the bone on the x axis.
         * @param x The amount to scale the bone on the y axis.
         * @param z The amount to scale the bone on the z axis.
         * @param scaleChildren Set this to true if children of the bone should be scaled.
         */
        scale(x: number, y: number, z: number, scaleChildren?: boolean): void;
        /**
         * Set the yaw, pitch, and roll of the bone in local or world space.
         * @param yaw The rotation of the bone on the y axis.
         * @param pitch The rotation of the bone on the x axis.
         * @param roll The rotation of the bone on the z axis.
         * @param space The space that the axes of rotation are in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setYawPitchRoll(yaw: number, pitch: number, roll: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Rotate the bone on an axis in local or world space.
         * @param axis The axis to rotate the bone on.
         * @param amount The amount to rotate the bone.
         * @param space The space that the axis is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        rotate(axis: Vector3, amount: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the rotation of the bone to a particular axis angle in local or world space.
         * @param axis The axis to rotate the bone on.
         * @param angle The angle that the bone should be rotated to.
         * @param space The space that the axis is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setAxisAngle(axis: Vector3, angle: number, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the euler rotation of the bone in local of world space.
         * @param rotation The euler rotation that the bone should be set to.
         * @param space The space that the rotation is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setRotation(rotation: Vector3, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the quaternion rotation of the bone in local of world space.
         * @param quat The quaternion rotation that the bone should be set to.
         * @param space The space that the rotation is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setRotationQuaternion(quat: Quaternion, space?: Space, mesh?: AbstractMesh): void;
        /**
         * Set the rotation matrix of the bone in local of world space.
         * @param rotMat The rotation matrix that the bone should be set to.
         * @param space The space that the rotation is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         */
        setRotationMatrix(rotMat: Matrix, space?: Space, mesh?: AbstractMesh): void;
        private _rotateWithMatrix(rmat, space?, mesh?);
        private _getNegativeRotationToRef(rotMatInv, space?, mesh?);
        /**
         * Get the scale of the bone
         * @returns the scale of the bone
         */
        getScale(): Vector3;
        /**
         * Copy the scale of the bone to a vector3.
         * @param result The vector3 to copy the scale to
         */
        getScaleToRef(result: Vector3): void;
        /**
         * Get the position of the bone in local or world space.
         * @param space The space that the returned position is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @returns The position of the bone
         */
        getPosition(space?: Space, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the position of the bone to a vector3 in local or world space.
         * @param space The space that the returned position is in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @param result The vector3 to copy the position to.
         */
        getPositionToRef(space: Space | undefined, mesh: Nullable<AbstractMesh>, result: Vector3): void;
        /**
         * Get the absolute position of the bone (world space).
         * @param mesh The mesh that this bone is attached to.
         * @returns The absolute position of the bone
         */
        getAbsolutePosition(mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the absolute position of the bone (world space) to the result param.
         * @param mesh The mesh that this bone is attached to.
         * @param result The vector3 to copy the absolute position to.
         */
        getAbsolutePositionToRef(mesh: AbstractMesh, result: Vector3): void;
        /**
         * Compute the absolute transforms of this bone and its children.
         */
        computeAbsoluteTransforms(): void;
        private _syncScaleVector();
        /**
         * Get the world direction from an axis that is in the local space of the bone.
         * @param localAxis The local direction that is used to compute the world direction.
         * @param mesh The mesh that this bone is attached to.
         * @returns The world direction
         */
        getDirection(localAxis: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the world direction to a vector3 from an axis that is in the local space of the bone.
         * @param localAxis The local direction that is used to compute the world direction.
         * @param mesh The mesh that this bone is attached to.
         * @param result The vector3 that the world direction will be copied to.
         */
        getDirectionToRef(localAxis: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the euler rotation of the bone in local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @returns The euler rotation
         */
        getRotation(space?: Space, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Copy the euler rotation of the bone to a vector3.  The rotation can be in either local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @param result The vector3 that the rotation should be copied to.
         */
        getRotationToRef(space: Space | undefined, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the quaternion rotation of the bone in either local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @returns The quaternion rotation
         */
        getRotationQuaternion(space?: Space, mesh?: Nullable<AbstractMesh>): Quaternion;
        /**
         * Copy the quaternion rotation of the bone to a quaternion.  The rotation can be in either local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @param result The quaternion that the rotation should be copied to.
         */
        getRotationQuaternionToRef(space: Space | undefined, mesh: AbstractMesh | null | undefined, result: Quaternion): void;
        /**
         * Get the rotation matrix of the bone in local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @returns The rotation matrix
         */
        getRotationMatrix(space: Space | undefined, mesh: AbstractMesh): Matrix;
        /**
         * Copy the rotation matrix of the bone to a matrix.  The rotation can be in either local or world space.
         * @param space The space that the rotation should be in.
         * @param mesh The mesh that this bone is attached to.  This is only used in world space.
         * @param result The quaternion that the rotation should be copied to.
         */
        getRotationMatrixToRef(space: Space | undefined, mesh: AbstractMesh, result: Matrix): void;
        /**
         * Get the world position of a point that is in the local space of the bone.
         * @param position The local position
         * @param mesh The mesh that this bone is attached to.
         * @returns The world position
         */
        getAbsolutePositionFromLocal(position: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Get the world position of a point that is in the local space of the bone and copy it to the result param.
         * @param position The local position
         * @param mesh The mesh that this bone is attached to.
         * @param result The vector3 that the world position should be copied to.
         */
        getAbsolutePositionFromLocalToRef(position: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
        /**
         * Get the local position of a point that is in world space.
         * @param position The world position
         * @param mesh The mesh that this bone is attached to.
         * @returns The local position
         */
        getLocalPositionFromAbsolute(position: Vector3, mesh?: Nullable<AbstractMesh>): Vector3;
        /**
         * Get the local position of a point that is in world space and copy it to the result param.
         * @param position The world position
         * @param mesh The mesh that this bone is attached to.
         * @param result The vector3 that the local position should be copied to.
         */
        getLocalPositionFromAbsoluteToRef(position: Vector3, mesh: AbstractMesh | null | undefined, result: Vector3): void;
    }
}

declare module BABYLON {
    class BoneIKController {
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        targetMesh: AbstractMesh;
        poleTargetMesh: AbstractMesh;
        poleTargetBone: Nullable<Bone>;
        targetPosition: Vector3;
        poleTargetPosition: Vector3;
        poleTargetLocalOffset: Vector3;
        poleAngle: number;
        mesh: AbstractMesh;
        slerpAmount: number;
        private _bone1Quat;
        private _bone1Mat;
        private _bone2Ang;
        private _bone1;
        private _bone2;
        private _bone1Length;
        private _bone2Length;
        private _maxAngle;
        private _maxReach;
        private _rightHandedSystem;
        private _bendAxis;
        private _slerping;
        private _adjustRoll;
        maxAngle: number;
        constructor(mesh: AbstractMesh, bone: Bone, options?: {
            targetMesh?: AbstractMesh;
            poleTargetMesh?: AbstractMesh;
            poleTargetBone?: Bone;
            poleTargetLocalOffset?: Vector3;
            poleAngle?: number;
            bendAxis?: Vector3;
            maxAngle?: number;
            slerpAmount?: number;
        });
        private _setMaxAngle(ang);
        update(): void;
    }
}

declare module BABYLON {
    class BoneLookController {
        private static _tmpVecs;
        private static _tmpQuat;
        private static _tmpMats;
        /**
         * The target Vector3 that the bone will look at.
         */
        target: Vector3;
        /**
         * The mesh that the bone is attached to.
         */
        mesh: AbstractMesh;
        /**
         * The bone that will be looking to the target.
         */
        bone: Bone;
        /**
         * The up axis of the coordinate system that is used when the bone is rotated.
         */
        upAxis: Vector3;
        /**
         * The space that the up axis is in - BABYLON.Space.BONE, BABYLON.Space.LOCAL (default), or BABYLON.Space.WORLD.
         */
        upAxisSpace: Space;
        /**
         * Used to make an adjustment to the yaw of the bone.
         */
        adjustYaw: number;
        /**
         * Used to make an adjustment to the pitch of the bone.
         */
        adjustPitch: number;
        /**
         * Used to make an adjustment to the roll of the bone.
         */
        adjustRoll: number;
        /**
         * The amount to slerp (spherical linear interpolation) to the target.  Set this to a value between 0 and 1 (a value of 1 disables slerp).
         */
        slerpAmount: number;
        private _minYaw;
        private _maxYaw;
        private _minPitch;
        private _maxPitch;
        private _minYawSin;
        private _minYawCos;
        private _maxYawSin;
        private _maxYawCos;
        private _midYawConstraint;
        private _minPitchTan;
        private _maxPitchTan;
        private _boneQuat;
        private _slerping;
        private _transformYawPitch;
        private _transformYawPitchInv;
        private _firstFrameSkipped;
        private _yawRange;
        private _fowardAxis;
        /**
         * Get/set the minimum yaw angle that the bone can look to.
         */
        minYaw: number;
        /**
         * Get/set the maximum yaw angle that the bone can look to.
         */
        maxYaw: number;
        /**
         * Get/set the minimum pitch angle that the bone can look to.
         */
        minPitch: number;
        /**
         * Get/set the maximum pitch angle that the bone can look to.
         */
        maxPitch: number;
        /**
         * Create a BoneLookController
         * @param mesh the mesh that the bone belongs to
         * @param bone the bone that will be looking to the target
         * @param target the target Vector3 to look at
         * @param settings optional settings:
         * - maxYaw: the maximum angle the bone will yaw to
         * - minYaw: the minimum angle the bone will yaw to
         * - maxPitch: the maximum angle the bone will pitch to
         * - minPitch: the minimum angle the bone will yaw to
         * - slerpAmount: set the between 0 and 1 to make the bone slerp to the target.
         * - upAxis: the up axis of the coordinate system
         * - upAxisSpace: the space that the up axis is in - BABYLON.Space.BONE, BABYLON.Space.LOCAL (default), or BABYLON.Space.WORLD.
         * - yawAxis: set yawAxis if the bone does not yaw on the y axis
         * - pitchAxis: set pitchAxis if the bone does not pitch on the x axis
         * - adjustYaw: used to make an adjustment to the yaw of the bone
         * - adjustPitch: used to make an adjustment to the pitch of the bone
         * - adjustRoll: used to make an adjustment to the roll of the bone
         **/
        constructor(mesh: AbstractMesh, bone: Bone, target: Vector3, options?: {
            maxYaw?: number;
            minYaw?: number;
            maxPitch?: number;
            minPitch?: number;
            slerpAmount?: number;
            upAxis?: Vector3;
            upAxisSpace?: Space;
            yawAxis?: Vector3;
            pitchAxis?: Vector3;
            adjustYaw?: number;
            adjustPitch?: number;
            adjustRoll?: number;
        });
        /**
         * Update the bone to look at the target.  This should be called before the scene is rendered (use scene.registerBeforeRender()).
         */
        update(): void;
        private _getAngleDiff(ang1, ang2);
        private _getAngleBetween(ang1, ang2);
        private _isAngleBetween(ang, ang1, ang2);
    }
}

declare module BABYLON {
    class Skeleton implements IAnimatable {
        name: string;
        id: string;
        bones: Bone[];
        dimensionsAtRest: Vector3;
        needInitialSkinMatrix: boolean;
        animations: Array<Animation>;
        private _scene;
        private _isDirty;
        private _transformMatrices;
        private _meshesWithPoseMatrix;
        private _animatables;
        private _identity;
        private _synchronizedWithMesh;
        private _ranges;
        private _lastAbsoluteTransformsUpdateId;
        /**
         * Specifies if the skeleton should be serialized.
         */
        doNotSerialize: boolean;
        /**
         * Gets or sets the animation properties override
         */
        animationPropertiesOverride: AnimationPropertiesOverride;
        /**
         * An event triggered before computing the skeleton's matrices
         */
        onBeforeComputeObservable: Observable<Skeleton>;
        constructor(name: string, id: string, scene: Scene);
        getTransformMatrices(mesh: AbstractMesh): Float32Array;
        getScene(): Scene;
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        /**
        * Get bone's index searching by name
        * @param {string} name is bone's name to search for
        * @return {number} Indice of the bone. Returns -1 if not found
        */
        getBoneIndexByName(name: string): number;
        createAnimationRange(name: string, from: number, to: number): void;
        deleteAnimationRange(name: string, deleteFrames?: boolean): void;
        getAnimationRange(name: string): Nullable<AnimationRange>;
        /**
         *  Returns as an Array, all AnimationRanges defined on this skeleton
         */
        getAnimationRanges(): Nullable<AnimationRange>[];
        /**
         *  note: This is not for a complete retargeting, only between very similar skeleton's with only possible bone length differences
         */
        copyAnimationRange(source: Skeleton, name: string, rescaleAsRequired?: boolean): boolean;
        returnToRest(): void;
        private _getHighestAnimationFrame();
        beginAnimation(name: string, loop?: boolean, speedRatio?: number, onAnimationEnd?: () => void): Nullable<Animatable>;
        _markAsDirty(): void;
        _registerMeshWithPoseMatrix(mesh: AbstractMesh): void;
        _unregisterMeshWithPoseMatrix(mesh: AbstractMesh): void;
        _computeTransformMatrices(targetMatrix: Float32Array, initialSkinMatrix: Nullable<Matrix>): void;
        prepare(): void;
        getAnimatables(): IAnimatable[];
        clone(name: string, id: string): Skeleton;
        enableBlending(blendingSpeed?: number): void;
        dispose(): void;
        serialize(): any;
        static Parse(parsedSkeleton: any, scene: Scene): Skeleton;
        computeAbsoluteTransforms(forceUpdate?: boolean): void;
        getPoseMatrix(): Nullable<Matrix>;
        sortBones(): void;
        private _sortBones(index, bones, visited);
    }
}

declare module BABYLON {
    class BoundingBox implements ICullable {
        minimum: Vector3;
        maximum: Vector3;
        vectors: Vector3[];
        center: Vector3;
        centerWorld: Vector3;
        extendSize: Vector3;
        extendSizeWorld: Vector3;
        directions: Vector3[];
        vectorsWorld: Vector3[];
        minimumWorld: Vector3;
        maximumWorld: Vector3;
        private _worldMatrix;
        constructor(minimum: Vector3, maximum: Vector3);
        getWorldMatrix(): Matrix;
        setWorldMatrix(matrix: Matrix): BoundingBox;
        _update(world: Matrix): void;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        intersectsPoint(point: Vector3): boolean;
        intersectsSphere(sphere: BoundingSphere): boolean;
        intersectsMinMax(min: Vector3, max: Vector3): boolean;
        static Intersects(box0: BoundingBox, box1: BoundingBox): boolean;
        static IntersectsSphere(minPoint: Vector3, maxPoint: Vector3, sphereCenter: Vector3, sphereRadius: number): boolean;
        static IsCompletelyInFrustum(boundingVectors: Vector3[], frustumPlanes: Plane[]): boolean;
        static IsInFrustum(boundingVectors: Vector3[], frustumPlanes: Plane[]): boolean;
    }
}

declare module BABYLON {
    interface ICullable {
        isInFrustum(frustumPlanes: Plane[]): boolean;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
    }
    class BoundingInfo implements ICullable {
        minimum: Vector3;
        maximum: Vector3;
        boundingBox: BoundingBox;
        boundingSphere: BoundingSphere;
        private _isLocked;
        constructor(minimum: Vector3, maximum: Vector3);
        isLocked: boolean;
        update(world: Matrix): void;
        /**
         * Recreate the bounding info to be centered around a specific point given a specific extend.
         * @param center New center of the bounding info
         * @param extend New extend of the bounding info
         */
        centerOn(center: Vector3, extend: Vector3): BoundingInfo;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Gets the world distance between the min and max points of the bounding box
         */
        readonly diagonalLength: number;
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        _checkCollision(collider: Collider): boolean;
        intersectsPoint(point: Vector3): boolean;
        intersects(boundingInfo: BoundingInfo, precise: boolean): boolean;
    }
}

declare module BABYLON {
    class BoundingSphere {
        minimum: Vector3;
        maximum: Vector3;
        center: Vector3;
        radius: number;
        centerWorld: Vector3;
        radiusWorld: number;
        private _tempRadiusVector;
        constructor(minimum: Vector3, maximum: Vector3);
        _update(world: Matrix): void;
        isInFrustum(frustumPlanes: Plane[]): boolean;
        intersectsPoint(point: Vector3): boolean;
        static Intersects(sphere0: BoundingSphere, sphere1: BoundingSphere): boolean;
    }
}

declare module BABYLON {
    class Ray {
        origin: Vector3;
        direction: Vector3;
        length: number;
        private _edge1;
        private _edge2;
        private _pvec;
        private _tvec;
        private _qvec;
        private _tmpRay;
        constructor(origin: Vector3, direction: Vector3, length?: number);
        intersectsBoxMinMax(minimum: Vector3, maximum: Vector3): boolean;
        intersectsBox(box: BoundingBox): boolean;
        intersectsSphere(sphere: BoundingSphere): boolean;
        intersectsTriangle(vertex0: Vector3, vertex1: Vector3, vertex2: Vector3): Nullable<IntersectionInfo>;
        intersectsPlane(plane: Plane): Nullable<number>;
        intersectsMesh(mesh: AbstractMesh, fastCheck?: boolean): PickingInfo;
        intersectsMeshes(meshes: Array<AbstractMesh>, fastCheck?: boolean, results?: Array<PickingInfo>): Array<PickingInfo>;
        private _comparePickingInfo(pickingInfoA, pickingInfoB);
        private static smallnum;
        private static rayl;
        /**
         * Intersection test between the ray and a given segment whithin a given tolerance (threshold)
         * @param sega the first point of the segment to test the intersection against
         * @param segb the second point of the segment to test the intersection against
         * @param threshold the tolerance margin, if the ray doesn't intersect the segment but is close to the given threshold, the intersection is successful
         * @return the distance from the ray origin to the intersection point if there's intersection, or -1 if there's no intersection
         */
        intersectionSegment(sega: Vector3, segb: Vector3, threshold: number): number;
        update(x: number, y: number, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Ray;
        static Zero(): Ray;
        static CreateNew(x: number, y: number, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Ray;
        /**
        * Function will create a new transformed ray starting from origin and ending at the end point. Ray's length will be set, and ray will be
        * transformed to the given world matrix.
        * @param origin The origin point
        * @param end The end point
        * @param world a matrix to transform the ray to. Default is the identity matrix.
        */
        static CreateNewFromTo(origin: Vector3, end: Vector3, world?: Matrix): Ray;
        static Transform(ray: Ray, matrix: Matrix): Ray;
        static TransformToRef(ray: Ray, matrix: Matrix, result: Ray): void;
    }
}

declare module BABYLON {
    class Collider {
        /** Define if a collision was found */
        collisionFound: boolean;
        /**
         * Define last intersection point in local space
         */
        intersectionPoint: Vector3;
        /**
         * Define last collided mesh
         */
        collidedMesh: Nullable<AbstractMesh>;
        private _collisionPoint;
        private _planeIntersectionPoint;
        private _tempVector;
        private _tempVector2;
        private _tempVector3;
        private _tempVector4;
        private _edge;
        private _baseToVertex;
        private _destinationPoint;
        private _slidePlaneNormal;
        private _displacementVector;
        _radius: Vector3;
        _retry: number;
        private _velocity;
        private _basePoint;
        private _epsilon;
        _velocityWorldLength: number;
        _basePointWorld: Vector3;
        private _velocityWorld;
        private _normalizedVelocity;
        _initialVelocity: Vector3;
        _initialPosition: Vector3;
        private _nearestDistance;
        private _collisionMask;
        collisionMask: number;
        /**
         * Gets the plane normal used to compute the sliding response (in local space)
         */
        readonly slidePlaneNormal: Vector3;
        _initialize(source: Vector3, dir: Vector3, e: number): void;
        _checkPointInTriangle(point: Vector3, pa: Vector3, pb: Vector3, pc: Vector3, n: Vector3): boolean;
        _canDoCollision(sphereCenter: Vector3, sphereRadius: number, vecMin: Vector3, vecMax: Vector3): boolean;
        _testTriangle(faceIndex: number, trianglePlaneArray: Array<Plane>, p1: Vector3, p2: Vector3, p3: Vector3, hasMaterial: boolean): void;
        _collide(trianglePlaneArray: Array<Plane>, pts: Vector3[], indices: IndicesArray, indexStart: number, indexEnd: number, decal: number, hasMaterial: boolean): void;
        _getResponse(pos: Vector3, vel: Vector3): void;
    }
}

declare module BABYLON {
    var CollisionWorker: string;
    interface ICollisionCoordinator {
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: Nullable<AbstractMesh>, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated(mesh: AbstractMesh): void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated(geometry: Geometry): void;
        onGeometryDeleted(geometry: Geometry): void;
    }
    interface SerializedMesh {
        id: string;
        name: string;
        uniqueId: number;
        geometryId: Nullable<string>;
        sphereCenter: Array<number>;
        sphereRadius: number;
        boxMinimum: Array<number>;
        boxMaximum: Array<number>;
        worldMatrixFromCache: any;
        subMeshes: Array<SerializedSubMesh>;
        checkCollisions: boolean;
    }
    interface SerializedSubMesh {
        position: number;
        verticesStart: number;
        verticesCount: number;
        indexStart: number;
        indexCount: number;
        hasMaterial: boolean;
        sphereCenter: Array<number>;
        sphereRadius: number;
        boxMinimum: Array<number>;
        boxMaximum: Array<number>;
    }
    /**
     * Interface describing the value associated with a geometry
     */
    interface SerializedGeometry {
        /**
         * Defines the unique ID of the geometry
         */
        id: string;
        /**
         * Defines the array containing the positions
         */
        positions: Float32Array;
        /**
         * Defines the array containing the indices
         */
        indices: Uint32Array;
        /**
         * Defines the array containing the normals
         */
        normals: Float32Array;
    }
    interface BabylonMessage {
        taskType: WorkerTaskType;
        payload: InitPayload | CollidePayload | UpdatePayload;
    }
    interface SerializedColliderToWorker {
        position: Array<number>;
        velocity: Array<number>;
        radius: Array<number>;
    }
    enum WorkerTaskType {
        INIT = 0,
        UPDATE = 1,
        COLLIDE = 2,
    }
    interface WorkerReply {
        error: WorkerReplyType;
        taskType: WorkerTaskType;
        payload?: any;
    }
    interface CollisionReplyPayload {
        newPosition: Array<number>;
        collisionId: number;
        collidedMeshUniqueId: number;
    }
    interface InitPayload {
    }
    interface CollidePayload {
        collisionId: number;
        collider: SerializedColliderToWorker;
        maximumRetry: number;
        excludedMeshUniqueId: Nullable<number>;
    }
    interface UpdatePayload {
        updatedMeshes: {
            [n: number]: SerializedMesh;
        };
        updatedGeometries: {
            [s: string]: SerializedGeometry;
        };
        removedMeshes: Array<number>;
        removedGeometries: Array<string>;
    }
    enum WorkerReplyType {
        SUCCESS = 0,
        UNKNOWN_ERROR = 1,
    }
    class CollisionCoordinatorWorker implements ICollisionCoordinator {
        private _scene;
        private _scaledPosition;
        private _scaledVelocity;
        private _collisionsCallbackArray;
        private _init;
        private _runningUpdated;
        private _worker;
        private _addUpdateMeshesList;
        private _addUpdateGeometriesList;
        private _toRemoveMeshesArray;
        private _toRemoveGeometryArray;
        constructor();
        static SerializeMesh: (mesh: AbstractMesh) => SerializedMesh;
        static SerializeGeometry: (geometry: Geometry) => SerializedGeometry;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: AbstractMesh, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated: (transformNode: TransformNode) => void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated: (geometry: Geometry) => void;
        onGeometryDeleted(geometry: Geometry): void;
        private _afterRender;
        private _onMessageFromWorker;
    }
    class CollisionCoordinatorLegacy implements ICollisionCoordinator {
        private _scene;
        private _scaledPosition;
        private _scaledVelocity;
        private _finalPosition;
        getNewPosition(position: Vector3, displacement: Vector3, collider: Collider, maximumRetry: number, excludedMesh: AbstractMesh, onNewPosition: (collisionIndex: number, newPosition: Vector3, collidedMesh: Nullable<AbstractMesh>) => void, collisionIndex: number): void;
        init(scene: Scene): void;
        destroy(): void;
        onMeshAdded(mesh: AbstractMesh): void;
        onMeshUpdated(mesh: AbstractMesh): void;
        onMeshRemoved(mesh: AbstractMesh): void;
        onGeometryAdded(geometry: Geometry): void;
        onGeometryUpdated(geometry: Geometry): void;
        onGeometryDeleted(geometry: Geometry): void;
        private _collideWithWorld(position, velocity, collider, maximumRetry, finalPosition, excludedMesh?);
    }
}

declare function importScripts(...urls: string[]): void;
declare const safePostMessage: any;
declare module BABYLON {
    var WorkerIncluded: boolean;
    class CollisionCache {
        private _meshes;
        private _geometries;
        getMeshes(): {
            [n: number]: SerializedMesh;
        };
        getGeometries(): {
            [s: number]: SerializedGeometry;
        };
        getMesh(id: any): SerializedMesh;
        addMesh(mesh: SerializedMesh): void;
        removeMesh(uniqueId: number): void;
        getGeometry(id: string): SerializedGeometry;
        addGeometry(geometry: SerializedGeometry): void;
        removeGeometry(id: string): void;
    }
    class CollideWorker {
        collider: Collider;
        private _collisionCache;
        private finalPosition;
        private collisionsScalingMatrix;
        private collisionTranformationMatrix;
        constructor(collider: Collider, _collisionCache: CollisionCache, finalPosition: Vector3);
        collideWithWorld(position: Vector3, velocity: Vector3, maximumRetry: number, excludedMeshUniqueId: Nullable<number>): void;
        private checkCollision(mesh);
        private processCollisionsForSubMeshes(transformMatrix, mesh);
        private collideForSubMesh(subMesh, transformMatrix, meshGeometry);
        private checkSubmeshCollision(subMesh);
    }
    interface ICollisionDetector {
        onInit(payload: InitPayload): void;
        onUpdate(payload: UpdatePayload): void;
        onCollision(payload: CollidePayload): void;
    }
    class CollisionDetectorTransferable implements ICollisionDetector {
        private _collisionCache;
        onInit(payload: InitPayload): void;
        onUpdate(payload: UpdatePayload): void;
        onCollision(payload: CollidePayload): void;
    }
}

declare module BABYLON {
    class IntersectionInfo {
        bu: Nullable<number>;
        bv: Nullable<number>;
        distance: number;
        faceId: number;
        subMeshId: number;
        constructor(bu: Nullable<number>, bv: Nullable<number>, distance: number);
    }
    class PickingInfo {
        hit: boolean;
        distance: number;
        pickedPoint: Nullable<Vector3>;
        pickedMesh: Nullable<AbstractMesh>;
        bu: number;
        bv: number;
        faceId: number;
        subMeshId: number;
        pickedSprite: Nullable<Sprite>;
        getNormal(useWorldCoordinates?: boolean, useVerticesNormals?: boolean): Nullable<Vector3>;
        getTextureCoordinates(): Nullable<Vector2>;
    }
}

declare module BABYLON {
    class ArcRotateCamera extends TargetCamera {
        alpha: number;
        beta: number;
        radius: number;
        protected _target: Vector3;
        protected _targetHost: Nullable<AbstractMesh>;
        target: Vector3;
        inertialAlphaOffset: number;
        inertialBetaOffset: number;
        inertialRadiusOffset: number;
        lowerAlphaLimit: Nullable<number>;
        upperAlphaLimit: Nullable<number>;
        lowerBetaLimit: number;
        upperBetaLimit: number;
        lowerRadiusLimit: Nullable<number>;
        upperRadiusLimit: Nullable<number>;
        inertialPanningX: number;
        inertialPanningY: number;
        pinchToPanMaxDistance: number;
        panningDistanceLimit: Nullable<number>;
        panningOriginTarget: Vector3;
        panningInertia: number;
        angularSensibilityX: number;
        angularSensibilityY: number;
        pinchPrecision: number;
        pinchDeltaPercentage: number;
        panningSensibility: number;
        keysUp: number[];
        keysDown: number[];
        keysLeft: number[];
        keysRight: number[];
        wheelPrecision: number;
        wheelDeltaPercentage: number;
        zoomOnFactor: number;
        targetScreenOffset: Vector2;
        allowUpsideDown: boolean;
        _viewMatrix: Matrix;
        _useCtrlForPanning: boolean;
        _panningMouseButton: number;
        inputs: ArcRotateCameraInputsManager;
        _reset: () => void;
        panningAxis: Vector3;
        protected _localDirection: Vector3;
        protected _transformedDirection: Vector3;
        private _bouncingBehavior;
        readonly bouncingBehavior: Nullable<BouncingBehavior>;
        useBouncingBehavior: boolean;
        private _framingBehavior;
        readonly framingBehavior: Nullable<FramingBehavior>;
        useFramingBehavior: boolean;
        private _autoRotationBehavior;
        readonly autoRotationBehavior: Nullable<AutoRotationBehavior>;
        useAutoRotationBehavior: boolean;
        onMeshTargetChangedObservable: Observable<Nullable<AbstractMesh>>;
        onCollide: (collidedMesh: AbstractMesh) => void;
        checkCollisions: boolean;
        collisionRadius: Vector3;
        protected _collider: Collider;
        protected _previousPosition: Vector3;
        protected _collisionVelocity: Vector3;
        protected _newPosition: Vector3;
        protected _previousAlpha: number;
        protected _previousBeta: number;
        protected _previousRadius: number;
        protected _collisionTriggered: boolean;
        protected _targetBoundingCenter: Nullable<Vector3>;
        private _computationVector;
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, scene: Scene);
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        protected _getTargetPosition(): Vector3;
        /**
         * Store current camera state (fov, position, etc..)
         */
        private _storedAlpha;
        private _storedBeta;
        private _storedRadius;
        private _storedTarget;
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         */
        _restoreStateValues(): boolean;
        _isSynchronizedViewMatrix(): boolean;
        attachControl(element: HTMLElement, noPreventDefault?: boolean, useCtrlForPanning?: boolean, panningMouseButton?: number): void;
        detachControl(element: HTMLElement): void;
        _checkInputs(): void;
        protected _checkLimits(): void;
        rebuildAnglesAndRadius(): void;
        setPosition(position: Vector3): void;
        setTarget(target: AbstractMesh | Vector3, toBoundingCenter?: boolean, allowSamePosition?: boolean): void;
        _getViewMatrix(): Matrix;
        protected _onCollisionPositionChange: (collisionId: number, newPosition: Vector3, collidedMesh?: Nullable<AbstractMesh>) => void;
        zoomOn(meshes?: AbstractMesh[], doNotUpdateMaxZ?: boolean): void;
        focusOn(meshesOrMinMaxVectorAndDistance: AbstractMesh[] | {
            min: Vector3;
            max: Vector3;
            distance: number;
        }, doNotUpdateMaxZ?: boolean): void;
        /**
         * @override
         * Override Camera.createRigCamera
         */
        createRigCamera(name: string, cameraIndex: number): Camera;
        /**
         * @override
         * Override Camera._updateRigCameras
         */
        _updateRigCameras(): void;
        dispose(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class ArcRotateCameraInputsManager extends CameraInputsManager<ArcRotateCamera> {
        constructor(camera: ArcRotateCamera);
        addMouseWheel(): ArcRotateCameraInputsManager;
        addPointers(): ArcRotateCameraInputsManager;
        addKeyboard(): ArcRotateCameraInputsManager;
        addGamepad(): ArcRotateCameraInputsManager;
        addVRDeviceOrientation(): ArcRotateCameraInputsManager;
    }
}

declare module BABYLON {
    class Camera extends Node {
        inputs: CameraInputsManager<Camera>;
        private static _PERSPECTIVE_CAMERA;
        private static _ORTHOGRAPHIC_CAMERA;
        private static _FOVMODE_VERTICAL_FIXED;
        private static _FOVMODE_HORIZONTAL_FIXED;
        private static _RIG_MODE_NONE;
        private static _RIG_MODE_STEREOSCOPIC_ANAGLYPH;
        private static _RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL;
        private static _RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED;
        private static _RIG_MODE_STEREOSCOPIC_OVERUNDER;
        private static _RIG_MODE_VR;
        private static _RIG_MODE_WEBVR;
        static readonly PERSPECTIVE_CAMERA: number;
        static readonly ORTHOGRAPHIC_CAMERA: number;
        /**
         * This is the default FOV mode for perspective cameras.
         * This setting aligns the upper and lower bounds of the viewport to the upper and lower bounds of the camera frustum.
         *
         */
        static readonly FOVMODE_VERTICAL_FIXED: number;
        /**
         * This setting aligns the left and right bounds of the viewport to the left and right bounds of the camera frustum.
         *
         */
        static readonly FOVMODE_HORIZONTAL_FIXED: number;
        static readonly RIG_MODE_NONE: number;
        static readonly RIG_MODE_STEREOSCOPIC_ANAGLYPH: number;
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_PARALLEL: number;
        static readonly RIG_MODE_STEREOSCOPIC_SIDEBYSIDE_CROSSEYED: number;
        static readonly RIG_MODE_STEREOSCOPIC_OVERUNDER: number;
        static readonly RIG_MODE_VR: number;
        static readonly RIG_MODE_WEBVR: number;
        static ForceAttachControlToAlwaysPreventDefault: boolean;
        static UseAlternateWebVRRendering: boolean;
        position: Vector3;
        /**
         * The vector the camera should consider as up.
         * (default is Vector3(0, 1, 0) aka Vector3.Up())
         */
        upVector: Vector3;
        orthoLeft: Nullable<number>;
        orthoRight: Nullable<number>;
        orthoBottom: Nullable<number>;
        orthoTop: Nullable<number>;
        /**
         * FOV is set in Radians. (default is 0.8)
         */
        fov: number;
        minZ: number;
        maxZ: number;
        inertia: number;
        mode: number;
        isIntermediate: boolean;
        viewport: Viewport;
        /**
         * Restricts the camera to viewing objects with the same layerMask.
         * A camera with a layerMask of 1 will render mesh.layerMask & camera.layerMask!== 0
         */
        layerMask: number;
        /**
         * fovMode sets the camera frustum bounds to the viewport bounds. (default is FOVMODE_VERTICAL_FIXED)
         */
        fovMode: number;
        cameraRigMode: number;
        interaxialDistance: number;
        isStereoscopicSideBySide: boolean;
        _cameraRigParams: any;
        _rigCameras: Camera[];
        _rigPostProcess: Nullable<PostProcess>;
        protected _webvrViewMatrix: Matrix;
        _skipRendering: boolean;
        _alternateCamera: Camera;
        customRenderTargets: RenderTargetTexture[];
        onViewMatrixChangedObservable: Observable<Camera>;
        onProjectionMatrixChangedObservable: Observable<Camera>;
        onAfterCheckInputsObservable: Observable<Camera>;
        onRestoreStateObservable: Observable<Camera>;
        private _computedViewMatrix;
        _projectionMatrix: Matrix;
        private _doNotComputeProjectionMatrix;
        private _worldMatrix;
        _postProcesses: Nullable<PostProcess>[];
        private _transformMatrix;
        _activeMeshes: SmartArray<AbstractMesh>;
        private _globalPosition;
        private _frustumPlanes;
        private _refreshFrustumPlanes;
        constructor(name: string, position: Vector3, scene: Scene);
        private _storedFov;
        private _stateStored;
        /**
         * Store current camera state (fov, position, etc..)
         */
        storeState(): Camera;
        /**
         * Restores the camera state values if it has been stored. You must call storeState() first
         */
        protected _restoreStateValues(): boolean;
        /**
         * Restored camera state. You must call storeState() first
         */
        restoreState(): boolean;
        getClassName(): string;
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        readonly globalPosition: Vector3;
        getActiveMeshes(): SmartArray<AbstractMesh>;
        isActiveMesh(mesh: Mesh): boolean;
        /**
         * Is this camera ready to be used/rendered
         * @param completeCheck defines if a complete check (including post processes) has to be done (false by default)
         * @return true if the camera is ready
         */
        isReady(completeCheck?: boolean): boolean;
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        _isSynchronized(): boolean;
        _isSynchronizedViewMatrix(): boolean;
        _isSynchronizedProjectionMatrix(): boolean;
        attachControl(element: HTMLElement, noPreventDefault?: boolean): void;
        detachControl(element: HTMLElement): void;
        update(): void;
        _checkInputs(): void;
        readonly rigCameras: Camera[];
        readonly rigPostProcess: Nullable<PostProcess>;
        /**
         * Internal, gets the first post proces.
         * @returns the first post process to be run on this camera.
         */
        _getFirstPostProcess(): Nullable<PostProcess>;
        private _cascadePostProcessesToRigCams();
        attachPostProcess(postProcess: PostProcess, insertAt?: Nullable<number>): number;
        detachPostProcess(postProcess: PostProcess): void;
        getWorldMatrix(): Matrix;
        _getViewMatrix(): Matrix;
        getViewMatrix(force?: boolean): Matrix;
        freezeProjectionMatrix(projection?: Matrix): void;
        unfreezeProjectionMatrix(): void;
        getProjectionMatrix(force?: boolean): Matrix;
        getTranformationMatrix(): Matrix;
        private updateFrustumPlanes();
        isInFrustum(target: ICullable): boolean;
        isCompletelyInFrustum(target: ICullable): boolean;
        getForwardRay(length?: number, transform?: Matrix, origin?: Vector3): Ray;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        readonly leftCamera: Nullable<FreeCamera>;
        readonly rightCamera: Nullable<FreeCamera>;
        getLeftTarget(): Nullable<Vector3>;
        getRightTarget(): Nullable<Vector3>;
        setCameraRigMode(mode: number, rigParams: any): void;
        private _getVRProjectionMatrix();
        protected _updateCameraRotationMatrix(): void;
        protected _updateWebVRCameraRotationMatrix(): void;
        /**
         * This function MUST be overwritten by the different WebVR cameras available.
         * The context in which it is running is the RIG camera. So 'this' is the TargetCamera, left or right.
         */
        protected _getWebVRProjectionMatrix(): Matrix;
        /**
         * This function MUST be overwritten by the different WebVR cameras available.
         * The context in which it is running is the RIG camera. So 'this' is the TargetCamera, left or right.
         */
        protected _getWebVRViewMatrix(): Matrix;
        setCameraRigParameter(name: string, value: any): void;
        /**
         * needs to be overridden by children so sub has required properties to be copied
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * May need to be overridden by children
         */
        _updateRigCameras(): void;
        _setupInputs(): void;
        serialize(): any;
        clone(name: string): Camera;
        getDirection(localAxis: Vector3): Vector3;
        getDirectionToRef(localAxis: Vector3, result: Vector3): void;
        static GetConstructorFromName(type: string, name: string, scene: Scene, interaxial_distance?: number, isStereoscopicSideBySide?: boolean): () => Camera;
        computeWorldMatrix(): Matrix;
        static Parse(parsedCamera: any, scene: Scene): Camera;
    }
}

declare module BABYLON {
    var CameraInputTypes: {};
    interface ICameraInput<TCamera extends Camera> {
        camera: Nullable<TCamera>;
        getClassName(): string;
        getSimpleName(): string;
        attachControl: (element: HTMLElement, noPreventDefault?: boolean) => void;
        detachControl: (element: Nullable<HTMLElement>) => void;
        checkInputs?: () => void;
    }
    interface CameraInputsMap<TCamera extends Camera> {
        [name: string]: ICameraInput<TCamera>;
        [idx: number]: ICameraInput<TCamera>;
    }
    class CameraInputsManager<TCamera extends Camera> {
        attached: CameraInputsMap<TCamera>;
        attachedElement: Nullable<HTMLElement>;
        noPreventDefault: boolean;
        camera: TCamera;
        checkInputs: () => void;
        constructor(camera: TCamera);
        /**
         * Add an input method to a camera.
         * builtin inputs example: camera.inputs.addGamepad();
         * custom inputs example: camera.inputs.add(new BABYLON.FreeCameraGamepadInput());
         * @param input camera input method
         */
        add(input: ICameraInput<TCamera>): void;
        /**
         * Remove a specific input method from a camera
         * example: camera.inputs.remove(camera.inputs.attached.mouse);
         * @param inputToRemove camera input method
         */
        remove(inputToRemove: ICameraInput<TCamera>): void;
        removeByType(inputType: string): void;
        private _addCheckInputs(fn);
        attachInput(input: ICameraInput<TCamera>): void;
        attachElement(element: HTMLElement, noPreventDefault?: boolean): void;
        detachElement(element: HTMLElement, disconnect?: boolean): void;
        rebuildInputCheck(): void;
        /**
         * Remove all attached input methods from a camera
         */
        clear(): void;
        serialize(serializedCamera: any): void;
        parse(parsedCamera: any): void;
    }
}

declare module BABYLON {
    /**
     * This is a camera specifically designed to react to device orientation events such as a modern mobile device
     * being tilted forward or back and left or right.
     */
    class DeviceOrientationCamera extends FreeCamera {
        private _initialQuaternion;
        private _quaternionCache;
        /**
         * Creates a new device orientation camera
         * @param name The name of the camera
         * @param position The start position camera
         * @param scene The scene the camera belongs to
         */
        constructor(name: string, position: Vector3, scene: Scene);
        /**
         * Gets the current instance class name ("DeviceOrientationCamera").
         * This helps avoiding instanceof at run time.
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Checks and applies the current values of the inputs to the camera. (Internal use only)
         */
        _checkInputs(): void;
        /**
         * Reset the camera to its default orientation on the specified axis only.
         * @param axis The axis to reset
         */
        resetToCurrentRotation(axis?: Axis): void;
    }
}

declare module BABYLON {
    class FollowCamera extends TargetCamera {
        radius: number;
        rotationOffset: number;
        heightOffset: number;
        cameraAcceleration: number;
        maxCameraSpeed: number;
        lockedTarget: Nullable<AbstractMesh>;
        constructor(name: string, position: Vector3, scene: Scene, lockedTarget?: Nullable<AbstractMesh>);
        private getRadians(degrees);
        private follow(cameraTarget);
        _checkInputs(): void;
        getClassName(): string;
    }
    class ArcFollowCamera extends TargetCamera {
        alpha: number;
        beta: number;
        radius: number;
        target: Nullable<AbstractMesh>;
        private _cartesianCoordinates;
        constructor(name: string, alpha: number, beta: number, radius: number, target: Nullable<AbstractMesh>, scene: Scene);
        private follow();
        _checkInputs(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class FreeCamera extends TargetCamera {
        ellipsoid: Vector3;
        ellipsoidOffset: Vector3;
        checkCollisions: boolean;
        applyGravity: boolean;
        inputs: FreeCameraInputsManager;
        /**
         * Gets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        /**
         * Sets the input sensibility for a mouse input. (default is 2000.0)
         * Higher values reduce sensitivity.
         */
        angularSensibility: number;
        keysUp: number[];
        keysDown: number[];
        keysLeft: number[];
        keysRight: number[];
        onCollide: (collidedMesh: AbstractMesh) => void;
        private _collider;
        private _needMoveForGravity;
        private _oldPosition;
        private _diffPosition;
        private _newPosition;
        _localDirection: Vector3;
        _transformedDirection: Vector3;
        constructor(name: string, position: Vector3, scene: Scene);
        attachControl(element: HTMLElement, noPreventDefault?: boolean): void;
        detachControl(element: HTMLElement): void;
        private _collisionMask;
        collisionMask: number;
        _collideWithWorld(displacement: Vector3): void;
        private _onCollisionPositionChange;
        _checkInputs(): void;
        _decideIfNeedsToMove(): boolean;
        _updatePosition(): void;
        dispose(): void;
        getClassName(): string;
    }
}

declare module BABYLON {
    class FreeCameraInputsManager extends CameraInputsManager<FreeCamera> {
        constructor(camera: FreeCamera);
        addKeyboard(): FreeCameraInputsManager;
        addMouse(touchEnabled?: boolean): FreeCameraInputsManager;
        addGamepad(): FreeCameraInputsManager;
        addDeviceOrientation(): FreeCameraInputsManager;
        addTouch(): FreeCameraInputsManager;
        addVirtualJoystick(): FreeCameraInputsManager;
    }
}

declare module BABYLON {
    class GamepadCamera extends UniversalCamera {
        gamepadAngularSensibility: number;
        gamepadMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

declare module BABYLON {
    class AnaglyphFreeCamera extends FreeCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        getClassName(): string;
    }
    class AnaglyphArcRotateCamera extends ArcRotateCamera {
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, scene: Scene);
        getClassName(): string;
    }
    class AnaglyphGamepadCamera extends GamepadCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        getClassName(): string;
    }
    class AnaglyphUniversalCamera extends UniversalCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, scene: Scene);
        getClassName(): string;
    }
    class StereoscopicFreeCamera extends FreeCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        getClassName(): string;
    }
    class StereoscopicArcRotateCamera extends ArcRotateCamera {
        constructor(name: string, alpha: number, beta: number, radius: number, target: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        getClassName(): string;
    }
    class StereoscopicGamepadCamera extends GamepadCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        getClassName(): string;
    }
    class StereoscopicUniversalCamera extends UniversalCamera {
        constructor(name: string, position: Vector3, interaxialDistance: number, isStereoscopicSideBySide: boolean, scene: Scene);
        getClassName(): string;
    }
}

declare module BABYLON {
    class TargetCamera extends Camera {
        cameraDirection: Vector3;
        cameraRotation: Vector2;
        rotation: Vector3;
        rotationQuaternion: Quaternion;
        speed: number;
        noRotationConstraint: boolean;
        lockedTarget: any;
        _currentTarget: Vector3;
        _viewMatrix: Matrix;
        _camMatrix: Matrix;
        _cameraTransformMatrix: Matrix;
        _cameraRotationMatrix: Matrix;
        private _rigCamTransformMatrix;
        _referencePoint: Vector3;
        private _currentUpVector;
        _transformedReferencePoint: Vector3;
        _lookAtTemp: Matrix;
        _tempMatrix: Matrix;
        _reset: () => void;
        constructor(name: string, position: Vector3, scene: Scene);
        getFrontPosition(distance: number): Vector3;
        _getLockedTargetPosition(): Nullable<Vector3>;
        /**
         * Store current camera state (fov, position, etc..)
         */
        private _storedPosition;
        private _storedRotation;
        private _storedRotationQuaternion;
        storeState(): Camera;
        /**
         * Restored camera state. You must call storeState() first
         */
        _restoreStateValues(): boolean;
        _initCache(): void;
        _updateCache(ignoreParentClass?: boolean): void;
        _isSynchronizedViewMatrix(): boolean;
        _computeLocalCameraSpeed(): number;
        setTarget(target: Vector3): void;
        /**
         * Return the current target position of the camera. This value is expressed in local space.
         */
        getTarget(): Vector3;
        _decideIfNeedsToMove(): boolean;
        _updatePosition(): void;
        _checkInputs(): void;
        protected _updateCameraRotationMatrix(): void;
        _getViewMatrix(): Matrix;
        /**
         * @override
         * Override Camera.createRigCamera
         */
        createRigCamera(name: string, cameraIndex: number): Nullable<Camera>;
        /**
         * @override
         * Override Camera._updateRigCameras
         */
        _updateRigCameras(): void;
        private _getRigCamPosition(halfSpace, result);
        getClassName(): string;
    }
}

declare module BABYLON {
    class TouchCamera extends FreeCamera {
        touchAngularSensibility: number;
        touchMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
        _setupInputs(): void;
    }
}

declare module BABYLON {
    class UniversalCamera extends TouchCamera {
        gamepadAngularSensibility: number;
        gamepadMoveSensibility: number;
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

declare module BABYLON {
    class VirtualJoysticksCamera extends FreeCamera {
        constructor(name: string, position: Vector3, scene: Scene);
        getClassName(): string;
    }
}

interface VRDisplay extends EventTarget {
    /**
     * Dictionary of capabilities describing the VRDisplay.
     */
    readonly capabilities: VRDisplayCapabilities;
    /**
     * z-depth defining the far plane of the eye view frustum
     * enables mapping of values in the render target depth
     * attachment to scene coordinates. Initially set to 10000.0.
     */
    depthFar: number;
    /**
     * z-depth defining the near plane of the eye view frustum
     * enables mapping of values in the render target depth
     * attachment to scene coordinates. Initially set to 0.01.
     */
    depthNear: number;
    /**
     * An identifier for this distinct VRDisplay. Used as an
     * association point in the Gamepad API.
     */
    readonly displayId: number;
    /**
     * A display name, a user-readable name identifying it.
     */
    readonly displayName: string;
    readonly isConnected: boolean;
    readonly isPresenting: boolean;
    /**
     * If this VRDisplay supports room-scale experiences, the optional
     * stage attribute contains details on the room-scale parameters.
     */
    readonly stageParameters: VRStageParameters | null;
    /**
     * Passing the value returned by `requestAnimationFrame` to
     * `cancelAnimationFrame` will unregister the callback.
     */
    cancelAnimationFrame(handle: number): void;
    /**
     * Stops presenting to the VRDisplay.
     */
    exitPresent(): Promise<void>;
    getEyeParameters(whichEye: string): VREyeParameters;
    /**
     * Populates the passed VRFrameData with the information required to render
     * the current frame.
     */
    getFrameData(frameData: VRFrameData): boolean;
    /**
     * Get the layers currently being presented.
     */
    getLayers(): VRLayer[];
    /**
     * Return a VRPose containing the future predicted pose of the VRDisplay
     * when the current frame will be presented. The value returned will not
     * change until JavaScript has returned control to the browser.
     *
     * The VRPose will contain the position, orientation, velocity,
     * and acceleration of each of these properties.
     */
    getPose(): VRPose;
    /**
     * Return the current instantaneous pose of the VRDisplay, with no
     * prediction applied.
     */
    getImmediatePose(): VRPose;
    /**
     * The callback passed to `requestAnimationFrame` will be called
     * any time a new frame should be rendered. When the VRDisplay is
     * presenting the callback will be called at the native refresh
     * rate of the HMD. When not presenting this function acts
     * identically to how window.requestAnimationFrame acts. Content should
     * make no assumptions of frame rate or vsync behavior as the HMD runs
     * asynchronously from other displays and at differing refresh rates.
     */
    requestAnimationFrame(callback: FrameRequestCallback): number;
    /**
     * Begin presenting to the VRDisplay. Must be called in response to a user gesture.
     * Repeat calls while already presenting will update the VRLayers being displayed.
     */
    requestPresent(layers: VRLayer[]): Promise<void>;
    /**
     * Reset the pose for this display, treating its current position and
     * orientation as the "origin/zero" values. VRPose.position,
     * VRPose.orientation, and VRStageParameters.sittingToStandingTransform may be
     * updated when calling resetPose(). This should be called in only
     * sitting-space experiences.
     */
    resetPose(): void;
    /**
     * The VRLayer provided to the VRDisplay will be captured and presented
     * in the HMD. Calling this function has the same effect on the source
     * canvas as any other operation that uses its source image, and canvases
     * created without preserveDrawingBuffer set to true will be cleared.
     */
    submitFrame(pose?: VRPose): void;
}
declare var VRDisplay: {
    prototype: VRDisplay;
    new (): VRDisplay;
};
interface VRLayer {
    leftBounds?: number[] | null;
    rightBounds?: number[] | null;
    source?: HTMLCanvasElement | null;
}
interface VRDisplayCapabilities {
    readonly canPresent: boolean;
    readonly hasExternalDisplay: boolean;
    readonly hasOrientation: boolean;
    readonly hasPosition: boolean;
    readonly maxLayers: number;
}
interface VREyeParameters {
    /** @deprecated */
    readonly fieldOfView: VRFieldOfView;
    readonly offset: Float32Array;
    readonly renderHeight: number;
    readonly renderWidth: number;
}
interface VRFieldOfView {
    readonly downDegrees: number;
    readonly leftDegrees: number;
    readonly rightDegrees: number;
    readonly upDegrees: number;
}
interface VRFrameData {
    readonly leftProjectionMatrix: Float32Array;
    readonly leftViewMatrix: Float32Array;
    readonly pose: VRPose;
    readonly rightProjectionMatrix: Float32Array;
    readonly rightViewMatrix: Float32Array;
    readonly timestamp: number;
}
interface VRPose {
    readonly angularAcceleration: Float32Array | null;
    readonly angularVelocity: Float32Array | null;
    readonly linearAcceleration: Float32Array | null;
    readonly linearVelocity: Float32Array | null;
    readonly orientation: Float32Array | null;
    readonly position: Float32Array | null;
    readonly timestamp: number;
}
interface VRStageParameters {
    sittingToStandingTransform?: Float32Array;
    sizeX?: number;
    sizeY?: number;
}
interface Navigator {
    getVRDisplays(): Promise<VRDisplay[]>;
    readonly activeVRDisplays: ReadonlyArray<VRDisplay>;
}
interface Window {
    onvrdisplayconnected: ((this: Window, ev: Event) => any) | null;
    onvrdisplaydisconnected: ((this: Window, ev: Event) => any) | null;
    onvrdisplaypresentchange: ((this: Window, ev: Event) => any) | null;
    addEventListener(type: "vrdisplayconnected", listener: (ev: Event) => any, useCapture?: boolean): void;
    addEventListener(type: "vrdisplaydisconnected", listener: (ev: Event) => any, useCapture?: boolean): void;
    addEventListener(type: "vrdisplaypresentchange", listener: (ev: Event) => any, useCapture?: boolean): void;
}
interface Gamepad {
    readonly displayId: number;
}

declare module BABYLON {
    class DebugLayer {
        private _scene;
        static InspectorURL: string;
        private _inspector;
        private BJSINSPECTOR;
        constructor(scene: Scene);
        /** Creates the inspector window. */
        private _createInspector(config?);
        isVisible(): boolean;
        hide(): void;
        show(config?: {
            popup?: boolean;
            initialTab?: number;
            parentElement?: HTMLElement;
            newColors?: {
                backgroundColor?: string;
                backgroundColorLighter?: string;
                backgroundColorLighter2?: string;
                backgroundColorLighter3?: string;
                color?: string;
                colorTop?: string;
                colorBot?: string;
            };
        }): void;
    }
}

declare module BABYLON {
    class Debug {
        static AxesViewer: {
            new (scene: Scene, scaleLines?: number): {
                _xline: Vector3[];
                _yline: Vector3[];
                _zline: Vector3[];
                _xmesh: Nullable<LinesMesh>;
                _ymesh: Nullable<LinesMesh>;
                _zmesh: Nullable<LinesMesh>;
                scene: Nullable<Scene>;
                scaleLines: number;
                update(position: Vector3, xaxis: Vector3, yaxis: Vector3, zaxis: Vector3): void;
                dispose(): void;
            };
        };
        static BoneAxesViewer: {
            new (scene: Scene, bone: Bone, mesh: Mesh, scaleLines?: number): {
                mesh: Nullable<Mesh>;
                bone: Nullable<Bone>;
                pos: Vector3;
                xaxis: Vector3;
                yaxis: Vector3;
                zaxis: Vector3;
                update(): void;
                dispose(): void;
                _xline: Vector3[];
                _yline: Vector3[];
                _zline: Vector3[];
                _xmesh: Nullable<LinesMesh>;
                _ymesh: Nullable<LinesMesh>;
                _zmesh: Nullable<LinesMesh>;
                scene: Nullable<Scene>;
                scaleLines: number;
            };
        };
        static PhysicsViewer: {
            new (scene: Scene): {
                _impostors: Nullable<PhysicsImpostor>[];
                _meshes: Nullable<AbstractMesh>[];
                _scene: Nullable<Scene>;
                _numMeshes: number;
                _physicsEnginePlugin: Nullable<IPhysicsEnginePlugin>;
                _renderFunction: () => void;
                _debugBoxMesh: Mesh;
                _debugSphereMesh: Mesh;
                _debugMaterial: StandardMaterial;
                _updateDebugMeshes(): void;
                showImpostor(impostor: PhysicsImpostor): void;
                hideImpostor(impostor: Nullable<PhysicsImpostor>): void;
                _getDebugMaterial(scene: Scene): Material;
                _getDebugBoxMesh(scene: Scene): AbstractMesh;
                _getDebugSphereMesh(scene: Scene): AbstractMesh;
                _getDebugMesh(impostor: PhysicsImpostor, scene: Scene): Nullable<AbstractMesh>;
                dispose(): void;
            };
        };
        static SkeletonViewer: {
            new (skeleton: Skeleton, mesh: AbstractMesh, scene: Scene, autoUpdateBonesMatrices?: boolean, renderingGroupId?: number): {
                color: Color3;
                _scene: Scene;
                _debugLines: Vector3[][];
                _debugMesh: Nullable<LinesMesh>;
                _isEnabled: boolean;
                _renderFunction: () => void;
                skeleton: Skeleton;
                mesh: AbstractMesh;
                autoUpdateBonesMatrices: boolean;
                renderingGroupId: number;
                isEnabled: boolean;
                _getBonePosition(position: Vector3, bone: Bone, meshMat: Matrix, x?: number, y?: number, z?: number): void;
                _getLinesForBonesWithLength(bones: Bone[], meshMat: Matrix): void;
                _getLinesForBonesNoLength(bones: Bone[], meshMat: Matrix): void;
                update(): void;
                dispose(): void;
            };
        };
    }
}

declare module BABYLON {
    class RayHelper {
        ray: Nullable<Ray>;
        private _renderPoints;
        private _renderLine;
        private _renderFunction;
        private _scene;
        private _updateToMeshFunction;
        private _attachedToMesh;
        private _meshSpaceDirection;
        private _meshSpaceOrigin;
        static CreateAndShow(ray: Ray, scene: Scene, color: Color3): RayHelper;
        constructor(ray: Ray);
        show(scene: Scene, color: Color3): void;
        hide(): void;
        private _render();
        attachToMesh(mesh: AbstractMesh, meshSpaceDirection?: Vector3, meshSpaceOrigin?: Vector3, length?: number): void;
        detachFromMesh(): void;
        private _updateToMesh();
        dispose(): void;
    }
}

declare module BABYLON {
    class InstancingAttributeInfo {
        /**
         * Index/offset of the attribute in the vertex shader
         */
        index: number;
        /**
         * size of the attribute, 1, 2, 3 or 4
         */
        attributeSize: number;
        /**
         * type of the attribute, gl.BYTE, gl.UNSIGNED_BYTE, gl.SHORT, gl.UNSIGNED_SHORT, gl.FIXED, gl.FLOAT.
         * default is FLOAT
         */
        attribyteType: number;
        /**
         * normalization of fixed-point data. behavior unclear, use FALSE, default is FALSE
         */
        normalized: boolean;
        /**
         * Offset of the data in the Vertex Buffer acting as the instancing buffer
         */
        offset: number;
        /**
         * Name of the GLSL attribute, for debugging purpose only
         */
        attributeName: string;
    }
    /**
     * Define options used to create a render target texture
     */
    class RenderTargetCreationOptions {
        /**
         * Specifies is mipmaps must be generated
         */
        generateMipMaps?: boolean;
        /** Specifies whether or not a depth should be allocated in the texture (true by default) */
        generateDepthBuffer?: boolean;
        /** Specifies whether or not a stencil should be allocated in the texture (false by default)*/
        generateStencilBuffer?: boolean;
        /** Defines texture type (int by default) */
        type?: number;
        /** Defines sampling mode (trilinear by default) */
        samplingMode?: number;
        /** Defines format (RGBA by default) */
        format?: number;
    }
    /**
     * Define options used to create a depth texture
     */
    class DepthTextureCreationOptions {
        /** Specifies whether or not a stencil should be allocated in the texture */
        generateStencil?: boolean;
        /** Specifies whether or not bilinear filtering is enable on the texture */
        bilinearFiltering?: boolean;
        /** Specifies the comparison function to set on the texture. If 0 or undefined, the texture is not in comparison mode */
        comparisonFunction?: number;
        /** Specifies if the created texture is a cube texture */
        isCube?: boolean;
    }
    /**
     * Regroup several parameters relative to the browser in use
     */
    class EngineCapabilities {
        /** The maximum textures image */
        maxTexturesImageUnits: number;
        maxVertexTextureImageUnits: number;
        maxCombinedTexturesImageUnits: number;
        /** The maximum texture size */
        maxTextureSize: number;
        maxCubemapTextureSize: number;
        maxRenderTextureSize: number;
        maxVertexAttribs: number;
        maxVaryingVectors: number;
        maxVertexUniformVectors: number;
        maxFragmentUniformVectors: number;
        standardDerivatives: boolean;
        s3tc: Nullable<WEBGL_compressed_texture_s3tc>;
        pvrtc: any;
        etc1: any;
        etc2: any;
        astc: any;
        textureFloat: boolean;
        vertexArrayObject: boolean;
        textureAnisotropicFilterExtension: Nullable<EXT_texture_filter_anisotropic>;
        maxAnisotropy: number;
        instancedArrays: boolean;
        uintIndices: boolean;
        highPrecisionShaderSupported: boolean;
        fragmentDepthSupported: boolean;
        textureFloatLinearFiltering: boolean;
        textureFloatRender: boolean;
        textureHalfFloat: boolean;
        textureHalfFloatLinearFiltering: boolean;
        textureHalfFloatRender: boolean;
        textureLOD: boolean;
        drawBuffersExtension: boolean;
        depthTextureExtension: boolean;
        colorBufferFloat: boolean;
        timerQuery: EXT_disjoint_timer_query;
        canUseTimestampForTimerQuery: boolean;
    }
    interface EngineOptions extends WebGLContextAttributes {
        limitDeviceRatio?: number;
        autoEnableWebVR?: boolean;
        disableWebGL2Support?: boolean;
        audioEngine?: boolean;
        deterministicLockstep?: boolean;
        lockstepMaxSteps?: number;
        doNotHandleContextLost?: boolean;
    }
    interface IDisplayChangedEventArgs {
        vrDisplay: Nullable<any>;
        vrSupported: boolean;
    }
    /**
     * The engine class is responsible for interfacing with all lower-level APIs such as WebGL and Audio.
     */
    class Engine {
        /** Use this array to turn off some WebGL2 features on known buggy browsers version */
        static ExceptionList: ({
            key: string;
            capture: string;
            captureConstraint: number;
            targets: string[];
        } | {
            key: string;
            capture: null;
            captureConstraint: null;
            targets: string[];
        })[];
        static Instances: Engine[];
        static readonly LastCreatedEngine: Nullable<Engine>;
        static readonly LastCreatedScene: Nullable<Scene>;
        /**
         * Will flag all materials in all scenes in all engines as dirty to trigger new shader compilation
         */
        static MarkAllMaterialsAsDirty(flag: number, predicate?: (mat: Material) => boolean): void;
        private static _ALPHA_DISABLE;
        private static _ALPHA_ADD;
        private static _ALPHA_COMBINE;
        private static _ALPHA_SUBTRACT;
        private static _ALPHA_MULTIPLY;
        private static _ALPHA_MAXIMIZED;
        private static _ALPHA_ONEONE;
        private static _ALPHA_PREMULTIPLIED;
        private static _ALPHA_PREMULTIPLIED_PORTERDUFF;
        private static _ALPHA_INTERPOLATE;
        private static _ALPHA_SCREENMODE;
        private static _DELAYLOADSTATE_NONE;
        private static _DELAYLOADSTATE_LOADED;
        private static _DELAYLOADSTATE_LOADING;
        private static _DELAYLOADSTATE_NOTLOADED;
        private static _TEXTUREFORMAT_ALPHA;
        private static _TEXTUREFORMAT_LUMINANCE;
        private static _TEXTUREFORMAT_LUMINANCE_ALPHA;
        private static _TEXTUREFORMAT_RGB;
        private static _TEXTUREFORMAT_RGBA;
        private static _TEXTUREFORMAT_R32F;
        private static _TEXTUREFORMAT_RG32F;
        private static _TEXTUREFORMAT_RGB32F;
        private static _TEXTUREFORMAT_RGBA32F;
        private static _TEXTURETYPE_UNSIGNED_INT;
        private static _TEXTURETYPE_FLOAT;
        private static _TEXTURETYPE_HALF_FLOAT;
        private static _NEVER;
        private static _ALWAYS;
        private static _LESS;
        private static _EQUAL;
        private static _LEQUAL;
        private static _GREATER;
        private static _GEQUAL;
        private static _NOTEQUAL;
        static readonly NEVER: number;
        static readonly ALWAYS: number;
        static readonly LESS: number;
        static readonly EQUAL: number;
        static readonly LEQUAL: number;
        static readonly GREATER: number;
        static readonly GEQUAL: number;
        static readonly NOTEQUAL: number;
        private static _KEEP;
        private static _REPLACE;
        private static _INCR;
        private static _DECR;
        private static _INVERT;
        private static _INCR_WRAP;
        private static _DECR_WRAP;
        static readonly KEEP: number;
        static readonly REPLACE: number;
        static readonly INCR: number;
        static readonly DECR: number;
        static readonly INVERT: number;
        static readonly INCR_WRAP: number;
        static readonly DECR_WRAP: number;
        static readonly ALPHA_DISABLE: number;
        static readonly ALPHA_ONEONE: number;
        static readonly ALPHA_ADD: number;
        static readonly ALPHA_COMBINE: number;
        static readonly ALPHA_SUBTRACT: number;
        static readonly ALPHA_MULTIPLY: number;
        static readonly ALPHA_MAXIMIZED: number;
        static readonly ALPHA_PREMULTIPLIED: number;
        static readonly ALPHA_PREMULTIPLIED_PORTERDUFF: number;
        static readonly ALPHA_INTERPOLATE: number;
        static readonly ALPHA_SCREENMODE: number;
        static readonly DELAYLOADSTATE_NONE: number;
        static readonly DELAYLOADSTATE_LOADED: number;
        static readonly DELAYLOADSTATE_LOADING: number;
        static readonly DELAYLOADSTATE_NOTLOADED: number;
        static readonly TEXTUREFORMAT_ALPHA: number;
        static readonly TEXTUREFORMAT_LUMINANCE: number;
        /**
         * R32F
         */
        static readonly TEXTUREFORMAT_R32F: number;
        /**
         * RG32F
         */
        static readonly TEXTUREFORMAT_RG32F: number;
        /**
         * RGB32F
         */
        static readonly TEXTUREFORMAT_RGB32F: number;
        /**
         * RGBA32F
         */
        static readonly TEXTUREFORMAT_RGBA32F: number;
        static readonly TEXTUREFORMAT_LUMINANCE_ALPHA: number;
        static readonly TEXTUREFORMAT_RGB: number;
        static readonly TEXTUREFORMAT_RGBA: number;
        static readonly TEXTURETYPE_UNSIGNED_INT: number;
        static readonly TEXTURETYPE_FLOAT: number;
        static readonly TEXTURETYPE_HALF_FLOAT: number;
        private static _SCALEMODE_FLOOR;
        private static _SCALEMODE_NEAREST;
        private static _SCALEMODE_CEILING;
        static readonly SCALEMODE_FLOOR: number;
        static readonly SCALEMODE_NEAREST: number;
        static readonly SCALEMODE_CEILING: number;
        static readonly Version: string;
        static CollisionsEpsilon: number;
        static CodeRepository: string;
        static ShadersRepository: string;
        forcePOTTextures: boolean;
        isFullscreen: boolean;
        isPointerLock: boolean;
        cullBackFaces: boolean;
        renderEvenInBackground: boolean;
        preventCacheWipeBetweenFrames: boolean;
        /**
         * To enable/disable IDB support and avoid XHR on .manifest
         **/
        enableOfflineSupport: boolean;
        scenes: Scene[];
        postProcesses: PostProcess[];
        /**
         * Observable event triggered each time the rendering canvas is resized
         */
        onResizeObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas loses focus
         */
        onCanvasBlurObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas gains focus
         */
        onCanvasFocusObservable: Observable<Engine>;
        /**
         * Observable event triggered each time the canvas receives pointerout event
         */
        onCanvasPointerOutObservable: Observable<PointerEvent>;
        /**
         * Observable event triggered before each texture is initialized
         */
        onBeforeTextureInitObservable: Observable<Texture>;
        private _vrDisplay;
        private _vrSupported;
        private _oldSize;
        private _oldHardwareScaleFactor;
        private _vrExclusivePointerMode;
        private _webVRInitPromise;
        readonly isInVRExclusivePointerMode: boolean;
        disableUniformBuffers: boolean;
        _uniformBuffers: UniformBuffer[];
        readonly supportsUniformBuffers: boolean;
        /**
         * Observable raised when the engine begins a new frame
         */
        onBeginFrameObservable: Observable<Engine>;
        /**
         * Observable raised when the engine ends the current frame
         */
        onEndFrameObservable: Observable<Engine>;
        /**
         * Observable raised when the engine is about to compile a shader
         */
        onBeforeShaderCompilationObservable: Observable<Engine>;
        /**
         * Observable raised when the engine has jsut compiled a shader
         */
        onAfterShaderCompilationObservable: Observable<Engine>;
        private _gl;
        private _renderingCanvas;
        private _windowIsBackground;
        private _webGLVersion;
        readonly needPOTTextures: boolean;
        private _badOS;
        readonly badOS: boolean;
        private _badDesktopOS;
        readonly badDesktopOS: boolean;
        /**
         * Gets or sets a value indicating if we want to disable texture binding optmization.
         * This could be required on some buggy drivers which wants to have textures bound in a progressive order
         * By default Babylon.js will try to let textures bound where they are and only update the samplers to point where the texture is.
         */
        disableTextureBindingOptimization: boolean;
        static audioEngine: AudioEngine;
        private _onFocus;
        private _onBlur;
        private _onCanvasPointerOut;
        private _onCanvasBlur;
        private _onCanvasFocus;
        private _onFullscreenChange;
        private _onPointerLockChange;
        private _onVRDisplayPointerRestricted;
        private _onVRDisplayPointerUnrestricted;
        private _onVrDisplayConnect;
        private _onVrDisplayDisconnect;
        private _onVrDisplayPresentChange;
        onVRDisplayChangedObservable: Observable<IDisplayChangedEventArgs>;
        onVRRequestPresentComplete: Observable<boolean>;
        onVRRequestPresentStart: Observable<Engine>;
        private _hardwareScalingLevel;
        protected _caps: EngineCapabilities;
        private _pointerLockRequested;
        private _isStencilEnable;
        private _colorWrite;
        private _loadingScreen;
        _drawCalls: PerfCounter;
        _textureCollisions: PerfCounter;
        private _glVersion;
        private _glRenderer;
        private _glVendor;
        private _videoTextureSupported;
        private _renderingQueueLaunched;
        private _activeRenderLoops;
        private _deterministicLockstep;
        private _lockstepMaxSteps;
        onContextLostObservable: Observable<Engine>;
        onContextRestoredObservable: Observable<Engine>;
        private _onContextLost;
        private _onContextRestored;
        private _contextWasLost;
        private _doNotHandleContextLost;
        private _performanceMonitor;
        private _fps;
        private _deltaTime;
        /**
         * Turn this value on if you want to pause FPS computation when in background
         */
        disablePerformanceMonitorInBackground: boolean;
        readonly performanceMonitor: PerformanceMonitor;
        protected _depthCullingState: _DepthCullingState;
        protected _stencilState: _StencilState;
        protected _alphaState: _AlphaState;
        protected _alphaMode: number;
        private _internalTexturesCache;
        protected _activeChannel: number;
        private _currentTextureChannel;
        protected _boundTexturesCache: {
            [key: string]: Nullable<InternalTexture>;
        };
        protected _currentEffect: Nullable<Effect>;
        protected _currentProgram: Nullable<WebGLProgram>;
        private _compiledEffects;
        private _vertexAttribArraysEnabled;
        protected _cachedViewport: Nullable<Viewport>;
        private _cachedVertexArrayObject;
        protected _cachedVertexBuffers: any;
        protected _cachedIndexBuffer: Nullable<WebGLBuffer>;
        protected _cachedEffectForVertexBuffers: Nullable<Effect>;
        protected _currentRenderTarget: Nullable<InternalTexture>;
        private _uintIndicesCurrentlySet;
        private _currentBoundBuffer;
        protected _currentFramebuffer: Nullable<WebGLFramebuffer>;
        private _currentBufferPointers;
        private _currentInstanceLocations;
        private _currentInstanceBuffers;
        private _textureUnits;
        private _firstBoundInternalTextureTracker;
        private _lastBoundInternalTextureTracker;
        private _workingCanvas;
        private _workingContext;
        private _rescalePostProcess;
        private _dummyFramebuffer;
        private _externalData;
        private _bindedRenderFunction;
        private _vaoRecordInProgress;
        private _mustWipeVertexAttributes;
        private _emptyTexture;
        private _emptyCubeTexture;
        private _emptyTexture3D;
        private _frameHandler;
        private _nextFreeTextureSlots;
        private _maxSimultaneousTextures;
        private _activeRequests;
        private _texturesSupported;
        private _textureFormatInUse;
        readonly texturesSupported: Array<string>;
        readonly textureFormatInUse: Nullable<string>;
        readonly currentViewport: Nullable<Viewport>;
        readonly emptyTexture: InternalTexture;
        readonly emptyTexture3D: InternalTexture;
        readonly emptyCubeTexture: InternalTexture;
        /**
         * @constructor
         * @param canvasOrContext defines the canvas or WebGL context to use for rendering
         * @param antialias defines enable antialiasing (default: false)
         * @param options defines further options to be sent to the getContext() function
         * @param adaptToDeviceRatio defines whether to adapt to the device's viewport characteristics (default: false)
         */
        constructor(canvasOrContext: Nullable<HTMLCanvasElement | WebGLRenderingContext>, antialias?: boolean, options?: EngineOptions, adaptToDeviceRatio?: boolean);
        private _rebuildInternalTextures();
        private _rebuildEffects();
        private _rebuildBuffers();
        private _initGLContext();
        readonly webGLVersion: number;
        /**
         * Returns true if the stencil buffer has been enabled through the creation option of the context.
         */
        readonly isStencilEnable: boolean;
        private _prepareWorkingCanvas();
        resetTextureCache(): void;
        isDeterministicLockStep(): boolean;
        getLockstepMaxSteps(): number;
        getGlInfo(): {
            vendor: string;
            renderer: string;
            version: string;
        };
        getAspectRatio(camera: Camera, useScreen?: boolean): number;
        getRenderWidth(useScreen?: boolean): number;
        getRenderHeight(useScreen?: boolean): number;
        getRenderingCanvas(): Nullable<HTMLCanvasElement>;
        getRenderingCanvasClientRect(): Nullable<ClientRect>;
        setHardwareScalingLevel(level: number): void;
        getHardwareScalingLevel(): number;
        getLoadedTexturesCache(): InternalTexture[];
        getCaps(): EngineCapabilities;
        /** The number of draw calls submitted last frame */
        readonly drawCalls: number;
        readonly drawCallsPerfCounter: Nullable<PerfCounter>;
        getDepthFunction(): Nullable<number>;
        setDepthFunction(depthFunc: number): void;
        setDepthFunctionToGreater(): void;
        setDepthFunctionToGreaterOrEqual(): void;
        setDepthFunctionToLess(): void;
        setDepthFunctionToLessOrEqual(): void;
        getStencilBuffer(): boolean;
        setStencilBuffer(enable: boolean): void;
        getStencilMask(): number;
        setStencilMask(mask: number): void;
        getStencilFunction(): number;
        getStencilFunctionReference(): number;
        getStencilFunctionMask(): number;
        setStencilFunction(stencilFunc: number): void;
        setStencilFunctionReference(reference: number): void;
        setStencilFunctionMask(mask: number): void;
        getStencilOperationFail(): number;
        getStencilOperationDepthFail(): number;
        getStencilOperationPass(): number;
        setStencilOperationFail(operation: number): void;
        setStencilOperationDepthFail(operation: number): void;
        setStencilOperationPass(operation: number): void;
        setDitheringState(value: boolean): void;
        setRasterizerState(value: boolean): void;
        /**
         * stop executing a render loop function and remove it from the execution array
         * @param {Function} [renderFunction] the function to be removed. If not provided all functions will be removed.
         */
        stopRenderLoop(renderFunction?: () => void): void;
        _renderLoop(): void;
        /**
         * Register and execute a render loop. The engine can have more than one render function.
         * @param {Function} renderFunction - the function to continuously execute starting the next render loop.
         * @example
         * engine.runRenderLoop(function () {
         *      scene.render()
         * })
         */
        runRenderLoop(renderFunction: () => void): void;
        /**
         * Toggle full screen mode.
         * @param {boolean} requestPointerLock - should a pointer lock be requested from the user
         * @param {any} options - an options object to be sent to the requestFullscreen function
         */
        switchFullscreen(requestPointerLock: boolean): void;
        clear(color: Nullable<Color4>, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        scissorClear(x: number, y: number, width: number, height: number, clearColor: Color4): void;
        /**
         * Set the WebGL's viewport
         * @param {BABYLON.Viewport} viewport - the viewport element to be used.
         * @param {number} [requiredWidth] - the width required for rendering. If not provided the rendering canvas' width is used.
         * @param {number} [requiredHeight] - the height required for rendering. If not provided the rendering canvas' height is used.
         */
        setViewport(viewport: Viewport, requiredWidth?: number, requiredHeight?: number): void;
        /**
         * Directly set the WebGL Viewport
         * The x, y, width & height are directly passed to the WebGL call
         * @return the current viewport Object (if any) that is being replaced by this call. You can restore this viewport later on to go back to the original state.
         */
        setDirectViewport(x: number, y: number, width: number, height: number): Nullable<Viewport>;
        beginFrame(): void;
        endFrame(): void;
        /**
         * resize the view according to the canvas' size.
         * @example
         *   window.addEventListener("resize", function () {
         *      engine.resize();
         *   });
         */
        resize(): void;
        /**
         * force a specific size of the canvas
         * @param {number} width - the new canvas' width
         * @param {number} height - the new canvas' height
         */
        setSize(width: number, height: number): void;
        isVRDevicePresent(): boolean;
        getVRDevice(): any;
        /**
         * Initializes a webVR display and starts listening to display change events.
         * The onVRDisplayChangedObservable will be notified upon these changes.
         * @returns The onVRDisplayChangedObservable.
         */
        initWebVR(): Observable<IDisplayChangedEventArgs>;
        /**
         * Initializes a webVR display and starts listening to display change events.
         * The onVRDisplayChangedObservable will be notified upon these changes.
         * @returns A promise containing a VRDisplay and if vr is supported.
         */
        initWebVRAsync(): Promise<IDisplayChangedEventArgs>;
        enableVR(): void;
        disableVR(): void;
        private _onVRFullScreenTriggered;
        private _getVRDisplaysAsync();
        /**
         * Binds the frame buffer to the specified texture.
         * @param texture The texture to render to or null for the default canvas
         * @param faceIndex The face of the texture to render to in case of cube texture
         * @param requiredWidth The width of the target to render to
         * @param requiredHeight The height of the target to render to
         * @param forceFullscreenViewport Forces the viewport to be the entire texture/screen if true
         * @param depthStencilTexture The depth stencil texture to use to render
         */
        bindFramebuffer(texture: InternalTexture, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean, depthStencilTexture?: InternalTexture): void;
        private bindUnboundFramebuffer(framebuffer);
        unBindFramebuffer(texture: InternalTexture, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        unBindMultiColorAttachmentFramebuffer(textures: InternalTexture[], disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        generateMipMapsForCubemap(texture: InternalTexture): void;
        flushFramebuffer(): void;
        restoreDefaultFramebuffer(): void;
        createUniformBuffer(elements: FloatArray): WebGLBuffer;
        createDynamicUniformBuffer(elements: FloatArray): WebGLBuffer;
        updateUniformBuffer(uniformBuffer: WebGLBuffer, elements: FloatArray, offset?: number, count?: number): void;
        private _resetVertexBufferBinding();
        createVertexBuffer(vertices: FloatArray): WebGLBuffer;
        createDynamicVertexBuffer(vertices: FloatArray): WebGLBuffer;
        updateDynamicIndexBuffer(indexBuffer: WebGLBuffer, indices: IndicesArray, offset?: number): void;
        updateDynamicVertexBuffer(vertexBuffer: WebGLBuffer, vertices: FloatArray, offset?: number, count?: number): void;
        private _resetIndexBufferBinding();
        createIndexBuffer(indices: IndicesArray, updatable?: boolean): WebGLBuffer;
        bindArrayBuffer(buffer: Nullable<WebGLBuffer>): void;
        bindUniformBuffer(buffer: Nullable<WebGLBuffer>): void;
        bindUniformBufferBase(buffer: WebGLBuffer, location: number): void;
        bindUniformBlock(shaderProgram: WebGLProgram, blockName: string, index: number): void;
        private bindIndexBuffer(buffer);
        private bindBuffer(buffer, target);
        updateArrayBuffer(data: Float32Array): void;
        private vertexAttribPointer(buffer, indx, size, type, normalized, stride, offset);
        private _bindIndexBufferWithCache(indexBuffer);
        private _bindVertexBuffersAttributes(vertexBuffers, effect);
        recordVertexArrayObject(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: Nullable<WebGLBuffer>, effect: Effect): WebGLVertexArrayObject;
        bindVertexArrayObject(vertexArrayObject: WebGLVertexArrayObject, indexBuffer: Nullable<WebGLBuffer>): void;
        bindBuffersDirectly(vertexBuffer: WebGLBuffer, indexBuffer: WebGLBuffer, vertexDeclaration: number[], vertexStrideSize: number, effect: Effect): void;
        private _unbindVertexArrayObject();
        bindBuffers(vertexBuffers: {
            [key: string]: Nullable<VertexBuffer>;
        }, indexBuffer: Nullable<WebGLBuffer>, effect: Effect): void;
        unbindInstanceAttributes(): void;
        releaseVertexArrayObject(vao: WebGLVertexArrayObject): void;
        _releaseBuffer(buffer: WebGLBuffer): boolean;
        createInstancesBuffer(capacity: number): WebGLBuffer;
        deleteInstancesBuffer(buffer: WebGLBuffer): void;
        updateAndBindInstancesBuffer(instancesBuffer: WebGLBuffer, data: Float32Array, offsetLocations: number[] | InstancingAttributeInfo[]): void;
        applyStates(): void;
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawPointClouds(verticesStart: number, verticesCount: number, instancesCount?: number): void;
        drawUnIndexed(useTriangles: boolean, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        private _drawMode(fillMode);
        _releaseEffect(effect: Effect): void;
        _deleteProgram(program: WebGLProgram): void;
        /**
         * @param baseName The base name of the effect (The name of file without .fragment.fx or .vertex.fx)
         * @param samplers An array of string used to represent textures
         */
        createEffect(baseName: any, attributesNamesOrOptions: string[] | EffectCreationOptions, uniformsNamesOrEngine: string[] | Engine, samplers?: string[], defines?: string, fallbacks?: EffectFallbacks, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void, indexParameters?: any): Effect;
        createEffectForParticles(fragmentName: string, uniformsNames?: string[], samplers?: string[], defines?: string, fallbacks?: EffectFallbacks, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): Effect;
        createRawShaderProgram(vertexCode: string, fragmentCode: string, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        createShaderProgram(vertexCode: string, fragmentCode: string, defines: Nullable<string>, context?: WebGLRenderingContext, transformFeedbackVaryings?: Nullable<string[]>): WebGLProgram;
        private _createShaderProgram(vertexShader, fragmentShader, context, transformFeedbackVaryings?);
        getUniforms(shaderProgram: WebGLProgram, uniformsNames: string[]): Nullable<WebGLUniformLocation>[];
        getAttributes(shaderProgram: WebGLProgram, attributesNames: string[]): number[];
        enableEffect(effect: Nullable<Effect>): void;
        setIntArray(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        setIntArray2(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        setIntArray3(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        setIntArray4(uniform: Nullable<WebGLUniformLocation>, array: Int32Array): void;
        setFloatArray(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        setFloatArray2(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        setFloatArray3(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        setFloatArray4(uniform: Nullable<WebGLUniformLocation>, array: Float32Array): void;
        setArray(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        setArray2(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        setArray3(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        setArray4(uniform: Nullable<WebGLUniformLocation>, array: number[]): void;
        setMatrices(uniform: Nullable<WebGLUniformLocation>, matrices: Float32Array): void;
        setMatrix(uniform: Nullable<WebGLUniformLocation>, matrix: Matrix): void;
        setMatrix3x3(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): void;
        setMatrix2x2(uniform: Nullable<WebGLUniformLocation>, matrix: Float32Array): void;
        setInt(uniform: Nullable<WebGLUniformLocation>, value: number): void;
        setFloat(uniform: Nullable<WebGLUniformLocation>, value: number): void;
        setFloat2(uniform: Nullable<WebGLUniformLocation>, x: number, y: number): void;
        setFloat3(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number): void;
        setBool(uniform: Nullable<WebGLUniformLocation>, bool: number): void;
        setFloat4(uniform: Nullable<WebGLUniformLocation>, x: number, y: number, z: number, w: number): void;
        setColor3(uniform: Nullable<WebGLUniformLocation>, color3: Color3): void;
        setColor4(uniform: Nullable<WebGLUniformLocation>, color3: Color3, alpha: number): void;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniform defines the uniform location
         * @param color4 defines the value to be set
         */
        setDirectColor4(uniform: Nullable<WebGLUniformLocation>, color4: Color4): void;
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean): void;
        setZOffset(value: number): void;
        getZOffset(): number;
        setDepthBuffer(enable: boolean): void;
        getDepthWrite(): boolean;
        setDepthWrite(enable: boolean): void;
        setColorWrite(enable: boolean): void;
        getColorWrite(): boolean;
        setAlphaConstants(r: number, g: number, b: number, a: number): void;
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        getAlphaMode(): number;
        wipeCaches(bruteForce?: boolean): void;
        /**
         * Set the compressed texture format to use, based on the formats you have, and the formats
         * supported by the hardware / browser.
         *
         * Khronos Texture Container (.ktx) files are used to support this.  This format has the
         * advantage of being specifically designed for OpenGL.  Header elements directly correspond
         * to API arguments needed to compressed textures.  This puts the burden on the container
         * generator to house the arcane code for determining these for current & future formats.
         *
         * for description see https://www.khronos.org/opengles/sdk/tools/KTX/
         * for file layout see https://www.khronos.org/opengles/sdk/tools/KTX/file_format_spec/
         *
         * Note: The result of this call is not taken into account when a texture is base64.
         *
         * @param {Array<string>} formatsAvailable- The list of those format families you have created
         * on your server.  Syntax: '-' + format family + '.ktx'.  (Case and order do not matter.)
         *
         * Current families are astc, dxt, pvrtc, etc2, & etc1.
         * @returns The extension selected.
         */
        setTextureFormatToUse(formatsAvailable: Array<string>): Nullable<string>;
        _createTexture(): WebGLTexture;
        /**
         * Usually called from BABYLON.Texture.ts.  Passed information to create a WebGLTexture.
         * @param {string} urlArg- This contains one of the following:
         *                         1. A conventional http URL, e.g. 'http://...' or 'file://...'
         *                         2. A base64 string of in-line texture data, e.g. 'data:image/jpg;base64,/...'
         *                         3. An indicator that data being passed using the buffer parameter, e.g. 'data:mytexture.jpg'
         *
         * @param {boolean} noMipmap- When true, no mipmaps shall be generated.  Ignored for compressed textures.  They must be in the file.
         * @param {boolean} invertY- When true, image is flipped when loaded.  You probably want true. Ignored for compressed textures.  Must be flipped in the file.
         * @param {Scene} scene- Needed for loading to the correct scene.
         * @param {number} samplingMode- Mode with should be used sample / access the texture.  Default: TRILINEAR
         * @param {callback} onLoad- Optional callback to be called upon successful completion.
         * @param {callback} onError- Optional callback to be called upon failure.
         * @param {ArrayBuffer | HTMLImageElement} buffer- A source of a file previously fetched as either an ArrayBuffer (compressed or image format) or HTMLImageElement (image format)
         * @param {WebGLTexture} fallback- An internal argument in case the function must be called again, due to etc1 not having alpha capabilities.
         * @param {number} format-  Internal format.  Default: RGB when extension is '.jpg' else RGBA.  Ignored for compressed textures.
         *
         * @returns {WebGLTexture} for assignment back into BABYLON.Texture
         */
        createTexture(urlArg: Nullable<string>, noMipmap: boolean, invertY: boolean, scene: Nullable<Scene>, samplingMode?: number, onLoad?: Nullable<() => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<ArrayBuffer | HTMLImageElement>, fallBack?: Nullable<InternalTexture>, format?: Nullable<number>): InternalTexture;
        private _rescaleTexture(source, destination, scene, internalFormat, onComplete);
        updateRawTexture(texture: Nullable<InternalTexture>, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>, type?: number): void;
        createRawTexture(data: Nullable<ArrayBufferView>, width: number, height: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>, type?: number): InternalTexture;
        createDynamicTexture(width: number, height: number, generateMipMaps: boolean, samplingMode: number): InternalTexture;
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        updateDynamicTexture(texture: Nullable<InternalTexture>, canvas: HTMLCanvasElement, invertY: boolean, premulAlpha?: boolean, format?: number): void;
        updateVideoTexture(texture: Nullable<InternalTexture>, video: HTMLVideoElement, invertY: boolean): void;
        /**
         * Updates a depth texture Comparison Mode and Function.
         * If the comparison Function is equal to 0, the mode will be set to none.
         * Otherwise, this only works in webgl 2 and requires a shadow sampler in the shader.
         * @param texture The texture to set the comparison function for
         * @param comparisonFunction The comparison function to set, 0 if no comparison required
         */
        updateTextureComparisonFunction(texture: InternalTexture, comparisonFunction: number): void;
        private _setupDepthStencilTexture(internalTexture, size, generateStencil, bilinearFiltering, comparisonFunction);
        /**
         * Creates a depth stencil texture.
         * This is only available in WebGL 2 or with the depth texture extension available.
         * @param size The size of face edge in the texture.
         * @param options The options defining the texture.
         * @returns The texture
         */
        createDepthStencilTexture(size: number | {
            width: number;
            height: number;
        }, options: DepthTextureCreationOptions): InternalTexture;
        /**
         * Creates a depth stencil texture.
         * This is only available in WebGL 2 or with the depth texture extension available.
         * @param size The size of face edge in the texture.
         * @param options The options defining the texture.
         * @returns The texture
         */
        private _createDepthStencilTexture(size, options);
        /**
         * Creates a depth stencil cube texture.
         * This is only available in WebGL 2.
         * @param size The size of face edge in the cube texture.
         * @param options The options defining the cube texture.
         * @returns The cube texture
         */
        private _createDepthStencilCubeTexture(size, options);
        /**
         * Sets the frame buffer Depth / Stencil attachement of the render target to the defined depth stencil texture.
         * @param renderTarget The render target to set the frame buffer for
         */
        setFrameBufferDepthStencilTexture(renderTarget: RenderTargetTexture): void;
        createRenderTargetTexture(size: number | {
            width: number;
            height: number;
        }, options: boolean | RenderTargetCreationOptions): InternalTexture;
        createMultipleRenderTarget(size: any, options: any): InternalTexture[];
        private _setupFramebufferDepthAttachments(generateStencilBuffer, generateDepthBuffer, width, height, samples?);
        updateRenderTargetTextureSampleCount(texture: Nullable<InternalTexture>, samples: number): number;
        updateMultipleRenderTargetTextureSampleCount(textures: Nullable<InternalTexture[]>, samples: number): number;
        _uploadDataToTexture(target: number, lod: number, internalFormat: number, width: number, height: number, format: number, type: number, data: ArrayBufferView): void;
        _uploadCompressedDataToTexture(target: number, lod: number, internalFormat: number, width: number, height: number, data: ArrayBufferView): void;
        createRenderTargetCubeTexture(size: number, options?: Partial<RenderTargetCreationOptions>): InternalTexture;
        createPrefilteredCubeTexture(rootUrl: string, scene: Nullable<Scene>, scale: number, offset: number, onLoad?: Nullable<(internalTexture: Nullable<InternalTexture>) => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, forcedExtension?: any): InternalTexture;
        createCubeTexture(rootUrl: string, scene: Nullable<Scene>, files: Nullable<string[]>, noMipmap?: boolean, onLoad?: Nullable<(data?: any) => void>, onError?: Nullable<(message?: string, exception?: any) => void>, format?: number, forcedExtension?: any): InternalTexture;
        private setCubeMapTextureParams(gl, loadMipmap);
        updateRawCubeTexture(texture: InternalTexture, data: ArrayBufferView[], format: number, type: number, invertY: boolean, compression?: Nullable<string>, level?: number): void;
        createRawCubeTexture(data: Nullable<ArrayBufferView[]>, size: number, format: number, type: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>): InternalTexture;
        createRawCubeTextureFromUrl(url: string, scene: Scene, size: number, format: number, type: number, noMipmap: boolean, callback: (ArrayBuffer: ArrayBuffer) => Nullable<ArrayBufferView[]>, mipmmapGenerator: Nullable<((faces: ArrayBufferView[]) => ArrayBufferView[][])>, onLoad?: Nullable<() => void>, onError?: Nullable<(message?: string, exception?: any) => void>, samplingMode?: number, invertY?: boolean): InternalTexture;
        updateRawTexture3D(texture: InternalTexture, data: Nullable<ArrayBufferView>, format: number, invertY: boolean, compression?: Nullable<string>): void;
        createRawTexture3D(data: Nullable<ArrayBufferView>, width: number, height: number, depth: number, format: number, generateMipMaps: boolean, invertY: boolean, samplingMode: number, compression?: Nullable<string>): InternalTexture;
        private _prepareWebGLTextureContinuation(texture, scene, noMipmap, isCompressed, samplingMode);
        private _prepareWebGLTexture(texture, scene, width, height, invertY, noMipmap, isCompressed, processFunction, samplingMode?);
        private _convertRGBtoRGBATextureData(rgbData, width, height, textureType);
        _releaseFramebufferObjects(texture: InternalTexture): void;
        _releaseTexture(texture: InternalTexture): void;
        private setProgram(program);
        private _boundUniforms;
        bindSamplers(effect: Effect): void;
        private _moveBoundTextureOnTop(internalTexture);
        private _getCorrectTextureChannel(channel, internalTexture);
        private _linkTrackers(previous, next);
        private _removeDesignatedSlot(internalTexture);
        private _activateCurrentTexture();
        protected _bindTextureDirectly(target: number, texture: Nullable<InternalTexture>, forTextureDataUpdate?: boolean): void;
        _bindTexture(channel: number, texture: Nullable<InternalTexture>): void;
        setTextureFromPostProcess(channel: number, postProcess: Nullable<PostProcess>): void;
        /**
         * Binds the output of the passed in post process to the texture channel specified
         * @param channel The channel the texture should be bound to
         * @param postProcess The post process which's output should be bound
         */
        setTextureFromPostProcessOutput(channel: number, postProcess: Nullable<PostProcess>): void;
        unbindAllTextures(): void;
        /**
         * Sets a texture to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The texture to apply
         */
        setTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<BaseTexture>): void;
        /**
         * Sets a depth stencil texture from a render target to the according uniform.
         * @param channel The texture channel
         * @param uniform The uniform to set
         * @param texture The render target texture containing the depth stencil texture to apply
         */
        setDepthStencilTexture(channel: number, uniform: Nullable<WebGLUniformLocation>, texture: Nullable<RenderTargetTexture>): void;
        private _bindSamplerUniformToChannel(sourceSlot, destination);
        private _setTexture(channel, texture, isPartOfTextureArray?, depthStencilTexture?);
        setTextureArray(channel: number, uniform: Nullable<WebGLUniformLocation>, textures: BaseTexture[]): void;
        _setAnisotropicLevel(key: number, texture: BaseTexture): void;
        readPixels(x: number, y: number, width: number, height: number): Uint8Array;
        /**
         * Add an externaly attached data from its key.
         * This method call will fail and return false, if such key already exists.
         * If you don't care and just want to get the data no matter what, use the more convenient getOrAddExternalDataWithFactory() method.
         * @param key the unique key that identifies the data
         * @param data the data object to associate to the key for this Engine instance
         * @return true if no such key were already present and the data was added successfully, false otherwise
         */
        addExternalData<T>(key: string, data: T): boolean;
        /**
         * Get an externaly attached data from its key
         * @param key the unique key that identifies the data
         * @return the associated data, if present (can be null), or undefined if not present
         */
        getExternalData<T>(key: string): T;
        /**
         * Get an externaly attached data from its key, create it using a factory if it's not already present
         * @param key the unique key that identifies the data
         * @param factory the factory that will be called to create the instance if and only if it doesn't exists
         * @return the associated data, can be null if the factory returned null.
         */
        getOrAddExternalDataWithFactory<T>(key: string, factory: (k: string) => T): T;
        /**
         * Remove an externaly attached data from the Engine instance
         * @param key the unique key that identifies the data
         * @return true if the data was successfully removed, false if it doesn't exist
         */
        removeExternalData(key: string): boolean;
        unbindAllAttributes(): void;
        releaseEffects(): void;
        dispose(): void;
        displayLoadingUI(): void;
        hideLoadingUI(): void;
        loadingScreen: ILoadingScreen;
        loadingUIText: string;
        loadingUIBackgroundColor: string;
        attachContextLostEvent(callback: ((event: WebGLContextEvent) => void)): void;
        attachContextRestoredEvent(callback: ((event: WebGLContextEvent) => void)): void;
        getVertexShaderSource(program: WebGLProgram): Nullable<string>;
        getFragmentShaderSource(program: WebGLProgram): Nullable<string>;
        getError(): number;
        getFps(): number;
        getDeltaTime(): number;
        private _measureFps();
        _readTexturePixels(texture: InternalTexture, width: number, height: number, faceIndex?: number): ArrayBufferView;
        private _canRenderToFloatFramebuffer();
        private _canRenderToHalfFloatFramebuffer();
        private _canRenderToFramebuffer(type);
        _getWebGLTextureType(type: number): number;
        private _getInternalFormat(format);
        /** @ignore */
        _getRGBABufferInternalSizedFormat(type: number, format?: number): number;
        _getRGBAMultiSampleBufferFormat(type: number): number;
        createQuery(): WebGLQuery;
        deleteQuery(query: WebGLQuery): Engine;
        isQueryResultAvailable(query: WebGLQuery): boolean;
        getQueryResult(query: WebGLQuery): number;
        beginOcclusionQuery(algorithmType: number, query: WebGLQuery): Engine;
        endOcclusionQuery(algorithmType: number): Engine;
        private _createTimeQuery();
        private _deleteTimeQuery(query);
        private _getTimeQueryResult(query);
        private _getTimeQueryAvailability(query);
        private _currentNonTimestampToken;
        startTimeQuery(): Nullable<_TimeToken>;
        endTimeQuery(token: _TimeToken): int;
        private getGlAlgorithmType(algorithmType);
        createTransformFeedback(): WebGLTransformFeedback;
        deleteTransformFeedback(value: WebGLTransformFeedback): void;
        bindTransformFeedback(value: Nullable<WebGLTransformFeedback>): void;
        beginTransformFeedback(usePoints?: boolean): void;
        endTransformFeedback(): void;
        setTranformFeedbackVaryings(program: WebGLProgram, value: string[]): void;
        bindTransformFeedbackBuffer(value: Nullable<WebGLBuffer>): void;
        _loadFile(url: string, onSuccess: (data: string | ArrayBuffer, responseURL?: string) => void, onProgress?: (data: any) => void, database?: Database, useArrayBuffer?: boolean, onError?: (request?: XMLHttpRequest, exception?: any) => void): IFileRequest;
        /** @ignore */
        _loadFileAsync(url: string, database?: Database, useArrayBuffer?: boolean): Promise<string | ArrayBuffer>;
        private _partialLoadFile(url, index, loadedFiles, scene, onfinish, onErrorCallBack?);
        private _cascadeLoadFiles(scene, onfinish, files, onError?);
        static isSupported(): boolean;
    }
}

declare module BABYLON {
    class NullEngineOptions {
        renderWidth: number;
        renderHeight: number;
        textureSize: number;
        deterministicLockstep: boolean;
        lockstepMaxSteps: number;
    }
    /**
     * The null engine class provides support for headless version of babylon.js.
     * This can be used in server side scenario or for testing purposes
     */
    class NullEngine extends Engine {
        private _options;
        isDeterministicLockStep(): boolean;
        getLockstepMaxSteps(): number;
        getHardwareScalingLevel(): number;
        constructor(options?: NullEngineOptions);
        createVertexBuffer(vertices: FloatArray): WebGLBuffer;
        createIndexBuffer(indices: IndicesArray): WebGLBuffer;
        clear(color: Color4, backBuffer: boolean, depth: boolean, stencil?: boolean): void;
        getRenderWidth(useScreen?: boolean): number;
        getRenderHeight(useScreen?: boolean): number;
        setViewport(viewport: Viewport, requiredWidth?: number, requiredHeight?: number): void;
        createShaderProgram(vertexCode: string, fragmentCode: string, defines: string, context?: WebGLRenderingContext): WebGLProgram;
        getUniforms(shaderProgram: WebGLProgram, uniformsNames: string[]): WebGLUniformLocation[];
        getAttributes(shaderProgram: WebGLProgram, attributesNames: string[]): number[];
        bindSamplers(effect: Effect): void;
        enableEffect(effect: Effect): void;
        setState(culling: boolean, zOffset?: number, force?: boolean, reverseSide?: boolean): void;
        setIntArray(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray2(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray3(uniform: WebGLUniformLocation, array: Int32Array): void;
        setIntArray4(uniform: WebGLUniformLocation, array: Int32Array): void;
        setFloatArray(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray2(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray3(uniform: WebGLUniformLocation, array: Float32Array): void;
        setFloatArray4(uniform: WebGLUniformLocation, array: Float32Array): void;
        setArray(uniform: WebGLUniformLocation, array: number[]): void;
        setArray2(uniform: WebGLUniformLocation, array: number[]): void;
        setArray3(uniform: WebGLUniformLocation, array: number[]): void;
        setArray4(uniform: WebGLUniformLocation, array: number[]): void;
        setMatrices(uniform: WebGLUniformLocation, matrices: Float32Array): void;
        setMatrix(uniform: WebGLUniformLocation, matrix: Matrix): void;
        setMatrix3x3(uniform: WebGLUniformLocation, matrix: Float32Array): void;
        setMatrix2x2(uniform: WebGLUniformLocation, matrix: Float32Array): void;
        setFloat(uniform: WebGLUniformLocation, value: number): void;
        setFloat2(uniform: WebGLUniformLocation, x: number, y: number): void;
        setFloat3(uniform: WebGLUniformLocation, x: number, y: number, z: number): void;
        setBool(uniform: WebGLUniformLocation, bool: number): void;
        setFloat4(uniform: WebGLUniformLocation, x: number, y: number, z: number, w: number): void;
        setColor3(uniform: WebGLUniformLocation, color3: Color3): void;
        setColor4(uniform: WebGLUniformLocation, color3: Color3, alpha: number): void;
        setAlphaMode(mode: number, noDepthWriteChange?: boolean): void;
        bindBuffers(vertexBuffers: {
            [key: string]: VertexBuffer;
        }, indexBuffer: WebGLBuffer, effect: Effect): void;
        wipeCaches(bruteForce?: boolean): void;
        draw(useTriangles: boolean, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawElementsType(fillMode: number, indexStart: number, indexCount: number, instancesCount?: number): void;
        drawArraysType(fillMode: number, verticesStart: number, verticesCount: number, instancesCount?: number): void;
        _createTexture(): WebGLTexture;
        createTexture(urlArg: string, noMipmap: boolean, invertY: boolean, scene: Scene, samplingMode?: number, onLoad?: Nullable<() => void>, onError?: Nullable<(message: string, exception: any) => void>, buffer?: Nullable<ArrayBuffer | HTMLImageElement>, fallBack?: InternalTexture, format?: number): InternalTexture;
        createRenderTargetTexture(size: any, options: boolean | RenderTargetCreationOptions): InternalTexture;
        updateTextureSamplingMode(samplingMode: number, texture: InternalTexture): void;
        bindFramebuffer(texture: InternalTexture, faceIndex?: number, requiredWidth?: number, requiredHeight?: number, forceFullscreenViewport?: boolean): void;
        unBindFramebuffer(texture: InternalTexture, disableGenerateMipMaps?: boolean, onBeforeUnbind?: () => void): void;
        createDynamicVertexBuffer(vertices: FloatArray): WebGLBuffer;
        updateDynamicIndexBuffer(indexBuffer: WebGLBuffer, indices: IndicesArray, offset?: number): void;
        updateDynamicVertexBuffer(vertexBuffer: WebGLBuffer, vertices: FloatArray, offset?: number, count?: number): void;
        protected _bindTextureDirectly(target: number, texture: InternalTexture): void;
        _bindTexture(channel: number, texture: InternalTexture): void;
        _releaseBuffer(buffer: WebGLBuffer): boolean;
    }
}

interface WebGLRenderingContext {
    readonly RASTERIZER_DISCARD: number;
    readonly DEPTH_COMPONENT24: number;
    readonly TEXTURE_3D: number;
    readonly TEXTURE_2D_ARRAY: number;
    readonly TEXTURE_COMPARE_FUNC: number;
    readonly TEXTURE_COMPARE_MODE: number;
    readonly COMPARE_REF_TO_TEXTURE: number;
    readonly TEXTURE_WRAP_R: number;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ArrayBufferView | null): void;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ArrayBufferView, offset: number): void;
    texImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, format: number, type: number, pixels: ImageBitmap | ImageData | HTMLVideoElement | HTMLImageElement | HTMLCanvasElement): void;
    compressedTexImage3D(target: number, level: number, internalformat: number, width: number, height: number, depth: number, border: number, data: ArrayBufferView, offset?: number, length?: number): void;
    readonly TRANSFORM_FEEDBACK: number;
    readonly INTERLEAVED_ATTRIBS: number;
    readonly TRANSFORM_FEEDBACK_BUFFER: number;
    createTransformFeedback(): WebGLTransformFeedback;
    deleteTransformFeedback(transformFeedbac: WebGLTransformFeedback): void;
    bindTransformFeedback(target: number, transformFeedback: WebGLTransformFeedback | null): void;
    beginTransformFeedback(primitiveMode: number): void;
    endTransformFeedback(): void;
    transformFeedbackVaryings(program: WebGLProgram, varyings: string[], bufferMode: number): void;
}
interface ImageBitmap {
    readonly width: number;
    readonly height: number;
    close(): void;
}
interface WebGLQuery extends WebGLObject {
}
declare var WebGLQuery: {
    prototype: WebGLQuery;
    new (): WebGLQuery;
};
interface WebGLSampler extends WebGLObject {
}
declare var WebGLSampler: {
    prototype: WebGLSampler;
    new (): WebGLSampler;
};
interface WebGLSync extends WebGLObject {
}
declare var WebGLSync: {
    prototype: WebGLSync;
    new (): WebGLSync;
};
interface WebGLTransformFeedback extends WebGLObject {
}
declare var WebGLTransformFeedback: {
    prototype: WebGLTransformFeedback;
    new (): WebGLTransformFeedback;
};
interface WebGLVertexArrayObject extends WebGLObject {
}
declare var WebGLVertexArrayObject: {
    prototype: WebGLVertexArrayObject;
    new (): WebGLVertexArrayObject;
};

declare module BABYLON {
    class KeyboardEventTypes {
        static _KEYDOWN: number;
        static _KEYUP: number;
        static readonly KEYDOWN: number;
        static readonly KEYUP: number;
    }
    class KeyboardInfo {
        type: number;
        event: KeyboardEvent;
        constructor(type: number, event: KeyboardEvent);
    }
    /**
     * This class is used to store keyboard related info for the onPreKeyboardObservable event.
     * Set the skipOnKeyboardObservable property to true if you want the engine to stop any process after this event is triggered, even not calling onKeyboardObservable
     */
    class KeyboardInfoPre extends KeyboardInfo {
        constructor(type: number, event: KeyboardEvent);
        skipOnPointerObservable: boolean;
    }
}

declare module BABYLON {
    class PointerEventTypes {
        static _POINTERDOWN: number;
        static _POINTERUP: number;
        static _POINTERMOVE: number;
        static _POINTERWHEEL: number;
        static _POINTERPICK: number;
        static _POINTERTAP: number;
        static _POINTERDOUBLETAP: number;
        static readonly POINTERDOWN: number;
        static readonly POINTERUP: number;
        static readonly POINTERMOVE: number;
        static readonly POINTERWHEEL: number;
        static readonly POINTERPICK: number;
        static readonly POINTERTAP: number;
        static readonly POINTERDOUBLETAP: number;
    }
    class PointerInfoBase {
        type: number;
        event: PointerEvent | MouseWheelEvent;
        constructor(type: number, event: PointerEvent | MouseWheelEvent);
    }
    /**
     * This class is used to store pointer related info for the onPrePointerObservable event.
     * Set the skipOnPointerObservable property to true if you want the engine to stop any process after this event is triggered, even not calling onPointerObservable
     */
    class PointerInfoPre extends PointerInfoBase {
        constructor(type: number, event: PointerEvent | MouseWheelEvent, localX: number, localY: number);
        localPosition: Vector2;
        skipOnPointerObservable: boolean;
    }
    /**
     * This type contains all the data related to a pointer event in Babylon.js.
     * The event member is an instance of PointerEvent for all types except PointerWheel and is of type MouseWheelEvent when type equals PointerWheel. The different event types can be found in the PointerEventTypes class.
     */
    class PointerInfo extends PointerInfoBase {
        pickInfo: Nullable<PickingInfo>;
        constructor(type: number, event: PointerEvent | MouseWheelEvent, pickInfo: Nullable<PickingInfo>);
    }
}

declare module BABYLON {
    class StickValues {
        x: number;
        y: number;
        constructor(x: number, y: number);
    }
    interface GamepadButtonChanges {
        changed: boolean;
        pressChanged: boolean;
        touchChanged: boolean;
        valueChanged: boolean;
    }
    class Gamepad {
        id: string;
        index: number;
        browserGamepad: any;
        type: number;
        private _leftStick;
        private _rightStick;
        _isConnected: boolean;
        private _leftStickAxisX;
        private _leftStickAxisY;
        private _rightStickAxisX;
        private _rightStickAxisY;
        private _onleftstickchanged;
        private _onrightstickchanged;
        static GAMEPAD: number;
        static GENERIC: number;
        static XBOX: number;
        static POSE_ENABLED: number;
        protected _invertLeftStickY: boolean;
        readonly isConnected: boolean;
        constructor(id: string, index: number, browserGamepad: any, leftStickX?: number, leftStickY?: number, rightStickX?: number, rightStickY?: number);
        onleftstickchanged(callback: (values: StickValues) => void): void;
        onrightstickchanged(callback: (values: StickValues) => void): void;
        leftStick: StickValues;
        rightStick: StickValues;
        update(): void;
        dispose(): void;
    }
    class GenericPad extends Gamepad {
        private _buttons;
        private _onbuttondown;
        private _onbuttonup;
        onButtonDownObservable: Observable<number>;
        onButtonUpObservable: Observable<number>;
        onbuttondown(callback: (buttonPressed: number) => void): void;
        onbuttonup(callback: (buttonReleased: number) => void): void;
        constructor(id: string, index: number, browserGamepad: any);
        private _setButtonValue(newValue, currentValue, buttonIndex);
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class GamepadManager {
        private _scene;
        private _babylonGamepads;
        private _oneGamepadConnected;
        _isMonitoring: boolean;
        private _gamepadEventSupported;
        private _gamepadSupport;
        onGamepadConnectedObservable: Observable<Gamepad>;
        onGamepadDisconnectedObservable: Observable<Gamepad>;
        private _onGamepadConnectedEvent;
        private _onGamepadDisconnectedEvent;
        constructor(_scene?: Scene | undefined);
        readonly gamepads: Gamepad[];
        getGamepadByType(type?: number): Nullable<Gamepad>;
        dispose(): void;
        private _addNewGamepad(gamepad);
        private _startMonitoringGamepads();
        private _stopMonitoringGamepads();
        _checkGamepadsStatus(): void;
        private _updateGamepadObjects();
    }
}

declare module BABYLON {
    enum Xbox360Button {
        A = 0,
        B = 1,
        X = 2,
        Y = 3,
        Start = 4,
        Back = 5,
        LB = 6,
        RB = 7,
        LeftStick = 8,
        RightStick = 9,
    }
    enum Xbox360Dpad {
        Up = 0,
        Down = 1,
        Left = 2,
        Right = 3,
    }
    class Xbox360Pad extends Gamepad {
        private _leftTrigger;
        private _rightTrigger;
        private _onlefttriggerchanged;
        private _onrighttriggerchanged;
        private _onbuttondown;
        private _onbuttonup;
        private _ondpaddown;
        private _ondpadup;
        onButtonDownObservable: Observable<Xbox360Button>;
        onButtonUpObservable: Observable<Xbox360Button>;
        onPadDownObservable: Observable<Xbox360Dpad>;
        onPadUpObservable: Observable<Xbox360Dpad>;
        private _buttonA;
        private _buttonB;
        private _buttonX;
        private _buttonY;
        private _buttonBack;
        private _buttonStart;
        private _buttonLB;
        private _buttonRB;
        private _buttonLeftStick;
        private _buttonRightStick;
        private _dPadUp;
        private _dPadDown;
        private _dPadLeft;
        private _dPadRight;
        private _isXboxOnePad;
        constructor(id: string, index: number, gamepad: any, xboxOne?: boolean);
        onlefttriggerchanged(callback: (value: number) => void): void;
        onrighttriggerchanged(callback: (value: number) => void): void;
        leftTrigger: number;
        rightTrigger: number;
        onbuttondown(callback: (buttonPressed: Xbox360Button) => void): void;
        onbuttonup(callback: (buttonReleased: Xbox360Button) => void): void;
        ondpaddown(callback: (dPadPressed: Xbox360Dpad) => void): void;
        ondpadup(callback: (dPadReleased: Xbox360Dpad) => void): void;
        private _setButtonValue(newValue, currentValue, buttonType);
        private _setDPadValue(newValue, currentValue, buttonType);
        buttonA: number;
        buttonB: number;
        buttonX: number;
        buttonY: number;
        buttonStart: number;
        buttonBack: number;
        buttonLB: number;
        buttonRB: number;
        buttonLeftStick: number;
        buttonRightStick: number;
        dPadUp: number;
        dPadDown: number;
        dPadLeft: number;
        dPadRight: number;
        update(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Represents the different options available during the creation of
     * a Environment helper.
     *
     * This can control the default ground, skybox and image processing setup of your scene.
     */
    interface IEnvironmentHelperOptions {
        /**
         * Specifies wether or not to create a ground.
         * True by default.
         */
        createGround: boolean;
        /**
         * Specifies the ground size.
         * 15 by default.
         */
        groundSize: number;
        /**
         * The texture used on the ground for the main color.
         * Comes from the BabylonJS CDN by default.
         *
         * Remarks: Can be either a texture or a url.
         */
        groundTexture: string | BaseTexture;
        /**
         * The color mixed in the ground texture by default.
         * BabylonJS clearColor by default.
         */
        groundColor: Color3;
        /**
         * Specifies the ground opacity.
         * 1 by default.
         */
        groundOpacity: number;
        /**
         * Enables the ground to receive shadows.
         * True by default.
         */
        enableGroundShadow: boolean;
        /**
         * Helps preventing the shadow to be fully black on the ground.
         * 0.5 by default.
         */
        groundShadowLevel: number;
        /**
         * Creates a mirror texture attach to the ground.
         * false by default.
         */
        enableGroundMirror: boolean;
        /**
         * Specifies the ground mirror size ratio.
         * 0.3 by default as the default kernel is 64.
         */
        groundMirrorSizeRatio: number;
        /**
         * Specifies the ground mirror blur kernel size.
         * 64 by default.
         */
        groundMirrorBlurKernel: number;
        /**
         * Specifies the ground mirror visibility amount.
         * 1 by default
         */
        groundMirrorAmount: number;
        /**
         * Specifies the ground mirror reflectance weight.
         * This uses the standard weight of the background material to setup the fresnel effect
         * of the mirror.
         * 1 by default.
         */
        groundMirrorFresnelWeight: number;
        /**
         * Specifies the ground mirror Falloff distance.
         * This can helps reducing the size of the reflection.
         * 0 by Default.
         */
        groundMirrorFallOffDistance: number;
        /**
         * Specifies the ground mirror texture type.
         * Unsigned Int by Default.
         */
        groundMirrorTextureType: number;
        /**
         * Specifies a bias applied to the ground vertical position to prevent z-fighyting with
         * the shown objects.
         */
        groundYBias: number;
        /**
         * Specifies wether or not to create a skybox.
         * True by default.
         */
        createSkybox: boolean;
        /**
         * Specifies the skybox size.
         * 20 by default.
         */
        skyboxSize: number;
        /**
         * The texture used on the skybox for the main color.
         * Comes from the BabylonJS CDN by default.
         *
         * Remarks: Can be either a texture or a url.
         */
        skyboxTexture: string | BaseTexture;
        /**
         * The color mixed in the skybox texture by default.
         * BabylonJS clearColor by default.
         */
        skyboxColor: Color3;
        /**
         * The background rotation around the Y axis of the scene.
         * This helps aligning the key lights of your scene with the background.
         * 0 by default.
         */
        backgroundYRotation: number;
        /**
         * Compute automatically the size of the elements to best fit with the scene.
         */
        sizeAuto: boolean;
        /**
         * Default position of the rootMesh if autoSize is not true.
         */
        rootPosition: Vector3;
        /**
         * Sets up the image processing in the scene.
         * true by default.
         */
        setupImageProcessing: boolean;
        /**
         * The texture used as your environment texture in the scene.
         * Comes from the BabylonJS CDN by default and in use if setupImageProcessing is true.
         *
         * Remarks: Can be either a texture or a url.
         */
        environmentTexture: string | BaseTexture;
        /**
         * The value of the exposure to apply to the scene.
         * 0.6 by default if setupImageProcessing is true.
         */
        cameraExposure: number;
        /**
         * The value of the contrast to apply to the scene.
         * 1.6 by default if setupImageProcessing is true.
         */
        cameraContrast: number;
        /**
         * Specifies wether or not tonemapping should be enabled in the scene.
         * true by default if setupImageProcessing is true.
         */
        toneMappingEnabled: boolean;
    }
    /**
     * The Environment helper class can be used to add a fully featuread none expensive background to your scene.
     * It includes by default a skybox and a ground relying on the BackgroundMaterial.
     * It also helps with the default setup of your imageProcessing configuration.
     */
    class EnvironmentHelper {
        /**
         * Default ground texture URL.
         */
        private static _groundTextureCDNUrl;
        /**
         * Default skybox texture URL.
         */
        private static _skyboxTextureCDNUrl;
        /**
         * Default environment texture URL.
         */
        private static _environmentTextureCDNUrl;
        /**
         * Creates the default options for the helper.
         */
        private static _getDefaultOptions();
        private _rootMesh;
        /**
         * Gets the root mesh created by the helper.
         */
        readonly rootMesh: Mesh;
        private _skybox;
        /**
         * Gets the skybox created by the helper.
         */
        readonly skybox: Nullable<Mesh>;
        private _skyboxTexture;
        /**
         * Gets the skybox texture created by the helper.
         */
        readonly skyboxTexture: Nullable<BaseTexture>;
        private _skyboxMaterial;
        /**
         * Gets the skybox material created by the helper.
         */
        readonly skyboxMaterial: Nullable<BackgroundMaterial>;
        private _ground;
        /**
         * Gets the ground mesh created by the helper.
         */
        readonly ground: Nullable<Mesh>;
        private _groundTexture;
        /**
         * Gets the ground texture created by the helper.
         */
        readonly groundTexture: Nullable<BaseTexture>;
        private _groundMirror;
        /**
         * Gets the ground mirror created by the helper.
         */
        readonly groundMirror: Nullable<MirrorTexture>;
        /**
         * Gets the ground mirror render list to helps pushing the meshes
         * you wish in the ground reflection.
         */
        readonly groundMirrorRenderList: Nullable<AbstractMesh[]>;
        private _groundMaterial;
        /**
         * Gets the ground material created by the helper.
         */
        readonly groundMaterial: Nullable<BackgroundMaterial>;
        /**
         * Stores the creation options.
         */
        private readonly _scene;
        private _options;
        /**
         * This observable will be notified with any error during the creation of the environment,
         * mainly texture creation errors.
         */
        onErrorObservable: Observable<{
            message?: string;
            exception?: any;
        }>;
        /**
         * constructor
         * @param options
         * @param scene The scene to add the material to
         */
        constructor(options: Partial<IEnvironmentHelperOptions>, scene: Scene);
        /**
         * Updates the background according to the new options
         * @param options
         */
        updateOptions(options: Partial<IEnvironmentHelperOptions>): void;
        /**
         * Sets the primary color of all the available elements.
         * @param color
         */
        setMainColor(color: Color3): void;
        /**
         * Setup the image processing according to the specified options.
         */
        private _setupImageProcessing();
        /**
         * Setup the environment texture according to the specified options.
         */
        private _setupEnvironmentTexture();
        /**
         * Setup the background according to the specified options.
         */
        private _setupBackground();
        /**
         * Get the scene sizes according to the setup.
         */
        private _getSceneSize();
        /**
         * Setup the ground according to the specified options.
         */
        private _setupGround(sceneSize);
        /**
         * Setup the ground material according to the specified options.
         */
        private _setupGroundMaterial();
        /**
         * Setup the ground diffuse texture according to the specified options.
         */
        private _setupGroundDiffuseTexture();
        /**
         * Setup the ground mirror texture according to the specified options.
         */
        private _setupGroundMirrorTexture(sceneSize);
        /**
         * Setup the ground to receive the mirror texture.
         */
        private _setupMirrorInGroundMaterial();
        /**
         * Setup the skybox according to the specified options.
         */
        private _setupSkybox(sceneSize);
        /**
         * Setup the skybox material according to the specified options.
         */
        private _setupSkyboxMaterial();
        /**
         * Setup the skybox reflection texture according to the specified options.
         */
        private _setupSkyboxReflectionTexture();
        private _errorHandler;
        /**
         * Dispose all the elements created by the Helper.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Display a 360 degree video on an approximately spherical surface, useful for VR applications or skyboxes.
     * As a subclass of Node, this allow parenting to the camera or multiple videos with different locations in the scene.
     * This class achieves its effect with a VideoTexture and a correctly configured BackgroundMaterial on an inverted sphere.
     * Potential additions to this helper include zoom and and non-infinite distance rendering effects.
     */
    class VideoDome extends Node {
        /**
         * The video texture being displayed on the sphere
         */
        protected _videoTexture: VideoTexture;
        /**
         * The skybox material
         */
        protected _material: BackgroundMaterial;
        /**
         * The surface used for the skybox
         */
        protected _mesh: Mesh;
        /**
         * The current fov(field of view) multiplier, 0.0 - 2.0. Defaults to 1.0. Lower values "zoom in" and higher values "zoom out".
         * Also see the options.resolution property.
         */
        fovMultiplier: number;
        /**
         * Create an instance of this class and pass through the parameters to the relevant classes, VideoTexture, StandardMaterial, and Mesh.
         * @param name Element's name, child elements will append suffixes for their own names.
         * @param urlsOrVideo
         * @param options An object containing optional or exposed sub element properties:
         * @param options **resolution=12** Integer, lower resolutions have more artifacts at extreme fovs
         * @param options **clickToPlay=false** Add a click to play listener to the video, does not prevent autoplay.
         * @param options **autoPlay=true** Automatically attempt to being playing the video.
         * @param options **loop=true** Automatically loop video on end.
         * @param options **size=1000** Physical radius to create the dome at, defaults to approximately half the far clip plane.
         */
        constructor(name: string, urlsOrVideo: string | string[] | HTMLVideoElement, options: {
            resolution?: number;
            clickToPlay?: boolean;
            autoPlay?: boolean;
            loop?: boolean;
            size?: number;
        }, scene: Scene);
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
    }
}

declare module BABYLON {
    /**
     * This class can be used to get instrumentation data from a Babylon engine
     */
    class EngineInstrumentation implements IDisposable {
        engine: Engine;
        private _captureGPUFrameTime;
        private _gpuFrameTimeToken;
        private _gpuFrameTime;
        private _captureShaderCompilationTime;
        private _shaderCompilationTime;
        private _onBeginFrameObserver;
        private _onEndFrameObserver;
        private _onBeforeShaderCompilationObserver;
        private _onAfterShaderCompilationObserver;
        /**
         * Gets the perf counter used for GPU frame time
         */
        readonly gpuFrameTimeCounter: PerfCounter;
        /**
         * Gets the GPU frame time capture status
         */
        /**
         * Enable or disable the GPU frame time capture
         */
        captureGPUFrameTime: boolean;
        /**
         * Gets the perf counter used for shader compilation time
         */
        readonly shaderCompilationTimeCounter: PerfCounter;
        /**
         * Gets the shader compilation time capture status
         */
        /**
         * Enable or disable the shader compilation time capture
         */
        captureShaderCompilationTime: boolean;
        constructor(engine: Engine);
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * This class can be used to get instrumentation data from a Babylon engine
     */
    class SceneInstrumentation implements IDisposable {
        scene: Scene;
        private _captureActiveMeshesEvaluationTime;
        private _activeMeshesEvaluationTime;
        private _captureRenderTargetsRenderTime;
        private _renderTargetsRenderTime;
        private _captureFrameTime;
        private _frameTime;
        private _captureRenderTime;
        private _renderTime;
        private _captureInterFrameTime;
        private _interFrameTime;
        private _captureParticlesRenderTime;
        private _particlesRenderTime;
        private _captureSpritesRenderTime;
        private _spritesRenderTime;
        private _capturePhysicsTime;
        private _physicsTime;
        private _captureAnimationsTime;
        private _animationsTime;
        private _onBeforeActiveMeshesEvaluationObserver;
        private _onAfterActiveMeshesEvaluationObserver;
        private _onBeforeRenderTargetsRenderObserver;
        private _onAfterRenderTargetsRenderObserver;
        private _onAfterRenderObserver;
        private _onBeforeDrawPhaseObserver;
        private _onAfterDrawPhaseObserver;
        private _onBeforeAnimationsObserver;
        private _onBeforeParticlesRenderingObserver;
        private _onAfterParticlesRenderingObserver;
        private _onBeforeSpritesRenderingObserver;
        private _onAfterSpritesRenderingObserver;
        private _onBeforePhysicsObserver;
        private _onAfterPhysicsObserver;
        private _onAfterAnimationsObserver;
        /**
         * Gets the perf counter used for active meshes evaluation time
         */
        readonly activeMeshesEvaluationTimeCounter: PerfCounter;
        /**
         * Gets the active meshes evaluation time capture status
         */
        /**
         * Enable or disable the active meshes evaluation time capture
         */
        captureActiveMeshesEvaluationTime: boolean;
        /**
         * Gets the perf counter used for render targets render time
         */
        readonly renderTargetsRenderTimeCounter: PerfCounter;
        /**
         * Gets the render targets render time capture status
         */
        /**
         * Enable or disable the render targets render time capture
         */
        captureRenderTargetsRenderTime: boolean;
        /**
         * Gets the perf counter used for particles render time
         */
        readonly particlesRenderTimeCounter: PerfCounter;
        /**
         * Gets the particles render time capture status
         */
        /**
         * Enable or disable the particles render time capture
         */
        captureParticlesRenderTime: boolean;
        /**
         * Gets the perf counter used for sprites render time
         */
        readonly spritesRenderTimeCounter: PerfCounter;
        /**
         * Gets the sprites render time capture status
         */
        /**
         * Enable or disable the sprites render time capture
         */
        captureSpritesRenderTime: boolean;
        /**
         * Gets the perf counter used for physics time
         */
        readonly physicsTimeCounter: PerfCounter;
        /**
         * Gets the physics time capture status
         */
        /**
         * Enable or disable the physics time capture
         */
        capturePhysicsTime: boolean;
        /**
         * Gets the perf counter used for animations time
         */
        readonly animationsTimeCounter: PerfCounter;
        /**
         * Gets the animations time capture status
         */
        /**
         * Enable or disable the animations time capture
         */
        captureAnimationsTime: boolean;
        /**
         * Gets the perf counter used for frame time capture
         */
        readonly frameTimeCounter: PerfCounter;
        /**
         * Gets the frame time capture status
         */
        /**
         * Enable or disable the frame time capture
         */
        captureFrameTime: boolean;
        /**
         * Gets the perf counter used for inter-frames time capture
         */
        readonly interFrameTimeCounter: PerfCounter;
        /**
         * Gets the inter-frames time capture status
         */
        /**
         * Enable or disable the inter-frames time capture
         */
        captureInterFrameTime: boolean;
        /**
         * Gets the perf counter used for render time capture
         */
        readonly renderTimeCounter: PerfCounter;
        /**
         * Gets the render time capture status
         */
        /**
         * Enable or disable the render time capture
         */
        captureRenderTime: boolean;
        /**
         * Gets the perf counter used for draw calls
         */
        readonly drawCallsCounter: PerfCounter;
        /**
         * Gets the perf counter used for texture collisions
         */
        readonly textureCollisionsCounter: PerfCounter;
        constructor(scene: Scene);
        dispose(): void;
    }
}

declare module BABYLON {
    class _TimeToken {
        _startTimeQuery: Nullable<WebGLQuery>;
        _endTimeQuery: Nullable<WebGLQuery>;
        _timeElapsedQuery: Nullable<WebGLQuery>;
        _timeElapsedQueryEnded: boolean;
    }
}

declare module BABYLON {
    /**
     * Effect layer options. This helps customizing the behaviour
     * of the effect layer.
     */
    interface IEffectLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure effect stability across devices.
         */
        mainTextureFixedSize?: number;
        /**
         * Alpha blending mode used to apply the blur. Default depends of the implementation.
         */
        alphaBlendingMode: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
    }
    /**
     * The effect layer Helps adding post process effect blended with the main pass.
     *
     * This can be for instance use to generate glow or higlight effects on the scene.
     *
     * The effect layer class can not be used directly and is intented to inherited from to be
     * customized per effects.
     */
    abstract class EffectLayer {
        /** The Friendly of the effect in the scene */
        name: string;
        private _vertexBuffers;
        private _indexBuffer;
        private _cachedDefines;
        private _effectLayerMapGenerationEffect;
        private _effectLayerOptions;
        private _mergeEffect;
        protected _scene: Scene;
        protected _engine: Engine;
        protected _maxSize: number;
        protected _mainTextureDesiredSize: ISize;
        protected _mainTexture: RenderTargetTexture;
        protected _shouldRender: boolean;
        protected _postProcesses: PostProcess[];
        protected _textures: BaseTexture[];
        protected _emissiveTextureAndColor: {
            texture: Nullable<BaseTexture>;
            color: Color4;
        };
        /**
         * The clear color of the texture used to generate the glow map.
         */
        neutralColor: Color4;
        /**
         * Specifies wether the highlight layer is enabled or not.
         */
        isEnabled: boolean;
        /**
         * Gets the camera attached to the layer.
         */
        readonly camera: Nullable<Camera>;
        /**
         * An event triggered when the effect layer has been disposed.
         */
        onDisposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the effect layer is about rendering the main texture with the glowy parts.
         */
        onBeforeRenderMainTextureObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the generated texture is being merged in the scene.
         */
        onBeforeComposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the generated texture has been merged in the scene.
         */
        onAfterComposeObservable: Observable<EffectLayer>;
        /**
         * An event triggered when the efffect layer changes its size.
         */
        onSizeChangedObservable: Observable<EffectLayer>;
        /**
         * Instantiates a new effect Layer and references it in the scene.
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         */
        constructor(
            /** The Friendly of the effect in the scene */
            name: string, scene: Scene);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        abstract getEffectName(): string;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @return true if ready otherwise, false
         */
        abstract isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         * @returns true if the effect requires stencil during the main canvas render pass.
         */
        abstract needStencil(): boolean;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         * @returns The effect containing the shader used to merge the effect on the  main canvas
         */
        protected abstract _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the effect layer.
         */
        protected abstract _createTextureAndPostProcesses(): void;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected abstract _internalRender(effect: Effect): void;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected abstract _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        abstract _disposeMesh(mesh: Mesh): void;
        /**
         * Initializes the effect layer with the required options.
         * @param options Sets of none mandatory options to use with the layer (see IEffectLayerOptions for more information)
         */
        protected _init(options: Partial<IEffectLayerOptions>): void;
        /**
         * Generates the index buffer of the full screen quad blending to the main canvas.
         */
        private _generateIndexBuffer();
        /**
         * Generates the vertex buffer of the full screen quad blending to the main canvas.
         */
        private _genrateVertexBuffer();
        /**
         * Sets the main texture desired size which is the closest power of two
         * of the engine canvas size.
         */
        private _setMainTextureSize();
        /**
         * Creates the main texture for the effect layer.
         */
        protected _createMainTexture(): void;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        protected _isReady(subMesh: SubMesh, useInstances: boolean, emissiveTexture: Nullable<BaseTexture>): boolean;
        /**
         * Renders the glowing part of the scene by blending the blurred glowing meshes on top of the rendered scene.
         */
        render(): void;
        /**
         * Determine if a given mesh will be used in the current effect.
         * @param mesh mesh to test
         * @returns true if the mesh will be used
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Returns true if the layer contains information to display, otherwise false.
         * @returns true if the glow layer should be rendered
         */
        shouldRender(): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderEmissiveTextureForMesh(mesh: Mesh): boolean;
        /**
         * Renders the submesh passed in parameter to the generation map.
         */
        protected _renderSubMesh(subMesh: SubMesh): void;
        /**
         * Rebuild the required buffers.
         * @ignore Internal use only.
         */
        _rebuild(): void;
        /**
         * Dispose only the render target textures and post process.
         */
        private _disposeTextureAndPostProcesses();
        /**
         * Dispose the highlight layer and free resources.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * Glow layer options. This helps customizing the behaviour
     * of the glow layer.
     */
    interface IGlowLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the glowing objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure resize independant blur.
         */
        mainTextureFixedSize?: number;
        /**
         * How big is the kernel of the blur texture.
         */
        blurKernelSize: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
        /**
         * Enable MSAA by chosing the number of samples.
         */
        mainTextureSamples?: number;
    }
    /**
     * The glow layer Helps adding a glow effect around the emissive parts of a mesh.
     *
     * Once instantiated in a scene, simply use the pushMesh or removeMesh method to add or remove
     * glowy meshes to your scene.
     *
     * Documentation: https://doc.babylonjs.com/how_to/glow_layer
     */
    class GlowLayer extends EffectLayer {
        name: string;
        /**
         * Effect Name of the layer.
         */
        static readonly EffectName: string;
        /**
         * The default blur kernel size used for the glow.
         */
        static DefaultBlurKernelSize: number;
        /**
         * The default texture size ratio used for the glow.
         */
        static DefaultTextureRatio: number;
        /**
         * Gets the kernel size of the blur.
         */
        /**
         * Sets the kernel size of the blur.
         */
        blurKernelSize: number;
        /**
         * Gets the glow intensity.
         */
        /**
         * Sets the glow intensity.
         */
        intensity: number;
        private _options;
        private _intensity;
        private _horizontalBlurPostprocess1;
        private _verticalBlurPostprocess1;
        private _horizontalBlurPostprocess2;
        private _verticalBlurPostprocess2;
        private _blurTexture1;
        private _blurTexture2;
        private _postProcesses1;
        private _postProcesses2;
        private _includedOnlyMeshes;
        private _excludedMeshes;
        /**
         * Callback used to let the user override the color selection on a per mesh basis
         */
        customEmissiveColorSelector: (mesh: Mesh, subMesh: SubMesh, material: Material, result: Color4) => void;
        /**
         * Callback used to let the user override the texture selection on a per mesh basis
         */
        customEmissiveTextureSelector: (mesh: Mesh, subMesh: SubMesh, material: Material) => Texture;
        /**
         * Instantiates a new glow Layer and references it to the scene.
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         * @param options Sets of none mandatory options to use with the layer (see IGlowLayerOptions for more information)
         */
        constructor(name: string, scene: Scene, options?: Partial<IGlowLayerOptions>);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        getEffectName(): string;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         */
        protected _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the glow layer.
         */
        protected _createTextureAndPostProcesses(): void;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         */
        needStencil(): boolean;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected _internalRender(effect: Effect): void;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Add a mesh in the exclusion list to prevent it to impact or being impacted by the glow layer.
         * @param mesh The mesh to exclude from the glow layer
         */
        addExcludedMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the exclusion list to let it impact or being impacted by the glow layer.
          * @param mesh The mesh to remove
          */
        removeExcludedMesh(mesh: Mesh): void;
        /**
         * Add a mesh in the inclusion list to impact or being impacted by the glow layer.
         * @param mesh The mesh to include in the glow layer
         */
        addIncludedOnlyMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the Inclusion list to prevent it to impact or being impacted by the glow layer.
          * @param mesh The mesh to remove
          */
        removeIncludedOnlyMesh(mesh: Mesh): void;
        /**
         * Determine if a given mesh will be used in the glow layer
         * @param mesh The mesh to test
         * @returns true if the mesh will be highlighted by the current glow layer
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        _disposeMesh(mesh: Mesh): void;
    }
}

declare module BABYLON {
    /**
     * Highlight layer options. This helps customizing the behaviour
     * of the highlight layer.
     */
    interface IHighlightLayerOptions {
        /**
         * Multiplication factor apply to the canvas size to compute the render target size
         * used to generated the glowing objects (the smaller the faster).
         */
        mainTextureRatio: number;
        /**
         * Enforces a fixed size texture to ensure resize independant blur.
         */
        mainTextureFixedSize?: number;
        /**
         * Multiplication factor apply to the main texture size in the first step of the blur to reduce the size
         * of the picture to blur (the smaller the faster).
         */
        blurTextureSizeRatio: number;
        /**
         * How big in texel of the blur texture is the vertical blur.
         */
        blurVerticalSize: number;
        /**
         * How big in texel of the blur texture is the horizontal blur.
         */
        blurHorizontalSize: number;
        /**
         * Alpha blending mode used to apply the blur. Default is combine.
         */
        alphaBlendingMode: number;
        /**
         * The camera attached to the layer.
         */
        camera: Nullable<Camera>;
        /**
         * Should we display highlight as a solid stroke?
         */
        isStroke?: boolean;
    }
    /**
     * The highlight layer Helps adding a glow effect around a mesh.
     *
     * Once instantiated in a scene, simply use the pushMesh or removeMesh method to add or remove
     * glowy meshes to your scene.
     *
     * !!! THIS REQUIRES AN ACTIVE STENCIL BUFFER ON THE CANVAS !!!
     */
    class HighlightLayer extends EffectLayer {
        name: string;
        /**
         * Effect Name of the highlight layer.
         */
        static readonly EffectName: string;
        /**
         * The neutral color used during the preparation of the glow effect.
         * This is black by default as the blend operation is a blend operation.
         */
        static NeutralColor: Color4;
        /**
         * Stencil value used for glowing meshes.
         */
        static GlowingMeshStencilReference: number;
        /**
         * Stencil value used for the other meshes in the scene.
         */
        static NormalMeshStencilReference: number;
        /**
         * Specifies whether or not the inner glow is ACTIVE in the layer.
         */
        innerGlow: boolean;
        /**
         * Specifies whether or not the outer glow is ACTIVE in the layer.
         */
        outerGlow: boolean;
        /**
         * Gets the horizontal size of the blur.
         */
        /**
         * Specifies the horizontal size of the blur.
         */
        blurHorizontalSize: number;
        /**
         * Gets the vertical size of the blur.
         */
        /**
         * Specifies the vertical size of the blur.
         */
        blurVerticalSize: number;
        /**
         * An event triggered when the highlight layer is being blurred.
         */
        onBeforeBlurObservable: Observable<HighlightLayer>;
        /**
         * An event triggered when the highlight layer has been blurred.
         */
        onAfterBlurObservable: Observable<HighlightLayer>;
        private _instanceGlowingMeshStencilReference;
        private _options;
        private _downSamplePostprocess;
        private _horizontalBlurPostprocess;
        private _verticalBlurPostprocess;
        private _blurTexture;
        private _meshes;
        private _excludedMeshes;
        /**
         * Instantiates a new highlight Layer and references it to the scene..
         * @param name The name of the layer
         * @param scene The scene to use the layer in
         * @param options Sets of none mandatory options to use with the layer (see IHighlightLayerOptions for more information)
         */
        constructor(name: string, scene: Scene, options?: Partial<IHighlightLayerOptions>);
        /**
         * Get the effect name of the layer.
         * @return The effect name
         */
        getEffectName(): string;
        /**
         * Create the merge effect. This is the shader use to blit the information back
         * to the main canvas at the end of the scene rendering.
         */
        protected _createMergeEffect(): Effect;
        /**
         * Creates the render target textures and post processes used in the highlight layer.
         */
        protected _createTextureAndPostProcesses(): void;
        /**
         * Returns wether or nood the layer needs stencil enabled during the mesh rendering.
         */
        needStencil(): boolean;
        /**
         * Checks for the readiness of the element composing the layer.
         * @param subMesh the mesh to check for
         * @param useInstances specify wether or not to use instances to render the mesh
         * @param emissiveTexture the associated emissive texture used to generate the glow
         * @return true if ready otherwise, false
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Implementation specific of rendering the generating effect on the main canvas.
         * @param effect The effect used to render through
         */
        protected _internalRender(effect: Effect): void;
        /**
         * Returns true if the layer contains information to display, otherwise false.
         */
        shouldRender(): boolean;
        /**
         * Returns true if the mesh should render, otherwise false.
         * @param mesh The mesh to render
         * @returns true if it should render otherwise false
         */
        protected _shouldRenderMesh(mesh: Mesh): boolean;
        /**
         * Sets the required values for both the emissive texture and and the main color.
         */
        protected _setEmissiveTextureAndColor(mesh: Mesh, subMesh: SubMesh, material: Material): void;
        /**
         * Add a mesh in the exclusion list to prevent it to impact or being impacted by the highlight layer.
         * @param mesh The mesh to exclude from the highlight layer
         */
        addExcludedMesh(mesh: Mesh): void;
        /**
          * Remove a mesh from the exclusion list to let it impact or being impacted by the highlight layer.
          * @param mesh The mesh to highlight
          */
        removeExcludedMesh(mesh: Mesh): void;
        /**
         * Determine if a given mesh will be highlighted by the current HighlightLayer
         * @param mesh mesh to test
         * @returns true if the mesh will be highlighted by the current HighlightLayer
         */
        hasMesh(mesh: AbstractMesh): boolean;
        /**
         * Add a mesh in the highlight layer in order to make it glow with the chosen color.
         * @param mesh The mesh to highlight
         * @param color The color of the highlight
         * @param glowEmissiveOnly Extract the glow from the emissive texture
         */
        addMesh(mesh: Mesh, color: Color3, glowEmissiveOnly?: boolean): void;
        /**
         * Remove a mesh from the highlight layer in order to make it stop glowing.
         * @param mesh The mesh to highlight
         */
        removeMesh(mesh: Mesh): void;
        /**
         * Force the stencil to the normal expected value for none glowing parts
         */
        private _defaultStencilReference(mesh);
        /**
         * Free any resources and references associated to a mesh.
         * Internal use
         * @param mesh The mesh to free.
         */
        _disposeMesh(mesh: Mesh): void;
        /**
         * Dispose the highlight layer and free resources.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class Layer {
        name: string;
        texture: Nullable<Texture>;
        isBackground: boolean;
        color: Color4;
        scale: Vector2;
        offset: Vector2;
        alphaBlendingMode: number;
        alphaTest: boolean;
        layerMask: number;
        private _scene;
        private _vertexBuffers;
        private _indexBuffer;
        private _effect;
        private _alphaTestEffect;
        /**
        * An event triggered when the layer is disposed.
        */
        onDisposeObservable: Observable<Layer>;
        private _onDisposeObserver;
        onDispose: () => void;
        /**
        * An event triggered before rendering the scene
        */
        onBeforeRenderObservable: Observable<Layer>;
        private _onBeforeRenderObserver;
        onBeforeRender: () => void;
        /**
        * An event triggered after rendering the scene
        */
        onAfterRenderObservable: Observable<Layer>;
        private _onAfterRenderObserver;
        onAfterRender: () => void;
        constructor(name: string, imgUrl: Nullable<string>, scene: Nullable<Scene>, isBackground?: boolean, color?: Color4);
        private _createIndexBuffer();
        _rebuild(): void;
        render(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class LensFlare {
        size: number;
        position: number;
        color: Color3;
        texture: Nullable<Texture>;
        alphaMode: number;
        private _system;
        static AddFlare(size: number, position: number, color: Color3, imgUrl: string, system: LensFlareSystem): LensFlare;
        constructor(size: number, position: number, color: Color3, imgUrl: string, system: LensFlareSystem);
        dispose(): void;
    }
}

declare module BABYLON {
    class LensFlareSystem {
        name: string;
        lensFlares: LensFlare[];
        borderLimit: number;
        viewportBorder: number;
        meshesSelectionPredicate: (mesh: AbstractMesh) => boolean;
        layerMask: number;
        id: string;
        private _scene;
        private _emitter;
        private _vertexBuffers;
        private _indexBuffer;
        private _effect;
        private _positionX;
        private _positionY;
        private _isEnabled;
        constructor(name: string, emitter: any, scene: Scene);
        isEnabled: boolean;
        getScene(): Scene;
        getEmitter(): any;
        setEmitter(newEmitter: any): void;
        getEmitterPosition(): Vector3;
        computeEffectivePosition(globalViewport: Viewport): boolean;
        _isVisible(): boolean;
        render(): boolean;
        dispose(): void;
        static Parse(parsedLensFlareSystem: any, scene: Scene, rootUrl: string): LensFlareSystem;
        serialize(): any;
    }
}

declare module BABYLON {
    /**
     * A directional light is defined by a direction (what a surprise!).
     * The light is emitted from everywhere in the specified direction, and has an infinite range.
     * An example of a directional light is when a distance planet is lit by the apparently parallel lines of light from its sun. Light in a downward direction will light the top of an object.
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class DirectionalLight extends ShadowLight {
        private _shadowFrustumSize;
        /**
         * Fix frustum size for the shadow generation. This is disabled if the value is 0.
         */
        /**
         * Specifies a fix frustum size for the shadow generation.
         */
        shadowFrustumSize: number;
        private _shadowOrthoScale;
        /**
         * Gets the shadow projection scale against the optimal computed one.
         * 0.1 by default which means that the projection window is increase by 10% from the optimal size.
         * This does not impact in fixed frustum size (shadowFrustumSize being set)
         */
        /**
         * Sets the shadow projection scale against the optimal computed one.
         * 0.1 by default which means that the projection window is increase by 10% from the optimal size.
         * This does not impact in fixed frustum size (shadowFrustumSize being set)
         */
        shadowOrthoScale: number;
        /**
         * Automatically compute the projection matrix to best fit (including all the casters)
         * on each frame.
         */
        autoUpdateExtends: boolean;
        private _orthoLeft;
        private _orthoRight;
        private _orthoTop;
        private _orthoBottom;
        /**
         * Creates a DirectionalLight object in the scene, oriented towards the passed direction (Vector3).
         * The directional light is emitted from everywhere in the given direction.
         * It can cast shawdows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The friendly name of the light
         * @param direction The direction of the light
         * @param scene The scene the light belongs to
         */
        constructor(name: string, direction: Vector3, scene: Scene);
        /**
         * Returns the string "DirectionalLight".
         * @return The class name
         */
        getClassName(): string;
        /**
         * Returns the integer 1.
         * @return The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Sets the passed matrix "matrix" as projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        /**
         * Sets the passed matrix "matrix" as fixed frustum projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultFixedFrustumShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix): void;
        /**
         * Sets the passed matrix "matrix" as auto extend projection matrix for the shadows cast by the light according to the passed view matrix.
         * Returns the DirectionalLight Shadow projection matrix.
         */
        protected _setDefaultAutoExtendShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect object with the DirectionalLight transformed position (or position if not parented) and the passed name.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The directional light
         */
        transferToEffect(effect: Effect, lightIndex: string): DirectionalLight;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         *
         * Values are fixed on directional lights as it relies on an ortho projection hence the need to convert being
         * -1 and 1 to 0 and 1 doing (depth + min) / (min + max) -> (depth + 1) / (1 + 1) -> (depth * 0.5) + 0.5.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         *
         * Values are fixed on directional lights as it relies on an ortho projection hence the need to convert being
         * -1 and 1 to 0 and 1 doing (depth + min) / (min + max) -> (depth + 1) / (1 + 1) -> (depth * 0.5) + 0.5.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
    }
}

declare module BABYLON {
    /**
     * The HemisphericLight simulates the ambient environment light,
     * so the passed direction is the light reflection direction, not the incoming direction.
     */
    class HemisphericLight extends Light {
        /**
         * The groundColor is the light in the opposite direction to the one specified during creation.
         * You can think of the diffuse and specular light as coming from the centre of the object in the given direction and the groundColor light in the opposite direction.
         */
        groundColor: Color3;
        /**
         * The light reflection direction, not the incoming direction.
         */
        direction: Vector3;
        private _worldMatrix;
        /**
         * Creates a HemisphericLight object in the scene according to the passed direction (Vector3).
         * The HemisphericLight simulates the ambient environment light, so the passed direction is the light reflection direction, not the incoming direction.
         * The HemisphericLight can't cast shadows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The friendly name of the light
         * @param direction The direction of the light reflection
         * @param scene The scene the light belongs to
         */
        constructor(name: string, direction: Vector3, scene: Scene);
        protected _buildUniformLayout(): void;
        /**
         * Returns the string "HemisphericLight".
         * @return The class name
         */
        getClassName(): string;
        /**
         * Sets the HemisphericLight direction towards the passed target (Vector3).
         * Returns the updated direction.
         * @param target The target the direction should point to
         * @return The computed direction
         */
        setDirectionToTarget(target: Vector3): Vector3;
        /**
         * Returns the shadow generator associated to the light.
         * @returns Always null for hemispheric lights because it does not support shadows.
         */
        getShadowGenerator(): Nullable<ShadowGenerator>;
        /**
         * Sets the passed Effect object with the HemisphericLight normalized direction and color and the passed name (string).
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The hemispheric light
         */
        transferToEffect(effect: Effect, lightIndex: string): HemisphericLight;
        /**
         * @ignore internal use only.
         */
        _getWorldMatrix(): Matrix;
        /**
         * Returns the integer 3.
         * @return The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
    }
}

declare module BABYLON {
    /**
     * Base class of all the lights in Babylon. It groups all the generic information about lights.
     * Lights are used, as you would expect, to affect how meshes are seen, in terms of both illumination and colour.
     * All meshes allow light to pass through them unless shadow generation is activated. The default number of lights allowed is four but this can be increased.
     */
    abstract class Light extends Node {
        private static _LIGHTMAP_DEFAULT;
        private static _LIGHTMAP_SPECULAR;
        private static _LIGHTMAP_SHADOWSONLY;
        /**
         * If every light affecting the material is in this lightmapMode,
         * material.lightmapTexture adds or multiplies
         * (depends on material.useLightmapAsShadowmap)
         * after every other light calculations.
         */
        static readonly LIGHTMAP_DEFAULT: number;
        /**
         * material.lightmapTexture as only diffuse lighting from this light
         * adds only specular lighting from this light
         * adds dynamic shadows
         */
        static readonly LIGHTMAP_SPECULAR: number;
        /**
         * material.lightmapTexture as only lighting
         * no light calculation from this light
         * only adds dynamic shadows from this light
         */
        static readonly LIGHTMAP_SHADOWSONLY: number;
        private static _INTENSITYMODE_AUTOMATIC;
        private static _INTENSITYMODE_LUMINOUSPOWER;
        private static _INTENSITYMODE_LUMINOUSINTENSITY;
        private static _INTENSITYMODE_ILLUMINANCE;
        private static _INTENSITYMODE_LUMINANCE;
        /**
         * Each light type uses the default quantity according to its type:
         *      point/spot lights use luminous intensity
         *      directional lights use illuminance
         */
        static readonly INTENSITYMODE_AUTOMATIC: number;
        /**
         * lumen (lm)
         */
        static readonly INTENSITYMODE_LUMINOUSPOWER: number;
        /**
         * candela (lm/sr)
         */
        static readonly INTENSITYMODE_LUMINOUSINTENSITY: number;
        /**
         * lux (lm/m^2)
         */
        static readonly INTENSITYMODE_ILLUMINANCE: number;
        /**
         * nit (cd/m^2)
         */
        static readonly INTENSITYMODE_LUMINANCE: number;
        private static _LIGHTTYPEID_POINTLIGHT;
        private static _LIGHTTYPEID_DIRECTIONALLIGHT;
        private static _LIGHTTYPEID_SPOTLIGHT;
        private static _LIGHTTYPEID_HEMISPHERICLIGHT;
        /**
         * Light type const id of the point light.
         */
        static readonly LIGHTTYPEID_POINTLIGHT: number;
        /**
         * Light type const id of the directional light.
         */
        static readonly LIGHTTYPEID_DIRECTIONALLIGHT: number;
        /**
         * Light type const id of the spot light.
         */
        static readonly LIGHTTYPEID_SPOTLIGHT: number;
        /**
         * Light type const id of the hemispheric light.
         */
        static readonly LIGHTTYPEID_HEMISPHERICLIGHT: number;
        /**
         * Diffuse gives the basic color to an object.
         */
        diffuse: Color3;
        /**
         * Specular produces a highlight color on an object.
         * Note: This is note affecting PBR materials.
         */
        specular: Color3;
        /**
         * Strength of the light.
         * Note: By default it is define in the framework own unit.
         * Note: In PBR materials the intensityMode can be use to chose what unit the intensity is defined in.
         */
        intensity: number;
        /**
         * Defines how far from the source the light is impacting in scene units.
         * Note: Unused in PBR material as the distance light falloff is defined following the inverse squared falloff.
         */
        range: number;
        /**
         * Cached photometric scale default to 1.0 as the automatic intensity mode defaults to 1.0 for every type
         * of light.
         */
        private _photometricScale;
        private _intensityMode;
        /**
         * Gets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        /**
         * Sets the photometric scale used to interpret the intensity.
         * This is only relevant with PBR Materials where the light intensity can be defined in a physical way.
         */
        intensityMode: number;
        private _radius;
        /**
         * Gets the light radius used by PBR Materials to simulate soft area lights.
         */
        /**
         * sets the light radius used by PBR Materials to simulate soft area lights.
         */
        radius: number;
        private _renderPriority;
        /**
         * Defines the rendering priority of the lights. It can help in case of fallback or number of lights
         * exceeding the number allowed of the materials.
         */
        renderPriority: number;
        /**
         * Defines wether or not the shadows are enabled for this light. This can help turning off/on shadow without detaching
         * the current shadow generator.
         */
        shadowEnabled: boolean;
        private _includedOnlyMeshes;
        /**
         * Gets the only meshes impacted by this light.
         */
        /**
         * Sets the only meshes impacted by this light.
         */
        includedOnlyMeshes: AbstractMesh[];
        private _excludedMeshes;
        /**
         * Gets the meshes not impacted by this light.
         */
        /**
         * Sets the meshes not impacted by this light.
         */
        excludedMeshes: AbstractMesh[];
        private _excludeWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        /**
         * Sets the layer id use to find what meshes are not impacted by the light.
         * Inactive if 0
         */
        excludeWithLayerMask: number;
        private _includeOnlyWithLayerMask;
        /**
         * Gets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        /**
         * Sets the layer id use to find what meshes are impacted by the light.
         * Inactive if 0
         */
        includeOnlyWithLayerMask: number;
        private _lightmapMode;
        /**
         * Gets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        /**
         * Sets the lightmap mode of this light (should be one of the constants defined by Light.LIGHTMAP_x)
         */
        lightmapMode: number;
        private _parentedWorldMatrix;
        /**
         * Shadow generator associted to the light.
         * Internal use only.
         */
        _shadowGenerator: Nullable<IShadowGenerator>;
        /**
         * @ignore Internal use only.
         */
        _excludedMeshesIds: string[];
        /**
         * @ignore Internal use only.
         */
        _includedOnlyMeshesIds: string[];
        /**
         * The current light unifom buffer.
         * @ignore Internal use only.
         */
        _uniformBuffer: UniformBuffer;
        /**
         * Creates a Light object in the scene.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The firendly name of the light
         * @param scene The scene the light belongs too
         */
        constructor(name: string, scene: Scene);
        protected abstract _buildUniformLayout(): void;
        /**
         * Sets the passed Effect "effect" with the Light information.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The light
         */
        abstract transferToEffect(effect: Effect, lightIndex: string): Light;
        /**
         * @ignore internal use only.
         */
        abstract _getWorldMatrix(): Matrix;
        /**
         * Returns the string "Light".
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Converts the light information to a readable string for debug purpose.
         * @param fullDetails Supports for multiple levels of logging within scene loading
         * @returns the human readable light info
         */
        toString(fullDetails?: boolean): string;
        /**
         * Set the enabled state of this node.
         * @param value - the new enabled state
         */
        setEnabled(value: boolean): void;
        /**
         * Returns the Light associated shadow generator if any.
         * @return the associated shadow generator.
         */
        getShadowGenerator(): Nullable<IShadowGenerator>;
        /**
         * Returns a Vector3, the absolute light position in the World.
         * @returns the world space position of the light
         */
        getAbsolutePosition(): Vector3;
        /**
         * Specifies if the light will affect the passed mesh.
         * @param mesh The mesh to test against the light
         * @return true the mesh is affected otherwise, false.
         */
        canAffectMesh(mesh: AbstractMesh): boolean;
        /**
         * Computes and Returns the light World matrix.
         * @returns the world matrix
         */
        getWorldMatrix(): Matrix;
        /**
         * Sort function to order lights for rendering.
         * @param a First Light object to compare to second.
         * @param b Second Light object to compare first.
         * @return -1 to reduce's a's index relative to be, 0 for no change, 1 to increase a's index relative to b.
         */
        static CompareLightsPriority(a: Light, b: Light): number;
        /**
         * Releases resources associated with this node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Returns the light type ID (integer).
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Returns the intensity scaled by the Photometric Scale according to the light type and intensity mode.
         * @returns the scaled intensity in intensity mode unit
         */
        getScaledIntensity(): number;
        /**
         * Returns a new Light object, named "name", from the current one.
         * @param name The name of the cloned light
         * @returns the new created light
         */
        clone(name: string): Nullable<Light>;
        /**
         * Serializes the current light into a Serialization object.
         * @returns the serialized object.
         */
        serialize(): any;
        /**
         * Creates a new typed light from the passed type (integer) : point light = 0, directional light = 1, spot light = 2, hemispheric light = 3.
         * This new light is named "name" and added to the passed scene.
         * @param type Type according to the types available in Light.LIGHTTYPEID_x
         * @param name The friendly name of the light
         * @param scene The scene the new light will belong to
         * @returns the constructor function
         */
        static GetConstructorFromName(type: number, name: string, scene: Scene): Nullable<() => Light>;
        /**
         * Parses the passed "parsedLight" and returns a new instanced Light from this parsing.
         * @param parsedLight The JSON representation of the light
         * @param scene The scene to create the parsed light in
         * @returns the created light after parsing
         */
        static Parse(parsedLight: any, scene: Scene): Nullable<Light>;
        private _hookArrayForExcluded(array);
        private _hookArrayForIncludedOnly(array);
        private _resyncMeshes();
        /**
         * Forces the meshes to update their light related information in their rendering used effects
         * @ignore Internal Use Only
         */
        _markMeshesAsLightDirty(): void;
        /**
         * Recomputes the cached photometric scale if needed.
         */
        private _computePhotometricScale();
        /**
         * Returns the Photometric Scale according to the light type and intensity mode.
         */
        private _getPhotometricScale();
        /**
         * Reorder the light in the scene according to their defined priority.
         * @ignore Internal Use Only
         */
        _reorderLightsInScene(): void;
    }
}

declare module BABYLON {
    /**
     * A point light is a light defined by an unique point in world space.
     * The light is emitted in every direction from this point.
     * A good example of a point light is a standard light bulb.
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class PointLight extends ShadowLight {
        private _shadowAngle;
        /**
         * Getter: In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         * This specifies what angle the shadow will use to be created.
         *
         * It default to 90 degrees to work nicely with the cube texture generation for point lights shadow maps.
         */
        /**
         * Setter: In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         * This specifies what angle the shadow will use to be created.
         *
         * It default to 90 degrees to work nicely with the cube texture generation for point lights shadow maps.
         */
        shadowAngle: number;
        /**
         * Gets the direction if it has been set.
         * In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         */
        /**
         * In case of direction provided, the shadow will not use a cube texture but simulate a spot shadow as a fallback
         */
        direction: Vector3;
        /**
         * Creates a PointLight object from the passed name and position (Vector3) and adds it in the scene.
         * A PointLight emits the light in every direction.
         * It can cast shadows.
         * If the scene camera is already defined and you want to set your PointLight at the camera position, just set it :
         * ```javascript
         * var pointLight = new BABYLON.PointLight("pl", camera.position, scene);
         * ```
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The light friendly name
         * @param position The position of the point light in the scene
         * @param scene The scene the lights belongs to
         */
        constructor(name: string, position: Vector3, scene: Scene);
        /**
         * Returns the string "PointLight"
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Returns the integer 0.
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Specifies wether or not the shadowmap should be a cube texture.
         * @returns true if the shadowmap needs to be a cube texture.
         */
        needCube(): boolean;
        /**
         * Returns a new Vector3 aligned with the PointLight cube system according to the passed cube face index (integer).
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Sets the passed matrix "matrix" as a left-handed perspective projection matrix with the following settings :
         * - fov = PI / 2
         * - aspect ratio : 1.0
         * - z-near and far equal to the active camera minZ and maxZ.
         * Returns the PointLight.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect "effect" with the PointLight transformed position (or position, if none) and passed name (string).
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The point light
         */
        transferToEffect(effect: Effect, lightIndex: string): PointLight;
    }
}

declare module BABYLON {
    /**
     * Interface describing all the common properties and methods a shadow light needs to implement.
     * This helps both the shadow generator and materials to genrate the corresponding shadow maps
     * as well as binding the different shadow properties to the effects.
     */
    interface IShadowLight extends Light {
        /**
         * The light id in the scene (used in scene.findLighById for instance)
         */
        id: string;
        /**
         * The position the shdow will be casted from.
         */
        position: Vector3;
        /**
         * In 2d mode (needCube being false), the direction used to cast the shadow.
         */
        direction: Vector3;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        /**
         * The friendly name of the light in the scene.
         */
        name: string;
        /**
         * Defines the shadow projection clipping minimum z value.
         */
        shadowMinZ: number;
        /**
         * Defines the shadow projection clipping maximum z value.
         */
        shadowMaxZ: number;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Gets the scene the light belongs to.
         * @returns The scene
         */
        getScene(): Scene;
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
        /**
         * Gets the current depth scale used in ESM.
         * @returns The scale
         */
        getDepthScale(): number;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
    }
    /**
     * Base implementation IShadowLight
     * It groups all the common behaviour in order to reduce dupplication and better follow the DRY pattern.
     */
    abstract class ShadowLight extends Light implements IShadowLight {
        protected abstract _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _position: Vector3;
        protected _setPosition(value: Vector3): void;
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        /**
         * Sets the position the shadow will be casted from. Also use as the light position for both
         * point and spot lights.
         */
        position: Vector3;
        protected _direction: Vector3;
        protected _setDirection(value: Vector3): void;
        /**
         * In 2d mode (needCube being false), gets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        /**
         * In 2d mode (needCube being false), sets the direction used to cast the shadow.
         * Also use as the light direction on spot and directional lights.
         */
        direction: Vector3;
        private _shadowMinZ;
        /**
         * Gets the shadow projection clipping minimum z value.
         */
        /**
         * Sets the shadow projection clipping minimum z value.
         */
        shadowMinZ: number;
        private _shadowMaxZ;
        /**
         * Sets the shadow projection clipping maximum z value.
         */
        /**
         * Gets the shadow projection clipping maximum z value.
         */
        shadowMaxZ: number;
        /**
         * Callback defining a custom Projection Matrix Builder.
         * This can be used to override the default projection matrix computation.
         */
        customProjectionMatrixBuilder: (viewMatrix: Matrix, renderList: Array<AbstractMesh>, result: Matrix) => void;
        /**
         * The transformed position. Position of the light in world space taking parenting in account.
         */
        transformedPosition: Vector3;
        /**
         * The transformed direction. Direction of the light in world space taking parenting in account.
         */
        transformedDirection: Vector3;
        private _worldMatrix;
        private _needProjectionMatrixCompute;
        /**
         * Computes the transformed information (transformedPosition and transformedDirection in World space) of the current light
         * @returns true if the information has been computed, false if it does not need to (no parenting)
         */
        computeTransformedInformation(): boolean;
        /**
         * Return the depth scale used for the shadow map.
         * @returns the depth scale.
         */
        getDepthScale(): number;
        /**
         * Get the direction to use to render the shadow map. In case of cube texture, the face index can be passed.
         * @param faceIndex The index of the face we are computed the direction to generate shadow
         * @returns The set direction in 2d mode otherwise the direction to the cubemap face if needCube() is true
         */
        getShadowDirection(faceIndex?: number): Vector3;
        /**
         * Returns the ShadowLight absolute position in the World.
         * @returns the position vector in world space
         */
        getAbsolutePosition(): Vector3;
        /**
         * Sets the ShadowLight direction toward the passed target.
         * @param target The point tot target in local space
         * @returns the updated ShadowLight direction
         */
        setDirectionToTarget(target: Vector3): Vector3;
        /**
         * Returns the light rotation in euler definition.
         * @returns the x y z rotation in local space.
         */
        getRotation(): Vector3;
        /**
         * Returns whether or not the shadow generation require a cube texture or a 2d texture.
         * @returns true if a cube texture needs to be use
         */
        needCube(): boolean;
        /**
         * Detects if the projection matrix requires to be recomputed this frame.
         * @returns true if it requires to be recomputed otherwise, false.
         */
        needProjectionMatrixCompute(): boolean;
        /**
         * Forces the shadow generator to recompute the projection matrix even if position and direction did not changed.
         */
        forceProjectionMatrixCompute(): void;
        /**
         * Get the world matrix of the sahdow lights.
         * @ignore Internal Use Only
         */
        _getWorldMatrix(): Matrix;
        /**
         * Gets the minZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the min for
         * @returns the depth min z
         */
        getDepthMinZ(activeCamera: Camera): number;
        /**
         * Gets the maxZ used for shadow according to both the scene and the light.
         * @param activeCamera The camera we are returning the max for
         * @returns the depth max z
         */
        getDepthMaxZ(activeCamera: Camera): number;
        /**
         * Sets the shadow projection matrix in parameter to the generated projection matrix.
         * @param matrix The materix to updated with the projection information
         * @param viewMatrix The transform matrix of the light
         * @param renderList The list of mesh to render in the map
         * @returns The current light
         */
        setShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): IShadowLight;
    }
}

declare module BABYLON {
    /**
     * A spot light is defined by a position, a direction, an angle, and an exponent.
     * These values define a cone of light starting from the position, emitting toward the direction.
     * The angle, in radians, defines the size (field of illumination) of the spotlight's conical beam,
     * and the exponent defines the speed of the decay of the light with distance (reach).
     * Documentation: https://doc.babylonjs.com/babylon101/lights
     */
    class SpotLight extends ShadowLight {
        private _angle;
        /**
         * Gets the cone angle of the spot light in Radians.
         */
        /**
         * Sets the cone angle of the spot light in Radians.
         */
        angle: number;
        private _shadowAngleScale;
        /**
         * Allows scaling the angle of the light for shadow generation only.
         */
        /**
         * Allows scaling the angle of the light for shadow generation only.
         */
        shadowAngleScale: number;
        /**
         * The light decay speed with the distance from the emission spot.
         */
        exponent: number;
        private _projectionTextureMatrix;
        /**
        * Allows reading the projecton texture
        */
        readonly projectionTextureMatrix: Matrix;
        protected _projectionTextureLightNear: number;
        /**
         * Gets the near clip of the Spotlight for texture projection.
         */
        /**
         * Sets the near clip of the Spotlight for texture projection.
         */
        projectionTextureLightNear: number;
        protected _projectionTextureLightFar: number;
        /**
         * Gets the far clip of the Spotlight for texture projection.
         */
        /**
         * Sets the far clip of the Spotlight for texture projection.
         */
        projectionTextureLightFar: number;
        protected _projectionTextureUpDirection: Vector3;
        /**
         * Gets the Up vector of the Spotlight for texture projection.
         */
        /**
         * Sets the Up vector of the Spotlight for texture projection.
         */
        projectionTextureUpDirection: Vector3;
        private _projectionTexture;
        /**
         * Gets the projection texture of the light.
        */
        /**
        * Sets the projection texture of the light.
        */
        projectionTexture: Nullable<BaseTexture>;
        private _projectionTextureViewLightDirty;
        private _projectionTextureProjectionLightDirty;
        private _projectionTextureDirty;
        private _projectionTextureViewTargetVector;
        private _projectionTextureViewLightMatrix;
        private _projectionTextureProjectionLightMatrix;
        private _projectionTextureScalingMatrix;
        /**
         * Creates a SpotLight object in the scene. A spot light is a simply light oriented cone.
         * It can cast shadows.
         * Documentation : http://doc.babylonjs.com/tutorials/lights
         * @param name The light friendly name
         * @param position The position of the spot light in the scene
         * @param direction The direction of the light in the scene
         * @param angle The cone angle of the light in Radians
         * @param exponent The light decay speed with the distance from the emission spot
         * @param scene The scene the lights belongs to
         */
        constructor(name: string, position: Vector3, direction: Vector3, angle: number, exponent: number, scene: Scene);
        /**
         * Returns the string "SpotLight".
         * @returns the class name
         */
        getClassName(): string;
        /**
         * Returns the integer 2.
         * @returns The light Type id as a constant defines in Light.LIGHTTYPEID_x
         */
        getTypeID(): number;
        /**
         * Overrides the direction setter to recompute the projection texture view light Matrix.
         */
        protected _setDirection(value: Vector3): void;
        /**
         * Overrides the position setter to recompute the projection texture view light Matrix.
         */
        protected _setPosition(value: Vector3): void;
        /**
         * Sets the passed matrix "matrix" as perspective projection matrix for the shadows and the passed view matrix with the fov equal to the SpotLight angle and and aspect ratio of 1.0.
         * Returns the SpotLight.
         */
        protected _setDefaultShadowProjectionMatrix(matrix: Matrix, viewMatrix: Matrix, renderList: Array<AbstractMesh>): void;
        protected _computeProjectionTextureViewLightMatrix(): void;
        protected _computeProjectionTextureProjectionLightMatrix(): void;
        /**
         * Main function for light texture projection matrix computing.
         */
        protected _computeProjectionTextureMatrix(): void;
        protected _buildUniformLayout(): void;
        /**
         * Sets the passed Effect object with the SpotLight transfomed position (or position if not parented) and normalized direction.
         * @param effect The effect to update
         * @param lightIndex The index of the light in the effect to update
         * @returns The spot light
         */
        transferToEffect(effect: Effect, lightIndex: string): SpotLight;
        /**
         * Disposes the light and the associated resources.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    interface ILoadingScreen {
        displayLoadingUI: () => void;
        hideLoadingUI: () => void;
        loadingUIBackgroundColor: string;
        loadingUIText: string;
    }
    class DefaultLoadingScreen implements ILoadingScreen {
        private _renderingCanvas;
        private _loadingText;
        private _loadingDivBackgroundColor;
        private _loadingDiv;
        private _loadingTextDiv;
        constructor(_renderingCanvas: HTMLCanvasElement, _loadingText?: string, _loadingDivBackgroundColor?: string);
        displayLoadingUI(): void;
        hideLoadingUI(): void;
        loadingUIText: string;
        loadingUIBackgroundColor: string;
        private _resizeLoadingUI;
    }
}

declare module BABYLON {
    class SceneLoaderProgressEvent {
        readonly lengthComputable: boolean;
        readonly loaded: number;
        readonly total: number;
        constructor(lengthComputable: boolean, loaded: number, total: number);
        static FromProgressEvent(event: ProgressEvent): SceneLoaderProgressEvent;
    }
    interface ISceneLoaderPluginExtensions {
        [extension: string]: {
            isBinary: boolean;
        };
    }
    interface ISceneLoaderPluginFactory {
        name: string;
        createPlugin(): ISceneLoaderPlugin | ISceneLoaderPluginAsync;
        canDirectLoad?: (data: string) => boolean;
    }
    interface ISceneLoaderPlugin {
        /**
         * The friendly name of this plugin.
         */
        name: string;
        /**
         * The file extensions supported by this plugin.
         */
        extensions: string | ISceneLoaderPluginExtensions;
        /**
         * Import meshes into a scene.
         * @param meshesNames An array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param scene The scene to import into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param meshes The meshes array to import into
         * @param particleSystems The particle systems array to import into
         * @param skeletons The skeletons array to import into
         * @param onError The callback when import fails
         * @returns True if successful or false otherwise
         */
        importMesh(meshesNames: any, scene: Scene, data: any, rootUrl: string, meshes: AbstractMesh[], particleSystems: ParticleSystem[], skeletons: Skeleton[], onError?: (message: string, exception?: any) => void): boolean;
        /**
         * Load into a scene.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onError The callback when import fails
         * @returns true if successful or false otherwise
         */
        load(scene: Scene, data: string, rootUrl: string, onError?: (message: string, exception?: any) => void): boolean;
        /**
         * The callback that returns true if the data can be directly loaded.
         */
        canDirectLoad?: (data: string) => boolean;
        /**
         * The callback that allows custom handling of the root url based on the response url.
         */
        rewriteRootURL?: (rootUrl: string, responseURL?: string) => string;
        /**
         * Load into an asset container.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onError The callback when import fails
         * @returns The loaded asset container
         */
        loadAssetContainer(scene: Scene, data: string, rootUrl: string, onError?: (message: string, exception?: any) => void): AssetContainer;
    }
    interface ISceneLoaderPluginAsync {
        /**
         * The friendly name of this plugin.
         */
        name: string;
        /**
         * The file extensions supported by this plugin.
         */
        extensions: string | ISceneLoaderPluginExtensions;
        /**
         * Import meshes into a scene.
         * @param meshesNames An array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param scene The scene to import into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns The loaded meshes, particle systems, and skeletons
         */
        importMeshAsync(meshesNames: any, scene: Scene, data: any, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<{
            meshes: AbstractMesh[];
            particleSystems: ParticleSystem[];
            skeletons: Skeleton[];
        }>;
        /**
         * Load into a scene.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns Nothing
         */
        loadAsync(scene: Scene, data: string, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<void>;
        /**
         * The callback that returns true if the data can be directly loaded.
         */
        canDirectLoad?: (data: string) => boolean;
        /**
         * The callback that allows custom handling of the root url based on the response url.
         */
        rewriteRootURL?: (rootUrl: string, responseURL?: string) => string;
        /**
         * Load into an asset container.
         * @param scene The scene to load into
         * @param data The data to import
         * @param rootUrl The root url for scene and resources
         * @param onProgress The callback when the load progresses
         * @returns The loaded asset container
         */
        loadAssetContainerAsync(scene: Scene, data: string, rootUrl: string, onProgress?: (event: SceneLoaderProgressEvent) => void): Promise<AssetContainer>;
    }
    class SceneLoader {
        private static _ForceFullSceneLoadingForIncremental;
        private static _ShowLoadingScreen;
        private static _CleanBoneMatrixWeights;
        static readonly NO_LOGGING: number;
        static readonly MINIMAL_LOGGING: number;
        static readonly SUMMARY_LOGGING: number;
        static readonly DETAILED_LOGGING: number;
        private static _loggingLevel;
        static ForceFullSceneLoadingForIncremental: boolean;
        static ShowLoadingScreen: boolean;
        static loggingLevel: number;
        static CleanBoneMatrixWeights: boolean;
        static OnPluginActivatedObservable: Observable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        private static _registeredPlugins;
        private static _getDefaultPlugin();
        private static _getPluginForExtension(extension);
        private static _getPluginForDirectLoad(data);
        private static _getPluginForFilename(sceneFilename);
        private static _getDirectLoad(sceneFilename);
        private static _loadData(rootUrl, sceneFilename, scene, onSuccess, onProgress, onError, onDispose, pluginExtension);
        static GetPluginForExtension(extension: string): ISceneLoaderPlugin | ISceneLoaderPluginAsync | ISceneLoaderPluginFactory;
        static IsPluginForExtensionAvailable(extension: string): boolean;
        static RegisterPlugin(plugin: ISceneLoaderPlugin | ISceneLoaderPluginAsync): void;
        /**
         * Import meshes into a scene
         * @param meshNames an array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
         * @param rootUrl a string that defines the root url for scene and resources
         * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
         * @param scene the instance of BABYLON.Scene to append to
         * @param onSuccess a callback with a list of imported meshes, particleSystems, and skeletons when import succeeds
         * @param onProgress a callback with a progress event for each file being loaded
         * @param onError a callback with the scene, a message, and possibly an exception when import fails
         * @param pluginExtension the extension used to determine the plugin
         * @returns The loaded plugin
         */
        static ImportMesh(meshNames: any, rootUrl: string, sceneFilename: string, scene: Scene, onSuccess?: Nullable<(meshes: AbstractMesh[], particleSystems: ParticleSystem[], skeletons: Skeleton[]) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Import meshes into a scene
        * @param meshNames an array of mesh names, a single mesh name, or empty string for all meshes that filter what meshes are imported
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded list of imported meshes, particleSystems, and skeletons
        */
        static ImportMeshAsync(meshNames: any, rootUrl: string, sceneFilename: string, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<{
            meshes: AbstractMesh[];
            particleSystems: ParticleSystem[];
            skeletons: Skeleton[];
        }>;
        /**
        * Load a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param engine is the instance of BABYLON.Engine to use to create the scene
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static Load(rootUrl: string, sceneFilename: any, engine: Engine, onSuccess?: Nullable<(scene: Scene) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Load a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param engine is the instance of BABYLON.Engine to use to create the scene
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded scene
        */
        static LoadAsync(rootUrl: string, sceneFilename: any, engine: Engine, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<Scene>;
        /**
        * Append a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static Append(rootUrl: string, sceneFilename: any, scene: Scene, onSuccess?: Nullable<(scene: Scene) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Append a scene
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The given scene
        */
        static AppendAsync(rootUrl: string, sceneFilename: any, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<Scene>;
        /**
        * Load a scene into an asset container
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onSuccess a callback with the scene when import succeeds
        * @param onProgress a callback with a progress event for each file being loaded
        * @param onError a callback with the scene, a message, and possibly an exception when import fails
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded plugin
        */
        static LoadAssetContainer(rootUrl: string, sceneFilename: any, scene: Scene, onSuccess?: Nullable<(assets: AssetContainer) => void>, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, onError?: Nullable<(scene: Scene, message: string, exception?: any) => void>, pluginExtension?: Nullable<string>): Nullable<ISceneLoaderPlugin | ISceneLoaderPluginAsync>;
        /**
        * Load a scene into an asset container
        * @param rootUrl a string that defines the root url for scene and resources
        * @param sceneFilename a string that defines the name of the scene file. can start with "data:" following by the stringified version of the scene
        * @param scene is the instance of BABYLON.Scene to append to
        * @param onProgress a callback with a progress event for each file being loaded
        * @param pluginExtension the extension used to determine the plugin
        * @returns The loaded asset container
        */
        static LoadAssetContainerAsync(rootUrl: string, sceneFilename: any, scene: Scene, onProgress?: Nullable<(event: SceneLoaderProgressEvent) => void>, pluginExtension?: Nullable<string>): Promise<AssetContainer>;
    }
}

declare module BABYLON {
    class Scalar {
        /**
         * Two pi constants convenient for computation.
         */
        static TwoPi: number;
        /**
         * Boolean : true if the absolute difference between a and b is lower than epsilon (default = 1.401298E-45)
         */
        static WithinEpsilon(a: number, b: number, epsilon?: number): boolean;
        /**
         * Returns a string : the upper case translation of the number i to hexadecimal.
         */
        static ToHex(i: number): string;
        /**
         * Returns -1 if value is negative and +1 is value is positive.
         * Returns the value itself if it's equal to zero.
         */
        static Sign(value: number): number;
        /**
         * Returns the value itself if it's between min and max.
         * Returns min if the value is lower than min.
         * Returns max if the value is greater than max.
         */
        static Clamp(value: number, min?: number, max?: number): number;
        /**
         * Returns the log2 of value.
         */
        static Log2(value: number): number;
        /**
        * Loops the value, so that it is never larger than length and never smaller than 0.
        *
        * This is similar to the modulo operator but it works with floating point numbers.
        * For example, using 3.0 for t and 2.5 for length, the result would be 0.5.
        * With t = 5 and length = 2.5, the result would be 0.0.
        * Note, however, that the behaviour is not defined for negative numbers as it is for the modulo operator
        */
        static Repeat(value: number, length: number): number;
        /**
        * Normalize the value between 0.0 and 1.0 using min and max values
        */
        static Normalize(value: number, min: number, max: number): number;
        /**
        * Denormalize the value from 0.0 and 1.0 using min and max values
        */
        static Denormalize(normalized: number, min: number, max: number): number;
        /**
        * Calculates the shortest difference between two given angles given in degrees.
        */
        static DeltaAngle(current: number, target: number): number;
        /**
        * PingPongs the value t, so that it is never larger than length and never smaller than 0.
        *
        * The returned value will move back and forth between 0 and length
        */
        static PingPong(tx: number, length: number): number;
        /**
        * Interpolates between min and max with smoothing at the limits.
        *
        * This function interpolates between min and max in a similar way to Lerp. However, the interpolation will gradually speed up
        * from the start and slow down toward the end. This is useful for creating natural-looking animation, fading and other transitions.
        */
        static SmoothStep(from: number, to: number, tx: number): number;
        /**
        * Moves a value current towards target.
        *
        * This is essentially the same as Mathf.Lerp but instead the function will ensure that the speed never exceeds maxDelta.
        * Negative values of maxDelta pushes the value away from target.
        */
        static MoveTowards(current: number, target: number, maxDelta: number): number;
        /**
        * Same as MoveTowards but makes sure the values interpolate correctly when they wrap around 360 degrees.
        *
        * Variables current and target are assumed to be in degrees. For optimization reasons, negative values of maxDelta
        *  are not supported and may cause oscillation. To push current away from a target angle, add 180 to that angle instead.
        */
        static MoveTowardsAngle(current: number, target: number, maxDelta: number): number;
        /**
            * Creates a new scalar with values linearly interpolated of "amount" between the start scalar and the end scalar.
            */
        static Lerp(start: number, end: number, amount: number): number;
        /**
        * Same as Lerp but makes sure the values interpolate correctly when they wrap around 360 degrees.
        * The parameter t is clamped to the range [0, 1]. Variables a and b are assumed to be in degrees.
        */
        static LerpAngle(start: number, end: number, amount: number): number;
        /**
        * Calculates the linear parameter t that produces the interpolant value within the range [a, b].
        */
        static InverseLerp(a: number, b: number, value: number): number;
        /**
         * Returns a new scalar located for "amount" (float) on the Hermite spline defined by the scalars "value1", "value3", "tangent1", "tangent2".
         */
        static Hermite(value1: number, tangent1: number, value2: number, tangent2: number, amount: number): number;
        /**
        * Returns a random float number between and min and max values
        */
        static RandomRange(min: number, max: number): number;
        /**
        * This function returns percentage of a number in a given range.
        *
        * RangeToPercent(40,20,60) will return 0.5 (50%)
        * RangeToPercent(34,0,100) will return 0.34 (34%)
        */
        static RangeToPercent(number: number, min: number, max: number): number;
        /**
        * This function returns number that corresponds to the percentage in a given range.
        *
        * PercentToRange(0.34,0,100) will return 34.
        */
        static PercentToRange(percent: number, min: number, max: number): number;
        /**
         * Returns the angle converted to equivalent value between -Math.PI and Math.PI radians.
         * @param angle The angle to normalize in radian.
         * @return The converted angle.
         */
        static NormalizeRadians(angle: number): number;
    }
}

declare module BABYLON {
    const ToGammaSpace: number;
    const ToLinearSpace = 2.2;
    const Epsilon = 0.001;
    /**
     * Class used to hold a RBG color
     */
    class Color3 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Creates a new Color3 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         */
        constructor(
            /**
             * Defines the red component (between 0 and 1, default is 0)
             */
            r?: number, 
            /**
             * Defines the green component (between 0 and 1, default is 0)
             */
            g?: number, 
            /**
             * Defines the blue component (between 0 and 1, default is 0)
             */
            b?: number);
        /**
         * Creates a string with the Color3 current values
         * @returns the string representation of the Color3 object
         */
        toString(): string;
        /**
         * Returns the string "Color3"
         * @returns "Color3"
         */
        getClassName(): string;
        /**
         * Compute the Color3 hash code
         * @returns an unique number that can be used to hash Color3 objects
         */
        getHashCode(): number;
        /**
         * Stores in the passed array from the passed starting index the red, green, blue values as successive elements
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color3 object
         */
        toArray(array: FloatArray, index?: number): Color3;
        /**
         * Returns a new {BABYLON.Color4} object from the current Color3 and the passed alpha
         * @param alpha defines the alpha component on the new {BABYLON.Color4} object (default is 1)
         * @returns a new {BABYLON.Color4} object
         */
        toColor4(alpha?: number): Color4;
        /**
         * Returns a new array populated with 3 numeric elements : red, green and blue values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Returns the luminance value
         * @returns a float value
         */
        toLuminance(): number;
        /**
         * Multiply each Color3 rgb values by the passed Color3 rgb values in a new Color3 object
         * @param otherColor defines the second operand
         * @returns the new Color3 object
         */
        multiply(otherColor: Color3): Color3;
        /**
         * Multiply the rgb values of the Color3 and the passed Color3 and stores the result in the object "result"
         * @param otherColor defines the second operand
         * @param result defines the Color3 object where to store the result
         * @returns the current Color3
         */
        multiplyToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Determines equality between Color3 objects
         * @param otherColor defines the second operand
         * @returns true if the rgb values are equal to the passed ones
         */
        equals(otherColor: Color3): boolean;
        /**
         * Determines equality between the current Color3 object and a set of r,b,g values
         * @param r defines the red component to check
         * @param g defines the green component to check
         * @param b defines the blue component to check
         * @returns true if the rgb values are equal to the passed ones
         */
        equalsFloats(r: number, g: number, b: number): boolean;
        /**
         * Multiplies in place each rgb value by scale
         * @param scale defines the scaling factor
         * @returns the updated Color3
         */
        scale(scale: number): Color3;
        /**
         * Multiplies the rgb values by scale and stores the result into "result"
         * @param scale defines the scaling factor
         * @param result defines the Color3 object where to store the result
         * @returns the unmodified current Color3
         */
        scaleToRef(scale: number, result: Color3): Color3;
        /**
         * Scale the current Color3 values by a factor and add the result to a given Color3
         * @param scale defines the scale factor
         * @param result defines color to store the result into
         * @returns the unmodified current Color3
         */
        scaleAndAddToRef(scale: number, result: Color3): Color3;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into
         * @returns the original Color3
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color3): Color3;
        /**
         * Creates a new Color3 set with the added values of the current Color3 and of the passed one
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        add(otherColor: Color3): Color3;
        /**
         * Stores the result of the addition of the current Color3 and passed one rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        addToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Returns a new Color3 set with the subtracted values of the passed one from the current Color3
         * @param otherColor defines the second operand
         * @returns the new Color3
         */
        subtract(otherColor: Color3): Color3;
        /**
         * Stores the result of the subtraction of passed one from the current Color3 rgb values into "result"
         * @param otherColor defines the second operand
         * @param result defines Color3 object to store the result into
         * @returns the unmodified current Color3
         */
        subtractToRef(otherColor: Color3, result: Color3): Color3;
        /**
         * Copy the current object
         * @returns a new Color3 copied the current one
         */
        clone(): Color3;
        /**
         * Copies the rgb values from the source in the current Color3
         * @param source defines the source Color3 object
         * @returns the updated Color3 object
         */
        copyFrom(source: Color3): Color3;
        /**
         * Updates the Color3 rgb values from the passed floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        copyFromFloats(r: number, g: number, b: number): Color3;
        /**
         * Updates the Color3 rgb values from the passed floats
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @returns the current Color3 object
         */
        set(r: number, g: number, b: number): Color3;
        /**
         * Compute the Color3 hexadecimal code as a string
         * @returns a string containing the hexadecimal representation of the Color3 object
         */
        toHexString(): string;
        /**
         * Computes a new Color3 converted from the current one to linear space
         * @returns a new Color3 object
         */
        toLinearSpace(): Color3;
        /**
         * Converts the Color3 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the linear space version
         * @returns the unmodified Color3
         */
        toLinearSpaceToRef(convertedColor: Color3): Color3;
        /**
         * Computes a new Color3 converted from the current one to gamma space
         * @returns a new Color3 object
         */
        toGammaSpace(): Color3;
        /**
         * Converts the Color3 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color3 object where to store the gamma space version
         * @returns the unmodified Color3
         */
        toGammaSpaceToRef(convertedColor: Color3): Color3;
        /**
         * Creates a new Color3 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color3 object
         */
        static FromHexString(hex: string): Color3;
        /**
         * Creates a new Vector3 from the starting index of the passed array
         * @param array defines the source array
         * @param offset defines an offset in the source array
         * @returns a new Color3 object
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Color3;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number): Color3;
        /**
         * Creates a new Color3 with values linearly interpolated of "amount" between the start Color3 and the end Color3
         * @param start defines the start Color3 value
         * @param end defines the end Color3 value
         * @param amount defines the gradient value between start and end
         * @returns a new Color3 object
         */
        static Lerp(start: Color3, end: Color3, amount: number): Color3;
        /**
         * Returns a Color3 value containing a red color
         * @returns a new Color3 object
         */
        static Red(): Color3;
        /**
         * Returns a Color3 value containing a green color
         * @returns a new Color3 object
         */
        static Green(): Color3;
        /**
         * Returns a Color3 value containing a blue color
         * @returns a new Color3 object
         */
        static Blue(): Color3;
        /**
         * Returns a Color3 value containing a black color
         * @returns a new Color3 object
         */
        static Black(): Color3;
        /**
         * Returns a Color3 value containing a white color
         * @returns a new Color3 object
         */
        static White(): Color3;
        /**
         * Returns a Color3 value containing a purple color
         * @returns a new Color3 object
         */
        static Purple(): Color3;
        /**
         * Returns a Color3 value containing a magenta color
         * @returns a new Color3 object
         */
        static Magenta(): Color3;
        /**
         * Returns a Color3 value containing a yellow color
         * @returns a new Color3 object
         */
        static Yellow(): Color3;
        /**
         * Returns a Color3 value containing a gray color
         * @returns a new Color3 object
         */
        static Gray(): Color3;
        /**
         * Returns a Color3 value containing a teal color
         * @returns a new Color3 object
         */
        static Teal(): Color3;
        /**
         * Returns a Color3 value containing a random color
         * @returns a new Color3 object
         */
        static Random(): Color3;
    }
    /**
     * Class used to hold a RBGA color
     */
    class Color4 {
        /**
         * Defines the red component (between 0 and 1, default is 0)
         */
        r: number;
        /**
         * Defines the green component (between 0 and 1, default is 0)
         */
        g: number;
        /**
         * Defines the blue component (between 0 and 1, default is 0)
         */
        b: number;
        /**
         * Defines the alpha component (between 0 and 1, default is 1)
         */
        a: number;
        /**
         * Creates a new Color4 object from red, green, blue values, all between 0 and 1
         * @param r defines the red component (between 0 and 1, default is 0)
         * @param g defines the green component (between 0 and 1, default is 0)
         * @param b defines the blue component (between 0 and 1, default is 0)
         * @param a defines the alpha component (between 0 and 1, default is 1)
         */
        constructor(
            /**
             * Defines the red component (between 0 and 1, default is 0)
             */
            r?: number, 
            /**
             * Defines the green component (between 0 and 1, default is 0)
             */
            g?: number, 
            /**
             * Defines the blue component (between 0 and 1, default is 0)
             */
            b?: number, 
            /**
             * Defines the alpha component (between 0 and 1, default is 1)
             */
            a?: number);
        /**
         * Adds in place the passed Color4 values to the current Color4 object
         * @param right defines the second operand
         * @returns the current updated Color4 object
         */
        addInPlace(right: Color4): Color4;
        /**
         * Creates a new array populated with 4 numeric elements : red, green, blue, alpha values
         * @returns the new array
         */
        asArray(): number[];
        /**
         * Stores from the starting index in the passed array the Color4 successive values
         * @param array defines the array where to store the r,g,b components
         * @param index defines an optional index in the target array to define where to start storing values
         * @returns the current Color4 object
         */
        toArray(array: number[], index?: number): Color4;
        /**
         * Creates a new Color4 set with the added values of the current Color4 and of the passed one
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        add(right: Color4): Color4;
        /**
         * Creates a new Color4 set with the subtracted values of the passed one from the current Color4
         * @param right defines the second operand
         * @returns a new Color4 object
         */
        subtract(right: Color4): Color4;
        /**
         * Subtracts the passed ones from the current Color4 values and stores the results in "result"
         * @param right defines the second operand
         * @param result defines the Color4 object where to store the result
         * @returns the current Color4 object
         */
        subtractToRef(right: Color4, result: Color4): Color4;
        /**
         * Creates a new Color4 with the current Color4 values multiplied by scale
         * @param scale defines the scaling factor to apply
         * @returns a new Color4 object
         */
        scale(scale: number): Color4;
        /**
         * Multiplies the current Color4 values by scale and stores the result in "result"
         * @param scale defines the scaling factor to apply
         * @param result defines the Color4 object where to store the result
         * @returns the current unmodified Color4
         */
        scaleToRef(scale: number, result: Color4): Color4;
        /**
         * Scale the current Color4 values by a factor and add the result to a given Color4
         * @param scale defines the scale factor
         * @param result defines the Color4 object where to store the result
         * @returns the unmodified current Color4
         */
        scaleAndAddToRef(scale: number, result: Color4): Color4;
        /**
         * Clamps the rgb values by the min and max values and stores the result into "result"
         * @param min defines minimum clamping value (default is 0)
         * @param max defines maximum clamping value (default is 1)
         * @param result defines color to store the result into.
         * @returns the cuurent Color4
         */
        clampToRef(min: number | undefined, max: number | undefined, result: Color4): Color4;
        /**
          * Multipy an Color4 value by another and return a new Color4 object
          * @param color defines the Color4 value to multiply by
          * @returns a new Color4 object
          */
        multiply(color: Color4): Color4;
        /**
         * Multipy a Color4 value by another and push the result in a reference value
         * @param color defines the Color4 value to multiply by
         * @param result defines the Color4 to fill the result in
         * @returns the result Color4
         */
        multiplyToRef(color: Color4, result: Color4): Color4;
        /**
         * Creates a string with the Color4 current values
         * @returns the string representation of the Color4 object
         */
        toString(): string;
        /**
         * Returns the string "Color4"
         * @returns "Color4"
         */
        getClassName(): string;
        /**
         * Compute the Color4 hash code
         * @returns an unique number that can be used to hash Color4 objects
         */
        getHashCode(): number;
        /**
         * Creates a new Color4 copied from the current one
         * @returns a new Color4 object
         */
        clone(): Color4;
        /**
         * Copies the passed Color4 values into the current one
         * @param source defines the source Color4 object
         * @returns the current updated Color4 object
         */
        copyFrom(source: Color4): Color4;
        /**
         * Copies the passed float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        copyFromFloats(r: number, g: number, b: number, a: number): Color4;
        /**
         * Copies the passed float values into the current one
         * @param r defines the red component to read from
         * @param g defines the green component to read from
         * @param b defines the blue component to read from
         * @param a defines the alpha component to read from
         * @returns the current updated Color4 object
         */
        set(r: number, g: number, b: number, a: number): Color4;
        /**
         * Compute the Color4 hexadecimal code as a string
         * @returns a string containing the hexadecimal representation of the Color4 object
         */
        toHexString(): string;
        /**
         * Computes a new Color4 converted from the current one to linear space
         * @returns a new Color4 object
         */
        toLinearSpace(): Color4;
        /**
         * Converts the Color4 values to linear space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the linear space version
         * @returns the unmodified Color4
         */
        toLinearSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Computes a new Color4 converted from the current one to gamma space
         * @returns a new Color4 object
         */
        toGammaSpace(): Color4;
        /**
         * Converts the Color4 values to gamma space and stores the result in "convertedColor"
         * @param convertedColor defines the Color4 object where to store the gamma space version
         * @returns the unmodified Color4
         */
        toGammaSpaceToRef(convertedColor: Color4): Color4;
        /**
         * Creates a new Color4 from the string containing valid hexadecimal values
         * @param hex defines a string containing valid hexadecimal values
         * @returns a new Color4 object
         */
        static FromHexString(hex: string): Color4;
        /**
         * Creates a new Color4 object set with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @returns a new Color4 object
         */
        static Lerp(left: Color4, right: Color4, amount: number): Color4;
        /**
         * Set the passed "result" with the linearly interpolated values of "amount" between the left Color4 object and the right Color4 object
         * @param left defines the start value
         * @param right defines the end value
         * @param amount defines the gradient factor
         * @param result defines the Color4 object where to store data
         */
        static LerpToRef(left: Color4, right: Color4, amount: number, result: Color4): void;
        /**
         * Creates a new Color4 from the starting index element of the passed array
         * @param array defines the source array to read from
         * @param offset defines the offset in the source array
         * @returns a new Color4 object
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Color4;
        /**
         * Creates a new Color3 from integer values (< 256)
         * @param r defines the red component to read from (value between 0 and 255)
         * @param g defines the green component to read from (value between 0 and 255)
         * @param b defines the blue component to read from (value between 0 and 255)
         * @param a defines the alpha component to read from (value between 0 and 255)
         * @returns a new Color3 object
         */
        static FromInts(r: number, g: number, b: number, a: number): Color4;
        /**
         * Check the content of a given array and convert it to an array containing RGBA data
         * If the original array was already containing count * 4 values then it is returned directly
         * @param colors defines the array to check
         * @param count defines the number of RGBA data to expect
         * @returns an array containing count * 4 values (RGBA)
         */
        static CheckColors4(colors: number[], count: number): number[];
    }
    class Vector2 {
        x: number;
        y: number;
        /**
         * Creates a new Vector2 from the passed x and y coordinates.
         */
        constructor(x: number, y: number);
        /**
         * Returns a string with the Vector2 coordinates.
         */
        toString(): string;
        /**
         * Returns the string "Vector2"
         */
        getClassName(): string;
        /**
         * Returns the Vector2 hash code as a number.
         */
        getHashCode(): number;
        /**
         * Sets the Vector2 coordinates in the passed array or Float32Array from the passed index.
         * Returns the Vector2.
         */
        toArray(array: FloatArray, index?: number): Vector2;
        /**
         * Returns a new array with 2 elements : the Vector2 coordinates.
         */
        asArray(): number[];
        /**
         *  Sets the Vector2 coordinates with the passed Vector2 coordinates.
         * Returns the updated Vector2.
         */
        copyFrom(source: Vector2): Vector2;
        /**
         * Sets the Vector2 coordinates with the passed floats.
         * Returns the updated Vector2.
         */
        copyFromFloats(x: number, y: number): Vector2;
        /**
         * Sets the Vector2 coordinates with the passed floats.
         * Returns the updated Vector2.
         */
        set(x: number, y: number): Vector2;
        /**
         * Returns a new Vector2 set with the addition of the current Vector2 and the passed one coordinates.
         */
        add(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the addition of the current Vector2 and the passed one coordinates.
         * Returns the Vector2.
         */
        addToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Set the Vector2 coordinates by adding the passed Vector2 coordinates.
         * Returns the updated Vector2.
         */
        addInPlace(otherVector: Vector2): Vector2;
        /**
         * Returns a new Vector2 by adding the current Vector2 coordinates to the passed Vector3 x, y coordinates.
         */
        addVector3(otherVector: Vector3): Vector2;
        /**
         * Returns a new Vector2 set with the subtracted coordinates of the passed one from the current Vector2.
         */
        subtract(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the subtraction of the passed one from the current Vector2 coordinates.
         * Returns the Vector2.
         */
        subtractToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Sets the current Vector2 coordinates by subtracting from it the passed one coordinates.
         * Returns the updated Vector2.
         */
        subtractInPlace(otherVector: Vector2): Vector2;
        /**
         * Multiplies in place the current Vector2 coordinates by the passed ones.
         * Returns the updated Vector2.
         */
        multiplyInPlace(otherVector: Vector2): Vector2;
        /**
         * Returns a new Vector2 set with the multiplication of the current Vector2 and the passed one coordinates.
         */
        multiply(otherVector: Vector2): Vector2;
        /**
         * Sets "result" coordinates with the multiplication of the current Vector2 and the passed one coordinates.
         * Returns the Vector2.
         */
        multiplyToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Returns a new Vector2 set with the Vector2 coordinates multiplied by the passed floats.
         */
        multiplyByFloats(x: number, y: number): Vector2;
        /**
         * Returns a new Vector2 set with the Vector2 coordinates divided by the passed one coordinates.
         */
        divide(otherVector: Vector2): Vector2;
        /**
         * Sets the "result" coordinates with the Vector2 divided by the passed one coordinates.
         * Returns the Vector2.
         */
        divideToRef(otherVector: Vector2, result: Vector2): Vector2;
        /**
         * Divides the current Vector3 coordinates by the passed ones.
         * Returns the updated Vector3.
         */
        divideInPlace(otherVector: Vector2): Vector2;
        /**
         * Returns a new Vector2 with current Vector2 negated coordinates.
         */
        negate(): Vector2;
        /**
         * Multiply the Vector2 coordinates by scale.
         * Returns the updated Vector2.
         */
        scaleInPlace(scale: number): Vector2;
        /**
         * Returns a new Vector2 scaled by "scale" from the current Vector2.
         */
        scale(scale: number): Vector2;
        /**
         * Scale the current Vector2 values by a factor to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleToRef(scale: number, result: Vector2): Vector2;
        /**
         * Scale the current Vector2 values by a factor and add the result to a given Vector2
         * @param scale defines the scale factor
         * @param result defines the Vector2 object where to store the result
         * @returns the unmodified current Vector2
         */
        scaleAndAddToRef(scale: number, result: Vector2): Vector2;
        /**
         * Boolean : True if the passed vector coordinates strictly equal the current Vector2 ones.
         */
        equals(otherVector: Vector2): boolean;
        /**
         * Boolean : True if the passed vector coordinates are close to the current ones by a distance of epsilon.
         */
        equalsWithEpsilon(otherVector: Vector2, epsilon?: number): boolean;
        /**
         * Returns the vector length (float).
         */
        length(): number;
        /**
         * Returns the vector squared length (float);
         */
        lengthSquared(): number;
        /**
         * Normalize the vector.
         * Returns the updated Vector2.
         */
        normalize(): Vector2;
        /**
         * Returns a new Vector2 copied from the Vector2.
         */
        clone(): Vector2;
        /**
         * Returns a new Vector2(0, 0)
         */
        static Zero(): Vector2;
        /**
         * Returns a new Vector2(1, 1)
         */
        static One(): Vector2;
        /**
         * Returns a new Vector2 set from the passed index element of the passed array.
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector2;
        /**
         * Sets "result" from the passed index element of the passed array.
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector2): void;
        /**
         * Retuns a new Vector2 located for "amount" (float) on the CatmullRom  spline defined by the passed four Vector2.
         */
        static CatmullRom(value1: Vector2, value2: Vector2, value3: Vector2, value4: Vector2, amount: number): Vector2;
        /**
         * Returns a new Vector2 set with same the coordinates than "value" ones if the vector "value" is in the square defined by "min" and "max".
         * If a coordinate of "value" is lower than "min" coordinates, the returned Vector2 is given this "min" coordinate.
         * If a coordinate of "value" is greater than "max" coordinates, the returned Vector2 is given this "max" coordinate.
         */
        static Clamp(value: Vector2, min: Vector2, max: Vector2): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the Hermite spline defined by the vectors "value1", "value3", "tangent1", "tangent2".
         */
        static Hermite(value1: Vector2, tangent1: Vector2, value2: Vector2, tangent2: Vector2, amount: number): Vector2;
        /**
         * Returns a new Vector2 located for "amount" (float) on the linear interpolation between the vector "start" adn the vector "end".
         */
        static Lerp(start: Vector2, end: Vector2, amount: number): Vector2;
        /**
         * Returns the dot product (float) of the vector "left" and the vector "right".
         */
        static Dot(left: Vector2, right: Vector2): number;
        /**
         * Returns a new Vector2 equal to the normalized passed vector.
         */
        static Normalize(vector: Vector2): Vector2;
        /**
         * Returns a new Vecto2 set with the minimal coordinate values from the "left" and "right" vectors.
         */
        static Minimize(left: Vector2, right: Vector2): Vector2;
        /**
         * Returns a new Vecto2 set with the maximal coordinate values from the "left" and "right" vectors.
         */
        static Maximize(left: Vector2, right: Vector2): Vector2;
        /**
         * Returns a new Vecto2 set with the transformed coordinates of the passed vector by the passed transformation matrix.
         */
        static Transform(vector: Vector2, transformation: Matrix): Vector2;
        /**
         * Transforms the passed vector coordinates by the passed transformation matrix and stores the result in the vector "result" coordinates.
         */
        static TransformToRef(vector: Vector2, transformation: Matrix, result: Vector2): void;
        /**
         * Boolean : True if the point "p" is in the triangle defined by the vertors "p0", "p1", "p2"
         */
        static PointInTriangle(p: Vector2, p0: Vector2, p1: Vector2, p2: Vector2): boolean;
        /**
         * Returns the distance (float) between the vectors "value1" and "value2".
         */
        static Distance(value1: Vector2, value2: Vector2): number;
        /**
         * Returns the squared distance (float) between the vectors "value1" and "value2".
         */
        static DistanceSquared(value1: Vector2, value2: Vector2): number;
        /**
         * Returns a new Vecto2 located at the center of the vectors "value1" and "value2".
         */
        static Center(value1: Vector2, value2: Vector2): Vector2;
        /**
         * Returns the shortest distance (float) between the point "p" and the segment defined by the two points "segA" and "segB".
         */
        static DistanceOfPointFromSegment(p: Vector2, segA: Vector2, segB: Vector2): number;
    }
    /**
     * Classed used to store (x,y,z) vector representation
     * A Vector3 is the main object used in 3D geometry
     * It can represent etiher the coordinates of a point the space, either a direction
     * Reminder: Babylon.js uses a left handed forward facing system
     */
    class Vector3 {
        /**
         * Defines the first coordinates (on X axis)
         */
        x: number;
        /**
         * Defines the second coordinates (on Y axis)
         */
        y: number;
        /**
         * Defines the third coordinates (on Z axis)
         */
        z: number;
        /**
         * Creates a new Vector3 object from the passed x, y, z (floats) coordinates.
         * @param x defines the first coordinates (on X axis)
         * @param y defines the second coordinates (on Y axis)
         * @param z defines the third coordinates (on Z axis)
         */
        constructor(
            /**
             * Defines the first coordinates (on X axis)
             */
            x: number, 
            /**
             * Defines the second coordinates (on Y axis)
             */
            y: number, 
            /**
             * Defines the third coordinates (on Z axis)
             */
            z: number);
        /**
         * Creates a string representation of the Vector3
         * @returns a string with the Vector3 coordinates.
         */
        toString(): string;
        /**
         * Gets the class name
         * @returns the string "Vector3"
         */
        getClassName(): string;
        /**
         * Creates the Vector3 hash code
         * @returns a number which tends to be unique between Vector3 instances
         */
        getHashCode(): number;
        /**
         * Creates an array containing three elements : the coordinates of the Vector3
         * @returns a new array of numbers
         */
        asArray(): number[];
        /**
         * Populates the passed array or Float32Array from the passed index with the successive coordinates of the Vector3
         * @param array defines the destination array
         * @param index defines the offset in the destination array
         * @returns the current Vector3
         */
        toArray(array: FloatArray, index?: number): Vector3;
        /**
         * Converts the current Vector3 into a quaternion (considering that the Vector3 contains Euler angles representation of a rotation)
         * @returns a new Quaternion object, computed from the Vector3 coordinates
         */
        toQuaternion(): Quaternion;
        /**
         * Adds the passed vector to the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        addInPlace(otherVector: Vector3): Vector3;
        /**
         * Gets a new Vector3, result of the addition the current Vector3 and the passed vector
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        add(otherVector: Vector3): Vector3;
        /**
         * Adds the current Vector3 to the passed one and stores the result in the vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        addToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Subtract the passed vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        subtractInPlace(otherVector: Vector3): Vector3;
        /**
         * Returns a new Vector3, result of the subtraction of the passed vector from the current Vector3
         * @param otherVector defines the second operand
         * @returns the resulting Vector3
         */
        subtract(otherVector: Vector3): Vector3;
        /**
         * Subtracts the passed vector from the current Vector3 and stores the result in the vector "result".
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the subtraction of the passed floats from the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the resulting Vector3
         */
        subtractFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Subtracts the passed floats from the current Vector3 coordinates and set the passed vector "result" with this result
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, result: Vector3): Vector3;
        /**
         * Gets a new Vector3 set with the current Vector3 negated coordinates
         * @returns a new Vector3
         */
        negate(): Vector3;
        /**
         * Multiplies the Vector3 coordinates by the float "scale"
         * @param scale defines the multiplier factor
         * @returns the current updated Vector3
         */
        scaleInPlace(scale: number): Vector3;
        /**
         * Returns a new Vector3 set with the current Vector3 coordinates multiplied by the float "scale"
         * @param scale defines the multiplier factor
         * @returns a new Vector3
         */
        scale(scale: number): Vector3;
        /**
         * Multiplies the current Vector3 coordinates by the float "scale" and stores the result in the passed vector "result" coordinates
         * @param scale defines the multiplier factor
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        scaleToRef(scale: number, result: Vector3): Vector3;
        /**
         * Scale the current Vector3 values by a factor and add the result to a given Vector3
         * @param scale defines the scale factor
         * @param result defines the Vector3 object where to store the result
         * @returns the unmodified current Vector3
         */
        scaleAndAddToRef(scale: number, result: Vector3): Vector3;
        /**
         * Returns true if the current Vector3 and the passed vector coordinates are strictly equal
         * @param otherVector defines the second operand
         * @returns true if both vectors are equals
         */
        equals(otherVector: Vector3): boolean;
        /**
         * Returns true if the current Vector3 and the passed vector coordinates are distant less than epsilon
         * @param otherVector defines the second operand
         * @param epsilon defines the minimal distance to define values as equals
         * @returns true if both vectors are distant less than epsilon
         */
        equalsWithEpsilon(otherVector: Vector3, epsilon?: number): boolean;
        /**
         * Returns true if the current Vector3 coordinates equals the passed floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns true if both vectors are equals
         */
        equalsToFloats(x: number, y: number, z: number): boolean;
        /**
         * Multiplies the current Vector3 coordinates by the passed ones
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        multiplyInPlace(otherVector: Vector3): Vector3;
        /**
         * Returns a new Vector3, result of the multiplication of the current Vector3 by the passed vector
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        multiply(otherVector: Vector3): Vector3;
        /**
         * Multiplies the current Vector3 by the passed one and stores the result in the passed vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        multiplyToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Returns a new Vector3 set with the result of the mulliplication of the current Vector3 coordinates by the passed floats
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the new Vector3
         */
        multiplyByFloats(x: number, y: number, z: number): Vector3;
        /**
         * Returns a new Vector3 set with the result of the division of the current Vector3 coordinates by the passed ones
         * @param otherVector defines the second operand
         * @returns the new Vector3
         */
        divide(otherVector: Vector3): Vector3;
        /**
         * Divides the current Vector3 coordinates by the passed ones and stores the result in the passed vector "result"
         * @param otherVector defines the second operand
         * @param result defines the Vector3 object where to store the result
         * @returns the current Vector3
         */
        divideToRef(otherVector: Vector3, result: Vector3): Vector3;
        /**
         * Divides the current Vector3 coordinates by the passed ones.
         * @param otherVector defines the second operand
         * @returns the current updated Vector3
         */
        divideInPlace(otherVector: Vector3): Vector3;
        /**
         * Updates the current Vector3 with the minimal coordinate values between its and the passed vector ones
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        minimizeInPlace(other: Vector3): Vector3;
        /**
         * Updates the current Vector3 with the maximal coordinate values between its and the passed vector ones.
         * @param other defines the second operand
         * @returns the current updated Vector3
         */
        maximizeInPlace(other: Vector3): Vector3;
        /**
         * Gets a boolean indicating that the vector is non uniform meaning x, y or z are not all the same
         */
        readonly isNonUniform: boolean;
        /**
         * Gets the length of the Vector3
         * @returns the length of the Vecto3
         */
        length(): number;
        /**
         * Gets the squared length of the Vector3
         * @returns squared length of the Vector3
         */
        lengthSquared(): number;
        /**
         * Normalize the current Vector3.
         * Please note that this is an in place operation.
         * @returns the current updated Vector3
         */
        normalize(): Vector3;
        /**
         * Normalize the current Vector3 to a new vector
         * @returns the new Vector3
         */
        normalizeToNew(): Vector3;
        /**
         * Normalize the current Vector3 to the reference
         * @param reference define the Vector3 to update
         * @returns the updated Vector3
         */
        normalizeToRef(reference: Vector3): Vector3;
        /**
         * Creates a new Vector3 copied from the current Vector3
         * @returns the new Vector3
         */
        clone(): Vector3;
        /**
         * Copies the passed vector coordinates to the current Vector3 ones
         * @param source defines the source Vector3
         * @returns the current updated Vector3
         */
        copyFrom(source: Vector3): Vector3;
        /**
         * Copies the passed floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        copyFromFloats(x: number, y: number, z: number): Vector3;
        /**
         * Copies the passed floats to the current Vector3 coordinates
         * @param x defines the x coordinate of the operand
         * @param y defines the y coordinate of the operand
         * @param z defines the z coordinate of the operand
         * @returns the current updated Vector3
         */
        set(x: number, y: number, z: number): Vector3;
        /**
         * Get the clip factor between two vectors
         * @param vector0 defines the first operand
         * @param vector1 defines the second operand
         * @param axis defines the axis to use
         * @param size defines the size along the axis
         * @returns the clip factor
         */
        static GetClipFactor(vector0: Vector3, vector1: Vector3, axis: Vector3, size: number): number;
        /**
         * Get angle between two vectors
         * @param vector0 angle between vector0 and vector1
         * @param vector1 angle between vector0 and vector1
         * @param normal direction of the normal
         * @return the angle between vector0 and vector1
         */
        static GetAngleBetweenVectors(vector0: Vector3, vector1: Vector3, normal: Vector3): number;
        /**
         * Returns a new Vector3 set from the index "offset" of the passed array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector3;
        /**
         * Returns a new Vector3 set from the index "offset" of the passed Float32Array
         * This function is deprecated.  Use FromArray instead
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @returns the new Vector3
         */
        static FromFloatArray(array: Float32Array, offset?: number): Vector3;
        /**
         * Sets the passed vector "result" with the element values from the index "offset" of the passed array
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector3): void;
        /**
         * Sets the passed vector "result" with the element values from the index "offset" of the passed Float32Array
         * This function is deprecated.  Use FromArrayToRef instead.
         * @param array defines the source array
         * @param offset defines the offset in the source array
         * @param result defines the Vector3 where to store the result
         */
        static FromFloatArrayToRef(array: Float32Array, offset: number, result: Vector3): void;
        /**
         * Sets the passed vector "result" with the passed floats.
         * @param x defines the x coordinate of the source
         * @param y defines the y coordinate of the source
         * @param z defines the z coordinate of the source
         * @param result defines the Vector3 where to store the result
         */
        static FromFloatsToRef(x: number, y: number, z: number, result: Vector3): void;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 0.0)
         * @returns a new empty Vector3
         */
        static Zero(): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 1.0, 1.0)
         * @returns a new unit Vector3
         */
        static One(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 1.0, 0.0)
         * @returns a new up Vector3
         */
        static Up(): Vector3;
        /**
         * Returns a new Vector3 set to (0.0, 0.0, 1.0)
         * @returns a new forward Vector3
         */
        static Forward(): Vector3;
        /**
         * Returns a new Vector3 set to (1.0, 0.0, 0.0)
         * @returns a new right Vector3
         */
        static Right(): Vector3;
        /**
         * Returns a new Vector3 set to (-1.0, 0.0, 0.0)
         * @returns a new left Vector3
         */
        static Left(): Vector3;
        /**
         * Returns a new Vector3 set with the result of the transformation by the passed matrix of the passed vector.
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the transformed Vector3
         */
        static TransformCoordinates(vector: Vector3, transformation: Matrix): Vector3;
        /**
         * Sets the passed vector "result" coordinates with the result of the transformation by the passed matrix of the passed vector
         * This method computes tranformed coordinates only, not transformed direction vectors (ie. it takes translation in account)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesToRef(vector: Vector3, transformation: Matrix, result: Vector3): void;
        /**
         * Sets the passed vector "result" coordinates with the result of the transformation by the passed matrix of the passed floats (x, y, z)
         * This method computes tranformed coordinates only, not transformed direction vectors
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformCoordinatesFromFloatsToRef(x: number, y: number, z: number, transformation: Matrix, result: Vector3): void;
        /**
         * Returns a new Vector3 set with the result of the normal transformation by the passed matrix of the passed vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @returns the new Vector3
         */
        static TransformNormal(vector: Vector3, transformation: Matrix): Vector3;
        /**
         * Sets the passed vector "result" with the result of the normal transformation by the passed matrix of the passed vector
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param vector defines the Vector3 to transform
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalToRef(vector: Vector3, transformation: Matrix, result: Vector3): void;
        /**
         * Sets the passed vector "result" with the result of the normal transformation by the passed matrix of the passed floats (x, y, z)
         * This methods computes transformed normalized direction vectors only (ie. it does not apply translation)
         * @param x define the x coordinate of the source vector
         * @param y define the y coordinate of the source vector
         * @param z define the z coordinate of the source vector
         * @param transformation defines the transformation matrix
         * @param result defines the Vector3 where to store the result
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, transformation: Matrix, result: Vector3): void;
        /**
         * Returns a new Vector3 located for "amount" on the CatmullRom interpolation spline defined by the vectors "value1", "value2", "value3", "value4"
         * @param value1 defines the first control point
         * @param value2 defines the second control point
         * @param value3 defines the third control point
         * @param value4 defines the fourth control point
         * @param amount defines the amount on the spline to use
         * @returns the new Vector3
         */
        static CatmullRom(value1: Vector3, value2: Vector3, value3: Vector3, value4: Vector3, amount: number): Vector3;
        /**
         * Returns a new Vector3 set with the coordinates of "value", if the vector "value" is in the cube defined by the vectors "min" and "max"
         * If a coordinate value of "value" is lower than one of the "min" coordinate, then this "value" coordinate is set with the "min" one
         * If a coordinate value of "value" is greater than one of the "max" coordinate, then this "value" coordinate is set with the "max" one
         * @param value defines the current value
         * @param min defines the lower range value
         * @param max defines the upper range value
         * @returns the new Vector3
         */
        static Clamp(value: Vector3, min: Vector3, max: Vector3): Vector3;
        /**
         * Returns a new Vector3 located for "amount" (float) on the Hermite interpolation spline defined by the vectors "value1", "tangent1", "value2", "tangent2"
         * @param value1 defines the first control point
         * @param tangent1 defines the first tangent vector
         * @param value2 defines the second control point
         * @param tangent2 defines the second tangent vector
         * @param amount defines the amount on the interpolation spline (between 0 and 1)
         * @returns the new Vector3
         */
        static Hermite(value1: Vector3, tangent1: Vector3, value2: Vector3, tangent2: Vector3, amount: number): Vector3;
        /**
         * Returns a new Vector3 located for "amount" (float) on the linear interpolation between the vectors "start" and "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @returns the new Vector3
         */
        static Lerp(start: Vector3, end: Vector3, amount: number): Vector3;
        /**
         * Sets the passed vector "result" with the result of the linear interpolation from the vector "start" for "amount" to the vector "end"
         * @param start defines the start value
         * @param end defines the end value
         * @param amount max defines amount between both (between 0 and 1)
         * @param result defines the Vector3 where to store the result
         */
        static LerpToRef(start: Vector3, end: Vector3, amount: number, result: Vector3): void;
        /**
         * Returns the dot product (float) between the vectors "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the dot product
         */
        static Dot(left: Vector3, right: Vector3): number;
        /**
         * Returns a new Vector3 as the cross product of the vectors "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @returns the cross product
         */
        static Cross(left: Vector3, right: Vector3): Vector3;
        /**
         * Sets the passed vector "result" with the cross product of "left" and "right"
         * The cross product is then orthogonal to both "left" and "right"
         * @param left defines the left operand
         * @param right defines the right operand
         * @param result defines the Vector3 where to store the result
         */
        static CrossToRef(left: Vector3, right: Vector3, result: Vector3): void;
        /**
         * Returns a new Vector3 as the normalization of the passed vector
         * @param vector defines the Vector3 to normalize
         * @returns the new Vector3
         */
        static Normalize(vector: Vector3): Vector3;
        /**
         * Sets the passed vector "result" with the normalization of the passed first vector
         * @param vector defines the Vector3 to normalize
         * @param result defines the Vector3 where to store the result
         */
        static NormalizeToRef(vector: Vector3, result: Vector3): void;
        private static _viewportMatrixCache;
        /**
         * Project a Vector3 onto screen space
         * @param vector defines the Vector3 to project
         * @param world defines the world matrix to use
         * @param transform defines the transform (view x projection) matrix to use
         * @param viewport defines the screen viewport to use
         * @returns the new Vector3
         */
        static Project(vector: Vector3, world: Matrix, transform: Matrix, viewport: Viewport): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param transform defines the transform (view x projection) matrix to use
         * @returns the new Vector3
         */
        static UnprojectFromTransform(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, transform: Matrix): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @returns the new Vector3
         */
        static Unproject(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix): Vector3;
        /**
         * Unproject from screen space to object space
         * @param source defines the screen space Vector3 to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectToRef(source: Vector3, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix, result: Vector3): void;
        /**
         * Unproject from screen space to object space
         * @param sourceX defines the screen space x coordinate to use
         * @param sourceY defines the screen space y coordinate to use
         * @param sourceZ defines the screen space z coordinate to use
         * @param viewportWidth defines the current width of the viewport
         * @param viewportHeight defines the current height of the viewport
         * @param world defines the world matrix to use (can be set to Identity to go to world space)
         * @param view defines the view matrix to use
         * @param projection defines the projection matrix to use
         * @param result defines the Vector3 where to store the result
         */
        static UnprojectFloatsToRef(sourceX: float, sourceY: float, sourceZ: float, viewportWidth: number, viewportHeight: number, world: Matrix, view: Matrix, projection: Matrix, result: Vector3): void;
        /**
         * Gets the minimal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Minimize(left: Vector3, right: Vector3): Vector3;
        /**
         * Gets the maximal coordinate values between two Vector3
         * @param left defines the first operand
         * @param right defines the second operand
         * @returns the new Vector3
         */
        static Maximize(left: Vector3, right: Vector3): Vector3;
        /**
         * Returns the distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the distance
         */
        static Distance(value1: Vector3, value2: Vector3): number;
        /**
         * Returns the squared distance between the vectors "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the squared distance
         */
        static DistanceSquared(value1: Vector3, value2: Vector3): number;
        /**
         * Returns a new Vector3 located at the center between "value1" and "value2"
         * @param value1 defines the first operand
         * @param value2 defines the second operand
         * @returns the new Vector3
         */
        static Center(value1: Vector3, value2: Vector3): Vector3;
        /**
         * Given three orthogonal normalized left-handed oriented Vector3 axis in space (target system),
         * RotationFromAxis() returns the rotation Euler angles (ex : rotation.x, rotation.y, rotation.z) to apply
         * to something in order to rotate it from its local system to the given target system
         * Note: axis1, axis2 and axis3 are normalized during this operation
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @returns a new Vector3
         */
        static RotationFromAxis(axis1: Vector3, axis2: Vector3, axis3: Vector3): Vector3;
        /**
         * The same than RotationFromAxis but updates the passed ref Vector3 parameter instead of returning a new Vector3
         * @param axis1 defines the first axis
         * @param axis2 defines the second axis
         * @param axis3 defines the third axis
         * @param ref defines the Vector3 where to store the result
         */
        static RotationFromAxisToRef(axis1: Vector3, axis2: Vector3, axis3: Vector3, ref: Vector3): void;
    }
    class Vector4 {
        x: number;
        y: number;
        z: number;
        w: number;
        /**
         * Creates a Vector4 object from the passed floats.
         */
        constructor(x: number, y: number, z: number, w: number);
        /**
         * Returns the string with the Vector4 coordinates.
         */
        toString(): string;
        /**
         * Returns the string "Vector4".
         */
        getClassName(): string;
        /**
         * Returns the Vector4 hash code.
         */
        getHashCode(): number;
        /**
         * Returns a new array populated with 4 elements : the Vector4 coordinates.
         */
        asArray(): number[];
        /**
         * Populates the passed array from the passed index with the Vector4 coordinates.
         * Returns the Vector4.
         */
        toArray(array: FloatArray, index?: number): Vector4;
        /**
         * Adds the passed vector to the current Vector4.
         * Returns the updated Vector4.
         */
        addInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 as the result of the addition of the current Vector4 and the passed one.
         */
        add(otherVector: Vector4): Vector4;
        /**
         * Updates the passed vector "result" with the result of the addition of the current Vector4 and the passed one.
         * Returns the current Vector4.
         */
        addToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Subtract in place the passed vector from the current Vector4.
         * Returns the updated Vector4.
         */
        subtractInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 with the result of the subtraction of the passed vector from the current Vector4.
         */
        subtract(otherVector: Vector4): Vector4;
        /**
         * Sets the passed vector "result" with the result of the subtraction of the passed vector from the current Vector4.
         * Returns the current Vector4.
         */
        subtractToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the result of the subtraction of the passed floats from the current Vector4 coordinates.
         */
        subtractFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Sets the passed vector "result" set with the result of the subtraction of the passed floats from the current Vector4 coordinates.
         * Returns the current Vector4.
         */
        subtractFromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 negated coordinates.
         */
        negate(): Vector4;
        /**
         * Multiplies the current Vector4 coordinates by scale (float).
         * Returns the updated Vector4.
         */
        scaleInPlace(scale: number): Vector4;
        /**
         * Returns a new Vector4 set with the current Vector4 coordinates multiplied by scale (float).
         */
        scale(scale: number): Vector4;
        /**
         * Sets the passed vector "result" with the current Vector4 coordinates multiplied by scale (float).
         * Returns the current Vector4.
         */
        scaleToRef(scale: number, result: Vector4): Vector4;
        /**
         * Scale the current Vector4 values by a factor and add the result to a given Vector4
         * @param scale defines the scale factor
         * @param result defines the Vector4 object where to store the result
         * @returns the unmodified current Vector4
         */
        scaleAndAddToRef(scale: number, result: Vector4): Vector4;
        /**
         * Boolean : True if the current Vector4 coordinates are stricly equal to the passed ones.
         */
        equals(otherVector: Vector4): boolean;
        /**
         * Boolean : True if the current Vector4 coordinates are each beneath the distance "epsilon" from the passed vector ones.
         */
        equalsWithEpsilon(otherVector: Vector4, epsilon?: number): boolean;
        /**
         * Boolean : True if the passed floats are strictly equal to the current Vector4 coordinates.
         */
        equalsToFloats(x: number, y: number, z: number, w: number): boolean;
        /**
         * Multiplies in place the current Vector4 by the passed one.
         * Returns the updated Vector4.
         */
        multiplyInPlace(otherVector: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the current Vector4 and the passed one.
         */
        multiply(otherVector: Vector4): Vector4;
        /**
         * Updates the passed vector "result" with the multiplication result of the current Vector4 and the passed one.
         * Returns the current Vector4.
         */
        multiplyToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the multiplication result of the passed floats and the current Vector4 coordinates.
         */
        multiplyByFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Returns a new Vector4 set with the division result of the current Vector4 by the passed one.
         */
        divide(otherVector: Vector4): Vector4;
        /**
         * Updates the passed vector "result" with the division result of the current Vector4 by the passed one.
         * Returns the current Vector4.
         */
        divideToRef(otherVector: Vector4, result: Vector4): Vector4;
        /**
         * Divides the current Vector3 coordinates by the passed ones.
         * @returns the updated Vector3.
         */
        divideInPlace(otherVector: Vector4): Vector4;
        /**
         * Updates the Vector4 coordinates with the minimum values between its own and the passed vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        minimizeInPlace(other: Vector4): Vector4;
        /**
         * Updates the Vector4 coordinates with the maximum values between its own and the passed vector ones
         * @param other defines the second operand
         * @returns the current updated Vector4
         */
        maximizeInPlace(other: Vector4): Vector4;
        /**
         * Returns the Vector4 length (float).
         */
        length(): number;
        /**
         * Returns the Vector4 squared length (float).
         */
        lengthSquared(): number;
        /**
         * Normalizes in place the Vector4.
         * Returns the updated Vector4.
         */
        normalize(): Vector4;
        /**
         * Returns a new Vector3 from the Vector4 (x, y, z) coordinates.
         */
        toVector3(): Vector3;
        /**
         * Returns a new Vector4 copied from the current one.
         */
        clone(): Vector4;
        /**
         * Updates the current Vector4 with the passed one coordinates.
         * Returns the updated Vector4.
         */
        copyFrom(source: Vector4): Vector4;
        /**
         * Updates the current Vector4 coordinates with the passed floats.
         * Returns the updated Vector4.
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Updates the current Vector4 coordinates with the passed floats.
         * Returns the updated Vector4.
         */
        set(x: number, y: number, z: number, w: number): Vector4;
        /**
         * Returns a new Vector4 set from the starting index of the passed array.
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Vector4;
        /**
         * Updates the passed vector "result" from the starting index of the passed array.
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Vector4): void;
        /**
         * Updates the passed vector "result" from the starting index of the passed Float32Array.
         */
        static FromFloatArrayToRef(array: Float32Array, offset: number, result: Vector4): void;
        /**
         * Updates the passed vector "result" coordinates from the passed floats.
         */
        static FromFloatsToRef(x: number, y: number, z: number, w: number, result: Vector4): void;
        /**
         * Returns a new Vector4 set to (0.0, 0.0, 0.0, 0.0)
         */
        static Zero(): Vector4;
        /**
         * Returns a new Vector4 set to (1.0, 1.0, 1.0, 1.0)
         */
        static One(): Vector4;
        /**
         * Returns a new normalized Vector4 from the passed one.
         */
        static Normalize(vector: Vector4): Vector4;
        /**
         * Updates the passed vector "result" from the normalization of the passed one.
         */
        static NormalizeToRef(vector: Vector4, result: Vector4): void;
        static Minimize(left: Vector4, right: Vector4): Vector4;
        static Maximize(left: Vector4, right: Vector4): Vector4;
        /**
         * Returns the distance (float) between the vectors "value1" and "value2".
         */
        static Distance(value1: Vector4, value2: Vector4): number;
        /**
         * Returns the squared distance (float) between the vectors "value1" and "value2".
         */
        static DistanceSquared(value1: Vector4, value2: Vector4): number;
        /**
         * Returns a new Vector4 located at the center between the vectors "value1" and "value2".
         */
        static Center(value1: Vector4, value2: Vector4): Vector4;
        /**
         * Returns a new Vector4 set with the result of the normal transformation by the passed matrix of the passed vector.
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormal(vector: Vector4, transformation: Matrix): Vector4;
        /**
         * Sets the passed vector "result" with the result of the normal transformation by the passed matrix of the passed vector.
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormalToRef(vector: Vector4, transformation: Matrix, result: Vector4): void;
        /**
         * Sets the passed vector "result" with the result of the normal transformation by the passed matrix of the passed floats (x, y, z, w).
         * This methods computes transformed normalized direction vectors only.
         */
        static TransformNormalFromFloatsToRef(x: number, y: number, z: number, w: number, transformation: Matrix, result: Vector4): void;
    }
    interface ISize {
        width: number;
        height: number;
    }
    class Size implements ISize {
        width: number;
        height: number;
        /**
         * Creates a Size object from the passed width and height (floats).
         */
        constructor(width: number, height: number);
        toString(): string;
        /**
         * Returns the string "Size"
         */
        getClassName(): string;
        /**
         * Returns the Size hash code.
         */
        getHashCode(): number;
        /**
         * Updates the current size from the passed one.
         * Returns the updated Size.
         */
        copyFrom(src: Size): void;
        /**
         * Updates in place the current Size from the passed floats.
         * Returns the updated Size.
         */
        copyFromFloats(width: number, height: number): Size;
        /**
         * Updates in place the current Size from the passed floats.
         * Returns the updated Size.
         */
        set(width: number, height: number): Size;
        /**
         * Returns a new Size set with the multiplication result of the current Size and the passed floats.
         */
        multiplyByFloats(w: number, h: number): Size;
        /**
         * Returns a new Size copied from the passed one.
         */
        clone(): Size;
        /**
         * Boolean : True if the current Size and the passed one width and height are strictly equal.
         */
        equals(other: Size): boolean;
        /**
         * Returns the surface of the Size : width * height (float).
         */
        readonly surface: number;
        /**
         * Returns a new Size set to (0.0, 0.0)
         */
        static Zero(): Size;
        /**
         * Returns a new Size set as the addition result of the current Size and the passed one.
         */
        add(otherSize: Size): Size;
        /**
         * Returns a new Size set as the subtraction result of  the passed one from the current Size.
         */
        subtract(otherSize: Size): Size;
        /**
         * Returns a new Size set at the linear interpolation "amount" between "start" and "end".
         */
        static Lerp(start: Size, end: Size, amount: number): Size;
    }
    class Quaternion {
        x: number;
        y: number;
        z: number;
        w: number;
        /**
         * Creates a new Quaternion from the passed floats.
         */
        constructor(x?: number, y?: number, z?: number, w?: number);
        /**
         * Returns a string with the Quaternion coordinates.
         */
        toString(): string;
        /**
         * Returns the string "Quaternion".
         */
        getClassName(): string;
        /**
         * Returns the Quaternion hash code.
         */
        getHashCode(): number;
        /**
         * Returns a new array populated with 4 elements : the Quaternion coordinates.
         */
        asArray(): number[];
        /**
         * Boolean : True if the current Quaterion and the passed one coordinates are strictly equal.
         */
        equals(otherQuaternion: Quaternion): boolean;
        /**
         * Returns a new Quaternion copied from the current one.
         */
        clone(): Quaternion;
        /**
         * Updates the current Quaternion from the passed one coordinates.
         * Returns the updated Quaterion.
         */
        copyFrom(other: Quaternion): Quaternion;
        /**
         * Updates the current Quaternion from the passed float coordinates.
         * Returns the updated Quaterion.
         */
        copyFromFloats(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Updates the current Quaternion from the passed float coordinates.
         * Returns the updated Quaterion.
         */
        set(x: number, y: number, z: number, w: number): Quaternion;
        /**
         * Returns a new Quaternion as the addition result of the passed one and the current Quaternion.
         */
        add(other: Quaternion): Quaternion;
        /**
         * Returns a new Quaternion as the subtraction result of the passed one from the current Quaternion.
         */
        subtract(other: Quaternion): Quaternion;
        /**
         * Returns a new Quaternion set by multiplying the current Quaterion coordinates by the float "scale".
         */
        scale(value: number): Quaternion;
        /**
         * Scale the current Quaternion values by a factor to a given Quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current Quaternion
         */
        scaleToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Scale the current Quaternion values by a factor and add the result to a given Quaternion
         * @param scale defines the scale factor
         * @param result defines the Quaternion object where to store the result
         * @returns the unmodified current Quaternion
         */
        scaleAndAddToRef(scale: number, result: Quaternion): Quaternion;
        /**
         * Returns a new Quaternion set as the quaternion mulplication result of the current one with the passed one "q1".
         */
        multiply(q1: Quaternion): Quaternion;
        /**
         * Sets the passed "result" as the quaternion mulplication result of the current one with the passed one "q1".
         * Returns the current Quaternion.
         */
        multiplyToRef(q1: Quaternion, result: Quaternion): Quaternion;
        /**
         * Updates the current Quaternion with the quaternion mulplication result of itself with the passed one "q1".
         * Returns the updated Quaternion.
         */
        multiplyInPlace(q1: Quaternion): Quaternion;
        /**
         * Sets the passed "ref" with the conjugation of the current Quaternion.
         * Returns the current Quaternion.
         */
        conjugateToRef(ref: Quaternion): Quaternion;
        /**
         * Conjugates in place the current Quaternion.
         * Returns the updated Quaternion.
         */
        conjugateInPlace(): Quaternion;
        /**
         * Returns a new Quaternion as the conjugate of the current Quaternion.
         */
        conjugate(): Quaternion;
        /**
         * Returns the Quaternion length (float).
         */
        length(): number;
        /**
         * Normalize in place the current Quaternion.
         * Returns the updated Quaternion.
         */
        normalize(): Quaternion;
        /**
         * Returns a new Vector3 set with the Euler angles translated from the current Quaternion
         * @param order is a reserved parameter and is ignore for now
         * @returns the new Vector3
         */
        toEulerAngles(order?: string): Vector3;
        /**
         * Sets the passed vector3 "result" with the Euler angles translated from the current Quaternion
         * @param result defines the vector which will be filled with the Euler angles
         * @param order is a reserved parameter and is ignore for now
         * @returns the current Quaternion
         */
        toEulerAnglesToRef(result: Vector3, order?: string): Quaternion;
        /**
         * Updates the passed rotation matrix with the current Quaternion values.
         * Returns the current Quaternion.
         */
        toRotationMatrix(result: Matrix): Quaternion;
        /**
         * Updates the current Quaternion from the passed rotation matrix values.
         * Returns the updated Quaternion.
         */
        fromRotationMatrix(matrix: Matrix): Quaternion;
        /**
         * Returns a new Quaternion set from the passed rotation matrix values.
         */
        static FromRotationMatrix(matrix: Matrix): Quaternion;
        /**
         * Updates the passed quaternion "result" with the passed rotation matrix values.
         */
        static FromRotationMatrixToRef(matrix: Matrix, result: Quaternion): void;
        /**
         * Returns a new Quaternion set to (0.0, 0.0, 0.0).
         */
        static Zero(): Quaternion;
        /**
         * Returns a new Quaternion as the inverted current Quaternion.
         */
        static Inverse(q: Quaternion): Quaternion;
        /**
         * Returns the identity Quaternion.
         */
        static Identity(): Quaternion;
        static IsIdentity(quaternion: Quaternion): boolean;
        /**
         * Returns a new Quaternion set from the passed axis (Vector3) and angle in radians (float).
         */
        static RotationAxis(axis: Vector3, angle: number): Quaternion;
        /**
         * Sets the passed quaternion "result" from the passed axis (Vector3) and angle in radians (float).
         */
        static RotationAxisToRef(axis: Vector3, angle: number, result: Quaternion): Quaternion;
        /**
         * Retuns a new Quaternion set from the starting index of the passed array.
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Quaternion;
        /**
         * Returns a new Quaternion set from the passed Euler float angles (y, x, z).
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Quaternion;
        /**
         * Sets the passed quaternion "result" from the passed float Euler angles (y, x, z).
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Quaternion): void;
        /**
         * Returns a new Quaternion from the passed float Euler angles expressed in z-x-z orientation
         */
        static RotationAlphaBetaGamma(alpha: number, beta: number, gamma: number): Quaternion;
        /**
         * Sets the passed quaternion "result" from the passed float Euler angles expressed in z-x-z orientation
         */
        static RotationAlphaBetaGammaToRef(alpha: number, beta: number, gamma: number, result: Quaternion): void;
        /**
         * Returns a new Quaternion as the quaternion rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system.
         * cf to Vector3.RotationFromAxis() documentation.
         * Note : axis1, axis2 and axis3 are normalized during this operation.
         */
        static RotationQuaternionFromAxis(axis1: Vector3, axis2: Vector3, axis3: Vector3, ref: Quaternion): Quaternion;
        /**
         * Sets the passed quaternion "ref" with the quaternion rotation value to reach the target (axis1, axis2, axis3) orientation as a rotated XYZ system.
         * cf to Vector3.RotationFromAxis() documentation.
         * Note : axis1, axis2 and axis3 are normalized during this operation.
         */
        static RotationQuaternionFromAxisToRef(axis1: Vector3, axis2: Vector3, axis3: Vector3, ref: Quaternion): void;
        static Slerp(left: Quaternion, right: Quaternion, amount: number): Quaternion;
        static SlerpToRef(left: Quaternion, right: Quaternion, amount: number, result: Quaternion): void;
        /**
         * Returns a new Quaternion located for "amount" (float) on the Hermite interpolation spline defined by the vectors "value1", "tangent1", "value2", "tangent2".
         */
        static Hermite(value1: Quaternion, tangent1: Quaternion, value2: Quaternion, tangent2: Quaternion, amount: number): Quaternion;
    }
    class Matrix {
        private static _tempQuaternion;
        private static _xAxis;
        private static _yAxis;
        private static _zAxis;
        private static _updateFlagSeed;
        private static _identityReadOnly;
        private _isIdentity;
        private _isIdentityDirty;
        updateFlag: number;
        m: Float32Array;
        _markAsUpdated(): void;
        constructor();
        /**
         * Boolean : True is the matrix is the identity matrix
         */
        isIdentity(considerAsTextureMatrix?: boolean): boolean;
        /**
         * Returns the matrix determinant (float).
         */
        determinant(): number;
        /**
         * Returns the matrix underlying array.
         */
        toArray(): Float32Array;
        /**
        * Returns the matrix underlying array.
        */
        asArray(): Float32Array;
        /**
         * Inverts in place the Matrix.
         * Returns the Matrix inverted.
         */
        invert(): Matrix;
        /**
         * Sets all the matrix elements to zero.
         * Returns the Matrix.
         */
        reset(): Matrix;
        /**
         * Returns a new Matrix as the addition result of the current Matrix and the passed one.
         */
        add(other: Matrix): Matrix;
        /**
         * Sets the passed matrix "result" with the ddition result of the current Matrix and the passed one.
         * Returns the Matrix.
         */
        addToRef(other: Matrix, result: Matrix): Matrix;
        /**
         * Adds in place the passed matrix to the current Matrix.
         * Returns the updated Matrix.
         */
        addToSelf(other: Matrix): Matrix;
        /**
         * Sets the passed matrix with the current inverted Matrix.
         * Returns the unmodified current Matrix.
         */
        invertToRef(other: Matrix): Matrix;
        /**
         * Inserts the translation vector (using 3 x floats) in the current Matrix.
         * Returns the updated Matrix.
         */
        setTranslationFromFloats(x: number, y: number, z: number): Matrix;
        /**
 * Inserts the translation vector in the current Matrix.
 * Returns the updated Matrix.
 */
        setTranslation(vector3: Vector3): Matrix;
        /**
         * Returns a new Vector3 as the extracted translation from the Matrix.
         */
        getTranslation(): Vector3;
        /**
         * Fill a Vector3 with the extracted translation from the Matrix.
         */
        getTranslationToRef(result: Vector3): Matrix;
        /**
         * Remove rotation and scaling part from the Matrix.
         * Returns the updated Matrix.
         */
        removeRotationAndScaling(): Matrix;
        /**
         * Returns a new Matrix set with the multiplication result of the current Matrix and the passed one.
         */
        multiply(other: Matrix): Matrix;
        /**
         * Updates the current Matrix from the passed one values.
         * Returns the updated Matrix.
         */
        copyFrom(other: Matrix): Matrix;
        /**
         * Populates the passed array from the starting index with the Matrix values.
         * Returns the Matrix.
         */
        copyToArray(array: Float32Array, offset?: number): Matrix;
        /**
         * Sets the passed matrix "result" with the multiplication result of the current Matrix and the passed one.
         */
        multiplyToRef(other: Matrix, result: Matrix): Matrix;
        /**
         * Sets the Float32Array "result" from the passed index "offset" with the multiplication result of the current Matrix and the passed one.
         */
        multiplyToArray(other: Matrix, result: Float32Array, offset: number): Matrix;
        /**
         * Boolean : True is the current Matrix and the passed one values are strictly equal.
         */
        equals(value: Matrix): boolean;
        /**
         * Returns a new Matrix from the current Matrix.
         */
        clone(): Matrix;
        /**
         * Returns the string "Matrix"
         */
        getClassName(): string;
        /**
         * Returns the Matrix hash code.
         */
        getHashCode(): number;
        /**
         * Decomposes the current Matrix into :
         * - a scale vector3 passed as a reference to update,
         * - a rotation quaternion passed as a reference to update,
         * - a translation vector3 passed as a reference to update.
         * Returns the true if operation was successful.
         */
        decompose(scale: Vector3, rotation: Quaternion, translation: Vector3): boolean;
        /**
         * Returns the index-th row of the current matrix as a new Vector4.
         */
        getRow(index: number): Nullable<Vector4>;
        /**
         * Sets the index-th row of the current matrix with the passed Vector4 values.
         * Returns the updated Matrix.
         */
        setRow(index: number, row: Vector4): Matrix;
        /**
         * Compute the transpose of the matrix.
         * Returns a new Matrix.
         */
        transpose(): Matrix;
        /**
         * Compute the transpose of the matrix.
         * Returns the current matrix.
         */
        transposeToRef(result: Matrix): Matrix;
        /**
         * Sets the index-th row of the current matrix with the passed 4 x float values.
         * Returns the updated Matrix.
         */
        setRowFromFloats(index: number, x: number, y: number, z: number, w: number): Matrix;
        /**
         * Compute a new Matrix set with the current Matrix values multiplied by scale (float)
         * @param scale defines the scale factor
         * @returns a new Matrix
         */
        scale(scale: number): Matrix;
        /**
         * Scale the current Matrix values by a factor to a given result Matrix
         * @param scale defines the scale factor
         * @param result defines the Matrix to store the result
         * @returns the current Matrix
         */
        scaleToRef(scale: number, result: Matrix): Matrix;
        /**
         * Scale the current Matrix values by a factor and add the result to a given Matrix
         * @param scale defines the scale factor
         * @param result defines the Matrix to store the result
         * @returns the current Matrix
         */
        scaleAndAddToRef(scale: number, result: Matrix): Matrix;
        /**
         * Writes to the given matrix a normal matrix, computed from this one (using values from identity matrix for fourth row and column).
         * @param ref matrix to store the result
         */
        toNormalMatrix(ref: Matrix): void;
        /**
         * Returns a new Matrix as the extracted rotation matrix from the current one.
         */
        getRotationMatrix(): Matrix;
        /**
         * Extracts the rotation matrix from the current one and sets it as the passed "result".
         * Returns the current Matrix.
         */
        getRotationMatrixToRef(result: Matrix): Matrix;
        /**
         * Returns a new Matrix set from the starting index of the passed array.
         */
        static FromArray(array: ArrayLike<number>, offset?: number): Matrix;
        /**
         * Sets the passed "result" matrix from the starting index of the passed array.
         */
        static FromArrayToRef(array: ArrayLike<number>, offset: number, result: Matrix): void;
        /**
         * Sets the passed "result" matrix from the starting index of the passed Float32Array by multiplying each element by the float "scale".
         */
        static FromFloat32ArrayToRefScaled(array: Float32Array, offset: number, scale: number, result: Matrix): void;
        /**
         * Sets the passed matrix "result" with the 16 passed floats.
         */
        static FromValuesToRef(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number, result: Matrix): void;
        /**
         * Static identity matrix to be used as readonly matrix
         * Must not be updated.
         */
        static readonly IdentityReadOnly: Matrix;
        /**
         * Returns a new Matrix set from the 16 passed floats.
         */
        static FromValues(initialM11: number, initialM12: number, initialM13: number, initialM14: number, initialM21: number, initialM22: number, initialM23: number, initialM24: number, initialM31: number, initialM32: number, initialM33: number, initialM34: number, initialM41: number, initialM42: number, initialM43: number, initialM44: number): Matrix;
        /**
         * Returns a new Matrix composed by the passed scale (vector3), rotation (quaternion) and translation (vector3).
         */
        static Compose(scale: Vector3, rotation: Quaternion, translation: Vector3): Matrix;
        /**
       * Update a Matrix with values composed by the passed scale (vector3), rotation (quaternion) and translation (vector3).
       */
        static ComposeToRef(scale: Vector3, rotation: Quaternion, translation: Vector3, result: Matrix): void;
        /**
         * Returns a new indentity Matrix.
         */
        static Identity(): Matrix;
        /**
         * Sets the passed "result" as an identity matrix.
         */
        static IdentityToRef(result: Matrix): void;
        /**
         * Returns a new zero Matrix.
         */
        static Zero(): Matrix;
        /**
         * Returns a new rotation matrix for "angle" radians around the X axis.
         */
        static RotationX(angle: number): Matrix;
        /**
         * Returns a new Matrix as the passed inverted one.
         */
        static Invert(source: Matrix): Matrix;
        /**
         * Sets the passed matrix "result" as a rotation matrix for "angle" radians around the X axis.
         */
        static RotationXToRef(angle: number, result: Matrix): void;
        /**
         * Returns a new rotation matrix for "angle" radians around the Y axis.
         */
        static RotationY(angle: number): Matrix;
        /**
         * Sets the passed matrix "result" as a rotation matrix for "angle" radians around the Y axis.
         */
        static RotationYToRef(angle: number, result: Matrix): void;
        /**
         * Returns a new rotation matrix for "angle" radians around the Z axis.
         */
        static RotationZ(angle: number): Matrix;
        /**
         * Sets the passed matrix "result" as a rotation matrix for "angle" radians around the Z axis.
         */
        static RotationZToRef(angle: number, result: Matrix): void;
        /**
         * Returns a new rotation matrix for "angle" radians around the passed axis.
         */
        static RotationAxis(axis: Vector3, angle: number): Matrix;
        /**
         * Sets the passed matrix "result" as a rotation matrix for "angle" radians around the passed axis.
         */
        static RotationAxisToRef(axis: Vector3, angle: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a rotation matrix from the Euler angles (y, x, z).
         */
        static RotationYawPitchRoll(yaw: number, pitch: number, roll: number): Matrix;
        /**
         * Sets the passed matrix "result" as a rotation matrix from the Euler angles (y, x, z).
         */
        static RotationYawPitchRollToRef(yaw: number, pitch: number, roll: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a scaling matrix from the passed floats (x, y, z).
         */
        static Scaling(x: number, y: number, z: number): Matrix;
        /**
         * Sets the passed matrix "result" as a scaling matrix from the passed floats (x, y, z).
         */
        static ScalingToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a translation matrix from the passed floats (x, y, z).
         */
        static Translation(x: number, y: number, z: number): Matrix;
        /**
         * Sets the passed matrix "result" as a translation matrix from the passed floats (x, y, z).
         */
        static TranslationToRef(x: number, y: number, z: number, result: Matrix): void;
        /**
         * Returns a new Matrix whose values are the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @returns the new matrix
         */
        static Lerp(startValue: Matrix, endValue: Matrix, gradient: number): Matrix;
        /**
         * Set the passed matrix "result" as the interpolated values for "gradient" (float) between the ones of the matrices "startValue" and "endValue".
         * @param startValue defines the start value
         * @param endValue defines the end value
         * @param gradient defines the gradient factor
         * @param result defines the Matrix object where to store data
         */
        static LerpToRef(startValue: Matrix, endValue: Matrix, gradient: number, result: Matrix): void;
        /**
         * Returns a new Matrix whose values are computed by :
         * - decomposing the the "startValue" and "endValue" matrices into their respective scale, rotation and translation matrices,
         * - interpolating for "gradient" (float) the values between each of these decomposed matrices between the start and the end,
         * - recomposing a new matrix from these 3 interpolated scale, rotation and translation matrices.
         */
        static DecomposeLerp(startValue: Matrix, endValue: Matrix, gradient: number): Matrix;
        /**
         * Returns a new rotation Matrix used to rotate a mesh so as it looks at the target Vector3, from the eye Vector3, the UP vector3 being orientated like "up".
         * This methods works for a Left-Handed system.
         */
        static LookAtLH(eye: Vector3, target: Vector3, up: Vector3): Matrix;
        /**
         * Sets the passed "result" Matrix as a rotation matrix used to rotate a mesh so as it looks at the target Vector3, from the eye Vector3, the UP vector3 being orientated like "up".
         * This methods works for a Left-Handed system.
         */
        static LookAtLHToRef(eye: Vector3, target: Vector3, up: Vector3, result: Matrix): void;
        /**
         * Returns a new rotation Matrix used to rotate a mesh so as it looks at the target Vector3, from the eye Vector3, the UP vector3 being orientated like "up".
         * This methods works for a Right-Handed system.
         */
        static LookAtRH(eye: Vector3, target: Vector3, up: Vector3): Matrix;
        /**
         * Sets the passed "result" Matrix as a rotation matrix used to rotate a mesh so as it looks at the target Vector3, from the eye Vector3, the UP vector3 being orientated like "up".
         * This methods works for a Left-Handed system.
         */
        static LookAtRHToRef(eye: Vector3, target: Vector3, up: Vector3, result: Matrix): void;
        /**
         * Returns a new Matrix as a left-handed orthographic projection matrix computed from the passed floats : width and height of the projection plane, z near and far limits.
         */
        static OrthoLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Sets the passed matrix "result" as a left-handed orthographic projection matrix computed from the passed floats : width and height of the projection plane, z near and far limits.
         */
        static OrthoLHToRef(width: number, height: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a left-handed orthographic projection matrix computed from the passed floats : left, right, top and bottom being the coordinates of the projection plane, z near and far limits.
         */
        static OrthoOffCenterLH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Sets the passed matrix "result" as a left-handed orthographic projection matrix computed from the passed floats : left, right, top and bottom being the coordinates of the projection plane, z near and far limits.
         */
        static OrthoOffCenterLHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a right-handed orthographic projection matrix computed from the passed floats : left, right, top and bottom being the coordinates of the projection plane, z near and far limits.
         */
        static OrthoOffCenterRH(left: number, right: number, bottom: number, top: number, znear: number, zfar: number): Matrix;
        /**
         * Sets the passed matrix "result" as a right-handed orthographic projection matrix computed from the passed floats : left, right, top and bottom being the coordinates of the projection plane, z near and far limits.
         */
        static OrthoOffCenterRHToRef(left: number, right: number, bottom: number, top: number, znear: number, zfar: number, result: Matrix): void;
        /**
         * Returns a new Matrix as a left-handed perspective projection matrix computed from the passed floats : width and height of the projection plane, z near and far limits.
         */
        static PerspectiveLH(width: number, height: number, znear: number, zfar: number): Matrix;
        /**
         * Returns a new Matrix as a left-handed perspective projection matrix computed from the passed floats : vertical angle of view (fov), width/height ratio (aspect), z near and far limits.
         */
        static PerspectiveFovLH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Sets the passed matrix "result" as a left-handed perspective projection matrix computed from the passed floats : vertical angle of view (fov), width/height ratio (aspect), z near and far limits.
         */
        static PerspectiveFovLHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Returns a new Matrix as a right-handed perspective projection matrix computed from the passed floats : vertical angle of view (fov), width/height ratio (aspect), z near and far limits.
         */
        static PerspectiveFovRH(fov: number, aspect: number, znear: number, zfar: number): Matrix;
        /**
         * Sets the passed matrix "result" as a right-handed perspective projection matrix computed from the passed floats : vertical angle of view (fov), width/height ratio (aspect), z near and far limits.
         */
        static PerspectiveFovRHToRef(fov: number, aspect: number, znear: number, zfar: number, result: Matrix, isVerticalFovFixed?: boolean): void;
        /**
         * Sets the passed matrix "result" as a left-handed perspective projection matrix  for WebVR computed from the passed floats : vertical angle of view (fov), width/height ratio (aspect), z near and far limits.
         */
        static PerspectiveFovWebVRToRef(fov: {
            upDegrees: number;
            downDegrees: number;
            leftDegrees: number;
            rightDegrees: number;
        }, znear: number, zfar: number, result: Matrix, rightHanded?: boolean): void;
        /**
         * Returns the final transformation matrix : world * view * projection * viewport
         */
        static GetFinalMatrix(viewport: Viewport, world: Matrix, view: Matrix, projection: Matrix, zmin: number, zmax: number): Matrix;
        /**
         * Returns a new Float32Array array with 4 elements : the 2x2 matrix extracted from the passed Matrix.
         */
        static GetAsMatrix2x2(matrix: Matrix): Float32Array;
        /**
         * Returns a new Float32Array array with 9 elements : the 3x3 matrix extracted from the passed Matrix.
         */
        static GetAsMatrix3x3(matrix: Matrix): Float32Array;
        /**
         * Compute the transpose of the passed Matrix.
         * Returns a new Matrix.
         */
        static Transpose(matrix: Matrix): Matrix;
        /**
         * Compute the transpose of the passed Matrix and store it in the result matrix.
         */
        static TransposeToRef(matrix: Matrix, result: Matrix): void;
        /**
         * Returns a new Matrix as the reflection  matrix across the passed plane.
         */
        static Reflection(plane: Plane): Matrix;
        /**
         * Sets the passed matrix "result" as the reflection matrix across the passed plane.
         */
        static ReflectionToRef(plane: Plane, result: Matrix): void;
        /**
         * Sets the passed matrix "mat" as a rotation matrix composed from the 3 passed  left handed axis.
         */
        static FromXYZAxesToRef(xaxis: Vector3, yaxis: Vector3, zaxis: Vector3, result: Matrix): void;
        /**
         * Sets the passed matrix "result" as a rotation matrix according to the passed quaternion.
         */
        static FromQuaternionToRef(quat: Quaternion, result: Matrix): void;
    }
    class Plane {
        normal: Vector3;
        d: number;
        /**
         * Creates a Plane object according to the passed floats a, b, c, d and the plane equation : ax + by + cz + d = 0
         */
        constructor(a: number, b: number, c: number, d: number);
        /**
         * Returns the plane coordinates as a new array of 4 elements [a, b, c, d].
         */
        asArray(): number[];
        /**
         * Returns a new plane copied from the current Plane.
         */
        clone(): Plane;
        /**
         * Returns the string "Plane".
         */
        getClassName(): string;
        /**
         * Returns the Plane hash code.
         */
        getHashCode(): number;
        /**
         * Normalize the current Plane in place.
         * Returns the updated Plane.
         */
        normalize(): Plane;
        /**
         * Returns a new Plane as the result of the transformation of the current Plane by the passed matrix.
         */
        transform(transformation: Matrix): Plane;
        /**
         * Returns the dot product (float) of the point coordinates and the plane normal.
         */
        dotCoordinate(point: Vector3): number;
        /**
         * Updates the current Plane from the plane defined by the three passed points.
         * Returns the updated Plane.
         */
        copyFromPoints(point1: Vector3, point2: Vector3, point3: Vector3): Plane;
        /**
         * Boolean : True is the vector "direction"  is the same side than the plane normal.
         */
        isFrontFacingTo(direction: Vector3, epsilon: number): boolean;
        /**
         * Returns the signed distance (float) from the passed point to the Plane.
         */
        signedDistanceTo(point: Vector3): number;
        /**
         * Returns a new Plane from the passed array.
         */
        static FromArray(array: ArrayLike<number>): Plane;
        /**
         * Returns a new Plane defined by the three passed points.
         */
        static FromPoints(point1: Vector3, point2: Vector3, point3: Vector3): Plane;
        /**
         * Returns a new Plane the normal vector to this plane at the passed origin point.
         * Note : the vector "normal" is updated because normalized.
         */
        static FromPositionAndNormal(origin: Vector3, normal: Vector3): Plane;
        /**
         * Returns the signed distance between the plane defined by the normal vector at the "origin"" point and the passed other point.
         */
        static SignedDistanceToPlaneFromPositionAndNormal(origin: Vector3, normal: Vector3, point: Vector3): number;
    }
    class Viewport {
        x: number;
        y: number;
        width: number;
        height: number;
        /**
         * Creates a Viewport object located at (x, y) and sized (width, height).
         */
        constructor(x: number, y: number, width: number, height: number);
        toGlobal(renderWidthOrEngine: number | Engine, renderHeight: number): Viewport;
        /**
         * Returns a new Viewport copied from the current one.
         */
        clone(): Viewport;
    }
    class Frustum {
        /**
         * Returns a new array of 6 Frustum planes computed by the passed transformation matrix.
         */
        static GetPlanes(transform: Matrix): Plane[];
        static GetNearPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetFarPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetLeftPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetRightPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetTopPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        static GetBottomPlaneToRef(transform: Matrix, frustumPlane: Plane): void;
        /**
         * Sets the passed array "frustumPlanes" with the 6 Frustum planes computed by the passed transformation matrix.
         */
        static GetPlanesToRef(transform: Matrix, frustumPlanes: Plane[]): void;
    }
    enum Space {
        LOCAL = 0,
        WORLD = 1,
        BONE = 2,
    }
    class Axis {
        static X: Vector3;
        static Y: Vector3;
        static Z: Vector3;
    }
    class BezierCurve {
        /**
         * Returns the cubic Bezier interpolated value (float) at "t" (float) from the passed x1, y1, x2, y2 floats.
         */
        static interpolate(t: number, x1: number, y1: number, x2: number, y2: number): number;
    }
    enum Orientation {
        CW = 0,
        CCW = 1,
    }
    class Angle {
        private _radians;
        /**
         * Creates an Angle object of "radians" radians (float).
         */
        constructor(radians: number);
        /**
         * Returns the Angle value in degrees (float).
         */
        degrees: () => number;
        /**
         * Returns the Angle value in radians (float).
         */
        radians: () => number;
        /**
         * Returns a new Angle object valued with the angle value in radians between the two passed vectors.
         */
        static BetweenTwoPoints(a: Vector2, b: Vector2): Angle;
        /**
         * Returns a new Angle object from the passed float in radians.
         */
        static FromRadians(radians: number): Angle;
        /**
         * Returns a new Angle object from the passed float in degrees.
         */
        static FromDegrees(degrees: number): Angle;
    }
    class Arc2 {
        startPoint: Vector2;
        midPoint: Vector2;
        endPoint: Vector2;
        centerPoint: Vector2;
        radius: number;
        angle: Angle;
        startAngle: Angle;
        orientation: Orientation;
        /**
         * Creates an Arc object from the three passed points : start, middle and end.
         */
        constructor(startPoint: Vector2, midPoint: Vector2, endPoint: Vector2);
    }
    class Path2 {
        private _points;
        private _length;
        closed: boolean;
        /**
         * Creates a Path2 object from the starting 2D coordinates x and y.
         */
        constructor(x: number, y: number);
        /**
         * Adds a new segment until the passed coordinates (x, y) to the current Path2.
         * Returns the updated Path2.
         */
        addLineTo(x: number, y: number): Path2;
        /**
         * Adds _numberOfSegments_ segments according to the arc definition (middle point coordinates, end point coordinates, the arc start point being the current Path2 last point) to the current Path2.
         * Returns the updated Path2.
         */
        addArcTo(midX: number, midY: number, endX: number, endY: number, numberOfSegments?: number): Path2;
        /**
         * Closes the Path2.
         * Returns the Path2.
         */
        close(): Path2;
        /**
         * Returns the Path2 total length (float).
         */
        length(): number;
        /**
         * Returns the Path2 internal array of points.
         */
        getPoints(): Vector2[];
        /**
         * Returns a new Vector2 located at a percentage of the Path2 total length on this path.
         */
        getPointAtLengthPosition(normalizedLengthPosition: number): Vector2;
        /**
         * Returns a new Path2 starting at the coordinates (x, y).
         */
        static StartingAt(x: number, y: number): Path2;
    }
    class Path3D {
        path: Vector3[];
        private _curve;
        private _distances;
        private _tangents;
        private _normals;
        private _binormals;
        private _raw;
        /**
        * new Path3D(path, normal, raw)
        * Creates a Path3D. A Path3D is a logical math object, so not a mesh.
        * please read the description in the tutorial :  http://doc.babylonjs.com/tutorials/How_to_use_Path3D
        * path : an array of Vector3, the curve axis of the Path3D
        * normal (optional) : Vector3, the first wanted normal to the curve. Ex (0, 1, 0) for a vertical normal.
        * raw (optional, default false) : boolean, if true the returned Path3D isn't normalized. Useful to depict path acceleration or speed.
        */
        constructor(path: Vector3[], firstNormal?: Nullable<Vector3>, raw?: boolean);
        /**
         * Returns the Path3D array of successive Vector3 designing its curve.
         */
        getCurve(): Vector3[];
        /**
         * Returns an array populated with tangent vectors on each Path3D curve point.
         */
        getTangents(): Vector3[];
        /**
         * Returns an array populated with normal vectors on each Path3D curve point.
         */
        getNormals(): Vector3[];
        /**
         * Returns an array populated with binormal vectors on each Path3D curve point.
         */
        getBinormals(): Vector3[];
        /**
         * Returns an array populated with distances (float) of the i-th point from the first curve point.
         */
        getDistances(): number[];
        /**
         * Forces the Path3D tangent, normal, binormal and distance recomputation.
         * Returns the same object updated.
         */
        update(path: Vector3[], firstNormal?: Nullable<Vector3>): Path3D;
        private _compute(firstNormal);
        private _getFirstNonNullVector(index);
        private _getLastNonNullVector(index);
        private _normalVector(v0, vt, va);
    }
    class Curve3 {
        private _points;
        private _length;
        /**
         * Returns a Curve3 object along a Quadratic Bezier curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#quadratic-bezier-curve
         * @param v0 (Vector3) the origin point of the Quadratic Bezier
         * @param v1 (Vector3) the control point
         * @param v2 (Vector3) the end point of the Quadratic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateQuadraticBezier(v0: Vector3, v1: Vector3, v2: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Cubic Bezier curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#cubic-bezier-curve
         * @param v0 (Vector3) the origin point of the Cubic Bezier
         * @param v1 (Vector3) the first control point
         * @param v2 (Vector3) the second control point
         * @param v3 (Vector3) the end point of the Cubic Bezier
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateCubicBezier(v0: Vector3, v1: Vector3, v2: Vector3, v3: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a Hermite Spline curve : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#hermite-spline
         * @param p1 (Vector3) the origin point of the Hermite Spline
         * @param t1 (Vector3) the tangent vector at the origin point
         * @param p2 (Vector3) the end point of the Hermite Spline
         * @param t2 (Vector3) the tangent vector at the end point
         * @param nbPoints (integer) the wanted number of points in the curve
         */
        static CreateHermiteSpline(p1: Vector3, t1: Vector3, p2: Vector3, t2: Vector3, nbPoints: number): Curve3;
        /**
         * Returns a Curve3 object along a CatmullRom Spline curve :
         * @param points (array of Vector3) the points the spline must pass through. At least, four points required.
         * @param nbPoints (integer) the wanted number of points between each curve control points.
         */
        static CreateCatmullRomSpline(points: Vector3[], nbPoints: number): Curve3;
        /**
         * A Curve3 object is a logical object, so not a mesh, to handle curves in the 3D geometric space.
         * A Curve3 is designed from a series of successive Vector3.
         * Tuto : http://doc.babylonjs.com/tutorials/How_to_use_Curve3#curve3-object
         */
        constructor(points: Vector3[]);
        /**
         * Returns the Curve3 stored array of successive Vector3
         */
        getPoints(): Vector3[];
        /**
         * Returns the computed length (float) of the curve.
         */
        length(): number;
        /**
         * Returns a new instance of Curve3 object : var curve = curveA.continue(curveB);
         * This new Curve3 is built by translating and sticking the curveB at the end of the curveA.
         * curveA and curveB keep unchanged.
         */
        continue(curve: Curve3): Curve3;
        private _computeLength(path);
    }
    class PositionNormalVertex {
        position: Vector3;
        normal: Vector3;
        constructor(position?: Vector3, normal?: Vector3);
        clone(): PositionNormalVertex;
    }
    class PositionNormalTextureVertex {
        position: Vector3;
        normal: Vector3;
        uv: Vector2;
        constructor(position?: Vector3, normal?: Vector3, uv?: Vector2);
        clone(): PositionNormalTextureVertex;
    }
    class Tmp {
        static Color3: Color3[];
        static Vector2: Vector2[];
        static Vector3: Vector3[];
        static Vector4: Vector4[];
        static Quaternion: Quaternion[];
        static Matrix: Matrix[];
    }
}

declare module BABYLON {
    class SphericalPolynomial {
        x: Vector3;
        y: Vector3;
        z: Vector3;
        xx: Vector3;
        yy: Vector3;
        zz: Vector3;
        xy: Vector3;
        yz: Vector3;
        zx: Vector3;
        addAmbient(color: Color3): void;
        static getSphericalPolynomialFromHarmonics(harmonics: SphericalHarmonics): SphericalPolynomial;
        scale(scale: number): void;
    }
    class SphericalHarmonics {
        L00: Vector3;
        L1_1: Vector3;
        L10: Vector3;
        L11: Vector3;
        L2_2: Vector3;
        L2_1: Vector3;
        L20: Vector3;
        L21: Vector3;
        L22: Vector3;
        addLight(direction: Vector3, color: Color3, deltaSolidAngle: number): void;
        scale(scale: number): void;
        convertIncidentRadianceToIrradiance(): void;
        convertIrradianceToLambertianRadiance(): void;
        static getsphericalHarmonicsFromPolynomial(polynomial: SphericalPolynomial): SphericalHarmonics;
    }
}

declare module BABYLON {
    /**
     * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
     * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
     * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
     * corresponding to low luminance, medium luminance, and high luminance areas respectively.
     */
    class ColorCurves {
        private _dirty;
        private _tempColor;
        private _globalCurve;
        private _highlightsCurve;
        private _midtonesCurve;
        private _shadowsCurve;
        private _positiveCurve;
        private _negativeCurve;
        private _globalHue;
        private _globalDensity;
        private _globalSaturation;
        private _globalExposure;
        /**
         * Gets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the global Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        globalHue: number;
        /**
         * Gets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the global Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        globalDensity: number;
        /**
         * Gets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the global Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        globalSaturation: number;
        private _highlightsHue;
        private _highlightsDensity;
        private _highlightsSaturation;
        private _highlightsExposure;
        /**
         * Gets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the highlights Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        highlightsHue: number;
        /**
         * Gets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the highlights Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        highlightsDensity: number;
        /**
         * Gets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the highlights Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        highlightsSaturation: number;
        /**
         * Gets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the highlights Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        highlightsExposure: number;
        private _midtonesHue;
        private _midtonesDensity;
        private _midtonesSaturation;
        private _midtonesExposure;
        /**
         * Gets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the midtones Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        midtonesHue: number;
        /**
         * Gets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the midtones Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        midtonesDensity: number;
        /**
         * Gets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the midtones Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        midtonesSaturation: number;
        /**
         * Gets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the midtones Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        midtonesExposure: number;
        private _shadowsHue;
        private _shadowsDensity;
        private _shadowsSaturation;
        private _shadowsExposure;
        /**
         * Gets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        /**
         * Sets the shadows Hue value.
         * The hue value is a standard HSB hue in the range [0,360] where 0=red, 120=green and 240=blue. The default value is 30 degrees (orange).
         */
        shadowsHue: number;
        /**
         * Gets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        /**
         * Sets the shadows Density value.
         * The density value is in range [-100,+100] where 0 means the color filter has no effect and +100 means the color filter has maximum effect.
         * Values less than zero provide a filter of opposite hue.
         */
        shadowsDensity: number;
        /**
         * Gets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        /**
         * Sets the shadows Saturation value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase saturation and negative values decrease saturation.
         */
        shadowsSaturation: number;
        /**
         * Gets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        /**
         * Sets the shadows Exposure value.
         * This is an adjustment value in the range [-100,+100], where the default value of 0.0 makes no adjustment, positive values increase exposure and negative values decrease exposure.
         */
        shadowsExposure: number;
        getClassName(): string;
        /**
         * Binds the color curves to the shader.
         * @param colorCurves The color curve to bind
         * @param effect The effect to bind to
         */
        static Bind(colorCurves: ColorCurves, effect: Effect, positiveUniform?: string, neutralUniform?: string, negativeUniform?: string): void;
        /**
         * Prepare the list of uniforms associated with the ColorCurves effects.
         * @param uniformsList The list of uniforms used in the effect
         */
        static PrepareUniforms(uniformsList: string[]): void;
        /**
         * Returns color grading data based on a hue, density, saturation and exposure value.
         * @param filterHue The hue of the color filter.
         * @param filterDensity The density of the color filter.
         * @param saturation The saturation.
         * @param exposure The exposure.
         * @param result The result data container.
         */
        private getColorGradingDataToRef(hue, density, saturation, exposure, result);
        /**
         * Takes an input slider value and returns an adjusted value that provides extra control near the centre.
         * @param value The input slider value in range [-100,100].
         * @returns Adjusted value.
         */
        private static applyColorGradingSliderNonlinear(value);
        /**
         * Returns an RGBA Color4 based on Hue, Saturation and Brightness (also referred to as value, HSV).
         * @param hue The hue (H) input.
         * @param saturation The saturation (S) input.
         * @param brightness The brightness (B) input.
         * @result An RGBA color represented as Vector4.
         */
        private static fromHSBToRef(hue, saturation, brightness, result);
        /**
         * Returns a value clamped between min and max
         * @param value The value to clamp
         * @param min The minimum of value
         * @param max The maximum of value
         * @returns The clamped value.
         */
        private static clamp(value, min, max);
        /**
         * Clones the current color curve instance.
         * @return The cloned curves
         */
        clone(): ColorCurves;
        /**
         * Serializes the current color curve instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the color curve from a json representation.
         * @param source the JSON source to parse
         * @return The parsed curves
         */
        static Parse(source: any): ColorCurves;
    }
}

declare module BABYLON {
    /**
     * EffectFallbacks can be used to add fallbacks (properties to disable) to certain properties when desired to improve performance.
     * (Eg. Start at high quality with reflection and fog, if fps is low, remove reflection, if still low remove fog)
     */
    class EffectFallbacks {
        private _defines;
        private _currentRank;
        private _maxRank;
        private _mesh;
        /**
         * Removes the fallback from the bound mesh.
         */
        unBindMesh(): void;
        /**
         * Adds a fallback on the specified property.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param define The name of the define in the shader
         */
        addFallback(rank: number, define: string): void;
        /**
         * Sets the mesh to use CPU skinning when needing to fallback.
         * @param rank The rank of the fallback (Lower ranks will be fallbacked to first)
         * @param mesh The mesh to use the fallbacks.
         */
        addCPUSkinningFallback(rank: number, mesh: AbstractMesh): void;
        /**
         * Checks to see if more fallbacks are still availible.
         */
        readonly isMoreFallbacks: boolean;
        /**
         * Removes the defines that shoould be removed when falling back.
         * @param currentDefines defines the current define statements for the shader.
         * @param effect defines the current effect we try to compile
         * @returns The resulting defines with defines of the current rank removed.
         */
        reduce(currentDefines: string, effect: Effect): string;
    }
    /**
     * Options to be used when creating an effect.
     */
    class EffectCreationOptions {
        /**
         * Atrributes that will be used in the shader.
         */
        attributes: string[];
        /**
         * Uniform varible names that will be set in the shader.
         */
        uniformsNames: string[];
        /**
         * Uniform buffer varible names that will be set in the shader.
         */
        uniformBuffersNames: string[];
        /**
         * Sampler texture variable names that will be set in the shader.
         */
        samplers: string[];
        /**
         * Define statements that will be set in the shader.
         */
        defines: any;
        /**
         * Possible fallbacks for this effect to improve performance when needed.
         */
        fallbacks: Nullable<EffectFallbacks>;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: Effect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: Effect, errors: string) => void>;
        /**
         * Parameters to be used with Babylons include syntax to iterate over an array (eg. {lights: 10})
         */
        indexParameters: any;
        /**
         * Max number of lights that can be used in the shader.
         */
        maxSimultaneousLights: number;
        /**
         * See https://developer.mozilla.org/en-US/docs/Web/API/WebGL2RenderingContext/transformFeedbackVaryings
         */
        transformFeedbackVaryings: Nullable<string[]>;
    }
    /**
     * Effect containing vertex and fragment shader that can be executed on an object.
     */
    class Effect {
        /**
         * Name of the effect.
         */
        name: any;
        /**
         * String container all the define statements that should be set on the shader.
         */
        defines: string;
        /**
         * Callback that will be called when the shader is compiled.
         */
        onCompiled: Nullable<(effect: Effect) => void>;
        /**
         * Callback that will be called if an error occurs during shader compilation.
         */
        onError: Nullable<(effect: Effect, errors: string) => void>;
        /**
         * Callback that will be called when effect is bound.
         */
        onBind: Nullable<(effect: Effect) => void>;
        /**
         * Unique ID of the effect.
         */
        uniqueId: number;
        /**
         * Observable that will be called when the shader is compiled.
         */
        onCompileObservable: Observable<Effect>;
        /**
         * Observable that will be called if an error occurs during shader compilation.
         */
        onErrorObservable: Observable<Effect>;
        /**
         * Observable that will be called when effect is bound.
         */
        onBindObservable: Observable<Effect>;
        private static _uniqueIdSeed;
        private _engine;
        private _uniformBuffersNames;
        private _uniformsNames;
        private _samplers;
        private _isReady;
        private _compilationError;
        private _attributesNames;
        private _attributes;
        private _uniforms;
        /**
         * Key for the effect.
         */
        _key: string;
        private _indexParameters;
        private _fallbacks;
        private _vertexSourceCode;
        private _fragmentSourceCode;
        private _vertexSourceCodeOverride;
        private _fragmentSourceCodeOverride;
        private _transformFeedbackVaryings;
        /**
         * Compiled shader to webGL program.
         */
        _program: WebGLProgram;
        private _valueCache;
        private static _baseCache;
        /**
         * Instantiates an effect.
         * An effect can be used to create/manage/execute vertex and fragment shaders.
         * @param baseName Name of the effect.
         * @param attributesNamesOrOptions List of attribute names that will be passed to the shader or set of all options to create the effect.
         * @param uniformsNamesOrEngine List of uniform variable names that will be passed to the shader or the engine that will be used to render effect.
         * @param samplers List of sampler variables that will be passed to the shader.
         * @param engine Engine to be used to render the effect
         * @param defines Define statements to be added to the shader.
         * @param fallbacks Possible fallbacks for this effect to improve performance when needed.
         * @param onCompiled Callback that will be called when the shader is compiled.
         * @param onError Callback that will be called if an error occurs during shader compilation.
         * @param indexParameters Parameters to be used with Babylons include syntax to iterate over an array (eg. {lights: 10})
         */
        constructor(baseName: any, attributesNamesOrOptions: string[] | EffectCreationOptions, uniformsNamesOrEngine: string[] | Engine, samplers?: Nullable<string[]>, engine?: Engine, defines?: Nullable<string>, fallbacks?: Nullable<EffectFallbacks>, onCompiled?: Nullable<(effect: Effect) => void>, onError?: Nullable<(effect: Effect, errors: string) => void>, indexParameters?: any);
        /**
         * Unique key for this effect
         */
        readonly key: string;
        /**
         * If the effect has been compiled and prepared.
         * @returns if the effect is compiled and prepared.
         */
        isReady(): boolean;
        /**
         * The engine the effect was initialized with.
         * @returns the engine.
         */
        getEngine(): Engine;
        /**
         * The compiled webGL program for the effect
         * @returns the webGL program.
         */
        getProgram(): WebGLProgram;
        /**
         * The set of names of attribute variables for the shader.
         * @returns An array of attribute names.
         */
        getAttributesNames(): string[];
        /**
         * Returns the attribute at the given index.
         * @param index The index of the attribute.
         * @returns The location of the attribute.
         */
        getAttributeLocation(index: number): number;
        /**
         * Returns the attribute based on the name of the variable.
         * @param name of the attribute to look up.
         * @returns the attribute location.
         */
        getAttributeLocationByName(name: string): number;
        /**
         * The number of attributes.
         * @returns the numnber of attributes.
         */
        getAttributesCount(): number;
        /**
         * Gets the index of a uniform variable.
         * @param uniformName of the uniform to look up.
         * @returns the index.
         */
        getUniformIndex(uniformName: string): number;
        /**
         * Returns the attribute based on the name of the variable.
         * @param uniformName of the uniform to look up.
         * @returns the location of the uniform.
         */
        getUniform(uniformName: string): Nullable<WebGLUniformLocation>;
        /**
         * Returns an array of sampler variable names
         * @returns The array of sampler variable neames.
         */
        getSamplers(): string[];
        /**
         * The error from the last compilation.
         * @returns the error string.
         */
        getCompilationError(): string;
        /**
         * Adds a callback to the onCompiled observable and call the callback imediatly if already ready.
         * @param func The callback to be used.
         */
        executeWhenCompiled(func: (effect: Effect) => void): void;
        /** @ignore */
        _loadVertexShader(vertex: any, callback: (data: any) => void): void;
        /** @ignore */
        _loadFragmentShader(fragment: any, callback: (data: any) => void): void;
        private _dumpShadersSource(vertexCode, fragmentCode, defines);
        private _processShaderConversion(sourceCode, isFragment, callback);
        private _processIncludes(sourceCode, callback);
        private _processPrecision(source);
        /**
         * Recompiles the webGL program
         * @param vertexSourceCode The source code for the vertex shader.
         * @param fragmentSourceCode The source code for the fragment shader.
         * @param onCompiled Callback called when completed.
         * @param onError Callback called on error.
         */
        _rebuildProgram(vertexSourceCode: string, fragmentSourceCode: string, onCompiled: (program: WebGLProgram) => void, onError: (message: string) => void): void;
        /**
         * Gets the uniform locations of the the specified variable names
         * @param names THe names of the variables to lookup.
         * @returns Array of locations in the same order as variable names.
         */
        getSpecificUniformLocations(names: string[]): Nullable<WebGLUniformLocation>[];
        /**
         * Prepares the effect
         */
        _prepareEffect(): void;
        /**
         * Checks if the effect is supported. (Must be called after compilation)
         */
        readonly isSupported: boolean;
        /**
         * Binds a texture to the engine to be used as output of the shader.
         * @param channel Name of the output variable.
         * @param texture Texture to bind.
         */
        _bindTexture(channel: string, texture: InternalTexture): void;
        /**
         * Sets a texture on the engine to be used in the shader.
         * @param channel Name of the sampler variable.
         * @param texture Texture to set.
         */
        setTexture(channel: string, texture: Nullable<BaseTexture>): void;
        /**
         * Sets a depth stencil texture from a render target on the engine to be used in the shader.
         * @param channel Name of the sampler variable.
         * @param texture Texture to set.
         */
        setDepthStencilTexture(channel: string, texture: Nullable<RenderTargetTexture>): void;
        /**
         * Sets an array of textures on the engine to be used in the shader.
         * @param channel Name of the variable.
         * @param textures Textures to set.
         */
        setTextureArray(channel: string, textures: BaseTexture[]): void;
        /**
         * Sets a texture to be the input of the specified post process. (To use the output, pass in the next post process in the pipeline)
         * @param channel Name of the sampler variable.
         * @param postProcess Post process to get the input texture from.
         */
        setTextureFromPostProcess(channel: string, postProcess: Nullable<PostProcess>): void;
        /**
         * (Warning! setTextureFromPostProcessOutput may be desired instead)
         * Sets the input texture of the passed in post process to be input of this effect. (To use the output of the passed in post process use setTextureFromPostProcessOutput)
         * @param channel Name of the sampler variable.
         * @param postProcess Post process to get the output texture from.
         */
        setTextureFromPostProcessOutput(channel: string, postProcess: Nullable<PostProcess>): void;
        /** @ignore */
        _cacheMatrix(uniformName: string, matrix: Matrix): boolean;
        /** @ignore */
        _cacheFloat2(uniformName: string, x: number, y: number): boolean;
        /** @ignore */
        _cacheFloat3(uniformName: string, x: number, y: number, z: number): boolean;
        /** @ignore */
        _cacheFloat4(uniformName: string, x: number, y: number, z: number, w: number): boolean;
        /**
         * Binds a buffer to a uniform.
         * @param buffer Buffer to bind.
         * @param name Name of the uniform variable to bind to.
         */
        bindUniformBuffer(buffer: WebGLBuffer, name: string): void;
        /**
         * Binds block to a uniform.
         * @param blockName Name of the block to bind.
         * @param index Index to bind.
         */
        bindUniformBlock(blockName: string, index: number): void;
        /**
         * Sets an interger value on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value Value to be set.
         * @returns this effect.
         */
        setInt(uniformName: string, value: number): Effect;
        /**
         * Sets an int array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray2(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray3(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an int array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setIntArray4(uniformName: string, array: Int32Array): Effect;
        /**
         * Sets an float array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray2(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray3(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an float array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setFloatArray4(uniformName: string, array: Float32Array): Effect;
        /**
         * Sets an array on a uniform variable.
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 2 on a uniform variable. (Array is specified as single array eg. [1,2,3,4] will result in [[1,2],[3,4]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray2(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 3 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6] will result in [[1,2,3],[4,5,6]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray3(uniformName: string, array: number[]): Effect;
        /**
         * Sets an array 4 on a uniform variable. (Array is specified as single array eg. [1,2,3,4,5,6,7,8] will result in [[1,2,3,4],[5,6,7,8]] in the shader)
         * @param uniformName Name of the variable.
         * @param array array to be set.
         * @returns this effect.
         */
        setArray4(uniformName: string, array: number[]): Effect;
        /**
         * Sets matrices on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrices matrices to be set.
         * @returns this effect.
         */
        setMatrices(uniformName: string, matrices: Float32Array): Effect;
        /**
         * Sets matrix on a uniform variable.
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix(uniformName: string, matrix: Matrix): Effect;
        /**
         * Sets a 3x3 matrix on a uniform variable. (Speicified as [1,2,3,4,5,6,7,8,9] will result in [1,2,3][4,5,6][7,8,9] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix3x3(uniformName: string, matrix: Float32Array): Effect;
        /**
         * Sets a 2x2 matrix on a uniform variable. (Speicified as [1,2,3,4] will result in [1,2][3,4] matrix)
         * @param uniformName Name of the variable.
         * @param matrix matrix to be set.
         * @returns this effect.
         */
        setMatrix2x2(uniformName: string, matrix: Float32Array): Effect;
        /**
         * Sets a float on a uniform variable.
         * @param uniformName Name of the variable.
         * @param value value to be set.
         * @returns this effect.
         */
        setFloat(uniformName: string, value: number): Effect;
        /**
         * Sets a boolean on a uniform variable.
         * @param uniformName Name of the variable.
         * @param bool value to be set.
         * @returns this effect.
         */
        setBool(uniformName: string, bool: boolean): Effect;
        /**
         * Sets a Vector2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector2 vector2 to be set.
         * @returns this effect.
         */
        setVector2(uniformName: string, vector2: Vector2): Effect;
        /**
         * Sets a float2 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float2.
         * @param y Second float in float2.
         * @returns this effect.
         */
        setFloat2(uniformName: string, x: number, y: number): Effect;
        /**
         * Sets a Vector3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector3 Value to be set.
         * @returns this effect.
         */
        setVector3(uniformName: string, vector3: Vector3): Effect;
        /**
         * Sets a float3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float3.
         * @param y Second float in float3.
         * @param z Third float in float3.
         * @returns this effect.
         */
        setFloat3(uniformName: string, x: number, y: number, z: number): Effect;
        /**
         * Sets a Vector4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param vector4 Value to be set.
         * @returns this effect.
         */
        setVector4(uniformName: string, vector4: Vector4): Effect;
        /**
         * Sets a float4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param x First float in float4.
         * @param y Second float in float4.
         * @param z Third float in float4.
         * @param w Fourth float in float4.
         * @returns this effect.
         */
        setFloat4(uniformName: string, x: number, y: number, z: number, w: number): Effect;
        /**
         * Sets a Color3 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @returns this effect.
         */
        setColor3(uniformName: string, color3: Color3): Effect;
        /**
         * Sets a Color4 on a uniform variable.
         * @param uniformName Name of the variable.
         * @param color3 Value to be set.
         * @param alpha Alpha value to be set.
         * @returns this effect.
         */
        setColor4(uniformName: string, color3: Color3, alpha: number): Effect;
        /**
         * Sets a Color4 on a uniform variable
         * @param uniformName defines the name of the variable
         * @param color4 defines the value to be set
         * @returns this effect.
         */
        setDirectColor4(uniformName: string, color4: Color4): Effect;
        /**
         * Store of each shader (The can be looked up using effect.key)
         */
        static ShadersStore: {
            [key: string]: string;
        };
        /**
         * Store of each included file for a shader (The can be looked up using effect.key)
         */
        static IncludesShadersStore: {
            [key: string]: string;
        };
        /**
         * Resets the cache of effects.
         */
        static ResetCache(): void;
    }
}

declare module BABYLON {
    class FresnelParameters {
        private _isEnabled;
        isEnabled: boolean;
        leftColor: Color3;
        rightColor: Color3;
        bias: number;
        power: number;
        clone(): FresnelParameters;
        serialize(): any;
        static Parse(parsedFresnelParameters: any): FresnelParameters;
    }
}

declare module BABYLON {
    /**
     * Interface to follow in your material defines to integrate easily the
     * Image proccessing functions.
     * @ignore
     */
    interface IImageProcessingConfigurationDefines {
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        CONTRAST: boolean;
        EXPOSURE: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
    }
    /**
     * This groups together the common properties used for image processing either in direct forward pass
     * or through post processing effect depending on the use of the image processing pipeline in your scene
     * or not.
     */
    class ImageProcessingConfiguration {
        /**
         * Color curves setup used in the effect if colorCurvesEnabled is set to true
         */
        colorCurves: Nullable<ColorCurves>;
        private _colorCurvesEnabled;
        /**
         * Gets wether the color curves effect is enabled.
         */
        /**
         * Sets wether the color curves effect is enabled.
         */
        colorCurvesEnabled: boolean;
        /**
         * Color grading LUT texture used in the effect if colorGradingEnabled is set to true
         */
        colorGradingTexture: Nullable<BaseTexture>;
        private _colorGradingEnabled;
        /**
         * Gets wether the color grading effect is enabled.
         */
        /**
         * Sets wether the color grading effect is enabled.
         */
        colorGradingEnabled: boolean;
        private _colorGradingWithGreenDepth;
        /**
         * Gets wether the color grading effect is using a green depth for the 3d Texture.
         */
        /**
         * Sets wether the color grading effect is using a green depth for the 3d Texture.
         */
        colorGradingWithGreenDepth: boolean;
        private _colorGradingBGR;
        /**
         * Gets wether the color grading texture contains BGR values.
         */
        /**
         * Sets wether the color grading texture contains BGR values.
         */
        colorGradingBGR: boolean;
        _exposure: number;
        /**
         * Gets the Exposure used in the effect.
         */
        /**
         * Sets the Exposure used in the effect.
         */
        exposure: number;
        private _toneMappingEnabled;
        /**
         * Gets wether the tone mapping effect is enabled.
         */
        /**
         * Sets wether the tone mapping effect is enabled.
         */
        toneMappingEnabled: boolean;
        protected _contrast: number;
        /**
         * Gets the contrast used in the effect.
         */
        /**
         * Sets the contrast used in the effect.
         */
        contrast: number;
        /**
         * Vignette stretch size.
         */
        vignetteStretch: number;
        /**
         * Vignette centre X Offset.
         */
        vignetteCentreX: number;
        /**
         * Vignette centre Y Offset.
         */
        vignetteCentreY: number;
        /**
         * Vignette weight or intensity of the vignette effect.
         */
        vignetteWeight: number;
        /**
         * Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        vignetteColor: Color4;
        /**
         * Camera field of view used by the Vignette effect.
         */
        vignetteCameraFov: number;
        private _vignetteBlendMode;
        /**
         * Gets the vignette blend mode allowing different kind of effect.
         */
        /**
         * Sets the vignette blend mode allowing different kind of effect.
         */
        vignetteBlendMode: number;
        private _vignetteEnabled;
        /**
         * Gets wether the vignette effect is enabled.
         */
        /**
         * Sets wether the vignette effect is enabled.
         */
        vignetteEnabled: boolean;
        private _applyByPostProcess;
        /**
         * Gets wether the image processing is applied through a post process or not.
         */
        /**
         * Sets wether the image processing is applied through a post process or not.
         */
        applyByPostProcess: boolean;
        private _isEnabled;
        /**
         * Gets wether the image processing is enabled or not.
         */
        /**
         * Sets wether the image processing is enabled or not.
         */
        isEnabled: boolean;
        /**
        * An event triggered when the configuration changes and requires Shader to Update some parameters.
        */
        onUpdateParameters: Observable<ImageProcessingConfiguration>;
        /**
         * Method called each time the image processing information changes requires to recompile the effect.
         */
        protected _updateParameters(): void;
        getClassName(): string;
        /**
         * Prepare the list of uniforms associated with the Image Processing effects.
         * @param uniformsList The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareUniforms(uniforms: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of samplers associated with the Image Processing effects.
         * @param uniformsList The list of uniforms used in the effect
         * @param defines the list of defines currently in use
         */
        static PrepareSamplers(samplersList: string[], defines: IImageProcessingConfigurationDefines): void;
        /**
         * Prepare the list of defines associated to the shader.
         * @param defines the list of defines to complete
         */
        prepareDefines(defines: IImageProcessingConfigurationDefines, forPostProcess?: boolean): void;
        /**
         * Returns true if all the image processing information are ready.
         */
        isReady(): boolean;
        /**
         * Binds the image processing to the shader.
         * @param effect The effect to bind to
         */
        bind(effect: Effect, aspectRatio?: number): void;
        /**
         * Clones the current image processing instance.
         * @return The cloned image processing
         */
        clone(): ImageProcessingConfiguration;
        /**
         * Serializes the current image processing instance to a json representation.
         * @return a JSON representation
         */
        serialize(): any;
        /**
         * Parses the image processing from a json representation.
         * @param source the JSON source to parse
         * @return The parsed image processing
         */
        static Parse(source: any): ImageProcessingConfiguration;
        private static _VIGNETTEMODE_MULTIPLY;
        private static _VIGNETTEMODE_OPAQUE;
        /**
         * Used to apply the vignette as a mix with the pixel color.
         */
        static readonly VIGNETTEMODE_MULTIPLY: number;
        /**
         * Used to apply the vignette as a replacement of the pixel color.
         */
        static readonly VIGNETTEMODE_OPAQUE: number;
    }
}

declare module BABYLON {
    /**
     * Manages the defines for the Material.
     */
    class MaterialDefines {
        private _keys;
        private _isDirty;
        _renderId: number;
        _areLightsDirty: boolean;
        _areAttributesDirty: boolean;
        _areTexturesDirty: boolean;
        _areFresnelDirty: boolean;
        _areMiscDirty: boolean;
        _areImageProcessingDirty: boolean;
        _normals: boolean;
        _uvs: boolean;
        _needNormals: boolean;
        _needUVs: boolean;
        /**
         * Specifies if the material needs to be re-calculated.
         */
        readonly isDirty: boolean;
        /**
         * Marks the material to indicate that it has been re-calculated.
         */
        markAsProcessed(): void;
        /**
         * Marks the material to indicate that it needs to be re-calculated.
         */
        markAsUnprocessed(): void;
        /**
         * Marks the material to indicate all of its defines need to be re-calculated.
         */
        markAllAsDirty(): void;
        /**
         * Marks the material to indicate that image processing needs to be re-calculated.
         */
        markAsImageProcessingDirty(): void;
        /**
         * Marks the material to indicate the lights need to be re-calculated.
         */
        markAsLightDirty(): void;
        /**
         * Marks the attribute state as changed.
         */
        markAsAttributesDirty(): void;
        /**
         * Marks the texture state as changed.
         */
        markAsTexturesDirty(): void;
        /**
         * Marks the fresnel state as changed.
         */
        markAsFresnelDirty(): void;
        /**
         * Marks the misc state as changed.
         */
        markAsMiscDirty(): void;
        /**
         * Rebuilds the material defines.
         */
        rebuild(): void;
        /**
         * Specifies if two material defines are equal.
         * @param other - A material define instance to compare to.
         * @returns - Boolean indicating if the material defines are equal (true) or not (false).
         */
        isEqual(other: MaterialDefines): boolean;
        /**
         * Clones this instance's defines to another instance.
         * @param other - material defines to clone values to.
         */
        cloneTo(other: MaterialDefines): void;
        /**
         * Resets the material define values.
         */
        reset(): void;
        /**
         * Converts the material define values to a string.
         * @returns - String of material define information.
         */
        toString(): string;
    }
    /**
     * This offers the main features of a material in BJS.
     */
    class Material implements IAnimatable {
        private static _TriangleFillMode;
        private static _WireFrameFillMode;
        private static _PointFillMode;
        private static _PointListDrawMode;
        private static _LineListDrawMode;
        private static _LineLoopDrawMode;
        private static _LineStripDrawMode;
        private static _TriangleStripDrawMode;
        private static _TriangleFanDrawMode;
        /**
         * Returns the triangle fill mode.
         */
        static readonly TriangleFillMode: number;
        /**
         * Returns the wireframe mode.
         */
        static readonly WireFrameFillMode: number;
        /**
         * Returns the point fill mode.
         */
        static readonly PointFillMode: number;
        /**
         * Returns the point list draw mode.
         */
        static readonly PointListDrawMode: number;
        /**
         * Returns the line list draw mode.
         */
        static readonly LineListDrawMode: number;
        /**
         * Returns the line loop draw mode.
         */
        static readonly LineLoopDrawMode: number;
        /**
         * Returns the line strip draw mode.
         */
        static readonly LineStripDrawMode: number;
        /**
         * Returns the triangle strip draw mode.
         */
        static readonly TriangleStripDrawMode: number;
        /**
         * Returns the triangle fan draw mode.
         */
        static readonly TriangleFanDrawMode: number;
        /**
         * Stores the clock-wise side orientation.
         */
        private static _ClockWiseSideOrientation;
        /**
         * Stores the counter clock-wise side orientation.
         */
        private static _CounterClockWiseSideOrientation;
        /**
         * Returns the clock-wise side orientation.
         */
        static readonly ClockWiseSideOrientation: number;
        /**
         * Returns the counter clock-wise side orientation.
         */
        static readonly CounterClockWiseSideOrientation: number;
        /**
         * The dirty texture flag value.
         */
        private static _TextureDirtyFlag;
        /**
         * The dirty light flag value.
         */
        private static _LightDirtyFlag;
        /**
         * The dirty fresnel flag value.
         */
        private static _FresnelDirtyFlag;
        /**
         * The dirty attribute flag value.
         */
        private static _AttributesDirtyFlag;
        /**
         * The dirty misc flag value.
         */
        private static _MiscDirtyFlag;
        /**
         * Returns the dirty texture flag value.
         */
        static readonly TextureDirtyFlag: number;
        /**
         * Returns the dirty light flag value.
         */
        static readonly LightDirtyFlag: number;
        /**
         * Returns the dirty fresnel flag value.
         */
        static readonly FresnelDirtyFlag: number;
        /**
         * Returns the dirty attributes flag value.
         */
        static readonly AttributesDirtyFlag: number;
        /**
         * Returns the dirty misc flag value.
         */
        static readonly MiscDirtyFlag: number;
        /**
         * The ID of the material.
         */
        id: string;
        /**
         * The name of the material.
         */
        name: string;
        /**
         * Specifies if the ready state should be checked on each call.
         */
        checkReadyOnEveryCall: boolean;
        /**
         * Specifies if the ready state should be checked once.
         */
        checkReadyOnlyOnce: boolean;
        /**
         * The state of the material.
         */
        state: string;
        /**
         * The alpha value of the material.
         */
        protected _alpha: number;
        /**
         * Gets the alpha value of the material.
         */
        /**
         * Sets the alpha value of the material.
         */
        alpha: number;
        /**
         * Specifies if back face culling is enabled.
         */
        protected _backFaceCulling: boolean;
        /**
         * Gets the back-face culling state.
         */
        /**
         * Sets the back-face culling state.
         */
        backFaceCulling: boolean;
        /**
         * Stores the value for side orientation.
         */
        sideOrientation: number;
        /**
         * Callback triggered when the material is compiled.
         */
        onCompiled: (effect: Effect) => void;
        /**
         * Callback triggered when an error occurs.
         */
        onError: (effect: Effect, errors: string) => void;
        /**
         * Callback triggered to get the render target textures.
         */
        getRenderTargetTextures: () => SmartArray<RenderTargetTexture>;
        /**
         * Specifies if the material should be serialized.
         */
        doNotSerialize: boolean;
        /**
         * Specifies if the effect should be stored on sub meshes.
         */
        storeEffectOnSubMeshes: boolean;
        /**
         * Stores the animations for the material.
         */
        animations: Array<Animation>;
        /**
        * An event triggered when the material is disposed.
        */
        onDisposeObservable: Observable<Material>;
        /**
         * An observer which watches for dispose events.
         */
        private _onDisposeObserver;
        /**
         * Called during a dispose event.
         */
        onDispose: () => void;
        /**
        * An event triggered when the material is bound.
        */
        onBindObservable: Observable<AbstractMesh>;
        /**
         * An observer which watches for bind events.
         */
        private _onBindObserver;
        /**
         * Called during a bind event.
         */
        onBind: (Mesh: AbstractMesh) => void;
        /**
        * An event triggered when the material is unbound.
        */
        onUnBindObservable: Observable<Material>;
        /**
         * Stores the value of the alpha mode.
         */
        private _alphaMode;
        /**
         * Gets the value of the alpha mode.
         */
        /**
         * Sets the value of the alpha mode.
         *
         * | Value | Type | Description |
         * | --- | --- | --- |
         * | 0 | ALPHA_DISABLE |   |
         * | 1 | ALPHA_ADD |   |
         * | 2 | ALPHA_COMBINE |   |
         * | 3 | ALPHA_SUBTRACT |   |
         * | 4 | ALPHA_MULTIPLY |   |
         * | 5 | ALPHA_MAXIMIZED |   |
         * | 6 | ALPHA_ONEONE |   |
         * | 7 | ALPHA_PREMULTIPLIED |   |
         * | 8 | ALPHA_PREMULTIPLIED_PORTERDUFF |   |
         * | 9 | ALPHA_INTERPOLATE |   |
         * | 10 | ALPHA_SCREENMODE |   |
         *
         */
        alphaMode: number;
        /**
         * Stores the state of the need depth pre-pass value.
         */
        private _needDepthPrePass;
        /**
         * Gets the depth pre-pass value.
         */
        /**
         * Sets the need depth pre-pass value.
         */
        needDepthPrePass: boolean;
        /**
         * Specifies if depth writing should be disabled.
         */
        disableDepthWrite: boolean;
        /**
         * Specifies if depth writing should be forced.
         */
        forceDepthWrite: boolean;
        /**
         * Specifies if there should be a separate pass for culling.
         */
        separateCullingPass: boolean;
        /**
         * Stores the state specifing if fog should be enabled.
         */
        private _fogEnabled;
        /**
         * Gets the value of the fog enabled state.
         */
        /**
         * Sets the state for enabling fog.
         */
        fogEnabled: boolean;
        /**
         * Stores the size of points.
         */
        pointSize: number;
        /**
         * Stores the z offset value.
         */
        zOffset: number;
        /**
         * Gets a value specifying if wireframe mode is enabled.
         */
        /**
         * Sets the state of wireframe mode.
         */
        wireframe: boolean;
        /**
         * Gets the value specifying if point clouds are enabled.
         */
        /**
         * Sets the state of point cloud mode.
         */
        pointsCloud: boolean;
        /**
         * Gets the material fill mode.
         */
        /**
         * Sets the material fill mode.
         */
        fillMode: number;
        /**
         * Stores the effects for the material.
         */
        _effect: Nullable<Effect>;
        /**
         * Specifies if the material was previously ready.
         */
        _wasPreviouslyReady: boolean;
        /**
         * Specifies if uniform buffers should be used.
         */
        private _useUBO;
        /**
         * Stores a reference to the scene.
         */
        private _scene;
        /**
         * Stores the fill mode state.
         */
        private _fillMode;
        /**
         * Specifies if the depth write state should be cached.
         */
        private _cachedDepthWriteState;
        /**
         * Stores the uniform buffer.
         */
        protected _uniformBuffer: UniformBuffer;
        /**
         * Creates a material instance.
         * @param name - The name of the material.
         * @param scene - The BJS scene to reference.
         * @param doNotAdd - Specifies if the material should be added to the scene.
         */
        constructor(name: string, scene: Scene, doNotAdd?: boolean);
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         * subclasses should override adding information pertainent to themselves.
         * @returns - String with material information.
         */
        toString(fullDetails?: boolean): string;
        /**
         * Gets the class name of the material.
         * @returns - String with the class name of the material.
         */
        getClassName(): string;
        /**
         * Specifies if updates for the material been locked.
         */
        readonly isFrozen: boolean;
        /**
         * Locks updates for the material.
         */
        freeze(): void;
        /**
         * Unlocks updates for the material.
         */
        unfreeze(): void;
        /**
         * Specifies if the material is ready to be used.
         * @param mesh - BJS mesh.
         * @param useInstances - Specifies if instances should be used.
         * @returns - Boolean indicating if the material is ready to be used.
         */
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        /**
         * Specifies that the submesh is ready to be used.
         * @param mesh - BJS mesh.
         * @param subMesh - A submesh of the BJS mesh.  Used to check if it is ready.
         * @param useInstances - Specifies that instances should be used.
         * @returns - boolean indicating that the submesh is ready or not.
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: BaseSubMesh, useInstances?: boolean): boolean;
        /**
         * Returns the material effect.
         * @returns - Nullable material effect.
         */
        getEffect(): Nullable<Effect>;
        /**
         * Returns the BJS scene.
         * @returns - BJS Scene.
         */
        getScene(): Scene;
        /**
         * Specifies if the material will require alpha blending
         * @returns - Boolean specifying if alpha blending is needed.
         */
        needAlphaBlending(): boolean;
        /**
         * Specifies if the mesh will require alpha blending.
         * @param mesh - BJS mesh.
         * @returns - Boolean specifying if alpha blending is needed for the mesh.
         */
        needAlphaBlendingForMesh(mesh: AbstractMesh): boolean;
        /**
         * Specifies if this material should be rendered in alpha test mode.
         * @returns - Boolean specifying if an alpha test is needed.
         */
        needAlphaTesting(): boolean;
        /**
         * Gets the texture used for the alpha test.
         * @returns - Nullable alpha test texture.
         */
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Marks the material to indicate that it needs to be re-calculated.
         */
        markDirty(): void;
        _preBind(effect?: Effect, overrideOrientation?: Nullable<number>): boolean;
        /**
         * Binds the material to the mesh.
         * @param world - World transformation matrix.
         * @param mesh - Mesh to bind the material to.
         */
        bind(world: Matrix, mesh?: Mesh): void;
        /**
         * Binds the submesh to the material.
         * @param world - World transformation matrix.
         * @param mesh - Mesh containing the submesh.
         * @param subMesh - Submesh to bind the material to.
         */
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        /**
         * Binds the world matrix to the material.
         * @param world - World transformation matrix.
         */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the scene's uniform buffer to the effect.
         * @param effect - Effect to bind to the scene uniform buffer.
         * @param sceneUbo - Scene uniform buffer.
         */
        bindSceneUniformBuffer(effect: Effect, sceneUbo: UniformBuffer): void;
        /**
         * Binds the view matrix to the effect.
         * @param effect - Effect to bind the view matrix to.
         */
        bindView(effect: Effect): void;
        /**
         * Binds the view projection matrix to the effect.
         * @param effect - Effect to bind the view projection matrix to.
         */
        bindViewProjection(effect: Effect): void;
        /**
         * Specifies if material alpha testing should be turned on for the mesh.
         * @param mesh - BJS mesh.
         */
        protected _shouldTurnAlphaTestOn(mesh: AbstractMesh): boolean;
        /**
         * Processes to execute after binding the material to a mesh.
         * @param mesh - BJS mesh.
         */
        protected _afterBind(mesh?: Mesh): void;
        /**
         * Unbinds the material from the mesh.
         */
        unbind(): void;
        /**
         * Gets the active textures from the material.
         * @returns - Array of textures.
         */
        getActiveTextures(): BaseTexture[];
        /**
         * Specifies if the material uses a texture.
         * @param texture - Texture to check against the material.
         * @returns - Boolean specifying if the material uses the texture.
         */
        hasTexture(texture: BaseTexture): boolean;
        /**
         * Makes a duplicate of the material, and gives it a new name.
         * @param name - Name to call the duplicated material.
         * @returns - Nullable cloned material
         */
        clone(name: string): Nullable<Material>;
        /**
         * Gets the meshes bound to the material.
         * @returns - Array of meshes bound to the material.
         */
        getBindedMeshes(): AbstractMesh[];
        /**
         * Force shader compilation
         * @param mesh - BJS mesh.
         * @param onCompiled - function to execute once the material is compiled.
         * @param options - options to pass to this function.
         */
        forceCompilation(mesh: AbstractMesh, onCompiled?: (material: Material) => void, options?: Partial<{
            clipPlane: boolean;
        }>): void;
        /**
         * Force shader compilation.
         * @param mesh The mesh that will use this material
         * @param options Additional options for compiling the shaders
         * @returns A promise that resolves when the compilation completes
         */
        forceCompilationAsync(mesh: AbstractMesh, options?: Partial<{
            clipPlane: boolean;
        }>): Promise<void>;
        /**
         * Marks a define in the material to indicate that it needs to be re-computed.
         * @param flag - Material define flag.
         */
        markAsDirty(flag: number): void;
        /**
         * Marks all submeshes of a material to indicate that their material defines need to be re-calculated.
         * @param func - function which checks material defines against the submeshes.
         */
        protected _markAllSubMeshesAsDirty(func: (defines: MaterialDefines) => void): void;
        /**
         * Indicates that image processing needs to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsImageProcessingDirty(): void;
        /**
         * Indicates that textures need to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsTexturesDirty(): void;
        /**
         * Indicates that fresnel needs to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsFresnelDirty(): void;
        /**
         * Indicates that fresnel and misc need to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsFresnelAndMiscDirty(): void;
        /**
         * Indicates that lights need to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsLightsDirty(): void;
        /**
         * Indicates that attributes need to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsAttributesDirty(): void;
        /**
         * Indicates that misc needs to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsMiscDirty(): void;
        /**
         * Indicates that textures and misc need to be re-calculated for all submeshes.
         */
        protected _markAllSubMeshesAsTexturesAndMiscDirty(): void;
        /**
         * Disposes the material.
         * @param forceDisposeEffect - Specifies if effects should be force disposed.
         * @param forceDisposeTextures - Specifies if textures should be force disposed.
         */
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        /**
         * Serializes this material.
         * @returns - serialized material object.
         */
        serialize(): any;
        /**
         * Creates a MultiMaterial from parse MultiMaterial data.
         * @param parsedMultiMaterial - Parsed MultiMaterial data.
         * @param scene - BJS scene.
         * @returns - MultiMaterial.
         */
        static ParseMultiMaterial(parsedMultiMaterial: any, scene: Scene): MultiMaterial;
        /**
         * Creates a material from parsed material data.
         * @param parsedMaterial - Parsed material data.
         * @param scene - BJS scene.
         * @param rootUrl - Root URL containing the material information.
         * @returns - Parsed material.
         */
        static Parse(parsedMaterial: any, scene: Scene, rootUrl: string): any;
    }
}

declare module BABYLON {
    /**
     * "Static Class" containing the most commonly used helper while dealing with material for
     * rendering purpose.
     *
     * It contains the basic tools to help defining defines, binding uniform for the common part of the materials.
     *
     * This works by convention in BabylonJS but is meant to be use only with shader following the in place naming rules and conventions.
     */
    class MaterialHelper {
        /**
         * Bind the current view position to an effect.
         * @param effect The effect to be bound
         * @param scene The scene the eyes position is used from
         */
        static BindEyePosition(effect: Effect, scene: Scene): void;
        /**
         * Helps preparing the defines values about the UVs in used in the effect.
         * UVs are shared as much as we can accross chanels in the shaders.
         * @param texture The texture we are preparing the UVs for
         * @param defines The defines to update
         * @param key The chanel key "diffuse", "specular"... used in the shader
         */
        static PrepareDefinesForMergedUV(texture: BaseTexture, defines: any, key: string): void;
        /**
         * Binds a texture matrix value to its corrsponding uniform
         * @param texture The texture to bind the matrix for
         * @param uniformBuffer The uniform buffer receivin the data
         * @param key The chanel key "diffuse", "specular"... used in the shader
         */
        static BindTextureMatrix(texture: BaseTexture, uniformBuffer: UniformBuffer, key: string): void;
        /**
         * Helper used to prepare the list of defines associated with misc. values for shader compilation
         * @param mesh defines the current mesh
         * @param scene defines the current scene
         * @param useLogarithmicDepth defines if logarithmic depth has to be turned on
         * @param pointsCloud defines if point cloud rendering has to be turned on
         * @param fogEnabled defines if fog has to be turned on
         * @param alphaTest defines if alpha testing has to be turned on
         * @param defines defines the current list of defines
         */
        static PrepareDefinesForMisc(mesh: AbstractMesh, scene: Scene, useLogarithmicDepth: boolean, pointsCloud: boolean, fogEnabled: boolean, alphaTest: boolean, defines: any): void;
        /**
         * Helper used to prepare the list of defines associated with frame values for shader compilation
         * @param scene defines the current scene
         * @param engine defines the current engine
         * @param defines specifies the list of active defines
         * @param useInstances defines if instances have to be turned on
         * @param useClipPlane defines if clip plane have to be turned on
         */
        static PrepareDefinesForFrameBoundValues(scene: Scene, engine: Engine, defines: any, useInstances: boolean, useClipPlane?: Nullable<boolean>): void;
        /**
         * Prepares the defines used in the shader depending on the attributes data available in the mesh
         * @param mesh The mesh containing the geometry data we will draw
         * @param defines The defines to update
         * @param useVertexColor Precise whether vertex colors should be used or not (override mesh info)
         * @param useBones Precise whether bones should be used or not (override mesh info)
         * @param useMorphTargets Precise whether morph targets should be used or not (override mesh info)
         * @param useVertexAlpha Precise whether vertex alpha should be used or not (override mesh info)
         * @returns false if defines are considered not dirty and have not been checked
         */
        static PrepareDefinesForAttributes(mesh: AbstractMesh, defines: any, useVertexColor: boolean, useBones: boolean, useMorphTargets?: boolean, useVertexAlpha?: boolean): boolean;
        /**
         * Prepares the defines related to the light information passed in parameter
         * @param scene The scene we are intending to draw
         * @param mesh The mesh the effect is compiling for
         * @param defines The defines to update
         * @param specularSupported Specifies whether specular is supported or not (override lights data)
         * @param maxSimultaneousLights Specfies how manuy lights can be added to the effect at max
         * @param disableLighting Specifies whether the lighting is disabled (override scene and light)
         * @returns true if normals will be required for the rest of the effect
         */
        static PrepareDefinesForLights(scene: Scene, mesh: AbstractMesh, defines: any, specularSupported: boolean, maxSimultaneousLights?: number, disableLighting?: boolean): boolean;
        /**
         * Prepares the uniforms and samplers list to be used in the effect. This can automatically remove from the list uniforms
         * that won t be acctive due to defines being turned off.
         * @param uniformsListOrOptions The uniform names to prepare or an EffectCreationOptions containing the liist and extra information
         * @param samplersList The samplers list
         * @param defines The defines helping in the list generation
         * @param maxSimultaneousLights The maximum number of simultanous light allowed in the effect
         */
        static PrepareUniformsAndSamplersList(uniformsListOrOptions: string[] | EffectCreationOptions, samplersList?: string[], defines?: any, maxSimultaneousLights?: number): void;
        /**
         * This helps decreasing rank by rank the shadow quality (0 being the highest rank and quality)
         * @param defines The defines to update while falling back
         * @param fallbacks The authorized effect fallbacks
         * @param maxSimultaneousLights The maximum number of lights allowed
         * @param rank the current rank of the Effect
         * @returns The newly affected rank
         */
        static HandleFallbacksForShadows(defines: any, fallbacks: EffectFallbacks, maxSimultaneousLights?: number, rank?: number): number;
        /**
         * Prepares the list of attributes required for morph targets according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param mesh The mesh to prepare the morph targets attributes for
         * @param defines The current Defines of the effect
         */
        static PrepareAttributesForMorphTargets(attribs: string[], mesh: AbstractMesh, defines: any): void;
        /**
         * Prepares the list of attributes required for bones according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param mesh The mesh to prepare the bones attributes for
         * @param defines The current Defines of the effect
         * @param fallbacks The current efffect fallback strategy
         */
        static PrepareAttributesForBones(attribs: string[], mesh: AbstractMesh, defines: any, fallbacks: EffectFallbacks): void;
        /**
         * Prepares the list of attributes required for instances according to the effect defines.
         * @param attribs The current list of supported attribs
         * @param defines The current Defines of the effect
         */
        static PrepareAttributesForInstances(attribs: string[], defines: any): void;
        /**
         * Binds the light shadow information to the effect for the given mesh.
         * @param light The light containing the generator
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param lightIndex The light index in the effect used to render the mesh
         * @param effect The effect we are binding the data to
         */
        static BindLightShadow(light: Light, scene: Scene, mesh: AbstractMesh, lightIndex: string, effect: Effect): void;
        /**
         * Binds the light information to the effect.
         * @param light The light containing the generator
         * @param effect The effect we are binding the data to
         * @param lightIndex The light index in the effect used to render
         */
        static BindLightProperties(light: Light, effect: Effect, lightIndex: number): void;
        /**
         * Binds the lights information from the scene to the effect for the given mesh.
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         * @param defines The generated defines for the effect
         * @param maxSimultaneousLights The maximum number of light that can be bound to the effect
         * @param usePhysicalLightFalloff Specifies whether the light falloff is defined physically or not
         */
        static BindLights(scene: Scene, mesh: AbstractMesh, effect: Effect, defines: any, maxSimultaneousLights?: number, usePhysicalLightFalloff?: boolean): void;
        /**
         * Binds the fog information from the scene to the effect for the given mesh.
         * @param scene The scene the lights belongs to
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindFogParameters(scene: Scene, mesh: AbstractMesh, effect: Effect): void;
        /**
         * Binds the bones information from the mesh to the effect.
         * @param mesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindBonesParameters(mesh?: AbstractMesh, effect?: Effect): void;
        /**
         * Binds the morph targets information from the mesh to the effect.
         * @param abstractMesh The mesh we are binding the information to render
         * @param effect The effect we are binding the data to
         */
        static BindMorphTargetParameters(abstractMesh: AbstractMesh, effect: Effect): void;
        /**
         * Binds the logarithmic depth information from the scene to the effect for the given defines.
         * @param defines The generated defines used in the effect
         * @param effect The effect we are binding the data to
         * @param scene The scene we are willing to render with logarithmic scale for
         */
        static BindLogDepth(defines: any, effect: Effect, scene: Scene): void;
        /**
         * Binds the clip plane information from the scene to the effect.
         * @param scene The scene the clip plane information are extracted from
         * @param effect The effect we are binding the data to
         */
        static BindClipPlane(effect: Effect, scene: Scene): void;
    }
}

declare module BABYLON {
    class MultiMaterial extends Material {
        private _subMaterials;
        subMaterials: Nullable<Material>[];
        constructor(name: string, scene: Scene);
        private _hookArray(array);
        getSubMaterial(index: number): Nullable<Material>;
        getActiveTextures(): BaseTexture[];
        getClassName(): string;
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: BaseSubMesh, useInstances?: boolean): boolean;
        clone(name: string, cloneChildren?: boolean): MultiMaterial;
        serialize(): any;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
    }
}

declare module BABYLON {
    class PushMaterial extends Material {
        protected _activeEffect: Effect;
        protected _normalMatrix: Matrix;
        constructor(name: string, scene: Scene);
        getEffect(): Effect;
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        /**
        * Binds the given world matrix to the active effect
        *
        * @param world the matrix to bind
        */
        bindOnlyWorldMatrix(world: Matrix): void;
        /**
         * Binds the given normal matrix to the active effect
         *
         * @param normalMatrix the matrix to bind
         */
        bindOnlyNormalMatrix(normalMatrix: Matrix): void;
        bind(world: Matrix, mesh?: Mesh): void;
        protected _afterBind(mesh: Mesh, effect?: Nullable<Effect>): void;
        protected _mustRebind(scene: Scene, effect: Effect, visibility?: number): boolean;
    }
}

declare module BABYLON {
    class ShaderMaterial extends Material {
        private _shaderPath;
        private _options;
        private _textures;
        private _textureArrays;
        private _floats;
        private _ints;
        private _floatsArrays;
        private _colors3;
        private _colors3Arrays;
        private _colors4;
        private _vectors2;
        private _vectors3;
        private _vectors4;
        private _matrices;
        private _matrices3x3;
        private _matrices2x2;
        private _vectors2Arrays;
        private _vectors3Arrays;
        private _cachedWorldViewMatrix;
        private _renderId;
        constructor(name: string, scene: Scene, shaderPath: any, options: any);
        getClassName(): string;
        needAlphaBlending(): boolean;
        needAlphaTesting(): boolean;
        private _checkUniform(uniformName);
        setTexture(name: string, texture: Texture): ShaderMaterial;
        setTextureArray(name: string, textures: Texture[]): ShaderMaterial;
        setFloat(name: string, value: number): ShaderMaterial;
        setInt(name: string, value: number): ShaderMaterial;
        setFloats(name: string, value: number[]): ShaderMaterial;
        setColor3(name: string, value: Color3): ShaderMaterial;
        setColor3Array(name: string, value: Color3[]): ShaderMaterial;
        setColor4(name: string, value: Color4): ShaderMaterial;
        setVector2(name: string, value: Vector2): ShaderMaterial;
        setVector3(name: string, value: Vector3): ShaderMaterial;
        setVector4(name: string, value: Vector4): ShaderMaterial;
        setMatrix(name: string, value: Matrix): ShaderMaterial;
        setMatrix3x3(name: string, value: Float32Array): ShaderMaterial;
        setMatrix2x2(name: string, value: Float32Array): ShaderMaterial;
        setArray2(name: string, value: number[]): ShaderMaterial;
        setArray3(name: string, value: number[]): ShaderMaterial;
        private _checkCache(scene, mesh?, useInstances?);
        isReady(mesh?: AbstractMesh, useInstances?: boolean): boolean;
        bindOnlyWorldMatrix(world: Matrix): void;
        bind(world: Matrix, mesh?: Mesh): void;
        getActiveTextures(): BaseTexture[];
        hasTexture(texture: BaseTexture): boolean;
        clone(name: string): ShaderMaterial;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        serialize(): any;
        static Parse(source: any, scene: Scene, rootUrl: string): ShaderMaterial;
    }
}

declare module BABYLON {
    /** @ignore */
    class StandardMaterialDefines extends MaterialDefines implements IImageProcessingConfigurationDefines {
        MAINUV1: boolean;
        MAINUV2: boolean;
        DIFFUSE: boolean;
        DIFFUSEDIRECTUV: number;
        AMBIENT: boolean;
        AMBIENTDIRECTUV: number;
        OPACITY: boolean;
        OPACITYDIRECTUV: number;
        OPACITYRGB: boolean;
        REFLECTION: boolean;
        EMISSIVE: boolean;
        EMISSIVEDIRECTUV: number;
        SPECULAR: boolean;
        SPECULARDIRECTUV: number;
        BUMP: boolean;
        BUMPDIRECTUV: number;
        PARALLAX: boolean;
        PARALLAXOCCLUSION: boolean;
        SPECULAROVERALPHA: boolean;
        CLIPPLANE: boolean;
        ALPHATEST: boolean;
        DEPTHPREPASS: boolean;
        ALPHAFROMDIFFUSE: boolean;
        POINTSIZE: boolean;
        FOG: boolean;
        SPECULARTERM: boolean;
        DIFFUSEFRESNEL: boolean;
        OPACITYFRESNEL: boolean;
        REFLECTIONFRESNEL: boolean;
        REFRACTIONFRESNEL: boolean;
        EMISSIVEFRESNEL: boolean;
        FRESNEL: boolean;
        NORMAL: boolean;
        UV1: boolean;
        UV2: boolean;
        VERTEXCOLOR: boolean;
        VERTEXALPHA: boolean;
        NUM_BONE_INFLUENCERS: number;
        BonesPerMesh: number;
        INSTANCES: boolean;
        GLOSSINESS: boolean;
        ROUGHNESS: boolean;
        EMISSIVEASILLUMINATION: boolean;
        LINKEMISSIVEWITHDIFFUSE: boolean;
        REFLECTIONFRESNELFROMSPECULAR: boolean;
        LIGHTMAP: boolean;
        LIGHTMAPDIRECTUV: number;
        OBJECTSPACE_NORMALMAP: boolean;
        USELIGHTMAPASSHADOWMAP: boolean;
        REFLECTIONMAP_3D: boolean;
        REFLECTIONMAP_SPHERICAL: boolean;
        REFLECTIONMAP_PLANAR: boolean;
        REFLECTIONMAP_CUBIC: boolean;
        USE_LOCAL_REFLECTIONMAP_CUBIC: boolean;
        REFLECTIONMAP_PROJECTION: boolean;
        REFLECTIONMAP_SKYBOX: boolean;
        REFLECTIONMAP_EXPLICIT: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR: boolean;
        REFLECTIONMAP_EQUIRECTANGULAR_FIXED: boolean;
        REFLECTIONMAP_MIRROREDEQUIRECTANGULAR_FIXED: boolean;
        INVERTCUBICMAP: boolean;
        LOGARITHMICDEPTH: boolean;
        REFRACTION: boolean;
        REFRACTIONMAP_3D: boolean;
        REFLECTIONOVERALPHA: boolean;
        TWOSIDEDLIGHTING: boolean;
        SHADOWFLOAT: boolean;
        MORPHTARGETS: boolean;
        MORPHTARGETS_NORMAL: boolean;
        MORPHTARGETS_TANGENT: boolean;
        NUM_MORPH_INFLUENCERS: number;
        NONUNIFORMSCALING: boolean;
        PREMULTIPLYALPHA: boolean;
        IMAGEPROCESSING: boolean;
        VIGNETTE: boolean;
        VIGNETTEBLENDMODEMULTIPLY: boolean;
        VIGNETTEBLENDMODEOPAQUE: boolean;
        TONEMAPPING: boolean;
        CONTRAST: boolean;
        COLORCURVES: boolean;
        COLORGRADING: boolean;
        COLORGRADING3D: boolean;
        SAMPLER3DGREENDEPTH: boolean;
        SAMPLER3DBGRMAP: boolean;
        IMAGEPROCESSINGPOSTPROCESS: boolean;
        EXPOSURE: boolean;
        constructor();
        setReflectionMode(modeToEnable: string): void;
    }
    class StandardMaterial extends PushMaterial {
        private _diffuseTexture;
        diffuseTexture: Nullable<BaseTexture>;
        private _ambientTexture;
        ambientTexture: Nullable<BaseTexture>;
        private _opacityTexture;
        opacityTexture: Nullable<BaseTexture>;
        private _reflectionTexture;
        reflectionTexture: Nullable<BaseTexture>;
        private _emissiveTexture;
        emissiveTexture: Nullable<BaseTexture>;
        private _specularTexture;
        specularTexture: Nullable<BaseTexture>;
        private _bumpTexture;
        bumpTexture: Nullable<BaseTexture>;
        private _lightmapTexture;
        lightmapTexture: Nullable<BaseTexture>;
        private _refractionTexture;
        refractionTexture: Nullable<BaseTexture>;
        ambientColor: Color3;
        diffuseColor: Color3;
        specularColor: Color3;
        emissiveColor: Color3;
        specularPower: number;
        private _useAlphaFromDiffuseTexture;
        useAlphaFromDiffuseTexture: boolean;
        private _useEmissiveAsIllumination;
        useEmissiveAsIllumination: boolean;
        private _linkEmissiveWithDiffuse;
        linkEmissiveWithDiffuse: boolean;
        private _useSpecularOverAlpha;
        useSpecularOverAlpha: boolean;
        private _useReflectionOverAlpha;
        useReflectionOverAlpha: boolean;
        private _disableLighting;
        disableLighting: boolean;
        private _useObjectSpaceNormalMap;
        /**
         * Allows using an object space normal map (instead of tangent space).
         */
        useObjectSpaceNormalMap: boolean;
        private _useParallax;
        useParallax: boolean;
        private _useParallaxOcclusion;
        useParallaxOcclusion: boolean;
        parallaxScaleBias: number;
        private _roughness;
        roughness: number;
        indexOfRefraction: number;
        invertRefractionY: boolean;
        private _useLightmapAsShadowmap;
        useLightmapAsShadowmap: boolean;
        private _diffuseFresnelParameters;
        diffuseFresnelParameters: FresnelParameters;
        private _opacityFresnelParameters;
        opacityFresnelParameters: FresnelParameters;
        private _reflectionFresnelParameters;
        reflectionFresnelParameters: FresnelParameters;
        private _refractionFresnelParameters;
        refractionFresnelParameters: FresnelParameters;
        private _emissiveFresnelParameters;
        emissiveFresnelParameters: FresnelParameters;
        private _useReflectionFresnelFromSpecular;
        useReflectionFresnelFromSpecular: boolean;
        private _useGlossinessFromSpecularMapAlpha;
        useGlossinessFromSpecularMapAlpha: boolean;
        private _maxSimultaneousLights;
        maxSimultaneousLights: number;
        /**
         * If sets to true, x component of normal map value will invert (x = 1.0 - x).
         */
        private _invertNormalMapX;
        invertNormalMapX: boolean;
        /**
         * If sets to true, y component of normal map value will invert (y = 1.0 - y).
         */
        private _invertNormalMapY;
        invertNormalMapY: boolean;
        /**
         * If sets to true and backfaceCulling is false, normals will be flipped on the backside.
         */
        private _twoSidedLighting;
        twoSidedLighting: boolean;
        /**
         * Default configuration related to image processing available in the standard Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the Standard Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>): void;
        /**
         * Gets wether the color curves effect is enabled.
         */
        /**
         * Sets wether the color curves effect is enabled.
         */
        cameraColorCurvesEnabled: boolean;
        /**
         * Gets wether the color grading effect is enabled.
         */
        /**
         * Gets wether the color grading effect is enabled.
         */
        cameraColorGradingEnabled: boolean;
        /**
         * Gets wether tonemapping is enabled or not.
         */
        /**
         * Sets wether tonemapping is enabled or not
         */
        cameraToneMappingEnabled: boolean;
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        /**
         * The camera exposure used on this material.
         * This property is here and not in the camera to allow controlling exposure without full screen post process.
         * This corresponds to a photographic exposure.
         */
        cameraExposure: number;
        /**
         * Gets The camera contrast used on this material.
         */
        /**
         * Sets The camera contrast used on this material.
         */
        cameraContrast: number;
        /**
         * Gets the Color Grading 2D Lookup Texture.
         */
        /**
         * Sets the Color Grading 2D Lookup Texture.
         */
        cameraColorGradingTexture: Nullable<BaseTexture>;
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        /**
         * The color grading curves provide additional color adjustmnent that is applied after any color grading transform (3D LUT).
         * They allow basic adjustment of saturation and small exposure adjustments, along with color filter tinting to provide white balance adjustment or more stylistic effects.
         * These are similar to controls found in many professional imaging or colorist software. The global controls are applied to the entire image. For advanced tuning, extra controls are provided to adjust the shadow, midtone and highlight areas of the image;
         * corresponding to low luminance, medium luminance, and high luminance areas respectively.
         */
        cameraColorCurves: Nullable<ColorCurves>;
        customShaderNameResolve: (shaderName: string, uniforms: string[], uniformBuffers: string[], samplers: string[], defines: StandardMaterialDefines) => string;
        protected _renderTargets: SmartArray<RenderTargetTexture>;
        protected _worldViewProjectionMatrix: Matrix;
        protected _globalAmbientColor: Color3;
        protected _useLogarithmicDepth: boolean;
        constructor(name: string, scene: Scene);
        getClassName(): string;
        useLogarithmicDepth: boolean;
        needAlphaBlending(): boolean;
        needAlphaTesting(): boolean;
        protected _shouldUseAlphaFromDiffuseTexture(): boolean;
        getAlphaTestTexture(): Nullable<BaseTexture>;
        /**
         * Child classes can use it to update shaders
         */
        isReadyForSubMesh(mesh: AbstractMesh, subMesh: SubMesh, useInstances?: boolean): boolean;
        buildUniformLayout(): void;
        unbind(): void;
        bindForSubMesh(world: Matrix, mesh: Mesh, subMesh: SubMesh): void;
        getAnimatables(): IAnimatable[];
        getActiveTextures(): BaseTexture[];
        hasTexture(texture: BaseTexture): boolean;
        dispose(forceDisposeEffect?: boolean, forceDisposeTextures?: boolean): void;
        clone(name: string): StandardMaterial;
        serialize(): any;
        static Parse(source: any, scene: Scene, rootUrl: string): StandardMaterial;
        static _DiffuseTextureEnabled: boolean;
        static DiffuseTextureEnabled: boolean;
        static _AmbientTextureEnabled: boolean;
        static AmbientTextureEnabled: boolean;
        static _OpacityTextureEnabled: boolean;
        static OpacityTextureEnabled: boolean;
        static _ReflectionTextureEnabled: boolean;
        static ReflectionTextureEnabled: boolean;
        static _EmissiveTextureEnabled: boolean;
        static EmissiveTextureEnabled: boolean;
        static _SpecularTextureEnabled: boolean;
        static SpecularTextureEnabled: boolean;
        static _BumpTextureEnabled: boolean;
        static BumpTextureEnabled: boolean;
        static _LightmapTextureEnabled: boolean;
        static LightmapTextureEnabled: boolean;
        static _RefractionTextureEnabled: boolean;
        static RefractionTextureEnabled: boolean;
        static _ColorGradingTextureEnabled: boolean;
        static ColorGradingTextureEnabled: boolean;
        static _FresnelEnabled: boolean;
        static FresnelEnabled: boolean;
    }
}

declare module BABYLON {
    class UniformBuffer {
        private _engine;
        private _buffer;
        private _data;
        private _bufferData;
        private _dynamic?;
        private _uniformLocations;
        private _uniformSizes;
        private _uniformLocationPointer;
        private _needSync;
        private _noUBO;
        private _currentEffect;
        private static _MAX_UNIFORM_SIZE;
        private static _tempBuffer;
        /**
         * Wrapper for updateUniform.
         * @method updateMatrix3x3
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Float32Array} matrix
         */
        updateMatrix3x3: (name: string, matrix: Float32Array) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Float32Array} matrix
         */
        updateMatrix2x2: (name: string, matrix: Float32Array) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         */
        updateFloat: (name: string, x: number) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat2: (name: string, x: number, y: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat3: (name: string, x: number, y: number, z: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         * @param {number} w
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateFloat4: (name: string, x: number, y: number, z: number, w: number, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Matrix} A 4x4 matrix.
         */
        updateMatrix: (name: string, mat: Matrix) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector3} vector
         */
        updateVector3: (name: string, vector: Vector3) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector4} vector
         */
        updateVector4: (name: string, vector: Vector4) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateColor3: (name: string, color: Color3, suffix?: string) => void;
        /**
         * Wrapper for updateUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {number} alpha
         * @param {string} [suffix] Suffix to add to the uniform name.
         */
        updateColor4: (name: string, color: Color3, alpha: number, suffix?: string) => void;
        /**
         * Uniform buffer objects.
         *
         * Handles blocks of uniform on the GPU.
         *
         * If WebGL 2 is not available, this class falls back on traditionnal setUniformXXX calls.
         *
         * For more information, please refer to :
         * https://www.khronos.org/opengl/wiki/Uniform_Buffer_Object
         */
        constructor(engine: Engine, data?: number[], dynamic?: boolean);
        /**
         * Indicates if the buffer is using the WebGL2 UBO implementation,
         * or just falling back on setUniformXXX calls.
         */
        readonly useUbo: boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is in sync
         * with the javascript cache data.
         */
        readonly isSync: boolean;
        /**
         * Indicates if the WebGL underlying uniform buffer is dynamic.
         * Also, a dynamic UniformBuffer will disable cache verification and always
         * update the underlying WebGL uniform buffer to the GPU.
         */
        isDynamic(): boolean;
        /**
         * The data cache on JS side.
         */
        getData(): Float32Array;
        /**
         * The underlying WebGL Uniform buffer.
         */
        getBuffer(): Nullable<WebGLBuffer>;
        /**
         * std140 layout specifies how to align data within an UBO structure.
         * See https://khronos.org/registry/OpenGL/specs/gl/glspec45.core.pdf#page=159
         * for specs.
         */
        private _fillAlignment(size);
        /**
         * Adds an uniform in the buffer.
         * Warning : the subsequents calls of this function must be in the same order as declared in the shader
         * for the layout to be correct !
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number|number[]} size Data size, or data directly.
         */
        addUniform(name: string, size: number | number[]): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Matrix} mat A 4x4 matrix.
         */
        addMatrix(name: string, mat: Matrix): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         */
        addFloat2(name: string, x: number, y: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {number} x
         * @param {number} y
         * @param {number} z
         */
        addFloat3(name: string, x: number, y: number, z: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         */
        addColor3(name: string, color: Color3): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Color3} color
         * @param {number} alpha
         */
        addColor4(name: string, color: Color3, alpha: number): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         * @param {Vector3} vector
         */
        addVector3(name: string, vector: Vector3): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix3x3(name: string): void;
        /**
         * Wrapper for addUniform.
         * @param {string} name Name of the uniform, as used in the uniform block in the shader.
         */
        addMatrix2x2(name: string): void;
        /**
         * Effectively creates the WebGL Uniform Buffer, once layout is completed with `addUniform`.
         */
        create(): void;
        _rebuild(): void;
        /**
         * Updates the WebGL Uniform Buffer on the GPU.
         * If the `dynamic` flag is set to true, no cache comparison is done.
         * Otherwise, the buffer will be updated only if the cache differs.
         */
        update(): void;
        /**
         * Updates the value of an uniform. The `update` method must be called afterwards to make it effective in the GPU.
         * @param {string} uniformName Name of the uniform, as used in the uniform block in the shader.
         * @param {number[]|Float32Array} data Flattened data
         * @param {number} size Size of the data.
         */
        updateUniform(uniformName: string, data: FloatArray, size: number): void;
        private _updateMatrix3x3ForUniform(name, matrix);
        private _updateMatrix3x3ForEffect(name, matrix);
        private _updateMatrix2x2ForEffect(name, matrix);
        private _updateMatrix2x2ForUniform(name, matrix);
        private _updateFloatForEffect(name, x);
        private _updateFloatForUniform(name, x);
        private _updateFloat2ForEffect(name, x, y, suffix?);
        private _updateFloat2ForUniform(name, x, y, suffix?);
        private _updateFloat3ForEffect(name, x, y, z, suffix?);
        private _updateFloat3ForUniform(name, x, y, z, suffix?);
        private _updateFloat4ForEffect(name, x, y, z, w, suffix?);
        private _updateFloat4ForUniform(name, x, y, z, w, suffix?);
        private _updateMatrixForEffect(name, mat);
        private _updateMatrixForUniform(name, mat);
        private _updateVector3ForEffect(name, vector);
        private _updateVector3ForUniform(name, vector);
        private _updateVector4ForEffect(name, vector);
        private _updateVector4ForUniform(name, vector);
        private _updateColor3ForEffect(name, color, suffix?);
        private _updateColor3ForUniform(name, color, suffix?);
        private _updateColor4ForEffect(name, color, alpha, suffix?);
        private _updateColor4ForUniform(name, color, alpha, suffix?);
        /**
         * Sets a sampler uniform on the effect.
         * @param {string} name Name of the sampler.
         * @param {Texture} texture
         */
        setTexture(name: string, texture: Nullable<BaseTexture>): void;
        /**
         * Directly updates the value of the uniform in the cache AND on the GPU.
         * @param {string} uniformName Name of the uniform, as used in the uniform block in the shader.
         * @param {number[]|Float32Array} data Flattened data
         */
        updateUniformDirectly(uniformName: string, data: FloatArray): void;
        /**
         * Binds this uniform buffer to an effect.
         * @param {Effect} effect
         * @param {string} name Name of the uniform block in the shader.
         */
        bindToEffect(effect: Effect, name: string): void;
        /**
         * Disposes the uniform buffer.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class AbstractMesh extends TransformNode implements IDisposable, ICullable, IGetSetVerticesData {
        static OCCLUSION_TYPE_NONE: number;
        static OCCLUSION_TYPE_OPTIMISTIC: number;
        static OCCLUSION_TYPE_STRICT: number;
        static OCCLUSION_ALGORITHM_TYPE_ACCURATE: number;
        static OCCLUSION_ALGORITHM_TYPE_CONSERVATIVE: number;
        static readonly BILLBOARDMODE_NONE: number;
        static readonly BILLBOARDMODE_X: number;
        static readonly BILLBOARDMODE_Y: number;
        static readonly BILLBOARDMODE_Z: number;
        static readonly BILLBOARDMODE_ALL: number;
        private _facetPositions;
        private _facetNormals;
        private _facetPartitioning;
        private _facetNb;
        private _partitioningSubdivisions;
        private _partitioningBBoxRatio;
        private _facetDataEnabled;
        private _facetParameters;
        private _bbSize;
        private _subDiv;
        private _facetDepthSort;
        private _facetDepthSortEnabled;
        private _depthSortedIndices;
        private _depthSortedFacets;
        private _facetDepthSortFunction;
        private _facetDepthSortFrom;
        private _facetDepthSortOrigin;
        private _invertedMatrix;
        /**
         * Read-only : the number of facets in the mesh
         */
        readonly facetNb: number;
        /**
         * The number (integer) of subdivisions per axis in the partioning space
         */
        partitioningSubdivisions: number;
        /**
         * The ratio (float) to apply to the bouding box size to set to the partioning space.
         * Ex : 1.01 (default) the partioning space is 1% bigger than the bounding box.
         */
        partitioningBBoxRatio: number;
        /**
         * Boolean : must the facet be depth sorted on next call to `updateFacetData()` ?
         * Works only for updatable meshes.
         * Doesn't work with multi-materials.
         */
        mustDepthSortFacets: boolean;
        /**
         * The location (Vector3) where the facet depth sort must be computed from.
         * By default, the active camera position.
         * Used only when facet depth sort is enabled.
         */
        facetDepthSortFrom: Vector3;
        /**
         * Read-only boolean : is the feature facetData enabled ?
         */
        readonly isFacetDataEnabled: boolean;
        _updateNonUniformScalingState(value: boolean): boolean;
        /**
        * An event triggered when this mesh collides with another one
        */
        onCollideObservable: Observable<AbstractMesh>;
        private _onCollideObserver;
        onCollide: () => void;
        /**
        * An event triggered when the collision's position changes
        */
        onCollisionPositionChangeObservable: Observable<Vector3>;
        private _onCollisionPositionChangeObserver;
        onCollisionPositionChange: () => void;
        /**
        * An event triggered when material is changed
        */
        onMaterialChangedObservable: Observable<AbstractMesh>;
        definedFacingForward: boolean;
        /**
        * This property determines the type of occlusion query algorithm to run in WebGl, you can use:

        * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_ACCURATE which is mapped to GL_ANY_SAMPLES_PASSED.

        * or

        * AbstractMesh.OCCLUSION_ALGORITHM_TYPE_CONSERVATIVE (Default Value) which is mapped to GL_ANY_SAMPLES_PASSED_CONSERVATIVE which is a false positive algorithm that is faster than GL_ANY_SAMPLES_PASSED but less accurate.

        * for more info check WebGl documentations
        */
        occlusionQueryAlgorithmType: number;
        /**
         * This property is responsible for starting the occlusion query within the Mesh or not, this property is also used     to determine what should happen when the occlusionRetryCount is reached. It has supports 3 values:

        * OCCLUSION_TYPE_NONE (Default Value): this option means no occlusion query whith the Mesh.

        * OCCLUSION_TYPE_OPTIMISTIC: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken show the mesh.

            * OCCLUSION_TYPE_STRICT: this option is means use occlusion query and if occlusionRetryCount is reached and the query is broken restore the last state of the mesh occlusion if the mesh was visible then show the mesh if was hidden then hide don't show.
         */
        occlusionType: number;
        /**
        * This number indicates the number of allowed retries before stop the occlusion query, this is useful if the        occlusion query is taking long time before to the query result is retireved, the query result indicates if the object is visible within the scene or not and based on that Babylon.Js engine decideds to show or hide the object.

        * The default value is -1 which means don't break the query and wait till the result.
        */
        occlusionRetryCount: number;
        private _occlusionInternalRetryCounter;
        protected _isOccluded: boolean;
        /**
        * Property isOccluded : Gets or sets whether the mesh is occluded or not, it is used also to set the intial state of the mesh to be occluded or not.
        */
        isOccluded: boolean;
        private _isOcclusionQueryInProgress;
        /**
        * Flag to check the progress status of the query
        */
        readonly isOcclusionQueryInProgress: boolean;
        private _occlusionQuery;
        private _visibility;
        /**
         * Gets or sets mesh visibility between 0 and 1 (defult is 1)
         */
        /**
         * Gets or sets mesh visibility between 0 and 1 (defult is 1)
         */
        visibility: number;
        alphaIndex: number;
        isVisible: boolean;
        isPickable: boolean;
        showBoundingBox: boolean;
        showSubMeshesBoundingBox: boolean;
        isBlocker: boolean;
        enablePointerMoveEvents: boolean;
        renderingGroupId: number;
        private _material;
        material: Nullable<Material>;
        private _receiveShadows;
        receiveShadows: boolean;
        renderOutline: boolean;
        outlineColor: Color3;
        outlineWidth: number;
        renderOverlay: boolean;
        overlayColor: Color3;
        overlayAlpha: number;
        private _hasVertexAlpha;
        hasVertexAlpha: boolean;
        private _useVertexColors;
        useVertexColors: boolean;
        private _computeBonesUsingShaders;
        computeBonesUsingShaders: boolean;
        private _numBoneInfluencers;
        numBoneInfluencers: number;
        private _applyFog;
        applyFog: boolean;
        useOctreeForRenderingSelection: boolean;
        useOctreeForPicking: boolean;
        useOctreeForCollisions: boolean;
        private _layerMask;
        layerMask: number;
        /**
         * True if the mesh must be rendered in any case.
         */
        alwaysSelectAsActiveMesh: boolean;
        /**
         * This scene's action manager
        */
        actionManager: Nullable<ActionManager>;
        physicsImpostor: Nullable<PhysicsImpostor>;
        private _checkCollisions;
        private _collisionMask;
        private _collisionGroup;
        ellipsoid: Vector3;
        ellipsoidOffset: Vector3;
        private _collider;
        private _oldPositionForCollisions;
        private _diffPositionForCollisions;
        collisionMask: number;
        collisionGroup: number;
        edgesWidth: number;
        edgesColor: Color4;
        _edgesRenderer: Nullable<EdgesRenderer>;
        private _collisionsTransformMatrix;
        private _collisionsScalingMatrix;
        _masterMesh: Nullable<AbstractMesh>;
        _boundingInfo: Nullable<BoundingInfo>;
        _renderId: number;
        subMeshes: SubMesh[];
        _submeshesOctree: Octree<SubMesh>;
        _intersectionsInProgress: AbstractMesh[];
        _unIndexed: boolean;
        _lightSources: Light[];
        readonly _positions: Nullable<Vector3[]>;
        _waitingActions: any;
        _waitingFreezeWorldMatrix: Nullable<boolean>;
        private _skeleton;
        _bonesTransformMatrices: Nullable<Float32Array>;
        skeleton: Nullable<Skeleton>;
        constructor(name: string, scene?: Nullable<Scene>);
        /**
         * Returns the string "AbstractMesh"
         */
        getClassName(): string;
        /**
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        _rebuild(): void;
        _resyncLightSources(): void;
        _resyncLighSource(light: Light): void;
        _removeLightSource(light: Light): void;
        private _markSubMeshesAsDirty(func);
        _markSubMeshesAsLightDirty(): void;
        _markSubMeshesAsAttributesDirty(): void;
        _markSubMeshesAsMiscDirty(): void;
        /**
        * Scaling property : a Vector3 depicting the mesh scaling along each local axis X, Y, Z.
        * Default : (1.0, 1.0, 1.0)
        */
        /**
         * Scaling property : a Vector3 depicting the mesh scaling along each local axis X, Y, Z.
         * Default : (1.0, 1.0, 1.0)
         */
        scaling: Vector3;
        /**
         * Disables the mesh edger rendering mode.
         * Returns the AbstractMesh.
         */
        disableEdgesRendering(): AbstractMesh;
        /**
         * Enables the edge rendering mode on the mesh.
         * This mode makes the mesh edges visible.
         * Returns the AbstractMesh.
         */
        enableEdgesRendering(epsilon?: number, checkVerticesInsteadOfIndices?: boolean): AbstractMesh;
        /**
         * Returns true if the mesh is blocked. Used by the class Mesh.
         * Returns the boolean `false` by default.
         */
        readonly isBlocked: boolean;
        /**
         * Returns the mesh itself by default, used by the class Mesh.
         * Returned type : AbstractMesh
         */
        getLOD(camera: Camera): AbstractMesh;
        /**
         * Returns 0 by default, used by the class Mesh.
         * Returns an integer.
         */
        getTotalVertices(): number;
        /**
         * Returns null by default, used by the class Mesh.
         * Returned type : integer array
         */
        getIndices(): Nullable<IndicesArray>;
        /**
         * Returns the array of the requested vertex data kind. Used by the class Mesh. Returns null here.
         * Returned type : float array or Float32Array
         */
        getVerticesData(kind: string): Nullable<FloatArray>;
        /**
         * Sets the vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, a new Geometry object is set to the mesh and then passed this vertex data.
         * The `data` are either a numeric array either a Float32Array.
         * The parameter `updatable` is passed as is to the underlying Geometry object constructor (if initianilly none) or updater.
         * The parameter `stride` is an optional positive integer, it is usually automatically deducted from the `kind` (3 for positions or normals, 2 for UV, etc).
         * Note that a new underlying VertexBuffer object is created each call.
         * If the `kind` is the `PositionKind`, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): AbstractMesh;
        /**
         * Updates the existing vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, it is simply returned as it is.
         * The `data` are either a numeric array either a Float32Array.
         * No new underlying VertexBuffer object is created.
         * If the `kind` is the `PositionKind` and if `updateExtends` is true, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         * If the parameter `makeItUnique` is true, a new global geometry is created from this positions and is set to the mesh.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): AbstractMesh;
        /**
         * Sets the mesh indices.
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array).
         * If the mesh has no geometry, a new Geometry object is created and set to the mesh.
         * This method creates a new index buffer each call.
         * Returns the Mesh.
         */
        setIndices(indices: IndicesArray, totalVertices: Nullable<number>): AbstractMesh;
        /** Returns false by default, used by the class Mesh.
         *  Returns a boolean
        */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns the mesh BoundingInfo object or creates a new one and returns it if undefined.
         * Returns a BoundingInfo
         */
        getBoundingInfo(): BoundingInfo;
        /**
         * Uniformly scales the mesh to fit inside of a unit cube (1 X 1 X 1 units).
         * @param includeDescendants Take the hierarchy's bounding box instead of the mesh's bounding box.
         */
        normalizeToUnitCube(includeDescendants?: boolean): AbstractMesh;
        /**
         * Sets a mesh new object BoundingInfo.
         * Returns the AbstractMesh.
         */
        setBoundingInfo(boundingInfo: BoundingInfo): AbstractMesh;
        readonly useBones: boolean;
        _preActivate(): void;
        _preActivateForIntermediateRendering(renderId: number): void;
        _activate(renderId: number): void;
        /**
         * Returns the latest update of the World matrix
         * Returns a Matrix.
         */
        getWorldMatrix(): Matrix;
        /**
         * Returns the latest update of the World matrix determinant.
         */
        protected _getWorldMatrixDeterminant(): number;
        /**
         * Perform relative position change from the point of view of behind the front of the mesh.
         * This is performed taking into account the meshes current rotation, so you do not have to care.
         * Supports definition of mesh facing forward or backward.
         * @param {number} amountRight
         * @param {number} amountUp
         * @param {number} amountForward
         *
         * Returns the AbstractMesh.
         */
        movePOV(amountRight: number, amountUp: number, amountForward: number): AbstractMesh;
        /**
         * Calculate relative position change from the point of view of behind the front of the mesh.
         * This is performed taking into account the meshes current rotation, so you do not have to care.
         * Supports definition of mesh facing forward or backward.
         * @param {number} amountRight
         * @param {number} amountUp
         * @param {number} amountForward
         *
         * Returns a new Vector3.
         */
        calcMovePOV(amountRight: number, amountUp: number, amountForward: number): Vector3;
        /**
         * Perform relative rotation change from the point of view of behind the front of the mesh.
         * Supports definition of mesh facing forward or backward.
         * @param {number} flipBack
         * @param {number} twirlClockwise
         * @param {number} tiltRight
         *
         * Returns the AbstractMesh.
         */
        rotatePOV(flipBack: number, twirlClockwise: number, tiltRight: number): AbstractMesh;
        /**
         * Calculate relative rotation change from the point of view of behind the front of the mesh.
         * Supports definition of mesh facing forward or backward.
         * @param {number} flipBack
         * @param {number} twirlClockwise
         * @param {number} tiltRight
         *
         * Returns a new Vector3.
         */
        calcRotatePOV(flipBack: number, twirlClockwise: number, tiltRight: number): Vector3;
        /**
         * Return the minimum and maximum world vectors of the entire hierarchy under current mesh
         * @param includeDescendants Include bounding info from descendants as well (true by default).
         */
        getHierarchyBoundingVectors(includeDescendants?: boolean): {
            min: Vector3;
            max: Vector3;
        };
        /**
         * Updates the mesh BoundingInfo object and all its children BoundingInfo objects also.
         * Returns the AbstractMesh.
         */
        _updateBoundingInfo(): AbstractMesh;
        /**
         * Update a mesh's children BoundingInfo objects only.
         * Returns the AbstractMesh.
         */
        _updateSubMeshesBoundingInfo(matrix: Matrix): AbstractMesh;
        protected _afterComputeWorldMatrix(): void;
        /**
         * Returns `true` if the mesh is within the frustum defined by the passed array of planes.
         * A mesh is in the frustum if its bounding box intersects the frustum.
         * Boolean returned.
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Returns `true` if the mesh is completely in the frustum defined be the passed array of planes.
         * A mesh is completely in the frustum if its bounding box it completely inside the frustum.
         * Boolean returned.
         */
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * True if the mesh intersects another mesh or a SolidParticle object.
         * Unless the parameter `precise` is set to `true` the intersection is computed according to Axis Aligned Bounding Boxes (AABB), else according to OBB (Oriented BBoxes)
         * includeDescendants can be set to true to test if the mesh defined in parameters intersects with the current mesh or any child meshes
         * Returns a boolean.
         */
        intersectsMesh(mesh: AbstractMesh | SolidParticle, precise?: boolean, includeDescendants?: boolean): boolean;
        /**
         * Returns true if the passed point (Vector3) is inside the mesh bounding box.
         * Returns a boolean.
         */
        intersectsPoint(point: Vector3): boolean;
        getPhysicsImpostor(): Nullable<PhysicsImpostor>;
        getPositionInCameraSpace(camera?: Nullable<Camera>): Vector3;
        /**
         * Returns the distance from the mesh to the active camera.
         * Returns a float.
         */
        getDistanceToCamera(camera?: Nullable<Camera>): number;
        applyImpulse(force: Vector3, contactPoint: Vector3): AbstractMesh;
        setPhysicsLinkWith(otherMesh: Mesh, pivot1: Vector3, pivot2: Vector3, options?: any): AbstractMesh;
        /**
         * Property checkCollisions : Boolean, whether the camera should check the collisions against the mesh.
         * Default `false`.
         */
        checkCollisions: boolean;
        /**
         * Gets Collider object used to compute collisions (not physics)
         */
        readonly collider: Collider;
        moveWithCollisions(displacement: Vector3): AbstractMesh;
        private _onCollisionPositionChange;
        /**
        * This function will create an octree to help to select the right submeshes for rendering, picking and collision computations.
        * Please note that you must have a decent number of submeshes to get performance improvements when using an octree.
        * Returns an Octree of submeshes.
        */
        createOrUpdateSubmeshesOctree(maxCapacity?: number, maxDepth?: number): Octree<SubMesh>;
        _collideForSubMesh(subMesh: SubMesh, transformMatrix: Matrix, collider: Collider): AbstractMesh;
        _processCollisionsForSubMeshes(collider: Collider, transformMatrix: Matrix): AbstractMesh;
        _checkCollision(collider: Collider): AbstractMesh;
        _generatePointsArray(): boolean;
        /**
         * Checks if the passed Ray intersects with the mesh.
         * Returns an object PickingInfo.
         */
        intersects(ray: Ray, fastCheck?: boolean): PickingInfo;
        /**
         * Clones the mesh, used by the class Mesh.
         * Just returns `null` for an AbstractMesh.
         */
        clone(name: string, newParent: Node, doNotCloneChildren?: boolean): Nullable<AbstractMesh>;
        /**
         * Disposes all the mesh submeshes.
         * Returns the AbstractMesh.
         */
        releaseSubMeshes(): AbstractMesh;
        /**
         * Releases resources associated with this abstract mesh.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Adds the passed mesh as a child to the current mesh.
         * Returns the AbstractMesh.
         */
        addChild(mesh: AbstractMesh): AbstractMesh;
        /**
         * Removes the passed mesh from the current mesh children list.
         * Returns the AbstractMesh.
         */
        removeChild(mesh: AbstractMesh): AbstractMesh;
        /**
         *  Initialize the facet data arrays : facetNormals, facetPositions and facetPartitioning.
         * Returns the AbstractMesh.
         */
        private _initFacetData();
        /**
         * Updates the mesh facetData arrays and the internal partitioning when the mesh is morphed or updated.
         * This method can be called within the render loop.
         * You don't need to call this method by yourself in the render loop when you update/morph a mesh with the methods CreateXXX() as they automatically manage this computation.
         * Returns the AbstractMesh.
         */
        updateFacetData(): AbstractMesh;
        /**
         * Returns the facetLocalNormals array.
         * The normals are expressed in the mesh local space.
         */
        getFacetLocalNormals(): Vector3[];
        /**
         * Returns the facetLocalPositions array.
         * The facet positions are expressed in the mesh local space.
         */
        getFacetLocalPositions(): Vector3[];
        /**
         * Returns the facetLocalPartioning array.
         */
        getFacetLocalPartitioning(): number[][];
        /**
         * Returns the i-th facet position in the world system.
         * This method allocates a new Vector3 per call.
         */
        getFacetPosition(i: number): Vector3;
        /**
         * Sets the reference Vector3 with the i-th facet position in the world system.
         * Returns the AbstractMesh.
         */
        getFacetPositionToRef(i: number, ref: Vector3): AbstractMesh;
        /**
         * Returns the i-th facet normal in the world system.
         * This method allocates a new Vector3 per call.
         */
        getFacetNormal(i: number): Vector3;
        /**
         * Sets the reference Vector3 with the i-th facet normal in the world system.
         * Returns the AbstractMesh.
         */
        getFacetNormalToRef(i: number, ref: Vector3): this;
        /**
         * Returns the facets (in an array) in the same partitioning block than the one the passed coordinates are located (expressed in the mesh local system).
         */
        getFacetsAtLocalCoordinates(x: number, y: number, z: number): Nullable<number[]>;
        /**
         * Returns the closest mesh facet index at (x,y,z) World coordinates, null if not found.
         * If the parameter projected (vector3) is passed, it is set as the (x,y,z) World projection on the facet.
         * If checkFace is true (default false), only the facet "facing" to (x,y,z) or only the ones "turning their backs", according to the parameter "facing" are returned.
         * If facing and checkFace are true, only the facet "facing" to (x, y, z) are returned : positive dot (x, y, z) * facet position.
         * If facing si false and checkFace is true, only the facet "turning their backs" to (x, y, z) are returned : negative dot (x, y, z) * facet position.
         */
        getClosestFacetAtCoordinates(x: number, y: number, z: number, projected?: Vector3, checkFace?: boolean, facing?: boolean): Nullable<number>;
        /**
         * Returns the closest mesh facet index at (x,y,z) local coordinates, null if not found.
         * If the parameter projected (vector3) is passed, it is set as the (x,y,z) local projection on the facet.
         * If checkFace is true (default false), only the facet "facing" to (x,y,z) or only the ones "turning their backs", according to the parameter "facing" are returned.
         * If facing and checkFace are true, only the facet "facing" to (x, y, z) are returned : positive dot (x, y, z) * facet position.
         * If facing si false and checkFace is true, only the facet "turning their backs"  to (x, y, z) are returned : negative dot (x, y, z) * facet position.
         */
        getClosestFacetAtLocalCoordinates(x: number, y: number, z: number, projected?: Vector3, checkFace?: boolean, facing?: boolean): Nullable<number>;
        /**
         * Returns the object "parameter" set with all the expected parameters for facetData computation by ComputeNormals()
         */
        getFacetDataParameters(): any;
        /**
         * Disables the feature FacetData and frees the related memory.
         * Returns the AbstractMesh.
         */
        disableFacetData(): AbstractMesh;
        /**
         * Updates the AbstractMesh indices array. Actually, used by the Mesh object.
         * Returns the mesh.
         */
        updateIndices(indices: IndicesArray): AbstractMesh;
        /**
         * The mesh Geometry. Actually used by the Mesh object.
         * Returns a blank geometry object.
         */
        /**
         * Creates new normals data for the mesh.
         * @param updatable.
         */
        createNormals(updatable: boolean): void;
        /**
         * Align the mesh with a normal.
         * Returns the mesh.
         */
        alignWithNormal(normal: Vector3, upDirection?: Vector3): AbstractMesh;
        protected checkOcclusionQuery(): void;
    }
}

declare module BABYLON {
    class Buffer {
        private _engine;
        private _buffer;
        private _data;
        private _updatable;
        private _strideSize;
        private _instanced;
        constructor(engine: any, data: FloatArray, updatable: boolean, stride: number, postponeInternalCreation?: boolean, instanced?: boolean);
        /**
         * Create a new {BABYLON.VertexBuffer} based on the current buffer
         * @param kind defines the vertex buffer kind (position, normal, etc.)
         * @param offset defines offset in the buffer (0 by default)
         * @param size defines the size in floats of attributes (position is 3 for instance)
         * @param stride defines the stride size in floats in the buffer (the offset to apply to reach next value when data is interleaved)
         * @param instanced defines if the vertex buffer contains indexed data
         * @returns the new vertex buffer
         */
        createVertexBuffer(kind: string, offset: number, size: number, stride?: number, instanced?: boolean): VertexBuffer;
        isUpdatable(): boolean;
        getData(): Nullable<FloatArray>;
        getBuffer(): Nullable<WebGLBuffer>;
        getStrideSize(): number;
        create(data?: Nullable<FloatArray>): void;
        _rebuild(): void;
        update(data: FloatArray): void;
        updateDirectly(data: Float32Array, offset: number, vertexCount?: number): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class CSG {
        private polygons;
        matrix: Matrix;
        position: Vector3;
        rotation: Vector3;
        rotationQuaternion: Nullable<Quaternion>;
        scaling: Vector3;
        static FromMesh(mesh: Mesh): CSG;
        private static FromPolygons(polygons);
        clone(): CSG;
        union(csg: CSG): CSG;
        unionInPlace(csg: CSG): void;
        subtract(csg: CSG): CSG;
        subtractInPlace(csg: CSG): void;
        intersect(csg: CSG): CSG;
        intersectInPlace(csg: CSG): void;
        inverse(): CSG;
        inverseInPlace(): void;
        copyTransformAttributes(csg: CSG): CSG;
        buildMeshGeometry(name: string, scene: Scene, keepSubMeshes: boolean): Mesh;
        toMesh(name: string, material: Nullable<Material>, scene: Scene, keepSubMeshes: boolean): Mesh;
    }
}

declare module BABYLON {
    /**
     * Class used to store geometry data (vertex buffers + index buffer)
     */
    class Geometry implements IGetSetVerticesData {
        /**
         * Gets or sets the unique ID of the geometry
         */
        id: string;
        /**
         * Gets the delay loading state of the geometry (none by default which means not delayed)
         */
        delayLoadState: number;
        /**
         * Gets the file containing the data to load when running in delay load state
         */
        delayLoadingFile: Nullable<string>;
        /**
         * Callback called when the geometry is updated
         */
        onGeometryUpdated: (geometry: Geometry, kind?: string) => void;
        private _scene;
        private _engine;
        private _meshes;
        private _totalVertices;
        private _indices;
        private _vertexBuffers;
        private _isDisposed;
        private _extend;
        private _boundingBias;
        /** @ignore */
        _delayInfo: Array<string>;
        private _indexBuffer;
        private _indexBufferIsUpdatable;
        /** @ignore */
        _boundingInfo: Nullable<BoundingInfo>;
        /** @ignore */
        _delayLoadingFunction: Nullable<(any: any, geometry: Geometry) => void>;
        /** @ignore */
        _softwareSkinningRenderId: number;
        private _vertexArrayObjects;
        private _updatable;
        /** @ignore */
        _positions: Nullable<Vector3[]>;
        /**
         *  Gets or sets the Bias Vector to apply on the bounding elements (box/sphere), the max extend is computed as v += v * bias.x + bias.y, the min is computed as v -= v * bias.x + bias.y
         */
        /**
         *  Gets or sets the Bias Vector to apply on the bounding elements (box/sphere), the max extend is computed as v += v * bias.x + bias.y, the min is computed as v -= v * bias.x + bias.y
         */
        boundingBias: Vector2;
        /**
         * Static function used to attach a new empty geometry to a mesh
         * @param mesh defines the mesh to attach the geometry to
         * @returns the new {BABYLON.Geometry}
         */
        static CreateGeometryForMesh(mesh: Mesh): Geometry;
        /**
         * Creates a new geometry
         * @param id defines the unique ID
         * @param scene defines the hosting scene
         * @param vertexData defines the {BABYLON.VertexData} used to get geometry data
         * @param updatable defines if geometry must be updatable (false by default)
         * @param mesh defines the mesh that will be associated with the geometry
         */
        constructor(id: string, scene: Scene, vertexData?: VertexData, updatable?: boolean, mesh?: Nullable<Mesh>);
        /**
         * Gets the current extend of the geometry
         */
        readonly extend: {
            minimum: Vector3;
            maximum: Vector3;
        };
        /**
         * Gets the hosting scene
         * @returns the hosting {BABYLON.Scene}
         */
        getScene(): Scene;
        /**
         * Gets the hosting engine
         * @returns the hosting {BABYLON.Engine}
         */
        getEngine(): Engine;
        /**
         * Defines if the geometry is ready to use
         * @returns true if the geometry is ready to be used
         */
        isReady(): boolean;
        /**
         * Gets a value indicating that the geometry should not be serialized
         */
        readonly doNotSerialize: boolean;
        /** @ignore */
        _rebuild(): void;
        /**
         * Affects all gemetry data in one call
         * @param vertexData defines the geometry data
         * @param updatable defines if the geometry must be flagged as updatable (false as default)
         */
        setAllVerticesData(vertexData: VertexData, updatable?: boolean): void;
        /**
         * Set specific vertex data
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the vertex data to use
         * @param updatable defines if the vertex must be flagged as updatable (false as default)
         * @param stride defines the stride to use (0 by default). This value is deduced from the kind value if not specified
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): void;
        /**
         * Removes a specific vertex data
         * @param kind defines the data kind (Position, normal, etc...)
         */
        removeVerticesData(kind: string): void;
        /**
         * Affect a vertex buffer to the geometry. the vertexBuffer.getKind() function is used to determine where to store the data
         * @param buffer defines the vertex buffer to use
         */
        setVerticesBuffer(buffer: VertexBuffer): void;
        /**
         * Update a specific vertex buffer
         * This function will directly update the underlying WebGLBuffer according to the passed numeric array or Float32Array
         * It will do nothing if the buffer is not updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the data to use
         * @param offset defines the offset in the target buffer where to store the data
         */
        updateVerticesDataDirectly(kind: string, data: Float32Array, offset: number): void;
        /**
         * Update a specific vertex buffer
         * This function will create a new buffer if the current one is not updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @param data defines the data to use
         * @param updateExtends defines if the geometry extends must be recomputed (false by default)
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean): void;
        private updateBoundingInfo(updateExtends, data);
        /** @ignore */
        _bind(effect: Nullable<Effect>, indexToBind?: Nullable<WebGLBuffer>): void;
        /**
         * Gets total number of vertices
         * @returns the total number of vertices
         */
        getTotalVertices(): number;
        /**
         * Gets a specific vertex data attached to this geometry
         * @param kind defines the data kind (Position, normal, etc...)
         * @param copyWhenShared defines if the returned array must be cloned upon returning it if the current geometry is shared between multiple meshes
         * @param forceCopy defines a boolean indicating that the returned array must be cloned upon returning it
         * @returns a float array containing vertex data
         */
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        /**
         * Returns a boolean defining if the vertex data for the requested `kind` is updatable
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns true if the vertex buffer with the specified kind is updatable
         */
        isVertexBufferUpdatable(kind: string): boolean;
        /**
         * Gets a specific vertex buffer
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns a {BABYLON.VertexBuffer}
         */
        getVertexBuffer(kind: string): Nullable<VertexBuffer>;
        /**
         * Returns all vertex buffers
         * @return an object holding all vertex buffers indexed by kind
         */
        getVertexBuffers(): Nullable<{
            [key: string]: VertexBuffer;
        }>;
        /**
         * Gets a boolean indicating if specific vertex buffer is present
         * @param kind defines the data kind (Position, normal, etc...)
         * @returns true if data is present
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Gets a list of all attached data kinds (Position, normal, etc...)
         * @returns a list of string containing all kinds
         */
        getVerticesDataKinds(): string[];
        /**
         * Update index buffer
         * @param indices defines the indices to store in the index buffer
         * @param offset defines the offset in the target buffer where to store the data
         */
        updateIndices(indices: IndicesArray, offset?: number): void;
        /**
         * Creates a new index buffer
         * @param indices defines the indices to store in the index buffer
         * @param totalVertices defines the total number of vertices (could be null)
         * @param updatable defines if the index buffer must be flagged as updatable (false by default)
         */
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>, updatable?: boolean): void;
        /**
         * Return the total number of indices
         * @returns the total number of indices
         */
        getTotalIndices(): number;
        /**
         * Gets the index buffer array
         * @param copyWhenShared defines if the returned array must be cloned upon returning it if the current geometry is shared between multiple meshes
         * @returns the index buffer array
         */
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        /**
         * Gets the index buffer
         * @return the index buffer
         */
        getIndexBuffer(): Nullable<WebGLBuffer>;
        /** @ignore */
        _releaseVertexArrayObject(effect?: Nullable<Effect>): void;
        /**
         * Release the associated resources for a specific mesh
         * @param mesh defines the source mesh
         * @param shouldDispose defines if the geometry must be disposed if there is no more mesh pointing to it
         */
        releaseForMesh(mesh: Mesh, shouldDispose?: boolean): void;
        /**
         * Apply current geometry to a given mesh
         * @param mesh defines the mesh to apply geometry to
         */
        applyToMesh(mesh: Mesh): void;
        private updateExtend(data?, stride?);
        private _applyToMesh(mesh);
        private notifyUpdate(kind?);
        /**
         * Load the geometry if it was flagged as delay loaded
         * @param scene defines the hosting scene
         * @param onLoaded defines a callback called when the geometry is loaded
         */
        load(scene: Scene, onLoaded?: () => void): void;
        private _queueLoad(scene, onLoaded?);
        /**
         * Invert the geometry to move from a right handed system to a left handed one.
         */
        toLeftHanded(): void;
        /** @ignore */
        _resetPointsArrayCache(): void;
        /** @ignore */
        _generatePointsArray(): boolean;
        /**
         * Gets a value indicating if the geometry is disposed
         * @returns true if the geometry was disposed
         */
        isDisposed(): boolean;
        private _disposeVertexArrayObjects();
        /**
         * Free all associated resources
         */
        dispose(): void;
        /**
         * Clone the current geometry into a new geometry
         * @param id defines the unique ID of the new geometry
         * @returns a new geometry object
         */
        copy(id: string): Geometry;
        /**
         * Serialize the current geometry info (and not the vertices data) into a JSON object
         * @return a JSON representation of the current geometry data (without the vertices data)
         */
        serialize(): any;
        private toNumberArray(origin);
        /**
         * Serialize all vertices data into a JSON oject
         * @returns a JSON representation of the current geometry data
         */
        serializeVerticeData(): any;
        /**
         * Extracts a clone of a mesh geometry
         * @param mesh defines the source mesh
         * @param id defines the unique ID of the new geometry object
         * @returns the new geometry object
         */
        static ExtractFromMesh(mesh: Mesh, id: string): Nullable<Geometry>;
        /**
         * You should now use Tools.RandomId(), this method is still here for legacy reasons.
         * Implementation from http://stackoverflow.com/questions/105034/how-to-create-a-guid-uuid-in-javascript/2117523#answer-2117523
         * Be aware Math.random() could cause collisions, but:
         * "All but 6 of the 128 bits of the ID are randomly generated, which means that for any two ids, there's a 1 in 2^^122 (or 5.3x10^^36) chance they'll collide"
         * @returns a string containing a new GUID
         */
        static RandomId(): string;
        /** @ignore */
        static _ImportGeometry(parsedGeometry: any, mesh: Mesh): void;
        private static _CleanMatricesWeights(parsedGeometry, mesh);
        /**
         * Create a new geometry from persisted data (Using .babylon file format)
         * @param parsedVertexData defines the persisted data
         * @param scene defines the hosting scene
         * @param rootUrl defines the root url to use to load assets (like delayed data)
         * @returns the new geometry object
         */
        static Parse(parsedVertexData: any, scene: Scene, rootUrl: string): Nullable<Geometry>;
    }
    /**
     * Abstract class used to provide common services for all typed geometries
     */
    class _PrimitiveGeometry extends Geometry {
        private _canBeRegenerated;
        private _beingRegenerated;
        /**
         * Creates a new typed geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param _canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, _canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        /**
         * Gets a value indicating if the geometry supports being regenerated with new parameters (false by default)
         * @returns true if the geometry can be regenerated
         */
        canBeRegenerated(): boolean;
        /**
         * If the geometry supports regeneration, the function will recreates the geometry with updated parameter values
         */
        regenerate(): void;
        /**
         * Clone the geometry
         * @param id defines the unique ID of the new geometry
         * @returns the new geometry
         */
        asNewGeometry(id: string): Geometry;
        setAllVerticesData(vertexData: VertexData, updatable?: boolean): void;
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean): void;
        /** @ignore */
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
    }
    /**
     * Creates a ribbon geometry
     * @description See http://doc.babylonjs.com/how_to/ribbon_tutorial, http://doc.babylonjs.com/resources/maths_make_ribbons
     */
    class RibbonGeometry extends _PrimitiveGeometry {
        /**
         * Defines the array of paths to use
         */
        pathArray: Vector3[][];
        /**
         * Defines if the last and first points of each path in your pathArray must be joined
         */
        closeArray: boolean;
        /**
         * Defines if the last and first points of each path in your pathArray must be joined
         */
        closePath: boolean;
        /**
         * Defines the offset between points
         */
        offset: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a ribbon geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param pathArray defines the array of paths to use
         * @param closeArray defines if the last path and the first path must be  joined
         * @param closePath defines if the last and first points of each path in your pathArray must be joined
         * @param offset defines the offset between points
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the array of paths to use
             */
            pathArray: Vector3[][], 
            /**
             * Defines if the last and first points of each path in your pathArray must be joined
             */
            closeArray: boolean, 
            /**
             * Defines if the last and first points of each path in your pathArray must be joined
             */
            closePath: boolean, 
            /**
             * Defines the offset between points
             */
            offset: number, canBeRegenerated?: boolean, mesh?: Mesh, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        /** @ignore */
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a box geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#box
     */
    class BoxGeometry extends _PrimitiveGeometry {
        /**
         * Defines the zise of the box (width, height and depth are the same)
         */
        size: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a box geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param size defines the zise of the box (width, height and depth are the same)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the zise of the box (width, height and depth are the same)
             */
            size: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedBox: any, scene: Scene): Nullable<BoxGeometry>;
    }
    /**
     * Creates a sphere geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#sphere
     */
    class SphereGeometry extends _PrimitiveGeometry {
        /**
         * Defines the number of segments to use to create the sphere
         */
        segments: number;
        /**
         * Defines the diameter of the sphere
         */
        diameter: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Create a new sphere geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param segments defines the number of segments to use to create the sphere
         * @param diameter defines the diameter of the sphere
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the number of segments to use to create the sphere
             */
            segments: number, 
            /**
             * Defines the diameter of the sphere
             */
            diameter: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedSphere: any, scene: Scene): Nullable<SphereGeometry>;
    }
    /**
     * Creates a disc geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#disc-or-regular-polygon
     */
    class DiscGeometry extends _PrimitiveGeometry {
        /**
         * Defines the radius of the disc
         */
        radius: number;
        /**
         * Defines the tesselation factor to apply to the disc
         */
        tessellation: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new disc geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param radius defines the radius of the disc
         * @param tessellation defines the tesselation factor to apply to the disc
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the radius of the disc
             */
            radius: number, 
            /**
             * Defines the tesselation factor to apply to the disc
             */
            tessellation: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a new cylinder geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#cylinder-or-cone
     */
    class CylinderGeometry extends _PrimitiveGeometry {
        /**
         * Defines the height of the cylinder
         */
        height: number;
        /**
         * Defines the diameter of the cylinder's top cap
         */
        diameterTop: number;
        /**
         * Defines the diameter of the cylinder's bottom cap
         */
        diameterBottom: number;
        /**
         * Defines the tessellation factor to apply to the cylinder
         */
        tessellation: number;
        /**
         * Defines the number of subdivisions to apply to the cylinder (1 by default)
         */
        subdivisions: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new cylinder geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param height defines the height of the cylinder
         * @param diameterTop defines the diameter of the cylinder's top cap
         * @param diameterBottom defines the diameter of the cylinder's bottom cap
         * @param tessellation defines the tessellation factor to apply to the cylinder (number of radial sides)
         * @param subdivisions defines the number of subdivisions to apply to the cylinder (number of rings) (1 by default)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the height of the cylinder
             */
            height: number, 
            /**
             * Defines the diameter of the cylinder's top cap
             */
            diameterTop: number, 
            /**
             * Defines the diameter of the cylinder's bottom cap
             */
            diameterBottom: number, 
            /**
             * Defines the tessellation factor to apply to the cylinder
             */
            tessellation: number, 
            /**
             * Defines the number of subdivisions to apply to the cylinder (1 by default)
             */
            subdivisions?: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedCylinder: any, scene: Scene): Nullable<CylinderGeometry>;
    }
    /**
     * Creates a new torus geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#torus
     */
    class TorusGeometry extends _PrimitiveGeometry {
        /**
         * Defines the diameter of the torus
         */
        diameter: number;
        /**
         * Defines the thickness of the torus (ie. internal diameter)
         */
        thickness: number;
        /**
         * Defines the tesselation factor to apply to the torus
         */
        tessellation: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a new torus geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param diameter defines the diameter of the torus
         * @param thickness defines the thickness of the torus (ie. internal diameter)
         * @param tessellation defines the tesselation factor to apply to the torus (number of segments along the circle)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the diameter of the torus
             */
            diameter: number, 
            /**
             * Defines the thickness of the torus (ie. internal diameter)
             */
            thickness: number, 
            /**
             * Defines the tesselation factor to apply to the torus
             */
            tessellation: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedTorus: any, scene: Scene): Nullable<TorusGeometry>;
    }
    /**
     * Creates a new ground geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#ground
     */
    class GroundGeometry extends _PrimitiveGeometry {
        /**
         * Defines the width of the ground
         */
        width: number;
        /**
         * Defines the height of the ground
         */
        height: number;
        /**
         * Defines the subdivisions to apply to the ground
         */
        subdivisions: number;
        /**
         * Creates a new ground geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param width defines the width of the ground
         * @param height defines the height of the ground
         * @param subdivisions defines the subdivisions to apply to the ground
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the width of the ground
             */
            width: number, 
            /**
             * Defines the height of the ground
             */
            height: number, 
            /**
             * Defines the subdivisions to apply to the ground
             */
            subdivisions: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedGround: any, scene: Scene): Nullable<GroundGeometry>;
    }
    /**
     * Creates a tiled ground geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#tiled-ground
     */
    class TiledGroundGeometry extends _PrimitiveGeometry {
        /**
         * Defines the minimum value on X axis
         */
        xmin: number;
        /**
         * Defines the minimum value on Z axis
         */
        zmin: number;
        /**
         * Defines the maximum value on X axis
         */
        xmax: number;
        /**
         * Defines the maximum value on Z axis
         */
        zmax: number;
        /**
         * Defines the subdivisions to apply to the ground
         */
        subdivisions: {
            w: number;
            h: number;
        };
        /**
         * Defines the precision to use when computing the tiles
         */
        precision: {
            w: number;
            h: number;
        };
        /**
         * Creates a tiled ground geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param xmin defines the minimum value on X axis
         * @param zmin defines the minimum value on Z axis
         * @param xmax defines the maximum value on X axis
         * @param zmax defines the maximum value on Z axis
         * @param subdivisions defines the subdivisions to apply to the ground (number of subdivisions (tiles) on the height and the width of the map)
         * @param precision defines the precision to use when computing the tiles
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the minimum value on X axis
             */
            xmin: number, 
            /**
             * Defines the minimum value on Z axis
             */
            zmin: number, 
            /**
             * Defines the maximum value on X axis
             */
            xmax: number, 
            /**
             * Defines the maximum value on Z axis
             */
            zmax: number, 
            /**
             * Defines the subdivisions to apply to the ground
             */
            subdivisions: {
            w: number;
            h: number;
        }, 
            /**
             * Defines the precision to use when computing the tiles
             */
            precision: {
            w: number;
            h: number;
        }, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
    }
    /**
     * Creates a plane geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#plane
     */
    class PlaneGeometry extends _PrimitiveGeometry {
        /**
         * Defines the size of the plane (width === height)
         */
        size: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a plane geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param size defines the size of the plane (width === height)
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the size of the plane (width === height)
             */
            size: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedPlane: any, scene: Scene): Nullable<PlaneGeometry>;
    }
    /**
     * Creates a torus knot geometry
     * @description see http://doc.babylonjs.com/how_to/set_shapes#torus-knot
     */
    class TorusKnotGeometry extends _PrimitiveGeometry {
        /**
         * Defines the radius of the torus knot
         */
        radius: number;
        /**
         * Defines the thickness of the torus knot tube
         */
        tube: number;
        /**
         * Defines the number of radial segments
         */
        radialSegments: number;
        /**
         * Defines the number of tubular segments
         */
        tubularSegments: number;
        /**
         * Defines the first number of windings
         */
        p: number;
        /**
         * Defines the second number of windings
         */
        q: number;
        /**
         * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        side: number;
        /**
         * Creates a torus knot geometry
         * @param id defines the unique ID of the geometry
         * @param scene defines the hosting scene
         * @param radius defines the radius of the torus knot
         * @param tube defines the thickness of the torus knot tube
         * @param radialSegments defines the number of radial segments
         * @param tubularSegments defines the number of tubular segments
         * @param p defines the first number of windings
         * @param q defines the second number of windings
         * @param canBeRegenerated defines if the geometry supports being regenerated with new parameters (false by default)
         * @param mesh defines the hosting mesh (can be null)
         * @param side defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
         */
        constructor(id: string, scene: Scene, 
            /**
             * Defines the radius of the torus knot
             */
            radius: number, 
            /**
             * Defines the thickness of the torus knot tube
             */
            tube: number, 
            /**
             * Defines the number of radial segments
             */
            radialSegments: number, 
            /**
             * Defines the number of tubular segments
             */
            tubularSegments: number, 
            /**
             * Defines the first number of windings
             */
            p: number, 
            /**
             * Defines the second number of windings
             */
            q: number, canBeRegenerated?: boolean, mesh?: Nullable<Mesh>, 
            /**
             * Defines if the created geometry is double sided or not (default is BABYLON.Mesh.DEFAULTSIDE)
             */
            side?: number);
        _regenerateVertexData(): VertexData;
        copy(id: string): Geometry;
        serialize(): any;
        static Parse(parsedTorusKnot: any, scene: Scene): Nullable<TorusKnotGeometry>;
    }
}

declare module BABYLON {
    class GroundMesh extends Mesh {
        generateOctree: boolean;
        private _heightQuads;
        _subdivisionsX: number;
        _subdivisionsY: number;
        _width: number;
        _height: number;
        _minX: number;
        _maxX: number;
        _minZ: number;
        _maxZ: number;
        constructor(name: string, scene: Scene);
        getClassName(): string;
        readonly subdivisions: number;
        readonly subdivisionsX: number;
        readonly subdivisionsY: number;
        optimize(chunksCount: number, octreeBlocksSize?: number): void;
        /**
         * Returns a height (y) value in the Worl system :
         * the ground altitude at the coordinates (x, z) expressed in the World system.
         * Returns the ground y position if (x, z) are outside the ground surface.
         */
        getHeightAtCoordinates(x: number, z: number): number;
        /**
         * Returns a normalized vector (Vector3) orthogonal to the ground
         * at the ground coordinates (x, z) expressed in the World system.
         * Returns Vector3(0.0, 1.0, 0.0) if (x, z) are outside the ground surface.
         */
        getNormalAtCoordinates(x: number, z: number): Vector3;
        /**
         * Updates the Vector3 passed a reference with a normalized vector orthogonal to the ground
         * at the ground coordinates (x, z) expressed in the World system.
         * Doesn't uptade the reference Vector3 if (x, z) are outside the ground surface.
         * Returns the GroundMesh.
         */
        getNormalAtCoordinatesToRef(x: number, z: number, ref: Vector3): GroundMesh;
        /**
        * Force the heights to be recomputed for getHeightAtCoordinates() or getNormalAtCoordinates()
        * if the ground has been updated.
        * This can be used in the render loop.
        * Returns the GroundMesh.
        */
        updateCoordinateHeights(): GroundMesh;
        private _getFacetAt(x, z);
        private _initHeightQuads();
        private _computeHeightQuads();
        serialize(serializationObject: any): void;
        static Parse(parsedMesh: any, scene: Scene): GroundMesh;
    }
}

declare module BABYLON {
    /**
     * Creates an instance based on a source mesh.
     */
    class InstancedMesh extends AbstractMesh {
        private _sourceMesh;
        private _currentLOD;
        constructor(name: string, source: Mesh);
        /**
         * Returns the string "InstancedMesh".
         */
        getClassName(): string;
        readonly receiveShadows: boolean;
        readonly material: Nullable<Material>;
        readonly visibility: number;
        readonly skeleton: Nullable<Skeleton>;
        renderingGroupId: number;
        /**
         * Returns the total number of vertices (integer).
         */
        getTotalVertices(): number;
        readonly sourceMesh: Mesh;
        /**
         * Is this node ready to be used/rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @return {boolean} is it ready
         */
        isReady(completeCheck?: boolean): boolean;
        /**
         * Returns a float array or a Float32Array of the requested kind of data : positons, normals, uvs, etc.
         */
        getVerticesData(kind: string, copyWhenShared?: boolean): Nullable<FloatArray>;
        /**
         * Sets the vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, a new Geometry object is set to the mesh and then passed this vertex data.
         * The `data` are either a numeric array either a Float32Array.
         * The parameter `updatable` is passed as is to the underlying Geometry object constructor (if initianilly none) or updater.
         * The parameter `stride` is an optional positive integer, it is usually automatically deducted from the `kind` (3 for positions or normals, 2 for UV, etc).
         * Note that a new underlying VertexBuffer object is created each call.
         * If the `kind` is the `PositionKind`, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): Mesh;
        /**
         * Updates the existing vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, it is simply returned as it is.
         * The `data` are either a numeric array either a Float32Array.
         * No new underlying VertexBuffer object is created.
         * If the `kind` is the `PositionKind` and if `updateExtends` is true, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         * If the parameter `makeItUnique` is true, a new global geometry is created from this positions and is set to the mesh.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): Mesh;
        /**
         * Sets the mesh indices.
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array).
         * If the mesh has no geometry, a new Geometry object is created and set to the mesh.
         * This method creates a new index buffer each call.
         * Returns the Mesh.
         */
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>): Mesh;
        /**
         * Boolean : True if the mesh owns the requested kind of data.
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns an array of indices (IndicesArray).
         */
        getIndices(): Nullable<IndicesArray>;
        readonly _positions: Nullable<Vector3[]>;
        /**
         * Sets a new updated BoundingInfo to the mesh.
         * Returns the mesh.
         */
        refreshBoundingInfo(): InstancedMesh;
        _preActivate(): InstancedMesh;
        _activate(renderId: number): InstancedMesh;
        /**
         * Returns the current associated LOD AbstractMesh.
         */
        getLOD(camera: Camera): AbstractMesh;
        _syncSubMeshes(): InstancedMesh;
        _generatePointsArray(): boolean;
        /**
         * Creates a new InstancedMesh from the current mesh.
         * - name (string) : the cloned mesh name
         * - newParent (optional Node) : the optional Node to parent the clone to.
         * - doNotCloneChildren (optional boolean, default `false`) : if `true` the model children aren't cloned.
         *
         * Returns the clone.
         */
        clone(name: string, newParent: Node, doNotCloneChildren?: boolean): InstancedMesh;
        /**
         * Disposes the InstancedMesh.
         * Returns nothing.
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
    }
}

declare module BABYLON {
    class LinesMesh extends Mesh {
        useVertexColor: boolean | undefined;
        useVertexAlpha: boolean | undefined;
        color: Color3;
        alpha: number;
        /**
         * The intersection Threshold is the margin applied when intersection a segment of the LinesMesh with a Ray.
         * This margin is expressed in world space coordinates, so its value may vary.
         * Default value is 0.1
         * @returns the intersection Threshold value.
         */
        /**
         * The intersection Threshold is the margin applied when intersection a segment of the LinesMesh with a Ray.
         * This margin is expressed in world space coordinates, so its value may vary.
         * @param value the new threshold to apply
         */
        intersectionThreshold: number;
        private _intersectionThreshold;
        private _colorShader;
        constructor(name: string, scene?: Nullable<Scene>, parent?: Nullable<Node>, source?: LinesMesh, doNotCloneChildren?: boolean, useVertexColor?: boolean | undefined, useVertexAlpha?: boolean | undefined);
        /**
         * Returns the string "LineMesh"
         */
        getClassName(): string;
        material: Material;
        readonly checkCollisions: boolean;
        createInstance(name: string): InstancedMesh;
        _bind(subMesh: SubMesh, effect: Effect, fillMode: number): LinesMesh;
        _draw(subMesh: SubMesh, fillMode: number, instancesCount?: number): LinesMesh;
        dispose(doNotRecurse?: boolean): void;
        /**
         * Returns a new LineMesh object cloned from the current one.
         */
        clone(name: string, newParent?: Node, doNotCloneChildren?: boolean): LinesMesh;
    }
}

declare module BABYLON {
    class _InstancesBatch {
        mustReturn: boolean;
        visibleInstances: Nullable<InstancedMesh[]>[];
        renderSelf: boolean[];
    }
    class Mesh extends AbstractMesh implements IGetSetVerticesData {
        static _FRONTSIDE: number;
        static _BACKSIDE: number;
        static _DOUBLESIDE: number;
        static _DEFAULTSIDE: number;
        static _NO_CAP: number;
        static _CAP_START: number;
        static _CAP_END: number;
        static _CAP_ALL: number;
        /**
         * Mesh side orientation : usually the external or front surface
         */
        static readonly FRONTSIDE: number;
        /**
         * Mesh side orientation : usually the internal or back surface
         */
        static readonly BACKSIDE: number;
        /**
         * Mesh side orientation : both internal and external or front and back surfaces
         */
        static readonly DOUBLESIDE: number;
        /**
         * Mesh side orientation : by default, `FRONTSIDE`
         */
        static readonly DEFAULTSIDE: number;
        /**
         * Mesh cap setting : no cap
         */
        static readonly NO_CAP: number;
        /**
         * Mesh cap setting : one cap at the beginning of the mesh
         */
        static readonly CAP_START: number;
        /**
         * Mesh cap setting : one cap at the end of the mesh
         */
        static readonly CAP_END: number;
        /**
         * Mesh cap setting : two caps, one at the beginning  and one at the end of the mesh
         */
        static readonly CAP_ALL: number;
        /**
         * An event triggered before rendering the mesh
         */
        onBeforeRenderObservable: Observable<Mesh>;
        /**
        * An event triggered after rendering the mesh
        */
        onAfterRenderObservable: Observable<Mesh>;
        /**
        * An event triggered before drawing the mesh
        */
        onBeforeDrawObservable: Observable<Mesh>;
        private _onBeforeDrawObserver;
        onBeforeDraw: () => void;
        delayLoadState: number;
        instances: InstancedMesh[];
        delayLoadingFile: string;
        _binaryInfo: any;
        private _LODLevels;
        onLODLevelSelection: (distance: number, mesh: Mesh, selectedLevel: Mesh) => void;
        private _morphTargetManager;
        morphTargetManager: Nullable<MorphTargetManager>;
        _geometry: Nullable<Geometry>;
        _delayInfo: Array<string>;
        _delayLoadingFunction: (any: any, mesh: Mesh) => void;
        _visibleInstances: any;
        private _renderIdForInstances;
        private _batchCache;
        private _instancesBufferSize;
        private _instancesBuffer;
        private _instancesData;
        private _overridenInstanceCount;
        private _effectiveMaterial;
        _shouldGenerateFlatShading: boolean;
        private _preActivateId;
        _originalBuilderSideOrientation: number;
        overrideMaterialSideOrientation: Nullable<number>;
        private _areNormalsFrozen;
        private _sourcePositions;
        private _sourceNormals;
        private _source;
        readonly source: Nullable<Mesh>;
        isUnIndexed: boolean;
        /**
         * @constructor
         * @param {string} name The value used by scene.getMeshByName() to do a lookup.
         * @param {Scene} scene The scene to add this mesh to.
         * @param {Node} parent The parent of this mesh, if it has one
         * @param {Mesh} source An optional Mesh from which geometry is shared, cloned.
         * @param {boolean} doNotCloneChildren When cloning, skip cloning child meshes of source, default False.
         *                  When false, achieved by calling a clone(), also passing False.
         *                  This will make creation of children, recursive.
         * @param {boolean} clonePhysicsImpostor When cloning, include cloning mesh physics impostor, default True.
         */
        constructor(name: string, scene?: Nullable<Scene>, parent?: Nullable<Node>, source?: Nullable<Mesh>, doNotCloneChildren?: boolean, clonePhysicsImpostor?: boolean);
        /**
         * Returns the string "Mesh".
         */
        getClassName(): string;
        /**
         * Returns a string.
         * @param {boolean} fullDetails - support for multiple levels of logging within scene loading
         */
        toString(fullDetails?: boolean): string;
        /**
         * True if the mesh has some Levels Of Details (LOD).
         * Returns a boolean.
         */
        readonly hasLODLevels: boolean;
        /**
         * Gets the list of {BABYLON.MeshLODLevel} associated with the current mesh
         * @returns an array of {BABYLON.MeshLODLevel}
         */
        getLODLevels(): MeshLODLevel[];
        private _sortLODLevels();
        /**
         * Add a mesh as LOD level triggered at the given distance.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * @param {number} distance The distance from the center of the object to show this level
         * @param {Mesh} mesh The mesh to be added as LOD level
         * @return {Mesh} This mesh (for chaining)
         */
        addLODLevel(distance: number, mesh: Mesh): Mesh;
        /**
         * Returns the LOD level mesh at the passed distance or null if not found.
         * It is related to the method `addLODLevel(distance, mesh)`.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * Returns an object Mesh or `null`.
         */
        getLODLevelAtDistance(distance: number): Nullable<Mesh>;
        /**
         * Remove a mesh from the LOD array
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         * @param {Mesh} mesh The mesh to be removed.
         * @return {Mesh} This mesh (for chaining)
         */
        removeLODLevel(mesh: Mesh): Mesh;
        /**
         * Returns the registered LOD mesh distant from the parameter `camera` position if any, else returns the current mesh.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_LOD
         */
        getLOD(camera: Camera, boundingSphere?: BoundingSphere): AbstractMesh;
        /**
         * Returns the mesh internal Geometry object.
         */
        readonly geometry: Nullable<Geometry>;
        /**
         * Returns a positive integer : the total number of vertices within the mesh geometry or zero if the mesh has no geometry.
         */
        getTotalVertices(): number;
        /**
         * Returns an array of integers or floats, or a Float32Array, depending on the requested `kind` (positions, indices, normals, etc).
         * If `copywhenShared` is true (default false) and if the mesh geometry is shared among some other meshes, the returned array is a copy of the internal one.
         * You can force the copy with forceCopy === true
         * Returns null if the mesh has no geometry or no vertex buffer.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        /**
         * Returns the mesh VertexBuffer object from the requested `kind` : positions, indices, normals, etc.
         * Returns `null` if the mesh has no geometry.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVertexBuffer(kind: string): Nullable<VertexBuffer>;
        /**
         * Returns a boolean depending on the existence of the Vertex Data for the requested `kind`.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        isVerticesDataPresent(kind: string): boolean;
        /**
         * Returns a boolean defining if the vertex data for the requested `kind` is updatable.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        isVertexBufferUpdatable(kind: string): boolean;
        /**
         * Returns a string : the list of existing `kinds` of Vertex Data for this mesh.
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         */
        getVerticesDataKinds(): string[];
        /**
         * Returns a positive integer : the total number of indices in this mesh geometry.
         * Returns zero if the mesh has no geometry.
         */
        getTotalIndices(): number;
        /**
         * Returns an array of integers or a typed array (Int32Array, Uint32Array, Uint16Array) populated with the mesh indices.
         * If the parameter `copyWhenShared` is true (default false) and and if the mesh geometry is shared among some other meshes, the returned array is a copy of the internal one.
         * Returns an empty array if the mesh has no geometry.
         */
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        readonly isBlocked: boolean;
        /**
         * Determine if the current mesh is ready to be rendered
         * @param completeCheck defines if a complete check (including materials and lights) has to be done (false by default)
         * @param forceInstanceSupport will check if the mesh will be ready when used with instances (false by default)
         * @returns true if all associated assets are ready (material, textures, shaders)
         */
        isReady(completeCheck?: boolean, forceInstanceSupport?: boolean): boolean;
        /**
         * Boolean : true if the normals aren't to be recomputed on next mesh `positions` array update.
         * This property is pertinent only for updatable parametric shapes.
         */
        readonly areNormalsFrozen: boolean;
        /**
         * This function affects parametric shapes on vertex position update only : ribbons, tubes, etc.
         * It has no effect at all on other shapes.
         * It prevents the mesh normals from being recomputed on next `positions` array update.
         * Returns the Mesh.
         */
        freezeNormals(): Mesh;
        /**
         * This function affects parametric shapes on vertex position update only : ribbons, tubes, etc.
         * It has no effect at all on other shapes.
         * It reactivates the mesh normals computation if it was previously frozen.
         * Returns the Mesh.
         */
        unfreezeNormals(): Mesh;
        /**
         * Overrides instance count. Only applicable when custom instanced InterleavedVertexBuffer are used rather than InstancedMeshs
         */
        overridenInstanceCount: number;
        _preActivate(): Mesh;
        _preActivateForIntermediateRendering(renderId: number): Mesh;
        _registerInstanceForRenderId(instance: InstancedMesh, renderId: number): Mesh;
        /**
         * This method recomputes and sets a new BoundingInfo to the mesh unless it is locked.
         * This means the mesh underlying bounding box and sphere are recomputed.
         * Returns the Mesh.
         */
        refreshBoundingInfo(): Mesh;
        _refreshBoundingInfo(applySkeleton: boolean): Mesh;
        private _getPositionData(applySkeleton);
        _createGlobalSubMesh(force: boolean): Nullable<SubMesh>;
        subdivide(count: number): void;
        /**
         * Sets the vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, a new Geometry object is set to the mesh and then passed this vertex data.
         * The `data` are either a numeric array either a Float32Array.
         * The parameter `updatable` is passed as is to the underlying Geometry object constructor (if initianilly none) or updater.
         * The parameter `stride` is an optional positive integer, it is usually automatically deducted from the `kind` (3 for positions or normals, 2 for UV, etc).
         * Note that a new underlying VertexBuffer object is created each call.
         * If the `kind` is the `PositionKind`, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        setVerticesData(kind: string, data: FloatArray, updatable?: boolean, stride?: number): Mesh;
        markVerticesDataAsUpdatable(kind: string, updatable?: boolean): void;
        /**
         * Sets the mesh VertexBuffer.
         * Returns the Mesh.
         */
        setVerticesBuffer(buffer: VertexBuffer): Mesh;
        /**
         * Updates the existing vertex data of the mesh geometry for the requested `kind`.
         * If the mesh has no geometry, it is simply returned as it is.
         * The `data` are either a numeric array either a Float32Array.
         * No new underlying VertexBuffer object is created.
         * If the `kind` is the `PositionKind` and if `updateExtends` is true, the mesh BoundingInfo is renewed, so the bounding box and sphere, and the mesh World Matrix is recomputed.
         * If the parameter `makeItUnique` is true, a new global geometry is created from this positions and is set to the mesh.
         *
         * Possible `kind` values :
         * - BABYLON.VertexBuffer.PositionKind
         * - BABYLON.VertexBuffer.UVKind
         * - BABYLON.VertexBuffer.UV2Kind
         * - BABYLON.VertexBuffer.UV3Kind
         * - BABYLON.VertexBuffer.UV4Kind
         * - BABYLON.VertexBuffer.UV5Kind
         * - BABYLON.VertexBuffer.UV6Kind
         * - BABYLON.VertexBuffer.ColorKind
         * - BABYLON.VertexBuffer.MatricesIndicesKind
         * - BABYLON.VertexBuffer.MatricesIndicesExtraKind
         * - BABYLON.VertexBuffer.MatricesWeightsKind
         * - BABYLON.VertexBuffer.MatricesWeightsExtraKind
         *
         * Returns the Mesh.
         */
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): Mesh;
        /**
         * This method updates the vertex positions of an updatable mesh according to the `positionFunction` returned values.
         * tuto : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#other-shapes-updatemeshpositions
         * The parameter `positionFunction` is a simple JS function what is passed the mesh `positions` array. It doesn't need to return anything.
         * The parameter `computeNormals` is a boolean (default true) to enable/disable the mesh normal recomputation after the vertex position update.
         * Returns the Mesh.
         */
        updateMeshPositions(positionFunction: (data: FloatArray) => void, computeNormals?: boolean): Mesh;
        /**
         * Creates a un-shared specific occurence of the geometry for the mesh.
         * Returns the Mesh.
         */
        makeGeometryUnique(): Mesh;
        /**
         * Sets the mesh indices.
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array).
         * Type is Uint16Array by default unless the mesh has more than 65536 vertices.
         * If the mesh has no geometry, a new Geometry object is created and set to the mesh.
         * This method creates a new index buffer each call.
         * Returns the Mesh.
         */
        setIndices(indices: IndicesArray, totalVertices?: Nullable<number>, updatable?: boolean): Mesh;
        /**
         * Update the current index buffer
         * Expects an array populated with integers or a typed array (Int32Array, Uint32Array, Uint16Array)
         * Returns the Mesh.
         */
        updateIndices(indices: IndicesArray, offset?: number): Mesh;
        /**
         * Invert the geometry to move from a right handed system to a left handed one.
         * Returns the Mesh.
         */
        toLeftHanded(): Mesh;
        _bind(subMesh: SubMesh, effect: Effect, fillMode: number): Mesh;
        _draw(subMesh: SubMesh, fillMode: number, instancesCount?: number, alternate?: boolean): Mesh;
        /**
         * Registers for this mesh a javascript function called just before the rendering process.
         * This function is passed the current mesh.
         * Return the Mesh.
         */
        registerBeforeRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Disposes a previously registered javascript function called before the rendering.
         * This function is passed the current mesh.
         * Returns the Mesh.
         */
        unregisterBeforeRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Registers for this mesh a javascript function called just after the rendering is complete.
         * This function is passed the current mesh.
         * Returns the Mesh.
         */
        registerAfterRender(func: (mesh: AbstractMesh) => void): Mesh;
        /**
         * Disposes a previously registered javascript function called after the rendering.
         * This function is passed the current mesh.
         * Return the Mesh.
         */
        unregisterAfterRender(func: (mesh: AbstractMesh) => void): Mesh;
        _getInstancesRenderList(subMeshId: number): _InstancesBatch;
        _renderWithInstances(subMesh: SubMesh, fillMode: number, batch: _InstancesBatch, effect: Effect, engine: Engine): Mesh;
        _processRendering(subMesh: SubMesh, effect: Effect, fillMode: number, batch: _InstancesBatch, hardwareInstancedRendering: boolean, onBeforeDraw: (isInstance: boolean, world: Matrix, effectiveMaterial?: Material) => void, effectiveMaterial?: Material): Mesh;
        /**
         * Triggers the draw call for the mesh.
         * Usually, you don't need to call this method by your own because the mesh rendering is handled by the scene rendering manager.
         * Returns the Mesh.
         */
        render(subMesh: SubMesh, enableAlphaMode: boolean): Mesh;
        private _onBeforeDraw(isInstance, world, effectiveMaterial?);
        /**
         * Returns an array populated with ParticleSystem objects whose the mesh is the emitter.
         */
        getEmittedParticleSystems(): IParticleSystem[];
        /**
         * Returns an array populated with ParticleSystem objects whose the mesh or its children are the emitter.
         */
        getHierarchyEmittedParticleSystems(): IParticleSystem[];
        _checkDelayState(): Mesh;
        private _queueLoad(scene);
        /**
         * Boolean, true is the mesh in the frustum defined by the Plane objects from the `frustumPlanes` array parameter.
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Sets the mesh material by the material or multiMaterial `id` property.
         * The material `id` is a string identifying the material or the multiMaterial.
         * This method returns the Mesh.
         */
        setMaterialByID(id: string): Mesh;
        /**
         * Returns as a new array populated with the mesh material and/or skeleton, if any.
         */
        getAnimatables(): IAnimatable[];
        /**
         * Modifies the mesh geometry according to the passed transformation matrix.
         * This method returns nothing but it really modifies the mesh even if it's originally not set as updatable.
         * The mesh normals are modified accordingly the same transformation.
         * tuto : http://doc.babylonjs.com/tutorials/How_Rotations_and_Translations_Work#baking-transform
         * Note that, under the hood, this method sets a new VertexBuffer each call.
         * Returns the Mesh.
         */
        bakeTransformIntoVertices(transform: Matrix): Mesh;
        /**
         * Modifies the mesh geometry according to its own current World Matrix.
         * The mesh World Matrix is then reset.
         * This method returns nothing but really modifies the mesh even if it's originally not set as updatable.
         * tuto : tuto : http://doc.babylonjs.com/resources/baking_transformations
         * Note that, under the hood, this method sets a new VertexBuffer each call.
         * Returns the Mesh.
         */
        bakeCurrentTransformIntoVertices(): Mesh;
        readonly _positions: Nullable<Vector3[]>;
        _resetPointsArrayCache(): Mesh;
        _generatePointsArray(): boolean;
        /**
         * Returns a new Mesh object generated from the current mesh properties.
         * This method must not get confused with createInstance().
         * The parameter `name` is a string, the name given to the new mesh.
         * The optional parameter `newParent` can be any Node object (default `null`).
         * The optional parameter `doNotCloneChildren` (default `false`) allows/denies the recursive cloning of the original mesh children if any.
         * The parameter `clonePhysicsImpostor` (default `true`)  allows/denies the cloning in the same time of the original mesh `body` used by the physics engine, if any.
         */
        clone(name: string, newParent?: Node, doNotCloneChildren?: boolean, clonePhysicsImpostor?: boolean): Mesh;
        /**
         * Releases resources associated with this mesh.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
        /**
         * Modifies the mesh geometry according to a displacement map.
         * A displacement map is a colored image. Each pixel color value (actually a gradient computed from red, green, blue values) will give the displacement to apply to each mesh vertex.
         * The mesh must be set as updatable. Its internal geometry is directly modified, no new buffer are allocated.
         * This method returns nothing.
         * The parameter `url` is a string, the URL from the image file is to be downloaded.
         * The parameters `minHeight` and `maxHeight` are the lower and upper limits of the displacement.
         * The parameter `onSuccess` is an optional Javascript function to be called just after the mesh is modified. It is passed the modified mesh and must return nothing.
         * The parameter `uvOffset` is an optional vector2 used to offset UV.
         * The parameter `uvScale` is an optional vector2 used to scale UV.
         *
         * Returns the Mesh.
         */
        applyDisplacementMap(url: string, minHeight: number, maxHeight: number, onSuccess?: (mesh: Mesh) => void, uvOffset?: Vector2, uvScale?: Vector2): Mesh;
        /**
         * Modifies the mesh geometry according to a displacementMap buffer.
         * A displacement map is a colored image. Each pixel color value (actually a gradient computed from red, green, blue values) will give the displacement to apply to each mesh vertex.
         * The mesh must be set as updatable. Its internal geometry is directly modified, no new buffer are allocated.
         * This method returns nothing.
         * The parameter `buffer` is a `Uint8Array` buffer containing series of `Uint8` lower than 255, the red, green, blue and alpha values of each successive pixel.
         * The parameters `heightMapWidth` and `heightMapHeight` are positive integers to set the width and height of the buffer image.
         * The parameters `minHeight` and `maxHeight` are the lower and upper limits of the displacement.
         * The parameter `uvOffset` is an optional vector2 used to offset UV.
         * The parameter `uvScale` is an optional vector2 used to scale UV.
         *
         * Returns the Mesh.
         */
        applyDisplacementMapFromBuffer(buffer: Uint8Array, heightMapWidth: number, heightMapHeight: number, minHeight: number, maxHeight: number, uvOffset?: Vector2, uvScale?: Vector2): Mesh;
        /**
         * Modify the mesh to get a flat shading rendering.
         * This means each mesh facet will then have its own normals. Usually new vertices are added in the mesh geometry to get this result.
         * This method returns the Mesh.
         * Warning : the mesh is really modified even if not set originally as updatable and, under the hood, a new VertexBuffer is allocated.
         */
        convertToFlatShadedMesh(): Mesh;
        /**
         * This method removes all the mesh indices and add new vertices (duplication) in order to unfold facets into buffers.
         * In other words, more vertices, no more indices and a single bigger VBO.
         * The mesh is really modified even if not set originally as updatable. Under the hood, a new VertexBuffer is allocated.
         * Returns the Mesh.
         */
        convertToUnIndexedMesh(): Mesh;
        /**
         * Inverses facet orientations and inverts also the normals with `flipNormals` (default `false`) if true.
         * This method returns the Mesh.
         * Warning : the mesh is really modified even if not set originally as updatable. A new VertexBuffer is created under the hood each call.
         */
        flipFaces(flipNormals?: boolean): Mesh;
        /**
         * Creates a new InstancedMesh object from the mesh model.
         * An instance shares the same properties and the same material than its model.
         * Only these properties of each instance can then be set individually :
         * - position
         * - rotation
         * - rotationQuaternion
         * - setPivotMatrix
         * - scaling
         * tuto : http://doc.babylonjs.com/tutorials/How_to_use_Instances
         * Warning : this method is not supported for Line mesh and LineSystem
         */
        createInstance(name: string): InstancedMesh;
        /**
         * Synchronises all the mesh instance submeshes to the current mesh submeshes, if any.
         * After this call, all the mesh instances have the same submeshes than the current mesh.
         * This method returns the Mesh.
         */
        synchronizeInstances(): Mesh;
        /**
         * Simplify the mesh according to the given array of settings.
         * Function will return immediately and will simplify async. It returns the Mesh.
         * @param settings a collection of simplification settings.
         * @param parallelProcessing should all levels calculate parallel or one after the other.
         * @param type the type of simplification to run.
         * @param successCallback optional success callback to be called after the simplification finished processing all settings.
         */
        simplify(settings: Array<ISimplificationSettings>, parallelProcessing?: boolean, simplificationType?: SimplificationType, successCallback?: (mesh?: Mesh, submeshIndex?: number) => void): Mesh;
        /**
         * Optimization of the mesh's indices, in case a mesh has duplicated vertices.
         * The function will only reorder the indices and will not remove unused vertices to avoid problems with submeshes.
         * This should be used together with the simplification to avoid disappearing triangles.
         * Returns the Mesh.
         * @param successCallback an optional success callback to be called after the optimization finished.
         */
        optimizeIndices(successCallback?: (mesh?: Mesh) => void): Mesh;
        serialize(serializationObject: any): void;
        _syncGeometryWithMorphTargetManager(): void;
        /**
         * Returns a new Mesh object parsed from the source provided.
         * The parameter `parsedMesh` is the source.
         * The parameter `rootUrl` is a string, it's the root URL to prefix the `delayLoadingFile` property with
         */
        static Parse(parsedMesh: any, scene: Scene, rootUrl: string): Mesh;
        /**
         * Creates a ribbon mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The ribbon is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         *
         * Please read this full tutorial to understand how to design a ribbon : http://doc.babylonjs.com/tutorials/Ribbon_Tutorial
         * The parameter `pathArray` is a required array of paths, what are each an array of successive Vector3. The pathArray parameter depicts the ribbon geometry.
         * The parameter `closeArray` (boolean, default false) creates a seam between the first and the last paths of the path array.
         * The parameter `closePath` (boolean, default false) creates a seam between the first and the last points of each path of the path array.
         * The parameter `offset` (positive integer, default : rounded half size of the pathArray length), is taken in account only if the `pathArray` is containing a single path.
         * It's the offset to join together the points from the same path. Ex : offset = 10 means the point 1 is joined to the point 11.
         * The optional parameter `instance` is an instance of an existing Ribbon object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#ribbon
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateRibbon(name: string, pathArray: Vector3[][], closeArray: boolean | undefined, closePath: boolean, offset: number, scene?: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates a plane polygonal mesh.  By default, this is a disc.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the radius size (float) of the polygon (default 0.5).
         * The parameter `tessellation` sets the number of polygon sides (positive integer, default 64). So a tessellation valued to 3 will build a triangle, to 4 a square, etc.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDisc(name: string, radius: number, tessellation: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a box mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `size` sets the size (float) of each box side (default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateBox(name: string, size: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a sphere mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `diameter` sets the diameter size (float) of the sphere (default 1).
         * The parameter `segments` sets the sphere number of horizontal stripes (positive integer, default 32).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateSphere(name: string, segments: number, diameter: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a cylinder or a cone mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `height` sets the height size (float) of the cylinder/cone (float, default 2).
         * The parameter `diameter` sets the diameter of the top and bottom cap at once (float, default 1).
         * The parameters `diameterTop` and `diameterBottom` overwrite the parameter `diameter` and set respectively the top cap and bottom cap diameter (floats, default 1). The parameter "diameterBottom" can't be zero.
         * The parameter `tessellation` sets the number of cylinder sides (positive integer, default 24). Set it to 3 to get a prism for instance.
         * The parameter `subdivisions` sets the number of rings along the cylinder height (positive integer, default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateCylinder(name: string, height: number, diameterTop: number, diameterBottom: number, tessellation: number, subdivisions: any, scene?: Scene, updatable?: any, sideOrientation?: number): Mesh;
        /**
         * Creates a torus mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `diameter` sets the diameter size (float) of the torus (default 1).
         * The parameter `thickness` sets the diameter size of the tube of the torus (float, default 0.5).
         * The parameter `tessellation` sets the number of torus sides (postive integer, default 16).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorus(name: string, diameter: number, thickness: number, tessellation: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a torus knot mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the global radius size (float) of the torus knot (default 2).
         * The parameter `radialSegments` sets the number of sides on each tube segments (positive integer, default 32).
         * The parameter `tubularSegments` sets the number of tubes to decompose the knot into (positive integer, default 32).
         * The parameters `p` and `q` are the number of windings on each axis (positive integers, default 2 and 3).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorusKnot(name: string, radius: number, tube: number, radialSegments: number, tubularSegments: number, p: number, q: number, scene?: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a line mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLines(name: string, points: Vector3[], scene?: Nullable<Scene>, updatable?: boolean, instance?: Nullable<LinesMesh>): LinesMesh;
        /**
         * Creates a dashed line mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A dashed line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The parameter `dashNb` is the intended total number of dashes (positive integer, default 200).
         * The parameter `dashSize` is the size of the dashes relatively the dash number (positive float, default 3).
         * The parameter `gapSize` is the size of the gap between two successive dashes relatively the dash number (positive float, default 1).
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDashedLines(name: string, points: Vector3[], dashSize: number, gapSize: number, dashNb: number, scene?: Nullable<Scene>, updatable?: boolean, instance?: LinesMesh): LinesMesh;
        /**
         * Creates a polygon mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The polygon's shape will depend on the input parameters and is constructed parallel to a ground mesh.
         * The parameter `shape` is a required array of successive Vector3 representing the corners of the polygon in th XoZ plane, that is y = 0 for all vectors.
         * You can set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * Remember you can only change the shape positions, not their number when updating a polygon.
         */
        static CreatePolygon(name: string, shape: Vector3[], scene: Scene, holes?: Vector3[][], updatable?: boolean, sideOrientation?: number): Mesh;
        /**
          * Creates an extruded polygon mesh, with depth in the Y direction.
          * Please consider using the same method from the MeshBuilder class instead.
         */
        static ExtrudePolygon(name: string, shape: Vector3[], depth: number, scene: Scene, holes?: Vector3[][], updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates an extruded shape mesh.
         * The extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         *
         * Please read this full tutorial to understand how to design an extruded shape : http://doc.babylonjs.com/tutorials/Parametric_Shapes#extrusion
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotation` (float, default 0 radians) is the angle value to rotate the shape each step (each path point), from the former step (so rotation added each step) along the curve.
         * The parameter `scale` (float, default 1) is the value to scale the shape.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShape(name: string, shape: Vector3[], path: Vector3[], scale: number, rotation: number, cap: number, scene?: Nullable<Scene>, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates an custom extruded shape mesh.
         * The custom extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         *
         * Please read this full tutorial to understand how to design a custom extruded shape : http://doc.babylonjs.com/tutorials/Parametric_Shapes#extrusion
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotationFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var rotationFunction = function(i, distance) {
         *     // do things
         *     return rotationValue; }
         * ```
         * It must returns a float value that will be the rotation in radians applied to the shape on each path point.
         * The parameter `scaleFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var scaleFunction = function(i, distance) {
         *     // do things
         *    return scaleValue;}
         * ```
         * It must returns a float value that will be the scale value applied to the shape on each path point.
         * The parameter `ribbonClosePath` (boolean, default false) forces the extrusion underlying ribbon to close all the paths in its `pathArray`.
         * The parameter `ribbonCloseArray` (boolean, default false) forces the extrusion underlying ribbon to close its `pathArray`.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShapeCustom(name: string, shape: Vector3[], path: Vector3[], scaleFunction: Function, rotationFunction: Function, ribbonCloseArray: boolean, ribbonClosePath: boolean, cap: number, scene: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates lathe mesh.
         * The lathe is a shape with a symetry axis : a 2D model shape is rotated around this axis to design the lathe.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be rotated in its local space : the shape must be designed in the xOy plane and will be
         * rotated around the Y axis. It's usually a 2D shape, so the Vector3 z coordinates are often set to zero.
         * The parameter `radius` (positive float, default 1) is the radius value of the lathe.
         * The parameter `tessellation` (positive integer, default 64) is the side number of the lathe.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLathe(name: string, shape: Vector3[], radius: number, tessellation: number, scene: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a plane mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `size` sets the size (float) of both sides of the plane at once (default 1).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePlane(name: string, size: number, scene: Scene, updatable?: boolean, sideOrientation?: number): Mesh;
        /**
         * Creates a ground mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameters `width` and `height` (floats, default 1) set the width and height sizes of the ground.
         * The parameter `subdivisions` (positive integer) sets the number of subdivisions per side.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGround(name: string, width: number, height: number, subdivisions: number, scene?: Scene, updatable?: boolean): Mesh;
        /**
         * Creates a tiled ground mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameters `xmin` and `xmax` (floats, default -1 and 1) set the ground minimum and maximum X coordinates.
         * The parameters `zmin` and `zmax` (floats, default -1 and 1) set the ground minimum and maximum Z coordinates.
         * The parameter `subdivisions` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 6, h: 6}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height. Each subdivision is called a tile.
         * The parameter `precision` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 2, h: 2}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height of each tile.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTiledGround(name: string, xmin: number, zmin: number, xmax: number, zmax: number, subdivisions: {
            w: number;
            h: number;
        }, precision: {
            w: number;
            h: number;
        }, scene: Scene, updatable?: boolean): Mesh;
        /**
         * Creates a ground mesh from a height map.
         * tuto : http://doc.babylonjs.com/tutorials/14._Height_Map
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `url` sets the URL of the height map image Resource.
         * The parameters `width` and `height` (positive floats, default 10) set the ground width and height sizes.
         * The parameter `subdivisions` (positive integer, default 1) sets the number of subdivision per side.
         * The parameter `minHeight` (float, default 0) is the minimum altitude on the ground.
         * The parameter `maxHeight` (float, default 1) is the maximum altitude on the ground.
         * The parameter `onReady` is a javascript callback function that will be called  once the mesh is just built (the height map download can last some time).
         * This function is passed the newly built mesh :
         * ```javascript
         * function(mesh) { // do things
         *     return; }
         * ```
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGroundFromHeightMap(name: string, url: string, width: number, height: number, subdivisions: number, minHeight: number, maxHeight: number, scene: Scene, updatable?: boolean, onReady?: (mesh: GroundMesh) => void): GroundMesh;
        /**
         * Creates a tube mesh.
         * The tube is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `path` is a required array of successive Vector3. It is the curve used as the axis of the tube.
         * The parameter `radius` (positive float, default 1) sets the tube radius size.
         * The parameter `tessellation` (positive float, default 64) is the number of sides on the tubular surface.
         * The parameter `radiusFunction` (javascript function, default null) is a vanilla javascript function. If it is not null, it overwrittes the parameter `radius`.
         * This function is called on each point of the tube path and is passed the index `i` of the i-th point and the distance of this point from the first point of the path.
         * It must return a radius value (positive float) :
         * ```javascript
         * var radiusFunction = function(i, distance) {
         *     // do things
         *     return radius; }
         * ```
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing Tube object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#tube
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTube(name: string, path: Vector3[], radius: number, tessellation: number, radiusFunction: {
            (i: number, distance: number): number;
        }, cap: number, scene: Scene, updatable?: boolean, sideOrientation?: number, instance?: Mesh): Mesh;
        /**
         * Creates a polyhedron mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `type` (positive integer, max 14, default 0) sets the polyhedron type to build among the 15 embbeded types. Please refer to the type sheet in the tutorial
         *  to choose the wanted type.
         * The parameter `size` (positive float, default 1) sets the polygon size.
         * You can overwrite the `size` on each dimension bu using the parameters `sizeX`, `sizeY` or `sizeZ` (positive floats, default to `size` value).
         * You can build other polyhedron types than the 15 embbeded ones by setting the parameter `custom` (`polyhedronObject`, default null). If you set the parameter `custom`, this overwrittes the parameter `type`.
         * A `polyhedronObject` is a formatted javascript object. You'll find a full file with pre-set polyhedra here : https://github.com/BabylonJS/Extensions/tree/master/Polyhedron
         * You can set the color and the UV of each side of the polyhedron with the parameters `faceColors` (Color4, default `(1, 1, 1, 1)`) and faceUV (Vector4, default `(0, 0, 1, 1)`).
         * To understand how to set `faceUV` or `faceColors`, please read this by considering the right number of faces of your polyhedron, instead of only 6 for the box : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * The parameter `flat` (boolean, default true). If set to false, it gives the polyhedron a single global face, so less vertices and shared normals. In this case, `faceColors` and `faceUV` are ignored.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePolyhedron(name: string, options: {
            type?: number;
            size?: number;
            sizeX?: number;
            sizeY?: number;
            sizeZ?: number;
            custom?: any;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
        }, scene: Scene): Mesh;
        /**
         * Creates a sphere based upon an icosahedron with 20 triangular faces which can be subdivided.
         * Please consider using the same method from the MeshBuilder class instead.
         * The parameter `radius` sets the radius size (float) of the icosphere (default 1).
         * You can set some different icosphere dimensions, for instance to build an ellipsoid, by using the parameters `radiusX`, `radiusY` and `radiusZ` (all by default have the same value than `radius`).
         * The parameter `subdivisions` sets the number of subdivisions (postive integer, default 4). The more subdivisions, the more faces on the icosphere whatever its size.
         * The parameter `flat` (boolean, default true) gives each side its own normals. Set it to false to get a smooth continuous light reflection on the surface.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateIcoSphere(name: string, options: {
            radius?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a decal mesh.
         * Please consider using the same method from the MeshBuilder class instead.
         * A decal is a mesh usually applied as a model onto the surface of another mesh. So don't forget the parameter `sourceMesh` depicting the decal.
         * The parameter `position` (Vector3, default `(0, 0, 0)`) sets the position of the decal in World coordinates.
         * The parameter `normal` (Vector3, default Vector3.Up) sets the normal of the mesh where the decal is applied onto in World coordinates.
         * The parameter `size` (Vector3, default `(1, 1, 1)`) sets the decal scaling.
         * The parameter `angle` (float in radian, default 0) sets the angle to rotate the decal.
         */
        static CreateDecal(name: string, sourceMesh: AbstractMesh, position: Vector3, normal: Vector3, size: Vector3, angle: number): Mesh;
        /**
         * @returns original positions used for CPU skinning.  Useful for integrating Morphing with skeletons in same mesh.
         */
        setPositionsForCPUSkinning(): Float32Array;
        /**
         * @returns original normals used for CPU skinning.  Useful for integrating Morphing with skeletons in same mesh.
         */
        setNormalsForCPUSkinning(): Float32Array;
        /**
         * Updates the vertex buffer by applying transformation from the bones.
         * Returns the Mesh.
         *
         * @param {skeleton} skeleton to apply
         */
        applySkeleton(skeleton: Skeleton): Mesh;
        /**
         * Returns an object `{min:` Vector3`, max:` Vector3`}`
         * This min and max Vector3 are the minimum and maximum vectors of each mesh bounding box from the passed array, in the World system
         */
        static MinMax(meshes: AbstractMesh[]): {
            min: Vector3;
            max: Vector3;
        };
        /**
         * Returns a Vector3, the center of the `{min:` Vector3`, max:` Vector3`}` or the center of MinMax vector3 computed from a mesh array.
         */
        static Center(meshesOrMinMaxVector: {
            min: Vector3;
            max: Vector3;
        } | AbstractMesh[]): Vector3;
        /**
         * Merge the array of meshes into a single mesh for performance reasons.
         * @param {Array<Mesh>} meshes - The vertices source.  They should all be of the same material.  Entries can empty
         * @param {boolean} disposeSource - When true (default), dispose of the vertices from the source meshes
         * @param {boolean} allow32BitsIndices - When the sum of the vertices > 64k, this must be set to true.
         * @param {Mesh} meshSubclass - When set, vertices inserted into this Mesh.  Meshes can then be merged into a Mesh sub-class.
         * @param {boolean} subdivideWithSubMeshes - When true (false default), subdivide mesh to his subMesh array with meshes source.
         */
        static MergeMeshes(meshes: Array<Mesh>, disposeSource?: boolean, allow32BitsIndices?: boolean, meshSubclass?: Mesh, subdivideWithSubMeshes?: boolean): Nullable<Mesh>;
    }
}

declare module BABYLON {
    interface IGetSetVerticesData {
        isVerticesDataPresent(kind: string): boolean;
        getVerticesData(kind: string, copyWhenShared?: boolean, forceCopy?: boolean): Nullable<FloatArray>;
        getIndices(copyWhenShared?: boolean): Nullable<IndicesArray>;
        setVerticesData(kind: string, data: FloatArray, updatable: boolean): void;
        updateVerticesData(kind: string, data: FloatArray, updateExtends?: boolean, makeItUnique?: boolean): void;
        setIndices(indices: IndicesArray, totalVertices: Nullable<number>, updatable?: boolean): void;
    }
    class VertexData {
        positions: Nullable<FloatArray>;
        normals: Nullable<FloatArray>;
        tangents: Nullable<FloatArray>;
        uvs: Nullable<FloatArray>;
        uvs2: Nullable<FloatArray>;
        uvs3: Nullable<FloatArray>;
        uvs4: Nullable<FloatArray>;
        uvs5: Nullable<FloatArray>;
        uvs6: Nullable<FloatArray>;
        colors: Nullable<FloatArray>;
        matricesIndices: Nullable<FloatArray>;
        matricesWeights: Nullable<FloatArray>;
        matricesIndicesExtra: Nullable<FloatArray>;
        matricesWeightsExtra: Nullable<FloatArray>;
        indices: Nullable<IndicesArray>;
        set(data: FloatArray, kind: string): void;
        /**
         * Associates the vertexData to the passed Mesh.
         * Sets it as updatable or not (default `false`).
         * Returns the VertexData.
         */
        applyToMesh(mesh: Mesh, updatable?: boolean): VertexData;
        /**
         * Associates the vertexData to the passed Geometry.
         * Sets it as updatable or not (default `false`).
         * Returns the VertexData.
         */
        applyToGeometry(geometry: Geometry, updatable?: boolean): VertexData;
        /**
         * Updates the associated mesh.
         * Returns the VertexData.
         */
        updateMesh(mesh: Mesh, updateExtends?: boolean, makeItUnique?: boolean): VertexData;
        /**
         * Updates the associated geometry.
         * Returns the VertexData.
         */
        updateGeometry(geometry: Geometry, updateExtends?: boolean, makeItUnique?: boolean): VertexData;
        private _applyTo(meshOrGeometry, updatable?);
        private _update(meshOrGeometry, updateExtends?, makeItUnique?);
        /**
         * Transforms each position and each normal of the vertexData according to the passed Matrix.
         * Returns the VertexData.
         */
        transform(matrix: Matrix): VertexData;
        /**
         * Merges the passed VertexData into the current one.
         * Returns the modified VertexData.
         */
        merge(other: VertexData): VertexData;
        private _mergeElement(source, other);
        private _validate();
        /**
         * Serializes the VertexData.
         * Returns a serialized object.
         */
        serialize(): any;
        /**
         * Returns the object VertexData associated to the passed mesh.
         */
        static ExtractFromMesh(mesh: Mesh, copyWhenShared?: boolean, forceCopy?: boolean): VertexData;
        /**
         * Returns the object VertexData associated to the passed geometry.
         */
        static ExtractFromGeometry(geometry: Geometry, copyWhenShared?: boolean, forceCopy?: boolean): VertexData;
        private static _ExtractFrom(meshOrGeometry, copyWhenShared?, forceCopy?);
        /**
         * Creates the vertexData of the Ribbon.
         */
        static CreateRibbon(options: {
            pathArray: Vector3[][];
            closeArray?: boolean;
            closePath?: boolean;
            offset?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            invertUV?: boolean;
            uvs?: Vector2[];
            colors?: Color4[];
        }): VertexData;
        /**
         * Creates the VertexData of the Box.
         */
        static CreateBox(options: {
            size?: number;
            width?: number;
            height?: number;
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Sphere.
         */
        static CreateSphere(options: {
            segments?: number;
            diameter?: number;
            diameterX?: number;
            diameterY?: number;
            diameterZ?: number;
            arc?: number;
            slice?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Cylinder or Cone.
         */
        static CreateCylinder(options: {
            height?: number;
            diameterTop?: number;
            diameterBottom?: number;
            diameter?: number;
            tessellation?: number;
            subdivisions?: number;
            arc?: number;
            faceColors?: Color4[];
            faceUV?: Vector4[];
            hasRings?: boolean;
            enclose?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Torus.
         */
        static CreateTorus(options: {
            diameter?: number;
            thickness?: number;
            tessellation?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the LineSystem.
         */
        static CreateLineSystem(options: {
            lines: Vector3[][];
            colors?: Nullable<Color4[][]>;
        }): VertexData;
        /**
         * Create the VertexData of the DashedLines.
         */
        static CreateDashedLines(options: {
            points: Vector3[];
            dashSize?: number;
            gapSize?: number;
            dashNb?: number;
        }): VertexData;
        /**
         * Creates the VertexData of the Ground.
         */
        static CreateGround(options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            subdivisionsX?: number;
            subdivisionsY?: number;
        }): VertexData;
        /**
         * Creates the VertexData of the TiledGround.
         */
        static CreateTiledGround(options: {
            xmin: number;
            zmin: number;
            xmax: number;
            zmax: number;
            subdivisions?: {
                w: number;
                h: number;
            };
            precision?: {
                w: number;
                h: number;
            };
        }): VertexData;
        /**
         * Creates the VertexData of the Ground designed from a heightmap.
         */
        static CreateGroundFromHeightMap(options: {
            width: number;
            height: number;
            subdivisions: number;
            minHeight: number;
            maxHeight: number;
            colorFilter: Color3;
            buffer: Uint8Array;
            bufferWidth: number;
            bufferHeight: number;
        }): VertexData;
        /**
         * Creates the VertexData of the Plane.
         */
        static CreatePlane(options: {
            size?: number;
            width?: number;
            height?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Disc or regular Polygon.
         */
        static CreateDisc(options: {
            radius?: number;
            tessellation?: number;
            arc?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Re-creates the VertexData of the Polygon for sideOrientation.
         */
        static CreatePolygon(polygon: Mesh, sideOrientation: number, fUV?: Vector4[], fColors?: Color4[], frontUVs?: Vector4, backUVs?: Vector4): VertexData;
        /**
         * Creates the VertexData of the IcoSphere.
         */
        static CreateIcoSphere(options: {
            radius?: number;
            radiusX?: number;
            radiusY?: number;
            radiusZ?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Polyhedron.
         */
        static CreatePolyhedron(options: {
            type?: number;
            size?: number;
            sizeX?: number;
            sizeY?: number;
            sizeZ?: number;
            custom?: any;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            flat?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * Creates the VertexData of the Torus Knot.
         */
        static CreateTorusKnot(options: {
            radius?: number;
            tube?: number;
            radialSegments?: number;
            tubularSegments?: number;
            p?: number;
            q?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }): VertexData;
        /**
         * @param {any} - positions (number[] or Float32Array)
         * @param {any} - indices   (number[] or Uint16Array)
         * @param {any} - normals   (number[] or Float32Array)
         * options (optional) :
         * facetPositions : optional array of facet positions (vector3)
         * facetNormals : optional array of facet normals (vector3)
         * facetPartitioning : optional partitioning array. facetPositions is required for facetPartitioning computation
         * subDiv : optional partitioning data about subdivsions on  each axis (int), required for facetPartitioning computation
         * ratio : optional partitioning ratio / bounding box, required for facetPartitioning computation
         * bbSize : optional bounding box size data, required for facetPartitioning computation
         * bInfo : optional bounding info, required for facetPartitioning computation
         * useRightHandedSystem: optional boolean to for right handed system computation
         * depthSort : optional boolean to enable the facet depth sort computation
         * distanceTo : optional Vector3 to compute the facet depth from this location
         * depthSortedFacets : optional array of depthSortedFacets to store the facet distances from the reference location
         */
        static ComputeNormals(positions: any, indices: any, normals: any, options?: {
            facetNormals?: any;
            facetPositions?: any;
            facetPartitioning?: any;
            ratio?: number;
            bInfo?: any;
            bbSize?: Vector3;
            subDiv?: any;
            useRightHandedSystem?: boolean;
            depthSort?: boolean;
            distanceTo?: Vector3;
            depthSortedFacets?: any;
        }): void;
        private static _ComputeSides(sideOrientation, positions, indices, normals, uvs, frontUVs?, backUVs?);
        /**
         * Creates a new VertexData from the imported parameters.
         */
        static ImportVertexData(parsedVertexData: any, geometry: Geometry): void;
    }
}

declare module BABYLON {
    class MeshBuilder {
        private static updateSideOrientation(orientation?);
        /**
         * Creates a box mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#box
         * The parameter `size` sets the size (float) of each box side (default 1).
         * You can set some different box dimensions by using the parameters `width`, `height` and `depth` (all by default have the same value than `size`).
         * You can set different colors and different images to each box side by using the parameters `faceColors` (an array of 6 Color3 elements) and `faceUV` (an array of 6 Vector4 elements).
         * Please read this tutorial : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateBox(name: string, options: {
            size?: number;
            width?: number;
            height?: number;
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a sphere mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#sphere
         * The parameter `diameter` sets the diameter size (float) of the sphere (default 1).
         * You can set some different sphere dimensions, for instance to build an ellipsoid, by using the parameters `diameterX`, `diameterY` and `diameterZ` (all by default have the same value than `diameter`).
         * The parameter `segments` sets the sphere number of horizontal stripes (positive integer, default 32).
         * You can create an unclosed sphere with the parameter `arc` (positive float, default 1), valued between 0 and 1, what is the ratio of the circumference (latitude) : 2 x PI x ratio
         * You can create an unclosed sphere on its height with the parameter `slice` (positive float, default1), valued between 0 and 1, what is the height ratio (longitude).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateSphere(name: string, options: {
            segments?: number;
            diameter?: number;
            diameterX?: number;
            diameterY?: number;
            diameterZ?: number;
            arc?: number;
            slice?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene: any): Mesh;
        /**
         * Creates a plane polygonal mesh.  By default, this is a disc.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#disc
         * The parameter `radius` sets the radius size (float) of the polygon (default 0.5).
         * The parameter `tessellation` sets the number of polygon sides (positive integer, default 64). So a tessellation valued to 3 will build a triangle, to 4 a square, etc.
         * You can create an unclosed polygon with the parameter `arc` (positive float, default 1), valued between 0 and 1, what is the ratio of the circumference : 2 x PI x ratio
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDisc(name: string, options: {
            radius?: number;
            tessellation?: number;
            arc?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a sphere based upon an icosahedron with 20 triangular faces which can be subdivided.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#icosphere
         * The parameter `radius` sets the radius size (float) of the icosphere (default 1).
         * You can set some different icosphere dimensions, for instance to build an ellipsoid, by using the parameters `radiusX`, `radiusY` and `radiusZ` (all by default have the same value than `radius`).
         * The parameter `subdivisions` sets the number of subdivisions (postive integer, default 4). The more subdivisions, the more faces on the icosphere whatever its size.
         * The parameter `flat` (boolean, default true) gives each side its own normals. Set it to false to get a smooth continuous light reflection on the surface.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateIcoSphere(name: string, options: {
            radius?: number;
            radiusX?: number;
            radiusY?: number;
            radiusZ?: number;
            flat?: boolean;
            subdivisions?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a ribbon mesh.
         * The ribbon is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         *
         * Please read this full tutorial to understand how to design a ribbon : http://doc.babylonjs.com/tutorials/Ribbon_Tutorial
         * The parameter `pathArray` is a required array of paths, what are each an array of successive Vector3. The pathArray parameter depicts the ribbon geometry.
         * The parameter `closeArray` (boolean, default false) creates a seam between the first and the last paths of the path array.
         * The parameter `closePath` (boolean, default false) creates a seam between the first and the last points of each path of the path array.
         * The parameter `offset` (positive integer, default : rounded half size of the pathArray length), is taken in account only if the `pathArray` is containing a single path.
         * It's the offset to join the points from the same path. Ex : offset = 10 means the point 1 is joined to the point 11.
         * The optional parameter `instance` is an instance of an existing Ribbon object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#ribbon
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * The parameter `uvs` is an optional flat array of `Vector2` to update/set each ribbon vertex with its own custom UV values instead of the computed ones.
         * The parameters `colors` is an optional flat array of `Color4` to set/update each ribbon vertex with its own custom color values.
         * Note that if you use the parameters `uvs` or `colors`, the passed arrays must be populated with the right number of elements, it is to say the number of ribbon vertices. Remember that
         * if you set `closePath` to `true`, there's one extra vertex per path in the geometry.
         * Moreover, you can use the parameter `color` with `instance` (to update the ribbon), only if you previously used it at creation time.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateRibbon(name: string, options: {
            pathArray: Vector3[][];
            closeArray?: boolean;
            closePath?: boolean;
            offset?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
            uvs?: Vector2[];
            colors?: Color4[];
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates a cylinder or a cone mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#cylinder-or-cone
         * The parameter `height` sets the height size (float) of the cylinder/cone (float, default 2).
         * The parameter `diameter` sets the diameter of the top and bottom cap at once (float, default 1).
         * The parameters `diameterTop` and `diameterBottom` overwrite the parameter `diameter` and set respectively the top cap and bottom cap diameter (floats, default 1). The parameter "diameterBottom" can't be zero.
         * The parameter `tessellation` sets the number of cylinder sides (positive integer, default 24). Set it to 3 to get a prism for instance.
         * The parameter `subdivisions` sets the number of rings along the cylinder height (positive integer, default 1).
         * The parameter `hasRings` (boolean, default false) makes the subdivisions independent from each other, so they become different faces.
         * The parameter `enclose`  (boolean, default false) adds two extra faces per subdivision to a sliced cylinder to close it around its height axis.
         * The parameter `arc` (float, default 1) is the ratio (max 1) to apply to the circumference to slice the cylinder.
         * You can set different colors and different images to each box side by using the parameters `faceColors` (an array of n Color3 elements) and `faceUV` (an array of n Vector4 elements).
         * The value of n is the number of cylinder faces. If the cylinder has only 1 subdivisions, n equals : top face + cylinder surface + bottom face = 3
         * Now, if the cylinder has 5 independent subdivisions (hasRings = true), n equals : top face + 5 stripe surfaces + bottom face = 2 + 5 = 7
         * Finally, if the cylinder has 5 independent subdivisions and is enclose, n equals : top face + 5 x (stripe surface + 2 closing faces) + bottom face = 2 + 5 * 3 = 17
         * Each array (color or UVs) is always ordered the same way : the first element is the bottom cap, the last element is the top cap. The other elements are each a ring surface.
         * If `enclose` is false, a ring surface is one element.
         * If `enclose` is true, a ring surface is 3 successive elements in the array : the tubular surface, then the two closing faces.
         * Example how to set colors and textures on a sliced cylinder : http://www.html5gamedevs.com/topic/17945-creating-a-closed-slice-of-a-cylinder/#comment-106379
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateCylinder(name: string, options: {
            height?: number;
            diameterTop?: number;
            diameterBottom?: number;
            diameter?: number;
            tessellation?: number;
            subdivisions?: number;
            arc?: number;
            faceColors?: Color4[];
            faceUV?: Vector4[];
            updatable?: boolean;
            hasRings?: boolean;
            enclose?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a torus mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#torus
         * The parameter `diameter` sets the diameter size (float) of the torus (default 1).
         * The parameter `thickness` sets the diameter size of the tube of the torus (float, default 0.5).
         * The parameter `tessellation` sets the number of torus sides (postive integer, default 16).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorus(name: string, options: {
            diameter?: number;
            thickness?: number;
            tessellation?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a torus knot mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#torus-knot
         * The parameter `radius` sets the global radius size (float) of the torus knot (default 2).
         * The parameter `radialSegments` sets the number of sides on each tube segments (positive integer, default 32).
         * The parameter `tubularSegments` sets the number of tubes to decompose the knot into (positive integer, default 32).
         * The parameters `p` and `q` are the number of windings on each axis (positive integers, default 2 and 3).
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTorusKnot(name: string, options: {
            radius?: number;
            tube?: number;
            radialSegments?: number;
            tubularSegments?: number;
            p?: number;
            q?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: any): Mesh;
        /**
         * Creates a line system mesh.
         * A line system is a pool of many lines gathered in a single mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#linesystem
         * A line system mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of lines as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineSystem to this static function.
         * The parameter `lines` is an array of lines, each line being an array of successive Vector3.
         * The optional parameter `instance` is an instance of an existing LineSystem object to be updated with the passed `lines` parameter. The way to update it is the same than for
         * The optional parameter `colors` is an array of line colors, each line colors being an array of successive Color4, one per line point.
         * The optional parameter `useVertexAlpha' is to be set to `false` (default `true`) when you don't need the alpha blending (faster).
         * updating a simple Line mesh, you just need to update every line in the `lines` array : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only line point positions can change, not the number of points, neither the number of lines.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLineSystem(name: string, options: {
            lines: Vector3[][];
            updatable: boolean;
            instance: Nullable<LinesMesh>;
            colors?: Nullable<Color4[][]>;
            useVertexAlpha?: boolean;
        }, scene: Nullable<Scene>): LinesMesh;
        /**
         * Creates a line mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#lines
         * A line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * The optional parameter `colors` is an array of successive Color4, one per line point.
         * The optional parameter `useVertexAlpha' is to be set to `false` (default `true`) when you don't need alpha blending (faster).
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLines(name: string, options: {
            points: Vector3[];
            updatable: boolean;
            instance: Nullable<LinesMesh>;
            colors?: Color4[];
            useVertexAlpha?: boolean;
        }, scene?: Nullable<Scene>): LinesMesh;
        /**
         * Creates a dashed line mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#dashed-lines
         * A dashed line mesh is considered as a parametric shape since it has no predefined original shape. Its shape is determined by the passed array of points as an input parameter.
         * Like every other parametric shape, it is dynamically updatable by passing an existing instance of LineMesh to this static function.
         * The parameter `points` is an array successive Vector3.
         * The parameter `dashNb` is the intended total number of dashes (positive integer, default 200).
         * The parameter `dashSize` is the size of the dashes relatively the dash number (positive float, default 3).
         * The parameter `gapSize` is the size of the gap between two successive dashes relatively the dash number (positive float, default 1).
         * The optional parameter `instance` is an instance of an existing LineMesh object to be updated with the passed `points` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#lines-and-dashedlines
         * When updating an instance, remember that only point positions can change, not the number of points.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateDashedLines(name: string, options: {
            points: Vector3[];
            dashSize?: number;
            gapSize?: number;
            dashNb?: number;
            updatable?: boolean;
            instance?: LinesMesh;
        }, scene?: Nullable<Scene>): LinesMesh;
        /**
         * Creates an extruded shape mesh.
         * The extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#extruded-shapes
         *
         * Please read this full tutorial to understand how to design an extruded shape : http://doc.babylonjs.com/tutorials/Parametric_Shapes#extrusion
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotation` (float, default 0 radians) is the angle value to rotate the shape each step (each path point), from the former step (so rotation added each step) along the curve.
         * The parameter `scale` (float, default 1) is the value to scale the shape.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShape(name: string, options: {
            shape: Vector3[];
            path: Vector3[];
            scale?: number;
            rotation?: number;
            cap?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
        }, scene?: Nullable<Scene>): Mesh;
        /**
         * Creates an custom extruded shape mesh.
         * The custom extrusion is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         * tuto :http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#custom-extruded-shapes
         *
         * Please read this full tutorial to understand how to design a custom extruded shape : http://doc.babylonjs.com/tutorials/Parametric_Shapes#extrusion
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be extruded in its local space : the shape must be designed in the xOy plane and will be
         * extruded along the Z axis.
         * The parameter `path` is a required array of successive Vector3. This is the axis curve the shape is extruded along.
         * The parameter `rotationFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var rotationFunction = function(i, distance) {
         *     // do things
         *     return rotationValue; }
         * ```
         * It must returns a float value that will be the rotation in radians applied to the shape on each path point.
         * The parameter `scaleFunction` (JS function) is a custom Javascript function called on each path point. This function is passed the position i of the point in the path
         * and the distance of this point from the begining of the path :
         * ```javascript
         * var scaleFunction = function(i, distance) {
         *     // do things
         *     return scaleValue;}
         * ```
         * It must returns a float value that will be the scale value applied to the shape on each path point.
         * The parameter `ribbonClosePath` (boolean, default false) forces the extrusion underlying ribbon to close all the paths in its `pathArray`.
         * The parameter `ribbonCloseArray` (boolean, default false) forces the extrusion underlying ribbon to close its `pathArray`.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing ExtrudedShape object to be updated with the passed `shape`, `path`, `scale` or `rotation` parameters : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#extruded-shape
         * Remember you can only change the shape or path point positions, not their number when updating an extruded shape.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static ExtrudeShapeCustom(name: string, options: {
            shape: Vector3[];
            path: Vector3[];
            scaleFunction?: any;
            rotationFunction?: any;
            ribbonCloseArray?: boolean;
            ribbonClosePath?: boolean;
            cap?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates lathe mesh.
         * The lathe is a shape with a symetry axis : a 2D model shape is rotated around this axis to design the lathe.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#lathe
         *
         * The parameter `shape` is a required array of successive Vector3. This array depicts the shape to be rotated in its local space : the shape must be designed in the xOy plane and will be
         * rotated around the Y axis. It's usually a 2D shape, so the Vector3 z coordinates are often set to zero.
         * The parameter `radius` (positive float, default 1) is the radius value of the lathe.
         * The parameter `tessellation` (positive integer, default 64) is the side number of the lathe.
         * The parameter `arc` (positive float, default 1) is the ratio of the lathe. 0.5 builds for instance half a lathe, so an opened shape.
         * The parameter `closed` (boolean, default true) opens/closes the lathe circumference. This should be set to false when used with the parameter "arc".
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateLathe(name: string, options: {
            shape: Vector3[];
            radius?: number;
            tessellation?: number;
            arc?: number;
            closed?: boolean;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            cap?: number;
            invertUV?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a plane mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#plane
         * The parameter `size` sets the size (float) of both sides of the plane at once (default 1).
         * You can set some different plane dimensions by using the parameters `width` and `height` (both by default have the same value than `size`).
         * The parameter `sourcePlane` is a Plane instance. It builds a mesh plane from a Math plane.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePlane(name: string, options: {
            size?: number;
            width?: number;
            height?: number;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            updatable?: boolean;
            sourcePlane?: Plane;
        }, scene: Scene): Mesh;
        /**
         * Creates a ground mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#plane
         * The parameters `width` and `height` (floats, default 1) set the width and height sizes of the ground.
         * The parameter `subdivisions` (positive integer) sets the number of subdivisions per side.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGround(name: string, options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            subdivisionsX?: number;
            subdivisionsY?: number;
            updatable?: boolean;
        }, scene: any): Mesh;
        /**
         * Creates a tiled ground mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#tiled-ground
         * The parameters `xmin` and `xmax` (floats, default -1 and 1) set the ground minimum and maximum X coordinates.
         * The parameters `zmin` and `zmax` (floats, default -1 and 1) set the ground minimum and maximum Z coordinates.
         * The parameter `subdivisions` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 6, h: 6}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height. Each subdivision is called a tile.
         * The parameter `precision` is a javascript object `{w: positive integer, h: positive integer}` (default `{w: 2, h: 2}`). `w` and `h` are the
         * numbers of subdivisions on the ground width and height of each tile.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTiledGround(name: string, options: {
            xmin: number;
            zmin: number;
            xmax: number;
            zmax: number;
            subdivisions?: {
                w: number;
                h: number;
            };
            precision?: {
                w: number;
                h: number;
            };
            updatable?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a ground mesh from a height map.
         * tuto : http://doc.babylonjs.com/tutorials/14._Height_Map
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#ground-from-a-height-map
         * The parameter `url` sets the URL of the height map image Resource.
         * The parameters `width` and `height` (positive floats, default 10) set the ground width and height sizes.
         * The parameter `subdivisions` (positive integer, default 1) sets the number of subdivision per side.
         * The parameter `minHeight` (float, default 0) is the minimum altitude on the ground.
         * The parameter `maxHeight` (float, default 1) is the maximum altitude on the ground.
         * The parameter `colorFilter` (optional Color3, default (0.3, 0.59, 0.11) ) is the filter to apply to the image pixel colors to compute the height.
         * The parameter `onReady` is a javascript callback function that will be called  once the mesh is just built (the height map download can last some time).
         * This function is passed the newly built mesh :
         * ```javascript
         * function(mesh) { // do things
         *     return; }
         * ```
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateGroundFromHeightMap(name: string, url: string, options: {
            width?: number;
            height?: number;
            subdivisions?: number;
            minHeight?: number;
            maxHeight?: number;
            colorFilter?: Color3;
            updatable?: boolean;
            onReady?: (mesh: GroundMesh) => void;
        }, scene: Scene): GroundMesh;
        /**
         * Creates a polygon mesh.
         * The polygon's shape will depend on the input parameters and is constructed parallel to a ground mesh.
         * The parameter `shape` is a required array of successive Vector3 representing the corners of the polygon in th XoZ plane, that is y = 0 for all vectors.
         * You can set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Remember you can only change the shape positions, not their number when updating a polygon.
         */
        static CreatePolygon(name: string, options: {
            shape: Vector3[];
            holes?: Vector3[][];
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: Scene): Mesh;
        /**
         * Creates an extruded polygon mesh, with depth in the Y direction.
         * You can set different colors and different images to the top, bottom and extruded side by using the parameters `faceColors` (an array of 3 Color3 elements) and `faceUV` (an array of 3 Vector4 elements).
         * Please read this tutorial : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
        */
        static ExtrudePolygon(name: string, options: {
            shape: Vector3[];
            holes?: Vector3[][];
            depth?: number;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: Scene): Mesh;
        /**
         * Creates a tube mesh.
         * The tube is a parametric shape :  http://doc.babylonjs.com/tutorials/Parametric_Shapes.  It has no predefined shape. Its final shape will depend on the input parameters.
         *
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#tube
         * The parameter `path` is a required array of successive Vector3. It is the curve used as the axis of the tube.
         * The parameter `radius` (positive float, default 1) sets the tube radius size.
         * The parameter `tessellation` (positive float, default 64) is the number of sides on the tubular surface.
         * The parameter `radiusFunction` (javascript function, default null) is a vanilla javascript function. If it is not null, it overwrittes the parameter `radius`.
         * This function is called on each point of the tube path and is passed the index `i` of the i-th point and the distance of this point from the first point of the path.
         * It must return a radius value (positive float) :
         * ```javascript
         * var radiusFunction = function(i, distance) {
         *     // do things
         *     return radius; }
         * ```
         * The parameter `arc` (positive float, maximum 1, default 1) is the ratio to apply to the tube circumference : 2 x PI x arc.
         * The parameter `cap` sets the way the extruded shape is capped. Possible values : BABYLON.Mesh.NO_CAP (default), BABYLON.Mesh.CAP_START, BABYLON.Mesh.CAP_END, BABYLON.Mesh.CAP_ALL
         * The optional parameter `instance` is an instance of an existing Tube object to be updated with the passed `pathArray` parameter : http://doc.babylonjs.com/tutorials/How_to_dynamically_morph_a_mesh#tube
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The optional parameter `invertUV` (boolean, default false) swaps in the geometry the U and V coordinates to apply a texture.
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreateTube(name: string, options: {
            path: Vector3[];
            radius?: number;
            tessellation?: number;
            radiusFunction?: {
                (i: number, distance: number): number;
            };
            cap?: number;
            arc?: number;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
            instance?: Mesh;
            invertUV?: boolean;
        }, scene: Scene): Mesh;
        /**
         * Creates a polyhedron mesh.
         *
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#polyhedron
         * The parameter `type` (positive integer, max 14, default 0) sets the polyhedron type to build among the 15 embbeded types. Please refer to the type sheet in the tutorial
         *  to choose the wanted type.
         * The parameter `size` (positive float, default 1) sets the polygon size.
         * You can overwrite the `size` on each dimension bu using the parameters `sizeX`, `sizeY` or `sizeZ` (positive floats, default to `size` value).
         * You can build other polyhedron types than the 15 embbeded ones by setting the parameter `custom` (`polyhedronObject`, default null). If you set the parameter `custom`, this overwrittes the parameter `type`.
         * A `polyhedronObject` is a formatted javascript object. You'll find a full file with pre-set polyhedra here : https://github.com/BabylonJS/Extensions/tree/master/Polyhedron
         * You can set the color and the UV of each side of the polyhedron with the parameters `faceColors` (Color4, default `(1, 1, 1, 1)`) and faceUV (Vector4, default `(0, 0, 1, 1)`).
         * To understand how to set `faceUV` or `faceColors`, please read this by considering the right number of faces of your polyhedron, instead of only 6 for the box : http://doc.babylonjs.com/tutorials/CreateBox_Per_Face_Textures_And_Colors
         * The parameter `flat` (boolean, default true). If set to false, it gives the polyhedron a single global face, so less vertices and shared normals. In this case, `faceColors` and `faceUV` are ignored.
         * You can also set the mesh side orientation with the values : BABYLON.Mesh.FRONTSIDE (default), BABYLON.Mesh.BACKSIDE or BABYLON.Mesh.DOUBLESIDE
         * If you create a double-sided mesh, you can choose what parts of the texture image to crop and stick respectively on the front and the back sides with the parameters `frontUVs` and `backUVs` (Vector4).
         * Detail here : http://doc.babylonjs.com/tutorials/02._Discover_Basic_Elements#side-orientation
         * The mesh can be set to updatable with the boolean parameter `updatable` (default false) if its internal geometry is supposed to change once created.
         */
        static CreatePolyhedron(name: string, options: {
            type?: number;
            size?: number;
            sizeX?: number;
            sizeY?: number;
            sizeZ?: number;
            custom?: any;
            faceUV?: Vector4[];
            faceColors?: Color4[];
            flat?: boolean;
            updatable?: boolean;
            sideOrientation?: number;
            frontUVs?: Vector4;
            backUVs?: Vector4;
        }, scene: Scene): Mesh;
        /**
         * Creates a decal mesh.
         * tuto : http://doc.babylonjs.com/tutorials/Mesh_CreateXXX_Methods_With_Options_Parameter#decals
         * A decal is a mesh usually applied as a model onto the surface of another mesh. So don't forget the parameter `sourceMesh` depicting the decal.
         * The parameter `position` (Vector3, default `(0, 0, 0)`) sets the position of the decal in World coordinates.
         * The parameter `normal` (Vector3, default `Vector3.Up`) sets the normal of the mesh where the decal is applied onto in World coordinates.
         * The parameter `size` (Vector3, default `(1, 1, 1)`) sets the decal scaling.
         * The parameter `angle` (float in radian, default 0) sets the angle to rotate the decal.
         */
        static CreateDecal(name: string, sourceMesh: AbstractMesh, options: {
            position?: Vector3;
            normal?: Vector3;
            size?: Vector3;
            angle?: number;
        }): Mesh;
        private static _ExtrudeShapeGeneric(name, shape, curve, scale, rotation, scaleFunction, rotateFunction, rbCA, rbCP, cap, custom, scene, updtbl, side, instance, invertUV, frontUVs, backUVs);
    }
}

declare module BABYLON {
    class MeshLODLevel {
        distance: number;
        mesh: Mesh;
        constructor(distance: number, mesh: Mesh);
    }
}

declare module BABYLON {
    /**
     * A simplifier interface for future simplification implementations.
     */
    interface ISimplifier {
        /**
         * Simplification of a given mesh according to the given settings.
         * Since this requires computation, it is assumed that the function runs async.
         * @param settings The settings of the simplification, including quality and distance
         * @param successCallback A callback that will be called after the mesh was simplified.
         * @param errorCallback in case of an error, this callback will be called. optional.
         */
        simplify(settings: ISimplificationSettings, successCallback: (simplifiedMeshes: Mesh) => void, errorCallback?: () => void): void;
    }
    /**
     * Expected simplification settings.
     * Quality should be between 0 and 1 (1 being 100%, 0 being 0%);
     */
    interface ISimplificationSettings {
        quality: number;
        distance: number;
        optimizeMesh?: boolean;
    }
    class SimplificationSettings implements ISimplificationSettings {
        quality: number;
        distance: number;
        optimizeMesh: boolean | undefined;
        constructor(quality: number, distance: number, optimizeMesh?: boolean | undefined);
    }
    interface ISimplificationTask {
        settings: Array<ISimplificationSettings>;
        simplificationType: SimplificationType;
        mesh: Mesh;
        successCallback?: () => void;
        parallelProcessing: boolean;
    }
    class SimplificationQueue {
        private _simplificationArray;
        running: boolean;
        constructor();
        addTask(task: ISimplificationTask): void;
        executeNext(): void;
        runSimplification(task: ISimplificationTask): void;
        private getSimplifier(task);
    }
    /**
     * The implemented types of simplification.
     * At the moment only Quadratic Error Decimation is implemented.
     */
    enum SimplificationType {
        QUADRATIC = 0,
    }
    class DecimationTriangle {
        vertices: Array<DecimationVertex>;
        normal: Vector3;
        error: Array<number>;
        deleted: boolean;
        isDirty: boolean;
        borderFactor: number;
        deletePending: boolean;
        originalOffset: number;
        constructor(vertices: Array<DecimationVertex>);
    }
    class DecimationVertex {
        position: Vector3;
        id: number;
        q: QuadraticMatrix;
        isBorder: boolean;
        triangleStart: number;
        triangleCount: number;
        originalOffsets: Array<number>;
        constructor(position: Vector3, id: number);
        updatePosition(newPosition: Vector3): void;
    }
    class QuadraticMatrix {
        data: Array<number>;
        constructor(data?: Array<number>);
        det(a11: number, a12: number, a13: number, a21: number, a22: number, a23: number, a31: number, a32: number, a33: number): number;
        addInPlace(matrix: QuadraticMatrix): void;
        addArrayInPlace(data: Array<number>): void;
        add(matrix: QuadraticMatrix): QuadraticMatrix;
        static FromData(a: number, b: number, c: number, d: number): QuadraticMatrix;
        static DataFromNumbers(a: number, b: number, c: number, d: number): number[];
    }
    class Reference {
        vertexId: number;
        triangleId: number;
        constructor(vertexId: number, triangleId: number);
    }
    /**
     * An implementation of the Quadratic Error simplification algorithm.
     * Original paper : http://www1.cs.columbia.edu/~cs4162/html05s/garland97.pdf
     * Ported mostly from QSlim and http://voxels.blogspot.de/2014/05/quadric-mesh-simplification-with-source.html to babylon JS
     * @author RaananW
     */
    class QuadraticErrorSimplification implements ISimplifier {
        private _mesh;
        private triangles;
        private vertices;
        private references;
        private _reconstructedMesh;
        syncIterations: number;
        aggressiveness: number;
        decimationIterations: number;
        boundingBoxEpsilon: number;
        constructor(_mesh: Mesh);
        simplify(settings: ISimplificationSettings, successCallback: (simplifiedMesh: Mesh) => void): void;
        private runDecimation(settings, submeshIndex, successCallback);
        private initWithMesh(submeshIndex, callback, optimizeMesh?);
        private init(callback);
        private reconstructMesh(submeshIndex);
        private initDecimatedMesh();
        private isFlipped(vertex1, vertex2, point, deletedArray, borderFactor, delTr);
        private updateTriangles(origVertex, vertex, deletedArray, deletedTriangles);
        private identifyBorder();
        private updateMesh(identifyBorders?);
        private vertexError(q, point);
        private calculateError(vertex1, vertex2, pointResult?, normalResult?, uvResult?, colorResult?);
    }
}

declare module BABYLON {
    class Polygon {
        static Rectangle(xmin: number, ymin: number, xmax: number, ymax: number): Vector2[];
        static Circle(radius: number, cx?: number, cy?: number, numberOfSides?: number): Vector2[];
        static Parse(input: string): Vector2[];
        static StartingAt(x: number, y: number): Path2;
    }
    class PolygonMeshBuilder {
        private _points;
        private _outlinepoints;
        private _holes;
        private _name;
        private _scene;
        private _epoints;
        private _eholes;
        private _addToepoint(points);
        constructor(name: string, contours: Path2, scene: Scene);
        constructor(name: string, contours: Vector2[], scene: Scene);
        addHole(hole: Vector2[]): PolygonMeshBuilder;
        build(updatable?: boolean, depth?: number): Mesh;
        private addSide(positions, normals, uvs, indices, bounds, points, depth, flip);
    }
}

declare module BABYLON {
    class BaseSubMesh {
        _materialDefines: Nullable<MaterialDefines>;
        _materialEffect: Nullable<Effect>;
        readonly effect: Nullable<Effect>;
        setEffect(effect: Nullable<Effect>, defines?: Nullable<MaterialDefines>): void;
    }
    class SubMesh extends BaseSubMesh implements ICullable {
        materialIndex: number;
        verticesStart: number;
        verticesCount: number;
        indexStart: number;
        indexCount: number;
        linesIndexCount: number;
        private _mesh;
        private _renderingMesh;
        private _boundingInfo;
        private _linesIndexBuffer;
        _lastColliderWorldVertices: Nullable<Vector3[]>;
        _trianglePlanes: Plane[];
        _lastColliderTransformMatrix: Matrix;
        _renderId: number;
        _alphaIndex: number;
        _distanceToCamera: number;
        _id: number;
        private _currentMaterial;
        static AddToMesh(materialIndex: number, verticesStart: number, verticesCount: number, indexStart: number, indexCount: number, mesh: AbstractMesh, renderingMesh?: Mesh, createBoundingBox?: boolean): SubMesh;
        constructor(materialIndex: number, verticesStart: number, verticesCount: number, indexStart: number, indexCount: number, mesh: AbstractMesh, renderingMesh?: Mesh, createBoundingBox?: boolean);
        readonly IsGlobal: boolean;
        /**
         * Returns the submesh BoudingInfo object.
         */
        getBoundingInfo(): BoundingInfo;
        /**
         * Sets the submesh BoundingInfo.
         * Return the SubMesh.
         */
        setBoundingInfo(boundingInfo: BoundingInfo): SubMesh;
        /**
         * Returns the mesh of the current submesh.
         */
        getMesh(): AbstractMesh;
        /**
         * Returns the rendering mesh of the submesh.
         */
        getRenderingMesh(): Mesh;
        /**
         * Returns the submesh material.
         */
        getMaterial(): Nullable<Material>;
        /**
         * Sets a new updated BoundingInfo object to the submesh.
         * Returns the SubMesh.
         */
        refreshBoundingInfo(): SubMesh;
        _checkCollision(collider: Collider): boolean;
        /**
         * Updates the submesh BoundingInfo.
         * Returns the Submesh.
         */
        updateBoundingInfo(world: Matrix): SubMesh;
        /**
         * True is the submesh bounding box intersects the frustum defined by the passed array of planes.
         * Boolean returned.
         */
        isInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * True is the submesh bounding box is completely inside the frustum defined by the passed array of planes.
         * Boolean returned.
         */
        isCompletelyInFrustum(frustumPlanes: Plane[]): boolean;
        /**
         * Renders the submesh.
         * Returns it.
         */
        render(enableAlphaMode: boolean): SubMesh;
        /**
         * Returns a new Index Buffer.
         * Type returned : WebGLBuffer.
         */
        getLinesIndexBuffer(indices: IndicesArray, engine: Engine): WebGLBuffer;
        /**
         * True is the passed Ray intersects the submesh bounding box.
         * Boolean returned.
         */
        canIntersects(ray: Ray): boolean;
        /**
         * Returns an object IntersectionInfo.
         */
        intersects(ray: Ray, positions: Vector3[], indices: IndicesArray, fastCheck?: boolean): Nullable<IntersectionInfo>;
        _rebuild(): void;
        /**
         * Creates a new Submesh from the passed Mesh.
         */
        clone(newMesh: AbstractMesh, newRenderingMesh?: Mesh): SubMesh;
        /**
         * Disposes the Submesh.
         * Returns nothing.
         */
        dispose(): void;
        /**
         * Creates a new Submesh from the passed parameters :
         * - materialIndex (integer) : the index of the main mesh material.
         * - startIndex (integer) : the index where to start the copy in the mesh indices array.
         * - indexCount (integer) : the number of indices to copy then from the startIndex.
         * - mesh (Mesh) : the main mesh to create the submesh from.
         * - renderingMesh (optional Mesh) : rendering mesh.
         */
        static CreateFromIndices(materialIndex: number, startIndex: number, indexCount: number, mesh: AbstractMesh, renderingMesh?: Mesh): SubMesh;
    }
}

declare module BABYLON {
    class TransformNode extends Node {
        static BILLBOARDMODE_NONE: number;
        static BILLBOARDMODE_X: number;
        static BILLBOARDMODE_Y: number;
        static BILLBOARDMODE_Z: number;
        static BILLBOARDMODE_ALL: number;
        private _forward;
        private _forwardInverted;
        private _up;
        private _right;
        private _rightInverted;
        private _rotation;
        private _rotationQuaternion;
        protected _scaling: Vector3;
        protected _isDirty: boolean;
        private _transformToBoneReferal;
        billboardMode: number;
        scalingDeterminant: number;
        infiniteDistance: boolean;
        position: Vector3;
        _poseMatrix: Matrix;
        private _localWorld;
        _worldMatrix: Matrix;
        _worldMatrixDeterminant: number;
        private _absolutePosition;
        private _pivotMatrix;
        private _pivotMatrixInverse;
        private _postMultiplyPivotMatrix;
        protected _isWorldMatrixFrozen: boolean;
        /**
        * An event triggered after the world matrix is updated
        */
        onAfterWorldMatrixUpdateObservable: Observable<TransformNode>;
        constructor(name: string, scene?: Nullable<Scene>, isPure?: boolean);
        /**
         * Gets a string idenfifying the name of the class
         * @returns "TransformNode" string
         */
        getClassName(): string;
        /**
          * Rotation property : a Vector3 depicting the rotation value in radians around each local axis X, Y, Z.
          * If rotation quaternion is set, this Vector3 will (almost always) be the Zero vector!
          * Default : (0.0, 0.0, 0.0)
          */
        rotation: Vector3;
        /**
         * Scaling property : a Vector3 depicting the mesh scaling along each local axis X, Y, Z.
         * Default : (1.0, 1.0, 1.0)
         */
        /**
         * Scaling property : a Vector3 depicting the mesh scaling along each local axis X, Y, Z.
         * Default : (1.0, 1.0, 1.0)
        */
        scaling: Vector3;
        /**
         * Rotation Quaternion property : this a Quaternion object depicting the mesh rotation by using a unit quaternion.
         * It's null by default.
         * If set, only the rotationQuaternion is then used to compute the mesh rotation and its property `.rotation\ is then ignored and set to (0.0, 0.0, 0.0)
         */
        rotationQuaternion: Nullable<Quaternion>;
        /**
         * The forward direction of that transform in world space.
         */
        readonly forward: Vector3;
        /**
         * The up direction of that transform in world space.
         */
        readonly up: Vector3;
        /**
         * The right direction of that transform in world space.
         */
        readonly right: Vector3;
        /**
         * Returns the latest update of the World matrix
         * Returns a Matrix.
         */
        getWorldMatrix(): Matrix;
        /**
         * Returns the latest update of the World matrix determinant.
         */
        protected _getWorldMatrixDeterminant(): number;
        /**
         * Returns directly the latest state of the mesh World matrix.
         * A Matrix is returned.
         */
        readonly worldMatrixFromCache: Matrix;
        /**
         * Copies the paramater passed Matrix into the mesh Pose matrix.
         * Returns the TransformNode.
         */
        updatePoseMatrix(matrix: Matrix): TransformNode;
        /**
         * Returns the mesh Pose matrix.
         * Returned object : Matrix
         */
        getPoseMatrix(): Matrix;
        _isSynchronized(): boolean;
        _initCache(): void;
        markAsDirty(property: string): TransformNode;
        /**
         * Returns the current mesh absolute position.
         * Retuns a Vector3.
         */
        readonly absolutePosition: Vector3;
        /**
         * Sets a new matrix to apply before all other transformation
         * @param matrix defines the transform matrix
         * @returns the current TransformNode
         */
        setPreTransformMatrix(matrix: Matrix): TransformNode;
        /**
         * Sets a new pivot matrix to the current node
         * @param matrix defines the new pivot matrix to use
         * @param postMultiplyPivotMatrix defines if the pivot matrix must be cancelled in the world matrix. When this parameter is set to true (default), the inverse of the pivot matrix is also applied at the end to cancel the transformation effect
         * @returns the current TransformNode
        */
        setPivotMatrix(matrix: Matrix, postMultiplyPivotMatrix?: boolean): TransformNode;
        /**
         * Returns the mesh pivot matrix.
         * Default : Identity.
         * A Matrix is returned.
         */
        getPivotMatrix(): Matrix;
        /**
         * Prevents the World matrix to be computed any longer.
         * Returns the TransformNode.
         */
        freezeWorldMatrix(): TransformNode;
        /**
         * Allows back the World matrix computation.
         * Returns the TransformNode.
         */
        unfreezeWorldMatrix(): this;
        /**
         * True if the World matrix has been frozen.
         * Returns a boolean.
         */
        readonly isWorldMatrixFrozen: boolean;
        /**
            * Retuns the mesh absolute position in the World.
            * Returns a Vector3.
            */
        getAbsolutePosition(): Vector3;
        /**
         * Sets the mesh absolute position in the World from a Vector3 or an Array(3).
         * Returns the TransformNode.
         */
        setAbsolutePosition(absolutePosition: Vector3): TransformNode;
        /**
           * Sets the mesh position in its local space.
           * Returns the TransformNode.
           */
        setPositionWithLocalVector(vector3: Vector3): TransformNode;
        /**
         * Returns the mesh position in the local space from the current World matrix values.
         * Returns a new Vector3.
         */
        getPositionExpressedInLocalSpace(): Vector3;
        /**
         * Translates the mesh along the passed Vector3 in its local space.
         * Returns the TransformNode.
         */
        locallyTranslate(vector3: Vector3): TransformNode;
        private static _lookAtVectorCache;
        /**
         * Orients a mesh towards a target point. Mesh must be drawn facing user.
         * @param targetPoint the position (must be in same space as current mesh) to look at
         * @param yawCor optional yaw (y-axis) correction in radians
         * @param pitchCor optional pitch (x-axis) correction in radians
         * @param rollCor optional roll (z-axis) correction in radians
         * @param space the choosen space of the target
         * @returns the TransformNode.
         */
        lookAt(targetPoint: Vector3, yawCor?: number, pitchCor?: number, rollCor?: number, space?: Space): TransformNode;
        /**
          * Returns a new Vector3 what is the localAxis, expressed in the mesh local space, rotated like the mesh.
          * This Vector3 is expressed in the World space.
          */
        getDirection(localAxis: Vector3): Vector3;
        /**
         * Sets the Vector3 "result" as the rotated Vector3 "localAxis" in the same rotation than the mesh.
         * localAxis is expressed in the mesh local space.
         * result is computed in the Wordl space from the mesh World matrix.
         * Returns the TransformNode.
         */
        getDirectionToRef(localAxis: Vector3, result: Vector3): TransformNode;
        /**
         * Sets a new pivot point to the current node
         * @param point defines the new pivot point to use
         * @param space defines if the point is in world or local space (local by default)
         * @returns the current TransformNode
        */
        setPivotPoint(point: Vector3, space?: Space): TransformNode;
        /**
         * Returns a new Vector3 set with the mesh pivot point coordinates in the local space.
         */
        getPivotPoint(): Vector3;
        /**
         * Sets the passed Vector3 "result" with the coordinates of the mesh pivot point in the local space.
         * Returns the TransformNode.
         */
        getPivotPointToRef(result: Vector3): TransformNode;
        /**
         * Returns a new Vector3 set with the mesh pivot point World coordinates.
         */
        getAbsolutePivotPoint(): Vector3;
        /**
         * Sets the Vector3 "result" coordinates with the mesh pivot point World coordinates.
         * Returns the TransformNode.
         */
        getAbsolutePivotPointToRef(result: Vector3): TransformNode;
        /**
         * Defines the passed node as the parent of the current node.
         * The node will remain exactly where it is and its position / rotation will be updated accordingly
         * Returns the TransformNode.
         */
        setParent(node: Nullable<Node>): TransformNode;
        private _nonUniformScaling;
        readonly nonUniformScaling: boolean;
        _updateNonUniformScalingState(value: boolean): boolean;
        /**
         * Attach the current TransformNode to another TransformNode associated with a bone
         * @param bone Bone affecting the TransformNode
         * @param affectedTransformNode TransformNode associated with the bone
         */
        attachToBone(bone: Bone, affectedTransformNode: TransformNode): TransformNode;
        detachFromBone(): TransformNode;
        private static _rotationAxisCache;
        /**
         * Rotates the mesh around the axis vector for the passed angle (amount) expressed in radians, in the given space.
         * space (default LOCAL) can be either BABYLON.Space.LOCAL, either BABYLON.Space.WORLD.
         * Note that the property `rotationQuaternion` is then automatically updated and the property `rotation` is set to (0,0,0) and no longer used.
         * The passed axis is also normalized.
         * Returns the TransformNode.
         */
        rotate(axis: Vector3, amount: number, space?: Space): TransformNode;
        /**
         * Rotates the mesh around the axis vector for the passed angle (amount) expressed in radians, in world space.
         * Note that the property `rotationQuaternion` is then automatically updated and the property `rotation` is set to (0,0,0) and no longer used.
         * The passed axis is also normalized.
         * Returns the TransformNode.
         * Method is based on http://www.euclideanspace.com/maths/geometry/affine/aroundPoint/index.htm
         */
        rotateAround(point: Vector3, axis: Vector3, amount: number): TransformNode;
        /**
         * Translates the mesh along the axis vector for the passed distance in the given space.
         * space (default LOCAL) can be either BABYLON.Space.LOCAL, either BABYLON.Space.WORLD.
         * Returns the TransformNode.
         */
        translate(axis: Vector3, distance: number, space?: Space): TransformNode;
        /**
         * Adds a rotation step to the mesh current rotation.
         * x, y, z are Euler angles expressed in radians.
         * This methods updates the current mesh rotation, either mesh.rotation, either mesh.rotationQuaternion if it's set.
         * This means this rotation is made in the mesh local space only.
         * It's useful to set a custom rotation order different from the BJS standard one YXZ.
         * Example : this rotates the mesh first around its local X axis, then around its local Z axis, finally around its local Y axis.
         * ```javascript
         * mesh.addRotation(x1, 0, 0).addRotation(0, 0, z2).addRotation(0, 0, y3);
         * ```
         * Note that `addRotation()` accumulates the passed rotation values to the current ones and computes the .rotation or .rotationQuaternion updated values.
         * Under the hood, only quaternions are used. So it's a little faster is you use .rotationQuaternion because it doesn't need to translate them back to Euler angles.
         * Returns the TransformNode.
         */
        addRotation(x: number, y: number, z: number): TransformNode;
        /**
         * Computes the mesh World matrix and returns it.
         * If the mesh world matrix is frozen, this computation does nothing more than returning the last frozen values.
         * If the parameter `force` is let to `false` (default), the current cached World matrix is returned.
         * If the parameter `force`is set to `true`, the actual computation is done.
         * Returns the mesh World Matrix.
         */
        computeWorldMatrix(force?: boolean): Matrix;
        protected _afterComputeWorldMatrix(): void;
        /**
        * If you'd like to be called back after the mesh position, rotation or scaling has been updated.
        * @param func: callback function to add
        *
        * Returns the TransformNode.
        */
        registerAfterWorldMatrixUpdate(func: (mesh: TransformNode) => void): TransformNode;
        /**
         * Removes a registered callback function.
         * Returns the TransformNode.
         */
        unregisterAfterWorldMatrixUpdate(func: (mesh: TransformNode) => void): TransformNode;
        /**
         * Clone the current transform node
         * Returns the new transform node
         * @param name Name of the new clone
         * @param newParent New parent for the clone
         * @param doNotCloneChildren Do not clone children hierarchy
         */
        clone(name: string, newParent: Node, doNotCloneChildren?: boolean): Nullable<TransformNode>;
        serialize(currentSerializationObject?: any): any;
        /**
         * Returns a new TransformNode object parsed from the source provided.
         * The parameter `parsedMesh` is the source.
         * The parameter `rootUrl` is a string, it's the root URL to prefix the `delayLoadingFile` property with
         */
        static Parse(parsedTransformNode: any, scene: Scene, rootUrl: string): TransformNode;
        /**
         * Releases resources associated with this transform node.
         * @param doNotRecurse Set to true to not recurse into each children (recurse into each children by default)
         * @param disposeMaterialAndTextures Set to true to also dispose referenced materials and textures (false by default)
         */
        dispose(doNotRecurse?: boolean, disposeMaterialAndTextures?: boolean): void;
    }
}

declare module BABYLON {
    class VertexBuffer {
        private _buffer;
        private _kind;
        private _offset;
        private _size;
        private _stride;
        private _ownsBuffer;
        private _instanced;
        private _instanceDivisor;
        /**
         * Gets or sets the instance divisor when in instanced mode
         */
        instanceDivisor: number;
        constructor(engine: any, data: FloatArray | Buffer, kind: string, updatable: boolean, postponeInternalCreation?: boolean, stride?: number, instanced?: boolean, offset?: number, size?: number);
        _rebuild(): void;
        /**
         * Returns the kind of the VertexBuffer (string).
         */
        getKind(): string;
        /**
         * Boolean : is the VertexBuffer updatable ?
         */
        isUpdatable(): boolean;
        /**
         * Returns an array of numbers or a Float32Array containing the VertexBuffer data.
         */
        getData(): Nullable<FloatArray>;
        /**
         * Returns the WebGLBuffer associated to the VertexBuffer.
         */
        getBuffer(): Nullable<WebGLBuffer>;
        /**
         * Returns the stride of the VertexBuffer (integer).
         */
        getStrideSize(): number;
        /**
         * Returns the offset (integer).
         */
        getOffset(): number;
        /**
         * Returns the VertexBuffer total size (integer).
         */
        getSize(): number;
        /**
         * Boolean : is the WebGLBuffer of the VertexBuffer instanced now ?
         */
        getIsInstanced(): boolean;
        /**
         * Returns the instancing divisor, zero for non-instanced (integer).
         */
        getInstanceDivisor(): number;
        /**
         * Creates the underlying WebGLBuffer from the passed numeric array or Float32Array.
         * Returns the created WebGLBuffer.
         */
        create(data?: FloatArray): void;
        /**
         * Updates the underlying WebGLBuffer according to the passed numeric array or Float32Array.
         * This function will create a new buffer if the current one is not updatable
         * Returns the updated WebGLBuffer.
         */
        update(data: FloatArray): void;
        /**
         * Updates directly the underlying WebGLBuffer according to the passed numeric array or Float32Array.
         * Returns the directly updated WebGLBuffer.
         */
        updateDirectly(data: Float32Array, offset: number): void;
        /**
         * Disposes the VertexBuffer and the underlying WebGLBuffer.
         */
        dispose(): void;
        private static _PositionKind;
        private static _NormalKind;
        private static _TangentKind;
        private static _UVKind;
        private static _UV2Kind;
        private static _UV3Kind;
        private static _UV4Kind;
        private static _UV5Kind;
        private static _UV6Kind;
        private static _ColorKind;
        private static _MatricesIndicesKind;
        private static _MatricesWeightsKind;
        private static _MatricesIndicesExtraKind;
        private static _MatricesWeightsExtraKind;
        static readonly PositionKind: string;
        static readonly NormalKind: string;
        static readonly TangentKind: string;
        static readonly UVKind: string;
        static readonly UV2Kind: string;
        static readonly UV3Kind: string;
        static readonly UV4Kind: string;
        static readonly UV5Kind: string;
        static readonly UV6Kind: string;
        static readonly ColorKind: string;
        static readonly MatricesIndicesKind: string;
        static readonly MatricesWeightsKind: string;
        static readonly MatricesIndicesExtraKind: string;
        static readonly MatricesWeightsExtraKind: string;
        /**
         * Deduces the stride given a kind.
         * @param kind The kind string to deduce
         * @returns The deduced stride
         */
        static DeduceStride(kind: string): number;
    }
}

declare module BABYLON {
    interface PhysicsImpostorJoint {
        mainImpostor: PhysicsImpostor;
        connectedImpostor: PhysicsImpostor;
        joint: PhysicsJoint;
    }
    class PhysicsEngine {
        private _physicsPlugin;
        gravity: Vector3;
        constructor(gravity: Nullable<Vector3>, _physicsPlugin?: IPhysicsEnginePlugin);
        setGravity(gravity: Vector3): void;
        /**
         * Set the time step of the physics engine.
         * default is 1/60.
         * To slow it down, enter 1/600 for example.
         * To speed it up, 1/30
         * @param {number} newTimeStep the new timestep to apply to this world.
         */
        setTimeStep(newTimeStep?: number): void;
        /**
         * Get the time step of the physics engine.
         */
        getTimeStep(): number;
        dispose(): void;
        getPhysicsPluginName(): string;
        static Epsilon: number;
        private _impostors;
        private _joints;
        /**
         * Adding a new impostor for the impostor tracking.
         * This will be done by the impostor itself.
         * @param {PhysicsImpostor} impostor the impostor to add
         */
        addImpostor(impostor: PhysicsImpostor): void;
        /**
         * Remove an impostor from the engine.
         * This impostor and its mesh will not longer be updated by the physics engine.
         * @param {PhysicsImpostor} impostor the impostor to remove
         */
        removeImpostor(impostor: PhysicsImpostor): void;
        /**
         * Add a joint to the physics engine
         * @param {PhysicsImpostor} mainImpostor the main impostor to which the joint is added.
         * @param {PhysicsImpostor} connectedImpostor the impostor that is connected to the main impostor using this joint
         * @param {PhysicsJoint} the joint that will connect both impostors.
         */
        addJoint(mainImpostor: PhysicsImpostor, connectedImpostor: PhysicsImpostor, joint: PhysicsJoint): void;
        removeJoint(mainImpostor: PhysicsImpostor, connectedImpostor: PhysicsImpostor, joint: PhysicsJoint): void;
        /**
         * Called by the scene. no need to call it.
         */
        _step(delta: number): void;
        getPhysicsPlugin(): IPhysicsEnginePlugin;
        getImpostors(): Array<PhysicsImpostor>;
        getImpostorForPhysicsObject(object: IPhysicsEnabledObject): Nullable<PhysicsImpostor>;
        getImpostorWithPhysicsBody(body: any): Nullable<PhysicsImpostor>;
    }
    interface IPhysicsEnginePlugin {
        world: any;
        name: string;
        setGravity(gravity: Vector3): void;
        setTimeStep(timeStep: number): void;
        getTimeStep(): number;
        executeStep(delta: number, impostors: Array<PhysicsImpostor>): void;
        applyImpulse(impostor: PhysicsImpostor, force: Vector3, contactPoint: Vector3): void;
        applyForce(impostor: PhysicsImpostor, force: Vector3, contactPoint: Vector3): void;
        generatePhysicsBody(impostor: PhysicsImpostor): void;
        removePhysicsBody(impostor: PhysicsImpostor): void;
        generateJoint(joint: PhysicsImpostorJoint): void;
        removeJoint(joint: PhysicsImpostorJoint): void;
        isSupported(): boolean;
        setTransformationFromPhysicsBody(impostor: PhysicsImpostor): void;
        setPhysicsBodyTransformation(impostor: PhysicsImpostor, newPosition: Vector3, newRotation: Quaternion): void;
        setLinearVelocity(impostor: PhysicsImpostor, velocity: Nullable<Vector3>): void;
        setAngularVelocity(impostor: PhysicsImpostor, velocity: Nullable<Vector3>): void;
        getLinearVelocity(impostor: PhysicsImpostor): Nullable<Vector3>;
        getAngularVelocity(impostor: PhysicsImpostor): Nullable<Vector3>;
        setBodyMass(impostor: PhysicsImpostor, mass: number): void;
        getBodyMass(impostor: PhysicsImpostor): number;
        getBodyFriction(impostor: PhysicsImpostor): number;
        setBodyFriction(impostor: PhysicsImpostor, friction: number): void;
        getBodyRestitution(impostor: PhysicsImpostor): number;
        setBodyRestitution(impostor: PhysicsImpostor, restitution: number): void;
        sleepBody(impostor: PhysicsImpostor): void;
        wakeUpBody(impostor: PhysicsImpostor): void;
        updateDistanceJoint(joint: PhysicsJoint, maxDistance: number, minDistance?: number): void;
        setMotor(joint: IMotorEnabledJoint, speed: number, maxForce?: number, motorIndex?: number): void;
        setLimit(joint: IMotorEnabledJoint, upperLimit: number, lowerLimit?: number, motorIndex?: number): void;
        getRadius(impostor: PhysicsImpostor): number;
        getBoxSizeToRef(impostor: PhysicsImpostor, result: Vector3): void;
        syncMeshWithImpostor(mesh: AbstractMesh, impostor: PhysicsImpostor): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class PhysicsHelper {
        private _scene;
        private _physicsEngine;
        constructor(scene: Scene);
        /**
         * @param {Vector3} origin the origin of the explosion
         * @param {number} radius the explosion radius
         * @param {number} strength the explosion strength
         * @param {PhysicsRadialImpulseFalloff} falloff possible options: Constant & Linear. Defaults to Constant
         */
        applyRadialExplosionImpulse(origin: Vector3, radius: number, strength: number, falloff?: PhysicsRadialImpulseFalloff): Nullable<PhysicsRadialExplosionEvent>;
        /**
         * @param {Vector3} origin the origin of the explosion
         * @param {number} radius the explosion radius
         * @param {number} strength the explosion strength
         * @param {PhysicsRadialImpulseFalloff} falloff possible options: Constant & Linear. Defaults to Constant
         */
        applyRadialExplosionForce(origin: Vector3, radius: number, strength: number, falloff?: PhysicsRadialImpulseFalloff): Nullable<PhysicsRadialExplosionEvent>;
        /**
         * @param {Vector3} origin the origin of the explosion
         * @param {number} radius the explosion radius
         * @param {number} strength the explosion strength
         * @param {PhysicsRadialImpulseFalloff} falloff possible options: Constant & Linear. Defaults to Constant
         */
        gravitationalField(origin: Vector3, radius: number, strength: number, falloff?: PhysicsRadialImpulseFalloff): Nullable<PhysicsGravitationalFieldEvent>;
        /**
         * @param {Vector3} origin the origin of the updraft
         * @param {number} radius the radius of the updraft
         * @param {number} strength the strength of the updraft
         * @param {number} height the height of the updraft
         * @param {PhysicsUpdraftMode} updraftMode possible options: Center & Perpendicular. Defaults to Center
         */
        updraft(origin: Vector3, radius: number, strength: number, height: number, updraftMode?: PhysicsUpdraftMode): Nullable<PhysicsUpdraftEvent>;
        /**
         * @param {Vector3} origin the of the vortex
         * @param {number} radius the radius of the vortex
         * @param {number} strength the strength of the vortex
         * @param {number} height   the height of the vortex
         */
        vortex(origin: Vector3, radius: number, strength: number, height: number): Nullable<PhysicsVortexEvent>;
    }
    /***** Radial explosion *****/
    class PhysicsRadialExplosionEvent {
        private _scene;
        private _sphere;
        private _sphereOptions;
        private _rays;
        private _dataFetched;
        constructor(scene: Scene);
        /**
         * Returns the data related to the radial explosion event (sphere & rays).
         * @returns {PhysicsRadialExplosionEventData}
         */
        getData(): PhysicsRadialExplosionEventData;
        /**
         * Returns the force and contact point of the impostor or false, if the impostor is not affected by the force/impulse.
         * @param impostor
         * @param {Vector3} origin the origin of the explosion
         * @param {number} radius the explosion radius
         * @param {number} strength the explosion strength
         * @param {PhysicsRadialImpulseFalloff} falloff possible options: Constant & Linear
         * @returns {Nullable<PhysicsForceAndContactPoint>}
         */
        getImpostorForceAndContactPoint(impostor: PhysicsImpostor, origin: Vector3, radius: number, strength: number, falloff: PhysicsRadialImpulseFalloff): Nullable<PhysicsForceAndContactPoint>;
        /**
         * Disposes the sphere.
         * @param {bolean} force
         */
        dispose(force?: boolean): void;
        /*** Helpers ***/
        private _prepareSphere();
        private _intersectsWithSphere(impostor, origin, radius);
    }
    /***** Gravitational Field *****/
    class PhysicsGravitationalFieldEvent {
        private _physicsHelper;
        private _scene;
        private _origin;
        private _radius;
        private _strength;
        private _falloff;
        private _tickCallback;
        private _sphere;
        private _dataFetched;
        constructor(physicsHelper: PhysicsHelper, scene: Scene, origin: Vector3, radius: number, strength: number, falloff?: PhysicsRadialImpulseFalloff);
        /**
         * Returns the data related to the gravitational field event (sphere).
         * @returns {PhysicsGravitationalFieldEventData}
         */
        getData(): PhysicsGravitationalFieldEventData;
        /**
         * Enables the gravitational field.
         */
        enable(): void;
        /**
         * Disables the gravitational field.
         */
        disable(): void;
        /**
         * Disposes the sphere.
         * @param {bolean} force
         */
        dispose(force?: boolean): void;
        private _tick();
    }
    /***** Updraft *****/
    class PhysicsUpdraftEvent {
        private _scene;
        private _origin;
        private _radius;
        private _strength;
        private _height;
        private _updraftMode;
        private _physicsEngine;
        private _originTop;
        private _originDirection;
        private _tickCallback;
        private _cylinder;
        private _cylinderPosition;
        private _dataFetched;
        constructor(_scene: Scene, _origin: Vector3, _radius: number, _strength: number, _height: number, _updraftMode: PhysicsUpdraftMode);
        /**
         * Returns the data related to the updraft event (cylinder).
         * @returns {PhysicsUpdraftEventData}
         */
        getData(): PhysicsUpdraftEventData;
        /**
         * Enables the updraft.
         */
        enable(): void;
        /**
         * Disables the cortex.
         */
        disable(): void;
        /**
         * Disposes the sphere.
         * @param {bolean} force
         */
        dispose(force?: boolean): void;
        private getImpostorForceAndContactPoint(impostor);
        private _tick();
        /*** Helpers ***/
        private _prepareCylinder();
        private _intersectsWithCylinder(impostor);
    }
    /***** Vortex *****/
    class PhysicsVortexEvent {
        private _scene;
        private _origin;
        private _radius;
        private _strength;
        private _height;
        private _physicsEngine;
        private _originTop;
        private _centripetalForceThreshold;
        private _updraftMultiplier;
        private _tickCallback;
        private _cylinder;
        private _cylinderPosition;
        private _dataFetched;
        constructor(_scene: Scene, _origin: Vector3, _radius: number, _strength: number, _height: number);
        /**
         * Returns the data related to the vortex event (cylinder).
         * @returns {PhysicsVortexEventData}
         */
        getData(): PhysicsVortexEventData;
        /**
         * Enables the vortex.
         */
        enable(): void;
        /**
         * Disables the cortex.
         */
        disable(): void;
        /**
         * Disposes the sphere.
         * @param {bolean} force
         */
        dispose(force?: boolean): void;
        private getImpostorForceAndContactPoint(impostor);
        private _tick();
        /*** Helpers ***/
        private _prepareCylinder();
        private _intersectsWithCylinder(impostor);
    }
    /***** Enums *****/
    /**
    * The strenght of the force in correspondence to the distance of the affected object
    */
    enum PhysicsRadialImpulseFalloff {
        Constant = 0,
        Linear = 1,
    }
    /**
     * The strenght of the force in correspondence to the distance of the affected object
     */
    enum PhysicsUpdraftMode {
        Center = 0,
        Perpendicular = 1,
    }
    /***** Data interfaces *****/
    interface PhysicsForceAndContactPoint {
        force: Vector3;
        contactPoint: Vector3;
    }
    interface PhysicsRadialExplosionEventData {
        sphere: Mesh;
        rays: Array<Ray>;
    }
    interface PhysicsGravitationalFieldEventData {
        sphere: Mesh;
    }
    interface PhysicsUpdraftEventData {
        cylinder: Mesh;
    }
    interface PhysicsVortexEventData {
        cylinder: Mesh;
    }
}

declare module BABYLON {
    interface PhysicsImpostorParameters {
        mass: number;
        friction?: number;
        restitution?: number;
        nativeOptions?: any;
        ignoreParent?: boolean;
        disableBidirectionalTransformation?: boolean;
    }
    interface IPhysicsEnabledObject {
        position: Vector3;
        rotationQuaternion: Nullable<Quaternion>;
        scaling: Vector3;
        rotation?: Vector3;
        parent?: any;
        getBoundingInfo(): BoundingInfo;
        computeWorldMatrix(force: boolean): Matrix;
        getWorldMatrix?(): Matrix;
        getChildMeshes?(directDescendantsOnly?: boolean): Array<AbstractMesh>;
        getVerticesData(kind: string): Nullable<Array<number> | Float32Array>;
        getIndices?(): Nullable<IndicesArray>;
        getScene?(): Scene;
        getAbsolutePosition(): Vector3;
        getAbsolutePivotPoint(): Vector3;
        rotate(axis: Vector3, amount: number, space?: Space): TransformNode;
        translate(axis: Vector3, distance: number, space?: Space): TransformNode;
        setAbsolutePosition(absolutePosition: Vector3): TransformNode;
        getClassName(): string;
    }
    class PhysicsImpostor {
        object: IPhysicsEnabledObject;
        type: number;
        private _options;
        private _scene;
        static DEFAULT_OBJECT_SIZE: Vector3;
        static IDENTITY_QUATERNION: Quaternion;
        private _physicsEngine;
        private _physicsBody;
        private _bodyUpdateRequired;
        private _onBeforePhysicsStepCallbacks;
        private _onAfterPhysicsStepCallbacks;
        private _onPhysicsCollideCallbacks;
        private _deltaPosition;
        private _deltaRotation;
        private _deltaRotationConjugated;
        private _parent;
        private _isDisposed;
        private static _tmpVecs;
        private static _tmpQuat;
        readonly isDisposed: boolean;
        mass: number;
        friction: number;
        restitution: number;
        uniqueId: number;
        private _joints;
        constructor(object: IPhysicsEnabledObject, type: number, _options?: PhysicsImpostorParameters, _scene?: Scene | undefined);
        /**
         * This function will completly initialize this impostor.
         * It will create a new body - but only if this mesh has no parent.
         * If it has, this impostor will not be used other than to define the impostor
         * of the child mesh.
         */
        _init(): void;
        private _getPhysicsParent();
        /**
         * Should a new body be generated.
         */
        isBodyInitRequired(): boolean;
        setScalingUpdated(updated: boolean): void;
        /**
         * Force a regeneration of this or the parent's impostor's body.
         * Use under cautious - This will remove all joints already implemented.
         */
        forceUpdate(): void;
        /**
         * Gets the body that holds this impostor. Either its own, or its parent.
         */
        /**
         * Set the physics body. Used mainly by the physics engine/plugin
         */
        physicsBody: any;
        parent: Nullable<PhysicsImpostor>;
        resetUpdateFlags(): void;
        getObjectExtendSize(): Vector3;
        getObjectCenter(): Vector3;
        /**
         * Get a specific parametes from the options parameter.
         */
        getParam(paramName: string): any;
        /**
         * Sets a specific parameter in the options given to the physics plugin
         */
        setParam(paramName: string, value: number): void;
        /**
         * Specifically change the body's mass option. Won't recreate the physics body object
         */
        setMass(mass: number): void;
        getLinearVelocity(): Nullable<Vector3>;
        setLinearVelocity(velocity: Nullable<Vector3>): void;
        getAngularVelocity(): Nullable<Vector3>;
        setAngularVelocity(velocity: Nullable<Vector3>): void;
        /**
         * Execute a function with the physics plugin native code.
         * Provide a function the will have two variables - the world object and the physics body object.
         */
        executeNativeFunction(func: (world: any, physicsBody: any) => void): void;
        /**
         * Register a function that will be executed before the physics world is stepping forward.
         */
        registerBeforePhysicsStep(func: (impostor: PhysicsImpostor) => void): void;
        unregisterBeforePhysicsStep(func: (impostor: PhysicsImpostor) => void): void;
        /**
         * Register a function that will be executed after the physics step
         */
        registerAfterPhysicsStep(func: (impostor: PhysicsImpostor) => void): void;
        unregisterAfterPhysicsStep(func: (impostor: PhysicsImpostor) => void): void;
        /**
         * register a function that will be executed when this impostor collides against a different body.
         */
        registerOnPhysicsCollide(collideAgainst: PhysicsImpostor | Array<PhysicsImpostor>, func: (collider: PhysicsImpostor, collidedAgainst: PhysicsImpostor) => void): void;
        unregisterOnPhysicsCollide(collideAgainst: PhysicsImpostor | Array<PhysicsImpostor>, func: (collider: PhysicsImpostor, collidedAgainst: PhysicsImpostor | Array<PhysicsImpostor>) => void): void;
        private _tmpQuat;
        private _tmpQuat2;
        getParentsRotation(): Quaternion;
        /**
         * this function is executed by the physics engine.
         */
        beforeStep: () => void;
        /**
         * this function is executed by the physics engine.
         */
        afterStep: () => void;
        /**
         * Legacy collision detection event support
         */
        onCollideEvent: Nullable<(collider: PhysicsImpostor, collidedWith: PhysicsImpostor) => void>;
        onCollide: (e: {
            body: any;
        }) => void;
        /**
         * Apply a force
         */
        applyForce(force: Vector3, contactPoint: Vector3): PhysicsImpostor;
        /**
         * Apply an impulse
         */
        applyImpulse(force: Vector3, contactPoint: Vector3): PhysicsImpostor;
        /**
         * A help function to create a joint.
         */
        createJoint(otherImpostor: PhysicsImpostor, jointType: number, jointData: PhysicsJointData): PhysicsImpostor;
        /**
         * Add a joint to this impostor with a different impostor.
         */
        addJoint(otherImpostor: PhysicsImpostor, joint: PhysicsJoint): PhysicsImpostor;
        /**
         * Will keep this body still, in a sleep mode.
         */
        sleep(): PhysicsImpostor;
        /**
         * Wake the body up.
         */
        wakeUp(): PhysicsImpostor;
        clone(newObject: IPhysicsEnabledObject): Nullable<PhysicsImpostor>;
        dispose(): void;
        setDeltaPosition(position: Vector3): void;
        setDeltaRotation(rotation: Quaternion): void;
        getBoxSizeToRef(result: Vector3): PhysicsImpostor;
        getRadius(): number;
        /**
         * Sync a bone with this impostor
         * @param bone The bone to sync to the impostor.
         * @param boneMesh The mesh that the bone is influencing.
         * @param jointPivot The pivot of the joint / bone in local space.
         * @param distToJoint Optional distance from the impostor to the joint.
         * @param adjustRotation Optional quaternion for adjusting the local rotation of the bone.
         */
        syncBoneWithImpostor(bone: Bone, boneMesh: AbstractMesh, jointPivot: Vector3, distToJoint?: number, adjustRotation?: Quaternion): void;
        /**
         * Sync impostor to a bone
         * @param bone The bone that the impostor will be synced to.
         * @param boneMesh The mesh that the bone is influencing.
         * @param jointPivot The pivot of the joint / bone in local space.
         * @param distToJoint Optional distance from the impostor to the joint.
         * @param adjustRotation Optional quaternion for adjusting the local rotation of the bone.
         * @param boneAxis Optional vector3 axis the bone is aligned with
         */
        syncImpostorWithBone(bone: Bone, boneMesh: AbstractMesh, jointPivot: Vector3, distToJoint?: number, adjustRotation?: Quaternion, boneAxis?: Vector3): void;
        static NoImpostor: number;
        static SphereImpostor: number;
        static BoxImpostor: number;
        static PlaneImpostor: number;
        static MeshImpostor: number;
        static CylinderImpostor: number;
        static ParticleImpostor: number;
        static HeightmapImpostor: number;
    }
}

declare module BABYLON {
    interface PhysicsJointData {
        mainPivot?: Vector3;
        connectedPivot?: Vector3;
        mainAxis?: Vector3;
        connectedAxis?: Vector3;
        collision?: boolean;
        nativeParams?: any;
    }
    /**
     * This is a holder class for the physics joint created by the physics plugin.
     * It holds a set of functions to control the underlying joint.
     */
    class PhysicsJoint {
        type: number;
        jointData: PhysicsJointData;
        private _physicsJoint;
        protected _physicsPlugin: IPhysicsEnginePlugin;
        constructor(type: number, jointData: PhysicsJointData);
        physicsJoint: any;
        physicsPlugin: IPhysicsEnginePlugin;
        /**
         * Execute a function that is physics-plugin specific.
         * @param {Function} func the function that will be executed.
         *                        It accepts two parameters: the physics world and the physics joint.
         */
        executeNativeFunction(func: (world: any, physicsJoint: any) => void): void;
        static DistanceJoint: number;
        static HingeJoint: number;
        static BallAndSocketJoint: number;
        static WheelJoint: number;
        static SliderJoint: number;
        static PrismaticJoint: number;
        static UniversalJoint: number;
        static Hinge2Joint: number;
        static PointToPointJoint: number;
        static SpringJoint: number;
        static LockJoint: number;
    }
    /**
     * A class representing a physics distance joint.
     */
    class DistanceJoint extends PhysicsJoint {
        constructor(jointData: DistanceJointData);
        /**
         * Update the predefined distance.
         */
        updateDistance(maxDistance: number, minDistance?: number): void;
    }
    class MotorEnabledJoint extends PhysicsJoint implements IMotorEnabledJoint {
        constructor(type: number, jointData: PhysicsJointData);
        /**
         * Set the motor values.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         * @param {number} force the force to apply
         * @param {number} maxForce max force for this motor.
         */
        setMotor(force?: number, maxForce?: number): void;
        /**
         * Set the motor's limits.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         */
        setLimit(upperLimit: number, lowerLimit?: number): void;
    }
    /**
     * This class represents a single hinge physics joint
     */
    class HingeJoint extends MotorEnabledJoint {
        constructor(jointData: PhysicsJointData);
        /**
         * Set the motor values.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         * @param {number} force the force to apply
         * @param {number} maxForce max force for this motor.
         */
        setMotor(force?: number, maxForce?: number): void;
        /**
         * Set the motor's limits.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         */
        setLimit(upperLimit: number, lowerLimit?: number): void;
    }
    /**
     * This class represents a dual hinge physics joint (same as wheel joint)
     */
    class Hinge2Joint extends MotorEnabledJoint {
        constructor(jointData: PhysicsJointData);
        /**
         * Set the motor values.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         * @param {number} force the force to apply
         * @param {number} maxForce max force for this motor.
         * @param {motorIndex} the motor's index, 0 or 1.
         */
        setMotor(force?: number, maxForce?: number, motorIndex?: number): void;
        /**
         * Set the motor limits.
         * Attention, this function is plugin specific. Engines won't react 100% the same.
         * @param {number} upperLimit the upper limit
         * @param {number} lowerLimit lower limit
         * @param {motorIndex} the motor's index, 0 or 1.
         */
        setLimit(upperLimit: number, lowerLimit?: number, motorIndex?: number): void;
    }
    interface IMotorEnabledJoint {
        physicsJoint: any;
        setMotor(force?: number, maxForce?: number, motorIndex?: number): void;
        setLimit(upperLimit: number, lowerLimit?: number, motorIndex?: number): void;
    }
    interface DistanceJointData extends PhysicsJointData {
        maxDistance: number;
    }
    interface SpringJointData extends PhysicsJointData {
        length: number;
        stiffness: number;
        damping: number;
    }
}

declare module BABYLON {
    /**
     * This represents a GPU particle system in Babylon
     * This is the fastest particle system in Babylon as it uses the GPU to update the individual particle data
     * @see https://www.babylonjs-playground.com/#PU4WYI#4
     */
    class GPUParticleSystem implements IDisposable, IParticleSystem, IAnimatable {
        /**
         * The id of the Particle system.
         */
        id: string;
        /**
         * The friendly name of the Particle system.
         */
        name: string;
        /**
         * The emitter represents the Mesh or position we are attaching the particle system to.
         */
        emitter: Nullable<AbstractMesh | Vector3>;
        /**
         * The rendering group used by the Particle system to chose when to render.
         */
        renderingGroupId: number;
        /**
         * The layer mask we are rendering the particles through.
         */
        layerMask: number;
        private _capacity;
        private _activeCount;
        private _currentActiveCount;
        private _renderEffect;
        private _updateEffect;
        private _buffer0;
        private _buffer1;
        private _spriteBuffer;
        private _updateVAO;
        private _renderVAO;
        private _targetIndex;
        private _sourceBuffer;
        private _targetBuffer;
        private _scene;
        private _engine;
        private _currentRenderId;
        private _started;
        private _stopped;
        private _timeDelta;
        private _randomTexture;
        private readonly _attributesStrideSize;
        private _updateEffectOptions;
        private _randomTextureSize;
        private _actualFrame;
        /**
         * List of animations used by the particle system.
         */
        animations: Animation[];
        /**
         * Gets a boolean indicating if the GPU particles can be rendered on current browser
         */
        static readonly IsSupported: boolean;
        /**
        * An event triggered when the system is disposed.
        */
        onDisposeObservable: Observable<GPUParticleSystem>;
        /**
         * The overall motion speed (0.01 is default update speed, faster updates = faster animation)
         */
        updateSpeed: number;
        /**
         * The amount of time the particle system is running (depends of the overall update speed).
         */
        targetStopDuration: number;
        /**
         * The texture used to render each particle. (this can be a spritesheet)
         */
        particleTexture: Nullable<Texture>;
        /**
         * Blend mode use to render the particle, it can be either ParticleSystem.BLENDMODE_ONEONE or ParticleSystem.BLENDMODE_STANDARD.
         */
        blendMode: number;
        /**
         * Minimum life time of emitting particles.
         */
        minLifeTime: number;
        /**
         * Maximum life time of emitting particles.
         */
        maxLifeTime: number;
        /**
         * Minimum Size of emitting particles.
         */
        minSize: number;
        /**
         * Maximum Size of emitting particles.
         */
        maxSize: number;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color1: Color4;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color2: Color4;
        /**
         * Color the particle will have at the end of its lifetime.
         */
        colorDead: Color4;
        /**
         * The maximum number of particles to emit per frame until we reach the activeParticleCount value
         */
        emitRate: number;
        /**
         * You can use gravity if you want to give an orientation to your particles.
         */
        gravity: Vector3;
        /**
         * Minimum power of emitting particles.
         */
        minEmitPower: number;
        /**
         * Maximum power of emitting particles.
         */
        maxEmitPower: number;
        /**
         * The particle emitter type defines the emitter used by the particle system.
         * It can be for example box, sphere, or cone...
         */
        particleEmitterType: Nullable<IParticleEmitterType>;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        direction1: Vector3;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        direction2: Vector3;
        /**
         * Minimum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        minEmitBox: Vector3;
        /**
         * Maximum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        maxEmitBox: Vector3;
        /**
         * Gets the maximum number of particles active at the same time.
         * @returns The max number of active particles.
         */
        getCapacity(): number;
        /**
         * Forces the particle to write their depth information to the depth buffer. This can help preventing other draw calls
         * to override the particles.
         */
        forceDepthWrite: boolean;
        /**
         * Gets or set the number of active particles
         */
        activeParticleCount: number;
        /**
         * Is this system ready to be used/rendered
         * @return true if the system is ready
         */
        isReady(): boolean;
        /**
         * Gets Wether the system has been started.
         * @returns True if it has been started, otherwise false.
         */
        isStarted(): boolean;
        /**
         * Starts the particle system and begins to emit.
         */
        start(): void;
        /**
         * Stops the particle system.
         */
        stop(): void;
        /**
         * Remove all active particles
         */
        reset(): void;
        /**
         * Returns the string "GPUParticleSystem"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Instantiates a GPU particle system.
         * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
         * @param name The name of the particle system
         * @param capacity The max number of particles alive at the same time
         * @param scene The scene the particle system belongs to
         */
        constructor(name: string, options: Partial<{
            capacity: number;
            randomTextureSize: number;
        }>, scene: Scene);
        private _createUpdateVAO(source);
        private _createRenderVAO(source, spriteSource);
        private _initialize(force?);
        /** @ignore */
        _recreateUpdateEffect(): void;
        /** @ignore */
        _recreateRenderEffect(): void;
        /**
         * Animates the particle system for the current frame by emitting new particles and or animating the living ones.
         */
        animate(): void;
        /**
         * Renders the particle system in its current state.
         * @returns the current number of particles
         */
        render(): number;
        /**
         * Rebuilds the particle system
         */
        rebuild(): void;
        private _releaseBuffers();
        private _releaseVAOs();
        /**
         * Disposes the particle system and free the associated resources
         * @param disposeTexture defines if the particule texture must be disposed as well (true by default)
         */
        dispose(disposeTexture?: boolean): void;
        /**
         * Clones the particle system.
         * @param name The name of the cloned object
         * @param newEmitter The new emitter to use
         * @returns the cloned particle system
         */
        clone(name: string, newEmitter: any): Nullable<GPUParticleSystem>;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parses a JSON object to create a GPU particle system.
         * @param parsedParticleSystem The JSON object to parse
         * @param scene The scene to create the particle system in
         * @param rootUrl The root url to use to load external dependencies like texture
         * @returns the parsed GPU particle system
         */
        static Parse(parsedParticleSystem: any, scene: Scene, rootUrl: string): GPUParticleSystem;
    }
}

declare module BABYLON {
    /**
     * Interface representing a particle system in Babylon.
     * This groups the common functionalities that needs to be implemented in order to create a particle system.
     * A particle system represents a way to manage particles from their emission to their animation and rendering.
     */
    interface IParticleSystem {
        /**
         * The id of the Particle system.
         */
        id: string;
        /**
         * The name of the Particle system.
         */
        name: string;
        /**
         * The emitter represents the Mesh or position we are attaching the particle system to.
         */
        emitter: Nullable<AbstractMesh | Vector3>;
        /**
         * The rendering group used by the Particle system to chose when to render.
         */
        renderingGroupId: number;
        /**
         * The layer mask we are rendering the particles through.
         */
        layerMask: number;
        /**
        * The overall motion speed (0.01 is default update speed, faster updates = faster animation)
        */
        updateSpeed: number;
        /**
         * The amount of time the particle system is running (depends of the overall update speed).
         */
        targetStopDuration: number;
        /**
         * The texture used to render each particle. (this can be a spritesheet)
         */
        particleTexture: Nullable<Texture>;
        /**
         * Blend mode use to render the particle, it can be either ParticleSystem.BLENDMODE_ONEONE or ParticleSystem.BLENDMODE_STANDARD.
         */
        blendMode: number;
        /**
         * Minimum life time of emitting particles.
         */
        minLifeTime: number;
        /**
         * Maximum life time of emitting particles.
         */
        maxLifeTime: number;
        /**
         * Minimum Size of emitting particles.
         */
        minSize: number;
        /**
         * Maximum Size of emitting particles.
         */
        maxSize: number;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color1: Color4;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color2: Color4;
        /**
         * Color the particle will have at the end of its lifetime.
         */
        colorDead: Color4;
        /**
         * The maximum number of particles to emit per frame until we reach the activeParticleCount value
         */
        emitRate: number;
        /**
         * You can use gravity if you want to give an orientation to your particles.
         */
        gravity: Vector3;
        /**
         * Minimum power of emitting particles.
         */
        minEmitPower: number;
        /**
         * Maximum power of emitting particles.
         */
        maxEmitPower: number;
        /**
         * The particle emitter type defines the emitter used by the particle system.
         * It can be for example box, sphere, or cone...
         */
        particleEmitterType: Nullable<IParticleEmitterType>;
        /**
         * Gets the maximum number of particles active at the same time.
         * @returns The max number of active particles.
         */
        getCapacity(): number;
        /**
         * Gets Wether the system has been started.
         * @returns True if it has been started, otherwise false.
         */
        isStarted(): boolean;
        /**
         * Gets if the particle system has been started.
         * @return true if the system has been started, otherwise false.
         */
        isStarted(): boolean;
        /**
         * Animates the particle system for this frame.
         */
        animate(): void;
        /**
         * Renders the particle system in its current state.
         * @returns the current number of particles
         */
        render(): number;
        /**
         * Dispose the particle system and frees its associated resources.
         * @param disposeTexture defines if the particule texture must be disposed as well (true by default)
         */
        dispose(disposeTexture?: boolean): void;
        /**
         * Clones the particle system.
         * @param name The name of the cloned object
         * @param newEmitter The new emitter to use
         * @returns the cloned particle system
         */
        clone(name: string, newEmitter: any): Nullable<IParticleSystem>;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Rebuild the particle system
         */
        rebuild(): void;
        /**
         * Starts the particle system and begins to emit.
         */
        start(): void;
        /**
         * Stops the particle system.
         */
        stop(): void;
        /**
         * Remove all active particles
         */
        reset(): void;
        /**
         * Is this system ready to be used/rendered
         * @return true if the system is ready
         */
        isReady(): boolean;
    }
}

declare module BABYLON {
    /**
     * A particle represents one of the element emitted by a particle system.
     * This is mainly define by its coordinates, direction, velocity and age.
     */
    class Particle {
        /**
         * particleSystem the particle system the particle belongs to.
         */
        particleSystem: ParticleSystem;
        /**
         * The world position of the particle in the scene.
         */
        position: Vector3;
        /**
         * The world direction of the particle in the scene.
         */
        direction: Vector3;
        /**
         * The color of the particle.
         */
        color: Color4;
        /**
         * The color change of the particle per step.
         */
        colorStep: Color4;
        /**
         * Defines how long will the life of the particle be.
         */
        lifeTime: number;
        /**
         * The current age of the particle.
         */
        age: number;
        /**
         * The current size of the particle.
         */
        size: number;
        /**
         * The current angle of the particle.
         */
        angle: number;
        /**
         * Defines how fast is the angle changing.
         */
        angularSpeed: number;
        /**
         * Defines the cell index used by the particle to be rendered from a sprite.
         */
        cellIndex: number;
        private _currentFrameCounter;
        /**
         * Creates a new instance Particle
         * @param particleSystem the particle system the particle belongs to
         */
        constructor(
            /**
             * particleSystem the particle system the particle belongs to.
             */
            particleSystem: ParticleSystem);
        private updateCellInfoFromSystem();
        /**
         * Defines how the sprite cell index is updated for the particle. This is
         * defined as a callback.
         */
        updateCellIndex: (scaledUpdateSpeed: number) => void;
        private _updateCellIndexWithSpeedCalculated(scaledUpdateSpeed);
        private _updateCellIndexWithCustomSpeed();
        /**
         * Copy the properties of particle to another one.
         * @param other the particle to copy the information to.
         */
        copyTo(other: Particle): void;
    }
}

declare module BABYLON {
    /**
     * This represents a particle system in Babylon.
     * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
     * Particles can take different shapes while emitted like box, sphere, cone or you can write your custom function.
     * @example https://doc.babylonjs.com/babylon101/particles
     */
    class ParticleSystem implements IDisposable, IAnimatable, IParticleSystem {
        /**
         * Source color is added to the destination color without alpha affecting the result.
         */
        static BLENDMODE_ONEONE: number;
        /**
         * Blend current color and particle color using particle’s alpha.
         */
        static BLENDMODE_STANDARD: number;
        /**
         * List of animations used by the particle system.
         */
        animations: Animation[];
        /**
         * The id of the Particle system.
         */
        id: string;
        /**
         * The friendly name of the Particle system.
         */
        name: string;
        /**
         * The rendering group used by the Particle system to chose when to render.
         */
        renderingGroupId: number;
        /**
         * The emitter represents the Mesh or position we are attaching the particle system to.
         */
        emitter: Nullable<AbstractMesh | Vector3>;
        /**
         * The maximum number of particles to emit per frame
         */
        emitRate: number;
        /**
         * If you want to launch only a few particles at once, that can be done, as well.
         */
        manualEmitCount: number;
        /**
         * The overall motion speed (0.01 is default update speed, faster updates = faster animation)
         */
        updateSpeed: number;
        /**
         * The amount of time the particle system is running (depends of the overall update speed).
         */
        targetStopDuration: number;
        /**
         * Specifies whether the particle system will be disposed once it reaches the end of the animation.
         */
        disposeOnStop: boolean;
        /**
         * Minimum power of emitting particles.
         */
        minEmitPower: number;
        /**
         * Maximum power of emitting particles.
         */
        maxEmitPower: number;
        /**
         * Minimum life time of emitting particles.
         */
        minLifeTime: number;
        /**
         * Maximum life time of emitting particles.
         */
        maxLifeTime: number;
        /**
         * Minimum Size of emitting particles.
         */
        minSize: number;
        /**
         * Maximum Size of emitting particles.
         */
        maxSize: number;
        /**
         * Minimum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        minAngularSpeed: number;
        /**
         * Maximum angular speed of emitting particles (Z-axis rotation for each particle).
         */
        maxAngularSpeed: number;
        /**
         * The texture used to render each particle. (this can be a spritesheet)
         */
        particleTexture: Nullable<Texture>;
        /**
         * The layer mask we are rendering the particles through.
         */
        layerMask: number;
        /**
         * This can help using your own shader to render the particle system.
         * The according effect will be created
         */
        customShader: any;
        /**
         * By default particle system starts as soon as they are created. This prevents the
         * automatic start to happen and let you decide when to start emitting particles.
         */
        preventAutoStart: boolean;
        /**
         * This function can be defined to provide custom update for active particles.
         * This function will be called instead of regular update (age, position, color, etc.).
         * Do not forget that this function will be called on every frame so try to keep it simple and fast :)
         */
        updateFunction: (particles: Particle[]) => void;
        /**
         * Callback triggered when the particle animation is ending.
         */
        onAnimationEnd: Nullable<() => void>;
        /**
         * Blend mode use to render the particle, it can be either ParticleSystem.BLENDMODE_ONEONE or ParticleSystem.BLENDMODE_STANDARD.
         */
        blendMode: number;
        /**
         * Forces the particle to write their depth information to the depth buffer. This can help preventing other draw calls
         * to override the particles.
         */
        forceDepthWrite: boolean;
        /**
         * You can use gravity if you want to give an orientation to your particles.
         */
        gravity: Vector3;
        /**
          * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
          * This only works when particleEmitterTyps is a BoxParticleEmitter
          */
        direction1: Vector3;
        /**
         * Random direction of each particle after it has been emitted, between direction1 and direction2 vectors.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        direction2: Vector3;
        /**
         * Minimum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        minEmitBox: Vector3;
        /**
         * Maximum box point around our emitter. Our emitter is the center of particles source, but if you want your particles to emit from more than one point, then you can tell it to do so.
         * This only works when particleEmitterTyps is a BoxParticleEmitter
         */
        maxEmitBox: Vector3;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color1: Color4;
        /**
         * Random color of each particle after it has been emitted, between color1 and color2 vectors.
         */
        color2: Color4;
        /**
         * Color the particle will have at the end of its lifetime.
         */
        colorDead: Color4;
        /**
         * An optional mask to filter some colors out of the texture, or filter a part of the alpha channel.
         */
        textureMask: Color4;
        /**
         * The particle emitter type defines the emitter used by the particle system.
         * It can be for example box, sphere, or cone...
         */
        particleEmitterType: IParticleEmitterType;
        /**
         * This function can be defined to specify initial direction for every new particle.
         * It by default use the emitterType defined function.
         */
        startDirectionFunction: (emitPower: number, worldMatrix: Matrix, directionToUpdate: Vector3, particle: Particle) => void;
        /**
         * This function can be defined to specify initial position for every new particle.
         * It by default use the emitterType defined function.
         */
        startPositionFunction: (worldMatrix: Matrix, positionToUpdate: Vector3, particle: Particle) => void;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines if the sprite animation should loop between startSpriteCellID and endSpriteCellID or not.
         */
        spriteCellLoop: boolean;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) and spriteCellLoop defines the speed of the sprite loop.
         */
        spriteCellChangeSpeed: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) and spriteCellLoop defines the first sprite cell to display.
         */
        startSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled) and spriteCellLoop defines the last sprite cell to display.
         */
        endSpriteCellID: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell width to use.
         */
        spriteCellWidth: number;
        /**
         * If using a spritesheet (isAnimationSheetEnabled), defines the sprite cell height to use.
         */
        spriteCellHeight: number;
        /**
        * An event triggered when the system is disposed.
        */
        onDisposeObservable: Observable<ParticleSystem>;
        private _onDisposeObserver;
        /**
         * Sets a callback that will be triggered when the system is disposed.
         */
        onDispose: () => void;
        /**
         * Gets wether an animation sprite sheet is enabled or not on the particle system.
         */
        readonly isAnimationSheetEnabled: Boolean;
        private _particles;
        private _epsilon;
        private _capacity;
        private _scene;
        private _stockParticles;
        private _newPartsExcess;
        private _vertexData;
        private _vertexBuffer;
        private _vertexBuffers;
        private _indexBuffer;
        private _effect;
        private _customEffect;
        private _cachedDefines;
        private _scaledColorStep;
        private _colorDiff;
        private _scaledDirection;
        private _scaledGravity;
        private _currentRenderId;
        private _alive;
        private _started;
        private _stopped;
        private _actualFrame;
        private _scaledUpdateSpeed;
        private _vertexBufferSize;
        private _isAnimationSheetEnabled;
        /**
         * this is the Sub-emitters templates that will be used to generate particle system when the particle dies, this property is used by the root particle system only.
         */
        subEmitters: ParticleSystem[];
        /**
        * The current active Sub-systems, this property is used by the root particle system only.
        */
        activeSubSystems: Array<ParticleSystem>;
        private _rootParticleSystem;
        /**
         * Gets the current list of active particles
         */
        readonly particles: Particle[];
        /**
         * Returns the string "ParticleSystem"
         * @returns a string containing the class name
         */
        getClassName(): string;
        /**
         * Instantiates a particle system.
         * Particles are often small sprites used to simulate hard-to-reproduce phenomena like fire, smoke, water, or abstract visual effects like magic glitter and faery dust.
         * @param name The name of the particle system
         * @param capacity The max number of particles alive at the same time
         * @param scene The scene the particle system belongs to
         * @param customEffect a custom effect used to change the way particles are rendered by default
         * @param isAnimationSheetEnabled Must be true if using a spritesheet to animate the particles texture
         * @param epsilon Offset used to render the particles
         */
        constructor(name: string, capacity: number, scene: Scene, customEffect?: Nullable<Effect>, isAnimationSheetEnabled?: boolean, epsilon?: number);
        private _createIndexBuffer();
        /**
         * Gets the maximum number of particles active at the same time.
         * @returns The max number of active particles.
         */
        getCapacity(): number;
        /**
         * Gets Wether there are still active particles in the system.
         * @returns True if it is alive, otherwise false.
         */
        isAlive(): boolean;
        /**
         * Gets Wether the system has been started.
         * @returns True if it has been started, otherwise false.
         */
        isStarted(): boolean;
        /**
         * Starts the particle system and begins to emit.
         */
        start(): void;
        /**
         * Stops the particle system.
         * @param stopSubEmitters if true it will stop the current system and all created sub-Systems if false it will stop the current root system only, this param is used by the root particle system only. the default value is true.
         */
        stop(stopSubEmitters?: boolean): void;
        /**
         * Remove all active particles
         */
        reset(): void;
        /**
         * @ignore (for internal use only)
         */
        _appendParticleVertex(index: number, particle: Particle, offsetX: number, offsetY: number): void;
        /**
         * @ignore (for internal use only)
         */
        _appendParticleVertexWithAnimation(index: number, particle: Particle, offsetX: number, offsetY: number): void;
        /**
         * "Recycles" one of the particle by copying it back to the "stock" of particles and removing it from the active list.
         * Its lifetime will start back at 0.
         */
        recycleParticle: (particle: Particle) => void;
        private _stopSubEmitters();
        private _createParticle;
        private _removeFromRoot();
        private _emitFromParticle;
        private _update(newParticles);
        private _getEffect();
        /**
         * Animates the particle system for the current frame by emitting new particles and or animating the living ones.
         */
        animate(): void;
        private _appendParticleVertexes;
        private _appenedParticleVertexesWithSheet(offset, particle);
        private _appenedParticleVertexesNoSheet(offset, particle);
        /**
         * Rebuilds the particle system.
         */
        rebuild(): void;
        /**
         * Is this system ready to be used/rendered
         * @return true if the system is ready
         */
        isReady(): boolean;
        /**
         * Renders the particle system in its current state.
         * @returns the current number of particles
         */
        render(): number;
        /**
         * Disposes the particle system and free the associated resources
         * @param disposeTexture defines if the particule texture must be disposed as well (true by default)
         */
        dispose(disposeTexture?: boolean): void;
        /**
         * Creates a Sphere Emitter for the particle system. (emits along the sphere radius)
         * @param radius The radius of the sphere to emit from
         * @returns the emitter
         */
        createSphereEmitter(radius?: number): SphereParticleEmitter;
        /**
         * Creates a Directed Sphere Emitter for the particle system. (emits between direction1 and direction2)
         * @param radius The radius of the sphere to emit from
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the sphere
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the sphere
         * @returns the emitter
         */
        createDirectedSphereEmitter(radius?: number, direction1?: Vector3, direction2?: Vector3): SphereDirectedParticleEmitter;
        /**
         * Creates a Cone Emitter for the particle system. (emits from the cone to the particle position)
         * @param radius The radius of the cone to emit from
         * @param angle The base angle of the cone
         * @returns the emitter
         */
        createConeEmitter(radius?: number, angle?: number): ConeParticleEmitter;
        /**
         * Creates a Box Emitter for the particle system. (emits between direction1 and direction2 from withing the box defined by minEmitBox and maxEmitBox)
         * @param direction1 Particles are emitted between the direction1 and direction2 from within the box
         * @param direction2 Particles are emitted between the direction1 and direction2 from within the box
         * @param minEmitBox Particles are emitted from the box between minEmitBox and maxEmitBox
         * @param maxEmitBox  Particles are emitted from the box between minEmitBox and maxEmitBox
         * @returns the emitter
         */
        createBoxEmitter(direction1: Vector3, direction2: Vector3, minEmitBox: Vector3, maxEmitBox: Vector3): BoxParticleEmitter;
        /**
         * Clones the particle system.
         * @param name The name of the cloned object
         * @param newEmitter The new emitter to use
         * @returns the cloned particle system
         */
        clone(name: string, newEmitter: any): ParticleSystem;
        /**
         * Serializes the particle system to a JSON object.
         * @returns the JSON object
         */
        serialize(): any;
        /**
         * Parses a JSON object to create a particle system.
         * @param parsedParticleSystem The JSON object to parse
         * @param scene The scene to create the particle system in
         * @param rootUrl The root url to use to load external dependencies like texture
         * @returns the Parsed particle system
         */
        static Parse(parsedParticleSystem: any, scene: Scene, rootUrl: string): ParticleSystem;
    }
}

declare module BABYLON {
    /**
     * Represents one particle of a solid particle system.
     */
    class SolidParticle {
        /**
         * particle global index
         */
        idx: number;
        /**
         * The color of the particle
         */
        color: Nullable<Color4>;
        /**
         * The world space position of the particle.
         */
        position: Vector3;
        /**
         * The world space rotation of the particle. (Not use if rotationQuaternion is set)
         */
        rotation: Vector3;
        /**
         * The world space rotation quaternion of the particle.
         */
        rotationQuaternion: Nullable<Quaternion>;
        /**
         * The scaling of the particle.
         */
        scaling: Vector3;
        /**
         * The uvs of the particle.
         */
        uvs: Vector4;
        /**
         * The current speed of the particle.
         */
        velocity: Vector3;
        /**
         * The pivot point in the particle local space.
         */
        pivot: Vector3;
        /**
         * Must the particle be translated from its pivot point in its local space ?
         * In this case, the pivot point is set at the origin of the particle local space and the particle is translated.
         * Default : false
         */
        translateFromPivot: boolean;
        /**
         * Is the particle active or not ?
         */
        alive: boolean;
        /**
         * Is the particle visible or not ?
         */
        isVisible: boolean;
        /**
         * Index of this particle in the global "positions" array (Internal use)
         */
        _pos: number;
        /**
         * Index of this particle in the global "indices" array (Internal use)
         */
        _ind: number;
        /**
         * ModelShape of this particle (Internal use)
         */
        _model: ModelShape;
        /**
         * ModelShape id of this particle
         */
        shapeId: number;
        /**
         * Index of the particle in its shape id (Internal use)
         */
        idxInShape: number;
        /**
         * Reference to the shape model BoundingInfo object (Internal use)
         */
        _modelBoundingInfo: BoundingInfo;
        /**
         * Particle BoundingInfo object (Internal use)
         */
        _boundingInfo: BoundingInfo;
        /**
         * Reference to the SPS what the particle belongs to (Internal use)
         */
        _sps: SolidParticleSystem;
        /**
         * Still set as invisible in order to skip useless computations (Internal use)
         */
        _stillInvisible: boolean;
        /**
         * Last computed particle rotation matrix
         */
        _rotationMatrix: number[];
        /**
         * Parent particle Id, if any.
         * Default null.
         */
        parentId: Nullable<number>;
        /**
         * Internal global position in the SPS.
         */
        _globalPosition: Vector3;
        /**
         * Creates a Solid Particle object.
         * Don't create particles manually, use instead the Solid Particle System internal tools like _addParticle()
         * @param particleIndex (integer) is the particle index in the Solid Particle System pool. It's also the particle identifier.
         * @param positionIndex (integer) is the starting index of the particle vertices in the SPS "positions" array.
         * @param indiceIndex (integer) is the starting index of the particle indices in the SPS "indices" array.
         * @param model (ModelShape) is a reference to the model shape on what the particle is designed.
         * @param shapeId (integer) is the model shape identifier in the SPS.
         * @param idxInShape (integer) is the index of the particle in the current model (ex: the 10th box of addShape(box, 30))
         * @param modelBoundingInfo is the reference to the model BoundingInfo used for intersection computations.
         */
        constructor(particleIndex: number, positionIndex: number, indiceIndex: number, model: Nullable<ModelShape>, shapeId: number, idxInShape: number, sps: SolidParticleSystem, modelBoundingInfo?: Nullable<BoundingInfo>);
        /**
         * Legacy support, changed scale to scaling
         */
        /**
         * Legacy support, changed scale to scaling
         */
        scale: Vector3;
        /**
         * Legacy support, changed quaternion to rotationQuaternion
         */
        /**
         * Legacy support, changed quaternion to rotationQuaternion
         */
        quaternion: Nullable<Quaternion>;
        /**
         * Returns a boolean. True if the particle intersects another particle or another mesh, else false.
         * The intersection is computed on the particle bounding sphere and Axis Aligned Bounding Box (AABB)
         * @param target is the object (solid particle or mesh) what the intersection is computed against.
         * @returns true if it intersects
         */
        intersectsMesh(target: Mesh | SolidParticle): boolean;
    }
    /**
     * Represents the shape of the model used by one particle of a solid particle system.
     * SPS internal tool, don't use it manually.
     */
    class ModelShape {
        /**
         * The shape id.
         */
        shapeID: number;
        /**
         * flat array of model positions (internal use)
         */
        _shape: Vector3[];
        /**
         * flat array of model UVs (internal use)
         */
        _shapeUV: number[];
        /**
         * length of the shape in the model indices array (internal use)
         */
        _indicesLength: number;
        /**
         * Custom position function (internal use)
         */
        _positionFunction: Nullable<(particle: SolidParticle, i: number, s: number) => void>;
        /**
         * Custom vertex function (internal use)
         */
        _vertexFunction: Nullable<(particle: SolidParticle, vertex: Vector3, i: number) => void>;
        /**
         * Creates a ModelShape object. This is an internal simplified reference to a mesh used as for a model to replicate particles from by the SPS.
         * SPS internal tool, don't use it manually.
         * @ignore
         */
        constructor(id: number, shape: Vector3[], indicesLength: number, shapeUV: number[], posFunction: Nullable<(particle: SolidParticle, i: number, s: number) => void>, vtxFunction: Nullable<(particle: SolidParticle, vertex: Vector3, i: number) => void>);
    }
    /**
     * Represents a Depth Sorted Particle in the solid particle system.
     */
    class DepthSortedParticle {
        /**
         * Index of the particle in the "indices" array
         */
        ind: number;
        /**
         * Length of the particle shape in the "indices" array
         */
        indicesLength: number;
        /**
         * Squared distance from the particle to the camera
         */
        sqDistance: number;
    }
}

declare module BABYLON {
    /**
     * The SPS is a single updatable mesh. The solid particles are simply separate parts or faces fo this big mesh.
     *As it is just a mesh, the SPS has all the same properties than any other BJS mesh : not more, not less. It can be scaled, rotated, translated, enlighted, textured, moved, etc.

     * The SPS is also a particle system. It provides some methods to manage the particles.
     * However it is behavior agnostic. This means it has no emitter, no particle physics, no particle recycler. You have to implement your own behavior.
     *
     * Full documentation here : http://doc.babylonjs.com/overviews/Solid_Particle_System
     */
    class SolidParticleSystem implements IDisposable {
        /**
         *  The SPS array of Solid Particle objects. Just access each particle as with any classic array.
         *  Example : var p = SPS.particles[i];
         */
        particles: SolidParticle[];
        /**
         * The SPS total number of particles. Read only. Use SPS.counter instead if you need to set your own value.
         */
        nbParticles: number;
        /**
         * If the particles must ever face the camera (default false). Useful for planar particles.
         */
        billboard: boolean;
        /**
         * Recompute normals when adding a shape
         */
        recomputeNormals: boolean;
        /**
         * This a counter ofr your own usage. It's not set by any SPS functions.
         */
        counter: number;
        /**
         * The SPS name. This name is also given to the underlying mesh.
         */
        name: string;
        /**
         * The SPS mesh. It's a standard BJS Mesh, so all the methods from the Mesh class are avalaible.
         */
        mesh: Mesh;
        /**
         * This empty object is intended to store some SPS specific or temporary values in order to lower the Garbage Collector activity.
         * Please read : http://doc.babylonjs.com/overviews/Solid_Particle_System#garbage-collector-concerns
         */
        vars: any;
        /**
         * This array is populated when the SPS is set as 'pickable'.
         * Each key of this array is a `faceId` value that you can get from a pickResult object.
         * Each element of this array is an object `{idx: int, faceId: int}`.
         * `idx` is the picked particle index in the `SPS.particles` array
         * `faceId` is the picked face index counted within this particle.
         * Please read : http://doc.babylonjs.com/overviews/Solid_Particle_System#pickable-particles
         */
        pickedParticles: {
            idx: number;
            faceId: number;
        }[];
        /**
         * This array is populated when `enableDepthSort` is set to true.
         * Each element of this array is an instance of the class DepthSortedParticle.
         */
        depthSortedParticles: DepthSortedParticle[];
        /**
         * If the particle intersection must be computed only with the bounding sphere (no bounding box computation, so faster). (Internal use only)
         */
        _bSphereOnly: boolean;
        /**
         * A number to multiply the boundind sphere radius by in order to reduce it for instance. (Internal use only)
         */
        _bSphereRadiusFactor: number;
        private _scene;
        private _positions;
        private _indices;
        private _normals;
        private _colors;
        private _uvs;
        private _indices32;
        private _positions32;
        private _normals32;
        private _fixedNormal32;
        private _colors32;
        private _uvs32;
        private _index;
        private _updatable;
        private _pickable;
        private _isVisibilityBoxLocked;
        private _alwaysVisible;
        private _depthSort;
        private _shapeCounter;
        private _copy;
        private _shape;
        private _shapeUV;
        private _color;
        private _computeParticleColor;
        private _computeParticleTexture;
        private _computeParticleRotation;
        private _computeParticleVertex;
        private _computeBoundingBox;
        private _depthSortParticles;
        private _cam_axisZ;
        private _cam_axisY;
        private _cam_axisX;
        private _axisZ;
        private _camera;
        private _particle;
        private _camDir;
        private _camInvertedPosition;
        private _rotMatrix;
        private _invertMatrix;
        private _rotated;
        private _quaternion;
        private _vertex;
        private _normal;
        private _yaw;
        private _pitch;
        private _roll;
        private _halfroll;
        private _halfpitch;
        private _halfyaw;
        private _sinRoll;
        private _cosRoll;
        private _sinPitch;
        private _cosPitch;
        private _sinYaw;
        private _cosYaw;
        private _mustUnrotateFixedNormals;
        private _minimum;
        private _maximum;
        private _minBbox;
        private _maxBbox;
        private _particlesIntersect;
        private _depthSortFunction;
        private _needs32Bits;
        private _pivotBackTranslation;
        private _scaledPivot;
        private _particleHasParent;
        private _parent;
        /**
         * Creates a SPS (Solid Particle System) object.
         * @param name (String) is the SPS name, this will be the underlying mesh name.
         * @param scene (Scene) is the scene in which the SPS is added.
         * @param updatable (optional boolean, default true) : if the SPS must be updatable or immutable.
         * @param isPickable (optional boolean, default false) : if the solid particles must be pickable.
         * @param enableDepthSort (optional boolean, default false) : if the solid particles must be sorted in the geometry according to their distance to the camera.
         * @param particleIntersection (optional boolean, default false) : if the solid particle intersections must be computed.
         * @param boundingSphereOnly (optional boolean, default false) : if the particle intersection must be computed only with the bounding sphere (no bounding box computation, so faster).
         * @param bSphereRadiusFactor (optional float, default 1.0) : a number to multiply the boundind sphere radius by in order to reduce it for instance.
         * @example bSphereRadiusFactor = 1.0 / Math.sqrt(3.0) => the bounding sphere exactly matches a spherical mesh.
         */
        constructor(name: string, scene: Scene, options?: {
            updatable?: boolean;
            isPickable?: boolean;
            enableDepthSort?: boolean;
            particleIntersection?: boolean;
            boundingSphereOnly?: boolean;
            bSphereRadiusFactor?: number;
        });
        /**
         * Builds the SPS underlying mesh. Returns a standard Mesh.
         * If no model shape was added to the SPS, the returned mesh is just a single triangular plane.
         * @returns the created mesh
         */
        buildMesh(): Mesh;
        /**
         * Digests the mesh and generates as many solid particles in the system as wanted. Returns the SPS.
         * These particles will have the same geometry than the mesh parts and will be positioned at the same localisation than the mesh original places.
         * Thus the particles generated from `digest()` have their property `position` set yet.
         * @param mesh ( Mesh ) is the mesh to be digested
         * @param options {facetNb} (optional integer, default 1) is the number of mesh facets per particle, this parameter is overriden by the parameter `number` if any
         * {delta} (optional integer, default 0) is the random extra number of facets per particle , each particle will have between `facetNb` and `facetNb + delta` facets
         * {number} (optional positive integer) is the wanted number of particles : each particle is built with `mesh_total_facets / number` facets
         * @returns the current SPS
         */
        digest(mesh: Mesh, options?: {
            facetNb?: number;
            number?: number;
            delta?: number;
        }): SolidParticleSystem;
        private _unrotateFixedNormals();
        private _resetCopy();
        private _meshBuilder(p, shape, positions, meshInd, indices, meshUV, uvs, meshCol, colors, meshNor, normals, idx, idxInShape, options);
        private _posToShape(positions);
        private _uvsToShapeUV(uvs);
        private _addParticle(idx, idxpos, idxind, model, shapeId, idxInShape, bInfo?);
        /**
         * Adds some particles to the SPS from the model shape. Returns the shape id.
         * Please read the doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#create-an-immutable-sps
         * @param mesh is any Mesh object that will be used as a model for the solid particles.
         * @param nb (positive integer) the number of particles to be created from this model
         * @param options {positionFunction} is an optional javascript function to called for each particle on SPS creation.
         * {vertexFunction} is an optional javascript function to called for each vertex of each particle on SPS creation
         * @returns the number of shapes in the system
         */
        addShape(mesh: Mesh, nb: number, options?: {
            positionFunction?: any;
            vertexFunction?: any;
        }): number;
        private _rebuildParticle(particle);
        /**
         * Rebuilds the whole mesh and updates the VBO : custom positions and vertices are recomputed if needed.
         * @returns the SPS.
         */
        rebuildMesh(): SolidParticleSystem;
        /**
         *  Sets all the particles : this method actually really updates the mesh according to the particle positions, rotations, colors, textures, etc.
         *  This method calls `updateParticle()` for each particle of the SPS.
         *  For an animated SPS, it is usually called within the render loop.
         * @param start The particle index in the particle array where to start to compute the particle property values _(default 0)_
         * @param end The particle index in the particle array where to stop to compute the particle property values _(default nbParticle - 1)_
         * @param update If the mesh must be finally updated on this call after all the particle computations _(default true)_
         * @returns the SPS.
         */
        setParticles(start?: number, end?: number, update?: boolean): SolidParticleSystem;
        private _quaternionRotationYPR();
        private _quaternionToRotationMatrix();
        /**
        * Disposes the SPS.
        */
        dispose(): void;
        /**
         * Visibilty helper : Recomputes the visible size according to the mesh bounding box
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         * @returns the SPS.
         */
        refreshVisibleSize(): SolidParticleSystem;
        /**
         * Visibility helper : Sets the size of a visibility box, this sets the underlying mesh bounding box.
         * @param size the size (float) of the visibility box
         * note : this doesn't lock the SPS mesh bounding box.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         */
        setVisibilityBox(size: number): void;
        /**
         * Gets whether the SPS as always visible or not
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         */
        /**
         * Sets the SPS as always visible or not
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         */
        isAlwaysVisible: boolean;
        /**
         * Gets if the SPS visibility box as locked or not. This enables/disables the underlying mesh bounding box updates.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         */
        /**
         * Sets the SPS visibility box as locked or not. This enables/disables the underlying mesh bounding box updates.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#sps-visibility
         */
        isVisibilityBoxLocked: boolean;
        /**
         * Gets if `setParticles()` computes the particle rotations or not.
         * Default value : true. The SPS is faster when it's set to false.
         * Note : the particle rotations aren't stored values, so setting `computeParticleRotation` to false will prevents the particle to rotate.
         */
        /**
         * Tells to `setParticles()` to compute the particle rotations or not.
         * Default value : true. The SPS is faster when it's set to false.
         * Note : the particle rotations aren't stored values, so setting `computeParticleRotation` to false will prevents the particle to rotate.
         */
        computeParticleRotation: boolean;
        /**
         * Gets if `setParticles()` computes the particle colors or not.
         * Default value : true. The SPS is faster when it's set to false.
         * Note : the particle colors are stored values, so setting `computeParticleColor` to false will keep yet the last colors set.
         */
        /**
         * Tells to `setParticles()` to compute the particle colors or not.
         * Default value : true. The SPS is faster when it's set to false.
         * Note : the particle colors are stored values, so setting `computeParticleColor` to false will keep yet the last colors set.
         */
        computeParticleColor: boolean;
        /**
         * Gets if `setParticles()` computes the particle textures or not.
         * Default value : true. The SPS is faster when it's set to false.
         * Note : the particle textures are stored values, so setting `computeParticleTexture` to false will keep yet the last colors set.
         */
        computeParticleTexture: boolean;
        /**
         * Gets if `setParticles()` calls the vertex function for each vertex of each particle, or not.
         * Default value : false. The SPS is faster when it's set to false.
         * Note : the particle custom vertex positions aren't stored values.
         */
        /**
         * Tells to `setParticles()` to call the vertex function for each vertex of each particle, or not.
         * Default value : false. The SPS is faster when it's set to false.
         * Note : the particle custom vertex positions aren't stored values.
         */
        computeParticleVertex: boolean;
        /**
         * Gets if `setParticles()` computes or not the mesh bounding box when computing the particle positions.
         */
        /**
         * Tells to `setParticles()` to compute or not the mesh bounding box when computing the particle positions.
         */
        computeBoundingBox: boolean;
        /**
         * Gets if `setParticles()` sorts or not the distance between each particle and the camera.
         * Skipped when `enableDepthSort` is set to `false` (default) at construction time.
         * Default : `true`
         */
        /**
         * Tells to `setParticles()` to sort or not the distance between each particle and the camera.
         * Skipped when `enableDepthSort` is set to `false` (default) at construction time.
         * Default : `true`
         */
        depthSortParticles: boolean;
        /**
         * This function does nothing. It may be overwritten to set all the particle first values.
         * The SPS doesn't call this function, you may have to call it by your own.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#particle-management
         */
        initParticles(): void;
        /**
         * This function does nothing. It may be overwritten to recycle a particle.
         * The SPS doesn't call this function, you may have to call it by your own.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#particle-management
         * @param particle The particle to recycle
         * @returns the recycled particle
         */
        recycleParticle(particle: SolidParticle): SolidParticle;
        /**
         * Updates a particle : this function should  be overwritten by the user.
         * It is called on each particle by `setParticles()`. This is the place to code each particle behavior.
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#particle-management
         * @example : just set a particle position or velocity and recycle conditions
         * @param particle The particle to update
         * @returns the updated particle
         */
        updateParticle(particle: SolidParticle): SolidParticle;
        /**
         * Updates a vertex of a particle : it can be overwritten by the user.
         * This will be called on each vertex particle by `setParticles()` if `computeParticleVertex` is set to true only.
         * @param particle the current particle
         * @param vertex the current index of the current particle
         * @param pt the index of the current vertex in the particle shape
         * doc : http://doc.babylonjs.com/overviews/Solid_Particle_System#update-each-particle-shape
         * @example : just set a vertex particle position
         * @returns the updated vertex
         */
        updateParticleVertex(particle: SolidParticle, vertex: Vector3, pt: number): Vector3;
        /**
         * This will be called before any other treatment by `setParticles()` and will be passed three parameters.
         * This does nothing and may be overwritten by the user.
         * @param start the particle index in the particle array where to stop to iterate, same than the value passed to setParticle()
         * @param stop the particle index in the particle array where to stop to iterate, same than the value passed to setParticle()
         * @param update the boolean update value actually passed to setParticles()
         */
        beforeUpdateParticles(start?: number, stop?: number, update?: boolean): void;
        /**
         * This will be called  by `setParticles()` after all the other treatments and just before the actual mesh update.
         * This will be passed three parameters.
         * This does nothing and may be overwritten by the user.
         * @param start the particle index in the particle array where to stop to iterate, same than the value passed to setParticle()
         * @param stop the particle index in the particle array where to stop to iterate, same than the value passed to setParticle()
         * @param update the boolean update value actually passed to setParticles()
         */
        afterUpdateParticles(start?: number, stop?: number, update?: boolean): void;
    }
}

declare module BABYLON {
    class MorphTarget {
        name: string;
        animations: Animation[];
        private _positions;
        private _normals;
        private _tangents;
        private _influence;
        onInfluenceChanged: Observable<boolean>;
        influence: number;
        constructor(name: string, influence?: number);
        readonly hasPositions: boolean;
        readonly hasNormals: boolean;
        readonly hasTangents: boolean;
        setPositions(data: Nullable<FloatArray>): void;
        getPositions(): Nullable<FloatArray>;
        setNormals(data: Nullable<FloatArray>): void;
        getNormals(): Nullable<FloatArray>;
        setTangents(data: Nullable<FloatArray>): void;
        getTangents(): Nullable<FloatArray>;
        /**
         * Serializes the current target into a Serialization object.
         * Returns the serialized object.
         */
        serialize(): any;
        static Parse(serializationObject: any): MorphTarget;
        static FromMesh(mesh: AbstractMesh, name?: string, influence?: number): MorphTarget;
    }
}

declare module BABYLON {
    class MorphTargetManager {
        private _targets;
        private _targetObservable;
        private _activeTargets;
        private _scene;
        private _influences;
        private _supportsNormals;
        private _supportsTangents;
        private _vertexCount;
        private _uniqueId;
        private _tempInfluences;
        constructor(scene?: Nullable<Scene>);
        readonly uniqueId: number;
        readonly vertexCount: number;
        readonly supportsNormals: boolean;
        readonly supportsTangents: boolean;
        readonly numTargets: number;
        readonly numInfluencers: number;
        readonly influences: Float32Array;
        getActiveTarget(index: number): MorphTarget;
        getTarget(index: number): MorphTarget;
        addTarget(target: MorphTarget): void;
        removeTarget(target: MorphTarget): void;
        /**
         * Serializes the current manager into a Serialization object.
         * Returns the serialized object.
         */
        serialize(): any;
        private _syncActiveTargets(needUpdate);
        static Parse(serializationObject: any, scene: Scene): MorphTargetManager;
    }
}

declare module BABYLON {
    class AnaglyphPostProcess extends PostProcess {
        private _passedProcess;
        constructor(name: string, options: number | PostProcessOptions, rigCameras: Camera[], samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    class BlackAndWhitePostProcess extends PostProcess {
        degree: number;
        constructor(name: string, options: number | PostProcessOptions, camera: Camera, samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    /**
     * The bloom effect spreads bright areas of an image to simulate artifacts seen in cameras
     */
    class BloomEffect extends PostProcessRenderEffect {
        private bloomScale;
        /**
         * Internal
         */
        _effects: Array<PostProcess>;
        /**
         * Internal
         */
        _downscale: ExtractHighlightsPostProcess;
        private _blurX;
        private _blurY;
        private _merge;
        /**
         * The luminance threshold to find bright areas of the image to bloom.
         */
        threshold: number;
        /**
         * The strength of the bloom.
         */
        weight: number;
        /**
         * Specifies the size of the bloom blur kernel, relative to the final output size
         */
        kernel: number;
        /**
         * Creates a new instance of @see BloomEffect
         * @param scene The scene the effect belongs to.
         * @param bloomScale The ratio of the blur texture to the input texture that should be used to compute the bloom.
         * @param bloomKernel The size of the kernel to be used when applying the blur.
         * @param bloomWeight The the strength of bloom.
         * @param pipelineTextureType The type of texture to be used when performing the post processing.
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(scene: Scene, bloomScale: number, bloomWeight: number, bloomKernel: number, pipelineTextureType?: number, blockCompilation?: boolean);
        /**
         * Disposes each of the internal effects for a given camera.
         * @param camera The camera to dispose the effect on.
         */
        disposeEffects(camera: Camera): void;
        /**
         * Internal
         */
        _updateEffects(): void;
        /**
         * Internal
         * @returns if all the contained post processes are ready.
         */
        _isReady(): boolean;
    }
}

declare module BABYLON {
    /**
     * The BloomMergePostProcess merges blurred images with the original based on the values of the circle of confusion.
     */
    class BloomMergePostProcess extends PostProcess {
        /** Weight of the bloom to be added to the original input. */ weight: number;
        /**
         * Creates a new instance of @see BloomMergePostProcess
         * @param name The name of the effect.
         * @param originalFromInput Post process which's input will be used for the merge.
         * @param blurred Blurred highlights post process which's output will be used.
         * @param weight Weight of the bloom to be added to the original input.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, originalFromInput: PostProcess, blurred: PostProcess, /** Weight of the bloom to be added to the original input. */ weight: number, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    /**
     * The Blur Post Process which blurs an image based on a kernel and direction.
     * Can be used twice in x and y directions to perform a guassian blur in two passes.
     */
    class BlurPostProcess extends PostProcess {
        /** The direction in which to blur the image. */ direction: Vector2;
        private blockCompilation;
        protected _kernel: number;
        protected _idealKernel: number;
        protected _packedFloat: boolean;
        private _staticDefines;
        /**
         * Gets the length in pixels of the blur sample region
         */
        /**
         * Sets the length in pixels of the blur sample region
         */
        kernel: number;
        /**
         * Gets wether or not the blur is unpacking/repacking floats
         */
        /**
         * Sets wether or not the blur needs to unpack/repack floats
         */
        packedFloat: boolean;
        /**
         * Creates a new instance BlurPostProcess
         * @param name The name of the effect.
         * @param direction The direction in which to blur the image.
         * @param kernel The size of the kernel to be used when computing the blur. eg. Size of 3 will blur the center pixel by 2 pixels surrounding it.
         * @param options The required width/height ratio to downsize to before computing the render pass. (Use 1.0 for full size)
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, /** The direction in which to blur the image. */ direction: Vector2, kernel: number, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, defines?: string, blockCompilation?: boolean);
        /**
         * Updates the effect with the current post process compile time values and recompiles the shader.
         * @param defines Define statements that should be added at the beginning of the shader. (default: null)
         * @param uniforms Set of uniform variables that will be passed to the shader. (default: null)
         * @param samplers Set of Texture2D variables that will be passed to the shader. (default: null)
         * @param indexParameters The index parameters to be used for babylons include syntax "#include<kernelBlurVaryingDeclaration>[0..varyingCount]". (default: undefined) See usage in babylon.blurPostProcess.ts and kernelBlur.vertex.fx
         * @param onCompiled Called when the shader has been compiled.
         * @param onError Called if there is an error when compiling a shader.
         */
        updateEffect(defines?: Nullable<string>, uniforms?: Nullable<string[]>, samplers?: Nullable<string[]>, indexParameters?: any, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): void;
        protected _updateParameters(onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): void;
        /**
         * Best kernels are odd numbers that when divided by 2, their integer part is even, so 5, 9 or 13.
         * Other odd kernels optimize correctly but require proportionally more samples, even kernels are
         * possible but will produce minor visual artifacts. Since each new kernel requires a new shader we
         * want to minimize kernel changes, having gaps between physical kernels is helpful in that regard.
         * The gaps between physical kernels are compensated for in the weighting of the samples
         * @param idealKernel Ideal blur kernel.
         * @return Nearest best kernel.
         */
        protected _nearestBestKernel(idealKernel: number): number;
        /**
         * Calculates the value of a Gaussian distribution with sigma 3 at a given point.
         * @param x The point on the Gaussian distribution to sample.
         * @return the value of the Gaussian function at x.
         */
        protected _gaussianWeight(x: number): number;
        /**
          * Generates a string that can be used as a floating point number in GLSL.
          * @param x Value to print.
          * @param decimalFigures Number of decimal places to print the number to (excluding trailing 0s).
          * @return GLSL float string.
          */
        protected _glslFloat(x: number, decimalFigures?: number): string;
    }
}

declare module BABYLON {
    /**
     * The ChromaticAberrationPostProcess separates the rgb channels in an image to produce chromatic distortion around the edges of the screen
     */
    class ChromaticAberrationPostProcess extends PostProcess {
        /**
         * The amount of seperation of rgb channels (default: 30)
         */
        aberrationAmount: number;
        /**
         * The amount the effect will increase for pixels closer to the edge of the screen. (default: 0)
         */
        radialIntensity: number;
        /**
         * The normilized direction in which the rgb channels should be seperated. If set to 0,0 radial direction will be used. (default: Vector2(0.707,0.707))
         */
        direction: Vector2;
        /**
         * The center position where the radialIntensity should be around. [0.5,0.5 is center of screen, 1,1 is top right corder] (default: Vector2(0.5 ,0.5))
         */
        centerPosition: Vector2;
        /**
         * Creates a new instance ChromaticAberrationPostProcess
         * @param name The name of the effect.
         * @param screenWidth The width of the screen to apply the effect on.
         * @param screenHeight The height of the screen to apply the effect on.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, screenWidth: number, screenHeight: number, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    /**
     * The CircleOfConfusionPostProcess computes the circle of confusion value for each pixel given required lens parameters. See https://en.wikipedia.org/wiki/Circle_of_confusion
     */
    class CircleOfConfusionPostProcess extends PostProcess {
        /**
         * Max lens size in scene units/1000 (eg. millimeter). Standard cameras are 50mm. (default: 50) The diamater of the resulting aperture can be computed by lensSize/fStop.
         */
        lensSize: number;
        /**
         * F-Stop of the effect's camera. The diamater of the resulting aperture can be computed by lensSize/fStop. (default: 1.4)
         */
        fStop: number;
        /**
         * Distance away from the camera to focus on in scene units/1000 (eg. millimeter). (default: 2000)
         */
        focusDistance: number;
        /**
         * Focal length of the effect's camera in scene units/1000 (eg. millimeter). (default: 50)
         */
        focalLength: number;
        private _depthTexture;
        /**
         * Creates a new instance CircleOfConfusionPostProcess
         * @param name The name of the effect.
         * @param depthTexture The depth texture of the scene to compute the circle of confusion. This must be set in order for this to function but may be set after initialization if needed.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, depthTexture: Nullable<RenderTargetTexture>, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
        /**
         * Depth texture to be used to compute the circle of confusion. This must be set here or in the constructor in order for the post process to function.
         */
        depthTexture: RenderTargetTexture;
    }
}

declare module BABYLON {
    class ColorCorrectionPostProcess extends PostProcess {
        private _colorTableTexture;
        constructor(name: string, colorTableUrl: string, options: number | PostProcessOptions, camera: Camera, samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    /**
     * The ConvolutionPostProcess applies a 3x3 kernel to every pixel of the
     * input texture to perform effects such as edge detection or sharpening
     * See http://en.wikipedia.org/wiki/Kernel_(image_processing)
     */
    class ConvolutionPostProcess extends PostProcess {
        /** Array of 9 values corrisponding to the 3x3 kernel to be applied */ kernel: number[];
        /**
         * Creates a new instance ConvolutionPostProcess
         * @param name The name of the effect.
         * @param kernel Array of 9 values corrisponding to the 3x3 kernel to be applied
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         */
        constructor(name: string, /** Array of 9 values corrisponding to the 3x3 kernel to be applied */ kernel: number[], options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number);
        /**
         * Edge detection 0 see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static EdgeDetect0Kernel: number[];
        /**
         * Edge detection 1 see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static EdgeDetect1Kernel: number[];
        /**
         * Edge detection 2 see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static EdgeDetect2Kernel: number[];
        /**
         * Kernel to sharpen an image see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static SharpenKernel: number[];
        /**
         * Kernel to emboss an image see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static EmbossKernel: number[];
        /**
         * Kernel to blur an image see https://en.wikipedia.org/wiki/Kernel_(image_processing)
         */
        static GaussianKernel: number[];
    }
}

declare module BABYLON {
    /**
     * The DepthOfFieldBlurPostProcess applied a blur in a give direction.
     * This blur differs from the standard BlurPostProcess as it attempts to avoid blurring pixels
     * based on samples that have a large difference in distance than the center pixel.
     * See section 2.6.2 http://fileadmin.cs.lth.se/cs/education/edan35/lectures/12dof.pdf
     */
    class DepthOfFieldBlurPostProcess extends BlurPostProcess {
        direction: Vector2;
        /**
         * Creates a new instance CircleOfConfusionPostProcess
         * @param name The name of the effect.
         * @param scene The scene the effect belongs to.
         * @param direction The direction the blur should be applied.
         * @param kernel The size of the kernel used to blur.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param circleOfConfusion The circle of confusion + depth map to be used to avoid blurring accross edges
         * @param imageToBlur The image to apply the blur to (default: Current rendered frame)
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, scene: Scene, direction: Vector2, kernel: number, options: number | PostProcessOptions, camera: Nullable<Camera>, circleOfConfusion: PostProcess, imageToBlur?: Nullable<PostProcess>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    /**
     * Specifies the level of max blur that should be applied when using the depth of field effect
     */
    enum DepthOfFieldEffectBlurLevel {
        /**
         * Subtle blur
         */
        Low = 0,
        /**
         * Medium blur
         */
        Medium = 1,
        /**
         * Large blur
         */
        High = 2,
    }
    /**
     * The depth of field effect applies a blur to objects that are closer or further from where the camera is focusing.
     */
    class DepthOfFieldEffect extends PostProcessRenderEffect {
        private _circleOfConfusion;
        /**
         * Internal, blurs from high to low
         */
        _depthOfFieldBlurX: Array<DepthOfFieldBlurPostProcess>;
        private _depthOfFieldBlurY;
        private _dofMerge;
        /**
         * Internal post processes in depth of field effect
         */
        _effects: Array<PostProcess>;
        /**
         * The focal the length of the camera used in the effect
         */
        focalLength: number;
        /**
         * F-Stop of the effect's camera. The diamater of the resulting aperture can be computed by lensSize/fStop. (default: 1.4)
         */
        fStop: number;
        /**
         * Distance away from the camera to focus on in scene units/1000 (eg. millimeter). (default: 2000)
         */
        focusDistance: number;
        /**
         * Max lens size in scene units/1000 (eg. millimeter). Standard cameras are 50mm. (default: 50) The diamater of the resulting aperture can be computed by lensSize/fStop.
         */
        lensSize: number;
        /**
         * Creates a new instance DepthOfFieldEffect
         * @param scene The scene the effect belongs to.
         * @param depthTexture The depth texture of the scene to compute the circle of confusion.This must be set in order for this to function but may be set after initialization if needed.
         * @param pipelineTextureType The type of texture to be used when performing the post processing.
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(scene: Scene, depthTexture: Nullable<RenderTargetTexture>, blurLevel?: DepthOfFieldEffectBlurLevel, pipelineTextureType?: number, blockCompilation?: boolean);
        /**
         * Depth texture to be used to compute the circle of confusion. This must be set here or in the constructor in order for the post process to function.
         */
        depthTexture: RenderTargetTexture;
        /**
         * Disposes each of the internal effects for a given camera.
         * @param camera The camera to dispose the effect on.
         */
        disposeEffects(camera: Camera): void;
        /**
         * Internal
         */
        _updateEffects(): void;
        /**
         * Internal
         * @returns if all the contained post processes are ready.
         */
        _isReady(): boolean;
    }
}

declare module BABYLON {
    /**
     * Options to be set when merging outputs from the default pipeline.
     */
    class DepthOfFieldMergePostProcessOptions {
        /**
         * The original image to merge on top of
         */
        originalFromInput: PostProcess;
        /**
         * Parameters to perform the merge of the depth of field effect
         */
        depthOfField?: {
            circleOfConfusion: PostProcess;
            blurSteps: Array<PostProcess>;
        };
        /**
         * Parameters to perform the merge of bloom effect
         */
        bloom?: {
            blurred: PostProcess;
            weight: number;
        };
    }
    /**
     * The DepthOfFieldMergePostProcess merges blurred images with the original based on the values of the circle of confusion.
     */
    class DepthOfFieldMergePostProcess extends PostProcess {
        private blurSteps;
        /**
         * Creates a new instance of DepthOfFieldMergePostProcess
         * @param name The name of the effect.
         * @param originalFromInput Post process which's input will be used for the merge.
         * @param circleOfConfusion Circle of confusion post process which's output will be used to blur each pixel.
         * @param blurSteps Blur post processes from low to high which will be mixed with the original image.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, originalFromInput: PostProcess, circleOfConfusion: PostProcess, blurSteps: Array<PostProcess>, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
        /**
         * Updates the effect with the current post process compile time values and recompiles the shader.
         * @param defines Define statements that should be added at the beginning of the shader. (default: null)
         * @param uniforms Set of uniform variables that will be passed to the shader. (default: null)
         * @param samplers Set of Texture2D variables that will be passed to the shader. (default: null)
         * @param indexParameters The index parameters to be used for babylons include syntax "#include<kernelBlurVaryingDeclaration>[0..varyingCount]". (default: undefined) See usage in babylon.blurPostProcess.ts and kernelBlur.vertex.fx
         * @param onCompiled Called when the shader has been compiled.
         * @param onError Called if there is an error when compiling a shader.
         */
        updateEffect(defines?: Nullable<string>, uniforms?: Nullable<string[]>, samplers?: Nullable<string[]>, indexParameters?: any, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): void;
    }
}

declare module BABYLON {
    class DisplayPassPostProcess extends PostProcess {
        constructor(name: string, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    /**
     * The extract highlights post process sets all pixels to black except pixels above the specified luminance threshold. Used as the first step for a bloom effect.
     */
    class ExtractHighlightsPostProcess extends PostProcess {
        /**
         * The luminance threshold, pixels below this value will be set to black.
         */
        threshold: number;
        /**
         * Internal
         */
        _exposure: number;
        /**
         * Post process which has the input texture to be used when performing highlight extraction
         */
        _inputPostProcess: Nullable<PostProcess>;
        constructor(name: string, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    class FilterPostProcess extends PostProcess {
        kernelMatrix: Matrix;
        constructor(name: string, kernelMatrix: Matrix, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    class FxaaPostProcess extends PostProcess {
        texelWidth: number;
        texelHeight: number;
        constructor(name: string, options: number | PostProcessOptions, camera?: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number);
    }
}

declare module BABYLON {
    /**
     * The GrainPostProcess adds noise to the image at mid luminance levels
     */
    class GrainPostProcess extends PostProcess {
        /**
         * The intensity of the grain added (default: 30)
         */
        intensity: number;
        /**
         * If the grain should be randomized on every frame
         */
        animated: boolean;
        /**
         * Creates a new instance of @see GrainPostProcess
         * @param name The name of the effect.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    class HighlightsPostProcess extends PostProcess {
        constructor(name: string, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number);
    }
}

declare module BABYLON {
    class ImageProcessingPostProcess extends PostProcess {
        /**
         * Default configuration related to image processing available in the PBR Material.
         */
        protected _imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Gets the image processing configuration used either in this material.
         */
        /**
         * Sets the Default image processing configuration used either in the this material.
         *
         * If sets to null, the scene one is in use.
         */
        imageProcessingConfiguration: ImageProcessingConfiguration;
        /**
         * Keep track of the image processing observer to allow dispose and replace.
         */
        private _imageProcessingObserver;
        /**
         * Attaches a new image processing configuration to the PBR Material.
         * @param configuration
         */
        protected _attachImageProcessingConfiguration(configuration: Nullable<ImageProcessingConfiguration>, doNotBuild?: boolean): void;
        /**
         * Gets Color curves setup used in the effect if colorCurvesEnabled is set to true .
         */
        /**
         * Sets Color curves setup used in the effect if colorCurvesEnabled is set to true .
         */
        colorCurves: Nullable<ColorCurves>;
        /**
         * Gets wether the color curves effect is enabled.
         */
        /**
         * Sets wether the color curves effect is enabled.
         */
        colorCurvesEnabled: boolean;
        /**
         * Gets Color grading LUT texture used in the effect if colorGradingEnabled is set to true.
         */
        /**
         * Sets Color grading LUT texture used in the effect if colorGradingEnabled is set to true.
         */
        colorGradingTexture: Nullable<BaseTexture>;
        /**
         * Gets wether the color grading effect is enabled.
         */
        /**
         * Gets wether the color grading effect is enabled.
         */
        colorGradingEnabled: boolean;
        /**
         * Gets exposure used in the effect.
         */
        /**
         * Sets exposure used in the effect.
         */
        exposure: number;
        /**
         * Gets wether tonemapping is enabled or not.
         */
        /**
         * Sets wether tonemapping is enabled or not
         */
        toneMappingEnabled: boolean;
        /**
         * Gets contrast used in the effect.
         */
        /**
         * Sets contrast used in the effect.
         */
        contrast: number;
        /**
         * Gets Vignette stretch size.
         */
        /**
         * Sets Vignette stretch size.
         */
        vignetteStretch: number;
        /**
         * Gets Vignette centre X Offset.
         */
        /**
         * Sets Vignette centre X Offset.
         */
        vignetteCentreX: number;
        /**
         * Gets Vignette centre Y Offset.
         */
        /**
         * Sets Vignette centre Y Offset.
         */
        vignetteCentreY: number;
        /**
         * Gets Vignette weight or intensity of the vignette effect.
         */
        /**
         * Sets Vignette weight or intensity of the vignette effect.
         */
        vignetteWeight: number;
        /**
         * Gets Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        /**
         * Sets Color of the vignette applied on the screen through the chosen blend mode (vignetteBlendMode)
         * if vignetteEnabled is set to true.
         */
        vignetteColor: Color4;
        /**
         * Gets Camera field of view used by the Vignette effect.
         */
        /**
         * Sets Camera field of view used by the Vignette effect.
         */
        vignetteCameraFov: number;
        /**
         * Gets the vignette blend mode allowing different kind of effect.
         */
        /**
         * Sets the vignette blend mode allowing different kind of effect.
         */
        vignetteBlendMode: number;
        /**
         * Gets wether the vignette effect is enabled.
         */
        /**
         * Sets wether the vignette effect is enabled.
         */
        vignetteEnabled: boolean;
        private _fromLinearSpace;
        /**
         * Gets wether the input of the processing is in Gamma or Linear Space.
         */
        /**
         * Sets wether the input of the processing is in Gamma or Linear Space.
         */
        fromLinearSpace: boolean;
        /**
         * Defines cache preventing GC.
         */
        private _defines;
        constructor(name: string, options: number | PostProcessOptions, camera?: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, imageProcessingConfiguration?: ImageProcessingConfiguration);
        getClassName(): string;
        protected _updateParameters(): void;
        dispose(camera?: Camera): void;
    }
}

declare module BABYLON {
    class PassPostProcess extends PostProcess {
        constructor(name: string, options: number | PostProcessOptions, camera?: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    type PostProcessOptions = {
        width: number;
        height: number;
    };
    /**
     * PostProcess can be used to apply a shader to a texture after it has been rendered
     * See https://doc.babylonjs.com/how_to/how_to_use_postprocesses
     */
    class PostProcess {
        /** Name of the PostProcess. */ name: string;
        /**
        * Width of the texture to apply the post process on
        */
        width: number;
        /**
        * Height of the texture to apply the post process on
        */
        height: number;
        /**
        * Internal, reference to the location where this postprocess was output to. (Typically the texture on the next postprocess in the chain)
        */
        _outputTexture: Nullable<InternalTexture>;
        /**
        * Sampling mode used by the shader
        * See https://doc.babylonjs.com/classes/3.1/texture
        */
        renderTargetSamplingMode: number;
        /**
        * Clear color to use when screen clearing
        */
        clearColor: Color4;
        /**
        * If the buffer needs to be cleared before applying the post process. (default: true)
        * Should be set to false if shader will overwrite all previous pixels.
        */
        autoClear: boolean;
        /**
        * Type of alpha mode to use when performing the post process (default: Engine.ALPHA_DISABLE)
        */
        alphaMode: number;
        /**
        * Sets the setAlphaBlendConstants of the babylon engine
        */
        alphaConstants: Color4;
        /**
        * Animations to be used for the post processing
        */
        animations: Animation[];
        /**
         * Enable Pixel Perfect mode where texture is not scaled to be power of 2.
         * Can only be used on a single postprocess or on the last one of a chain. (default: false)
         */
        enablePixelPerfectMode: boolean;
        /**
        * Scale mode for the post process (default: Engine.SCALEMODE_FLOOR)
        */
        scaleMode: number;
        /**
        * Force textures to be a power of two (default: false)
        */
        alwaysForcePOT: boolean;
        /**
        * Number of sample textures (default: 1)
        */
        samples: number;
        /**
        * Modify the scale of the post process to be the same as the viewport (default: false)
        */
        adaptScaleToCurrentViewport: boolean;
        private _camera;
        private _scene;
        private _engine;
        private _options;
        private _reusable;
        private _textureType;
        /**
        * Smart array of input and output textures for the post process.
        */
        _textures: SmartArray<InternalTexture>;
        /**
        * The index in _textures that corresponds to the output texture.
        */
        _currentRenderTextureInd: number;
        private _effect;
        private _samplers;
        private _fragmentUrl;
        private _vertexUrl;
        private _parameters;
        private _scaleRatio;
        protected _indexParameters: any;
        private _shareOutputWithPostProcess;
        private _texelSize;
        private _forcedOutputTexture;
        /**
        * An event triggered when the postprocess is activated.
        */
        onActivateObservable: Observable<Camera>;
        private _onActivateObserver;
        /**
        * A function that is added to the onActivateObservable
        */
        onActivate: Nullable<(camera: Camera) => void>;
        /**
        * An event triggered when the postprocess changes its size.
        */
        onSizeChangedObservable: Observable<PostProcess>;
        private _onSizeChangedObserver;
        /**
        * A function that is added to the onSizeChangedObservable
        */
        onSizeChanged: (postProcess: PostProcess) => void;
        /**
        * An event triggered when the postprocess applies its effect.
        */
        onApplyObservable: Observable<Effect>;
        private _onApplyObserver;
        /**
        * A function that is added to the onApplyObservable
        */
        onApply: (effect: Effect) => void;
        /**
        * An event triggered before rendering the postprocess
        */
        onBeforeRenderObservable: Observable<Effect>;
        private _onBeforeRenderObserver;
        /**
        * A function that is added to the onBeforeRenderObservable
        */
        onBeforeRender: (effect: Effect) => void;
        /**
        * An event triggered after rendering the postprocess
        */
        onAfterRenderObservable: Observable<Effect>;
        private _onAfterRenderObserver;
        /**
        * A function that is added to the onAfterRenderObservable
        */
        onAfterRender: (efect: Effect) => void;
        /**
        * The input texture for this post process and the output texture of the previous post process. When added to a pipeline the previous post process will
        * render it's output into this texture and this texture will be used as textureSampler in the fragment shader of this post process.
        */
        inputTexture: InternalTexture;
        /**
        * Gets the camera which post process is applied to.
        * @returns The camera the post process is applied to.
        */
        getCamera(): Camera;
        /**
        * Gets the texel size of the postprocess.
        * See https://en.wikipedia.org/wiki/Texel_(graphics)
        */
        readonly texelSize: Vector2;
        /**
         * Creates a new instance PostProcess
         * @param name The name of the PostProcess.
         * @param fragmentUrl The url of the fragment shader to be used.
         * @param parameters Array of the names of uniform non-sampler2D variables that will be passed to the shader.
         * @param samplers Array of the names of uniform sampler2D variables that will be passed to the shader.
         * @param options The required width/height ratio to downsize to before computing the render pass. (Use 1.0 for full size)
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param defines String of defines that will be set when running the fragment shader. (default: null)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param vertexUrl The url of the vertex shader to be used. (default: "postprocess")
         * @param indexParameters The index parameters to be used for babylons include syntax "#include<kernelBlurVaryingDeclaration>[0..varyingCount]". (default: undefined) See usage in babylon.blurPostProcess.ts and kernelBlur.vertex.fx
         * @param blockCompilation If the shader should not be compiled imediatly. (default: false)
         */
        constructor(/** Name of the PostProcess. */ name: string, fragmentUrl: string, parameters: Nullable<string[]>, samplers: Nullable<string[]>, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, defines?: Nullable<string>, textureType?: number, vertexUrl?: string, indexParameters?: any, blockCompilation?: boolean);
        /**
         * Gets the engine which this post process belongs to.
         * @returns The engine the post process was enabled with.
         */
        getEngine(): Engine;
        /**
         * The effect that is created when initializing the post process.
         * @returns The created effect corrisponding the the postprocess.
         */
        getEffect(): Effect;
        /**
         * To avoid multiple redundant textures for multiple post process, the output the output texture for this post process can be shared with another.
         * @param postProcess The post process to share the output with.
         * @returns This post process.
         */
        shareOutputWith(postProcess: PostProcess): PostProcess;
        /**
         * Reverses the effect of calling shareOutputWith and returns the post process back to its original state.
         * This should be called if the post process that shares output with this post process is disabled/disposed.
         */
        useOwnOutput(): void;
        /**
         * Updates the effect with the current post process compile time values and recompiles the shader.
         * @param defines Define statements that should be added at the beginning of the shader. (default: null)
         * @param uniforms Set of uniform variables that will be passed to the shader. (default: null)
         * @param samplers Set of Texture2D variables that will be passed to the shader. (default: null)
         * @param indexParameters The index parameters to be used for babylons include syntax "#include<kernelBlurVaryingDeclaration>[0..varyingCount]". (default: undefined) See usage in babylon.blurPostProcess.ts and kernelBlur.vertex.fx
         * @param onCompiled Called when the shader has been compiled.
         * @param onError Called if there is an error when compiling a shader.
         */
        updateEffect(defines?: Nullable<string>, uniforms?: Nullable<string[]>, samplers?: Nullable<string[]>, indexParameters?: any, onCompiled?: (effect: Effect) => void, onError?: (effect: Effect, errors: string) => void): void;
        /**
         * The post process is reusable if it can be used multiple times within one frame.
         * @returns If the post process is reusable
         */
        isReusable(): boolean;
        /** invalidate frameBuffer to hint the postprocess to create a depth buffer */
        markTextureDirty(): void;
        /**
         * Activates the post process by intializing the textures to be used when executed. Notifies onActivateObservable.
         * When this post process is used in a pipeline, this is call will bind the input texture of this post process to the output of the previous.
         * @param camera The camera that will be used in the post process. This camera will be used when calling onActivateObservable.
         * @param sourceTexture The source texture to be inspected to get the width and height if not specified in the post process constructor. (default: null)
         * @param forceDepthStencil If true, a depth and stencil buffer will be generated. (default: false)
         * @returns The target texture that was bound to be written to.
         */
        activate(camera: Nullable<Camera>, sourceTexture?: Nullable<InternalTexture>, forceDepthStencil?: boolean): InternalTexture;
        /**
         * If the post process is supported.
         */
        readonly isSupported: boolean;
        /**
         * The aspect ratio of the output texture.
         */
        readonly aspectRatio: number;
        /**
         * Get a value indicating if the post-process is ready to be used
         * @returns true if the post-process is ready (shader is compiled)
         */
        isReady(): boolean;
        /**
         * Binds all textures and uniforms to the shader, this will be run on every pass.
         * @returns the effect corrisponding to this post process. Null if not compiled or not ready.
         */
        apply(): Nullable<Effect>;
        private _disposeTextures();
        /**
         * Disposes the post process.
         * @param camera The camera to dispose the post process on.
         */
        dispose(camera?: Camera): void;
    }
}

declare module BABYLON {
    /**
     * PostProcessManager is used to manage one or more post processes or post process pipelines
     * See https://doc.babylonjs.com/how_to/how_to_use_postprocesses
     */
    class PostProcessManager {
        private _scene;
        private _indexBuffer;
        private _vertexBuffers;
        /**
         * Creates a new instance PostProcess
         * @param scene The scene that the post process is associated with.
         */
        constructor(scene: Scene);
        private _prepareBuffers();
        private _buildIndexBuffer();
        /**
         * Rebuilds the vertex buffers of the manager.
         */
        _rebuild(): void;
        /**
         * Prepares a frame to be run through a post process.
         * @param sourceTexture The input texture to the post procesess. (default: null)
         * @param postProcesses An array of post processes to be run. (default: null)
         * @returns True if the post processes were able to be run.
         */
        _prepareFrame(sourceTexture?: Nullable<InternalTexture>, postProcesses?: Nullable<PostProcess[]>): boolean;
        /**
         * Manually render a set of post processes to a texture.
         * @param postProcesses An array of post processes to be run.
         * @param targetTexture The target texture to render to.
         * @param forceFullscreenViewport force gl.viewport to be full screen eg. 0,0,textureWidth,textureHeight
         */
        directRender(postProcesses: PostProcess[], targetTexture?: Nullable<InternalTexture>, forceFullscreenViewport?: boolean): void;
        /**
         * Finalize the result of the output of the postprocesses.
         * @param doNotPresent If true the result will not be displayed to the screen.
         * @param targetTexture The target texture to render to.
         * @param faceIndex The index of the face to bind the target texture to.
         * @param postProcesses The array of post processes to render.
         * @param forceFullscreenViewport force gl.viewport to be full screen eg. 0,0,textureWidth,textureHeight (default: false)
         */
        _finalizeFrame(doNotPresent?: boolean, targetTexture?: InternalTexture, faceIndex?: number, postProcesses?: Array<PostProcess>, forceFullscreenViewport?: boolean): void;
        /**
         * Disposes of the post process manager.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class RefractionPostProcess extends PostProcess {
        color: Color3;
        depth: number;
        colorLevel: number;
        private _refTexture;
        private _ownRefractionTexture;
        /**
         * Gets or sets the refraction texture
         * Please note that you are responsible for disposing the texture if you set it manually
         */
        refractionTexture: Texture;
        constructor(name: string, refractionTextureUrl: string, color: Color3, depth: number, colorLevel: number, options: number | PostProcessOptions, camera: Camera, samplingMode?: number, engine?: Engine, reusable?: boolean);
        dispose(camera: Camera): void;
    }
}

declare module BABYLON {
    /**
     * The SharpenPostProcess applies a sharpen kernel to every pixel
     * See http://en.wikipedia.org/wiki/Kernel_(image_processing)
     */
    class SharpenPostProcess extends PostProcess {
        /**
         * How much of the original color should be applied. Setting this to 0 will display edge detection. (default: 1)
         */
        colorAmount: number;
        /**
         * How much sharpness should be applied (default: 0.3)
         */
        edgeAmount: number;
        /**
         * Creates a new instance ConvolutionPostProcess
         * @param name The name of the effect.
         * @param options The required width/height ratio to downsize to before computing the render pass.
         * @param camera The camera to apply the render pass to.
         * @param samplingMode The sampling mode to be used when computing the pass. (default: 0)
         * @param engine The engine which the post process will be applied. (default: current engine)
         * @param reusable If the post process can be reused on the same frame. (default: false)
         * @param textureType Type of textures used when performing the post process. (default: 0)
         * @param blockCompilation If compilation of the shader should not be done in the constructor. The updateEffect method can be used to compile the shader at a later time. (default: false)
         */
        constructor(name: string, options: number | PostProcessOptions, camera: Nullable<Camera>, samplingMode?: number, engine?: Engine, reusable?: boolean, textureType?: number, blockCompilation?: boolean);
    }
}

declare module BABYLON {
    class StereoscopicInterlacePostProcess extends PostProcess {
        private _stepSize;
        private _passedProcess;
        constructor(name: string, rigCameras: Camera[], isStereoscopicHoriz: boolean, samplingMode?: number, engine?: Engine, reusable?: boolean);
    }
}

declare module BABYLON {
    enum TonemappingOperator {
        Hable = 0,
        Reinhard = 1,
        HejiDawson = 2,
        Photographic = 3,
    }
    class TonemapPostProcess extends PostProcess {
        private _operator;
        exposureAdjustment: number;
        constructor(name: string, _operator: TonemappingOperator, exposureAdjustment: number, camera: Camera, samplingMode?: number, engine?: Engine, textureFormat?: number);
    }
}

declare module BABYLON {
    class VolumetricLightScatteringPostProcess extends PostProcess {
        private _volumetricLightScatteringPass;
        private _volumetricLightScatteringRTT;
        private _viewPort;
        private _screenCoordinates;
        private _cachedDefines;
        /**
        * If not undefined, the mesh position is computed from the attached node position
        */
        attachedNode: {
            position: Vector3;
        };
        /**
        * Custom position of the mesh. Used if "useCustomMeshPosition" is set to "true"
        */
        customMeshPosition: Vector3;
        /**
        * Set if the post-process should use a custom position for the light source (true) or the internal mesh position (false)
        */
        useCustomMeshPosition: boolean;
        /**
        * If the post-process should inverse the light scattering direction
        */
        invert: boolean;
        /**
        * The internal mesh used by the post-process
        */
        mesh: Mesh;
        useDiffuseColor: boolean;
        /**
        * Array containing the excluded meshes not rendered in the internal pass
        */
        excludedMeshes: AbstractMesh[];
        /**
        * Controls the overall intensity of the post-process
        */
        exposure: number;
        /**
        * Dissipates each sample's contribution in range [0, 1]
        */
        decay: number;
        /**
        * Controls the overall intensity of each sample
        */
        weight: number;
        /**
        * Controls the density of each sample
        */
        density: number;
        /**
         * @constructor
         * @param {string} name - The post-process name
         * @param {any} ratio - The size of the post-process and/or internal pass (0.5 means that your postprocess will have a width = canvas.width 0.5 and a height = canvas.height 0.5)
         * @param {BABYLON.Camera} camera - The camera that the post-process will be attached to
         * @param {BABYLON.Mesh} mesh - The mesh used to create the light scattering
         * @param {number} samples - The post-process quality, default 100
         * @param {number} samplingMode - The post-process filtering mode
         * @param {BABYLON.Engine} engine - The babylon engine
         * @param {boolean} reusable - If the post-process is reusable
         * @param {BABYLON.Scene} scene - The constructor needs a scene reference to initialize internal components. If "camera" is null (RenderPipelineà, "scene" must be provided
         */
        constructor(name: string, ratio: any, camera: Camera, mesh?: Mesh, samples?: number, samplingMode?: number, engine?: Engine, reusable?: boolean, scene?: Scene);
        getClassName(): string;
        private _isReady(subMesh, useInstances);
        /**
         * Sets the new light position for light scattering effect
         * @param {BABYLON.Vector3} The new custom light position
         */
        setCustomMeshPosition(position: Vector3): void;
        /**
         * Returns the light position for light scattering effect
         * @return {BABYLON.Vector3} The custom light position
         */
        getCustomMeshPosition(): Vector3;
        /**
         * Disposes the internal assets and detaches the post-process from the camera
         */
        dispose(camera: Camera): void;
        /**
         * Returns the render target texture used by the post-process
         * @return {BABYLON.RenderTargetTexture} The render target texture used by the post-process
         */
        getPass(): RenderTargetTexture;
        private _meshExcluded(mesh);
        private _createPass(scene, ratio);
        private _updateMeshScreenCoordinates(scene);
        /**
        * Creates a default mesh for the Volumeric Light Scattering post-process
        * @param {string} The mesh name
        * @param {BABYLON.Scene} The scene where to create the mesh
        * @return {BABYLON.Mesh} the default mesh
        */
        static CreateDefaultMesh(name: string, scene: Scene): Mesh;
    }
}

declare module BABYLON {
    class VRDistortionCorrectionPostProcess extends PostProcess {
        private _isRightEye;
        private _distortionFactors;
        private _postProcessScaleFactor;
        private _lensCenterOffset;
        private _scaleIn;
        private _scaleFactor;
        private _lensCenter;
        constructor(name: string, camera: Camera, isRightEye: boolean, vrMetrics: VRCameraMetrics);
    }
}

declare module BABYLON {
    class ReflectionProbe {
        name: string;
        private _scene;
        private _renderTargetTexture;
        private _projectionMatrix;
        private _viewMatrix;
        private _target;
        private _add;
        private _attachedMesh;
        private _invertYAxis;
        position: Vector3;
        constructor(name: string, size: number, scene: Scene, generateMipMaps?: boolean);
        samples: number;
        refreshRate: number;
        getScene(): Scene;
        readonly cubeTexture: RenderTargetTexture;
        readonly renderList: Nullable<AbstractMesh[]>;
        attachToMesh(mesh: AbstractMesh): void;
        /**
         * Specifies whether or not the stencil and depth buffer are cleared between two rendering groups.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param autoClearDepthStencil Automatically clears depth and stencil between groups if true.
         */
        setRenderingAutoClearDepthStencil(renderingGroupId: number, autoClearDepthStencil: boolean): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class BoundingBoxRenderer {
        frontColor: Color3;
        backColor: Color3;
        showBackLines: boolean;
        renderList: SmartArray<BoundingBox>;
        private _scene;
        private _colorShader;
        private _vertexBuffers;
        private _indexBuffer;
        constructor(scene: Scene);
        private _prepareRessources();
        private _createIndexBuffer();
        _rebuild(): void;
        reset(): void;
        render(): void;
        renderOcclusionBoundingBox(mesh: AbstractMesh): void;
        dispose(): void;
    }
}

declare module BABYLON {
    /**
     * This represents a depth renderer in Babylon.
     * A depth renderer will render to it's depth map every frame which can be displayed or used in post processing
     */
    class DepthRenderer {
        private _scene;
        private _depthMap;
        private _effect;
        private _cachedDefines;
        private _camera;
        /**
         * Instantiates a depth renderer
         * @param scene The scene the renderer belongs to
         * @param type The texture type of the depth map (default: Engine.TEXTURETYPE_FLOAT)
         * @param camera The camera to be used to render the depth map (default: scene's active camera)
         */
        constructor(scene: Scene, type?: number, camera?: Nullable<Camera>);
        /**
         * Creates the depth rendering effect and checks if the effect is ready.
         * @param subMesh The submesh to be used to render the depth map of
         * @param useInstances If multiple world instances should be used
         * @returns if the depth renderer is ready to render the depth map
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Gets the texture which the depth map will be written to.
         * @returns The depth map texture
         */
        getDepthMap(): RenderTargetTexture;
        /**
         * Disposes of the depth renderer.
         */
        dispose(): void;
    }
}

declare module BABYLON {
    class EdgesRenderer {
        edgesWidthScalerForOrthographic: number;
        edgesWidthScalerForPerspective: number;
        private _source;
        private _linesPositions;
        private _linesNormals;
        private _linesIndices;
        private _epsilon;
        private _indicesCount;
        private _lineShader;
        private _ib;
        private _buffers;
        private _checkVerticesInsteadOfIndices;
        constructor(source: AbstractMesh, epsilon?: number, checkVerticesInsteadOfIndices?: boolean);
        private _prepareRessources();
        _rebuild(): void;
        dispose(): void;
        private _processEdgeForAdjacencies(pa, pb, p0, p1, p2);
        private _processEdgeForAdjacenciesWithVertices(pa, pb, p0, p1, p2);
        private _checkEdge(faceIndex, edge, faceNormals, p0, p1);
        _generateEdgesLines(): void;
        render(): void;
    }
}

declare module BABYLON {
    /**
     * This renderer is helpfull to fill one of the render target with a geometry buffer.
     */
    class GeometryBufferRenderer {
        private _scene;
        private _multiRenderTarget;
        private _ratio;
        private _enablePosition;
        protected _effect: Effect;
        protected _cachedDefines: string;
        /**
         * Set the render list (meshes to be rendered) used in the G buffer.
         */
        renderList: Mesh[];
        /**
         * Gets wether or not G buffer are supported by the running hardware.
         * This requires draw buffer supports
         */
        readonly isSupported: boolean;
        /**
         * Gets wether or not position are enabled for the G buffer.
         */
        /**
         * Sets wether or not position are enabled for the G buffer.
         */
        enablePosition: boolean;
        /**
         * Gets the scene associated with the buffer.
         */
        readonly scene: Scene;
        /**
         * Gets the ratio used by the buffer during its creation.
         * How big is the buffer related to the main canvas.
         */
        readonly ratio: number;
        /**
         * Creates a new G Buffer for the scene
         * @param scene The scene the buffer belongs to
         * @param ratio How big is the buffer related to the main canvas.
         */
        constructor(scene: Scene, ratio?: number);
        /**
         * Checks wether everything is ready to render a submesh to the G buffer.
         * @param subMesh the submesh to check readiness for
         * @param useInstances is the mesh drawn using instance or not
         * @returns true if ready otherwise false
         */
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
        /**
         * Gets the current underlying G Buffer.
         * @returns the buffer
         */
        getGBuffer(): MultiRenderTarget;
        /**
         * Gets the number of samples used to render the buffer (anti aliasing).
         */
        /**
         * Sets the number of samples used to render the buffer (anti aliasing).
         */
        samples: number;
        /**
         * Disposes the renderer and frees up associated resources.
         */
        dispose(): void;
        protected _createRenderTargets(): void;
    }
}

declare module BABYLON {
    class OutlineRenderer {
        private _scene;
        private _effect;
        private _cachedDefines;
        zOffset: number;
        constructor(scene: Scene);
        render(subMesh: SubMesh, batch: _InstancesBatch, useOverlay?: boolean): void;
        isReady(subMesh: SubMesh, useInstances: boolean): boolean;
    }
}

declare module BABYLON {
    class RenderingGroup {
        index: number;
        private _scene;
        private _opaqueSubMeshes;
        private _transparentSubMeshes;
        private _alphaTestSubMeshes;
        private _depthOnlySubMeshes;
        private _particleSystems;
        private _spriteManagers;
        private _opaqueSortCompareFn;
        private _alphaTestSortCompareFn;
        private _transparentSortCompareFn;
        private _renderOpaque;
        private _renderAlphaTest;
        private _renderTransparent;
        private _edgesRenderers;
        onBeforeTransparentRendering: () => void;
        /**
         * Set the opaque sort comparison function.
         * If null the sub meshes will be render in the order they were created
         */
        opaqueSortCompareFn: Nullable<(a: SubMesh, b: SubMesh) => number>;
        /**
         * Set the alpha test sort comparison function.
         * If null the sub meshes will be render in the order they were created
         */
        alphaTestSortCompareFn: Nullable<(a: SubMesh, b: SubMesh) => number>;
        /**
         * Set the transparent sort comparison function.
         * If null the sub meshes will be render in the order they were created
         */
        transparentSortCompareFn: Nullable<(a: SubMesh, b: SubMesh) => number>;
        /**
         * Creates a new rendering group.
         * @param index The rendering group index
         * @param opaqueSortCompareFn The opaque sort comparison function. If null no order is applied
         * @param alphaTestSortCompareFn The alpha test sort comparison function. If null no order is applied
         * @param transparentSortCompareFn The transparent sort comparison function. If null back to front + alpha index sort is applied
         */
        constructor(index: number, scene: Scene, opaqueSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, alphaTestSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, transparentSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>);
        /**
         * Render all the sub meshes contained in the group.
         * @param customRenderFunction Used to override the default render behaviour of the group.
         * @returns true if rendered some submeshes.
         */
        render(customRenderFunction: Nullable<(opaqueSubMeshes: SmartArray<SubMesh>, transparentSubMeshes: SmartArray<SubMesh>, alphaTestSubMeshes: SmartArray<SubMesh>, depthOnlySubMeshes: SmartArray<SubMesh>) => void>, renderSprites: boolean, renderParticles: boolean, activeMeshes: Nullable<AbstractMesh[]>): void;
        /**
         * Renders the opaque submeshes in the order from the opaqueSortCompareFn.
         * @param subMeshes The submeshes to render
         */
        private renderOpaqueSorted(subMeshes);
        /**
         * Renders the opaque submeshes in the order from the alphatestSortCompareFn.
         * @param subMeshes The submeshes to render
         */
        private renderAlphaTestSorted(subMeshes);
        /**
         * Renders the opaque submeshes in the order from the transparentSortCompareFn.
         * @param subMeshes The submeshes to render
         */
        private renderTransparentSorted(subMeshes);
        /**
         * Renders the submeshes in a specified order.
         * @param subMeshes The submeshes to sort before render
         * @param sortCompareFn The comparison function use to sort
         * @param cameraPosition The camera position use to preprocess the submeshes to help sorting
         * @param transparent Specifies to activate blending if true
         */
        private static renderSorted(subMeshes, sortCompareFn, camera, transparent);
        /**
         * Renders the submeshes in the order they were dispatched (no sort applied).
         * @param subMeshes The submeshes to render
         */
        private static renderUnsorted(subMeshes);
        /**
         * Build in function which can be applied to ensure meshes of a special queue (opaque, alpha test, transparent)
         * are rendered back to front if in the same alpha index.
         *
         * @param a The first submesh
         * @param b The second submesh
         * @returns The result of the comparison
         */
        static defaultTransparentSortCompare(a: SubMesh, b: SubMesh): number;
        /**
         * Build in function which can be applied to ensure meshes of a special queue (opaque, alpha test, transparent)
         * are rendered back to front.
         *
         * @param a The first submesh
         * @param b The second submesh
         * @returns The result of the comparison
         */
        static backToFrontSortCompare(a: SubMesh, b: SubMesh): number;
        /**
         * Build in function which can be applied to ensure meshes of a special queue (opaque, alpha test, transparent)
         * are rendered front to back (prevent overdraw).
         *
         * @param a The first submesh
         * @param b The second submesh
         * @returns The result of the comparison
         */
        static frontToBackSortCompare(a: SubMesh, b: SubMesh): number;
        /**
         * Resets the different lists of submeshes to prepare a new frame.
         */
        prepare(): void;
        dispose(): void;
        /**
         * Inserts the submesh in its correct queue depending on its material.
         * @param subMesh The submesh to dispatch
         * @param [mesh] Optional reference to the submeshes's mesh. Provide if you have an exiting reference to improve performance.
         * @param [material] Optional reference to the submeshes's material. Provide if you have an exiting reference to improve performance.
         */
        dispatch(subMesh: SubMesh, mesh?: AbstractMesh, material?: Nullable<Material>): void;
        dispatchSprites(spriteManager: SpriteManager): void;
        dispatchParticles(particleSystem: IParticleSystem): void;
        private _renderParticles(activeMeshes);
        private _renderSprites();
    }
}

declare module BABYLON {
    class RenderingManager {
        /**
         * The max id used for rendering groups (not included)
         */
        static MAX_RENDERINGGROUPS: number;
        /**
         * The min id used for rendering groups (included)
         */
        static MIN_RENDERINGGROUPS: number;
        /**
         * Used to globally prevent autoclearing scenes.
         */
        static AUTOCLEAR: boolean;
        private _scene;
        private _renderingGroups;
        private _depthStencilBufferAlreadyCleaned;
        private _autoClearDepthStencil;
        private _customOpaqueSortCompareFn;
        private _customAlphaTestSortCompareFn;
        private _customTransparentSortCompareFn;
        private _renderinGroupInfo;
        constructor(scene: Scene);
        private _clearDepthStencilBuffer(depth?, stencil?);
        render(customRenderFunction: Nullable<(opaqueSubMeshes: SmartArray<SubMesh>, transparentSubMeshes: SmartArray<SubMesh>, alphaTestSubMeshes: SmartArray<SubMesh>, depthOnlySubMeshes: SmartArray<SubMesh>) => void>, activeMeshes: Nullable<AbstractMesh[]>, renderParticles: boolean, renderSprites: boolean): void;
        reset(): void;
        dispose(): void;
        /**
         * Clear the info related to rendering groups preventing retention points during dispose.
         */
        freeRenderingGroups(): void;
        private _prepareRenderingGroup(renderingGroupId);
        dispatchSprites(spriteManager: SpriteManager): void;
        dispatchParticles(particleSystem: IParticleSystem): void;
        /**
         * @param subMesh The submesh to dispatch
         * @param [mesh] Optional reference to the submeshes's mesh. Provide if you have an exiting reference to improve performance.
         * @param [material] Optional reference to the submeshes's material. Provide if you have an exiting reference to improve performance.
         */
        dispatch(subMesh: SubMesh, mesh?: AbstractMesh, material?: Nullable<Material>): void;
        /**
         * Overrides the default sort function applied in the renderging group to prepare the meshes.
         * This allowed control for front to back rendering or reversly depending of the special needs.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param opaqueSortCompareFn The opaque queue comparison function use to sort.
         * @param alphaTestSortCompareFn The alpha test queue comparison function use to sort.
         * @param transparentSortCompareFn The transparent queue comparison function use to sort.
         */
        setRenderingOrder(renderingGroupId: number, opaqueSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, alphaTestSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>, transparentSortCompareFn?: Nullable<(a: SubMesh, b: SubMesh) => number>): void;
        /**
         * Specifies whether or not the stencil and depth buffer are cleared between two rendering groups.
         *
         * @param renderingGroupId The rendering group id corresponding to its index
         * @param autoClearDepthStencil Automatically clears depth and stencil between groups if true.
         * @param depth Automatically clears depth between groups if true and autoClear is true.
         * @param stencil Automatically clears stencil between groups if true and autoClear is true.
         */
        setRenderingAutoClearDepthStencil(renderingGroupId: number, autoClearDepthStencil: boolean, depth?: boolean, stencil?: boolean): void;
    }
}

declare module BABYLON {
    class Sprite {
        name: string;
        position: Vector3;
        color: Color4;
        width: number;
        height: number;
        angle: number;
        cellIndex: number;
        invertU: number;
        invertV: number;
        disposeWhenFinishedAnimating: boolean;
        animations: Animation[];
        isPickable: boolean;
        actionManager: ActionManager;
        private _animationStarted;
        private _loopAnimation;
        private _fromIndex;
        private _toIndex;
        private _delay;
        private _direction;
        private _manager;
        private _time;
        private _onAnimationEnd;
        size: number;
        constructor(name: string, manager: SpriteManager);
        playAnimation(from: number, to: number, loop: boolean, delay: number, onAnimationEnd: () => void): void;
        stopAnimation(): void;
        _animate(deltaTime: number): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class SpriteManager {
        name: string;
        sprites: Sprite[];
        renderingGroupId: number;
        layerMask: number;
        fogEnabled: boolean;
        isPickable: boolean;
        cellWidth: number;
        cellHeight: number;
        /**
        * An event triggered when the manager is disposed.
        */
        onDisposeObservable: Observable<SpriteManager>;
        private _onDisposeObserver;
        onDispose: () => void;
        private _capacity;
        private _spriteTexture;
        private _epsilon;
        private _scene;
        private _vertexData;
        private _buffer;
        private _vertexBuffers;
        private _indexBuffer;
        private _effectBase;
        private _effectFog;
        texture: Texture;
        constructor(name: string, imgUrl: string, capacity: number, cellSize: any, scene: Scene, epsilon?: number, samplingMode?: number);
        private _appendSpriteVertex(index, sprite, offsetX, offsetY, rowSize);
        intersects(ray: Ray, camera: Camera, predicate?: (sprite: Sprite) => boolean, fastCheck?: boolean): Nullable<PickingInfo>;
        render(): void;
        dispose(): void;
    }
}

declare module BABYLON {
    class _AlphaState {
        private _isAlphaBlendDirty;
        private _isBlendFunctionParametersDirty;
        private _isBlendEquationParametersDirty;
        private _isBlendConstantsDirty;
        private _alphaBlend;
        private _blendFunctionParameters;
        private _blendEquationParameters;
        private _blendConstants;
        /**
         * Initializes the state.
         */
        constructor();
        readonly isDirty: boolean;
        alphaBlend: boolean;
        setAlphaBlendConstants(r: number, g: number, b: number, a: number): void;
        setAlphaBlendFunctionParameters(value0: number, value1: number, value2: number, value3: number): void;
        setAlphaEquationParameters(rgb: number, alpha: number): void;
        reset(): void;
        apply(gl: WebGLRenderingContext): void;
    }
}

declare module BABYLON {
    class _DepthCullingState {
        private _isDepthTestDirty;
        private _isDepthMaskDirty;
        private _isDepthFuncDirty;
        private _isCullFaceDirty;
        private _isCullDirty;
        private _isZOffsetDirty;
        private _isFrontFaceDirty;
        private _depthTest;
        private _depthMask;
        private _depthFunc;
        private _cull;
        private _cullFace;
        private _zOffset;
        private _frontFace;
        /**
         * Initializes the state.
         */
        constructor();
        readonly isDirty: boolean;
        zOffset: number;
        cullFace: Nullable<number>;
        cull: Nullable<boolean>;
        depthFunc: Nullable<number>;
        depthMask: boolean;
        depthTest: boolean;
        frontFace: Nullable<number>;
        reset(): void;
        apply(gl: WebGLRenderingContext): void;
    }
}

declare module BABYLON {
    class _StencilState {
        private _isStencilTestDirty;
        private _isStencilMaskDirty;
        private _isStencilFuncDirty;
        private _isStencilOpDirty;
        private _stencilTest;
        private _stencilMask;
        private _stencilFunc;
        private _stencilFuncRef;
        private _stencilFuncMask;
        private _stencilOpStencilFail;
        private _stencilOpDepthFail;
        private _stencilOpStencilDepthPass;
        readonly isDirty: boolean;
        stencilFunc: number;
        stencilFuncRef: number;
        stencilFuncMask: number;
        stencilOpStencilFail: number;
        stencilOpDepthFail: number;
        stencilOpStencilDepthPass: number;
        stencilMask: number;
        stencilTest: boolean;
        constructor();
        reset(): void;
        apply(gl: WebGLRenderingContext): void;
    }
}

declare module BABYLON {
    class AndOrNotEvaluator {
        static Eval(query: string, evaluateCallback: (val: any) => boolean): boolean;
        private static _HandleParenthesisContent(parenthesisContent, evaluateCallback);
        private static _SimplifyNegation(booleanString);
    }
}

declare module BABYLON {
    /**
     * Defines the list of states available for a task inside a {BABYLON.AssetsManager}
     */
    enum AssetTaskState {
        /**
         * Initialization
         */
        INIT = 0,
        /**
         * Running
         */
        RUNNING = 1,
        /**
         * Done
         */
        DONE = 2,
        /**
         * Error
         */
        ERROR = 3,
    }
    /**
     * Define an abstract asset task used with a {BABYLON.AssetsManager} class to load assets into a scene
     */
    abstract class AbstractAssetTask {
        /**
         * Task name
         */ name: string;
        /**
         * Callback called when the task is successful
         */
        onSuccess: (task: any) => void;
        /**
         * Callback called when the task is not successful
         */
        onError: (task: any, message?: string, exception?: any) => void;
        /**
         * Creates a new {BABYLON.AssetsManager}
         * @param name defines the name of the task
         */
        constructor(
            /**
             * Task name
             */ name: string);
        private _isCompleted;
        private _taskState;
        private _errorObject;
        /**
         * Get if the task is completed
         */
        readonly isCompleted: boolean;
        /**
         * Gets the current state of the task
         */
        readonly taskState: AssetTaskState;
        /**
         * Gets the current error object (if task is in error)
         */
        readonly errorObject: {
            message?: string;
            exception?: any;
        };
        /**
         * Internal only
         * @ignore
         */
        _setErrorObject(message?: string, exception?: any): void;
        /**
         * Execute the current task
         * @param scene defines the scene where you want your assets to be loaded
         * @param onSuccess is a callback called when the task is successfully executed
         * @param onError is a callback called if an error occurs
         */
        run(scene: Scene, onSuccess: () => void, onError: (message?: string, exception?: any) => void): void;
        /**
         * Execute the current task
         * @param scene defines the scene where you want your assets to be loaded
         * @param onSuccess is a callback called when the task is successfully executed
         * @param onError is a callback called if an error occurs
         */
        runTask(scene: Scene, onSuccess: () => void, onError: (message?: string, exception?: any) => void): void;
        private onErrorCallback(onError, message?, exception?);
        private onDoneCallback(onSuccess, onError);
    }
    /**
     * Define the interface used by progress events raised during assets loading
     */
    interface IAssetsProgressEvent {
        /**
         * Defines the number of remaining tasks to process
         */
        remainingCount: number;
        /**
         * Defines the total number of tasks
         */
        totalCount: number;
        /**
         * Defines the task that was just processed
         */
        task: AbstractAssetTask;
    }
    /**
     * Class used to share progress information about assets loading
     */
    class AssetsProgressEvent implements IAssetsProgressEvent {
        /**
         * Defines the number of remaining tasks to process
         */
        remainingCount: number;
        /**
         * Defines the total number of tasks
         */
        totalCount: number;
        /**
         * Defines the task that was just processed
         */
        task: AbstractAssetTask;
        /**
         * Creates a {BABYLON.AssetsProgressEvent}
         * @param remainingCount defines the number of remaining tasks to process
         * @param totalCount defines the total number of tasks
         * @param task defines the task that was just processed
         */
        constructor(remainingCount: number, totalCount: number, task: AbstractAssetTask);
    }
    /**
     * Define a task used by {BABYLON.AssetsManager} to load meshes
     */
    class MeshAssetTask extends AbstractAssetTask {
        /**
         * Defines the name of the task
         */
        name: string;
        /**
         * Defines the list of mesh's names you want to load
         */
        meshesNames: any;
        /**
         * Defines the root url to use as a base to load your meshes and associated resources
         */
        rootUrl: string;
        /**
         * Defines the filename of the scene to load from
         */
        sceneFilename: string;
        /**
         * Gets the list of loaded meshes
         */
        loadedMeshes: Array<AbstractMesh>;
        /**
         * Gets the list of loaded particle systems
         */
        loadedParticleSystems: Array<ParticleSystem>;
        /**
         * Gets the list of loaded skeletons
         */
        loadedSkeletons: Array<Skeleton>;
        /**
         * Callback called when the task is successful
         */
        onSuccess: (task: MeshAssetTask) => void;
        /**
         * Callback called when the task is successful
         */
        onError: (task: MeshAssetTask, message?: string, exception?: any) => void;
        /**
         * Creates a new {BABYLON.MeshAssetTask}
         * @param name defines the name of the task
         * @param meshesNames defines the list of mesh's names you want to load
         * @param rootUrl defines the root url to use as a base to load your meshes and associated resources
         * @param sceneFilename defines the filename of the scene to load from
         */
        constructor(
            /**
             * Defines the name of the task
             */
            name: string, 
            /**
             * Defines the list of mesh's names you want to load
             */
            meshesNames: any, 
            /**
             * Defines the root url to use as a base to load your meshes and associated resources
             */
            rootUrl: string, 
            /**
             * Defines the filename of the scene to load from
             */
            sceneFilename: string);
        /**
         * Execute the current task
         * @param scene defines the scene where you want your assets to be loaded
         * @param onSuccess is a callback called when the task is successfully executed
         * @param onError is a callback called if an error occurs
         */
        runTask(scene: Scene, onSuccess: () => void, onError: (message?: string, exception?: any) => void): void;
    }
    /**
     * Define a task used by {BABYLON.AssetsManager} to load text content
     */
    class TextFileAssetTask extends AbstractAssetTask {
        /**
         * Defines the name of the task
         */
        name: string;
        /**
         * Defines the location of the file to load
         */
        url: string;
        /**
         * Gets the loaded text string
         */
        text: string;
        /**
         * Callback called when the task is successful
         */
        onSuccess: (task: TextFileAssetTask) => void;
        /**
         * Callback called when the task is successful
         */
        onError: (task: TextFileAssetTask, message?: string, exception?: any) => void;
        /**
         * Creates a new 