import {Data} from '../data/Data';
import {TwLang} from "../util/TwLang";
import {DataManager} from "./DataManager";
import {NetName} from '../enum/NetName';
import {ABNetManager} from '../../tc/manager/ABNetManager';
import {INetData} from '../../tc/interface/INetData';
import {GTime} from '../../tc/util/GTime';
import {IExposureData} from '../data/common/IExposureData';
import {GDispatcher} from '../../tc/util/GDispatcher';
import {onNotSuccess} from "../../../src/comm/onNotSuccess";

export class NetManager extends ABNetManager {
    private static instance: NetManager;

    public static get ins(): NetManager {
        if (!this.instance) {
            this.instance = new NetManager();
        }
        return this.instance;
    }

    private isInit: boolean;

    constructor() {
        super();
        if (this.isInit) {
            throw Error(TwLang.lang_001);
        }
        this.isInit = true;
    }

    /**
     * 获取集卡数据
     * @param callback
     * @param {number} collectRuleId 集卡规则ID
     */
    public getCollectRule(callback: Function, collectRuleId: number): void {
        const net: INetData = {
            name: NetName.COLLECT_RULE,
            uri: '/plugin/collectRule/getCollectRule',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------游戏-----------------------------------------

    /**
     * 游戏基础信息
     */
    public getInfo(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_INFO,
            uri: '/ngame/new/getInfo',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 夏日活动游戏基础信息
     */
    public getSummerInfo(callback: Function, rankBaseConfigId: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMERINFO,
            uri: '/summer/getUserActivityInfo',
            type: 'get',
            dataType: 'json',
            param: {
                rankBaseConfigId: rankBaseConfigId,
                isMainActivity: false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  购买鱼钩鱼线道具
     * @param callback
     * @param operatingActivityId 活动oaid
     * @param type 道具类型 0：鱼线 1：鱼钩
     */
    public summerBuyProp(callback: Function, operatingActivityId: number, type: number): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_BUYPROP,
            uri: '/summer/buyProp',
            type: 'post',
            dataType: 'json',
            param: {
                operatingActivityId: operatingActivityId,
                type: type
            },
            callback: callback, hideMsg: true
        };
        this.send(net);
    }

    /**
     * 获取鱼钩鱼线数量
     * @param operatingActivityId
     * @param callback
     */
    public summerGetToyInfo(operatingActivityId, callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_TOY_INFO,
            uri: '/summer/getToyInfo',
            type: 'get',
            dataType: 'json',
            param: {operatingActivityId: operatingActivityId},
            callback: callback
        };
        this.send(net);
    }


    /**
     * 开始游戏
     * @param callback
     * @param {string} isAgain 是否是再来一次
     * @param {number} credits 主动要求花费多少积分玩游戏
     * @param {number} customizedType xx类型
     */
    public doStart(callback: Function, isAgain: boolean, credits?: number, customizedType?: number): void {

        let addUrl = '';
        if (isAgain) {
            addUrl += '?dpm=' + DataManager.ins.gameGetSubmitResultData.againExposure.dpm;
        }

        const param: any = {
            id: DataManager.ins.gameCfgData.gameInfo.gameId,
            oaId: DataManager.ins.gameCfgData.gameInfo.oaId
        };

        if (credits) {
            param.credits = credits;
        }

        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_START,
            uri: '/ngapi/dostart',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            addUrl: addUrl
        };
        this.getToken(net);
    }

    /**
     * 签到接口
     * @param {Function} callback
     * @param {number} signActivityId 签到活动ID
     * @param {number} activityId 插件活动ID，用于加抽奖次数，不传则使用签到身上配置的插件活动ID，否则不加抽奖次数
     */
    public doSign(callback: Function, signActivityId: number, activityId?: number): void {
        var data: any;
        if (activityId) {
            data = {
                id: signActivityId,
                activityId: activityId
            };
        } else {
            data = {
                id: signActivityId
            };
        }
        const net: INetData = {
            name: NetName.SIGN_DO_SIGN,
            uri: '/signactivity/doSign',
            type: 'post',
            dataType: 'json',
            param: data,
            callback: callback
        };
        this.send(net);
    }

    /**
     *  获取奖项
     */
    public getOptions(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_OPTIONS,
            uri: '/ngapi/getOptions',
            type: 'post',
            dataType: 'json',
            param: {
                id: DataManager.ins.customCfgData.actId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询开始游戏状态
     * @param callback
     * @param {number} orderId 订单ID
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param {number} pollingCount 最大轮询次数
     * @param {number} customizedType
     */
    public getStartStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderId: orderId
        };
        if (customizedType) {
            param.customizedType = customizedType
        }

        const net: INetData = {
            name: NetName.GAME_START_STATUS_NEW,
            uri: '/hdtool/recon/ngame/getNgameStartStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck,
            // hideMsg:true,
        };
        this.send(net);
    }

    /**
     * 获取订单状态
     * @param callback 回调很熟
     * @param orderNum
     * @param pollingCheck 轮询回调函数
     * @param pollingCount 轮询次数
     * @param customizedType
     */
    public getOrderStatus(callback: Function, orderNum: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderId: orderNum
        };
        const net: INetData = {
            name: NetName.GET_ORDER_STATUS,
            uri: '/hdtool/recon/getOrderStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck,
            hideMsg: true,
        };
        this.send(net);
    }

    public getSummerOrderStatus(callback: Function, orderNum: number, pollingCheck: Function, pollingCount = 5, customizedType?: number): void {
        const param: any = {
            orderNum: orderNum
        };

        const net: INetData = {
            name: NetName.GAME_SUMMER_GET_ORDER_STATUS,
            uri: '/summer/getOrderStatus',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 复活
     */
    public doRevive(callback: Function, collectRuleId: number, itemId: number): void {

        const net: INetData = {
            name: NetName.GAME_REVIVE,
            uri: '/summer/doRevive',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };
        this.getToken(net);
    }

    /**
     * 获得复活卡数量
     */
    public getReviveCardNum(callback: Function, collectRuleId: number, itemId: number): void {
        const net: INetData = {
            name: NetName.GAME_REVIVE_STATUS,
            uri: '/summer/getReviveCardNum',
            type: 'post',
            dataType: 'json',
            param: {
                collectRuleId: collectRuleId,
                itemId: itemId
            },
            callback: callback,
        };

        this.getToken(net);
    }

    /**
     * 猜扑克
     * @param callback
     * @param plugId
     * @param {number} ticketId 订单ID
     * @param {number} prizeId 集卡规则ID
     * @param {number} betting 下注方式 0:大 1:小 2:红 3:黑
     */
    public guessPoker(callback: Function, plugId: number, ticketId: number, prizeId: number, betting: number): void {
        const net: INetData = {
            name: NetName.GAME_GUESS_POKER,
            uri: '/ngapi/guessPoker',
            type: 'post',
            dataType: 'json',
            param: {
                pluginId: plugId,
                ticketId: ticketId,
                prizeId: prizeId,
                betting: betting
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 游戏阶段性数据提交
     * @param callback
     * @param ticketId
     * @param dynamicData 阶段性交互数据
     */
    public datapash(callback: Function, ticketId: number, dynamicData: string): void {
        const net: INetData = {
            name: NetName.GAME_DATA_PASH,
            uri: '/ngame/new/datapash',
            type: 'post',
            dataType: 'json',
            param: {
                ticketId: ticketId,
                gameId: DataManager.ins.gameCfgData.gameInfo.id,
                dynamicData: dynamicData
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 提交游戏数据
     *
     * @param callback
     * @param {number} ticketId 订单ID
     * @param {number} score 分数
     * @param {any} gameData 防作弊信息
     * @param {string} submitToken
     * @param {string} dynamicData 行为数据
     * @param {boolean} checkScore 是否校验分数
     * @param {number} customizedType
     */
    public gameSubmit(callback: Function,
                      ticketId: number,
                      score: number,
                      gameData: any,
                      submitToken: string,
                      dynamicData: string,
                      checkScore?: boolean,
                      customizedType?: number): void {
        const sign = this.createSgin(ticketId, score, gameData, submitToken);
        const param: any = {
            ticketId: ticketId,
            score: score,
            gameData: gameData,
            sgin: sign,
            dynamicData: dynamicData
        };
        if (checkScore) {
            param.checkScore = checkScore;
        }
        if (customizedType) {
            param.customizedType = customizedType;
        }

        const net: INetData = {
            name: NetName.GAME_SUBMIT,
            uri: '/ngame/new/submit',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 生成签名
     * @param {number} ticketId
     * @param {number} score
     * @param {any} gameData
     * @param {string} submitToken
     * @returns {string} 签名
     */
    private createSgin(ticketId: number, score: number, gameData: any, submitToken: string): string {
        return window['duiba_md5'](ticketId + '' + score + '' + gameData + '' + submitToken);
    }

    /**
     * 游戏结束获取奖品数据
     * @param callback
     * @param {number} orderId 订单ID
     * @param {Function} pollingCheck 轮询条件 返回true继续轮询
     * @param pollingCount
     */
    public getSubmitResult(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.GAME_SUBMIT_STATUS,
            uri: '/ngame/new/getSubmitResult',
            type: 'get',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 查看中奖名单
     * @param {boolean} showName 是否返回昵称
     */
    public winRanks(callback: Function, showName = false): void {
        const net: INetData = {
            name: NetName.GAME_RANKS,
            uri: '/ngapi/winranks',
            type: 'post',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                showName: showName
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     *  获取规则
     */
    public getRuleNg(callback: Function): void {
        const net: INetData = {
            name: NetName.GAME_RULE,
            uri: '/ngapi/getRuleNg',
            type: 'post',
            dataType: 'html',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 实时排行榜
     * @param {number} type 0总排行榜 1今日排行榜 2 多游戏总排行榜 3 昨日排行榜
     * @param {number} count 返回榜单长度 最大50
     */
    public realtimerank(callback: Function, type: number, count = 50): void {
        const net: INetData = {
            name: NetName.GAME_REAL_TIME_RANK,
            uri: '/ngapi/realtimerank',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.gameCfgData.gameInfo.gameId,
                count: count,
            },
            callback: callback,
            addUrl: '/' + type
        };
        this.send(net);
    }

    /**
     * 获取成长值
     * @param {number} gametotalid 游戏中心榜单ID
     */
    public getUserTotalScore(callback: Function, gametotalid: number): void {
        const net: INetData = {
            name: NetName.GAME_TOTAL_SCORE,
            uri: '/ngapi/getUserTotalScore',
            type: 'get',
            dataType: 'json',
            param: {
                id: gametotalid
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------插件-----------------------------------------

    /**
     * 插件抽奖
     * @param {number} activityId 插件ID
     */
    public doJoinPlugDraw(callback: Function, activityId: number, deductCredits?: boolean): void {
        const net: INetData = {
            name: NetName.PLUG_DO_JOIN,
            uri: '/activityPlugDrawInfo/doJoinPlugdraw',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: activityId,
                deductCredits: deductCredits ? deductCredits : false
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询插件抽奖订单
     * @param {number} orderId 订单ID
     * @param {number} prizeLevel 开奖等级，暂未开放
     */
    public getPlugOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5, prizeLevel?: number): void {
        const net: INetData = {
            name: NetName.PLUG_ORDER_STATUS,
            uri: '/plugin/getOrderStatus',
            type: 'get',
            dataType: 'json',
            hideMsg: true,
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 查询插件信息
     * @param callback
     * @param plugId
     */
    public getPrizeInfo(callback: Function, plugId: number): void {
        const net: INetData = {
            name: NetName.PLUG_PRIZE_INFO,
            uri: '/activityPlugDrawInfo/getPrizeInfo',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                activityId: plugId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件-批量查询奖项信息
     * @param callback
     * @param {string} ids 插件ID 用，间隔
     */
    public optionInfo(callback: Function, ids: string): void {
        const net: INetData = {
            name: NetName.PLUG_OPTION_INFO,
            uri: '/activityPlugin/optionInfo',
            type: 'get',
            dataType: 'json',
            param: {
                ids: ids
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 插件校验解锁纪录
     * @param callback
     * @param {number} id 插件ID
     */
    public unblocking(callback: Function, id: number): void {
        const net: INetData = {
            name: NetName.PLUG_UNBLOCKING,
            uri: 'activityPlugin/unblocking',
            type: 'post',
            dataType: 'json',
            param: {
                id: id
            },
            callback: callback
        };
        this.send(net);
    }

    //--------------------------------------------自定义活动工具-----------------------------------------

    /**
     * 获取基础信息
     */
    public ajaxElement(callback: Function): void {
        const net: INetData = {
            name: NetName.CUSTOM_ELEMENT,
            uri: window['CFG'].getElement,
            type: 'post',
            dataType: 'json',
            param: {
                hdType: DataManager.ins.customCfgData.hdType,
                hdToolId: DataManager.ins.customCfgData.hdToolId,
                actId: DataManager.ins.customCfgData.actId,
                preview: DataManager.ins.customCfgData.preview
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询闯关游戏 配置信息
     * @param {number} throughId
     */
    public ajaxThroughInfo(callback: Function, throughId = 1): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_INFO,
            uri: window['CFG'].ajaxThroughInfo,
            type: 'post',
            dataType: 'json',
            param: {
                hdtoolId: DataManager.ins.customCfgData.hdToolId,
                throughId: 1
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 页面停留时长
     * @param remain
     * @param first
     * @param from
     */
    public remainDuration(remain, first, from = 1): void {
        const net: INetData = {
            name: 'remainDuration',
            uri: window['duiba'] + '/customActivity/api/remainDuration',
            type: 'get',
            dataType: 'json',
            param: {
                id: DataManager.ins.customCfgData.hdToolId,
                activityId: DataManager.ins.customCfgData.actId,
                activityType: 'hdtool',
                remain: remain,
                first: first,
                from: from
            },
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 停留时长
     * @param remain 停留时长
     * @param first 是否第一次进入
     * @param from 来源
     */
    public remainDurationProjectX(remain, first, from = 1): void {
        const net: INetData = {
            name: 'remainDuration',
            uri: '/customActivity/api/remainDuration',
            type: 'get',
            dataType: 'json',
            param: {
                activityId: window['projectId'],
                activityType: '60000',
                remain: remain,
                first: first,
                from: from
            },
            callback: null,
            hideMsg: true
        };
        NetManager.ins.send(net);
    }

    /**
     * 活动工具抽奖
     * @param {number} collectRuleId 集卡规则ID，活动与集卡规则关联才能获得发卡资格
     */
    public doJoin(callback: Function, collectRuleId?: number): void {
        const param: any = {
            activityId: DataManager.ins.customCfgData.actId,
            oaId: DataManager.ins.customCfgData.oaId
        };
        if (collectRuleId) {
            param.collectRuleId = collectRuleId;
        }
        const net: INetData = {
            name: NetName.DO_JOIN_DATA,
            uri: "/hdtool/recon/doJoin",
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback,
            // hideMsg:false,
        };
        this.getToken(net);
    }

    /**
     * 活动工具查询订单结果
     * @param orderId
     */
    public getCustomOrderStatus(callback: Function, orderId: number, pollingCheck: Function, pollingCount = 5): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_STATUS,
            uri: window['CFG'].quireOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount
        };
        this.send(net);
    }

    /**
     * 获取抽奖前置信息
     * @param {string} orderId 订单ID
     */
    public getOrderInfo(callback: Function, orderId: number, showMsg = true): void {
        const net: INetData = {
            name: NetName.CUSTOM_ORDER_INFO,
            uri: window['CFG'].gameGetOrder,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 前置开奖提交
     * @param {string} orderId 订单ID
     * @param {string} facePrice 前置开奖所需分值
     */
    public beforSubmit(callback: Function, orderId: number, facePrice: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_BEFOR_SUBMIT,
            uri: window['CFG'].gameSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId,
                facePrice: facePrice
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 答题提交
     */
    public questionSubmit(): void {

    }

    /**
     * 闯关游戏提交
     */
    public throughSubmit(callback: Function, orderId: number): void {
        const net: INetData = {
            name: NetName.CUSTOM_THROUGH_SUBMIT,
            uri: window['CFG'].throughSubmit,
            type: 'post',
            dataType: 'json',
            param: {
                orderId: orderId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获取token
     * @param net
     */
    public getToken(net: INetData): void {
        if (window['getDuibaToken']) {
            window['getDuibaToken']((tokenObj: any) => {
                net.param.token = tokenObj.token;
                this.send(net);
            }, (key: string, messageObj: any) => {
                this.onError(net);
            });
        } else {
            this.send(net);
        }
    }

    /**
     * 查询我的奖品记录
     * @param callback
     * @param {number} page 当前页数
     */
    public getRecord(callback: Function, page: number): void {
        const net: INetData = {
            name: NetName.GET_RECORD,
            uri: (window['recordUrl'] ? window['recordUrl'] : '') + '/Crecord/getRecord',
            type: 'get',
            dataType: 'json',
            param: {
                page: page
            },
            callback: callback
        };

        let gTime: string = '?_=' + GTime.getTimestamp();
        let realUrl: string = net.uri;
        if (realUrl.indexOf('?') != -1) {
            gTime = '&_=' + GTime.getTimestamp();
        }

        //url加参数等特殊需求（例如再玩一次需要在dostart接口的url上加埋点）
        if (net.addUrl) {
            realUrl += net.addUrl;
        }

        window['$'].ajax({
            type: net.type,
            // url: realUrl + gTime,
            url: realUrl,
            dataType: net.dataType,
            data: net.param,
            async: true,
            xhrFields: {
                withCredentials: true
            },
            crossDomain: true,
            cache: false,
            success: (result) => {
                this.onResponse(net, result);
            },
            error: (message) => {
                this.onError(net);
            }
        });
    }

    /**
     * 推啊曝光埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmshow(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmshow',
            uri: domain + 'spmshow',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 推啊点击埋点(福袋)
     * @param {IExposureData} exposure
     */
    public spmclick(exposure: IExposureData): void {
        const domain = (exposure.domain ? exposure.domain : '') + '/engine/';
        const net: INetData = {
            name: 'spmclick',
            uri: domain + 'spmclick',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧点击埋点
     * @param {IExposureData} exposure
     */
    public clickLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'clickLog',
            uri: '/log/click',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 兑吧曝光埋点
     * @param {IExposureData} exposure
     */
    public showLog(exposure: IExposureData): void {
        const net: INetData = {
            name: 'showLog',
            uri: exposure.domain + '/exposure/standard',
            type: 'get',
            dataType: 'jsonp',
            param: exposure,
            callback: null,
            hideMsg: true
        };
        this.send(net);
    }

    /**
     * 消息响应
     * @param net
     * @param result 结果
     */
    protected onResponse(net: INetData, result: any): void {
        //数据处理
        // console.error("数据检查233====》",net.uri,JSON.stringify(result));
        const data: Data = DataManager.ins.updateData(net.name, result, net.param);
        if (net.pollingCheck) {
            console.log('result=====', result);
            console.log('data=====', data);
        }
        //接口成功
        if (net.pollingCount && net.pollingCheck(data)) {
            net.pollingCount -= 1;
            //轮询接口特殊处理
            setTimeout(() => {
                this.send(net);
            }, 500);
        } else if (net.callback) {
            if (net.pollingCheck && !net.pollingCount && net.pollingCheck(data)) {
                net.callback(false, 'pollingFaild');
            } else {
                net.callback(data.success, data || result);
            }

        }
        if (!data.success && !net.hideMsg) {
            console.log('test-->接口错误--》》',net.uri,data)
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net, result.message || result.desc || result.msg, result);
            onNotSuccess(data, net.name);
        }
    }

    /**
     * 通讯底层错误
     * @param net
     */
    protected onError(net: INetData): void {
        if (net.callback) {
            net.callback(false);
        }

        if (!net.hideMsg) {
            GDispatcher.dispatchEvent(ABNetManager.ERROR, net);
            onNotSuccess();
        }
    }

    // tslint:disable-next-line:max-file-line-count


    //////////////////////////////////////// projectX ////////////////////////////////////////

    /**
     * 获取域名
     * @param callback
     */
    public getDomain(callback: Function) {
        const net: INetData = {
            name: NetName.GET_DOMAIN,
            uri: '/projectx/getDomain.query',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询发奖规则的奖项
     * @param callback
     * @param strategyId 发奖规则id
     * @param optionId 奖项id，不传查询所有奖品
     */
    public queryPrizeList(callback: Function, strategyId, optionId?) {
        const net: INetData = {
            name: NetName.QUERY_PRIZE_LIST,
            uri: '/projectx/{projectId}/' + strategyId + '.query',
            type: 'get',
            dataType: 'json',
            param: optionId ? {
                optionId: optionId
            } : {},
            callback: callback
        };
        this.send(net);
    }


    /**
     * 预扣积分
     * @param callback
     * @param toPlaywayId
     * @param toActionId
     * @param desc
     * @param credits
     */
    public creditsCost(callback: Function, toPlaywayId, toActionId, credits, desc) {
        const param: any = {
            toPlaywayId: toPlaywayId,
            toActionId: toActionId,
            credits: credits,
        };

        if (desc) {
            param.desc = desc;
        }

        const net: INetData = {
            name: NetName.CREDITS_COST,
            uri: '/projectx/{projectId}/credits/creditsCost.do',
            type: 'post',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询扣积分状态
     * @param callback
     * @param ticketNum
     * @param pollingCheck
     * @param pollingCount
     */
    public queryStatus(callback: Function, ticketNum, pollingCheck: Function, pollingCount: number = 10) {
        const net: INetData = {
            name: NetName.QUERY_STATUS,
            uri: '/projectx/{projectId}/credits/queryStatus.do',
            type: 'get',
            dataType: 'json',
            param: {
                ticketNum: ticketNum,
            },
            callback: callback,
            pollingCheck: pollingCheck,
            pollingCount: pollingCount,
        };
        this.send(net);
    }

    /**
     * 查询奖品信息
     * @param callback
     * @param ids
     */
    public queryPrizeInfo(callback: Function, ids) {
        const net: INetData = {
            name: NetName.QUERY_PRIZE_INFO,
            uri: '/projectx/{projectId}/prize.query',
            type: 'get',
            dataType: 'json',
            param: {
                ids: ids,
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询用户拥有的道具
     */
    public queryUserSpList(callback: Function) {
        const net: INetData = {
            name: NetName.QUERY_PRIZE_INFO,
            uri: '/projectx/{projectId}/getUserSpList.query',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询优惠卷信息
     * @param callback
     * @param projectOrderNo
     * @param pollingCheck
     * @param pollingCount
     */
    public getCouponInfo(callback: Function, projectOrderNo, pollingCheck: Function, pollingCount: number = 10) {
        const net: INetData = {
            name: NetName.GET_COUPON_INFO,
            uri: '/aaw/projectx/getCouponInfo',
            type: 'get',
            dataType: 'json',
            param: {
                projectOrderNo: projectOrderNo,
            },
            callback: callback,
            pollingCount: pollingCount,
            pollingCheck: pollingCheck
        };
        this.send(net);
    }

    /**
     * 查询我的奖品列表
     * @param callback - 回调函数
     * @param credits - 是否包含加积分，传参数代表包含加积分
     */
    public queryRecords(callback: Function, credits?) {
        let param = {};
        credits ? credits['credits'] = credits : 0;
        const net: INetData = {
            name: NetName.QUERY_RECORDS,
            uri: '/projectx/{projectId}/records.query',
            type: 'get',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获得用户信息
     * @param callback
     */
    public getProjectUserInfo(callback: Function) {
        const net: INetData = {
            name: NetName.GET_PROJECTUSERINFO,
            uri: '/ctool/getProjectUserInfo',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 获得规则
     * @param callback
     */
    public getRule(callback: Function) {
        const net: INetData = {
            name: NetName.GET_RULE,
            uri: '/projectx/{projectId}/projectRule.query',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }


    /**
     * // 聚合查询
     * querys = {
     *     key:{    key唯一 返回值会设置到key中	是	[object]
     *         path:    原先接口的路径	是	[string]
     *         params:  原先接口的参数	是	[string]
     *     }
     * }
     */
    public mergeQuery(callback: Function, querys) {
        const net: INetData = {
            name: NetName.MERGE_QUERY,
            uri: '/projectx/{projectId}/merge.query',
            type: 'post',
            dataType: 'json',
            param: {
                querys: JSON.stringify(querys)
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查询发奖规则的中奖记录
     * @param {Function} callback
     * @param strategyId 发奖规则
     * @param limit 条数
     * @param credits 传则包含积分奖品，不传则不包含
     */
    public strategyRecords(callback: Function, strategyId, limit, credits?) {
        let param = {
            strategyId: strategyId,
            limit: limit
        };
        credits ? (param['credits'] = credits) : 0;
        const net: INetData = {
            name: NetName.STRATEGY_RECORDS,
            uri: '/projectx/{projectId}/strategyRecords.query',
            type: 'get',
            dataType: 'json',
            param: param,
            callback: callback
        };
        this.send(net);
    }


    ///////////////////////////////////////////////////////////////////

    //// 工行欢趣庄园
    /**
     * 获取地块信息
     * @param callback
     */
    public getLandInfo(callback: Function) {
        const net: INetData = {
            name: NetName.GET_LANDINFO,
            uri: '/projectx/{projectId}/join_1/landInfo.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 种植
     * @param callback
     * @param landId 地块Id
     * @param petId 植物Id
     */
    public doPlant(callback: Function, landId, petId) {
        const net: INetData = {
            name: NetName.DO_PLANT,
            uri: '/projectx/{projectId}/join_1/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                landId: landId,
                petId: petId
            },
            callback: callback
        };
        this.send(net);
    }

    public doFeed(callback: Function, landId) {
        const net: INetData = {
            name: NetName.DO_FEED,
            uri: '/projectx/{projectId}/join_2/join.do',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                landId: landId,
            },
            callback: callback
        };
        this.send(net);
    }

    public priceInfo(callback: Function) {
        const net: INetData = {
            name: NetName.DO_FEED,
            uri: '/projectx/{projectId}/join_4/priceInfo.do',
            type: 'get',
            dataType: 'json',
            hideMsg: true,
            param: {},
            callback: callback
        };
        this.send(net);
    }

    public doHarvest(callback: Function, landId) {
        const net: INetData = {
            name: NetName.DO_HARVEST,
            uri: '/projectx/{projectId}/join_3/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                landId: landId,
            },
            callback: callback
        };
        this.send(net);
    }

    public doSell(callback: Function, spId, count) {
        const net: INetData = {
            name: NetName.DO_SELL,
            uri: '/projectx/{projectId}/join_4/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                spId: spId,
                count: count
            },
            callback: callback
        };
        this.send(net);
    }

    public getSellInfo(callback: Function) {
        const net: INetData = {
                name: NetName.GET_SELLINFO,
                uri: '/projectx/{projectId}/join_4/saleInfo.do',
                type: 'get',
                dataType: 'json',
                param: {},
                callback: callback
            }
        ;
        this.send(net);
    }


    public getConsumerInfo(callback: Function) {
        const net: INetData = {
            name: NetName.Get_CONSUMERINFO,
            uri: '/projectx/{projectId}/join_1/consumerInfo.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        }
        this.send(net);
    }

    public newUserPrize(callback: Function, token) {
        const net: INetData = {
            name: NetName.NEW_USER_PRIZE,
            uri: '/projectx/{projectId}/join_9/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                token: token,
            },
            callback: callback
        };
        this.send(net);
    }

    public doTask(callback: Function, taskId) {
        const net: INetData = {
            name: NetName.DO_TASK,
            uri: '/projectx/{projectId}/join_5/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                taskId: taskId,
            },
            callback: callback
        };
        this.send(net);
    }

    public getTaskInfo(callback: Function) {
        const net: INetData = {
            name: NetName.GET_TASKINFO,
            uri: '/projectx/{projectId}/join_5/taskInfo.do',
            type: 'post',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    public prizeExchange(callback: Function, strategyId) {
        const net: INetData = {
            name: NetName.PRIZE_EXCHANGE,
            uri: '/projectx/{projectId}/join_6/join.do',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                strategyId: strategyId
            },
            callback: callback
        };
        this.send(net);
    }

    public prizeInfo(callback: Function, prizeIds) {
        const net: INetData = {
            name: NetName.GET_PRIZEINFO,
            uri: '/projectx/{projectId}/join_6/prizeInfo.do',
            type: 'get',
            dataType: 'json',
            param: {
                prizeIds: prizeIds
            },
            callback: callback
        };
        this.send(net);
    }

    public propExchange(callback: Function, spId, ticket, num) {
        const net: INetData = {
            name: NetName.PROP_EXCHANGE,
            uri: '/projectx/{projectId}/join_7/join.do',
            type: 'post',
            dataType: 'json',
            hideMsg: true,
            param: {
                spId,
                ticket,
                num
            },
            callback: callback
        };
        this.send(net);
    }

    public propInfo(callback: Function) {
        const net: INetData = {
            name: NetName.PROPINFO,
            uri: '/projectx/{projectId}/join_7/propInfo.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 使用道具
     * @param callback
     * @param spId
     * @param landId
     */
    public useProp(callback: Function, spId, landId) {
        const net: INetData = {
            name: NetName.USER_PROP,
            uri: '/projectx/{projectId}/join_8/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                spId: spId,
                landId: landId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 土地价格
     * @param callback
     */
    public areaPrice(callback: Function) {
        const net: INetData = {
            name: NetName.AREA_PRICE,
            uri: '/projectx/{projectId}/area/areaPrice.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 问题列表
     * @param callback
     */
    public questionList(callback: Function) {
        const net: INetData = {
            name: NetName.QUESTION_LIST,
            uri: '/projectx/{projectId}/questions/join.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback
        };
        this.send(net);
    }

    /**
     * 解锁土地
     * @param callback
     * @param ticketNum
     * @param landId
     */
    public unlockLand(callback: Function, ticketNum, landId) {
        const net: INetData = {
            name: NetName.UNLOCK_LAND,
            uri: '/projectx/{projectId}/area/join.do',
            type: 'post',
            dataType: 'json',
            param: {
                ticketNum,
                landId
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 用户成就
     * @param callback
     */
    public achieves(callback: Function) {
        const net: INetData = {
            name: NetName.ACHIEVES,
            uri: '/projectx/{projectId}/join_1/achieves.do',
            type: 'get',
            dataType: 'json',
            param: {},
            callback: callback,
            hideMsg: true,
        };
        this.send(net);
    }

    /**
     * 用户去玩游戏合集里的游戏，发接口告诉后端打一个标记，用于后端发奖
     * @param callback
     * @param activity
     */
    public gameGroupMark(callback: Function, activity) {
        const net: INetData = {
            name: NetName.GAMEGROUP_MARK,
            uri: '/projectx/{projectId}/game/mark.do',
            type: 'post',
            dataType: 'json',
            param: {
                activity
            },
            callback: callback,
            hideMsg: true,
        };
        this.send(net);
    }


    /**
     * 进游戏或者用户玩游戏合集里的游戏回来，发接口告诉后端，后端判断是否发奖
     * @param callback
     */
    public gameGroupPrize(callback: Function) {
        const net: INetData = {
            name: NetName.GAMEGROUP_PRIZE,
            uri: '/projectx/{projectId}/game/join.do',
            type: 'post',
            dataType: 'json',
            param: {},
            callback: callback,
            hideMsg: true,
        };
        this.send(net);
    }
    /**
     * 
     * @param callback 
     */
    public frontVariable(callback: Function,keys) {
        const net: INetData = {
            name: NetName.frontVariable,
            uri: '/projectx/{projectId}/frontVariable.query',
            type: 'get',
            dataType: 'json',
            param: keys,
            callback: callback
        };
        this.send(net);
    }

    /**
     * 用户信息
     * @param callback 
     */
    public customInfo(callback:Function){
        const net: INetData = {
            name: NetName.customInfo,
            uri: '/projectx/{projectId}/join_5/customInfo.do',
            type: 'get',
            dataType: 'json',
            param: {
                
            },
            callback: callback
        };
        this.send(net);
    }

    /**
     * 查看运营配置项
     * @callback
     */
    public frontInfo(callback:Function){
        const net: INetData = {
            name: NetName.customInfo,
            uri: '/projectx/{projectId}/join_5/frontInfo.do',
            type: 'get',
            dataType: 'json',
            param: {
                
            },
            callback: callback
        };
        this.send(net);
    }
}

// window['nt'] = NetManager.ins;

const regular = "(\\S+)//(\\S+)/(\\S+)/(\\S+)/(\\S+)";
const re = new RegExp(regular, "g");
window['projectId'] = window.location.href.replace(re, "$4");
console.log(window['projectId']);
