package cn.com.duiba.jdactivity.common.utils;

import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/5/17 18:47
 */
public class RequestTool {


    private static final String UNKNOWN = "unknown";
    private static final String IPHONE = "iphone";
    private static final String ANDROID = "android";
    private static final String IOS = "ios";
    private static final String IPAD = "ipad";
    private static final String MACINTOSH = "macintosh";
    private static final String LINUX = "linux";
    private static final String WINDOWS = "windows";


    /**
     * OS map
     */
    private static Map<String, String> osMap = new LinkedHashMap<>();

    static {
        osMap.put(IPHONE, "iPhone");
        osMap.put(ANDROID, "Android");
        osMap.put(IPAD, "iPad");
        osMap.put(MACINTOSH, "Mac");
        osMap.put(WINDOWS, "Windows");
        osMap.put(LINUX, "Linux");
    }

    /**
     * 获得User-Agent
     *
     * @param req
     * @return
     */
    public static String getUserAgent(HttpServletRequest req) {
        if (req != null) {
            return req.getHeader("User-Agent");
        }
        return null;
    }

    /**
     * 获得完整的url（包含请求url中的参数，但是不包含post提交的参数部分）
     *
     * @param request
     * @return
     */
    public static String getFullRequestUrl(HttpServletRequest request) {
        String url = request.getRequestURL().toString();
        String queryString = request.getQueryString();
        if (queryString == null) {
            return url;
        } else {
            return url + "?" + queryString;
        }
    }

    /**
     * 如果非forward请求，则简单返回request.getRequestURL() (不包含参数)
     * 如果是forward请求，则返回完整的url（包含请求url中的参数，如果是post请求，还会包含post提交的参数部分）
     *
     * @param request
     * @return
     */
    public static String getRequestUrl(HttpServletRequest request) {
        String path = (String) request.getAttribute("javax.servlet.forward.request_uri");//如果是forward过来的，则这里非空
        if (path == null) {
            return request.getRequestURL().toString();
        }

        StringBuilder url = new StringBuilder(getServerPath(request)).append(path);

        Map<String, String[]> map = request.getParameterMap();
        if (!map.isEmpty()) {
            url.append("?");
        }
        int i = 0;
        for (Map.Entry<String, String[]> entry : map.entrySet()) {
            if (entry.getValue().length > 0) {
                String key = entry.getKey();
                if (i > 0) {
                    url.append("&");
                }
                url.append(key).append("=").append(entry.getValue()[0]);
                i++;
            }
        }
        return url.toString();
    }

    /**
     * 获取用户的ip地址
     *
     * @param request
     * @return
     */
    public static String getIpAddr(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip != null && ip.trim().length() > 0) {
            String[] ips = ip.trim().split(",");
            int size = ips.length;
            if (size > 0) {
                ip = ips[0].trim();
            }
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("X-Real-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeader("Cdn-Src-Ip");
        }
        if (isInvalidIp(ip)) {
            ip = request.getRemoteAddr();
        }
        if (ip != null && ip.startsWith("0:0:0:0")) {
            ip = "127.0.0.1";//NOSONAR
        }
        return ip;
    }

    private static boolean isInvalidIp(String ip) {
        return ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip);
    }

    /**
     * 是否经过了HTTP代理
     *
     * @param request
     * @return
     */
    public static boolean isProxy(HttpServletRequest request) {
        //代理服务器ip
        String httpVia = request.getHeader("HTTP_VIA");
        if (httpVia != null && !"".equals(httpVia)) {
            return true;
        }

        String xForwardedFor = request.getHeader("x-forwarded-for");
        if (xForwardedFor != null && xForwardedFor.trim().length() > 0) {
            String[] ips = xForwardedFor.trim().split(",");
            long sizeOfForwardFor = ips.length;
            if (sizeOfForwardFor > 1) {
                return true;
            }
        }

        return false;
    }

    /**
     * 获取操作系统类型： Android,iPhone,iPad,Mac,Windows,Linux，Unknown
     *
     * @param request
     * @return
     */
    public static String getOS(HttpServletRequest request) {
        String ua = getUserAgent(request);
        if (ua == null) {
            return UNKNOWN;
        }
        String ua1 = ua.toLowerCase();
        for (Map.Entry<String, String> entry : osMap.entrySet()) {
            if (ua1.contains(entry.getKey())) {
                return entry.getValue();
            }
        }
        return UNKNOWN;
    }

    /**
     * 获取操作系统类型： Android,iPhone,iPad,Mac,Windows,Linux，Unknown
     *
     * @return
     */
    public static String getOS(String ua) {
        if (ua == null) {
            return UNKNOWN;
        }
        String ua1 = ua.toLowerCase();
        for (Map.Entry<String, String> entry : osMap.entrySet()) {
            if (ua1.contains(entry.getKey())) {
                return entry.getValue();
            }
        }
        return UNKNOWN;
    }

    /**
     * 获取简洁ua,对ios,mac,ipad合并为ios，返回ios/android/unknown
     *
     * @param request
     * @return
     * @since JDK 1.6
     */
    public static String getSimpleOs(HttpServletRequest request) {
        String ua = getUserAgent(request);
        if (ua != null) {
            String ua1 = ua.toLowerCase();
            if (ua1.contains(IPAD) || ua1.contains(IPHONE) || ua1.contains(MACINTOSH)) {
                return IOS;
            } else if (ua1.contains(ANDROID)) {
                return ANDROID;
            }
        }
        return UNKNOWN;
    }

    /**
     * 获取操作系统类型：
     * Android,iPhone,iPad,Mac,Windows,Linux
     *
     * @param request
     * @return
     */
    public static String getOSNew(HttpServletRequest request) {
        String ua = getUserAgent(request);
        return getOSNew(ua);
    }

    /**
     * 获取操作系统类型，返回数字类型：4：iPad，1：iPhone，0：Android，5：Linux，2：Windows,3：Mac，6：Unknown，
     * Android,iPhone,iPad,Mac,Windows,Linux
     *
     * @return
     */
    public static String getOSNew(String ua) {
        //Unknown
        if (ua == null) {
            return "6";
        }
        String ua1 = ua.toLowerCase();
        if (ua1.contains("ipad")) {
            //iPad
            return "4";
        } else if (ua1.contains("iphone")) {
            //iPhone
            return "1";
        } else if (ua1.contains("android")) {
            //Android
            return "0";
        } else if (ua1.contains("linux")) {
            //Linux
            return "5";
        } else if (ua1.contains("windows")) {
            //Windows
            return "2";
        } else if (ua1.contains("macintosh")) {
            //Mac
            return "3";
        } else if (ua1.contains("ios")) {
            //ios
            return "6";
        }
        return "6";
    }

    /**
     * 判断来源ip是不是本地ip(126.0.0.1)
     *
     * @param request
     * @return
     */
    public static boolean isLocalRequest(HttpServletRequest request) {
        String ip = getIpAddr(request);
        if (ip.startsWith("127.0.0.1") || ip.startsWith("0:0:0:0")) {//NOSONAR
            return true;
        }
        return false;
    }

    /**
     * 获取指定的cookie
     *
     * @param request
     * @param name
     * @return
     */
    public static String getCookie(HttpServletRequest request, String name) {
        Cookie[] cookies = request.getCookies();
        if (cookies == null) {
            return null;
        }
        for (Cookie c : cookies) {
            if (name.equals(c.getName())) {
                return c.getValue();
            }
        }
        return null;
    }

    /**
     * 获得url，不包含path和参数
     *
     * @param request
     * @return
     */
    public static String getServerPath(HttpServletRequest request) {
        String port = String.valueOf(request.getServerPort());
        if (("80".equals(port) && "http".equals(request.getScheme())) || ("443".equals(port) && "https".equals(request.getScheme()))) {
            port = "";
        } else {
            port = ":" + port;
        }
        return request.getScheme() + "://" + request.getServerName() + port;
    }

    /**
     * 判断请求来源是否是ios系统
     *
     * @return
     */
    public static boolean isIos(HttpServletRequest request) {
        if (request == null) {
            return false;
        }
        String ua = getUserAgent(request);
        if (ua == null) {
            return false;
        }
        ua = ua.toLowerCase();
        return ua.contains(IPHONE) || ua.contains(IPAD) || ua.contains(MACINTOSH);
    }

    /**
     * 是否是android系统
     *
     * @return
     */
    public static boolean isAndroid(HttpServletRequest request) {
        if (request == null) {
            return false;
        }
        String ua = getUserAgent(request);
        if (ua == null) {
            return false;
        }
        ua = ua.toLowerCase();
        return ua.contains(ANDROID);
    }

    /**
     * 从request header获取referer头
     *
     * @param request
     * @return
     */
    public static String getReferer(HttpServletRequest request) {
        if (request == null) {
            return null;
        }
        return request.getHeader("referer");
    }

    /**
     * 提取request中的参数
     *
     * @param request
     * @return
     */
    public static Map<String, String> getRequestParamMap(HttpServletRequest request) {
        Map<String, String[]> map = request.getParameterMap();
        Map<String, String> resultMap = new HashMap<>();
        for (Map.Entry<String, String[]> entry : map.entrySet()) {
            resultMap.put(entry.getKey(), entry.getValue()[0]);
        }
        return resultMap;
    }


    /**
     * 工具方法，获取HttpServletRequest 线程安全
     */
    public static HttpServletRequest getHttpServletRequest() {
        return ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getRequest();
    }

    /**
     * 工具方法，获取HttpServletRequest 线程安全
     */
    public static HttpServletResponse getHttpServletResponse() {
        return ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getResponse();
    }

    /**
     * 如果ContextPath为空，则返回request.getRequestURI()，否则返回request.getRequestURI()减去request.getContextPath()的部分;
     * <br/>
     * 比如http://localhost/context/getUser返回getUser（其中 /context 是 request.getContextPath() ）;
     * 比如http://localhost/getUser返回getUser
     *
     * @param request
     * @return
     */
    public static String getRequestPath(HttpServletRequest request) {
        String uri = request.getRequestURI();
        String contextPath = request.getContextPath();
        if (contextPath != null && contextPath.length() > 0) {
            uri = uri.substring(contextPath.length());
        }

        return uri;
    }
}
