package cn.com.duiba.jdactivity.dao.redis;

import org.springframework.beans.factory.FactoryBean;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisNode;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import redis.clients.jedis.JedisPoolConfig;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

public class JedisConnectionFactoryBean implements FactoryBean<JedisConnectionFactory> {

    @Resource
    private RedisProperties redisProperties;


    public JedisConnectionFactoryBean() {
    }

    private RedisProperties getRedisProperties() {
        return redisProperties;
    }

    @Override
    public JedisConnectionFactory getObject() throws Exception {
        JedisConnectionFactory f = applyProperties(createJedisConnectionFactory());
        f.afterPropertiesSet();
        return f;
    }

    @Override
    public Class<?> getObjectType() {
        return JedisConnectionFactory.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    private JedisConnectionFactory createJedisConnectionFactory() {
        JedisPoolConfig poolConfig = (this.getRedisProperties().getJedis().getPool() != null) ? jedisPoolConfig()
                : new JedisPoolConfig();
        if (getSentinelConfig() != null) {
            return new JedisConnectionFactory(getSentinelConfig(), poolConfig);
        }
        if (getClusterConfiguration() != null) {
            return new JedisConnectionFactory(getClusterConfiguration(), poolConfig);
        }
        return new JedisConnectionFactory(poolConfig);
    }

    private JedisPoolConfig jedisPoolConfig() {
        JedisPoolConfig config = new JedisPoolConfig();
        RedisProperties.Pool props = this.getRedisProperties().getJedis().getPool();
        config.setMaxTotal(props.getMaxActive());
        config.setMaxIdle(props.getMaxIdle());
        config.setMinIdle(props.getMinIdle());
        config.setMaxWaitMillis(props.getMaxWait().toMillis());
        config.setTestWhileIdle(true);
        config.setTimeBetweenEvictionRunsMillis(90000);//每隔90S testWhileIdle一次

        // 按如下文章优化: https://mp.weixin.qq.com/s/6cJ5JuEgEWmMBzJFBDsSMg
        config.setSoftMinEvictableIdleTimeMillis(180000);
        config.setMinEvictableIdleTimeMillis(-1);
        config.setNumTestsPerEvictionRun(-1);

        return config;
    }

    protected final JedisConnectionFactory applyProperties(
            JedisConnectionFactory factory) {
        factory.setHostName(this.getRedisProperties().getHost());
        factory.setPort(this.getRedisProperties().getPort());
        if (this.getRedisProperties().getPassword() != null) {
            factory.setPassword(this.getRedisProperties().getPassword());
        }
        if (this.getRedisProperties().isSsl()) {
            factory.setUseSsl(true);
        }
        factory.setDatabase(this.getRedisProperties().getDatabase());
        if (this.getRedisProperties().getTimeout().toMillis() > 0) {
            factory.setTimeout((int) this.getRedisProperties().getTimeout().toMillis());
        }
        return factory;
    }

    protected final RedisSentinelConfiguration getSentinelConfig() {
        RedisProperties.Sentinel sentinelProperties = this.getRedisProperties().getSentinel();
        if (sentinelProperties != null) {
            RedisSentinelConfiguration config = new RedisSentinelConfiguration();
            config.master(sentinelProperties.getMaster());
            config.setSentinels(createSentinels(sentinelProperties));
            return config;
        }
        return null;
    }

    /**
     * Create a {@link RedisClusterConfiguration} if necessary.
     *
     * @return {@literal null} if no cluster settings are set.
     */
    protected final RedisClusterConfiguration getClusterConfiguration() {
        if (this.getRedisProperties().getCluster() == null) {
            return null;
        }
        RedisProperties.Cluster clusterProperties = this.getRedisProperties().getCluster();
        RedisClusterConfiguration config = new RedisClusterConfiguration(clusterProperties.getNodes());

        if (clusterProperties.getMaxRedirects() != null) {
            config.setMaxRedirects(clusterProperties.getMaxRedirects());
        }
        return config;
    }

    private List<RedisNode> createSentinels(RedisProperties.Sentinel sentinel) {
        List<RedisNode> sentinels = new ArrayList<>();
        for (String node : sentinel.getNodes()) {
            try {
                String[] parts = StringUtils.split(node, ":");
                Assert.state(parts.length == 2, "Must be defined as 'host:port'");
                sentinels.add(new RedisNode(parts[0], Integer.valueOf(parts[1])));
            } catch (RuntimeException ex) {
                throw new IllegalStateException(
                        "Invalid redis sentinel " + "property '" + node + "'", ex);
            }
        }
        return sentinels;
    }
}
