package cn.com.duiba.jdactivity.interceptor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.UUID;

/**
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/6/3 14:06
 */
public class RecordInterceptor implements HandlerInterceptor {
    public static final Logger LOGGER = LoggerFactory.getLogger(RecordInterceptor.class);

    public static final String REQUEST_ID_KEY = "request-id";

    private static String createUUID() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

    public static String getRequestId(HttpServletRequest request) {
        String requestId;
        String parameterRequestId = request.getParameter(REQUEST_ID_KEY);
        String headerRequestId = request.getHeader(REQUEST_ID_KEY);
        // 根据请求参数或请求头判断是否有“request-id”，有则使用，无则创建
        if (parameterRequestId == null && headerRequestId == null) {
            LOGGER.debug("no request-id in request parameter or header");
            requestId = createUUID();
        } else {
            requestId = parameterRequestId != null ? parameterRequestId : headerRequestId;
        }

        return requestId;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        MDC.put(REQUEST_ID_KEY, getRequestId(request));
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        //把requestId添加到响应头，以便其它应用使用
        response.addHeader(REQUEST_ID_KEY, MDC.get(REQUEST_ID_KEY));
        //请求完成，从MDC中移除requestId
        MDC.remove(REQUEST_ID_KEY);
    }
}
