package cn.com.duiba.jdactivity.developer.jd.utils;

import cn.com.duiba.jdactivity.common.utils.RequestTool;
import cn.com.duiba.jdactivity.common.vo.Result;
import cn.com.duiba.jdactivity.common.vo.ResultBuilder;
import cn.com.duiba.jdactivity.developer.jd.Main;
import cn.com.duiba.jdactivity.developer.jd.constant.JdAppEnum;
import cn.com.duiba.jdactivity.developer.jd.constant.JdServerUrlEnum;
import com.alibaba.fastjson.JSON;
import com.jd.open.api.sdk.DefaultJdClient;
import com.jd.open.api.sdk.JdClient;
import com.jd.open.api.sdk.domain.crm.VenderCustomerClientJsfService.response.getCustomer.EssentialCustomerInfo;
import com.jd.open.api.sdk.domain.gongzhonghao.JzoneJosService.response.addCartItemByPin.ResultData;
import com.jd.open.api.sdk.domain.market.PurchaseServiceProvider.response.getPurchaseInfo.JmServiceResult;
import com.jd.open.api.sdk.domain.market.PurchaseServiceProvider.response.getPurchaseInfo.PurchaseForJos;
import com.jd.open.api.sdk.domain.seller.ShopSafService.response.query.ShopJosResult;
import com.jd.open.api.sdk.domain.sku.BeanWriteService.response.sendBean.BeanSendResult;
import com.jd.open.api.sdk.domain.user.CrmCustomerService.response.getMemberInVender.CommonResult;
import com.jd.open.api.sdk.domain.user.CrmCustomerService.response.getMemberInVender.CustomerInfoEs;
import com.jd.open.api.sdk.domain.user.UserRelatedRpcService.response.getUserInfoByOpenId.OAuthUserInfo;
import com.jd.open.api.sdk.request.JdRequest;
import com.jd.open.api.sdk.request.crm.PointsJosSendPointsRequest;
import com.jd.open.api.sdk.request.crm.PopCrmCustomerGetCustomerRequest;
import com.jd.open.api.sdk.request.gongzhonghao.JzoneAddCartItemByPinRequest;
import com.jd.open.api.sdk.request.market.GetPurchaseInfoRequest;
import com.jd.open.api.sdk.request.openid.JosTokenSourceToOpenIdRequest;
import com.jd.open.api.sdk.request.promotion.SellerCouponWriteCreateRequest;
import com.jd.open.api.sdk.request.promotion.SellerCouponWriteLockCouponRequest;
import com.jd.open.api.sdk.request.promotion.SellerCouponWritePushCouponRequest;
import com.jd.open.api.sdk.request.seller.VenderShopQueryRequest;
import com.jd.open.api.sdk.request.sku.PopBeanSendBeanRequest;
import com.jd.open.api.sdk.request.user.*;
import com.jd.open.api.sdk.request.ware.SkuReadFindSkuByIdRequest;
import com.jd.open.api.sdk.request.ware.WareReadFindWareByIdRequest;
import com.jd.open.api.sdk.request.ware.WareReadSearchWare4ValidRequest;
import com.jd.open.api.sdk.response.crm.PointsJosSendPointsResponse;
import com.jd.open.api.sdk.response.crm.PopCrmCustomerGetCustomerResponse;
import com.jd.open.api.sdk.response.gongzhonghao.JzoneAddCartItemByPinResponse;
import com.jd.open.api.sdk.response.market.GetPurchaseInfoResponse;
import com.jd.open.api.sdk.response.openid.JosTokenSourceToOpenIdResponse;
import com.jd.open.api.sdk.response.promotion.SellerCouponWriteCreateResponse;
import com.jd.open.api.sdk.response.promotion.SellerCouponWriteLockCouponResponse;
import com.jd.open.api.sdk.response.promotion.SellerCouponWritePushCouponResponse;
import com.jd.open.api.sdk.response.seller.VenderShopQueryResponse;
import com.jd.open.api.sdk.response.sku.PopBeanSendBeanResponse;
import com.jd.open.api.sdk.response.user.*;
import com.jd.open.api.sdk.response.ware.SkuReadFindSkuByIdResponse;
import com.jd.open.api.sdk.response.ware.WareReadFindWareByIdResponse;
import com.jd.open.api.sdk.response.ware.WareReadSearchWare4ValidResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;

/**
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/5/13 15:08
 */
@Component
public class JdApiUtil {
    public static final Logger LOGGER = LoggerFactory.getLogger(JdApiUtil.class);
    private static final String CODE_SUCCESS = "0";
    private static final DateTimeFormatter formatTime = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private static DefaultJdClient getJdClient(JdAppEnum jdAppEnum, String accessToken) {
        return new DefaultJdClient(JdServerUrlEnum.API_URL.getUrl(), accessToken, jdAppEnum.getAppKey(), jdAppEnum.getAppSecret());
    }

    private static Result<Boolean> sendPoints(JdAppEnum jdAppEnum, String pin, String businessId,
                                              short sourceType,
                                              Integer points, String desc, String accessToken) {
        try {
            PointsJosSendPointsRequest request = new PointsJosSendPointsRequest();
            request.setPin(pin);
            request.setBusinessId(businessId);
            /*
            发放积分：
            27 - 发放积分，
            29 - 店铺签到发放，
            30 - 关注店铺发放，
            31 - 互动积分发放，
            32 - 其他渠道发放

            扣减积分：
            26 - 消费积分，
            33 - 兑换优惠券消耗，
            34 - 兑换红包消耗，
            35 - 兑换京豆消耗，
            36 - 兑换其他权益消耗，
            37 - 互动消耗积分
             */
            request.setSourceType(sourceType);
            request.setPoints(points);
            request.setSourceComment(StringUtils.substring(desc, 0, 10));

            Result<PointsJosSendPointsResponse> result = execute(jdAppEnum, accessToken, request);
            if (!result.getSuccess()) {
                return ResultBuilder.fail(result.getDesc());
            }
            PointsJosSendPointsResponse response = result.getData();

            return Optional.ofNullable(response)
                    .map(PointsJosSendPointsResponse::getJsfResult)
                    .map(resultData -> ResultBuilder.success(Objects.equals("200", resultData.getCode())))
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("积分发放接口,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 底层
     */
    private static <T extends com.jd.open.api.sdk.response.AbstractResponse> Result<T> execute(JdAppEnum jdAppEnum, String accessToken,
                                                                                               JdRequest<T> request) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            T response = client.execute(request);
            LOGGER.info("调用京东api,结果,request={},response={}", JSON.toJSONString(request), JSON.toJSONString(response));
            if (!CODE_SUCCESS.equals(response.getCode())) {
                return ResultBuilder.fail("失败:" + Optional.ofNullable(response.getZhDesc()).orElse(""));
            }
            return ResultBuilder.success(response);
        } catch (Exception e) {
            LOGGER.warn("调用京东api发生异常,exception,request:{}", JSON.toJSONString(request), e);
            return ResultBuilder.fail("调用京东api发生异常");
        }
    }

    /**
     * 查询商家基本信息
     */
    public Result<ShopJosResult> venderShopQuery(JdAppEnum jdAppEnum, String accessToken) {
        try {
            VenderShopQueryRequest request = new VenderShopQueryRequest();
            JdClient client = getJdClient(jdAppEnum, accessToken);
            VenderShopQueryResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getShopJosResult());
        } catch (Exception e) {
            LOGGER.warn("venderShopQuery,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 查询用户明文PIN
     *
     * @param source 01: 京东 App，02：微信
     */
    public Result<String> getPinByToken(JdAppEnum jdAppEnum, String accessToken, String source, String token) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            PopJmCenterUserGetOpenIdRequest request = new PopJmCenterUserGetOpenIdRequest();
            request.setSource(source);
            request.setToken(token);
            PopJmCenterUserGetOpenIdResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getReturnType().getPin());
        } catch (Exception e) {
            LOGGER.warn("getPinByToken,exception", e);
            return ResultBuilder.fail("发生异常");
        }

    }

    /**
     * 查询商品详情
     *
     * @param
     */
    public Result<String> getGoodsDetail(JdAppEnum jdAppEnum, String accessToken, String wareId) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            WareReadSearchWare4ValidRequest request = new WareReadSearchWare4ValidRequest();
            request.setWareId(wareId);
            WareReadSearchWare4ValidResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getMsg());
            }
            return ResultBuilder.success(response.getMsg());
        } catch (Exception e) {
            LOGGER.warn("getPinByToken,exception", e);
            return ResultBuilder.fail("发生异常");
        }

    }

    /**
     * 查询商品详情
     *
     * @param
     */
    public Result<String> findWareById(JdAppEnum jdAppEnum, String accessToken, Long wareId) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            WareReadFindWareByIdRequest request = new WareReadFindWareByIdRequest();
            request.setWareId(wareId);
            WareReadFindWareByIdResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getMsg());
            }
            return ResultBuilder.success(response.getMsg());
        } catch (Exception e) {
            LOGGER.warn("getPinByToken,exception", e);
            return ResultBuilder.fail("发生异常");
        }

    }

    /**
     * 查询用户加密PIN
     *
     * @param source 01: 京东 App，02：微信
     */
    public Result<String> getEncryptPinNewByToken(JdAppEnum jdAppEnum, String accessToken, String source, String token) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            PopJmCenterUserGetEncryptPinNewRequest request = new PopJmCenterUserGetEncryptPinNewRequest();
            request.setSource(source);
            request.setToken(token);
            PopJmCenterUserGetEncryptPinNewResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getReturnType().getPin());
        } catch (Exception e) {
            LOGGER.warn("venderShopQuery,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 查询用户openId
     *
     * @param source 01: 京东 App，02：微信
     */
    public Result<String> getOpenIdByToken(JdAppEnum jdAppEnum, String source, String token) {
        try {
            JdClient client = getJdClient(jdAppEnum, "");
            JosTokenSourceToOpenIdRequest request = new JosTokenSourceToOpenIdRequest();
            request.setToken(token);
            request.setSource(source);
            request.setAppKey(jdAppEnum.getAppKey());
            JosTokenSourceToOpenIdResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getResult().getOpenId());
        } catch (Exception e) {
            LOGGER.warn("venderShopQuery,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 根据 openId 获取用户信息
     * 性别 (0: 男，1: 女)
     */
    public Result<OAuthUserInfo> getUserInfoByOpenId(JdAppEnum jdAppEnum, String accessToken, String openId) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            UserGetUserInfoByOpenIdRequest request = new UserGetUserInfoByOpenIdRequest();
            request.setOpenId(openId);
            UserGetUserInfoByOpenIdResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getGetuserinfobyappidandopenidResult().getData());
        } catch (Exception e) {
            LOGGER.warn("venderShopQuery,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 查询 pin 在商家下的会员信息
     * 性别 (0: 男，1: 女)
     */
    public Result<EssentialCustomerInfo> getCustomer(JdAppEnum jdAppEnum, String accessToken, String pin) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            PopCrmCustomerGetCustomerRequest request = new PopCrmCustomerGetCustomerRequest();
            request.setCustomerPin(pin);
            request.setOpenIdBuyer(pin);
            PopCrmCustomerGetCustomerResponse response = client.execute(request);
            LOGGER.info("查询 pin 在商家下的会员信息,pin={},response={}", pin, JSON.toJSONString(response));
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return ResultBuilder.success(response.getReturnResult().getData());
        } catch (Exception e) {
            LOGGER.warn("venderShopQuery,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 返回用户在某个商家下的会员信息
     *
     * @deprecated 注意：立刻成为会员后，该接口有延迟
     */
    @Deprecated
    public Result<CustomerInfoEs> getMemberInVender(JdAppEnum jdAppEnum, String accessToken, String pin) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            PopCrmGetMemberInVenderRequest request = new PopCrmGetMemberInVenderRequest();
            request.setCustomerPin(pin);
            request.setOpenIdBuyer(pin);
            PopCrmGetMemberInVenderResponse response = client.execute(request);
            LOGGER.info("返回用户在某个商家下的会员信息,pin={},response={}", pin, JSON.toJSONString(response));
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return Optional.ofNullable(response)
                    .map(PopCrmGetMemberInVenderResponse::getGetmemberinvenderResult)
                    .map(CommonResult::getCustomerInfoEs)
                    .map(ResultBuilder::success)
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));

        } catch (Exception e) {
            LOGGER.warn("getMemberInVender,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 是否已关注该店铺
     */
    public Result<Boolean> isFollowByPinAndVid(JdAppEnum jdAppEnum, String pin, Long shopId) {
        try {
            JdClient client = getJdClient(jdAppEnum, "");
            FollowVenderReadIsFollowByPinAndVidRequest request = new FollowVenderReadIsFollowByPinAndVidRequest();
            request.setOpenIdBuyer(pin);
            request.setPin(pin);
            request.setShopId(shopId);
            FollowVenderReadIsFollowByPinAndVidResponse response = client.execute(request);
            /*
            {"jingdong_follow_vender_read_isFollowByPinAndVid_responce":{"isfollowbypinandvid_result":{"code":"F10000","data":true},"code":"0"}}
             */
            LOGGER.info("是否已关注该店铺,pin={},shopId={},response={}", pin, shopId, JSON.toJSONString(response));
            return Optional.ofNullable(response)
                    .filter(followVenderReadIsFollowByPinAndVidResponse -> Objects.equals(CODE_SUCCESS, followVenderReadIsFollowByPinAndVidResponse.getCode()))
                    .map(FollowVenderReadIsFollowByPinAndVidResponse::getIsfollowbypinandvidResult)
                    .map(result -> ResultBuilder.success(result.getData()))
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));

        } catch (Exception e) {
            LOGGER.warn("isFollowByPinAndVid,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 关注该店铺
     * 如果已关注会返回false
     */
    public Result<Boolean> followByPinAndVid(JdAppEnum jdAppEnum, String pin, Long shopId, String accessToken) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            FollowVenderWriteFollowByPinAndVidRequest request = new FollowVenderWriteFollowByPinAndVidRequest();
            request.setPin(pin);
            request.setShopId(shopId);
            request.setOpenIdBuyer(pin);
            FollowVenderWriteFollowByPinAndVidResponse response = client.execute(request);
            LOGGER.info("关注该店铺,pin={},shopId={},response={}", pin, shopId, JSON.toJSONString(response));
            return Optional.ofNullable(response)
                    .map(FollowVenderWriteFollowByPinAndVidResponse::getFollowbypinandvidResult)
                    .map((Function<com.jd.open.api.sdk.domain.user.FollowVenderWriteRpcService.response.followByPinAndVid.Result, Result<Boolean>>)
                            result -> {
                                Boolean data = result.getData();
                                String code = result.getCode();
                                String msg = result.getMsg();
                                if (!data) {
                                    // 已关注情况,会返回false,要手动修复为true
                                    if (Objects.equals(code, "F0402") || Objects.equals(msg, "已关注")) {
                                        return ResultBuilder.success(Boolean.TRUE);
                                    } else if (Objects.equals(code, "F0410")) {
                                        return ResultBuilder.fail("用户关注店铺数量已达上限500");
                                    } else {
                                        return ResultBuilder.fail(code, msg);
                                    }
                                } else {
                                    return ResultBuilder.success(Boolean.TRUE);
                                }
                            })
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("followByPinAndVid,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 通过用户pin加入购物车
     */
    public Result<Boolean> addCartItemByPin(JdAppEnum jdAppEnum, String pin, String itemId, Integer num, String accessToken) {
        try {
            JzoneAddCartItemByPinRequest request = new JzoneAddCartItemByPinRequest();
            request.setPin(pin);
            request.setItemId(itemId);
            request.setNum(num);

            Result<JzoneAddCartItemByPinResponse> result = execute(jdAppEnum, accessToken, request);
            if (!result.getSuccess()) {
                return ResultBuilder.fail(result.getDesc());
            }
            JzoneAddCartItemByPinResponse response = result.getData();

            return Optional.ofNullable(response)
                    .map(JzoneAddCartItemByPinResponse::getReturnType)
                    .map((Function<ResultData, Result<Boolean>>) resultData -> {
                        boolean success = Objects.equals("0", resultData.getCode());
                        if (success) {
                            return ResultBuilder.success(Boolean.TRUE);
                        }
                        return ResultBuilder.fail(resultData.getMessage());
                    })
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("通过用户pin加入购物车,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 收藏商品
     */
    public Result<Boolean> followGood(JdAppEnum jdAppEnum, String pin, Long productId, String accessToken) {
        try {
            FollowProductComplexWriteFollowRequest request = new FollowProductComplexWriteFollowRequest();
            request.setPin(pin);
            request.setProductId(productId);

            Result<FollowProductComplexWriteFollowResponse> result = execute(jdAppEnum, accessToken, request);
            if (!result.getSuccess()) {
                return ResultBuilder.fail(result.getDesc());
            }
            FollowProductComplexWriteFollowResponse response = result.getData();


            return Optional.ofNullable(response)
                    .map(FollowProductComplexWriteFollowResponse::getFollowResult)
                    .map((Function<com.jd.open.api.sdk.domain.user.FollowProductWriteRpcService.response.follow.Result,Result<Boolean>>) resultData -> {
                        boolean success = Objects.equals("F10000", resultData.getCode());
                        if (success) {
                            return ResultBuilder.success(Boolean.TRUE);
                        }
                        return ResultBuilder.fail(resultData.getMsg());
                    }).orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("关注商品,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }


    /**
     * 互动积分发放积分
     *
     * @param businessId 防重 ID，建议是 uuid
     */
    public Result<Boolean> sendPoints(JdAppEnum jdAppEnum, String pin, String businessId,
                                      Integer points, String desc, String accessToken) {
        if (points <= 0) {
            return ResultBuilder.fail("发放积分须是正数");
        }
        return sendPoints(jdAppEnum, pin, businessId, (short) 31, points, desc, accessToken);
    }

    /**
     * 互动消耗积分
     *
     * @param businessId 防重 ID，建议是 uuid
     */
    public Result<Boolean> consumePoints(JdAppEnum jdAppEnum, String pin, String businessId,
                                         Integer points, String desc, String accessToken) {
        if (points >= 0) {
            return ResultBuilder.fail("消费积分须是负数");
        }
        return sendPoints(jdAppEnum, pin, businessId, (short) 37, points, desc, accessToken);
    }

    /**
     * 获取当前服务当前用户的有效订购记录
     */
    public Result<List<PurchaseForJos>> getPurchaseInfo(JdAppEnum jdAppEnum, String accessToken) {
        try {
            JdClient client = getJdClient(jdAppEnum, accessToken);
            GetPurchaseInfoRequest request = new GetPurchaseInfoRequest();
            GetPurchaseInfoResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("失败:request:{},return_code:{},return_msg:{}", JSON.toJSONString(request), response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("失败:" + response.getZhDesc());
            }
            return Optional.ofNullable(response)
                    .map(GetPurchaseInfoResponse::getReturnType)
                    .map(JmServiceResult::getPurchaseInfoList)
                    .map(ResultBuilder::success)
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("getPurchaseInfo,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    public Result<Boolean> lockCoupon(JdAppEnum jdAppEnum, String accessToken, Long couponId) {
        try {
            LOGGER.info("锁券：couponId:{}", couponId);
            JdClient client = getJdClient(jdAppEnum, accessToken);
            SellerCouponWriteLockCouponRequest request = new SellerCouponWriteLockCouponRequest();
            request.setRequestId(UUID.randomUUID().toString());
            request.setTime(System.currentTimeMillis());
            request.setPort(getPort());
            //request.setPurpose("无");
            request.setOperateTime(formatTime.format(LocalDateTime.now()));
            request.setCouponId(couponId);

            SellerCouponWriteLockCouponResponse response = client.execute(request);

            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("锁券失败:couponId:{},return_code:{},return_msg:{}", couponId, response.getCode(), response.getZhDesc());
                return ResultBuilder.fail("锁券失败:" + response.getZhDesc());
            }
            LOGGER.info("锁券成功：couponId:{}", couponId);
            return ResultBuilder.success();
        } catch (Exception e) {
            LOGGER.warn("lockCoupon,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 优惠券发放
     * 同一个人：重复发券
     * 不同一个人，同一个uuid：发放成功,
     * 不同一个人，同一个requestId：发放成功,
     * 库存不足：发券失败
     * 不锁直接发：优惠券未锁定不能发放
     *
     * @param jdAppEnum
     * @param accessToken
     * @param couponId
     * @param pin
     * @param uuid
     * @return
     */
    public Result<Boolean> pushCoupon(JdAppEnum jdAppEnum, String accessToken, Long couponId, String pin, String uuid) {
        try {
            LOGGER.info("发券：couponId:{}，pin：{}", couponId, pin);
            JdClient client = getJdClient(jdAppEnum, accessToken);
            SellerCouponWritePushCouponRequest request = new SellerCouponWritePushCouponRequest();

            request.setPort(getPort());
            request.setRequestId(uuid);
            request.setPin(pin);
            request.setDistrTime(formatTime.format(LocalDateTime.now()));
            request.setCouponId(couponId);
            request.setUuid(uuid);
            request.setOpenIdBuyer(pin);
            SellerCouponWritePushCouponResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("发券失败:couponId:{},return_code:{},return_msg:{}", couponId, response.getCode(), response.getMsg());
                return ResultBuilder.fail(response.getCode(), response.getZhDesc());
            }
            LOGGER.info("发券成功：couponId:{}，pin：{}", couponId, pin);
            return ResultBuilder.success(Boolean.TRUE);
        } catch (Exception e) {
            LOGGER.warn("pushCoupon,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    /**
     * 新建优惠券 (不支持自营)
     *
     * @param jdAppEnum
     * @param accessToken
     * @return
     */
    public Result<Long> createCoupon(JdAppEnum jdAppEnum,
                                     String accessToken,
                                     String couponName,
                                     Integer couponNumber) {
        try {
            LOGGER.warn("创建优惠券:couponName:{},couponNumber={}", couponName, couponNumber);
            JdClient client = getJdClient(jdAppEnum, accessToken);
            SellerCouponWriteCreateRequest request = new SellerCouponWriteCreateRequest();
            request.setIp(RequestTool.getIpAddr(RequestTool.getHttpServletRequest()));
            request.setPort(getPort());
            request.setName(couponName);
            // 优惠券类型 0 京券 1 东券
            request.setType(0);
            // 绑定类型 1 全店参加 2 指定 sku 参加
            request.setBindType(1);
            // 发放类型 3 免费领取 5 互动平台 【仅允许这两种】
            request.setGrantType(5);
            request.setNum(couponNumber);
            // 优惠券面额（限制：[1,5000)）
            request.setDiscount(new BigDecimal(1));
            // 优惠限额（限制：[1,100000)）, 京券无门槛传 0
            request.setQuota(new BigDecimal(0));
            // 有效期类型 1 相对时间 5 绝对时间
            request.setValidityType(1);

            // 有效期
            if (request.getValidityType() == 1) {
                request.setDays(1);
            } else {
                request.setBeginTime(System.currentTimeMillis() + 30 * 60 * 1000);
                request.setEndTime(request.getBeginTime() + 30 * 60 * 1000);
            }


            // 会员等级 50 注册会员 56 铜牌 61 银牌 62 金牌 105 钻石 110VIP 90 企业会员
            request.setMember(50);
            // 领券开始时间（晚于当前）
            // request.setTakeBeginTime(System.currentTimeMillis() + 30 * 60 * 1000);
            request.setTakeBeginTime(System.currentTimeMillis() - 1000 * 60 * 60);
            // 领券结束时间（活动时间最长 90 天）
            // request.setTakeEndTime(request.getTakeBeginTime() + 24 * 60 * 60 * 1000);
            request.setTakeEndTime(System.currentTimeMillis() + 1000 * 60 * 60);
            // 领券规则 5 活动期间限领一张 4 活动内每天限领一张【仅免 grantType 为 3，且使用链接领取时有效】
            request.setTakeRule(5);
            // 限制条件内可以领取张数
            request.setTakeNum(1);
            // 是否公开 1 不公开 3 公开
            request.setDisplay(1);
            if (request.getGrantType() == 5) {
                // grantType 如设值 5 此参数必须为 3
                request.setDisplay(3);
            }
            request.setPlatformType(-1);
            // 优惠券使用平台
            // request.setPlatform("[]");
            // 分享类型 1 分享 2 不分享（如设置京券 type=0,2 不分享
            request.setShareType(2);
            // 活动返回链接
            // request.setActivityLink("www.jd.com");
            // 会员类别
            // request.setUserClass(20000);
            // request.setPaidMembers("[]");
            // request.setSkuId(3254235);
            SellerCouponWriteCreateResponse response = client.execute(request);
            if (!CODE_SUCCESS.equals(response.getCode())) {
                LOGGER.warn("创建优惠券失败:request:{},response:{}", JSON.toJSONString(request), JSON.toJSONString(response));
                return ResultBuilder.fail("创建优惠券失败：" + response.getZhDesc());
            }
            Long couponId = response.getCouponId();
            LOGGER.warn("创建优惠券成功:couponId:{}", couponId);
            if (request.getGrantType() == 5) {
                // 为 5 时券需已锁定
                lockCoupon(jdAppEnum, accessToken, couponId);
            }
            return ResultBuilder.success(couponId);
        } catch (Exception e) {
            LOGGER.warn("创建优惠券,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

    private String getPort() {
        return String.valueOf(RequestTool.getHttpServletRequest().getServerPort());
    }

   /* public static void main(String[] args) throws Exception {
        JdApiUtil util = new JdApiUtil();
        JdAppEnum jdAppEnum = JdAppEnum.getAppByAppKey(Main.appKey);
        String token = Main.accessToken;

        Long couponId = Long.valueOf("364829065");
        String pin = "isv_test_acc102";

        //util.create();
        //util.lockCoupon(jdAppEnum,token,couponId);
        util.pushCoupon(jdAppEnum,token,couponId,pin,UUID.randomUUID().toString());
    }*/

    //https://jos.jd.com/apilist?apiGroupId=141&apiId=13752&apiName=null
    public void create() throws Exception {
        JdClient client = new DefaultJdClient(Main.SERVER_URL, Main.accessToken, Main.appKey, Main.appSecret);
        SellerCouponWriteCreateRequest request = new SellerCouponWriteCreateRequest();
        //request.setIp("");
        request.setPort("80");
        request.setName("测试券002");
        request.setType(0);
        request.setBindType(1);
        request.setGrantType(5);
        request.setNum(5);
        request.setDiscount(new BigDecimal(1));
        request.setQuota(new BigDecimal(0));

        //券有效去期
        request.setValidityType(1);
        request.setDays(1);

        request.setMember(50);
        request.setTakeBeginTime(System.currentTimeMillis() - 1000 * 60 * 60);
        request.setTakeEndTime(System.currentTimeMillis() + 1000 * 60 * 60);
        request.setTakeRule(5);
        //限制条件内能领取张数量
        request.setTakeNum(1);
        request.setDisplay(3);
        request.setPlatformType(1);
        request.setShareType(2);
        request.setUserClass(20000);
        SellerCouponWriteCreateResponse response = client.execute(request);
        System.out.println(response.getCouponId().toString());
    }

    /**
     * 发送京豆接口
     */
    public Result<Boolean> sendBean(JdAppEnum jdAppEnum, String accessToken,
                                    String requestId,
                                    Long beanNum,
                                    Long venderId,
                                    Long planId,
                                    String pin,
                                    String desc) {
        try {

            PopBeanSendBeanRequest request = new PopBeanSendBeanRequest();
            request.setRequestId(requestId);
            request.setBeanNum(beanNum);
            // request.setAccountId(0);
            request.setAccountCode(String.valueOf(venderId));
            request.setSendWay(1);
            request.setSendCode(pin);
            request.setAccountType(1);
            request.setContext(desc);
            request.setPlanId(planId);

            Result<PopBeanSendBeanResponse> result = execute(jdAppEnum, accessToken, request);
            if (!result.getSuccess()) {
                return ResultBuilder.fail(result.getDesc());
            }
            PopBeanSendBeanResponse response = result.getData();

            return Optional.of(result.getData())
                    .map(PopBeanSendBeanResponse::getBeanSendResult)
                    .map((Function<BeanSendResult, Result<Boolean>>) beanSendResult -> {
                        boolean success = Objects.equals("成功", beanSendResult.getDesc());
                        if (success) {
                            return ResultBuilder.success(Boolean.TRUE);
                        }
                        return ResultBuilder.fail(beanSendResult.getDesc());
                    })
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("通过用户pin加入购物车,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }


    /**
     * 获取用户积分
     *
     * @param jdAppEnum
     * @param pin
     * @param accessToken
     * @return
     */
    public Result<Long> getCustomerPoints(JdAppEnum jdAppEnum, String pin, String accessToken) {
        try {
            PopCrmGetCustomerPointsRequest request = new PopCrmGetCustomerPointsRequest();
            request.setCustomerPin(pin);
            Result<PopCrmGetCustomerPointsResponse> result = execute(jdAppEnum, accessToken, request);
            if (!result.getSuccess()) {
                return ResultBuilder.fail(result.getDesc());
            }
            PopCrmGetCustomerPointsResponse response = result.getData();

            return Optional.ofNullable(response)
                    .map(r -> ResultBuilder.success(r.getGetcustomerpointsResult()))
                    .orElse(ResultBuilder.fail(JSON.toJSONString(response)));
        } catch (Exception e) {
            LOGGER.warn("获取用户积分接口,exception", e);
            return ResultBuilder.fail("发生异常");
        }
    }

}
