package cn.com.duiba.jdactivity.developer.jd.utils;

import cn.com.duiba.jdactivity.common.utils.HttpClientUtil;
import cn.com.duiba.jdactivity.common.utils.UrlUtils;
import cn.com.duiba.jdactivity.developer.jd.constant.JdAppEnum;
import cn.com.duiba.jdactivity.developer.jd.constant.JdServerUrlEnum;
import cn.com.duiba.jdactivity.developer.jd.domain.AccessTokenResponse;
import cn.com.duiba.jdactivity.dto.ShopParam;
import cn.com.duiba.jdactivity.dto.TbShopAccessTokenDto;
import cn.com.duiba.jdactivity.exception.BizException;
import cn.com.duiba.jdactivity.service.TbShopAccessTokenService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.jd.open.api.sdk.domain.seller.ShopSafService.response.query.ShopJosResult;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

/**
 * // TODO 定时刷新
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/5/11 15:00
 */
@Component
public class AccessTokenUtils {
    public static final Logger LOGGER = LoggerFactory.getLogger(AccessTokenUtils.class);

    private static final String TO_LOGIN_SUFFIX = "/to_login";
    private static final String ACCESS_TOKEN_SUFFIX = "/access_token";
    private static final String NULL = "null";

    @Resource
    private HttpClientUtil httpClientUtil;
    @Resource
    private TbShopAccessTokenService tbShopAccessTokenService;


    private static final Cache<String, TbShopAccessTokenDto> ACCESS_TOKEN_CACHE = Caffeine.newBuilder()
            // 缓存一天，不考虑过期时间
            .expireAfterWrite(1, TimeUnit.DAYS)
            .initialCapacity(10)
            .maximumSize(100)
            .recordStats()
            .build();

    public String generateAuthUrl(JdAppEnum appKey, String state) {
        Map<String, String> param = new TreeMap<>();
        param.put("app_key", appKey.getAppKey());
        param.put("response_type", "code");
        param.put("redirect_uri", appKey.getUrl());
        param.put("scope", "snsapi_base");
        param.put("state", state);
        return UrlUtils.assembleUrl(JdServerUrlEnum.OAUTH_URL.getUrl() + TO_LOGIN_SUFFIX, param);
    }


    public AccessTokenResponse getAccessTokenByCode(JdAppEnum appKey, String code) {
        Map<String, String> param = new TreeMap<>();
        param.put("app_key", appKey.getAppKey());
        param.put("app_secret", appKey.getAppSecret());
        param.put("grant_type", "authorization_code");
        param.put("code", code);
        String url = UrlUtils.assembleUrl(JdServerUrlEnum.OAUTH_URL.getUrl() + ACCESS_TOKEN_SUFFIX, param);
        String s = httpClientUtil.sendGet(url);

        LOGGER.info("getAccessTokenByCode,code={},url={},s={}", code, url, s);

        if (StringUtils.isNotBlank(s)
                && !StringUtils.equalsIgnoreCase(NULL, s)
                && JSONValidator.from(s).validate()) {
            return JSONObject.parseObject(s, AccessTokenResponse.class);
        }

        return null;
    }

    private static String getCacheKey(Long venderId, Long shopId) {
        return String.format("%s_%s", venderId, shopId);
    }

    private TbShopAccessTokenDto assembleTbShopAccessTokenDto(JdAppEnum jdAppEnum, AccessTokenResponse accessTokenByCode, ShopJosResult data) {
        TbShopAccessTokenDto tbShopAccessTokenDto = new TbShopAccessTokenDto();
        tbShopAccessTokenDto.setVenderId(data.getVenderId());
        tbShopAccessTokenDto.setShopId(data.getShopId());
        tbShopAccessTokenDto.setShopName(data.getShopName());
        tbShopAccessTokenDto.setAccessToken(accessTokenByCode.getAccess_token());
        tbShopAccessTokenDto.setRefreshToken(accessTokenByCode.getRefresh_token());

        Long expires_in = accessTokenByCode.getExpires_in();
        Long time = accessTokenByCode.getTime();
        // 过期时间=当前时间+过期时长-5分钟buffer
        long expireTime = time + expires_in * 1000 - 5 * 60 * 1000;
        tbShopAccessTokenDto.setExpiresTime(new Date(expireTime));
        tbShopAccessTokenDto.setAccessTokenOriginal(JSON.toJSONString(accessTokenByCode));
        tbShopAccessTokenDto.setAppKey(jdAppEnum.getAppKey());
        return tbShopAccessTokenDto;
    }

    private static Pair<String, String> splitCacheKey(String key) throws BizException {
        String[] split = key.split("_");
        if (split.length != 2) {
            throw new BizException("key不合法" + key);
        }
        return Pair.of(split[0], split[1]);
    }

    public Long saveAccessToken(JdAppEnum jdAppEnum, AccessTokenResponse accessTokenByCode, ShopJosResult data) {
        ACCESS_TOKEN_CACHE.invalidate(getCacheKey(data.getVenderId(), data.getShopId()));
        TbShopAccessTokenDto tbShopAccessTokenDto = assembleTbShopAccessTokenDto(jdAppEnum, accessTokenByCode, data);
        return tbShopAccessTokenService.insert(tbShopAccessTokenDto);
    }

    public TbShopAccessTokenDto getAccessTokenWithCache(Long venderId, Long shopId) throws BizException {
        if (venderId == null || shopId == null || venderId == 0L || shopId == 0L) {
            throw new BizException("店铺参数不存在");
        }

        String key = getCacheKey(venderId, shopId);
        TbShopAccessTokenDto tbShopAccessTokenDto = ACCESS_TOKEN_CACHE.get(key, s -> {
            try {
                LOGGER.info("开始加载缓存,venderId={},shopId={}", venderId, shopId);
                return getAccessTokenWithoutCache(venderId, shopId);
            } catch (BizException e) {
                LOGGER.warn("加载缓存失败", e);
                return null;
            }
        });

        LOGGER.info("获取缓存结果,venderId={},shopId={},tbShopAccessTokenDto={}", venderId, shopId, tbShopAccessTokenDto != null);

        if (tbShopAccessTokenDto == null) {
            throw new BizException("店铺授权获取失败" + shopId);
        }

        if (tbShopAccessTokenDto.getExpiresTime().before(new Date())) {
            ACCESS_TOKEN_CACHE.invalidate(key);
            throw new BizException("店铺授权获取失败" + shopId);
        }

        return tbShopAccessTokenDto;
    }

    public TbShopAccessTokenDto getAccessTokenWithCache(ShopParam param) throws BizException {
        return getAccessTokenWithCache(param.getVenderId(), param.getShopId());
    }

    private TbShopAccessTokenDto getAccessTokenWithoutCache(Long venderId, Long shopId) throws BizException {
        if (venderId == null || shopId == null || venderId == 0L || shopId == 0L) {
            throw new BizException("店铺参数不存在");
        }
        Date now = new Date();
        List<TbShopAccessTokenDto> tbShopAccessTokenDtos = tbShopAccessTokenService.queryByVenderIdShopId(venderId, shopId);
        if (CollectionUtils.isEmpty(tbShopAccessTokenDtos)) {
            throw new BizException("店铺未授权" + shopId);
        }

        tbShopAccessTokenDtos.sort(Comparator.comparing(TbShopAccessTokenDto::getExpiresTime));

        for (int i = tbShopAccessTokenDtos.size() - 1; i >= 0; i--) {
            TbShopAccessTokenDto tbShopAccessTokenDto = tbShopAccessTokenDtos.get(i);
            if (tbShopAccessTokenDto.getExpiresTime().after(now)) {
                return tbShopAccessTokenDto;
            }
        }

        throw new BizException("店铺授权已失效" + shopId);
    }

    /*
    {"error_response":{"code":"19","en_desc":"Invalid access_token(Solution reference: https://jos.jd.com/commondoc?listId=171)","zh_desc":"token已过期或者不存在，请重新授权,access_token:c99865d29305423b87693b8c511c6799zdi5（解决方案参考: https://jos.jd.com/commondoc?listId=171）"}}
     */

}
