package cn.com.duiba.jdactivity.exception;

/**
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/5/18 20:30
 */
public class BizException extends Exception {
    private static final int MAX_DEPTH = 25;

    /**
     * 异常错误码, 比如用户已存在可以表示为 user.already.exists, 图形验证码错误可以表示为 invalid.verify.code，命名请尽量有意义些
     * <br/>
     * 这个字段可以用于表示国际化的key。
     */
    private String code;

    /**
     * 构造一个业务异常，通常而言这个异常的message是应该直接回显给用户的，比如"用户已存在".
     *
     * @param message 默认异常消息，表示具体的错误信息，比如『用户不存在』(后期国际化以后，这个可以用于表示默认异常消息，服务端优先根据key去找到对应语言的错误消息并替换message【服务端从RpcContext中获得当前语言环境】，找不到则不替换)
     */
    public BizException(String message) {
        super(message);
    }

    /**
     * 判断异常的根cause是不是BizException，如果是的话，返回这个BizException，如果不是，原样抛出异常
     *
     * @param e
     * @return
     */
    public static BizException getOrPropagate(Throwable e) {
        Throwable finalE = getFinalCause(e);
        if (finalE instanceof BizException) {
            return (BizException) finalE;
        }

        throw propagate(e);
    }

    private static Throwable getFinalCause(Throwable e) {
        int i = 0;
        Throwable e1 = e;
        while (e1.getCause() != null) {
            if (i++ >= MAX_DEPTH) {
                // stack too deep to get final cause
                return new RuntimeException("Stack too deep to get final cause");
            }
            e1 = e1.getCause();
        }
        return e1;
    }

    private static RuntimeException propagate(Throwable t) {
        /*
         * The return type of RuntimeException is a trick for code to be like this:
         *
         * throw Exceptions.propagate(e);
         *
         * Even though nothing will return and throw via that 'throw', it allows the code to look like it
         * so it's easy to read and understand that it will always result in a throw.
         */
        if (t instanceof RuntimeException) {
            throw (RuntimeException) t;
        } else if (t instanceof Error) {
            throw (Error) t;
        } else {
            throw new RuntimeException(t); // NOPMD
        }
    }

    /**
     * 设置异常错误码
     *
     * @param code
     * @return
     */
    public BizException withCode(String code) {
        this.code = code;
        return this;
    }

    /**
     * 异常错误码, 比如用户已存在可以表示为 user.already.exists, 图形验证码错误可以表示为 invalid.verify.code，命名请尽量有意义些
     * <br/>
     * 后期国际化的时候，这个字段可以用于表示国际化的key。
     */
    public String getCode() {
        return code;
    }
}
