package cn.com.duiba.jdactivity.common.utils;

import com.alibaba.fastjson.JSON;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * URLUtils
 * <p/>
 * Created by wangyuanjie on 2018/3/21.
 */
public final class UrlUtils {

    /**
     * 拼接签名字符串
     *
     * @param prefix      前缀
     * @param queryString 参数
     * @return 拼接后的字符串
     */
    public static String assembleUrl(String prefix, String queryString) {
        if (StringUtils.isBlank(prefix)) {
            return queryString;
        } else {
            return prefix + (prefix.contains("?") ? "&" : "?") + queryString;
        }
    }

    /**
     * 拼接签名字符串
     *
     * @param prefix    前缀
     * @param paramsMap 参数
     * @return 拼接后的字符串
     */
    public static String assembleUrl(String prefix, Map<String, ?> paramsMap) {
        return assembleUrl(prefix, paramsMap, true);
    }

    /**
     * 拼接签名字符串
     *
     * @param prefix      前缀
     * @param paramsMap   参数
     * @param isUrlEncode 是否urlEncode
     * @return 拼接后的字符串
     */
    public static String assembleUrl(String prefix, Map<String, ?> paramsMap, boolean isUrlEncode) {
        return assembleUrl(prefix, mapToQueryString(paramsMap, true, isUrlEncode));
    }

    /**
     * 将map转为queryString
     *
     * @param map         参数
     * @param isSort      是否排序
     * @param isUrlEncode 是否需要UrlEncode
     * @return queryString
     */
    public static String mapToQueryString(Map<String, ?> map, boolean isSort, boolean isUrlEncode) {
        Map<String, ?> tempMap;
        if (isSort) {
            tempMap = new TreeMap<>(map);
        } else {
            tempMap = map;
        }

        // map拼接
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, ?> entry : tempMap.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            String valueStr = objectToString(value);
            if (StringUtils.isBlank(valueStr)) {
                continue;
            }
            if (isUrlEncode) {
                valueStr = urlEncode(valueStr);
            }
            // 剔除空字符串
            if (StringUtils.isNotBlank(valueStr)) {
                sb.append(key).append("=").append(valueStr).append("&");
            }
        }

        String queryString = sb.toString();
        if (queryString.length() > 1) {
            queryString = queryString.substring(0, queryString.length() - 1);
        }
        return queryString;
    }

    /**
     * 将对象转换为String
     * 若对象为集合，则转换为json
     *
     * @param object
     * @return
     */
    public static String objectToString(Object object) {
        if (null == object) {
            return "";
        }
        return (object instanceof Map || object instanceof Collection)
                ? JSON.toJSONString(object) : object + "";
    }

    /**
     * urlEncode
     *
     * @param str 字符串
     * @return urlEncode之后的字符串
     */
    public static String urlEncode(String str) {
        try {
            String[] temp = str.split("\\ ");
            StringBuilder sb = new StringBuilder();
            for (int t = 0; t < temp.length; t++) {
                sb.append(URLEncoder.encode(temp[t], StandardCharsets.UTF_8.name()));
                if (t < temp.length - 1) {
                    sb.append("%20");
                }
            }
            return sb.toString();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public static String urlDecode(String url) {
        if (StringUtils.isBlank(url)) {
            return "";
        }
        try {
            return URLDecoder.decode(StringEscapeUtils.unescapeHtml3(url), "utf-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return url;
    }

    public static Map<String, String> explainURLParams(String params) {
        Map<String, String> mapRequest = new HashMap<>();
        String[] arrSplit;
        // 每个键值为一组
        arrSplit = params.split("[&]");
        for (String strSplit : arrSplit) {
            String[] arrSplitEqual;
            arrSplitEqual = strSplit.split("[=]");
            // 解析出键值
            if (arrSplitEqual.length > 1) {
                // 正确解析
                mapRequest.put(arrSplitEqual[0], arrSplitEqual[1]);
            } else {
                if (arrSplitEqual[0] != "") {
                    // 只有参数没有值，加入空字符
                    mapRequest.put(arrSplitEqual[0], "");
                }
            }
        }
        return mapRequest;
    }

    /**
     * 为url添加http或https协议
     *
     * @param url
     * @return
     */
    public static String getFullUrl(String url) {
        url = StringUtils.trimToEmpty(url);
        String requestScheme = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest().getScheme();
        if (url.startsWith("http")) {
            return url;
        } else if (url.startsWith("//")) {
            return requestScheme + ":" + url;
        } else {
            return requestScheme + "://" + url;
        }
    }
}
