/**
 * Project Name:activity-cms-web File Name:HttpClientUtil.java Package Name:com.duiba.activity.cmsweb.tool
 * Date:2017年3月13日上午11:27:27 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.jdactivity.common.utils;

import cn.com.duiba.jdactivity.common.log.HttpRequestLog;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.net.ssl.SSLContext;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * ClassName:HttpClientUtil <br/>
 * HttpCilent工具类 Date: 2017年3月13日 上午11:27:27 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Component
public class HttpClientUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpClientUtil.class);

    private static final String CHARACTER_ENCODE = "UTF-8";
    private static final String APPLICATION_JSON = "application/json";
    private static final int MAX_TIMEOUT = 5000;
    private static PoolingHttpClientConnectionManager connMgr;
    private static RequestConfig requestConfig;

    static {
        // 设置连接池
        connMgr = new PoolingHttpClientConnectionManager();
        // 设置连接池大小
        connMgr.setMaxTotal(100);
        connMgr.setDefaultMaxPerRoute(connMgr.getMaxTotal());
        connMgr.setValidateAfterInactivity(3000);

        RequestConfig.Builder configBuilder = RequestConfig.custom();
        // 设置连接超时
        configBuilder.setConnectTimeout(MAX_TIMEOUT);
        // 设置读取超时
        configBuilder.setSocketTimeout(MAX_TIMEOUT);
        // 设置从连接池获取连接实例的超时
        configBuilder.setConnectionRequestTimeout(MAX_TIMEOUT);
        // 在提交请求之前 测试连接是否可用
        requestConfig = configBuilder.build();
    }

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    /**
     * 设置每次请求的超时时间，默认5000,10000,10
     */
    private static void setTimeoutIfAbsent(HttpRequestBase requestBase) {
        final int connectTimeout = 5000;
        // fixme 暂时统一到10s，观察下，再降低到5s
        final int socketTimeout = 10000;
        final int connectionRequestTimeout = 10;

        RequestConfig config;
        RequestConfig existConfig = requestBase.getConfig();
        if (existConfig == null) {
            // new出一个config来设置
            config = RequestConfig.custom()
                    .setConnectTimeout(connectTimeout)
                    .setSocketTimeout(socketTimeout)
                    .setConnectionRequestTimeout(connectionRequestTimeout)
                    .build();
        } else {
            // 复制出一个config来设置
            RequestConfig.Builder builder = RequestConfig.copy(existConfig);
            if (existConfig.getConnectTimeout() == 0 || existConfig.getConnectTimeout() == -1) {
                builder.setConnectTimeout(connectTimeout);
            }
            if (existConfig.getSocketTimeout() == 0 || existConfig.getSocketTimeout() == -1) {
                builder.setSocketTimeout(socketTimeout);
            }
            if (existConfig.getConnectionRequestTimeout() == 0 || existConfig.getConnectionRequestTimeout() == -1) {
                builder.setConnectionRequestTimeout(connectionRequestTimeout);
            }
            config = builder.build();
        }
        requestBase.setConfig(config);
    }

    /**
     * 使用默认httpClient发送post请求
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 请求返回结果
     */
    public String sendPost(String url, Map<String, String> params) {
        Charset encoding = StandardCharsets.UTF_8;
        return sendPost(url, params, encoding, null);
    }

    /**
     * 使用默认httpClient发送post请求
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 请求返回结果
     */
    public String sendPostWithHeaders(String url, Map<String, String> params, Header[] headers) {
        Charset encoding = StandardCharsets.UTF_8;
        return sendPost(url, params, headers, encoding, null);
    }

    /**
     * 使用默认httpClient发送post请求,此时参数增加一个超时时间设置的参数
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 请求返回结果
     */
    public String sendPost(String url, Map<String, String> params, Integer socketTimeOut) {
        Charset encoding = StandardCharsets.UTF_8;
        return sendPost(url, params, encoding, socketTimeOut);
    }

    /**
     * 发送 POST 请求（HTTP），JSON形式
     *
     * @param url
     * @param json json对象
     * @return
     */
    public String sendPost(String url, String json) {
        HttpRequestLog.log(url, json, null);
        HttpPost httpPost = new HttpPost(url);
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        return sendPost(httpPost);
    }

    public String sendPost(String url, String json, Integer socketTimeOut) {
        HttpRequestLog.log(url, json, null);
        HttpPost httpPost = new HttpPost(url);
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        httpPost.setConfig(RequestConfig.custom().
                setSocketTimeout(socketTimeOut == null ? 5000 : socketTimeOut).
                setConnectionRequestTimeout(100).
                build());
        return sendPost(httpPost);
    }

    /**
     * 上海联通专用，跳过对方线上ssl证书
     * 发送 POST 请求（HTTP），JSON形式
     */
    public String sendPost4ChinaUnicom(String url, String json) {
        HttpRequestLog.log(url, json, null);
        HttpPost httpPost = new HttpPost(url);
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        String resp = sendPostSslForChinaUnicom(httpPost);
        HttpRequestLog.log(httpPost.getURI().getPath(), httpPost.getEntity().toString(), resp);
        return resp;
    }

    /**
     * 发送 POST 请求（HTTP），JSON形式
     *
     * @param url
     * @param json json对象
     * @return
     */
    public String sendPostJsonWithHeaders(String url, String json, Header[] headers) {
        HttpPost httpPost = new HttpPost(url);
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        if (null != headers && headers.length > 0) {
            httpPost.setHeaders(headers);
        }
        httpPost.setEntity(stringEntity);
        return sendPost(httpPost);
    }

    public String sendPost(HttpPost httpPost) {
        String resp = getResponse(httpClient, httpPost);
        HttpRequestLog.log(httpPost.getURI().getPath(), httpPost.getEntity().toString(), resp);
        return resp;
    }

    private String getResponse(CloseableHttpClient httpClient, HttpRequestBase requestBase) {
        setTimeoutIfAbsent(requestBase);
        String resp = "";
        try (CloseableHttpResponse response = httpClient.execute(requestBase)) {
            //do something with resp
            HttpEntity entity = response.getEntity();
            if (entity == null) {
                return null;
            }
            int statusCode = response.getStatusLine().getStatusCode();
            resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
            if (statusCode != HttpStatus.SC_OK) {
                LOGGER.warn("request={},resp={}", requestBase, resp);
                return null;
            }
        } catch (IOException e) {
            LOGGER.warn("发送请求失败", e);
        }
        return resp;
    }

    /**
     * 发送post请求
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 请求结果
     */
    public String sendPostForm(String url, Map<String, Object> params, Integer socketTimeOut) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().
                setSocketTimeout(socketTimeOut == null ? 5000 : socketTimeOut).
                setConnectionRequestTimeout(100).
                build());

        if (params.size() > 0) {
            List<NameValuePair> formParams = new ArrayList<>();
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                formParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue() != null ? String.valueOf(entry.getValue()) : null));
            }
            UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(formParams, StandardCharsets.UTF_8);
            httpPost.setEntity(postEntity);
        }
        String resp = getResponse(httpClient, httpPost);
        HttpRequestLog.log(url, params.toString(), resp);
        return resp;
    }

    /**
     * 发送post请求
     *
     * @param url      请求地址
     * @param params   请求参数
     * @param encoding 编码
     * @return 请求结果
     */
    private String sendPost(String url, Map<String, String> params, Charset encoding, Integer socketTimeOut) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().
                setSocketTimeout(socketTimeOut == null ? 5000 : socketTimeOut).
                setConnectionRequestTimeout(100).
                build());
        if (params.size() > 0) {
            List<NameValuePair> formParams = new ArrayList<>();
            for (Map.Entry<String, String> entry : params.entrySet()) {
                formParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
            UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(formParams, encoding);
            httpPost.setEntity(postEntity);
        }
        String resp = getResponse(httpClient, httpPost);
        HttpRequestLog.log(url, params.toString(), resp);
        return resp;
    }

    /**
     * 发送post请求sendPost
     *
     * @param url      请求地址
     * @param params   请求参数
     * @param encoding 编码
     * @return 请求结果
     */
    private String sendPost(String url, Map<String, String> params, Header[] headers, Charset encoding, Integer socketTimeOut) {
        HttpRequestLog.log(url, params.toString(), null);
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().
                setSocketTimeout(socketTimeOut == null ? 5000 : socketTimeOut).
                setConnectionRequestTimeout(100).
                build());
        httpPost.setHeaders(headers);
        if (params.size() > 0) {
            List<NameValuePair> formParams = new ArrayList<>();
            for (Map.Entry<String, String> entry : params.entrySet()) {
                formParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
            UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(formParams, encoding);
            httpPost.setEntity(postEntity);
        }
        return getResponse(httpClient, httpPost);
    }

    /**
     * 发送 POST 请求（HTTP），JSON形式
     *
     * @param url
     * @param json json对象
     * @return
     */
    public String sendPostSsl(String url, String json) {
        HttpRequestLog.log(url, json, null);
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().setSocketTimeout(5000).setConnectionRequestTimeout(100).build());
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        return sendPostSsl(httpPost);
    }

    /**
     * 百信专用  忽略https 签名
     * 发送 POST 请求（HTTP），JSON形式
     *
     * @param url
     * @param json json对象
     * @return
     */
    public String sendPostSslWithHeaders(String url, String json, Header[] headers) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().setSocketTimeout(5000).setConnectionRequestTimeout(100).build());
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        if (headers != null && headers.length > 0) {
            httpPost.setHeaders(headers);
        }
        httpPost.setEntity(stringEntity);
        String resp = sendPostSslForBaiXin(httpPost);
        HttpRequestLog.log(url, json, resp);
        return resp;
    }


    /**
     * 使用默认httpClient发送post请求
     *
     * @param url    请求地址
     * @param params 请求参数
     * @return 请求返回结果
     */
    public String sendPostSsl(String url, Map<String, String> params) {
        Charset encoding = StandardCharsets.UTF_8;
        return sendPostSsl(url, params, encoding);
    }

    private String sendPostSsl(String url, Map<String, String> params, Charset encoding) {
        HttpRequestLog.log(url, params.toString(), null);
        HttpPost httpPost = new HttpPost(url);
        httpPost.setConfig(RequestConfig.custom().
                setSocketTimeout(5000).
                setConnectionRequestTimeout(100).
                build());
        if (params.size() > 0) {
            List<NameValuePair> formParams = new ArrayList<>();
            for (Map.Entry<String, String> entry : params.entrySet()) {
                formParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
            UrlEncodedFormEntity postEntity = new UrlEncodedFormEntity(formParams, encoding);
            httpPost.setEntity(postEntity);
        }
        return sendPostSsl(httpPost);
    }


    /**
     * 发送 SSL GET 请求（HTTPS）
     *
     * @param url API接口URL
     * @return
     */
    public String sendGet(String url) {
        HttpRequestLog.log(url, null, null);
        HttpGet httpGet = new HttpGet(url);
        httpGet.setConfig(RequestConfig.custom().
                setSocketTimeout(5000).
                setConnectionRequestTimeout(100).
                build());
        String httpStr = getResponse(httpClient, httpGet);
        HttpRequestLog.log(url, null, httpStr);
        return httpStr;
    }

    /**
     * 发送 SSL GET 请求（HTTPS）
     *
     * @param url API接口URL
     * @return
     */
    public String sendGet(String url, Integer timeout) {
        HttpRequestLog.log(url, null, null);
        HttpGet httpGet = new HttpGet(url);
        httpGet.setConfig(RequestConfig.custom().
                setSocketTimeout(null == timeout ? 5000 : timeout).
                setConnectionRequestTimeout(100).
                build());
        String httpStr = getResponse(httpClient, httpGet);
        HttpRequestLog.log(url, null, httpStr);
        return httpStr;
    }


    /**
     * 发送 SSL GET 请求（HTTPS）
     *
     * @param url API接口URL
     * @return
     */
    public String sendGet(String url, Header[] headers, Map<String, String> params) {
        String requestUrl = UrlUtils.assembleUrl(url, params);
        requestUrl = requestUrl.substring(0, requestUrl.length() - 1);
        HttpRequestLog.log(url, null, null);
        HttpGet httpGet = new HttpGet(requestUrl);
        httpGet.setHeaders(headers);
        httpGet.setConfig(RequestConfig.custom().
                setSocketTimeout(5000).
                setConnectionRequestTimeout(100).
                build());
        String httpStr = getResponse(httpClient, httpGet);
        HttpRequestLog.log(requestUrl, null, httpStr);
        return httpStr;
    }

    /**
     * 自定义设置请求头,发送 GET 请求（HTTPS）
     *
     * @return
     */
    public String sendHourGet(HttpGet httpGet) {
        httpGet.setConfig(RequestConfig.custom().
                setSocketTimeout(5000).
                setConnectionRequestTimeout(100).
                build());
        String resp = getResponse(httpClient, httpGet);
        HttpRequestLog.log(httpGet.getURI().getPath(), null, resp);
        return resp;
    }

    public String sendGet(String url, Map<String, String> params) {
        String requestUrl = UrlUtils.assembleUrl(url, params);
        requestUrl = requestUrl.substring(0, requestUrl.length() - 1);
        HttpRequestLog.log(url, null, null);
        return sendGet(requestUrl);
    }

    public String sendGet(String url, Map<String, String> params, Integer timeout) {
        String requestUrl = UrlUtils.assembleUrl(url, params);
        requestUrl = requestUrl.substring(0, requestUrl.length() - 1);
        HttpRequestLog.log(url, null, null);
        return sendGet(requestUrl, timeout);
    }

    private String sendPostSsl(HttpRequestBase requestBase) {
        String resp = "";
        CloseableHttpClient closeableHttpClient = HttpClients.custom().setSSLSocketFactory(createSslConnSocketFactory())
                .setConnectionManager(connMgr).setDefaultRequestConfig(requestConfig).build();
        try (CloseableHttpResponse response = closeableHttpClient.execute(requestBase)) {
            HttpEntity entity = response.getEntity();
            resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
        } catch (IOException e) {
            LOGGER.warn("发送post请求失败", e);
        }
        HttpRequestLog.log(requestBase.getURI().getQuery(), null, resp);
        return resp;
    }

    private String sendPostSslForBaiXin(HttpRequestBase requestBase) {
        String resp = "";
        try {
            SSLContext sslContext = SSLContexts.custom().loadTrustMaterial(null, (TrustStrategy) (x509Certificates, s) -> true).build();
            CloseableHttpClient closeableHttpClient = HttpClients.custom().setSSLContext(sslContext).
                    setSSLHostnameVerifier(new NoopHostnameVerifier()).build();
            try (CloseableHttpResponse response = closeableHttpClient.execute(requestBase)) {
                HttpEntity entity = response.getEntity();
                resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
            }
        } catch (Exception e) {
            LOGGER.warn("发送post请求失败", e);
        }
        HttpRequestLog.log(requestBase.getURI().getQuery(), null, resp);
        return resp;
    }


    private String sendPostSslForChinaUnicom(HttpRequestBase requestBase) {
        String resp = "";
        try {
            // 忽略 cookie 访问
            RequestConfig customizedRequestConfig = RequestConfig.custom()
                    .setSocketTimeout(5000)
                    .setConnectTimeout(5000)
                    .setConnectionRequestTimeout(100)
                    .setCookieSpec(CookieSpecs.IGNORE_COOKIES).build();
            SSLContext sslContext = SSLContexts.custom().loadTrustMaterial(null, (TrustStrategy) (x509Certificates, s) -> true).build();
            CloseableHttpClient closeableHttpClient = HttpClients.custom().setSSLContext(sslContext).setDefaultRequestConfig(customizedRequestConfig).
                    setSSLHostnameVerifier(new NoopHostnameVerifier()).build();
            try (CloseableHttpResponse response = closeableHttpClient.execute(requestBase)) {
                HttpEntity entity = response.getEntity();
                resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
            }
        } catch (Exception e) {
            LOGGER.warn("发送post请求失败,上海联通", e);
        }
        HttpRequestLog.log(requestBase.getURI().getQuery(), null, resp);
        return resp;
    }


    public String sendPostSsl(HttpRequestBase requestBase, ResponseHandler<String> responseHandler) {
        String resp = "";
        CloseableHttpClient closeableHttpClient = HttpClients.custom().setSSLSocketFactory(createSslConnSocketFactory())
                .setConnectionManager(connMgr).setDefaultRequestConfig(requestConfig).build();
        try {
            resp = closeableHttpClient.execute(requestBase, responseHandler);
        } catch (IOException e) {
            LOGGER.warn("发送post请求失败", e);
        }
        LOGGER.info("post请求结果为:{}.", resp);
        return resp;
    }

    /**
     * 创建SSL安全连接
     *
     * @return
     */
    private SSLConnectionSocketFactory createSslConnSocketFactory() {
        SSLConnectionSocketFactory sslsf = null;
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, (chain, authType) -> true).build();
            sslsf = new SSLConnectionSocketFactory(sslContext, (arg0, arg1) -> true);
        } catch (GeneralSecurityException e) {
            LOGGER.error("创建ssl连接发生异常", e);
        }
        return sslsf;
    }
}
