"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var PlayGame_1 = require("./PlayGame");
var Background_1 = require("./Background");
var Road_1 = require("./Road");
var Player_1 = require("./player/Player");
var utils_1 = require("./utils");
var Block_1 = require("./Block");
var ObjectPool = engine.ObjectPool;
exports.PoolName = 'block';
ObjectPool.registerPool(exports.PoolName, function () {
    return new Block_1.Block();
}, function (item, data) {
    item.reset(data);
});
var GameView = (function (_super) {
    __extends(GameView, _super);
    function GameView() {
        var _this = _super.call(this) || this;
        _this.offset = { x: -100, y: 320 };
        _this.worldFPS = 60;
        _this._viewport = {
            x: 0,
            y: 0,
        };
        return _this;
    }
    Object.defineProperty(GameView, "instance", {
        get: function () {
            if (this._instance == undefined) {
                this._instance = new GameView();
            }
            return this._instance;
        },
        enumerable: true,
        configurable: true
    });
    GameView.prototype.eventStart = function () {
        var _this = this;
        if (this.hasPreSetup) {
            return;
        }
        this.hasPreSetup = true;
        var background = this.background = new Background_1.Background();
        background.setup();
        this.addChild(background);
        var road = this.road = new Road_1.Road();
        road.y = utils_1.getStage().height / 2 + 310;
        this.addChild(road);
        this.road.preSetup();
        this.setupWorld();
        this.player.fade(0, false);
        this.viewport = 0;
        this.startWorld();
        this.reset();
        this.setup();
        var playHint = new engine.Label();
        playHint.fillColor = 'block';
        playHint.size = 30;
        playHint.text = '注意不要踩到线和陷阱哦，后果很严重哦~';
        playHint.x = (utils_1.getStage().width - playHint.width) / 2;
        playHint.y = utils_1.getStage().height / 2 + 500;
        this.addChild(playHint);
        this.playGame = new PlayGame_1.PlayGame();
        setTimeout(function () {
            _this.playGame.GameView = _this;
            _this.addChild(_this.playGame);
            _this.start();
        }, 500);
    };
    GameView.prototype.setup = function () {
        if (this.hasSetup) {
            return;
        }
        this.hasSetup = true;
        this.road.setup();
    };
    GameView.prototype.setupWorld = function () {
        var OTHER = Math.pow(2, 1), BODYPARTS = Math.pow(2, 2), GROUND = Math.pow(2, 3);
        var world = this.world = new p2.World({});
        world.sleepMode = p2.World.BODY_SLEEPING;
        var planeShape = new p2.Plane();
        var ground = new p2.Body({
            position: [0, 0],
        });
        ground.addShape(planeShape);
        planeShape.collisionGroup = GROUND;
        planeShape.collisionMask = BODYPARTS | OTHER;
        world.addBody(ground);
        var player = this.player = new Player_1.Player();
        player.setup({
            collisionGroup: BODYPARTS,
            collisionMask: GROUND | OTHER,
            world: world,
            ground: ground,
        });
        player.addEventListener('move', this.onMove, this);
        player.addEventListener('foot_down', this.onFootDown, this);
    };
    GameView.prototype.onTicker = function (dt) {
        var world = this.world;
        this.world.step(this.worldFPS / 1000);
        var l = world.bodies.length;
        for (var i = 0; i < l; i++) {
            var body = world.bodies[i];
            var display = body.displays ? body.displays[0] : null;
            if (display) {
                if (!this.contains(display)) {
                    this.addChild(display);
                }
                display.localUpdate(body.position[0] * utils_1.getfactor() + utils_1.getStage().width / 2 + this.offset.x - this.viewport, utils_1.getStage().height / 2 - body.position[1] * utils_1.getfactor() + this.offset.y, 360 - (body.angle + body.shapes[0].angle) * 180 / Math.PI, body.sleepState == p2.Body.SLEEPING);
            }
        }
        return false;
    };
    GameView.prototype.reset = function (revive) {
        if (revive === void 0) { revive = false; }
        this.background.reset(revive);
        this.road.reset(revive);
        this.player.reset(revive);
        this.player.fade(0);
        this.viewport = revive ? this.revivePos : 0;
    };
    GameView.prototype.start = function (revive) {
        if (revive === void 0) { revive = false; }
        this.road.start(revive);
        this.player.fade(1);
        this.playGame.start(revive);
    };
    GameView.prototype.startWorld = function () {
        if (this.playing) {
            return;
        }
        this.playing = true;
        engine.gameStage.addEventListener(engine.Event.ENTER_FRAME, this.onTicker, this);
    };
    GameView.prototype.stopWorld = function () {
        if (!this.playing) {
            return;
        }
        this.playing = false;
        engine.gameStage.removeEventListener(engine.Event.ENTER_FRAME, this.onTicker, this);
    };
    Object.defineProperty(GameView.prototype, "viewport", {
        get: function () {
            return this._viewport.x;
        },
        set: function (v) {
            this._viewport.x = v;
            this.updateViewport();
        },
        enumerable: true,
        configurable: true
    });
    GameView.prototype.updateViewport = function () {
        var _a = this._viewport, x = _a.x, y = _a.y;
        var pos = Math.max(0, x);
        this.background.setViewport(pos);
        this.road.setViewport(pos);
    };
    GameView.prototype.onMove = function (event) {
        var _a = event.data, pos = _a.pos, bodyPos = _a.bodyPos;
        this.viewport = pos;
        if (this.road.hitReward(bodyPos)) {
            this.dispatchEvent('scoreChange', {
                rewardScore: 100,
                passType: 10,
            }, true);
        }
    };
    GameView.prototype.down = function () {
        if (this.lock) {
            return;
        }
        this.player.down();
    };
    GameView.prototype.up = function () {
        if (this.lock) {
            return;
        }
        this.player.up();
    };
    GameView.prototype.onFootDown = function (event) {
        var _a = event.data, mileage = _a.mileage, pos = _a.pos;
        this.revivePos = pos;
        var _b = this.road.hitBlock(pos), target = _b.target, lastTarget = _b.lastTarget;
        this.road.playDust(pos);
        if (target) {
            this.lock = true;
            this.stopWorld();
            this.dispatchEvent(engine.Event.COMPLETE, null, true);
            target.playFadeOut();
            target.playEffect();
            if (target.type == 4) {
                this.player.setSkin('gray');
            }
            this.road.removeBlock(target);
            this.player.switchFace(target.type);
            if (target.type == 0) {
                this.player.fall();
            }
            this.lock = false;
            this.startWorld();
        }
        else {
            this.player.switchFace('normal');
            var score = Math.floor(mileage / 50 * 5);
            this.dispatchEvent('scoreChange', {
                score: score,
                passType: lastTarget ? lastTarget.type : -1,
            }, true);
        }
    };
    GameView.prototype.setWitchFace = function (type) {
        this.player.switchFace(type);
    };
    return GameView;
}(engine.Container));
exports.default = GameView;
