(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory(require('tslib')) :
	typeof define === 'function' && define.amd ? define(['tslib'], factory) :
	(global = global || self, global['walking-cat'] = factory(global.tslib));
}(this, (function (tslib) { 'use strict';

	function getTexture(uuid) {
	    return engine.Texture.from(getAssetByUUID(uuid).uuid);
	}
	function getTextureByName(name) {
	    return getTexture(engine.getAssetByName(name).uuid);
	}
	function getStage() {
	    return engine.gameStage.stage;
	}
	function getfactor() {
	    return 10;
	}
	//# sourceMappingURL=utils.js.map

	var props = {};
	function prepareProps() {
	    var metaProps = getProps();
	    engine.injectProp(props, metaProps);
	}
	function injectProps(p) {
	    engine.injectProp(props, p);
	}
	//# sourceMappingURL=props.js.map

	var ScoreLabel = (function (_super) {
	    tslib.__extends(ScoreLabel, _super);
	    function ScoreLabel() {
	        return _super !== null && _super.apply(this, arguments) || this;
	    }
	    ScoreLabel.prototype.updateScore = function (score, formatFun, animation) {
	        if (formatFun === void 0) { formatFun = null; }
	        if (animation === void 0) { animation = true; }
	        this.formatFun = formatFun;
	        this._value = score;
	        if (animation) {
	            engine.Tween.get(this, null, null, true)
	                .to({ temp: score }, 300);
	        }
	        else {
	            this.updateLocal(score);
	        }
	    };
	    Object.defineProperty(ScoreLabel.prototype, "score", {
	        get: function () {
	            return this._value;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Object.defineProperty(ScoreLabel.prototype, "temp", {
	        get: function () {
	            return this._temp;
	        },
	        set: function (v) {
	            this._temp = v;
	            this.updateLocal(v);
	        },
	        enumerable: true,
	        configurable: true
	    });
	    ScoreLabel.prototype.updateLocal = function (v) {
	        this.text = this.formatFun && this.formatFun(v) || v;
	    };
	    return ScoreLabel;
	}(engine.Label));
	//# sourceMappingURL=ScoreLabel.js.map

	var PlayGame = (function (_super) {
	    tslib.__extends(PlayGame, _super);
	    function PlayGame() {
	        var _this = _super !== null && _super.apply(this, arguments) || this;
	        _this.addlabelScore = null;
	        _this.formatScore = function (s) {
	            var t = (Math.ceil(s * 10) / 10).toString();
	            t += t.indexOf('.') < 0 ? '.0' : '';
	            return t;
	        };
	        return _this;
	    }
	    Object.defineProperty(PlayGame, "instance", {
	        get: function () {
	            if (this._instance == undefined) {
	                this._instance = new PlayGame();
	            }
	            return this._instance;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    PlayGame.prototype.start = function (revive) {
	        if (revive === void 0) { revive = false; }
	        if (!revive) {
	            this.normalScore = 0;
	            this.rewardScore = 0;
	            this.lastDialogTs = 0;
	            this.initTouch();
	            this.initLabScore();
	            this.initAddlabel();
	            this.initDialog();
	        }
	        this.registerEvent();
	    };
	    PlayGame.prototype.registerEvent = function () {
	        this.touchLayer.addEventListener(engine.MouseEvent.MOUSE_DOWN, this.onMouseDown, this);
	        this.touchLayer.addEventListener(engine.MouseEvent.MOUSE_UP, this.onMouseUp, this);
	        this.GameView.addEventListener(engine.Event.COMPLETE, this.onGameComplete, this);
	        this.GameView.addEventListener('scoreChange', this.onScoreChange, this);
	    };
	    PlayGame.prototype.initTouch = function () {
	        this.touchLayer = new engine.Rect();
	        this.addChild(this.touchLayer);
	        this.touchLayer.width = getStage().width;
	        this.touchLayer.height = getStage().height;
	        this.touchLayer.x = 0;
	        this.touchLayer.y = 0;
	        this.touchLayer.alpha = 0;
	    };
	    PlayGame.prototype.removeListener = function () {
	        this.touchLayer.removeEventListener(engine.MouseEvent.MOUSE_DOWN, this.onMouseDown, this);
	        this.touchLayer.removeEventListener(engine.MouseEvent.MOUSE_UP, this.onMouseUp, this);
	        this.GameView.removeEventListener(engine.Event.COMPLETE, this.onGameComplete, this);
	        this.GameView.removeEventListener('scoreChange', this.onScoreChange, this);
	    };
	    PlayGame.prototype.initLabScore = function () {
	        this.labScore = new ScoreLabel();
	        this.labScore.visible = props.showScore;
	        this.labScore.updateScore(0, this.formatScore, false);
	        this.labScore.fillColor = 'yellow';
	        this.labScore.size = 46;
	        this.labScore.x = 610;
	        engine.Tween.get(this.labScore, { loop: false })
	            .set({ y: -100 })
	            .wait(300)
	            .to({ y: 40 }, 200, engine.Ease.backOut);
	        this.addChild(this.labScore);
	    };
	    PlayGame.prototype.initAddlabel = function () {
	        this.addlabelScore = new engine.Label();
	        this.addlabelScore.fillColor = 'yellow';
	        this.addlabelScore.size = 40;
	        this.addlabelScore.text = '';
	        this.addlabelScore.x = 378;
	        this.addlabelScore.y = 582;
	        this.addlabelScore.visible = false;
	        this.addChild(this.addlabelScore);
	    };
	    PlayGame.prototype.initDialog = function () {
	        this.dialog = new engine.Image();
	        this.dialog.x = 390;
	        this.dialog.y = 323;
	        this.addChild(this.dialog);
	    };
	    PlayGame.prototype.onMouseDown = function () {
	        this.GameView.down();
	    };
	    PlayGame.prototype.onMouseUp = function () {
	        this.GameView.up();
	    };
	    PlayGame.prototype.onGameComplete = function () {
	        console.log('死亡');
	        this.removeListener();
	        var score = this.getScore();
	        engine.globalEvent.dispatchEvent('walking-cat-game-end', { score: score });
	    };
	    PlayGame.prototype.getScore = function (round) {
	        if (round === void 0) { round = false; }
	        var score = (this.normalScore + this.rewardScore) / 10;
	        score *= (props.scoreTimes || 1);
	        if (round) {
	            return Math.round(score);
	        }
	        return score;
	    };
	    PlayGame.prototype.onScoreChange = function (event) {
	        var _a = event.data, score = _a.score, passType = _a.passType, rewardScore = _a.rewardScore;
	        var scoreAdd;
	        if (rewardScore) {
	            this.rewardScore += rewardScore;
	            scoreAdd = rewardScore;
	        }
	        else {
	            scoreAdd = score - this.normalScore;
	            this.normalScore = score;
	        }
	        var curScore = this.getScore();
	        this.labScore.updateScore(curScore, this.formatScore);
	        if (passType > 0 && passType <= 4) {
	            var now = Date.now();
	            if (now - this.lastDialogTs > 15000) {
	                this.play(1);
	                this.lastDialogTs = now;
	            }
	        }
	        else if (passType == 10) {
	            this.play(0);
	            this.lastDialogTs = Date.now();
	            this.GameView.setWitchFace('happy');
	        }
	        if (scoreAdd > 0) {
	            var labScoreAdd_1 = this.addlabelScore;
	            labScoreAdd_1.visible = props.showScore;
	            labScoreAdd_1.alpha = 0;
	            labScoreAdd_1.verticalCenter = 0;
	            labScoreAdd_1.text = '+' + (scoreAdd / 10 * (props.scoreTimes || 1));
	            engine.Tween.get(labScoreAdd_1, null, null, true)
	                .to({ verticalCenter: -50, alpha: 1, }, 200, engine.Ease.cubicIn)
	                .wait(300)
	                .to({ verticalCenter: -100, alpha: 0, }, 100, engine.Ease.cubicOut)
	                .call(function () {
	                labScoreAdd_1.visible = false;
	            }, this);
	        }
	        engine.globalEvent.dispatchEvent('walking-cat-game-changScore', { score: curScore });
	    };
	    PlayGame.prototype.play = function (id) {
	        var img = this.dialog;
	        img.texture = getTextureByName('dialog_' + id);
	        img.scaleX = img.scaleY = 0;
	        img.alpha = 1;
	        img.visible = true;
	        engine.Tween.get(img, null, null, true)
	            .to({
	            scaleX: 1, scaleY: 1
	        }, 500, engine.Ease.backOut)
	            .wait(500)
	            .to({
	            alpha: 0,
	        }, 300)
	            .call(function () {
	            img.visible = false;
	        });
	    };
	    PlayGame.prototype.revive = function () {
	        this.GameView.reset(true);
	        this.GameView.start(true);
	    };
	    return PlayGame;
	}(engine.Container));
	//# sourceMappingURL=PlayGame.js.map

	var LoopComponent = (function (_super) {
	    tslib.__extends(LoopComponent, _super);
	    function LoopComponent() {
	        var _this = _super.call(this) || this;
	        _this.onceInfo = { width: 0, height: 0, count: 0 };
	        _this.loopInfo = { width: 0, height: 0, count: 0 };
	        _this.parts = [];
	        return _this;
	    }
	    LoopComponent.prototype.setup = function (parts, info) {
	        var _this = this;
	        parts.forEach(function (item, index) {
	            var part;
	            if (typeof item == 'string') {
	                part = new engine.Sprite(getTextureByName(item));
	            }
	            else if (item instanceof engine.Texture) {
	                part = new engine.Sprite(item);
	            }
	            else {
	                part = item;
	            }
	            _this.addChild(part);
	            _this.parts.push(part);
	            if (index == 0) {
	                info.width = part.width;
	                info.height = part.height;
	            }
	        });
	        info.count = parts.length;
	        this.setViewport(0, 0);
	    };
	    LoopComponent.prototype.setupOnce = function (resArr) {
	        this.setup(resArr, this.onceInfo);
	    };
	    LoopComponent.prototype.setupLoop = function (resArr) {
	        this.setup(resArr, this.loopInfo);
	    };
	    LoopComponent.prototype.setViewport = function (x, y) {
	        if (y === void 0) { y = 0; }
	        var sizeField = y == 0 ? 'width' : 'height';
	        var posField = y == 0 ? 'x' : 'y';
	        var pos = y == 0 ? x : y;
	        var index;
	        var onceSize = this.onceInfo[sizeField];
	        var onceCount = this.onceInfo.count;
	        var loopCount = this.loopInfo.count;
	        var loopSize = this.loopInfo[sizeField];
	        index = onceSize == 0 ? -1 : Math.floor(pos / onceSize);
	        if (index < 0 || index >= onceCount) {
	            var loopIndex = Math.floor((pos - onceCount * onceSize) / this.loopInfo[sizeField]);
	            index = onceCount + loopIndex;
	        }
	        var formerIndex = index < onceCount ? index : (index) % loopCount + onceCount;
	        var latterIndex = index < onceCount ? index + 1 : (index - onceCount + 1) % loopCount + onceCount;
	        var former = this.parts[formerIndex];
	        var latter = this.parts[latterIndex];
	        if (!former || !latter) {
	            console.log(formerIndex, latterIndex);
	        }
	        former[posField] = -(pos - loopSize * (index < onceCount ? index : index - onceCount) - (index < onceCount ? 0 : onceCount * onceSize));
	        latter[posField] = former.x + (index < onceCount ? onceSize : loopSize);
	    };
	    return LoopComponent;
	}(engine.Container));
	//# sourceMappingURL=LoopComponent.js.map

	var Background = (function (_super) {
	    tslib.__extends(Background, _super);
	    function Background() {
	        var _this = _super !== null && _super.apply(this, arguments) || this;
	        _this.partRes = ['bg_far', 'bg_mid', 'bg_near'];
	        _this.speeds = [0.05, 0.5, 1];
	        return _this;
	    }
	    Background.prototype.setup = function () {
	        var bottomBg = new engine.Rect();
	        bottomBg.x = 0;
	        bottomBg.y = 0;
	        bottomBg.width = getStage().width;
	        bottomBg.height = 0;
	        bottomBg.fillColor = 0x75c9f5;
	        this.addChild(bottomBg);
	        var parts = [];
	        for (var i = 0; i < 3; i++) {
	            var part = new LoopComponent();
	            part.setupLoop([
	                this.partRes[i],
	                this.partRes[i],
	            ]);
	            parts.push(part);
	            this.addChild(part);
	        }
	        var height = getStage().height;
	        parts[0].y = (height - parts[0].height) / 2 - 400;
	        parts[1].y = (height - parts[1].height) / 2 - 120;
	        parts[2].y = (height - parts[2].height) / 2 + 300;
	        bottomBg.y = parts[2].y;
	        bottomBg.height = height - parts[2].y;
	        var moon = new engine.Sprite(getTextureByName('moon'));
	        moon.x = 100;
	        moon.y = 100;
	        this.addChild(moon);
	    };
	    Background.prototype.setViewport = function (pos) {
	        for (var i = 0; i < 3; i++) {
	            var part = this.getChildAt(i + 1);
	            var speed = this.speeds[i];
	            part.setViewport(pos * speed);
	        }
	    };
	    Background.prototype.reset = function (revive) {
	    };
	    return Background;
	}(engine.Container));
	//# sourceMappingURL=Background.js.map

	var localWave = (function () {
	    function localWave() {
	        this._oldProperties = {};
	        this._t = 0;
	    }
	    localWave.prototype.init = function (target, duration, calProps, loop, autoPlay, reverse, delay, offset) {
	        if (calProps === void 0) { calProps = null; }
	        if (loop === void 0) { loop = 0; }
	        if (autoPlay === void 0) { autoPlay = true; }
	        if (reverse === void 0) { reverse = false; }
	        if (delay === void 0) { delay = 0; }
	        if (offset === void 0) { offset = 0; }
	        this.target = target;
	        this._calProps = calProps ? calProps : localWave.round;
	        this.duration = duration;
	        this.loop = loop;
	        this.reverse = reverse;
	        this.delay = delay;
	        this.offset = offset;
	        this.updateRegisterPos();
	        if (autoPlay) {
	            this.play();
	        }
	    };
	    localWave.prototype.updateRegisterPos = function () {
	        this._oldProperties.x = this.target.x;
	        this._oldProperties.y = this.target.y;
	        this._oldProperties.scaleX = this.target.scaleX;
	        this._oldProperties.scaleY = this.target.scaleY;
	        this._oldProperties.skewX = this.target.skewX;
	        this._oldProperties.skewY = this.target.skewY;
	        this._oldProperties.rotation = this.target.rotation;
	        this._oldProperties.alpha = this.target.alpha;
	    };
	    localWave.prototype.play = function () {
	        if (this._tween) {
	            return this._tween;
	        }
	        this._count = 0;
	        return this._playStep();
	    };
	    localWave.prototype._playStep = function () {
	        if (this.loop > 0 && this._count >= this.loop) {
	            this.stop();
	            return;
	        }
	        this._count++;
	        this.t = this.reverse ? Math.PI * 2 : 0;
	        this._tween = engine.Tween.get(this);
	        this._tween.wait(this.delay).to({ t: this.reverse ? 0 : Math.PI * 2 }, this.duration).call(this._playStep, this);
	        return this._tween;
	    };
	    Object.defineProperty(localWave.prototype, "t", {
	        get: function () {
	            return this._t;
	        },
	        set: function (value) {
	            if (!this.target.stage) {
	                return;
	            }
	            this._t = value;
	            var props = this._calProps.call(this, this._t + this.offset);
	            if (props.hasOwnProperty('x')) {
	                this.target.x = (props.x || 0) + this._oldProperties.x;
	            }
	            if (props.hasOwnProperty('y')) {
	                this.target.y = (props.y || 0) + this._oldProperties.y;
	            }
	            if (props.hasOwnProperty('sx')) {
	                this.target.scaleX = props.sx;
	            }
	            if (props.hasOwnProperty('sy')) {
	                this.target.scaleY = props.sy;
	            }
	            if (props.hasOwnProperty('skewX')) {
	                this.target.skewX = props.skewX;
	            }
	            if (props.hasOwnProperty('skewY')) {
	                this.target.skewY = props.skewY;
	            }
	            if (props.hasOwnProperty('r')) {
	                this.target.rotation = props.r;
	            }
	            if (props.hasOwnProperty('alpha')) {
	                this.target.alpha = props.alpha;
	            }
	        },
	        enumerable: true,
	        configurable: true
	    });
	    localWave.prototype.stop = function (recovery, animation, duration) {
	        if (recovery === void 0) { recovery = false; }
	        if (duration === void 0) { duration = 1000; }
	        if (!this._tween) {
	            return;
	        }
	        engine.Tween.removeTweens(this);
	        if (recovery) {
	            engine.Tween.get(this.target).to(this._oldProperties, duration);
	        }
	        this._tween = null;
	    };
	    Object.defineProperty(localWave.prototype, "playing", {
	        get: function () {
	            return this._tween != null;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    localWave.round = function (h, t) {
	        return { x: Math.cos(t) * h, y: Math.sin(t) * h };
	    };
	    localWave.cos = function (h, t) {
	        return { x: Math.cos(t) * h, y: 0 };
	    };
	    localWave.sin = function (h, t) {
	        h = h || 1;
	        return { x: 0, y: Math.sin(t) * h };
	    };
	    localWave.rotate = function (t) {
	        return { r: 360 * t / Math.PI / 2 };
	    };
	    localWave.shake = function (angle, count, t) {
	        return { r: Math.sin(t * count) * angle };
	    };
	    localWave.breath = function (scale, t) {
	        return { sx: Math.sin(t) * scale + 1, sy: -Math.sin(t + Math.PI / 4) * scale + 1 };
	    };
	    localWave.zoom = function (scale, t) {
	        scale = scale || 0.1;
	        return { sx: Math.sin(t) * scale + 1, sy: Math.sin(t) * scale + 1 };
	    };
	    localWave.fade = function (base, t) {
	        return { alpha: (Math.sin(t) + 1) * 0.5 + base };
	    };
	    return localWave;
	}());
	//# sourceMappingURL=localWave.js.map

	var Block = (function (_super) {
	    tslib.__extends(Block, _super);
	    function Block() {
	        var _this = _super.call(this) || this;
	        _this.bmp = new engine.Sprite();
	        _this.addChild(_this.bmp = new engine.Sprite());
	        return _this;
	    }
	    Block.prototype.reset = function (_a) {
	        var type = _a.type;
	        this.type = type;
	        var bmp = this.bmp;
	        bmp.alpha = 1;
	        bmp.texture = this.getBlock();
	        var _b = Block.configs[type], width = _b.width, off = _b.off;
	        bmp.x = -(off + width / 2);
	        bmp.y = -(bmp.height / 2);
	    };
	    Block.prototype.getBlock = function () {
	        return getTextureByName('block_' + this.type);
	    };
	    Block.prototype.playEffect = function () {
	    };
	    Block.prototype.playFadeOut = function () {
	        return this.getTweenPromise(engine.Tween.get(this.bmp)
	            .to({ alpha: 0 }, 100)
	            .to({ alpha: 1 }, 100)
	            .to({ alpha: 0 }, 100));
	    };
	    Block.prototype.onEffectComplete = function (event) {
	        this.effectResolve && this.effectResolve();
	        this.effectResolve = null;
	    };
	    Block.prototype.getTweenPromise = function (tween) {
	        return new Promise(function (resolve) {
	            tween.call(resolve);
	        });
	    };
	    Block.configs = [
	        { ratio: 0.6, width: 15, off: 32, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0.1, width: 10, off: 18, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0.15, width: 30, off: 32, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0.1, width: 86, off: 28, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0.5, width: 70, off: 20, effect: '', effect_offset: { x: 0, y: -20 } },
	        { ratio: 0, width: 0, off: 0, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0, width: 0, off: 0, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0, width: 0, off: 0, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0, width: 0, off: 0, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0, width: 0, off: 0, effect: '', effect_offset: { x: 0, y: 0 } },
	        { ratio: 0, width: 123, off: 23, effect: '', effect_offset: { x: 0, y: 0 } },
	    ];
	    return Block;
	}(engine.Container));
	//# sourceMappingURL=Block.js.map

	var ObjectPool = engine.ObjectPool;
	var Road = (function (_super) {
	    tslib.__extends(Road, _super);
	    function Road() {
	        var _this = _super !== null && _super.apply(this, arguments) || this;
	        _this.dustIndex = 0;
	        _this.index = 0;
	        _this.minDistance = parseInt(props.blockMinDistance) || 100;
	        _this.maxDistance = parseInt(props.blockMaxDistance) || 300;
	        _this.basePos = getStage().width / 2 - 100;
	        _this.padding = 500;
	        return _this;
	    }
	    Road.prototype.preSetup = function () {
	        this.addChild(this.container = new engine.Container());
	        var flag = this.flag = new engine.Sprite(getTextureByName('flag_record'));
	        this.anchorRate(flag, 0.5, 1);
	        flag.y = -230;
	        flag.x = this.transScoreToFlagPos();
	        flag.visible = flag.x > 0;
	        this.addChild(flag);
	    };
	    Road.prototype.setup = function () {
	        this.dustNode = new engine.Container();
	        this.dustSprite = new engine.Sprite();
	        this.dustSprite.visible = false;
	        this.dustNode.y = -1000;
	        this.dustNode.x = -450;
	        this.dustNode.addChild(this.dustSprite);
	        this.addChild(this.dustNode);
	        var reward = this.reward = new engine.Sprite();
	        reward.visible = false;
	        reward.y = -500;
	        this.addChild(reward);
	        this.aniReward = new localWave();
	        this.aniReward.init(reward, 2000, localWave.sin.bind(null, 10));
	    };
	    Road.prototype.start = function (revive) {
	        if (revive === void 0) { revive = false; }
	        if (revive) ;
	        else {
	            this.rewardIndex = 0;
	            this.showReward();
	        }
	    };
	    Road.prototype.reset = function (revive) {
	        if (revive === void 0) { revive = false; }
	        if (!revive) {
	            this.index = 0;
	            this.lastPos = 0;
	            this.clear();
	            this.fill();
	            this.flag.x = this.transScoreToFlagPos();
	            this.flag.visible = this.flag.x > 0;
	        }
	    };
	    Road.prototype.anchorRate = function (target, rx, ry, width, height, resetPos) {
	        if (width === void 0) { width = 0; }
	        if (height === void 0) { height = 0; }
	        if (resetPos === void 0) { resetPos = true; }
	        if (width == 0) {
	            width = target.width;
	        }
	        if (height == 0) {
	            height = target.height;
	        }
	        if (resetPos) {
	            if (rx == 0) {
	                target.x -= target.anchorOffsetX;
	            }
	            if (ry == 0) {
	                target.y -= target.anchorOffsetY;
	            }
	        }
	        target.anchorOffsetX = width * rx;
	        target.anchorOffsetY = height * ry;
	        if (resetPos) {
	            if (rx > 0) {
	                target.x += target.anchorOffsetX;
	            }
	            if (ry > 0) {
	                target.y += target.anchorOffsetY;
	            }
	        }
	    };
	    Road.prototype.transScoreToFlagPos = function () {
	        return -1;
	    };
	    Road.prototype.clear = function () {
	        while (this.container.children.length > 0) {
	            this.removeBlockAt(0);
	        }
	    };
	    Road.prototype.addBlock = function () {
	        var type = 0, v;
	        if (this.index > 10) {
	            do {
	                var rv = Math.random();
	                v = 0;
	                for (var i = 0; i < 5; i++) {
	                    v += Block.configs[i].ratio;
	                    if (rv < v) {
	                        type = i;
	                        break;
	                    }
	                }
	            } while (this.lastBlockType == type);
	            this.lastBlockType = type;
	        }
	        var block = ObjectPool.getObject('block', { type: type });
	        var width = Block.configs[type].width;
	        var distance = this.index == 0 ?
	            this.maxDistance :
	            this.makeRandomFloat(this.maxDistance, this.minDistance);
	        var pos = this.lastPos + distance;
	        block.x = pos;
	        this.container.addChild(block);
	        this.lastPos = pos + width;
	        this.index++;
	    };
	    Road.prototype.removeBlockAt = function (index) {
	        var block = this.container.removeChildAt(index);
	        ObjectPool.recycleObject('block', block);
	    };
	    Road.prototype.removeBlock = function (block) {
	        this.container.removeChild(block);
	        ObjectPool.recycleObject('block', block);
	    };
	    Road.prototype.fill = function () {
	        if (isNaN(this.lastPos)) {
	            return;
	        }
	        while (true) {
	            if (this.lastPos + this.x > getStage().width + this.padding) {
	                break;
	            }
	            this.addBlock();
	        }
	    };
	    Road.prototype.makeRandomFloat = function (max, min) {
	        if (min === void 0) { min = 0; }
	        return Math.random() * (max - min) + min;
	    };
	    Road.prototype.setViewport = function (pos) {
	        this.x = this.basePos - pos;
	        while (this.container.children.length > 0) {
	            if (this.container.getChildAt(0).x >= pos - this.padding) {
	                break;
	            }
	            this.removeBlockAt(0);
	        }
	        this.fill();
	    };
	    Road.prototype.showReward = function (type) {
	        this.rewardIndex++;
	        var pos = this.rewardIndex * getStage().width * 3;
	        var reward = this.reward;
	        reward.visible = true;
	        reward.alpha = 1;
	        reward.texture = getTextureByName('menu_block_10');
	        reward.x = pos;
	        this.aniReward.updateRegisterPos();
	    };
	    Road.prototype.hideReward = function () {
	        var _this = this;
	        var reward = this.reward;
	        this.tweenReward = engine.Tween.get(reward)
	            .to({ alpha: 0 }, 200)
	            .call(function () {
	            _this.tweenReward = null;
	            _this.showReward();
	        });
	    };
	    Road.prototype.hitReward = function (pos) {
	        if (this.reward.visible && pos >= this.reward.x && !this.tweenReward) {
	            this.hideReward();
	            return true;
	        }
	    };
	    Road.prototype.playDust = function (pos) {
	        this.dustSprite.x = pos;
	        this.dustSprite.visible = true;
	        this.addEventListener(engine.Event.ENTER_FRAME, this.onEnterFrame, this);
	    };
	    Road.prototype.onEnterFrame = function () {
	        if (this.dustIndex < 12) {
	            this.dustSprite.texture = getTextureByName('dust_' + this.dustIndex);
	            this.dustIndex++;
	        }
	        else {
	            this.dustIndex = 0;
	            this.dustSprite.visible = false;
	            this.removeEventListener(engine.Event.ENTER_FRAME, this.onEnterFrame, this);
	        }
	    };
	    Road.prototype.hitBlock = function (pos) {
	        var _this = this;
	        var target, lastTarget;
	        var foot = { a: pos - 84 / 2, b: pos + 84 / 2 };
	        this.container.children.some(function (block, i) {
	            var width = Block.configs[block.type].width;
	            var line = { a: block.x - width / 2, b: block.x + width / 2 };
	            if (_this.lineHitTest(foot, line)) {
	                target = block;
	                return true;
	            }
	        });
	        if (!target) {
	            for (var i = this.container.children.length; i > 0; i--) {
	                var block = this.container.getChildAt(i - 1);
	                if (block.x < pos) {
	                    lastTarget = block;
	                    break;
	                }
	            }
	        }
	        return {
	            target: target,
	            lastTarget: lastTarget,
	        };
	    };
	    Road.prototype.lineHitTest = function (l0, l1) {
	        return l0.a < l1.b && l0.b > l1.a;
	    };
	    return Road;
	}(engine.Container));
	//# sourceMappingURL=Road.js.map

	var factor = 10;
	var Head = (function (_super) {
	    tslib.__extends(Head, _super);
	    function Head() {
	        var _this = _super.call(this) || this;
	        _this.imgFace = new engine.Sprite(getTextureByName('head_0'));
	        _this.addChild(_this.imgFace);
	        _this.imgFace.anchorTexture.set(107 / _this.imgFace.width, 155 / _this.imgFace.height);
	        _this.imgFace.y = 65;
	        _this._headType = '';
	        return _this;
	    }
	    Head.prototype.localUpdate = function (x, y, rotation, sleeping) {
	        this.x = x;
	        this.y = y;
	        this.rotation = rotation;
	    };
	    Object.defineProperty(Head.prototype, "fWidth", {
	        get: function () {
	            return this.width / factor;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Object.defineProperty(Head.prototype, "fHeight", {
	        get: function () {
	            return this.height / factor;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Head.prototype.switchFace = function (type, playAni) {
	        if (type == this._headType) {
	            return;
	        }
	        var headName = '';
	        if (type == 'normal') {
	            headName = 'head_0';
	        }
	        else if (type == 'happy') {
	            headName = 'head_1';
	        }
	        else if (type == 1 || type == 0 || type == 2) {
	            headName = 'head_2';
	        }
	        else if (type == 3) {
	            headName = 'head_3';
	        }
	        else if (type == 4) {
	            headName = 'head_4';
	        }
	        this._headType = type;
	        this.imgFace.texture = getTextureByName(headName);
	    };
	    return Head;
	}(engine.Container));
	//# sourceMappingURL=Head.js.map

	var PlayerPart = (function (_super) {
	    tslib.__extends(PlayerPart, _super);
	    function PlayerPart(name) {
	        var _this = _super.call(this) || this;
	        _this.changeRes(name);
	        return _this;
	    }
	    PlayerPart.prototype.changeRes = function (name) {
	        this.name = name;
	        this.texture = getTextureByName('body_normal_' + name);
	    };
	    PlayerPart.prototype.setSkin = function (name) {
	        this.texture = getTextureByName('body_' + name + '_' + this.name);
	    };
	    PlayerPart.prototype.localUpdate = function (x, y, rotation, sleeping) {
	        this.x = x;
	        this.y = y;
	        this.rotation = rotation;
	    };
	    Object.defineProperty(PlayerPart.prototype, "fSize", {
	        get: function () {
	            return { width: this.fWidth, height: this.fHeight };
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Object.defineProperty(PlayerPart.prototype, "fWidth", {
	        get: function () {
	            return this.width / getfactor();
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Object.defineProperty(PlayerPart.prototype, "fHeight", {
	        get: function () {
	            return this.height / getfactor();
	        },
	        enumerable: true,
	        configurable: true
	    });
	    PlayerPart.prototype.reset = function () {
	    };
	    return PlayerPart;
	}(engine.Sprite));
	//# sourceMappingURL=PlayerPart.js.map

	var PlayerStat;
	(function (PlayerStat) {
	    PlayerStat[PlayerStat["RELEASE"] = 0] = "RELEASE";
	    PlayerStat[PlayerStat["RISING"] = 1] = "RISING";
	    PlayerStat[PlayerStat["FALL"] = 2] = "FALL";
	})(PlayerStat || (PlayerStat = {}));
	var Player = (function (_super) {
	    tslib.__extends(Player, _super);
	    function Player() {
	        var _this = _super !== null && _super.apply(this, arguments) || this;
	        _this.partMass = 0.1;
	        _this.footMass = 0.1;
	        _this.shouldersDistance = 50 / getfactor();
	        _this.speed = 30;
	        _this.skin = '';
	        _this.originPos = {};
	        _this.safePos = 0;
	        _this.onPostStep = function (e) {
	            var side = _this.currentSide;
	            var footHang = _this.foots[_this.currentSide];
	            var footFall = _this.foots[_this.anotherSide];
	            switch (_this.state) {
	                case PlayerStat.FALL:
	                    break;
	                case PlayerStat.RISING:
	                    var footOffset = footHang.position[0] - footFall.position[0];
	                    var speed = _this.speed;
	                    _this.upperLegs[side].angularVelocity = 10;
	                    _this.upperLegs[side].velocity = [
	                        footOffset < 0 ? speed - 10 : speed,
	                        0
	                    ];
	                    _this.lowerLegs[side].velocity = [
	                        footOffset < 0 ? speed - 10 : speed,
	                        footOffset < 0 ? 3 : 2
	                    ];
	                    _this.head.velocity = [speed * 0.5, 20];
	                    _this.dispatchEvent('move', {
	                        pos: (_this.leftFoot.position[0] + _this.rightFoot.position[0]) / 2 * getfactor(),
	                        bodyPos: _this.body.position[0] * getfactor(),
	                    }, true);
	                    if (footOffset > 50) {
	                        _this.up();
	                    }
	                    break;
	                case PlayerStat.RELEASE:
	                    _this.head.velocity = [0, 30];
	                    break;
	            }
	        };
	        _this.onFootDown = function () {
	            var pos = _this.foots[_this.currentSide].position[0] * getfactor();
	            var mileage = Math.max(_this.foots[0].position[0], _this.foots[1].position[0]) * getfactor();
	            _this.safePos = _this.foots[_this.currentSide].position[0];
	            _this.dispatchEvent('foot_down', {
	                mileage: mileage,
	                pos: pos,
	            }, true);
	        };
	        return _this;
	    }
	    Player.prototype.setup = function (options) {
	        var collisionGroup = options.collisionGroup, collisionMask = options.collisionMask, world = options.world, ground = options.ground;
	        this.world = world;
	        world.on('postStep', this.onPostStep);
	        var bodyPartShapes = [];
	        var leftFootDisplay = this.createDisplay('foot');
	        var lowerLeftLegDisplay = this.createDisplay('lower_left_leg');
	        var upperLeftLegDisplay = this.createDisplay('upper_left_leg');
	        var rightFootDisplay = this.createDisplay('foot');
	        var lowerRightLegDisplay = this.createDisplay('lower_right_leg');
	        var upperRightLegDisplay = this.createDisplay('upper_right_leg');
	        var bodyDisplay = this.createDisplay('body', { x: 50, y: 0 });
	        var headDisplay = new Head();
	        var upperLeftArmDisplay = this.createDisplay('upper_left_arm');
	        var lowerLeftArmDisplay = this.createDisplay('lower_left_arm', { x: -20, y: 0 });
	        var upperRightArmDisplay = this.createDisplay('upper_right_arm');
	        var lowerRightArmDisplay = this.createDisplay('lower_right_arm', { x: -39, y: 0 });
	        var leftFootShape = new p2.Box(leftFootDisplay.fSize);
	        bodyPartShapes.push(leftFootShape);
	        this.originPos['leftFoot'] = [
	            -this.shouldersDistance / 2,
	            leftFootDisplay.fHeight / 2
	        ];
	        var leftFoot = this.leftFoot = new p2.Body({
	            allowSleep: false,
	            mass: this.footMass,
	            position: this.originPos['leftFoot'],
	        });
	        leftFoot.displays = [leftFootDisplay];
	        leftFoot.addShape(leftFootShape);
	        var lowerLeftLegShape = new p2.Box(lowerLeftLegDisplay.fSize);
	        bodyPartShapes.push(lowerLeftLegShape);
	        this.originPos['lowerLeftLeg'] = [
	            -this.shouldersDistance / 2,
	            leftFoot.position[1] + lowerLeftLegDisplay.fHeight / 2 + 5 / getfactor()
	        ];
	        var lowerLeftLeg = this.lowerLeftLeg = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['lowerLeftLeg'],
	        });
	        lowerLeftLeg.displays = [lowerLeftLegDisplay];
	        lowerLeftLeg.addShape(lowerLeftLegShape);
	        var leftAnkleJoint = new p2.RevoluteConstraint(lowerLeftLeg, leftFoot, {
	            localPivotA: [0, -lowerLeftLegDisplay.fHeight / 2 + 5 / getfactor()],
	            localPivotB: [-leftFootDisplay.fWidth / 2 + 18 / getfactor(), 0],
	        });
	        leftAnkleJoint.setLimits(-Math.PI / 3, Math.PI / 3);
	        var leftFootGroundCons = this.leftFootGroundCons = new p2.RevoluteConstraint(leftFoot, ground, {
	            localPivotA: [0, -leftFootDisplay.fHeight / 2],
	            localPivotB: [leftFoot.position[0], 0],
	        });
	        leftFootGroundCons.setLimits(0, 0);
	        var upperLeftLegShape = new p2.Box(upperLeftLegDisplay.fSize);
	        bodyPartShapes.push(upperLeftLegShape);
	        this.originPos['upperLeftLeg'] = [
	            -this.shouldersDistance / 2,
	            lowerLeftLeg.position[1] + lowerLeftLegDisplay.fHeight / 2 + upperLeftLegDisplay.fHeight / 2 - 40 / getfactor()
	        ];
	        var upperLeftLeg = this.upperLeftLeg = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['upperLeftLeg'],
	        });
	        upperLeftLeg.displays = [upperLeftLegDisplay];
	        upperLeftLeg.addShape(upperLeftLegShape);
	        var leftKneeJoint = new p2.RevoluteConstraint(upperLeftLeg, lowerLeftLeg, {
	            localPivotA: [0, -upperLeftLegDisplay.fHeight / 2 + 20 / getfactor()],
	            localPivotB: [0, lowerLeftLegDisplay.fHeight / 2 - 20 / getfactor()],
	        });
	        leftKneeJoint.setLimits(-Math.PI, 0);
	        var rightFootShape = new p2.Box(rightFootDisplay.fSize);
	        bodyPartShapes.push(rightFootShape);
	        this.originPos['rightFoot'] = [
	            this.shouldersDistance / 2,
	            rightFootDisplay.fHeight / 2
	        ];
	        var rightFoot = this.rightFoot = new p2.Body({
	            allowSleep: false,
	            mass: this.footMass,
	            position: this.originPos['rightFoot'],
	        });
	        rightFoot.displays = [rightFootDisplay];
	        rightFoot.addShape(rightFootShape);
	        var lowerRightLegShape = new p2.Box(lowerRightLegDisplay.fSize);
	        bodyPartShapes.push(lowerRightLegShape);
	        this.originPos['lowerRightLeg'] = [
	            this.shouldersDistance / 2,
	            rightFoot.position[1] + lowerRightLegDisplay.fHeight / 2 + 5 / getfactor()
	        ];
	        var lowerRightLeg = this.lowerRightLeg = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['lowerRightLeg'],
	        });
	        lowerRightLeg.displays = [lowerRightLegDisplay];
	        lowerRightLeg.addShape(lowerRightLegShape);
	        var rightAnkleJoint = new p2.RevoluteConstraint(lowerRightLeg, rightFoot, {
	            localPivotA: [0, -lowerRightLegDisplay.fHeight / 2 + 5 / getfactor()],
	            localPivotB: [-rightFootDisplay.fWidth / 2 + 18 / getfactor(), 0],
	        });
	        rightAnkleJoint.setLimits(-Math.PI / 3, Math.PI / 3);
	        var rightFootGroundCons = this.rightFootGroundCons = new p2.RevoluteConstraint(rightFoot, ground, {
	            localPivotA: [0, -rightFootDisplay.fHeight / 2],
	            localPivotB: [rightFoot.position[0], 0],
	        });
	        rightFootGroundCons.setLimits(0, 0);
	        var upperRightLegShape = new p2.Box(upperRightLegDisplay.fSize);
	        bodyPartShapes.push(upperRightLegShape);
	        this.originPos['upperRightLeg'] = [
	            this.shouldersDistance / 2,
	            lowerRightLeg.position[1] + lowerRightLegDisplay.fHeight / 2 + upperRightLegDisplay.fHeight / 2 - 40 / getfactor()
	        ];
	        var upperRightLeg = this.upperRightLeg = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['upperRightLeg'],
	        });
	        upperRightLeg.displays = [upperRightLegDisplay];
	        upperRightLeg.addShape(upperRightLegShape);
	        var rightKneeJoint = new p2.RevoluteConstraint(upperRightLeg, lowerRightLeg, {
	            localPivotA: [0, -upperRightLegDisplay.fHeight / 2 + 20 / getfactor()],
	            localPivotB: [0, lowerRightLegDisplay.fHeight / 2 - 20 / getfactor()],
	        });
	        rightKneeJoint.setLimits(-Math.PI, 0);
	        var bodyWidth = 120 / getfactor();
	        var bodyHeight = 250 / getfactor();
	        var bodyShape = new p2.Box({ width: bodyWidth, height: bodyHeight });
	        bodyPartShapes.push(bodyShape);
	        this.originPos['body'] = [
	            0,
	            upperRightLeg.position[1] + upperRightLegDisplay.fHeight / 2 + bodyHeight / 2 - 60 / getfactor()
	        ];
	        var body = this.body = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['body'],
	        });
	        body.displays = [bodyDisplay];
	        body.addShape(bodyShape);
	        var leftHipJoint = new p2.RevoluteConstraint(body, upperLeftLeg, {
	            localPivotA: [-20 / getfactor(), -bodyHeight / 2 + 30 / getfactor()],
	            localPivotB: [0, upperLeftLegDisplay.fHeight / 2 - 30 / getfactor()],
	        });
	        leftHipJoint.setLimits(-Math.PI / 6 * 2, Math.PI / 6 * 2);
	        var rightHipJoint = new p2.RevoluteConstraint(body, upperRightLeg, {
	            localPivotA: [-20 / getfactor(), -bodyHeight / 2 + 30 / getfactor()],
	            localPivotB: [0, upperRightLegDisplay.fHeight / 2 - 30 / getfactor()],
	        });
	        rightHipJoint.setLimits(-Math.PI / 6 * 2, Math.PI / 6 * 2);
	        var headShape = new p2.Circle({ radius: 85 / getfactor() });
	        bodyPartShapes.push(headShape);
	        this.originPos['head'] = [
	            0,
	            body.position[1] + bodyHeight / 2 + 85 / getfactor()
	        ];
	        var head = this.head = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['head'],
	        });
	        head.addShape(headShape);
	        head.displays = [headDisplay];
	        var headJoint = new p2.RevoluteConstraint(body, head, {
	            localPivotA: [0, bodyHeight / 2 - 10 / getfactor()],
	            localPivotB: [0, (-85 + 10) / getfactor()],
	        });
	        headJoint.setLimits(-Math.PI / 6 * 2, Math.PI / 6 * 2);
	        var upperLeftArmShape = new p2.Box(upperLeftArmDisplay.fSize);
	        bodyPartShapes.push(upperLeftArmShape);
	        this.originPos['upperLeftArm'] = [
	            -this.shouldersDistance / 2 - 10 / getfactor(),
	            body.position[1] + bodyHeight / 2 - upperLeftArmDisplay.fHeight / 2
	        ];
	        var upperLeftArm = this.upperLeftArm = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['upperLeftArm'],
	        });
	        upperLeftArm.displays = [upperLeftArmDisplay];
	        upperLeftArm.addShape(upperLeftArmShape);
	        var leftShoulderJoint = new p2.RevoluteConstraint(body, upperLeftArm, {
	            localPivotA: [-this.shouldersDistance / 2 + 10 / getfactor(), bodyHeight / 2],
	            localPivotB: [0, upperLeftArmDisplay.fHeight / 2],
	        });
	        leftShoulderJoint.setLimits(-Math.PI / 2, 0);
	        var width = 50 / getfactor();
	        var lowerLeftArmShape = new p2.Box({ width: width, height: lowerLeftArmDisplay.fHeight });
	        bodyPartShapes.push(lowerLeftArmShape);
	        this.originPos['lowerLeftArm'] = [
	            -this.shouldersDistance / 2 - 10 / getfactor() + 10 / getfactor(),
	            upperLeftArm.position[1] - upperLeftArmDisplay.fHeight / 2 - lowerLeftArmDisplay.fHeight / 2 + 20 / getfactor()
	        ];
	        var lowerLeftArm = this.lowerLeftArm = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['lowerLeftArm'],
	        });
	        lowerLeftArm.displays = [lowerLeftArmDisplay];
	        lowerLeftArm.addShape(lowerLeftArmShape);
	        var leftElbowJoint = new p2.RevoluteConstraint(upperLeftArm, lowerLeftArm, {
	            localPivotA: [0, -upperLeftArmDisplay.fHeight / 2 + 10 / getfactor()],
	            localPivotB: [-10 / getfactor(), lowerLeftLegDisplay.fHeight / 2 - 10 / getfactor()],
	        });
	        leftElbowJoint.setLimits(0, Math.PI * 2 / 3);
	        var upperRightArmShape = new p2.Box(upperRightArmDisplay.fSize);
	        bodyPartShapes.push(upperRightArmShape);
	        this.originPos['upperRightArm'] = [
	            this.shouldersDistance / 2,
	            body.position[1] + bodyHeight / 2 - upperRightArmDisplay.fHeight / 2
	        ];
	        var upperRightArm = this.upperRightArm = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['upperRightArm'],
	        });
	        upperRightArm.displays = [upperRightArmDisplay];
	        upperRightArm.addShape(upperRightArmShape);
	        var rightShoulderJoint = new p2.RevoluteConstraint(body, upperRightArm, {
	            localPivotA: [this.shouldersDistance / 2 + 10 / getfactor(), bodyHeight / 2],
	            localPivotB: [0, upperRightArmDisplay.fHeight / 2],
	        });
	        rightShoulderJoint.setLimits(-Math.PI / 2, 0);
	        var width = 30 / getfactor();
	        var lowerRightArmShape = new p2.Box({ width: width, height: lowerRightArmDisplay.fHeight });
	        bodyPartShapes.push(lowerRightArmShape);
	        this.originPos['lowerRightArm'] = [
	            this.shouldersDistance / 2,
	            upperRightArm.position[1] - upperRightArmDisplay.fHeight / 2 - lowerRightArmDisplay.fHeight / 2 + 40 / getfactor()
	        ];
	        var lowerRightArm = this.lowerRightArm = new p2.Body({
	            allowSleep: false,
	            mass: this.partMass,
	            position: this.originPos['lowerRightArm'],
	        });
	        lowerRightArm.displays = [lowerRightArmDisplay];
	        lowerRightArm.addShape(lowerRightArmShape);
	        var rightElbowJoint = new p2.RevoluteConstraint(upperRightArm, lowerRightArm, {
	            localPivotA: [0, -upperRightArmDisplay.fHeight / 2 + 20 / getfactor()],
	            localPivotB: [0, lowerRightLegDisplay.fHeight / 2 - 20 / getfactor()],
	        });
	        rightElbowJoint.setLimits(0, Math.PI * 2 / 3);
	        var bodies = [
	            lowerRightArm, upperRightArm,
	            lowerRightLeg, upperRightLeg, rightFoot,
	            body,
	            lowerLeftLeg, upperLeftLeg, leftFoot,
	            lowerLeftArm, upperLeftArm,
	            head,
	        ];
	        bodies.forEach(function (body) {
	            world.addBody(body);
	        });
	        var joints = [
	            leftAnkleJoint,
	            leftKneeJoint,
	            rightAnkleJoint,
	            rightKneeJoint,
	            headJoint,
	            leftHipJoint,
	            rightHipJoint,
	            leftShoulderJoint,
	            leftElbowJoint,
	            rightShoulderJoint,
	            rightElbowJoint,
	            leftFootGroundCons,
	            rightFootGroundCons,
	        ];
	        joints.forEach(function (joint) {
	            world.addConstraint(joint);
	        });
	        for (var i = 0; i < bodyPartShapes.length; i++) {
	            var s = bodyPartShapes[i];
	            s.collisionGroup = collisionGroup;
	            s.collisionMask = collisionMask;
	        }
	        this.foots = [leftFoot, rightFoot];
	        this.lowerLegs = [lowerLeftLeg, lowerRightLeg];
	        this.upperLegs = [upperLeftLeg, upperRightLeg];
	        this.footGroundCons = [leftFootGroundCons, rightFootGroundCons];
	    };
	    Player.prototype.reset = function (revive) {
	        var _this = this;
	        if (revive === void 0) { revive = false; }
	        if (revive) ;
	        else {
	            this.step = -1;
	            this.safePos = 0;
	            [
	                'lowerLeftLeg',
	                'upperLeftLeg',
	                'lowerRightLeg',
	                'upperRightLeg',
	                'body',
	                'head',
	                'upperLeftArm',
	                'lowerLeftArm',
	                'upperRightArm',
	                'lowerRightArm',
	            ].forEach(function (partName) {
	                var part = _this[partName];
	                part.position[0] = _this.safePos + _this.originPos[partName][0];
	            });
	        }
	        for (var i = 0; i < 2; i++) {
	            var cons = this.footGroundCons[i];
	            cons.pivotB[0] = this.foots[i].position[0] = this.safePos + (i == 0 ? -1 : 1) * this.shouldersDistance / 2;
	            if (this.state == PlayerStat.FALL) {
	                this.world.addConstraint(cons);
	            }
	        }
	        this.switchFace('normal', false);
	        this.setSkin('normal');
	        this.state = PlayerStat.RELEASE;
	    };
	    Player.prototype.createDisplay = function (resName, offset) {
	        if (offset === void 0) { offset = null; }
	        var bmp = new PlayerPart(resName);
	        var anchorX = bmp.width / 2 + (offset ? offset.x : 0);
	        var anchorY = bmp.height / 2 + (offset ? offset.y : 0);
	        bmp.anchorTexture.set(anchorX / bmp.width, anchorY / bmp.height);
	        return bmp;
	    };
	    Object.defineProperty(Player.prototype, "currentSide", {
	        get: function () {
	            return this.step % 2;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Object.defineProperty(Player.prototype, "anotherSide", {
	        get: function () {
	            return (this.step + 1) % 2;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    Player.prototype.down = function () {
	        if (this.state == PlayerStat.RISING) {
	            return;
	        }
	        this.step++;
	        var cons = this.footGroundCons[this.currentSide];
	        this.world.removeConstraint(cons);
	        this.state = PlayerStat.RISING;
	    };
	    Player.prototype.up = function () {
	        if (this.state == PlayerStat.RELEASE) {
	            return;
	        }
	        var side = this.currentSide;
	        this.foots[side].velocity = [0, 0];
	        this.body.velocity = [0, 0];
	        var currentFoot = this.foots[side];
	        var cons = this.footGroundCons[side];
	        cons.pivotB[0] = currentFoot.position[0];
	        this.world.addConstraint(cons);
	        this.state = PlayerStat.RELEASE;
	        setTimeout(this.onFootDown, 200);
	    };
	    Player.prototype.fall = function () {
	        this.body.velocity = [0, -30];
	        this.head.velocity = [-50, -50];
	        this.world.removeConstraint(this.footGroundCons[0]);
	        this.world.removeConstraint(this.footGroundCons[1]);
	        this.foots[this.currentSide].velocity[0] = 60;
	        this.foots[this.anotherSide].velocity[0] = 70;
	        this.state = PlayerStat.FALL;
	    };
	    Player.prototype.switchFace = function (type, playAni) {
	        if (playAni === void 0) { playAni = true; }
	        var headDisplay = this.head.displays[0];
	        headDisplay.switchFace(type, playAni);
	    };
	    Player.prototype.setSkin = function (name) {
	        var _this = this;
	        if (this.skin == name) {
	            return;
	        }
	        this.skin = name;
	        [
	            'leftFoot',
	            'lowerLeftLeg',
	            'upperLeftLeg',
	            'rightFoot',
	            'lowerRightLeg',
	            'upperRightLeg',
	            'body',
	            'upperLeftArm',
	            'lowerLeftArm',
	            'upperRightArm',
	            'lowerRightArm',
	        ].forEach(function (partName) {
	            var part = _this[partName];
	            part.displays[0].setSkin(name);
	        });
	    };
	    Player.prototype.fade = function (alpha, animation) {
	        var _this = this;
	        if (animation === void 0) { animation = true; }
	        [
	            'leftFoot',
	            'lowerLeftLeg',
	            'upperLeftLeg',
	            'rightFoot',
	            'lowerRightLeg',
	            'upperRightLeg',
	            'body',
	            'head',
	            'upperLeftArm',
	            'lowerLeftArm',
	            'upperRightArm',
	            'lowerRightArm',
	        ].forEach(function (partName) {
	            var display = _this[partName].displays[0];
	            if (animation) {
	                engine.Tween.get(display, null, null, true)
	                    .to({ alpha: alpha }, 200);
	            }
	            else {
	                display.alpha = alpha;
	            }
	        });
	    };
	    return Player;
	}(engine.EventDispatcher));
	//# sourceMappingURL=Player.js.map

	var ObjectPool$1 = engine.ObjectPool;
	var PoolName = 'block';
	ObjectPool$1.registerPool(PoolName, function () {
	    return new Block();
	}, function (item, data) {
	    item.reset(data);
	});
	var GameView = (function (_super) {
	    tslib.__extends(GameView, _super);
	    function GameView() {
	        var _this = _super.call(this) || this;
	        _this.offset = { x: -100, y: 320 };
	        _this.worldFPS = 60;
	        _this._viewport = {
	            x: 0,
	            y: 0,
	        };
	        return _this;
	    }
	    Object.defineProperty(GameView, "instance", {
	        get: function () {
	            if (this._instance == undefined) {
	                this._instance = new GameView();
	            }
	            return this._instance;
	        },
	        enumerable: true,
	        configurable: true
	    });
	    GameView.prototype.eventStart = function () {
	        var _this = this;
	        if (this.hasPreSetup) {
	            return;
	        }
	        this.hasPreSetup = true;
	        var background = this.background = new Background();
	        background.setup();
	        this.addChild(background);
	        var road = this.road = new Road();
	        road.y = getStage().height / 2 + 310;
	        this.addChild(road);
	        this.road.preSetup();
	        this.setupWorld();
	        this.player.fade(0, false);
	        this.viewport = 0;
	        this.startWorld();
	        this.reset();
	        this.setup();
	        var playHint = new engine.Label();
	        playHint.fillColor = 'block';
	        playHint.size = 30;
	        playHint.text = '注意不要踩到线和陷阱哦，后果很严重哦~';
	        playHint.x = (getStage().width - playHint.width) / 2;
	        playHint.y = getStage().height / 2 + 500;
	        this.addChild(playHint);
	        this.playGame = new PlayGame();
	        setTimeout(function () {
	            _this.playGame.GameView = _this;
	            _this.addChild(_this.playGame);
	            _this.start();
	        }, 500);
	    };
	    GameView.prototype.setup = function () {
	        if (this.hasSetup) {
	            return;
	        }
	        this.hasSetup = true;
	        this.road.setup();
	    };
	    GameView.prototype.setupWorld = function () {
	        var OTHER = Math.pow(2, 1), BODYPARTS = Math.pow(2, 2), GROUND = Math.pow(2, 3);
	        var world = this.world = new p2.World({});
	        world.sleepMode = p2.World.BODY_SLEEPING;
	        var planeShape = new p2.Plane();
	        var ground = new p2.Body({
	            position: [0, 0],
	        });
	        ground.addShape(planeShape);
	        planeShape.collisionGroup = GROUND;
	        planeShape.collisionMask = BODYPARTS | OTHER;
	        world.addBody(ground);
	        var player = this.player = new Player();
	        player.setup({
	            collisionGroup: BODYPARTS,
	            collisionMask: GROUND | OTHER,
	            world: world,
	            ground: ground,
	        });
	        player.addEventListener('move', this.onMove, this);
	        player.addEventListener('foot_down', this.onFootDown, this);
	    };
	    GameView.prototype.onTicker = function (dt) {
	        var world = this.world;
	        this.world.step(this.worldFPS / 1000);
	        var l = world.bodies.length;
	        for (var i = 0; i < l; i++) {
	            var body = world.bodies[i];
	            var display = body.displays ? body.displays[0] : null;
	            if (display) {
	                if (!this.contains(display)) {
	                    this.addChild(display);
	                }
	                display.localUpdate(body.position[0] * getfactor() + getStage().width / 2 + this.offset.x - this.viewport, getStage().height / 2 - body.position[1] * getfactor() + this.offset.y, 360 - (body.angle + body.shapes[0].angle) * 180 / Math.PI, body.sleepState == p2.Body.SLEEPING);
	            }
	        }
	        return false;
	    };
	    GameView.prototype.reset = function (revive) {
	        if (revive === void 0) { revive = false; }
	        this.background.reset(revive);
	        this.road.reset(revive);
	        this.player.reset(revive);
	        this.player.fade(0);
	        this.viewport = revive ? this.revivePos : 0;
	    };
	    GameView.prototype.start = function (revive) {
	        if (revive === void 0) { revive = false; }
	        this.road.start(revive);
	        this.player.fade(1);
	        this.playGame.start(revive);
	    };
	    GameView.prototype.startWorld = function () {
	        if (this.playing) {
	            return;
	        }
	        this.playing = true;
	        engine.gameStage.addEventListener(engine.Event.ENTER_FRAME, this.onTicker, this);
	    };
	    GameView.prototype.stopWorld = function () {
	        if (!this.playing) {
	            return;
	        }
	        this.playing = false;
	        engine.gameStage.removeEventListener(engine.Event.ENTER_FRAME, this.onTicker, this);
	    };
	    Object.defineProperty(GameView.prototype, "viewport", {
	        get: function () {
	            return this._viewport.x;
	        },
	        set: function (v) {
	            this._viewport.x = v;
	            this.updateViewport();
	        },
	        enumerable: true,
	        configurable: true
	    });
	    GameView.prototype.updateViewport = function () {
	        var _a = this._viewport, x = _a.x, y = _a.y;
	        var pos = Math.max(0, x);
	        this.background.setViewport(pos);
	        this.road.setViewport(pos);
	    };
	    GameView.prototype.onMove = function (event) {
	        var _a = event.data, pos = _a.pos, bodyPos = _a.bodyPos;
	        this.viewport = pos;
	        if (this.road.hitReward(bodyPos)) {
	            this.dispatchEvent('scoreChange', {
	                rewardScore: 100,
	                passType: 10,
	            }, true);
	        }
	    };
	    GameView.prototype.down = function () {
	        if (this.lock) {
	            return;
	        }
	        this.player.down();
	    };
	    GameView.prototype.up = function () {
	        if (this.lock) {
	            return;
	        }
	        this.player.up();
	    };
	    GameView.prototype.onFootDown = function (event) {
	        var _a = event.data, mileage = _a.mileage, pos = _a.pos;
	        this.revivePos = pos;
	        var _b = this.road.hitBlock(pos), target = _b.target, lastTarget = _b.lastTarget;
	        this.road.playDust(pos);
	        if (target) {
	            this.lock = true;
	            this.stopWorld();
	            this.dispatchEvent(engine.Event.COMPLETE, null, true);
	            target.playFadeOut();
	            target.playEffect();
	            if (target.type == 4) {
	                this.player.setSkin('gray');
	            }
	            this.road.removeBlock(target);
	            this.player.switchFace(target.type);
	            if (target.type == 0) {
	                this.player.fall();
	            }
	            this.lock = false;
	            this.startWorld();
	        }
	        else {
	            this.player.switchFace('normal');
	            var score = Math.floor(mileage / 50 * 5);
	            this.dispatchEvent('scoreChange', {
	                score: score,
	                passType: lastTarget ? lastTarget.type : -1,
	            }, true);
	        }
	    };
	    GameView.prototype.setWitchFace = function (type) {
	        this.player.switchFace(type);
	    };
	    return GameView;
	}(engine.Container));
	//# sourceMappingURL=GameView.js.map

	var GameWrapper = (function (_super) {
	    tslib.__extends(GameWrapper, _super);
	    function GameWrapper() {
	        var _this = _super.call(this) || this;
	        engine.globalEvent.addEventListener('walking-cat-start', _this.start, _this);
	        engine.globalEvent.addEventListener('walking-cat-revive', _this.revive, _this);
	        engine.globalEvent.addEventListener('walking-cat-showScore', _this.revive, _this);
	        var gameView = _this._gameView = new GameView();
	        _this.addChild(gameView);
	        return _this;
	    }
	    GameWrapper.prototype.start = function (event) {
	        injectProps(event.data);
	        this._gameView.eventStart();
	    };
	    GameWrapper.prototype.revive = function () {
	        console.log('复活');
	        this._gameView.reset(true);
	        this._gameView.start(true);
	    };
	    return GameWrapper;
	}(engine.Container));
	//# sourceMappingURL=GameWrapper.js.map

	function index (props) {
	    prepareProps();
	    injectProps(props);
	    var instance = new GameWrapper();
	    return instance;
	}
	//# sourceMappingURL=index.js.map

	return index;

})));

//# sourceMappingURL=main.js.map