export default class SpriteFontLabel extends engine.Container {
    public set text(v: string) {
        if (typeof v != "string") return;
        if (v != this._text) {
            this._text = v;
            this.onDraw();
        }
    }
    public get text() {
        return this._text;
    }
    private _text: string = "";

    public font: SpriteFont = null;

    private sprites: engine.Sprite[] = [];

    constructor(font: SpriteFont, text?: string) {
        super();
        this.font = font;
        this.text = text;
    }

    private onDraw() {
        if (!this.font) {
            console.error("font is null");
            return;
        }

        let sprite: engine.Sprite = null;
        let tex: engine.Texture = null;
        let char: string = null;
        let curX: number = 0;
        for (let i = 0; i < this._text.length; i++) {
            char = this._text[i];
            tex = this.font.getWord(char);
            sprite = this.sprites[i];
            if (tex) {
                if (!sprite) {
                    sprite = new engine.Sprite();
                    this.addChild(sprite);
                    this.sprites.push(sprite);
                }
                sprite.texture = tex;
                sprite.x = curX;
                curX += sprite.width;
            } else {
                sprite.texture = null;
                console.warn("con not resolve char:" + char);
            }
        }

        if (this.sprites.length > this._text.length) {
            //删除多余的sprite
            let rest = this.sprites.slice(this._text.length);
            for (let i of rest) {
                i.destroy();
            }
            this.sprites = this.sprites.slice(0, this._text.length);
        }
    }
}

const fontCaches: {
    [name: string]: {
        [char: string]: engine.Texture
    }
} = {}

type SpriteFontConfig = {
    name: string,
    textures: {
        [char: string]: {
            x: number,  //x,y,w,h为图集上的切图位置数据
            y: number,
            w: number,
            h: number,
            ox: number,  //ox,oy为偏移数据，trim裁剪的数据，在orig上的偏移{x:0,y:0,width:200,height:200}
            oy: number,
            sw: number, //sw,sh为原始宽高
            sh: number,
        }
    }
}

export class SpriteFont {
    public static load(texture: engine.Texture, config: SpriteFontConfig) {
        if (!fontCaches[config.name]) {
            let font: {
                [char: string]: engine.Texture
            } = {};
            let altaData = {};
            let chars = Object.keys(config.textures);
            for (let char of chars) {
                altaData[config.name + "_" + char] = config.textures[char];
            }

            engine.createTextureSheet(texture.baseTexture, altaData);

            for (let char of chars) {
                font[char] = RES.getRes(config.name + "_" + char);
            }

            fontCaches[config.name] = font;
        }

    }

    private words: {
        [char: string]: engine.Texture
    } = null;

    constructor(name: string) {
        if (fontCaches[name]) {
            this.words = fontCaches[name];
        } else {
            console.error("unloaded font:" + name);
        }
    }

    getWord(char: string) {
        return this.words[char];
    }
}