(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory(require('tslib')) :
    typeof define === 'function' && define.amd ? define(['tslib'], factory) :
    (global = global || self, global['shoot-planet-2'] = factory(global.tslib));
}(this, (function (tslib) { 'use strict';

    var GameObject = (function (_super) {
        tslib.__extends(GameObject, _super);
        function GameObject() {
            var _this = _super.call(this) || this;
            _this.name = "";
            _this.componentList = [];
            _this.mouseEnabled = false;
            _this.mouseChildren = false;
            return _this;
        }
        GameObject.prototype.addComponent = function (cls) {
            if (this.getComponent(cls) != null) {
                console.error("component is existent");
                return;
            }
            var ins = new cls(this);
            ins.enabled = true;
            this.componentList.push(ins);
            return ins;
        };
        GameObject.prototype.getComponent = function (cls) {
            for (var _i = 0, _a = this.componentList; _i < _a.length; _i++) {
                var i = _a[_i];
                if (i instanceof cls) {
                    return i;
                }
            }
            return null;
        };
        GameObject.prototype.removeComponent = function (cls) {
            var _loop_1 = function (i) {
                if (i instanceof cls) {
                    i.enabled = false;
                    this_1.componentList = this_1.componentList.filter(function (e) { return e != i; });
                    return { value: void 0 };
                }
            };
            var this_1 = this;
            for (var _i = 0, _a = this.componentList; _i < _a.length; _i++) {
                var i = _a[_i];
                var state_1 = _loop_1(i);
                if (typeof state_1 === "object")
                    return state_1.value;
            }
        };
        GameObject.prototype.dispose = function () {
            this.destroy();
            this.disableAllComponents();
        };
        GameObject.prototype.disableAllComponents = function () {
            var _this = this;
            Object.keys(this.componentList).forEach(function (e) { return _this.componentList[e].enabled = false; });
        };
        GameObject.prototype.enableAllComponents = function () {
            var _this = this;
            Object.keys(this.componentList).forEach(function (e) { return _this.componentList[e].enabled = true; });
        };
        GameObject.prototype.onCollisionEnter = function (other) {
        };
        GameObject.prototype.onCollisionStay = function (other) {
        };
        GameObject.prototype.onCollisionExit = function (other) {
        };
        return GameObject;
    }(engine.Container));

    var GameComponent = (function () {
        function GameComponent(owner) {
            this._enabled = false;
            this._owner = null;
            this._owner = owner;
        }
        Object.defineProperty(GameComponent.prototype, "enabled", {
            get: function () {
                return this._enabled;
            },
            set: function (v) {
                if (v == true && this._enabled == false) {
                    this.onEnabled();
                    this._enabled = v;
                }
                else if (v == false && this._enabled == true) {
                    this.onDisabled();
                    this._enabled = v;
                }
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GameComponent.prototype, "owner", {
            get: function () {
                return this._owner;
            },
            enumerable: true,
            configurable: true
        });
        return GameComponent;
    }());

    var DataMgr;
    (function (DataMgr) {
        DataMgr.game = null;
        DataMgr.minEnableCaptchaScore = null;
    })(DataMgr || (DataMgr = {}));

    var Physics = (function (_super) {
        tslib.__extends(Physics, _super);
        function Physics() {
            var _this = _super !== null && _super.apply(this, arguments) || this;
            _this.rotateVelocity = 0;
            _this.acceleration = new engine.Point();
            _this.velocity = new engine.Point();
            _this._collider = null;
            return _this;
        }
        Object.defineProperty(Physics.prototype, "collider", {
            get: function () {
                return this._collider;
            },
            set: function (v) {
                this._collider = v;
                v.physics = this;
            },
            enumerable: true,
            configurable: true
        });
        Physics.prototype.onFixedUpdate = function (dtFactor) {
            this.velocity.x += this.acceleration.x * dtFactor;
            this.velocity.y += this.acceleration.y * dtFactor;
            if (this.velocity.length > 0) {
                this.moveTo(this.owner.x += this.velocity.x * dtFactor, this.owner.y += this.velocity.y * dtFactor);
            }
            if (this.rotateVelocity != 0) {
                this.owner.rotation += this.rotateVelocity * dtFactor;
            }
        };
        Physics.prototype.moveTo = function (x, y) {
            if (typeof x == "number") {
                if (!y) {
                    console.error("invalid param");
                    return;
                }
                this.owner.x = x;
                this.owner.y = y;
            }
            else {
                this.owner.x = x.x;
                this.owner.y = x.y;
            }
            this.onMoved && this.onMoved(this.owner);
        };
        Physics.prototype.onDisabled = function () {
            DataMgr.game._phycicsSystem.remove(this);
        };
        Physics.prototype.onEnabled = function () {
            DataMgr.game._phycicsSystem.add(this);
        };
        Physics.prototype.onColliderResize = function (collider) {
            var center = collider.getCenter();
            this.owner.anchorX = center[0];
            this.owner.anchorY = center[1];
        };
        return Physics;
    }(GameComponent));

    var Collider = (function (_super) {
        tslib.__extends(Collider, _super);
        function Collider() {
            var _this = _super !== null && _super.apply(this, arguments) || this;
            _this.physics = null;
            _this._group = null;
            _this.type = null;
            _this.worldPos = new engine.Point();
            return _this;
        }
        Object.defineProperty(Collider.prototype, "group", {
            get: function () {
                return this._group;
            },
            set: function (group) {
                if (this._group == null) {
                    this._group = group;
                    DataMgr.game._phycicsSystem.addCollider(this);
                }
                else {
                    this._group = group;
                }
            },
            enumerable: true,
            configurable: true
        });
        Collider.prototype.setData = function () {
            var ags = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                ags[_i] = arguments[_i];
            }
            this.physics && this.physics.onColliderResize(this);
        };
        Collider.prototype.onEnabled = function () {
            if (this.group)
                DataMgr.game._phycicsSystem.addCollider(this);
            var physics = this.owner.getComponent(Physics);
            if (physics) {
                this.physics = physics;
                physics.collider = this;
            }
        };
        Collider.prototype.onDisabled = function () {
            DataMgr.game._phycicsSystem.removeCollider(this);
        };
        return Collider;
    }(GameComponent));
    var CircleCollider = (function (_super) {
        tslib.__extends(CircleCollider, _super);
        function CircleCollider(owner) {
            var _this = _super.call(this, owner) || this;
            _this.radius = 0;
            _this.center = [];
            _this.t = {
                x: 0,
                y: 0
            };
            _this.type = ColliderType.Circle;
            return _this;
        }
        CircleCollider.prototype.setData = function (x, y, radius) {
            this.radius = radius;
            this.center[0] = x;
            this.center[1] = y;
            _super.prototype.setData.call(this);
        };
        CircleCollider.prototype.getCenter = function () {
            return this.center;
        };
        CircleCollider.prototype.getWorldPosition = function (out) {
            out[0] = this.owner.x + this.center[0];
            out[1] = this.owner.y + this.center[1];
        };
        return CircleCollider;
    }(Collider));
    var RectCollider = (function (_super) {
        tslib.__extends(RectCollider, _super);
        function RectCollider(owner) {
            var _this = _super.call(this, owner) || this;
            _this.rect = new engine.Rectangle();
            _this.type = ColliderType.Rect;
            return _this;
        }
        RectCollider.prototype.setData = function (x, y, width, height) {
            this.rect.x = x;
            this.rect.y = y;
            this.rect.width = width;
            this.rect.height = height;
            _super.prototype.setData.call(this);
        };
        RectCollider.prototype.getCenter = function () {
            return [this.rect.x + this.rect.width / 2, this.rect.y + this.rect.height / 2];
        };
        RectCollider.prototype.getWorldPosition = function (out) {
            out[0] = this.owner.x + this.rect.left;
            out[1] = this.owner.y + this.rect.top;
        };
        return RectCollider;
    }(Collider));
    var PointCollider = (function (_super) {
        tslib.__extends(PointCollider, _super);
        function PointCollider(owner) {
            var _this = _super.call(this, owner) || this;
            _this.center = [];
            _this.type = ColliderType.Point;
            return _this;
        }
        PointCollider.prototype.setData = function (x, y) {
            this.center[0] = x;
            this.center[1] = y;
            _super.prototype.setData.call(this);
        };
        PointCollider.prototype.getCenter = function () {
            return this.center;
        };
        PointCollider.prototype.getWorldPosition = function (out) {
            var owner = this.owner;
            out[0] = this.owner.x + this.center[0];
            out[1] = this.owner.y + this.center[1];
        };
        return PointCollider;
    }(Collider));
    var ColliderGroup;
    (function (ColliderGroup) {
        ColliderGroup[ColliderGroup["None"] = 0] = "None";
        ColliderGroup[ColliderGroup["Ball"] = 1] = "Ball";
        ColliderGroup[ColliderGroup["Bullet"] = 2] = "Bullet";
        ColliderGroup[ColliderGroup["Wall"] = 3] = "Wall";
        ColliderGroup[ColliderGroup["Ground"] = 4] = "Ground";
        ColliderGroup[ColliderGroup["Drop"] = 5] = "Drop";
        ColliderGroup[ColliderGroup["Car"] = 6] = "Car";
        ColliderGroup[ColliderGroup["Top"] = 7] = "Top";
    })(ColliderGroup || (ColliderGroup = {}));
    var ColliderType;
    (function (ColliderType) {
        ColliderType[ColliderType["Circle"] = 0] = "Circle";
        ColliderType[ColliderType["Rect"] = 1] = "Rect";
        ColliderType[ColliderType["Point"] = 2] = "Point";
    })(ColliderType || (ColliderType = {}));

    var MConst;
    (function (MConst) {
        MConst.MaxColorIndex = 5;
        MConst.DesignResolution = {
            width: 750,
            height: 1624
        };
        MConst.GroundLine = 1110;
        MConst.BulletFireSpeed = {
            min: 10,
            max: 20
        };
        MConst.Gravity = 0.15;
        MConst.BallVelocityX = 2;
        MConst.BallVelocityY = 11;
        MConst.BallSplitVelocityY = 6.8;
        MConst.DropVelocityX = {
            x: 2,
            offset: 0.5
        };
        MConst.BallRotateSpeed = 0.7;
        MConst.BallInitPosY = 1206 * 0.25;
        MConst.CreateBallInterval = 1000;
        MConst.ReviveCardId = "sp_1";
        MConst.ConstantSubmitScoreNum = 100;
        MConst.DefaultBallWidth = 224;
        MConst.DefaultStarSize = {
            width: 76 * 0.744,
            height: 72 * 0.744
        };
        MConst.captchaId = "a869bfdfb9bd4cdf88e1ff2f8667a114";
        MConst.BallVelocityXRandomFactor = 0.1;
        MConst.BallVelocityYRandomFactor = 0.08;
        MConst.DropRandomFactor = 0.07;
        MConst.ReviveInvincibleDuration = 3;
        MConst.WenzhouLocation = new engine.Rectangle(119.37, 27.03, 1.81, 1.33);
        MConst.HangzhouLocation = new engine.Rectangle(118.21, 29.11, 2.09, 1.22);
    })(MConst || (MConst = {}));

    var MoveObjcet = (function (_super) {
        tslib.__extends(MoveObjcet, _super);
        function MoveObjcet() {
            var _this = _super.call(this) || this;
            _this.physics = null;
            _this.physics = _this.addComponent(Physics);
            return _this;
        }
        return MoveObjcet;
    }(GameObject));

    var MEvent = (function () {
        function MEvent() {
            this.list = [];
            this.isTraversing = false;
            this.deleteList = [];
        }
        MEvent.prototype.add = function (callback, thisObj) {
            this.list.push({
                callback: callback,
                thisObj: thisObj,
                bindFunc: thisObj ? callback.bind(thisObj) : undefined
            });
        };
        MEvent.prototype.clear = function () {
            if (this.isTraversing) {
                this.deleteList.concat(this.list);
            }
            else {
                this.list = [];
            }
        };
        MEvent.prototype.remove = function (callback, thisObj) {
            if (this.isTraversing) {
                this.deleteList.push({
                    callback: callback,
                    thisObj: thisObj
                });
            }
            else {
                this.list = this.list.filter(function (e) { return e.callback !== callback || e.thisObj !== thisObj; });
            }
        };
        MEvent.prototype.call = function () {
            var _this = this;
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i] = arguments[_i];
            }
            this.isTraversing = true;
            var item = null;
            for (var i = 0; i < this.list.length; i++) {
                item = this.list[i];
                if (item.bindFunc) {
                    item.bindFunc.apply(item, args);
                }
                else {
                    item.callback.apply(item, args);
                }
            }
            var _loop_1 = function (i) {
                this_1.list = this_1.list.filter(function (e) { return e.callback !== _this.deleteList[i].callback || e.thisObj !== _this.deleteList[i].thisObj; });
            };
            var this_1 = this;
            for (var i = 0; i < this.deleteList.length; i++) {
                _loop_1(i);
            }
            this.deleteList = [];
            this.isTraversing = false;
        };
        return MEvent;
    }());

    var MTimer = (function () {
        function MTimer() {
        }
        MTimer.init = function () {
            startTime = Date.now();
            engine.gameStage.addEventListener(engine.Event.ENTER_FRAME, onUpdate, MTimer);
            document.addEventListener("visibilitychange", onVisibilityChange);
        };
        MTimer.destroy = function () {
            _onFrame.clear();
            engine.gameStage.removeEventListener(engine.Event.ENTER_FRAME, onUpdate, MTimer);
            document.removeEventListener("visibilitychange", onVisibilityChange);
        };
        Object.defineProperty(MTimer, "deltaTime", {
            get: function () {
                return _deltaTime;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(MTimer, "dtFactor", {
            get: function () {
                return MTimer.deltaTime / DefaultDeltaTime;
            },
            enumerable: true,
            configurable: true
        });
        MTimer.onFrame = function (key, callback, thisObj) {
            if (key) {
                if (onFrameCaches[key]) {
                    _onFrame.remove(onFrameCaches[key].callback, onFrameCaches[key].thisObj);
                    delete onFrameCaches[key];
                }
                onFrameCaches[key] = {
                    callback: callback,
                    thisObj: thisObj
                };
            }
            _onFrame.add(callback, thisObj);
        };
        MTimer.removeOnFrame = function (key) {
            if (onFrameCaches[key]) {
                _onFrame.remove(onFrameCaches[key].callback, onFrameCaches[key].thisObj);
                delete onFrameCaches[key];
            }
        };
        MTimer.setFrameTimer = function (frame, onTimeOut) {
            var count = 0;
            var callback = function () {
                if (count >= frame) {
                    onTimeOut();
                    timerTickEvent.remove(callback);
                }
                else {
                    count += MTimer.dtFactor;
                }
            };
            timerTickEvent.add(callback);
        };
        return MTimer;
    }());
    var onFrameCaches = {};
    var _onFrame = new MEvent();
    var timerTickEvent = new MEvent();
    var DefaultDeltaTime = 1000 / 60;
    var _deltaTime = 0;
    var startTime = 0;
    var lastTime = 0;
    var curTime = 0;
    function updateDeltaTime() {
        lastTime = curTime;
        curTime = Date.now() - startTime;
        _deltaTime = curTime - lastTime;
    }
    function onVisibilityChange() {
        if (document.visibilityState == "visible") {
            curTime = Date.now() - startTime;
        }
    }
    function onUpdate() {
        updateDeltaTime();
        _onFrame.call(MTimer.deltaTime);
        timerTickEvent.call();
    }

    var GameMgr = (function () {
        function GameMgr(name) {
            this.name = name;
            this.pause = false;
        }
        GameMgr.prototype.onInit = function () {
            MTimer.onFrame(this.name, this.onUpdate, this);
        };
        GameMgr.prototype.onDestroy = function () {
            MTimer.removeOnFrame(this.name);
        };
        GameMgr.prototype.onUpdate = function (dt) {
        };
        return GameMgr;
    }());

    function getTexture(uuid) {
        var config = getAssetByUUID(uuid);
        return engine.Texture.from(config ? config.uuid : uuid);
    }
    function getAsset(uuid) {
        var config = getAssetByUUID(uuid);
        return engine.globalLoader.get(config.uuid);
    }
    function clamp(target, min, max) {
        return Math.min(max, Math.max(min, target));
    }

    var bulletSpeedValue = 20;
    var frameCountThatBulletsMoveInX = 4;
    var Bullet = (function (_super) {
        tslib.__extends(Bullet, _super);
        function Bullet() {
            var _this = _super.call(this) || this;
            _this.power = 1;
            _this.addChild(new engine.Image(getTexture("09bdff41-2fe3-4298-ad3a-1d2eaf6af778")));
            var collider = _this.addComponent(PointCollider);
            collider.group = ColliderGroup.Bullet;
            collider.setData(2, 0);
            _this.physics.velocity.y = -bulletSpeedValue;
            return _this;
        }
        Bullet.prototype.init = function (power) {
            this.power = power;
            return this;
        };
        Bullet.prototype.onElementInit = function () {
            this.enableAllComponents();
            this.visible = true;
        };
        Bullet.prototype.onElementRecycle = function () {
            this.disableAllComponents();
            this.visible = false;
        };
        Bullet.prototype.onCollisionEnter = function (other) {
            if (other.group == ColliderGroup.Ball || other.group == ColliderGroup.Top) {
                DataMgr.game._pool.bullet.recycle(this);
            }
        };
        return Bullet;
    }(MoveObjcet));
    var BulletGroup = (function () {
        function BulletGroup(position, rowBulletNum) {
            this.moveDatas = [];
            this.restFrameCountThatBulletsMoveInX = frameCountThatBulletsMoveInX;
            for (var i = 0; i < rowBulletNum; i++) {
                var bullet = DataMgr.game._pool.bullet.spwan().init(Math.floor(DataMgr.game.powerScore / 100));
                bullet.x = position.x - bullet.width / 2;
                bullet.y = position.y + bullet.height / 2;
                var frameOffset = ((i - (rowBulletNum - 1) / 2) * (bullet.width + 3)) / frameCountThatBulletsMoveInX;
                this.moveDatas.push({
                    bullet: bullet,
                    frameOffset: frameOffset,
                });
            }
            DataMgr.game._horizontalMoveMgr.movingList.push(this);
        }
        BulletGroup.prototype.onUpdate = function () {
            var data = null;
            for (var i = 0; i < this.moveDatas.length; i++) {
                data = this.moveDatas[i];
                data.bullet.x += data.frameOffset;
            }
            this.restFrameCountThatBulletsMoveInX--;
        };
        return BulletGroup;
    }());
    var HorizontalMoveMgr = (function (_super) {
        tslib.__extends(HorizontalMoveMgr, _super);
        function HorizontalMoveMgr() {
            var _this = _super.call(this, "HorizontalMoveMgr") || this;
            _this.movingList = [];
            return _this;
        }
        HorizontalMoveMgr.prototype.onUpdate = function () {
            if (this.pause)
                return;
            for (var i = 0; i < this.movingList.length; i++) {
                this.movingList[i].onUpdate();
            }
            this.movingList = this.movingList.filter(function (e) { return e.restFrameCountThatBulletsMoveInX > 0; });
        };
        return HorizontalMoveMgr;
    }(GameMgr));

    var SoundMgr = (function () {
        function SoundMgr() {
        }
        Object.defineProperty(SoundMgr, "instance", {
            get: function () {
                if (!this._instance) {
                    this._instance = new SoundMgr();
                }
                return this._instance;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(SoundMgr.prototype, "enabled", {
            get: function () {
                return engine.env.soundEnabled;
            },
            enumerable: true,
            configurable: true
        });
        SoundMgr.prototype.playEffect = function (name, loop) {
            if (!this.enabled)
                return;
            engine.playSound(getAssetByUUID(nameToUuid[name]).uuid, {
                loop: false,
                keep: true
            }, name);
        };
        SoundMgr._instance = null;
        return SoundMgr;
    }());
    var nameToUuid = {
        shoot: "b01dc39a-2886-4887-b3bb-0f8c801003a9",
        collect: "7004616a-0aba-4826-9245-895e7fdf8d31",
        dong: "7fc04e43-1465-4336-92a3-d6039ee88cb3",
        boom: "ca6b799f-be85-4e94-99df-812f31801490"
    };

    var props = {};
    function prepareProps() {
        var metaProps = getProps();
        engine.injectProp(props, metaProps);
    }
    function injectProps(p) {
        engine.injectProp(props, p);
    }

    var Car = (function (_super) {
        tslib.__extends(Car, _super);
        function Car() {
            var _this = _super.call(this) || this;
            _this.imgParticel = null;
            _this.fireTimer = null;
            _this.touchWall = 0;
            _this.isDied = false;
            _this.fireParticleTimer = 0;
            _this.invincible = false;
            _this.rowBulletNum = 1;
            _this.fireSpeed = MConst.BulletFireSpeed.min;
            var carCollidePaddingX = props.carCollidePaddingX, carCollidePaddingY = props.carCollidePaddingY;
            var sprite = new engine.Sprite(getTexture("ddf5128e-be66-4c00-8505-e4167aac37bc"));
            sprite.x = 0;
            sprite.y = 16;
            _this.addChild(sprite);
            var particle = new engine.Sprite(getTexture("d8daa113-0d75-4139-9e81-c54b7522c890"));
            particle.x = (sprite.width - particle.width) / 2;
            particle.y = 0;
            _this.addChild(particle);
            _this.imgParticel = particle;
            MTimer.onFrame("Car", _this.onUpdate, _this);
            var collider = _this.collider = _this.addComponent(RectCollider);
            collider.setData(carCollidePaddingX, carCollidePaddingY, sprite.width - carCollidePaddingX * 2, sprite.height - carCollidePaddingY * 2);
            collider.group = ColliderGroup.Car;
            return _this;
        }
        Car.prototype.fire = function () {
            SoundMgr.instance.playEffect("shoot");
            new BulletGroup(new engine.Point(this.x + this.width / 2, this.y), this.rowBulletNum);
        };
        Car.prototype.die = function () {
            if (this.invincible)
                return;
            if (this.isDied)
                return;
            this.isDied = true;
            this.onDied && this.onDied();
        };
        Car.prototype.onCollisionEnter = function (other) {
            if (other.group == ColliderGroup.Ball) {
                this.die();
            }
            if (other.group == ColliderGroup.Wall) {
                this.touchWall = other.owner.x < 0 ? 1 : -1;
            }
        };
        Car.prototype.onCollisionStay = function (other) {
            if (other.group == ColliderGroup.Ball) {
                this.die();
            }
            if (other.group == ColliderGroup.Wall) {
                this.touchWall = other.owner.x < 0 ? 1 : -1;
            }
        };
        Car.prototype.onceBlink = function () {
            var _this = this;
            engine.Tween.get(this)
                .to({ alpha: 0 }, 1)
                .wait(100)
                .to({ alpha: 1 }, 1)
                .wait(100)
                .call(function () {
                _this.onceBlink();
            });
        };
        Car.prototype.revive = function () {
            var _this = this;
            this.invincible = true;
            this.isDied = false;
            engine.Tween.removeTweens(this);
            this.onceBlink();
            MTimer.setFrameTimer(MConst.ReviveInvincibleDuration * 60, function () {
                _this.invincible = false;
                _this.alpha = 1;
                engine.Tween.removeTweens(_this);
            });
        };
        Car.prototype.move = function (deltaX) {
            if (this.touchWall == 1 && deltaX <= 0) {
                return;
            }
            else if (this.touchWall == -1 && deltaX >= 0) {
                return;
            }
            var carCollidePaddingX = props.carCollidePaddingX;
            var _a = this.collider.rect, x = _a.x, width = _a.width;
            this.x = Math.max(-x, Math.min(750 - width - carCollidePaddingX, this.x + deltaX));
        };
        Car.prototype.onCollisionExit = function (other) {
            if (other.group == ColliderGroup.Wall) {
                this.touchWall = 0;
            }
        };
        Car.prototype.onUpdate = function () {
            if (DataMgr.game.pause)
                return;
            var score = DataMgr.game.bulletScore;
            if (score <= 20) {
                var max = MConst.BulletFireSpeed.max;
                var min = MConst.BulletFireSpeed.min;
                this.fireSpeed = min + (max - min) * (score / 20);
            }
            else {
                this.fireSpeed = MConst.BulletFireSpeed.max;
            }
            this.rowBulletNum = Math.ceil((clamp(score, 15, 140) / 2) / 10);
            this.fireTimer += 1 * MTimer.dtFactor;
            if (this.fireTimer >= 60 / this.fireSpeed) {
                this.fire();
                this.fireTimer = 0;
            }
            if (this.fireParticleTimer % 3 == 0) {
                this.imgParticel.visible = !this.imgParticel.visible;
            }
            this.fireParticleTimer++;
        };
        return Car;
    }(GameObject));

    var PlayerController = (function (_super) {
        tslib.__extends(PlayerController, _super);
        function PlayerController() {
            var _this = _super.call(this) || this;
            _this.onTouchMove = null;
            _this.touchId = null;
            _this.lastX = null;
            _this.width = MConst.DesignResolution.width;
            _this.height = MConst.DesignResolution.height;
            _this.addEventListener(engine.MouseEvent.MOUSE_DOWN, function (e) {
                _this.touchId = e.instanceId;
                _this.lastX = e.clientX;
            }, _this);
            _this.addEventListener(engine.MouseEvent.MOUSE_MOVE, function (e) {
                if (e.instanceId != _this.touchId || _this.touchId == null || _this.lastX == null)
                    return;
                _this.onTouchMove && _this.onTouchMove(e.clientX - _this.lastX);
                _this.lastX = e.clientX;
            }, _this);
            _this.addEventListener(engine.MouseEvent.MOUSE_UP, function (e) {
                if (e.instanceId != _this.touchId)
                    return;
                _this.touchId = null;
                _this.lastX = null;
            }, _this);
            var rect = new engine.Rect();
            rect.width = 750;
            rect.height = 1624;
            rect.fillColor = "rgba(255,255,255,1)";
            rect.alpha = 0;
            _this.addChild(rect);
            return _this;
        }
        return PlayerController;
    }(engine.Container));

    var MConfigs;
    (function (MConfigs) {
        function updateConfig() {
            var mconfig = window["mconfig"];
            if (typeof mconfig != "object")
                return;
            Object.keys(mconfig).forEach(function (e) {
                MConfigs[e] = mconfig[e];
            });
        }
        MConfigs.updateConfig = updateConfig;
        MConfigs.size = [0.3125, 0.5357, 0.7321, 1];
        MConfigs.drop = [
            {
                type: "bullet",
                value: 1,
                textColor: 0x5dcefa,
                fileIndex: "0"
            },
            {
                type: "bullet",
                value: 2,
                textColor: 0x69e7fc,
                fileIndex: "1"
            },
            {
                type: "power",
                value: 30,
                textColor: 0xe83cf7,
                fileIndex: "2"
            },
            {
                type: "power",
                value: 20,
                textColor: 0xeb3924,
                fileIndex: "1"
            },
            {
                type: "power",
                value: 10,
                textColor: 0xfef552,
                fileIndex: "0"
            }
        ];
        MConfigs.dropPool = {
            0: {
                num: 3,
                factor: {
                    0: 40,
                    1: 0,
                    2: 0,
                    3: 15,
                    4: 50
                }
            },
            1: {
                num: 2,
                factor: {
                    0: 40,
                    1: 0,
                    2: 5,
                    3: 15,
                    4: 50
                }
            },
            2: {
                num: 1,
                factor: {
                    0: 40,
                    1: 0,
                    2: 5,
                    3: 15,
                    4: 50
                }
            }
        };
        MConfigs.guide = {
            0: {
                mask: "guide_mask_0_png",
                text: {
                    str: "左右移动炮车进行射击和躲避",
                    offsetToMask: {
                        x: 8,
                        y: 290
                    }
                }
            },
            1: {
                mask: "guide_mask_1_png",
                text: {
                    str: "拾取金币可获得更多的炮弹输出和得分加成",
                    offsetToMask: {
                        x: -65,
                        y: 290
                    }
                }
            },
        };
    })(MConfigs || (MConfigs = {}));

    var MUtils;
    (function (MUtils) {
        function random(min, max) {
            return Math.random() * (max - min) + min;
        }
        MUtils.random = random;
        function randomInt(min, max) {
            return Math.floor(Math.random() * (max - min) + min);
        }
        MUtils.randomInt = randomInt;
        function toRadians(degrees) {
            return degrees / 180 * Math.PI;
        }
        MUtils.toRadians = toRadians;
        function toDegress(radians) {
            return radians / Math.PI * 180;
        }
        MUtils.toDegress = toDegress;
        function degreesAngelAdd(a, b) {
            return (a + b) % 360;
        }
        MUtils.degreesAngelAdd = degreesAngelAdd;
        function dot(vec1, vec2) {
            return vec1.x * vec2.x + vec1.y * vec2.y;
        }
        MUtils.dot = dot;
        function getAngle(point) {
            var radians = Math.acos(point.x / point.length);
            radians = point.y > 0 ? radians : -radians;
            return toDegress(radians);
        }
        MUtils.getAngle = getAngle;
        function getVector(angle) {
            angle = toRadians(angle);
            return new engine.Point(Math.cos(angle), Math.sin(angle));
        }
        MUtils.getVector = getVector;
        function stringOverEllipsis(str, limit) {
            if (str.length > limit) {
                return str.substring(0, limit) + "...";
            }
            else {
                return str;
            }
        }
        MUtils.stringOverEllipsis = stringOverEllipsis;
        function getClassName(cls) {
            return cls["prototype"]["constructor"]["name"];
        }
        MUtils.getClassName = getClassName;
        function getInstanceClassName(instance) {
            return getClassName(instance["constructor"]);
        }
        MUtils.getInstanceClassName = getInstanceClassName;
        function setColorFilter(image, color) {
        }
        MUtils.setColorFilter = setColorFilter;
    })(MUtils || (MUtils = {}));

    function getBallScore(bulletScore, powerScore, colorIndex) {
        var getScoreFromRange = function (_a) {
            var a = _a[0], b = _a[1];
            a = Math.ceil(a * 0.1);
            b = Math.ceil(b * 0.1);
            var n = Math.abs(b - a) + 1;
            n = Math.floor(n / 6);
            var m = [];
            for (var i = 0; i < 5; i++) {
                m.push([a + n * i, a + n * i + n - 1]);
            }
            m.push([a + n * 4 + n, b]);
            var _b = m[colorIndex], a0 = _b[0], b0 = _b[1];
            var result = Math.ceil(Math.random() * (b0 - a0 + 1)) + a0;
            return result;
        };
        if (bulletScore <= 20)
            return getScoreFromRange([0.01 * powerScore, 0.2 * powerScore]);
        if (bulletScore > 20 && bulletScore <= 40) {
            return getScoreFromRange([0.2 * powerScore, 0.6 * powerScore]);
        }
        var getRange = function (list, props) {
            return getScoreFromRange(getProp(list, props)[0]);
        };
        if (bulletScore > 40 && bulletScore <= 60) {
            var list_1 = [
                [0.2 * powerScore, 0.6 * powerScore],
                [0.6 * powerScore, 0.8 * powerScore],
                [0.8 * powerScore, 1.5 * powerScore],
            ];
            return getRange(list_1, [3, 5, 2]);
        }
        if (bulletScore > 60 && bulletScore <= 80) {
            var list_2 = [
                [0.6 * powerScore, 0.8 * powerScore],
                [0.8 * powerScore, 1.5 * powerScore],
                [1.5 * powerScore, 2.2 * powerScore],
            ];
            return getRange(list_2, [2, 5, 2]);
        }
        if (bulletScore > 80 && bulletScore <= 100) {
            var list_3 = [
                [1.5 * powerScore, 2.2 * powerScore],
                [2.2 * powerScore, 3.2 * powerScore],
                [3.2 * powerScore, 4.2 * powerScore],
            ];
            return getRange(list_3, [1, 5, 4]);
        }
        if (bulletScore > 100 && bulletScore <= 120) {
            var list_4 = [
                [2.2 * powerScore, 3.2 * powerScore],
                [3.2 * powerScore, 4.2 * powerScore],
                [4.2 * powerScore, 5.2 * powerScore],
            ];
            return getRange(list_4, [1, 4, 5]);
        }
        var list = [
            [2.2 * powerScore, 3.2 * powerScore],
            [3.2 * powerScore, 4.2 * powerScore],
            [4.2 * powerScore, 5.2 * powerScore],
        ];
        return getRange(list, [1, 3, 6]);
    }
    function getProp(items, props, n) {
        if (n === void 0) { n = 1; }
        var pool = [];
        for (var i in items) {
            for (var j = 0; j < props[i]; j++) {
                pool.push(items[i]);
            }
        }
        var result = [];
        for (var i = 0; i < n; i++) {
            result.push(pool[MUtils.randomInt(0, pool.length)]);
        }
        return result;
    }
    function arrayRemove(array, target) {
        var newArray = [];
        for (var i = 0; i < array.length; i++) {
            if (array[i] !== target) {
                newArray.push(array[i]);
            }
        }
        return newArray;
    }

    var UILayer = (function (_super) {
        tslib.__extends(UILayer, _super);
        function UILayer() {
            var _this = _super.call(this) || this;
            _this.width = MConst.DesignResolution.width;
            _this.height = MConst.DesignResolution.height;
            return _this;
        }
        return UILayer;
    }(engine.Container));

    var DebugMgr = (function (_super) {
        tslib.__extends(DebugMgr, _super);
        function DebugMgr() {
            var _this = _super.call(this) || this;
            _this.enabled = true;
            _this.enabledGraphic = true;
            _this.shapeMap = {};
            _this.graphicDebugLayer = null;
            _this.uiLayer = null;
            _this.log = null;
            _this.test = null;
            _this.labelFPS = new engine.Label();
            _this.onClick = function () { };
            _this.lastFrameData = { runTime: 0, frameCount: 0 };
            _this.curFrameData = { runTime: 0, frameCount: 0 };
            _this.resetTimer = 0;
            if (!_this.enabled)
                return _this;
            _this.graphicDebugLayer = new UILayer();
            _this.addChild(_this.graphicDebugLayer);
            _this.uiLayer = new engine.Container();
            _this.addChild(_this.uiLayer);
            _this.labelFPS.size = 30;
            _this.labelFPS.fillColor = "rgba(255,255,255,1)";
            _this.labelFPS.text = "0";
            _this.labelFPS.x = 20;
            _this.labelFPS.y = 20;
            _this.uiLayer.addChild(_this.labelFPS);
            return _this;
        }
        DebugMgr.prototype.setLog = function (msg) {
            if (!this.enabled)
                return;
            this.log.text = msg;
        };
        DebugMgr.prototype.drawRect = function (rect, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = new engine.Shape();
            shape.beginStroke(style.color);
            shape.drawRect(rect.x, rect.y, rect.width, rect.height);
            shape.endStroke();
            this.graphicDebugLayer.addChild(shape);
        };
        DebugMgr.prototype.drawCircle = function (x, y, radius, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = new engine.Shape();
            shape.beginStroke(style.color);
            shape.drawCircle(x, y, radius);
            shape.endStroke();
            this.graphicDebugLayer.addChild(shape);
        };
        DebugMgr.prototype.drawLine = function (line, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = new engine.Shape();
            shape.beginStroke(style.color);
            shape.moveTo(line.startX, line.startY);
            shape.lineTo(line.endX, line.endY);
            shape.endStroke();
            this.graphicDebugLayer.addChild(shape);
        };
        DebugMgr.prototype.updateLine = function (key, line, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = this.shapeMap[key];
            if (!shape) {
                shape = new engine.Shape();
                this.graphicDebugLayer.addChild(shape);
                this.shapeMap[key] = shape;
            }
            shape.clear();
            shape.beginStroke(style.color);
            shape.moveTo(line.startX, line.startY);
            shape.lineTo(line.endX, line.endY);
            shape.endStroke();
        };
        DebugMgr.prototype.updateRect = function (key, rect, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = this.shapeMap[key];
            if (!shape) {
                shape = new engine.Shape();
                this.graphicDebugLayer.addChild(shape);
                this.shapeMap[key] = shape;
            }
            shape.clear();
            shape.beginStroke(style.color);
            shape.drawRect(rect.x, rect.y, rect.width, rect.height);
            shape.endStroke();
        };
        DebugMgr.prototype.updateCircle = function (key, x, y, radius, style) {
            if (style === void 0) { style = { color: 0x00ff00, lineSize: 1 }; }
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = this.shapeMap[key];
            if (!shape) {
                shape = new engine.Shape();
                this.graphicDebugLayer.addChild(shape);
                this.shapeMap[key] = shape;
            }
            shape.clear();
            shape.beginStroke(style.color);
            shape.drawCircle(x, y, radius);
            shape.endStroke();
        };
        DebugMgr.prototype.clearShape = function (key) {
            if (!this.enabled || !this.enabledGraphic)
                return;
            var shape = this.shapeMap[key];
            if (!shape) {
                return;
            }
            delete this.shapeMap[key];
            shape.dispose();
        };
        DebugMgr.prototype.init = function (context) {
            context.addChild(this);
            MTimer.onFrame("DebugMgr", this.onUpdate, this);
        };
        DebugMgr.prototype.onUpdate = function (dt) {
            this.curFrameData.runTime += dt;
            this.curFrameData.frameCount += 1;
            this.resetTimer += dt;
            var average = (this.curFrameData.runTime + this.lastFrameData.runTime) / (this.curFrameData.frameCount + this.lastFrameData.frameCount);
            if (this.resetTimer > 200) {
                this.labelFPS.text = Math.round(1000 / average).toString();
                this.lastFrameData.runTime = this.curFrameData.runTime;
                this.lastFrameData.frameCount = this.curFrameData.frameCount;
                this.curFrameData.runTime = 0;
                this.curFrameData.frameCount = 0;
                this.resetTimer = 0;
            }
        };
        Object.defineProperty(DebugMgr, "instance", {
            get: function () {
                if (!this._instance) {
                    this._instance = new DebugMgr();
                }
                return this._instance;
            },
            enumerable: true,
            configurable: true
        });
        DebugMgr._instance = null;
        return DebugMgr;
    }(UILayer));

    var instanceId1 = null;
    var instanceId2 = null;
    var PhycicsSystem = (function (_super) {
        tslib.__extends(PhycicsSystem, _super);
        function PhycicsSystem() {
            var _this = _super.call(this, "PhycicsSystem") || this;
            _this.phycicsList = [];
            _this.colliderList = [];
            _this.collisions = [];
            return _this;
        }
        PhycicsSystem.prototype.onInit = function () {
            _super.prototype.onInit.call(this);
            for (var i = 0; i <= GroupMaxIndex; i++) {
                this.colliderList[i] = [];
            }
        };
        PhycicsSystem.prototype.onDestroy = function () {
            _super.prototype.onDestroy.call(this);
            for (var i = 0; i <= GroupMaxIndex; i++) {
                this.colliderList[i] = [];
            }
            this.phycicsList = [];
        };
        PhycicsSystem.prototype.onUpdate = function () {
            if (this.pause)
                return;
            for (var _i = 0, _a = this.phycicsList; _i < _a.length; _i++) {
                var i = _a[_i];
                i.onFixedUpdate(MTimer.dtFactor);
            }
            this.detectCollision();
            return false;
        };
        PhycicsSystem.prototype.add = function (phycics) {
            this.phycicsList.push(phycics);
        };
        PhycicsSystem.prototype.remove = function (phycics) {
            this.phycicsList = arrayRemove(this.phycicsList, phycics);
        };
        PhycicsSystem.prototype.getCollisionIndex = function (collider1, collider2) {
            for (var i = 0; i < this.collisions.length; i++) {
                if ((this.collisions[i][0] == collider1 && this.collisions[i][1] == collider2) || (this.collisions[i][0] == collider2 && this.collisions[i][1] == collider1)) {
                    return i;
                }
            }
            return -1;
        };
        PhycicsSystem.prototype.addCollider = function (collider) {
            if (typeof this.colliderList[collider.group] == "undefined") {
                this.colliderList[collider.group] = [];
            }
            this.colliderList[collider.group].push(collider);
        };
        PhycicsSystem.prototype.removeCollider = function (collider) {
            this.collisions = this.collisions.filter(function (e) {
                if (e[0] != collider.owner.instanceId && e[1] != collider.owner.instanceId) {
                    return true;
                }
                else
                    return false;
            });
            this.colliderList[collider.group] = arrayRemove(this.colliderList[collider.group], collider);
        };
        PhycicsSystem.prototype.detectCollision = function () {
            var i = 0, j = 0, m = 0, n = 0, length = this.colliderList.length, group1 = null, group2 = null;
            for (i = 0; i <= length - 1; i++) {
                for (j = i + 1; j <= length - 1; j++) {
                    if (CollisionMap[i] & 1 << j) {
                        group1 = this.colliderList[i];
                        group2 = this.colliderList[j];
                        for (m = 0; m <= group1.length - 1; m++) {
                            for (n = 0; n <= group2.length - 1; n++) {
                                this.detectTraverse(group1[m], group2[n]);
                            }
                        }
                    }
                }
            }
        };
        PhycicsSystem.prototype.detectTraverse = function (collider1, collider2) {
            instanceId1 = collider1.owner.instanceId;
            instanceId2 = collider2.owner.instanceId;
            var collisionIndex = this.getCollisionIndex(instanceId1, instanceId2);
            if (callDetectFunc(collider1, collider2)) {
                if (collisionIndex >= 0) {
                    collider1.owner.onCollisionStay(collider2);
                    collider2.owner.onCollisionStay(collider1);
                }
                else {
                    this.collisions.push([instanceId1, instanceId2]);
                    collider1.owner.onCollisionEnter(collider2);
                    collider2.owner.onCollisionEnter(collider1);
                }
            }
            else {
                if (collisionIndex >= 0) {
                    this.collisions.splice(collisionIndex, 1);
                    collider1.owner.onCollisionExit(collider2);
                    collider2.owner.onCollisionExit(collider1);
                }
            }
        };
        PhycicsSystem.prototype.drawCollider = function (collider) {
            switch (collider.type) {
                case ColliderType.Circle:
                    var circleCollider = collider;
                    var worldCenter = [];
                    collider.getWorldPosition(worldCenter);
                    DebugMgr.instance.updateCircle(collider.owner.instanceId.toString(), worldCenter[0], worldCenter[1], circleCollider.radius);
                    break;
                case ColliderType.Rect:
                    var rectCollider = collider;
                    var worldPos = [];
                    rectCollider.getWorldPosition(worldPos);
                    DebugMgr.instance.updateRect(rectCollider.owner.instanceId.toString(), new engine.Rectangle(worldPos[0], worldPos[1], rectCollider.rect.width, rectCollider.rect.height));
                    break;
            }
        };
        return PhycicsSystem;
    }(GameMgr));
    function callDetectFunc(collider1, collider2) {
        if (collider1.type == ColliderType.Circle) {
            if (collider2.type == ColliderType.Circle) {
                return circleToCircle(collider1, collider2);
            }
            else if (collider2.type == ColliderType.Rect) {
                return circleToRect(collider1, collider2);
            }
            else if (collider2.type == ColliderType.Point) {
                return circleToPoint(collider1, collider2);
            }
        }
        else if (collider1.type == ColliderType.Rect) {
            if (collider2.type == ColliderType.Circle) {
                return circleToRect(collider2, collider1);
            }
            else if (collider2.type == ColliderType.Rect) {
                return rectToRect(collider1, collider2);
            }
            else if (collider2.type == ColliderType.Point) {
                return pointToRect(collider2, collider1);
            }
        }
        else if (collider1.type == ColliderType.Point) {
            if (collider2.type == ColliderType.Circle) {
                return circleToPoint(collider2, collider1);
            }
            else if (collider2.type == ColliderType.Rect) {
                return pointToRect(collider1, collider2);
            }
        }
        return false;
    }
    var tempPoint1 = [];
    var tempPoint2 = [];
    function distance(p1, p2) {
        return Math.sqrt((p1[0] - p2[0]) * (p1[0] - p2[0]) + (p1[1] - p2[1]) * (p1[1] - p2[1]));
    }
    function circleToCircle(collider1, collider2) {
        collider1.getWorldPosition(tempPoint1);
        collider2.getWorldPosition(tempPoint2);
        return distance(tempPoint1, tempPoint2) < (collider1.radius + collider2.radius);
    }
    function circleToRect(circleCollider, rectCollider) {
        circleCollider.getWorldPosition(tempPoint1);
        rectCollider.getWorldPosition(tempPoint2);
        var closestPoint = new engine.Point();
        if (tempPoint1[1] < tempPoint2[1]) {
            closestPoint.y = tempPoint2[1];
        }
        else if (tempPoint1[1] > tempPoint2[1] + rectCollider.rect.height) {
            closestPoint.y = tempPoint2[1] + rectCollider.rect.height;
        }
        else {
            closestPoint.y = tempPoint1[1];
        }
        if (tempPoint1[0] < tempPoint2[0]) {
            closestPoint.x = tempPoint2[0];
        }
        else if (tempPoint1[0] > tempPoint2[0] + rectCollider.rect.width) {
            closestPoint.x = tempPoint2[0] + rectCollider.rect.width;
        }
        else {
            closestPoint.x = tempPoint1[0];
        }
        return distance([closestPoint.x, closestPoint.y], tempPoint1) < circleCollider.radius;
    }
    function rectToRect(collider1, collider2) {
        var rect1 = collider1.rect, rect2 = collider2.rect;
        collider1.getWorldPosition(tempPoint1);
        collider2.getWorldPosition(tempPoint2);
        var rectW1 = tempPoint1[0] + rect1.width, rectW2 = tempPoint2[0] + rect2.width, rectH1 = rect1.height + tempPoint1[1], rectH2 = tempPoint2[1] + rect2.height;
        if (tempPoint1[0] <= rectW2 &&
            rectW1 >= tempPoint2[0] &&
            tempPoint1[1] <= rectH2 &&
            rectH1 >= tempPoint2[1]) {
            return true;
        }
        else
            return false;
    }
    function circleToPoint(circle, point) {
        circle.getWorldPosition(tempPoint1);
        point.getWorldPosition(tempPoint2);
        if (distance(tempPoint1, tempPoint2) < circle.radius) {
            return true;
        }
        else {
            return false;
        }
    }
    function pointToRect(point, rect) {
        rect.getWorldPosition(tempPoint1);
        point.getWorldPosition(tempPoint2);
        if (tempPoint2[1] < tempPoint1[1] + rect.rect.height
            && tempPoint2[1] > tempPoint1[1]
            && tempPoint2[0] > tempPoint1[0]
            && tempPoint2[0] < tempPoint1[0] + rect.rect.width) {
            return true;
        }
        else {
            return false;
        }
    }
    var CollisionMap = {
        0: 0,
        1: 92,
        2: 130,
        3: 98,
        4: 34,
        5: 88,
        6: 42,
        7: 2
    };
    var GroupMaxIndex = 7;

    var PoolGroup = (function () {
        function PoolGroup(layer) {
            this.layer = layer;
            this.data = {};
        }
        PoolGroup.prototype.recycle = function (key, element) {
            this.data[key].recycle(element);
        };
        return PoolGroup;
    }());
    var Pool = (function () {
        function Pool(layer) {
            this.layer = layer;
            this.data = [];
        }
        Pool.prototype.recycle = function (element) {
            element.onElementRecycle();
            this.data.push(element);
        };
        return Pool;
    }());

    var padding = 10;
    var GuideMask = (function (_super) {
        tslib.__extends(GuideMask, _super);
        function GuideMask(index, x1, y1) {
            var _this = _super.call(this) || this;
            var config = MConfigs.guide[index];
            var mask = new engine.Sprite(getTexture("3cc0aa86-0581-4caa-b662-266e58fd39b7"));
            var w = mask.width, h = mask.height;
            x1 = Math.floor(x1);
            y1 = Math.floor(y1);
            var x = x1 - w / 2;
            var y = y1 - h / 2;
            mask.x = x;
            mask.y = y;
            _this.addChild(mask);
            var sw = MConst.DesignResolution.width, sh = MConst.DesignResolution.height;
            var color = 0x000000;
            var alpha = 0.7;
            var top = new engine.Shape();
            top.beginFill(color, alpha);
            top.drawRect(0, -sh, sw, y + sh);
            top.endFill();
            _this.addChild(top);
            var bottom = new engine.Shape();
            bottom.beginFill(color, alpha);
            bottom.drawRect(x, y + h, w, sh - h - y);
            bottom.endFill();
            _this.addChild(bottom);
            var left = new engine.Shape();
            left.beginFill(color, alpha);
            left.drawRect(0, y, x, sh - y);
            left.endFill();
            _this.addChild(left);
            var right = new engine.Shape();
            right.beginFill(color, alpha);
            right.drawRect(x + w, y, sw - x - w, sh - y);
            right.endFill();
            _this.addChild(right);
            var offsetToMask = config.text.offsetToMask;
            var label = new engine.Label();
            label.text = props['guideText' + index];
            label.size = 24;
            label.fillColor = "rgba(255,255,255,1)";
            label.x = mask.x + (w - label.width) / 2;
            label.y = offsetToMask.y + mask.y;
            _this.addChild(label);
            if (index == 0) {
                var left_1 = new engine.Sprite(getTexture("93992a62-e3c4-46d5-99f4-f7ba35fe4f4f"));
                left_1.x = x1 + 200;
                left_1.y = y1 - left_1.height / 2;
                _this.addChild(left_1);
                var right_1 = new engine.Sprite(getTexture("93992a62-e3c4-46d5-99f4-f7ba35fe4f4f"));
                right_1.x = x1 - 200;
                right_1.scaleX = -1;
                right_1.y = y1 - right_1.height / 2;
                _this.addChild(right_1);
            }
            if (label.x < padding) {
                label.x = padding;
            }
            else if (label.x + label.width > 750 - padding) {
                label.x = 750 - label.width - padding;
            }
            _this.addEventListener(engine.MouseEvent.MOUSE_DOWN, function () {
                _this.onClick && _this.onClick(_this);
            }, _this);
            return _this;
        }
        return GuideMask;
    }(engine.Container));

    var GuideMgr = (function () {
        function GuideMgr() {
            this.done = [];
            this.currentGuideMask = null;
        }
        Object.defineProperty(GuideMgr, "instance", {
            get: function () {
                if (!GuideMgr._instance) {
                    GuideMgr._instance = new GuideMgr();
                }
                return GuideMgr._instance;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GuideMgr.prototype, "guideFlag", {
            get: function () {
                return localStorage.getItem("shoot-planet_" + props.guideFlagKey) != "false";
            },
            set: function (v) {
                localStorage.setItem("shoot-planet_" + props.guideFlagKey, v ? "true" : "false");
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GuideMgr.prototype, "container", {
            set: function (c) {
                this._container = c;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GuideMgr.prototype, "guideDone", {
            get: function () {
                return !this.guideFlag || this.done.length >= Object.keys(MConfigs.guide).length;
            },
            enumerable: true,
            configurable: true
        });
        GuideMgr.prototype.runGuide = function (index, x, y, end) {
            var _this = this;
            if (end === void 0) { end = false; }
            if (this.done[index] || this.guideFlag == false)
                return;
            this.done.push(index);
            DataMgr.game.pause = true;
            this.currentGuideMask = new GuideMask(index, x, y);
            this.currentGuideMask.onClick = function (context) {
                context.dispose();
                DataMgr.game.pause = false;
                if (_this.done.length >= Object.keys(MConfigs.guide).length) {
                    DataMgr.game.onGuideDone();
                }
                if (end) {
                    _this.guideFlag = false;
                }
            };
            this._container.addChild(this.currentGuideMask);
        };
        GuideMgr._instance = null;
        return GuideMgr;
    }());

    var tempPower = null;
    var imageNames = [
        "39843f75-8caa-4cee-bff5-91fb759b691b",
        "bd94b3e6-3c99-490f-95ef-ca0c056d7727",
        "d8eab9fc-104e-4b39-88d6-455ed8632703",
        "48f98d84-d30e-4dfe-aa92-c45bde5cefd8",
        "8d42fd9e-907b-4587-9a6f-a634909b8b8a",
    ];
    var textureSize = {
        width: 210,
        height: 213
    };
    var Ball = (function (_super) {
        tslib.__extends(Ball, _super);
        function Ball(sizeIndex) {
            var _this = _super.call(this) || this;
            _this.colorIndex = 0;
            _this.sizeIndex = 0;
            _this.bitmap = null;
            _this.labelScore = null;
            _this.scaleRatio = 1;
            _this.initScore = null;
            _this.initColorIndex = 0;
            _this.collider = null;
            _this.isBornStage = false;
            _this.poolKey = null;
            _this._score = 0;
            _this.dir = 1;
            _this.scoreUpdateFlag = false;
            var scaleRatio = MConfigs.size[sizeIndex];
            var bitmapWidth = textureSize.width * scaleRatio;
            var bitmapHeight = textureSize.height * scaleRatio;
            _this.bitmap = new engine.Sprite();
            _this.addChild(_this.bitmap);
            _this.bitmap.width = bitmapWidth;
            _this.bitmap.height = bitmapHeight;
            _this.anchorX = bitmapWidth / 2;
            _this.anchorY = bitmapHeight / 2;
            _this.bitmap.x = -bitmapWidth / 2;
            _this.bitmap.y = -bitmapHeight / 2;
            var labelScore = new engine.BitmapText();
            labelScore.font = props.fontName;
            labelScore.scaleX = labelScore.scaleY = scaleRatio;
            _this.addChild(labelScore);
            var collider = _this.addComponent(CircleCollider);
            collider.group = ColliderGroup.Ball;
            _this.collider = collider;
            collider.setData(0, 0, bitmapHeight / 2);
            _this.scaleRatio = scaleRatio;
            _this.sizeIndex = sizeIndex;
            _this.labelScore = labelScore;
            return _this;
        }
        Object.defineProperty(Ball.prototype, "score", {
            get: function () {
                return this._score;
            },
            set: function (v) {
                if (v < 0)
                    v = 0;
                if (this._score != v) {
                    this.scoreUpdateFlag = true;
                }
                this._score = v;
            },
            enumerable: true,
            configurable: true
        });
        Ball.prototype.onScoreIsZero = function () {
            if (GuideMgr.instance.guideFlag == true) {
                this.drop();
            }
            else {
                if (this.sizeIndex > 0) {
                    this.split();
                }
                else {
                    this.drop();
                }
            }
            DataMgr.game._playBoomEffect(this.position, this.scaleRatio);
            SoundMgr.instance.playEffect("boom");
            DataMgr.game._pool.ball.recycle(this.poolKey, this);
        };
        Ball.prototype.init = function (colorIndex, direction, score) {
            this.colorIndex = colorIndex;
            this.initColorIndex = colorIndex;
            this.bitmap.texture = getTexture(imageNames[colorIndex]);
            this.initScore = score;
            this.score = score;
            this.physics.rotateVelocity = direction * MConst.BallRotateSpeed * Math.pow(1 / this.scaleRatio, 0.5);
            return this;
        };
        Ball.prototype.startBornStage = function (dir) {
            this.isBornStage = true;
            if (dir == 1) {
                this.x = 0 - this.width / 2 - 10;
            }
            else if (dir == -1) {
                this.x = MConst.DesignResolution.width + this.width / 2 + 10;
            }
            this.y = MConst.BallInitPosY;
            this.physics.velocity.x = this.getRandomVelocityX(dir) + 1 * dir;
            this.dir = dir;
            this.physics.onMoved = this.onBornStageMoved.bind(this);
        };
        Ball.prototype.onBornStageMoved = function (owner) {
            if (this.dir == 1) {
                if (this.x > this.width / 2) {
                    this.startGravityStage(this.dir);
                    this.physics.onMoved = null;
                }
            }
            else if (this.dir == -1) {
                if (this.x < MConst.DesignResolution.width - this.width / 2) {
                    this.startGravityStage(this.dir);
                    this.physics.onMoved = null;
                }
            }
        };
        Ball.prototype.getRandomVelocityX = function (direction) {
            return direction * MConst.BallVelocityX * (1 + ((1 / this.scaleRatio) - 1) * MUtils.random(0, MConst.BallVelocityXRandomFactor));
        };
        Ball.prototype.startGravityStage = function (direction) {
            this.isBornStage = false;
            this.physics.velocity.x = this.getRandomVelocityX(direction);
            this.physics.acceleration.y = MConst.Gravity;
        };
        Ball.prototype.onCollisionEnter = function (other) {
            if (other.group == ColliderGroup.Bullet) {
                tempPower = other.owner.power;
                this.score = this.score - tempPower;
                DataMgr.game._score += tempPower;
            }
            if (other.group == ColliderGroup.Ground) {
                this.physics.velocity.y = -(MConst.BallVelocityY * (1 + ((1 / this.scaleRatio) - 1) * MUtils.random(0, MConst.BallVelocityYRandomFactor)));
                var clip = DataMgr.game._createAnimation("duang");
                clip.x = this.x - 140 * this.scaleRatio;
                clip.y = this.y - 50 * this.scaleRatio;
                clip.scaleX = clip.scaleY = this.scaleRatio;
                clip.play();
                if (this.sizeIndex >= MConfigs.size.length - 2) {
                    DataMgr.game._shake();
                    SoundMgr.instance.playEffect("dong");
                }
            }
            if (other.group == ColliderGroup.Wall && !this.isBornStage) {
                this.physics.velocity.x = -this.physics.velocity.x;
                this.physics.rotateVelocity = -this.physics.rotateVelocity;
            }
        };
        Ball.prototype.onCollisionStay = function (other) {
            if (other.group == ColliderGroup.Wall && !this.isBornStage) {
                var dir = other.owner.x < 0 ? 1 : -1;
                this.physics.velocity.x = dir * Math.abs(this.physics.velocity.x);
                this.physics.rotateVelocity = dir * Math.abs(this.physics.rotateVelocity);
            }
        };
        Ball.prototype.onElementRecycle = function () {
            this.visible = false;
            this.physics.velocity.y = 0;
            this.physics.velocity.x = 0;
            this.physics.acceleration.y = 0;
            this.physics.acceleration.x = 0;
            this.disableAllComponents();
        };
        Ball.prototype.onElementInit = function () {
            this.visible = true;
            this.enableAllComponents();
        };
        Ball.prototype.split = function () {
            var _this = this;
            var sizeIndex = this.sizeIndex - 1;
            var score = Math.ceil(this.initScore / 2 * (Math.random() * 0.4 + 0.8));
            var colorIndex = Math.max(this.initColorIndex - 1, 0);
            var callback = function (direction) {
                var ball = DataMgr.game._pool.ball.spwan(sizeIndex).init(colorIndex, direction, score);
                ball.x = _this.x;
                ball.y = _this.y;
                ball.startGravityStage(direction);
                ball.physics.velocity.y = -MConst.BallSplitVelocityY;
            };
            callback(1);
            callback(-1);
        };
        Ball.prototype.updateScoreLabel = function () {
            if (this.scoreUpdateFlag == true) {
                var labelScore = this.labelScore;
                var score = this.score;
                if (score >= 1000) {
                    labelScore.text = (Math.floor(score / 100) / 10).toString() + "k";
                }
                else {
                    labelScore.text = score.toString();
                }
                labelScore.x = -labelScore.width / 2;
                labelScore.y = -labelScore.height / 2;
                this.scoreUpdateFlag = false;
                if (this.score <= 0) {
                    this.onScoreIsZero();
                }
            }
        };
        Ball.prototype.drop = function () {
            var index = 1;
            var bulletScore = DataMgr.game.bulletScore;
            if (bulletScore <= 30)
                index = 0;
            else if (bulletScore > 30 && bulletScore <= 70)
                index = 1;
            else if (bulletScore > 70)
                index = 2;
            var config = MConfigs.dropPool[index];
            var keys = Object.keys(config.factor);
            var values = keys.map(function (e) { return config.factor[e]; });
            var dropIds = getProp(keys, values, config.num);
            var drops = [];
            for (var _i = 0, dropIds_1 = dropIds; _i < dropIds_1.length; _i++) {
                var id = dropIds_1[_i];
                drops.push(DataMgr.game._pool.drop.spwan(id.toString()));
            }
            var dir = this.x > MConst.DesignResolution.width / 2 ? -1 : 1;
            var _loop_1 = function (i) {
                var drop = drops[i];
                drop.x = this_1.x - drop.bitmap.width / 2;
                drop.y = this_1.y - drop.bitmap.height / 2;
                var offsetRatio = (i - 1) + (2 - (drops.length - 1));
                var x = MConst.DropVelocityX.x + offsetRatio * MConst.DropVelocityX.offset;
                x *= MUtils.random(1 - MConst.DropRandomFactor, 1 + MConst.DropRandomFactor);
                drop.physics.velocity.x = dir * x;
                drop.physics.velocity.y = -(x * 5 * MUtils.random(0.9, 1.1));
                if (GuideMgr.instance.guideFlag && i == 1) {
                    MTimer.setFrameTimer(20, function () {
                        GuideMgr.instance.runGuide(1, drop.x, drop.y, true);
                    });
                }
            };
            var this_1 = this;
            for (var i = 0; i < drops.length; i++) {
                _loop_1(i);
            }
        };
        Ball.textures = [];
        return Ball;
    }(MoveObjcet));

    var BallPool = (function (_super) {
        tslib.__extends(BallPool, _super);
        function BallPool() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        BallPool.prototype.spwan = function (sizeIndex) {
            var element = null;
            if (this.data.length > 0) {
                element = this.data.shift();
                element.onElementInit();
            }
            else {
                element = new Ball(sizeIndex);
                this.layer.addChild(element);
            }
            DataMgr.game._ballList.push(element);
            return element;
        };
        BallPool.prototype.recycle = function (element) {
            _super.prototype.recycle.call(this, element);
            DataMgr.game._ballList = DataMgr.game._ballList.filter(function (e) { return e !== element; });
        };
        return BallPool;
    }(Pool));
    var BallPoolGroup = (function (_super) {
        tslib.__extends(BallPoolGroup, _super);
        function BallPoolGroup() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        BallPoolGroup.prototype.spwan = function (sizeIndex) {
            var key = sizeIndex.toString();
            if (!this.data[key]) {
                var ball = this.data[key] = new BallPool(this.layer);
            }
            var element = this.data[key].spwan(sizeIndex);
            element.poolKey = key;
            return element;
        };
        return BallPoolGroup;
    }(PoolGroup));

    var FrameAnimationMgr = (function (_super) {
        tslib.__extends(FrameAnimationMgr, _super);
        function FrameAnimationMgr() {
            var _this = _super.call(this, "FrameAnimationMgr") || this;
            _this.playingAnimations = [];
            return _this;
        }
        FrameAnimationMgr.prototype.onUpdate = function () {
            if (this.pause)
                return;
            var filters = [];
            var anim = null;
            for (var i = 0; i < this.playingAnimations.length; i++) {
                anim = this.playingAnimations[i];
                anim.onUpdate();
                if (anim.curPos >= anim.frameCount) {
                    anim.onCompleted && anim.onCompleted();
                }
                else {
                    filters.push(anim);
                }
            }
            this.playingAnimations = filters;
        };
        return FrameAnimationMgr;
    }(GameMgr));
    var FrameAnimation = (function (_super) {
        tslib.__extends(FrameAnimation, _super);
        function FrameAnimation(key) {
            var _this = _super.call(this) || this;
            _this.frames = [];
            _this.frameCount = 0;
            _this.curPos = 0;
            _this.frameTimer = 0;
            _this.frameRate = 60;
            _this.sprite = new engine.Sprite;
            if (!FrameAnimation.caches[key]) {
                console.error("unloaded animation:" + key);
                return _this;
            }
            _this.frames = FrameAnimation.caches[key].frames;
            _this.frameCount = Object.keys(_this.frames).length;
            _this.frameRate = FrameAnimation.caches[key].frameRate;
            _this.addChild(_this.sprite);
            return _this;
        }
        FrameAnimation.loadRes = function (texture, egretData) {
            var name = null;
            var mc = egretData.mc;
            for (var key in mc) {
                name = key;
                break;
            }
            if (!FrameAnimation.caches[name]) {
                var altaData = {};
                for (var i in egretData.res) {
                    altaData[name + "_" + i] = egretData.res[i];
                }
                engine.createTextureSheet(texture.baseTexture, altaData);
                var frames = [];
                var framesData = mc[name].frames;
                for (var i in framesData) {
                    frames[i] = {
                        x: framesData[i].x,
                        y: framesData[i].y,
                        texture: getTexture(name + "_" + framesData[i].res)
                    };
                }
                FrameAnimation.caches[name] = {
                    frameRate: mc[name].frameRate,
                    frameCount: Object.keys(altaData).length,
                    frames: frames
                };
            }
        };
        FrameAnimation.prototype.onUpdate = function () {
            if (this.frameTimer >= 60 / this.frameRate) {
                if (this.curPos >= this.frameCount) {
                    return;
                }
                else {
                    var frameData = this.frames[this.curPos];
                    this.sprite.texture = frameData.texture;
                    this.sprite.x = frameData.x;
                    this.sprite.y = frameData.y;
                }
                this.curPos++;
                this.frameTimer = 0;
            }
            else {
                this.frameTimer++;
            }
        };
        FrameAnimation.prototype.play = function () {
            DataMgr.game._animationMgr.playingAnimations.push(this);
        };
        FrameAnimation.caches = {};
        return FrameAnimation;
    }(engine.Container));

    var PoolFrameAnimation = (function (_super) {
        tslib.__extends(PoolFrameAnimation, _super);
        function PoolFrameAnimation() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        PoolFrameAnimation.prototype.onElementInit = function () {
            this.visible = true;
            this.curPos = 0;
        };
        PoolFrameAnimation.prototype.onElementRecycle = function () {
            this.visible = false;
        };
        return PoolFrameAnimation;
    }(FrameAnimation));
    var AnimationPool = (function (_super) {
        tslib.__extends(AnimationPool, _super);
        function AnimationPool() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        AnimationPool.prototype.spwan = function (key) {
            if (this.data.length > 0) {
                var element = this.data.shift();
                element.onElementInit();
                return element;
            }
            else {
                var anim = new PoolFrameAnimation(key);
                this.layer.addChild(anim);
                return anim;
            }
        };
        return AnimationPool;
    }(Pool));
    var AnimationPoolGroup = (function (_super) {
        tslib.__extends(AnimationPoolGroup, _super);
        function AnimationPoolGroup() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        AnimationPoolGroup.prototype.spwan = function (key) {
            if (!this.data[key]) {
                this.data[key] = new AnimationPool(this.layer);
            }
            var element = this.data[key].spwan(key);
            return element;
        };
        return AnimationPoolGroup;
    }(PoolGroup));

    var BulletPool = (function (_super) {
        tslib.__extends(BulletPool, _super);
        function BulletPool() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        BulletPool.prototype.spwan = function () {
            if (this.data.length > 0) {
                var element = this.data.shift();
                element.onElementInit();
                return element;
            }
            else {
                var b = new Bullet();
                this.layer.addChild(b);
                return b;
            }
        };
        return BulletPool;
    }(Pool));

    var TextureMgr;
    (function (TextureMgr) {
        var uuids = {
            "star": [
                "e79f4e51-dbca-45fb-b95a-f52a708e0990",
                "1a1b529f-bf68-42cb-8a3e-01b21119b322",
                "999d65ba-f7c8-4939-9868-5f974a278752",
                "c5743fdf-aadf-4139-b03a-cfd047f85220",
                "2aaa2535-d4b9-44c4-8fc2-700c8310e534"
            ],
            "ring": [
                "b3c78176-5650-4d49-bc00-cfaf048c5940",
                "134e2eec-e393-4ba3-b8cc-f72270bf17c2",
                "49092176-3b1c-4ea5-8e02-6050fc5809dd",
                "15c46813-ac79-4c5f-8a52-e8cded6dadff",
                "64e64fd6-0426-493e-b003-dcafc7e06eb5"
            ],
            "fireworks_line_0": [
                "c76f6d87-7f84-45ed-a9ac-cf8d4dd47026",
                "1fc4fb96-73b7-4250-8886-3c6976eb1139",
                "65be7718-f6cd-44d9-89d0-5daf81068316"
            ],
            "fireworks_line_1": [
                "10901339-4976-4d38-a3b9-47b7e9025144",
                "0692b423-6f14-4ba7-884a-7c177f1583c8",
                "2ddd9477-12de-4f13-bb35-5b255f1624bd"
            ]
        };
        function get(type, index) {
            var uuid = uuids[type][index];
            return getTexture(uuid);
        }
        TextureMgr.get = get;
    })(TextureMgr || (TextureMgr = {}));

    var InitSpeedValue = 0.08;
    var alphaDuration = 300;
    var alphaPerFrame = 1 / (alphaDuration / (1000 / 60));
    var noAlphaDuration = 200 / (1000 / 60);
    var Line = (function () {
        function Line(parent, type, color, rotation) {
            this.sprite = null;
            this.direction = null;
            this.speedValue = InitSpeedValue;
            this.frameTimer = 0;
            var sprite = new engine.Sprite(TextureMgr.get(("fireworks_line_" + type), color));
            parent.addChild(sprite);
            sprite.anchorX = sprite.width / 2;
            sprite.rotation = rotation;
            this.sprite = sprite;
            var dir = MUtils.getVector(rotation + 90);
            this.direction = dir;
        }
        Line.prototype.init = function () {
            this.sprite.y = -this.sprite.height;
            this.sprite.alpha = 1;
            this.sprite.x = 0;
            this.sprite.y = 0;
            this.frameTimer = 0;
            this.speedValue = InitSpeedValue;
        };
        Line.prototype.onUpdate = function (dt) {
            this.sprite.x += this.direction.x * this.speedValue * dt;
            this.sprite.y += this.direction.y * this.speedValue * dt;
            if (this.frameTimer > noAlphaDuration) {
                this.sprite.alpha -= alphaPerFrame * MTimer.dtFactor;
            }
            this.frameTimer++;
        };
        return Line;
    }());
    var SpBoomEffectMgr = (function (_super) {
        tslib.__extends(SpBoomEffectMgr, _super);
        function SpBoomEffectMgr() {
            var _this = _super.call(this, "SpBoomEffectMgr") || this;
            _this.list = [];
            return _this;
        }
        SpBoomEffectMgr.prototype.onUpdate = function (dt) {
            if (this.pause)
                return;
            var filter = [];
            for (var _i = 0, _a = this.list; _i < _a.length; _i++) {
                var i = _a[_i];
                if (i.onUpdate(dt)) {
                    filter.push(i);
                }
            }
            this.list = filter;
        };
        return SpBoomEffectMgr;
    }(GameMgr));
    var SpBoomEffect = (function (_super) {
        tslib.__extends(SpBoomEffect, _super);
        function SpBoomEffect(color, size) {
            var _this = _super.call(this) || this;
            _this.lines = [];
            _this.scaleX = _this.scaleY = 0.5 * size + 0.5;
            _this.rotation = MUtils.random(0, 90);
            for (var i = 0; i < 16; i++) {
                var line = new Line(_this, i % 2, color, i * 22.5);
                _this.lines.push(line);
            }
            DataMgr.game._boomEffectMgr.list.push(_this);
            return _this;
        }
        SpBoomEffect.prototype.onUpdate = function (dt) {
            for (var _i = 0, _a = this.lines; _i < _a.length; _i++) {
                var i = _a[_i];
                i.onUpdate(dt);
            }
            if (this.lines[0].sprite.alpha <= 0) {
                DataMgr.game._pool.boomEffect.recycle(this.poolKey, this);
                return false;
            }
            return true;
        };
        SpBoomEffect.prototype.onElementInit = function () {
            this.visible = true;
            this.alpha = 1;
            for (var _i = 0, _a = this.lines; _i < _a.length; _i++) {
                var i = _a[_i];
                i.init();
            }
            DataMgr.game._boomEffectMgr.list.push(this);
        };
        SpBoomEffect.prototype.onElementRecycle = function () {
            this.visible = false;
        };
        return SpBoomEffect;
    }(engine.Container));
    var SpBoomEffectPool = (function (_super) {
        tslib.__extends(SpBoomEffectPool, _super);
        function SpBoomEffectPool() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        SpBoomEffectPool.prototype.spwan = function (color, size) {
            if (this.data.length > 0) {
                var element = this.data.shift();
                element.onElementInit();
                return element;
            }
            else {
                var e = new SpBoomEffect(color, size);
                this.layer.addChild(e);
                return e;
            }
        };
        return SpBoomEffectPool;
    }(Pool));
    var SpBoomEffectPoolGroup = (function (_super) {
        tslib.__extends(SpBoomEffectPoolGroup, _super);
        function SpBoomEffectPoolGroup() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        SpBoomEffectPoolGroup.prototype.spwan = function (color, size) {
            var key = color.toString() + size;
            if (!this.data[key]) {
                this.data[key] = new SpBoomEffectPool(this.layer);
            }
            var element = this.data[key].spwan(color, size);
            element.poolKey = key;
            return element;
        };
        return SpBoomEffectPoolGroup;
    }(PoolGroup));

    var dropImgNames = [
        "f7221f86-f376-40ce-b0e8-f7ea573ec780",
        "c9dbd728-1844-4910-b1ef-e84ce10dc422",
        "41182131-7050-4b95-9dce-45275b70738d",
        "17d8e000-b290-4972-bcab-cb819a6cf51b",
        "3f7faebf-64da-44a4-8152-0c35fcc29fe5"
    ];
    var Drop = (function (_super) {
        tslib.__extends(Drop, _super);
        function Drop(id) {
            var _this = _super.call(this) || this;
            _this.type = null;
            _this.label = null;
            _this.bitmap = null;
            _this.scoreValue = null;
            _this.collider = null;
            _this.isGround = false;
            _this.stayGroundCount = 0;
            _this.id = null;
            _this.id = id;
            var bitmap = new engine.Image();
            _this.addChild(bitmap);
            _this.bitmap = bitmap;
            var label = new engine.Label();
            label.size = 24;
            _this.addChild(label);
            _this.label = label;
            _this.collider = _this.addComponent(PointCollider);
            _this.collider.group = ColliderGroup.Drop;
            var config = MConfigs.drop[id];
            _this.type = config.type;
            _this.scoreValue = config.value;
            _this.physics.acceleration.y = 0.3;
            var texture = getTexture(dropImgNames[id]);
            _this.bitmap.texture = texture;
            var str = "+" + config.value;
            if (config.type == "power")
                str = str + "%";
            label.text = str;
            label.fillColor = config.textColor;
            label.visible = false;
            label.x = texture.width + 2;
            label.y = texture.height / 2 - label.height / 2;
            _this.collider.setData(texture.width / 2, texture.height / 2);
            return _this;
        }
        Drop.prototype.eated = function () {
            var _this = this;
            SoundMgr.instance.playEffect("collect");
            this.physics.rotateVelocity = 0;
            DataMgr.game._dropBlinkMgr.remove(this);
            this.visible = true;
            this.rotation = 0;
            this.physics.enabled = false;
            this.getComponent(PointCollider).enabled = false;
            this.label.visible = true;
            if (this.type == "bullet") {
                DataMgr.game._BulletScore += this.scoreValue;
            }
            else if (this.type == "power") {
                DataMgr.game._PowerScore += this.scoreValue;
            }
            engine.Tween.removeTweens(this);
            engine.Tween.get(this)
                .to({ y: this.y - 200, alpha: 0 }, 1000)
                .call(function () {
                DataMgr.game._pool.drop.recycle(_this.id.toString(), _this);
                engine.Tween.removeTweens(_this);
            });
        };
        Drop.prototype.onCollisionEnter = function (other) {
            if (other.group == ColliderGroup.Car) {
                this.eated();
            }
            if (other.group == ColliderGroup.Ground && !this.isGround) {
                this.stayGroundCount = 0;
                this.physics.velocity.y = -(this.physics.velocity.y * 0.4);
            }
            if (other.group == ColliderGroup.Wall) {
                this.physics.velocity.x = -this.physics.velocity.x;
                if (this.isGround) {
                    this.physics.rotateVelocity = -this.physics.rotateVelocity;
                }
            }
        };
        Drop.prototype.onCollisionStay = function (other) {
            if (other.group == ColliderGroup.Ground && !this.isGround) {
                this.stayGroundCount++;
                if (this.stayGroundCount > 2) {
                    this.isGround = true;
                    this.physics.velocity.y = 0;
                    this.physics.acceleration.y = 0;
                    this.physics.velocity.x *= 0.5;
                    DataMgr.game._dropBlinkMgr.add(this);
                    this.physics.rotateVelocity = this.physics.velocity.x * (180 / (Math.PI * 13));
                }
            }
        };
        Drop.prototype.onCollisionExit = function (other) {
            if (other.group == ColliderGroup.Ground) {
                this.stayGroundCount = 0;
            }
        };
        Drop.prototype.onElementInit = function () {
            this.visible = true;
            this.stayGroundCount = 0;
            this.isGround = false;
            this.enableAllComponents();
            this.label.visible = false;
            this.physics.acceleration.y = 0.3;
        };
        Drop.prototype.onElementRecycle = function () {
            this.alpha = 1;
            this.visible = false;
            this.disableAllComponents();
        };
        return Drop;
    }(MoveObjcet));

    var DropPool = (function (_super) {
        tslib.__extends(DropPool, _super);
        function DropPool() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        DropPool.prototype.spwan = function (id) {
            if (this.data.length > 0) {
                var element = this.data.shift();
                element.onElementInit();
                return element;
            }
            else {
                var d = new Drop(id);
                this.layer.addChild(d);
                return d;
            }
        };
        return DropPool;
    }(Pool));
    var DropPoolGroup = (function (_super) {
        tslib.__extends(DropPoolGroup, _super);
        function DropPoolGroup() {
            return _super !== null && _super.apply(this, arguments) || this;
        }
        DropPoolGroup.prototype.spwan = function (id) {
            if (!this.data[id]) {
                this.data[id] = new DropPool(this.layer);
            }
            return this.data[id].spwan(id);
        };
        return DropPoolGroup;
    }(PoolGroup));

    var DropBlinkMgr = (function (_super) {
        tslib.__extends(DropBlinkMgr, _super);
        function DropBlinkMgr() {
            var _this = _super.call(this, "DropBlinkMgr") || this;
            _this.list = [];
            _this.frameTimer = 0;
            return _this;
        }
        DropBlinkMgr.prototype.add = function (drop) {
            this.list.push({
                drop: drop,
                count: 0
            });
        };
        DropBlinkMgr.prototype.remove = function (drop) {
            this.list = this.list.filter(function (e) { return e.drop !== drop; });
        };
        DropBlinkMgr.prototype.onUpdate = function () {
            if (this.pause)
                return;
            if (this.frameTimer % BlinkDuration == 0) {
                var temp = null;
                for (var i in this.list) {
                    temp = this.list[i];
                    temp.drop.visible = !temp.drop.visible;
                    temp.count++;
                }
                this.list = this.list.filter(function (e) {
                    if (e.count < 60 / BlinkDuration * 2) {
                        return true;
                    }
                    else {
                        e.drop.dispose();
                        return false;
                    }
                });
            }
            this.frameTimer++;
        };
        return DropBlinkMgr;
    }(GameMgr));
    var BlinkDuration = 3;

    var Game = (function () {
        function Game(parent) {
            var _this = this;
            this._pause = false;
            this._updateData = {};
            this.mgrs = [];
            this._phycicsSystem = new PhycicsSystem();
            this._boomEffectMgr = new SpBoomEffectMgr();
            this._animationMgr = new FrameAnimationMgr();
            this._horizontalMoveMgr = new HorizontalMoveMgr();
            this.needSubmitCount = 0;
            this._localScore = 0;
            this._pool = null;
            this._dropBlinkMgr = new DropBlinkMgr();
            this.layers = {
                bullet: new engine.Container(),
                ball: new engine.Container(),
                drop: new engine.Container(),
                effect: new engine.Container(),
                animation: new engine.Container()
            };
            this._car = null;
            this.__score = 0;
            this.curMaxBallNum = 1;
            this.timing = false;
            this.createBallCD = 0;
            this.bulletScoreUpdateFlag = false;
            this._ballList = [];
            this.node = new engine.Container();
            this._bulletScore = 0;
            this._powerScore = 100;
            DataMgr.game = this;
            this.initMgrs();
            this.pause = false;
            this.loadRes();
            parent.addChild(this.node);
            GuideMgr.instance.container = parent;
            this.node.width = MConst.DesignResolution.width;
            this.node.height = MConst.DesignResolution.height;
            var bg = new engine.Image(getTexture("d99368b8-af5d-4d9e-981e-7bce3e1c1e84"));
            this.node.addChild(bg);
            this.node.addChild(this.layers.bullet);
            var car = new Car();
            car.x = this.node.width / 2 - car.width / 2;
            car.y = MConst.GroundLine - car.height + 30;
            this.node.addChild(car);
            car.addComponent(Physics);
            car.onDied = function () {
                _this.over();
            };
            this._car = car;
            this.node.addChild(this.layers.ball);
            this.node.addChild(this.layers.drop);
            this.node.addChild(this.layers.effect);
            this.node.addChild(this.layers.animation);
            for (var k in this.layers) {
                this.layers[k].mouseChildren = false;
                this.layers[k].mouseEnabled = false;
            }
            var playerController = new PlayerController();
            playerController.onTouchMove = function (deltaX) {
                if (!_this.pause) {
                    car.move(deltaX);
                }
            };
            this.node.addChild(playerController);
            MTimer.onFrame("Game", this.onUpdate, this);
            this.createPools();
            this.createWall();
            this.timing = true;
        }
        Object.defineProperty(Game.prototype, "score", {
            get: function () {
                return this._score;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Game.prototype, "bulletScore", {
            get: function () {
                return this._bulletScore;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Game.prototype, "powerScore", {
            get: function () {
                return this._powerScore;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Game.prototype, "pause", {
            get: function () {
                return this._pause;
            },
            set: function (v) {
                this._pause = v;
                this.timing = !this._pause;
                for (var _i = 0, _a = this.mgrs; _i < _a.length; _i++) {
                    var mgr = _a[_i];
                    mgr.pause = this._pause;
                }
            },
            enumerable: true,
            configurable: true
        });
        Game.prototype.revive = function () {
            this._car.revive();
            this.pause = false;
        };
        Game.prototype.destroy = function () {
            this.node.dispose();
            for (var _i = 0, _a = this.mgrs; _i < _a.length; _i++) {
                var mgr = _a[_i];
                mgr.onDestroy();
            }
            MTimer.removeOnFrame("Game");
            MTimer.removeOnFrame("Car");
        };
        Object.defineProperty(Game.prototype, "localScore", {
            get: function () {
                return this._localScore;
            },
            set: function (v) {
                this._localScore = v;
                if (this._localScore >= MConst.ConstantSubmitScoreNum) {
                    var count = Math.floor(this._localScore / MConst.ConstantSubmitScoreNum);
                    this.needSubmitCount += count;
                    this._localScore -= MConst.ConstantSubmitScoreNum * count;
                }
            },
            enumerable: true,
            configurable: true
        });
        Game.prototype.loadRes = function () {
            FrameAnimation.loadRes(getTexture("217164f4-a185-429c-8706-818137a4e438"), getAsset('763d2430-ad16-42c6-8bf0-d7337b05b247'));
            FrameAnimation.loadRes(getTexture("53c65221-3fbc-41d9-8cef-8a846876fe06"), getAsset('b82d6a84-8423-4be2-add5-ed9129e8c700'));
        };
        Game.prototype.initMgrs = function () {
            this.mgrs = [
                this._dropBlinkMgr,
                this._phycicsSystem,
                this._boomEffectMgr,
                this._animationMgr,
                this._horizontalMoveMgr
            ];
            for (var _i = 0, _a = this.mgrs; _i < _a.length; _i++) {
                var mgr = _a[_i];
                mgr.onInit();
            }
        };
        Game.prototype.over = function () {
            this.pause = true;
            engine.globalEvent.dispatchEvent("game-over", this._updateData);
            this.onGameOver && this.onGameOver();
        };
        Game.prototype.createWall = function () {
            var topWall = new GameObject();
            topWall.x = -500;
            topWall.y = -500;
            this.node.addChild(topWall);
            var topWallCollider = topWall.addComponent(RectCollider);
            topWallCollider.setData(0, 0, MConst.DesignResolution.width + 1000, 490);
            topWallCollider.group = ColliderGroup.Top;
            var ground = new GameObject();
            ground.x = -500;
            ground.y = MConst.GroundLine;
            this.node.addChild(ground);
            var groundCollider = ground.addComponent(RectCollider);
            groundCollider.setData(0, 0, MConst.DesignResolution.width + 1000, 1000);
            groundCollider.group = ColliderGroup.Ground;
            var leftWall = new GameObject();
            leftWall.x = -500;
            leftWall.y = -500;
            this.node.addChild(leftWall);
            var leftWallCollider = leftWall.addComponent(RectCollider);
            leftWallCollider.setData(0, 0, 500, MConst.DesignResolution.height + 1000);
            leftWallCollider.group = ColliderGroup.Wall;
            var rightWall = new GameObject();
            rightWall.x = MConst.DesignResolution.width;
            rightWall.y = -500;
            this.node.addChild(rightWall);
            var rightWallCollider = rightWall.addComponent(RectCollider);
            rightWallCollider.setData(0, 0, 500, MConst.DesignResolution.height + 1000);
            rightWallCollider.group = ColliderGroup.Wall;
        };
        Game.prototype.createBall = function () {
            var size = MUtils.randomInt(0, MConfigs.size.length);
            if (GuideMgr.instance.guideFlag == true) {
                size = 3;
            }
            var color = MUtils.randomInt(size, MConst.MaxColorIndex);
            var ball = this._pool.ball.spwan(size);
            var dir = Math.random() > 0.5 ? -1 : 1;
            var score = getBallScore(this._BulletScore, this._PowerScore, color);
            ball.init(color, dir, score);
            ball.startBornStage(dir);
            this.createBallCD = 1500;
        };
        Game.prototype.onUpdate = function () {
            for (var i = 0; i < this._ballList.length; i++) {
                this._ballList[i].updateScoreLabel();
            }
            if (this.bulletScoreUpdateFlag) {
                var score = this._BulletScore;
                this.updateMaxBallNum(score);
                this.bulletScoreUpdateFlag = false;
            }
            if (this.createBallCD > 0) {
                this.createBallCD -= MTimer.deltaTime;
            }
            if (this._ballList.length < this.curMaxBallNum) {
                if (this.createBallCD <= 0) {
                    this.createBall();
                }
            }
            if (!this.pause) {
                this.outputEvent();
            }
        };
        Game.prototype.outputEvent = function () {
            var updateData = {
                score: this.score,
                bulletScore: this.bulletScore,
                powerScore: this.powerScore,
            };
            var needSend = false;
            for (var key in updateData) {
                if (updateData[key] != this._updateData[key]) {
                    this._updateData[key] = updateData[key];
                    needSend = true;
                }
            }
            if (needSend) {
                engine.globalEvent.dispatchEvent("game-update", updateData);
            }
        };
        Game.prototype.updateMaxBallNum = function (bulletScore) {
            var num = 0;
            if (bulletScore <= 20)
                num = 1;
            else if (bulletScore <= 60)
                num = 2;
            else if (bulletScore <= 100)
                num = 3;
            else if (bulletScore <= 130)
                num = 4;
            else if (Math.random() < 0.3)
                num = 6;
            else
                num = 7;
            this.curMaxBallNum = num;
        };
        Game.prototype._shake = function () {
            var _this = this;
            var count = 0;
            var callback = function () {
                if (count > 1)
                    return;
                count++;
                _this.node.x = 10;
                engine.Tween.get(_this.node)
                    .wait(1, true)
                    .to({ x: 0, y: -10 }, 1)
                    .to({ x: -10, y: 0 }, 1)
                    .to({ x: 0, y: 10 }, 1)
                    .to({ x: 0, y: 0 }, 1)
                    .call(callback);
            };
            callback();
        };
        Game.prototype.createPools = function () {
            this._pool = {
                ball: new BallPoolGroup(this.layers.ball),
                bullet: new BulletPool(this.layers.bullet),
                animation: new AnimationPoolGroup(this.layers.animation),
                boomEffect: new SpBoomEffectPoolGroup(this.layers.effect),
                drop: new DropPoolGroup(this.layers.drop)
            };
        };
        Game.prototype._playBoomEffect = function (position, size) {
            var effect = this._pool.boomEffect.spwan(MUtils.randomInt(0, 3), size);
            effect.position = position;
        };
        Game.prototype._createAnimation = function (key) {
            var _this = this;
            var clip = this._pool.animation.spwan(key);
            clip.onCompleted = function () {
                _this._pool.animation.recycle(key, clip);
            };
            return clip;
        };
        Object.defineProperty(Game.prototype, "_score", {
            get: function () {
                return this.__score;
            },
            set: function (v) {
                this.localScore += (v - this.__score);
                this.__score = v;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Game.prototype, "_BulletScore", {
            get: function () {
                return this._bulletScore;
            },
            set: function (v) {
                if (v == this._bulletScore)
                    return;
                v = Math.max(v, 0);
                this._bulletScore = v;
                this.bulletScoreUpdateFlag = true;
            },
            enumerable: true,
            configurable: true
        });
        Game.prototype.onGuideDone = function () {
            engine.globalEvent.dispatchEvent('game-start');
        };
        Object.defineProperty(Game.prototype, "_PowerScore", {
            get: function () {
                return this._powerScore;
            },
            set: function (v) {
                v = Math.max(v, 100);
                this._powerScore = v;
            },
            enumerable: true,
            configurable: true
        });
        return Game;
    }());

    var ShootPlanet = (function (_super) {
        tslib.__extends(ShootPlanet, _super);
        function ShootPlanet() {
            var _this = _super.call(this) || this;
            _this.game = null;
            _this.customProperty();
            _this.width = 750;
            _this.height = 1624;
            _this.init();
            engine.globalEvent.addEventListener('game-create', _this.onActive, _this);
            engine.globalEvent.addEventListener('game-destroy', _this.onSleep, _this);
            engine.globalEvent.addEventListener('game-revive', _this.onRevive, _this);
            engine.globalEvent.addEventListener('game-pause', _this.onPause, _this);
            engine.globalEvent.addEventListener('game-resume', _this.onResume, _this);
            return _this;
        }
        ShootPlanet.prototype.onActive = function () {
            if (!this.game) {
                this.game = new Game(this);
            }
            if (GuideMgr.instance.guideFlag == true) {
                var car = this.game._car;
                GuideMgr.instance.runGuide(0, car.x + car.width / 2, car.y + car.height / 2);
            }
            if (GuideMgr.instance.guideDone) {
                this.game.onGuideDone();
            }
        };
        ShootPlanet.prototype.onSleep = function () {
            this.game.destroy();
            this.game = null;
        };
        ShootPlanet.prototype.onRevive = function () {
            this.game.revive();
        };
        ShootPlanet.prototype.onPause = function () {
            this.game.pause = true;
        };
        ShootPlanet.prototype.onResume = function () {
            this.game.pause = false;
        };
        ShootPlanet.prototype.init = function () {
            MTimer.init();
            DebugMgr.instance.init(this);
        };
        ShootPlanet.prototype.customProperty = function () {
            Object.defineProperty(engine.Point.prototype, 'length', {
                get: function () {
                    return Math.sqrt(this.x * this.x + this.y * this.y);
                }
            });
            engine.DisplayObject.prototype.dispose = function () {
                this.parent.removeChild(this);
            };
        };
        return ShootPlanet;
    }(engine.Container));

    function index (props) {
        prepareProps();
        injectProps(props);
        var instance = new ShootPlanet();
        return instance;
    }

    return index;

})));

//# sourceMappingURL=main.js.map