/**
 * Created by rockyl on 2018/8/16.
 */

import {props} from "../props";
import {playSound} from "./utils";
import {GuideLayer} from "./GuideLayer";
import ObjectPool = engine.ObjectPool;
import {Turntable} from "./Turntable";
import {Stick} from "./Stick";
import {PoolName} from "./object-pool-init";

export default class GameView extends engine.Container {
	turntable: Turntable;
	guideLayer: GuideLayer;

	private _counting;
	_touchEnabled;
	private _hasSetup;
	private _countdown;

	private _stick: Stick;
	private _timer;

	constructor() {
		super();

		this.once(engine.Event.ADDED_TO_STAGE, this.setup, this);
	}

	setup() {
		if (this._hasSetup) {
			return;
		}
		this._hasSetup = true;

		const turntable = this.turntable = new Turntable();
		this.addChild(turntable);

		const guideLayer = this.guideLayer = new GuideLayer();
		this.addChild(guideLayer);

		//this.reset();
	}

	async reset(revive = false, options = null) {
		if (options) {
			this.turntable.reset(options);
			this.recycleStick();
		} else {
			this.prepareStick();
		}
	}

	async start() {
		/*let guideFlagKey = 'lipstick-turntable-guide_' + props.guideFlagKey;
		let guideFlag = localStorage.getItem(guideFlagKey);
		if (!guideFlag) {
			localStorage.setItem(guideFlagKey, '1');
			await this.guideLayer.show('', {y: 100});
		}*/

		this._countdown = props.countDown;
		this._counting = props.lipstickCount;
		this.turntable.start();
		this.prepareStick();
		this.countDownStep(false);

		this.startCountDown();
	}

	startCountDown() {
		this._timer = setInterval(this.countDownStep, 1000);
	}

	stopCountDown() {
		if (this._timer) {
			clearInterval(this._timer);
			this._timer = null;
		}
	}

	prepareStick() {
		let stick = this._stick = <Stick>ObjectPool.getObject(PoolName);
		stick.y = props.stickDistance;
		this.addChildAt(stick, 0);

		this._touchEnabled = true;
	}

	recycleStick() {
		let stick = this._stick;
		if (stick) {
			this._stick = null;
			this.removeChild(stick);
			ObjectPool.recycleObject(PoolName, stick);
		}
	}

	pause() {
		this.turntable.pause();
		this._stick.stopSend();
		this._stick.playFall();
		this.stopCountDown();
	}

	resume() {
		this.recycleStick();
		this.turntable.resume();
		this.startCountDown();
	}

	async revive() {
		this.recycleStick();
		this.prepareStick();
		this.turntable.resume();
	}

	private countDownStep = (decrease = true) => {
		if (decrease) {
			this._countdown--;
			if (this._countdown <= 0) {
				this.stopCountDown();
				this.onTimeout();
			}
		}
		engine.globalEvent.dispatchEvent('lipstick-turntable-countdown', {
			second: this._countdown,
		});
	};

	private onTimeout() {
		this.pause();
		this._touchEnabled = false;

		engine.globalEvent.dispatchEvent('lipstick-turntable-game-failed', {
			reason: 1,
		});
	}

	private onHit = async () => {
		let stick = this._stick;
		if (this.turntable.addStick(stick)) {
			this.stopCountDown();
			await stick.playFall();
			engine.globalEvent.dispatchEvent('lipstick-turntable-game-failed', {
				reason: 2,
			});
		} else {
			this._counting--;
			engine.globalEvent.dispatchEvent('lipstick-turntable-hit', {
				remain: this._counting,
			});
			if (this._counting === 0) {
				this.turntable.hideSticks();
				this.stopCountDown();
				await this.turntable.playBreakUp();
				engine.globalEvent.dispatchEvent('lipstick-turntable-game-success');
			} else {
				this.prepareStick();
			}
		}
	};

	tap(e) {
		if (!this._touchEnabled) {
			return;
		}

		this._touchEnabled = false;

		this._stick.playSend(this.turntable.radius - props.stickInsertDepth, this.onHit);
	}
}
