declare module engine {
	export const VERSION = "1.0";

	export const devicePixelRatio: number;

	export function computeProps(props, options);

	export let StageScaleMode: {
		EXACT_FIT: string;
		NO_BORDER: string;
		NO_SCALE: string;
		SHOW_ALL: string;
		FIXED_WIDTH: string;
		FIXED_HEIGHT: string;
	};

	export const osType: string;

	export function cos(angle: number): number;

	export function sin(angle: number): number;

	export const PI_2: number;

	export const RAD_TO_DEG: number;

	export const DEG_TO_RAD: number;

	export function transPoint(str, sep = ',');

	export enum RENDERER_TYPE {
		UNKNOWN = "unknown",
		WEBGL = "webgl",
		CANVAS = "canvas"
	}

	export enum SHAPES {
		POLY = 0,
		RECT = 1,
		CIRC = 2,
		ELIP = 3,
		RREC = 4
	}

	export enum PRECISION {
		LOW = "lowp",
		MEDIUM = "mediump",
		HIGH = "highp"
	}

	export enum TEXT_GRADIENT {
		LINEAR_VERTICAL = 0,
		LINEAR_HORIZONTAL = 1
	}

	export enum TEXT_ALIGN {
		CENTER = "center",
		LEFT = "left",
		RIGHT = "right"
	}

	export enum VERTICAL_ALIGN {
		MIDDLE = "middle",
		UP = "up",
		DOWN = "down"
	}

	export enum TEXT_lINETYPE {
		SINGLE = "single",
		MULTI = "multi"
	}

	export enum LINE_ALIGNMENT {
		middle = 0.5,
		outter = 1,
		inner = 0
	}

	export enum LINE_CAP {
		BUTT = "butt",
		ROUND = "round",
		SQUARE = "square"
	}

	export enum LINE_JOIN {
		MITER = "miter",
		ROUND = "round",
		BEVEL = "bevel"
	}

	export enum BLEND_MODES {
		NORMAL = 0,
		ADD = 1,
		MULTIPLY = 2,
		SCREEN = 3,
		OVERLAY = 4,
		DARKEN = 5,
		LIGHTEN = 6,
		COLOR_DODGE = 7,
		COLOR_BURN = 8,
		HARD_LIGHT = 9,
		SOFT_LIGHT = 10,
		DIFFERENCE = 11,
		EXCLUSION = 12,
		HUE = 13,
		SATURATION = 14,
		COLOR = 15,
		LUMINOSITY = 16,
		NORMAL_NPM = 17,
		ADD_NPM = 18,
		SCREEN_NPM = 19
	}

	export enum DRAW_MODES {
		POINTS = 0,
		LINES = 1,
		LINE_LOOP = 2,
		LINE_STRIP = 3,
		TRIANGLES = 4,
		TRIANGLE_STRIP = 5,
		TRIANGLE_FAN = 6
	}

	export enum SCALE_MODES {
		LINEAR = 0,
		NEAREST = 1
	}

	export enum WRAP_MODES {
		CLAMP = 33071,
		REPEAT = 10497,
		MIRRORED_REPEAT = 33648
	}

	export enum GC_MODES {
		AUTO = 0,
		MANUAL = 1
	}

	export const URL_FILE_EXTENSION: any;

	export const DATA_URI: any;

	export enum SCROLL_DIRECTION {
		HORIZONTAL = "horizontal",
		VERTICAL = "vertical"
	}

	export abstract class HashObject {
		protected _instanceId: number;
		protected _instanceType: string;
		protected static _object_id: number;

		constructor();

		/**
		 * 每一个对象都会有一个唯一的id码。
		 * @property instanceId
		 * @public
		 * @since 1.0.0
		 * @return {number}
		 * @readonly
		 * @example
		 *      //获取 对象唯一码
		 *      trace(this.instanceId);
		 */
		readonly instanceId: number;
		/**
		 * 每一个类都有一个实例类型字符串，通过这个字符串，你能知道这个实例是从哪个类实例而来
		 * @property instanceType
		 * @since 1.0.3
		 * @public
		 * @return {string}
		 * @readonly
		 */
		readonly instanceType: string;

		/**
		 * 销毁一个对象
		 * 销毁之前一定要从显示对象移除，否则将会出错
		 * @method destroy
		 * @since 2.0.0
		 * @public
		 * @return {void}
		 */
		abstract destroy(): void;
	}

	export class Event extends HashObject {
		/**
		 * 舞台尺寸发生变化时触发
		 * @Event
		 * @property RESIZE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static RESIZE: string;
		/**
		 * ScrollPage组件滑动到开始位置事件
		 * @property ON_SCROLL_TO_HEAD
		 * @static
		 * @since 1.1.0
		 * @type {string}
		 */
		static ON_SCROLL_TO_HEAD: string;
		/**
		 * ScrollPage组件停止滑动事件
		 * @property ON_SCROLL_STOP
		 * @static
		 * @since 1.1.0
		 * @type {string}
		 */
		static ON_SCROLL_STOP: string;
		/**
		 * ScrollPage组件开始滑动事件
		 * @property ON_SCROLL_START
		 * @static
		 * @since 1.1.0
		 * @type {string}
		 */
		static ON_SCROLL_START: string;
		/**
		 * ScrollPage组件滑动到结束位置事件
		 * @property ON_SCROLL_TO_END
		 * @static
		 * @since 1.1.0
		 * @type {string}
		 */
		static ON_SCROLL_TO_END: string;
		/**
		 * 舞台初始化完成后会触发的事件
		 * @property ON_INIT_STAGE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static ON_INIT_STAGE: string;
		/**
		 * 显示对象加入到舞台事件
		 * @Event
		 * @property ADD_TO_STAGE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static ADDED_TO_STAGE: string;
		/**
		 * 显示对象从舞台移出事件
		 * @Event
		 * @property REMOVE_TO_STAGE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static REMOVED_FROM_STAGE: string;
		/**
		 * 显示对象 循环帧事件
		 * @Event
		 * @property ENTER_FRAME
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static ENTER_FRAME: string;
		/**
		 * MovieClip 播放完成事件
		 * @Event
		 * @property END_FRAME
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static END_FRAME: string;
		/**
		 * MovieClip 帧标签事件
		 * @Event
		 * @property CALL_FRAME
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static CALL_FRAME: string;
		/**
		 * 可见事件
		 * @Event
		 * @property VISIBLE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static VISIBLE: string;
		/**
		 * 隐藏事件
		 * @Event
		 * @property HIDDEN
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static HIDDEN: string;
		/**
		 * 完成事件
		 * @Event
		 * @property COMPLETE
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static COMPLETE: string;
		/**
		 * 加载过程事件
		 * @Event
		 * @property PROGRESS
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static PROGRESS: string;
		/**
		 * 获取焦点
		 * @Event
		 * @property FOCUS
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static FOCUS: string;
		/**
		 * 失去焦点
		 * @Event
		 * @property BLUR
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static BLUR: string;
		/**
		 * 内容变化时触发
		 * @Event
		 * @property BLUR
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static CHANGING: string;
		/**
		 * 出错事件
		 * @Event
		 * @property ERROR
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static ERROR: string;
		/**
		 * 中断事件
		 * @Event
		 * @property ABORT
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static ABORT: string;
		/**
		 * 开始事件
		 * @Event
		 * @property START
		 * @type {string}
		 * @static
		 * @public
		 * @since 1.0.0
		 */
		static START: string;
		/**
		 * 定时器触发事件
		 * @property TIMER
		 * @static
		 * @since 1.0.9
		 * @public
		 * @type {string}
		 */
		static TIMER: string;
		/**
		 * 定时器完成事件
		 * @property TIMER_COMPLETE
		 * @since 1.0.9
		 * @static
		 * @public
		 * @type {string}
		 */
		static TIMER_COMPLETE: string;
		/**
		 * 数据更新事件
		 * @property UPDATE_DATA
		 * @since 1.0.0
		 * @static
		 * @public
		 * @type {string}
		 */
		static UPDATE_DATA: string;
		/**
		 * 事件类型名
		 * @property type
		 * @type {string}
		 * @public
		 * @since 1.0.0
		 */
		type: string;
		/**
		 * 触发此事件的对象
		 * @property target
		 * @public
		 * @since 1.0.0
		 * @type {any}
		 */
		target: any;
		/**
		 * 随着事件一起附带的信息对象
		 * 所有需要随事件一起发送的信息都可以放在此对象中
		 * @property data
		 * @public
		 * @since 1.0.0
		 * @type {any}
		 * @default null
		 */
		data: any;

		/**
		 * @method Event
		 * @param {string} type 事件类型
		 * @public
		 * @since 1.0.0
		 */
		constructor(type: string);

		/**
		 * 防止对事件流中当前节点中和所有后续节点中的事件侦听器进行处理。
		 * @method stopImmediatePropagation
		 * @public
		 * @return {void}
		 * @since 2.0.0
		 */
		stopImmediatePropagation(): void;

		/**
		 * 防止对事件流中当前节点的后续节点中的所有事件侦听器进行处理。
		 * @method stopPropagation
		 * @public
		 * @since 2.0.0
		 * @return {void}
		 */
		stopPropagation(): void;

		private _bpd;
		/**
		 * 是否阻止事件向下冒泡
		 * @property _pd
		 * @type {boolean}
		 * @private
		 * @since 1.0.0
		 */
		private _pd;

		destroy(): void;

		/**
		 * 重围事件到初始状态方便重复利用
		 * @method reset
		 * @param {string} type
		 * @param target
		 * @since 2.0.0
		 * @return {void}
		 * @public
		 */
		reset(type: string, target: any): void;
	}

	export class EventDispatcher extends HashObject {
		/**
		 * 捕获阶段事件名
		 */
		protected eventTypes: any;
		/**
		 * 冒泡阶段事件名
		 */
		protected eventTypes1: any;

		constructor();

		/**
		 * 全局的鼠标事件的监听数对象表，比如{"onMouseMove":9,"onMouseDown":7}
		 * @property _MECO
		 * @private
		 * @since 1.0.0
		 */
		private static _MECO;
		/**
		 * 所有鼠标事件的数量
		 */
		static _totalMEC: number;

		/**
		 * 看看有多少mouse或者touch侦听数
		 * @method getMouseEventCount
		 * @return {number}
		 * @static
		 * @private
		 * @since 1.0.0
		 * @param {string} type 获取事件类型，默认是所有
		 */
		static getMouseEventCount(type?: string): number;

		/**
		 * 给对象添加一个侦听
		 * @method addEventListener
		 * @public
		 * @since 1.0.0
		 * @param {string} type 侦听类形
		 * @param {Function}listener 侦听后的回调方法,如果这个方法是类实例的方法,为了this引用的正确性,请在方法参数后加上.bind(this);
		 * @param context thisObject
		 * @param {boolean} useCapture true 捕获阶段 false 冒泡阶段 默认 true
		 * @example
		 *      this.addEventListener(Event.ADD_TO_STAGE,function(e){trace(this);}.bind(this));
		 */
		addEventListener(type: string, listener: Function, context?: any, useCapture?: boolean): void;

		/**
		 * 监听一次
		 * @param type
		 * @param listener
		 * @param context
		 * @param useCapture
		 */
		once(type: string, listener: Function, context?: any, useCapture?: boolean): void;

		/**
		 * 增加或删除相应mouse或touch侦听记数
		 * @method _changeMouseCount
		 * @private
		 * @since 1.0.0
		 * @param {string} type
		 * @param {boolean} isAdd
		 */
		private _changeMouseCount;
		private _defaultEvent;

		/**
		 * 广播侦听
		 * @method dispatchEvent
		 * @public
		 * @since 1.0.0
		 * @param {Event|string} event 广播所带的事件对象,如果传的是字符串则直接自动生成一个的事件对象,事件类型就是你传入进来的字符串的值
		 * @param {Object} data 广播后跟着事件一起传过去的其他任信息,默认值为null，在传参中
		 * @param {boolean} useCapture true 捕获阶段 false 冒泡阶段 默认 true
		 * @return {boolean} 如果有收听者则返回true
		 * @example
		 *      var mySprite=new Sprite(),
		 *          yourEvent=new Event("yourCustomerEvent");
		 *       yourEvent.data='false2x';
		 *       mySprite.addEventListener("yourCustomerEvent",function(e){
		 *          trace(e.data);
		 *        })
		 *       mySprite.dispatchEvent(yourEvent);
		 */
		dispatchEvent(event: any, data?: any, useCapture?: boolean): boolean;

		/**
		 * 是否有添加过此类形的侦听
		 * @method hasEventListener
		 * @public
		 * @since 1.0.0
		 * @param {string} type 侦听类形
		 * @param {boolean} useCapture true 捕获阶段 false 冒泡阶段 默认 true
		 * @return {boolean} 如果有则返回true
		 */
		hasEventListener(type: string, useCapture?: boolean): boolean;

		/**
		 * 移除对应类型的侦听
		 * @method removeEventListener
		 * @public
		 * @since 1.0.0
		 * @param {string} type 要移除的侦听类型
		 * @param {Function} listener 及侦听时绑定的回调方法
		 * @param context listener和context都相等的才移除，默认自身
		 * @param {boolean} useCapture true 捕获阶段 false 冒泡阶段 默认 true
		 */
		removeEventListener(type: string, listener: Function, context?: any, useCapture?: boolean): void;

		/**
		 * 移除对象中所有的侦听
		 * @method removeAllEventListener
		 * @public
		 * @since 1.0.0
		 */
		removeAllEventListener(): void;

		destroy(): void;
	}

	export class Point extends HashObject {
		destroy(): void;

		/**
		 * 构造函数
		 * @method Point
		 * @public
		 * @since 1.0.0
		 * @param x
		 * @param y
		 */
		constructor(x?: number, y?: number);

		/**
		 * 水平坐标
		 * @property x
		 * @public
		 * @since 1.0.0
		 * @type{number}
		 */
		x: number;
		/**
		 * 垂直坐标
		 * @property y
		 * @since 1.0.0
		 * @public
		 * @type {number}
		 */
		y: number;

		/**
		 * 求两点之间的距离
		 * @method distance
		 * @param args 可变参数 传两个参数的话就是两个Point类型 传四个参数的话分别是两个点的x y x y
		 * @return {number}
		 * @static
		 */
		static distance(...args: any[]): number;

		set(x: number, y: number): void;
	}

	export class ObservablePoint extends HashObject {
		_x: number;
		_y: number;
		cb: any;
		scope: any;

		/**
		 * @param {Function} cb - 值改变时的回调
		 * @param {object} scope - 回调里的上下文this
		 * @param {number} [x=0] - x
		 * @param {number} [y=0] - y
		 */
		constructor(cb: Function, scope: any, x?: number, y?: number);

		/**
		 * 设置xy
		 * @param {number} [x=0]
		 * @param {number} [y=0]
		 */
		set(x?: number, y?: number): void;

		/**
		 * 从一个点复制xy
		 *
		 * @param {Point|ObservablePoint} point
		 */
		copy(point: Point | ObservablePoint): void;

		x: number;
		y: number;

		destroy(): void;
	}

	export class Matrix extends HashObject {
		/**
		 * @property a
		 * @type {number}
		 * @public
		 * @default 1
		 * @since 1.0.0
		 */
		a: number;
		/**
		 * @property b
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		b: number;
		/**
		 * @property c
		 * @type {number}
		 * @public
		 * @since 1.0.0
		 */
		c: number;
		/**
		 * @property d
		 * @type {number}
		 * @public
		 * @since 1.0.0
		 */
		d: number;
		/**
		 * @property tx
		 * @type {number}
		 * @public
		 * @since 1.0.0
		 */
		tx: number;
		/**
		 * @property ty
		 * @type {number}
		 * @since 1.0.0
		 * @public
		 */
		ty: number;
		array: any;

		/**
		 * 构造函数
		 * @method Matrix
		 * @param {number} a
		 * @param {number} b
		 * @param {number} c
		 * @param {number} d
		 * @param {number} tx
		 * @param {number} ty
		 * @public
		 * @since 1.0.0
		 */
		constructor(a?: number, b?: number, c?: number, d?: number, tx?: number, ty?: number);

		/**
		 * 复制一个矩阵
		 * @method clone
		 * @since 1.0.0
		 * @public
		 * @return {Matrix}
		 */
		clone(): Matrix;

		/**
		 * 复制一个矩阵的所有属性
		 * @param matrix
		 */
		copy(matrix: Matrix | any): this;

		/**
		 * 将一个点通过矩阵变换后的点，世界矩阵应用于局部坐标，转化为世界坐标
		 * @method transformPoint
		 * @param {number} x
		 * @param {number} y
		 * @param {Point} 默认为空，如果不为null，则返回的是Point就是此对象，如果为null，则返回来的Point是新建的对象
		 * @return {Point}
		 * @public
		 * @since 1.0.0
		 */
		transformPoint(x: number, y: number, bp?: Point): Point;

		/**
		 * Get a new position with the inverse of the current transformation applied.
		 * Can be used to go from the world coordinate space to a child's coordinate space. (e.g. input)
		 * 用于世界坐标转化为局部坐标
		 * @param {number} x
		 * @param {number} y
		 * @param {Point} 默认为空，如果不为null，则返回的是Point就是此对象，如果为null，则返回来的Point是新建的对象
		 * @return {Point}
		 */
		transformPointInverse(x: number, y: number, bp?: Point): Point;

		/**
		 * 从一个矩阵里赋值给这个矩阵
		 * @method setFrom
		 * @param {Matrix} mtx
		 * @public
		 * @since 1.0.0
		 */
		setFrom(mtx: Matrix): void;

		/**
		 * 将矩阵恢复成原始矩阵
		 * @method identity
		 * @public
		 * @since 1.0.0
		 */
		identity(): void;

		/**
		 * 反转一个矩阵
		 * @method invert
		 * @return {Matrix}
		 * @since 1.0.0
		 * @public
		 */
		invert(): Matrix;

		/**
		 * 设置一个矩阵通过普通的显示对象的相关九大属性，锚点不影响坐标原点,暂时不用
		 * @method createBox
		 * @param {number} x
		 * @param {number} y
		 * @param {number} scaleX
		 * @param {number} scaleY
		 * @param {number} rotation 角度制
		 * @param {number} skewX 角度制
		 * @param {number} skewY 角度制
		 * @param {number} ax
		 * @param {number} ay
		 * @since 1.0.0
		 * @public
		 */
		createBox(x: number, y: number, scaleX: number, scaleY: number, rotation: number, skewX: number, skewY: number, ax: number, ay: number): void;

		/**
		 * 矩阵相乘
		 * @method prepend
		 * @public
		 * @since 1.0.0
		 * @param {Matrix} mtx
		 */
		prepend: (mtx: Matrix) => void;

		/**
		 * Appends the given Matrix to this Matrix.
		 *
		 * @param {Matrix} matrix - The matrix to append.
		 * @return {Matrix} This matrix. Good for chaining method calls.
		 */
		append(matrix: Matrix): void;

		/**
		 * 判断两个矩阵是否相等
		 * @method isEqual
		 * @static
		 * @public
		 * @since 1.0.0
		 * @param {Matrix} m1
		 * @param {Matrix} m2
		 * @return {boolean}
		 */
		static isEqual(m1: Matrix, m2: Matrix): boolean;

		concat(mtx: Matrix): void;

		/**
		 * 对矩阵应用旋转转换。
		 * @method rotate
		 * @param angle 弧度制
		 * @since 1.0.3
		 * @public
		 */
		rotate(angle: number): void;

		/**
		 * 对矩阵应用缩放转换。
		 * @method scale
		 * @param {Number} sx 用于沿 x 轴缩放对象的乘数。
		 * @param {Number} sy 用于沿 y 轴缩放对象的乘数。
		 * @since 1.0.3
		 * @public
		 */
		scale(sx: number, sy: number): void;

		/**
		 * 沿 x 和 y 轴平移矩阵，由 dx 和 dy 参数指定。
		 * @method translate
		 * @public
		 * @since 1.0.3
		 * @param {Number} dx 沿 x 轴向右移动的量（以像素为单位
		 * @param {Number} dy 沿 y 轴向右移动的量（以像素为单位
		 */
		translate(dx: number, dy: number): void;

		set(a: any, b: any, c: any, d: any, tx: any, ty: any): this;

		/**
		 * 获得角度,角度制,
		 * 其他的x,y,就是tx,ty
		 * scale就是a,d
		 * skew基本不用
		 */
		getRotation(): number;

		/**
		 * Creates an array from the current Matrix object.与glsl中的mat3对应,注意行列主序执行transpose;
		 *
		 * @param {boolean} transpose - Whether we need to transpose the matrix or not
		 * @param {Float32Array} [out=new Float32Array(9)] - If provided the array will be assigned to out
		 * @return {number[]} the newly created array which contains the matrix
		 */
		toArray(transpose?: boolean, out?: any): any;

		/**
		 * A default (identity) matrix
		 *
		 * @static
		 * @const
		 */
		static readonly IDENTITY: Matrix;
		/**
		 * A temp matrix
		 *
		 * @static
		 * @const
		 */
		static readonly TEMP_MATRIX: Matrix;

		destroy(): void;
	}

	export class Transform extends HashObject {
		/**
		 * 世界矩阵
		 */
		worldMatrix: Matrix;
		/**
		 * 本地矩阵
		 */
		localMatrix: Matrix;
		/**
		 * 记录是否更新worldMatrix
		 */
		_worldID: number;
		/**
		 * 记录是否和父级更新的_worldID一致
		 */
		_parentID: number;
		/**
		 * 位置
		 */
		position: ObservablePoint;
		/**
		 * 缩放
		 */
		scale: ObservablePoint;
		/**
		 * 锚点
		 * 不改变坐标原点
		 */
		anchor: ObservablePoint;
		/**
		 * 斜切值
		 */
		skew: ObservablePoint;
		/**
		 * 弧度制
		 */
		_rotation: number;
		_cx: number;
		_sx: number;
		_cy: number;
		_sy: number;
		/**
		 * 记录的本地坐标id
		 */
		_localID: number;
		/**
		 * 当前本地坐标id
		 */
		_currentLocalID: number;

		constructor();

		/**
		 * 任何属性更改
		 * @private
		 */
		onChange(): void;

		/**
		 * 当斜切改变时，先记录，优化计算
		 * @private
		 */
		updateSkew(): void;

		/**
		 * 更新本地矩阵
		 */
		updateLocalMatrix(): void;

		/**
		 * 更新世界矩阵,跟随父级修改
		 * @param {Transform} parentTransform - 父级矩阵
		 */
		updateWorldMatrix(parentTransform: Transform): void;

		/**
		 * 弧度制
		 *
		 * @member {number}
		 */
		rotation: number;

		destroy(): void;
	}

	export class Rectangle extends HashObject {
		/**
		 * 类型
		 */
		type: number;

		/**
		 * 构造函数
		 * @method Rectangle
		 * @param {number} x
		 * @param {number} y
		 * @param {number} width
		 * @param {number} height
		 */
		constructor(x?: number, y?: number, width?: number, height?: number);

		clear(): void;

		copy(rect: Rectangle): this;

		clone(): Rectangle;

		/**
		 * 矩形左上角的 x 坐标
		 * @property x
		 * @public
		 * @since 1.0.0
		 * @type{number}
		 * @default 0
		 */
		x: number;
		/**
		 * 矩形左上角的 y 坐标
		 * @property y
		 * @public
		 * @since 1.0.0
		 * @type{number}
		 * @default 0
		 */
		y: number;
		/**
		 * 矩形的宽度（以像素为单位）
		 * @property width
		 * @public
		 * @since 1.0.0
		 * @type{number}
		 * @default 0
		 */
		width: number;
		/**
		 * 矩形的高度（以像素为单位）
		 * @property height
		 * @public
		 * @since 1.0.0
		 * @type{number}
		 * @default 0
		 */
		height: number;
		/**
		 * returns the left edge of the rectangle
		 */
		readonly left: number;
		/**
		 * returns the right edge of the rectangle
		 */
		readonly right: number;
		/**
		 * returns the top edge of the rectangle
		 */
		readonly top: number;
		/**
		 * returns the bottom edge of the rectangle
		 */
		readonly bottom: number;

		/**
		 * 判断一个点是否在矩形内包括边
		 * @method isPointIn
		 * @param {Point} point
		 * @return {boolean}
		 * @public
		 * @since 1.0.0
		 */
		isPointIn(point: Point): boolean;

		/**
		 * Fits this rectangle around the passed one.
		 *
		 * @param {Rectangle} rectangle - The rectangle to fit.
		 */
		fit(rectangle: Rectangle): void;

		/**
		 * 将多个矩形合成为一个矩形,并将结果存到第一个矩形参数，并返回
		 * @method createFromRects
		 * @param {Rectangle} rect
		 * @param {..arg} arg
		 * @public
		 * @since 1.0.0
		 * @static
		 */
		static createFromRects(...arg: Rectangle[]): Rectangle;

		/**
		 * 通过一系列点来生成一个矩形
		 * 返回包含所有给定的点的最小矩形
		 * @method createFromPoints
		 * @static
		 * @public
		 * @since 1.0.0
		 * @param {Point} p1
		 * @param {..arg} ary
		 * @return {Rectangle}
		 */
		static createFromPoints(rect: Rectangle, ...arg: Point[]): Rectangle;

		/**
		 * 通过顶点数据  [0,1,
		 *               2,3,
		 *               1,3,
		 *               1,0]
		 * @param rect
		 * @param vertexData 一般为8长度
		 */
		static createFromVertexData(rect: Rectangle, vertexData: Float32Array): Rectangle;

		/**
		 * 通过两个点来确定一个矩形
		 * @method createRectform2Point
		 * @static
		 * @param rect
		 * @param p1
		 * @param p2
		 * @return
		 */
		static createRectfrom2Point(rect: Rectangle, p1: Point, p2: Point): Rectangle;

		/**
		 * 判读两个矩形是否相交
		 * @method testRectCross
		 * @public
		 * @since 1.0.2
		 * @param r1
		 * @param r2
		 * @return {boolean}
		 */
		static testRectCross(ra: Rectangle, rb: Rectangle): boolean;

		destroy(): void;
	}

	export const GroupD8: {
		E: number;
		SE: number;
		S: number;
		SW: number;
		W: number;
		NW: number;
		N: number;
		NE: number;
		MIRROR_VERTICAL: number;
		MIRROR_HORIZONTAL: number;
		uX: (ind: any) => number;
		uY: (ind: any) => number;
		vX: (ind: any) => number;
		vY: (ind: any) => number;
		inv: (rotation: any) => number;
		add: (rotationSecond: any, rotationFirst: any) => any;
		sub: (rotationSecond: any, rotationFirst: any) => any;
		/**
		 * Adds 180 degrees to rotation. Commutative operation.
		 *
		 * @memberof GroupD8
		 * @param {number} rotation - The number to rotate.
		 * @returns {number} rotated number
		 */
		rotate180: (rotation: any) => number;
		/**
		 * Direction of main vector can be horizontal, vertical or diagonal.
		 * Some objects work with vertical directions different.
		 *
		 * @memberof GroupD8
		 * @param {number} rotation - The number to check.
		 * @returns {boolean} Whether or not the direction is vertical
		 */
		isVertical: (rotation: any) => boolean;
		/**
		 * @memberof GroupD8
		 * @param {number} dx - TODO
		 * @param {number} dy - TODO
		 *
		 * @return {number} TODO
		 */
		byDirection: (dx: any, dy: any) => number;
		/**
		 * Helps sprite to compensate texture packer rotation.
		 *
		 * @memberof GroupD8
		 * @param {Matrix} matrix - sprite world matrix
		 * @param {number} rotation - The rotation factor to use.
		 * @param {number} tx - sprite anchoring
		 * @param {number} ty - sprite anchoring
		 */
		matrixAppendRotationInv: (matrix: any, rotation: any, tx?: number, ty?: number) => void;
	}

	export class Circle {
		/**
		 * @member {number}
		 * @default 0
		 */
		x: number;
		/**
		 * @member {number}
		 * @default 0
		 */
		y: number;
		/**
		 * @member {number}
		 * @default 0
		 */
		radius: number;
		/**
		 * 类型
		 * @member {number}
		 * @readOnly
		 * @default SHAPES.CIRC
		 * @see SHAPES
		 */
		type: number;

		/**
		 * @param {number} [x=0] - The X coordinate of the center of this circle
		 * @param {number} [y=0] - The Y coordinate of the center of this circle
		 * @param {number} [radius=0] - The radius of the circle
		 */
		constructor(x?: number, y?: number, radius?: number);

		/**
		 * Creates a clone of this Circle instance
		 *
		 * @return {Circle} a copy of the Circle
		 */
		clone(): Circle;

		/**
		 * @param {Point} point - The point to test
		 * @return {boolean} Whether the x/y coordinates are within this Circle
		 */
		isPointIn(point: Point): boolean;

		/**
		 * Returns the framing rectangle of the circle as a Rectangle object
		 *
		 * @return {Rectangle} the framing rectangle
		 */
		getBounds(): Rectangle;
	}

	export class Ellipse {
		/**
		 * @member {number}
		 * @default 0
		 */
		x: number;
		/**
		 * @member {number}
		 * @default 0
		 */
		y: number;
		/**
		 * @member {number}
		 * @default 0
		 */
		width: number;
		/**
		 * @member {number}
		 * @default 0
		 */
		height: number;
		/**
		 * The type of the object, mainly used to avoid `instanceof` checks
		 *
		 * @member {number}
		 * @readOnly
		 * @default SHAPES.ELIP
		 * @see SHAPES
		 */
		type: number;

		/**
		 * @param {number} [x=0] - The X coordinate of the center of this circle
		 * @param {number} [y=0] - The Y coordinate of the center of this circle
		 * @param {number} [width=0] - The half width of this ellipse
		 * @param {number} [height=0] - The half height of this ellipse
		 */
		constructor(x?: number, y?: number, width?: number, height?: number);

		/**
		 * Creates a clone of this Ellipse instance
		 *
		 * @return {Ellipse} a copy of the ellipse
		 */
		clone(): Ellipse;

		/**
		 * Checks whether the x and y coordinates given are contained within this ellipse
		 *
		 * @param {Point} point - The  point to test
		 * @return {boolean} Whether the x/y coords are within this ellipse
		 */
		isPointIn(point: Point): boolean;

		/**
		 * Returns the framing rectangle of the ellipse as a Rectangle object
		 *
		 * @return {Rectangle} the framing rectangle
		 */
		getBounds(): Rectangle;
	}

	export class Polygon {
		/**
		 * 是否闭合
		 */
		closed: boolean;
		/**
		 * [0,1, 2,3, 2,3, 2,3]
		 * An array of the points of this polygon
		 * @member {number[]}
		 */
		points: number[];
		/**
		 * The type of the object, mainly used to avoid `instanceof` checks
		 *
		 * @member {number}
		 * @readOnly
		 * @default SHAPES.POLY
		 * @see SHAPES
		 */
		type: number;

		/**
		 * new Polygon(new Point(), new Point(), ...)
		 *
		 * new Polygon([x,y, x,y, ...])
		 * new Polygon(x,y, x,y, x,y, ...)
		 * @param {Point[]|number[]} points - This can be an array of Points
		 *  that form the polygon, a flat array of numbers that will be interpreted as [x,y, x,y, ...], or
		 *  the arguments passed can be all the points of the polygon e.g.
		 *  `new Polygon(new Point(), new Point(), ...)`, or the arguments passed can be flat
		 *  x,y values e.g. `new Polygon(x,y, x,y, x,y, ...)` where `x` and `y` are Numbers.
		 */
		constructor(...points: any[]);

		/**
		 * Creates a clone of this polygon
		 *
		 * @return {Polygon} a copy of the polygon
		 */
		clone(): Polygon;

		/**
		 * Closes the polygon, adding points if necessary.
		 *
		 */
		close(): void;

		/**
		 * Checks whether the x and y coordinates passed to this function are contained within this polygon
		 *
		 * @param {Point} point - The  point to test
		 * @return {boolean} Whether the x/y coordinates are within this polygon
		 */
		isPointIn(point: Point): boolean;
	}

	export class RoundedRectangle {
		x: number;
		y: number;
		width: number;
		height: number;
		/**
		 * @member {number}
		 * @default 20
		 */
		radius: number;
		type: number;

		/**
		 * @param {number} [x=0] - The X coordinate of the upper-left corner of the rounded rectangle
		 * @param {number} [y=0] - The Y coordinate of the upper-left corner of the rounded rectangle
		 * @param {number} [width=0] - The overall width of this rounded rectangle
		 * @param {number} [height=0] - The overall height of this rounded rectangle
		 * @param {number} [radius=20] - Controls the radius of the rounded corners
		 */
		constructor(x?: number, y?: number, width?: number, height?: number, radius?: number);

		/**
		 * Creates a clone of this Rounded Rectangle
		 *
		 * @return {RoundedRectangle} a copy of the rounded rectangle
		 */
		clone(): RoundedRectangle;

		/**
		 * Checks whether the x and y coordinates given are contained within this ellipse
		 *
		 * @param {Point} point - The  point to test
		 * @return {boolean} Whether the x/y coords are within this ellipse
		 */
		isPointIn(point: Point): boolean;
	}

	export const INT_BITS1 = 32;

	export const INT_MAX = 2147483647;

	export const INT_MIN: number;

	export function abs(v: any): number;

	export function min(x: any, y: any): number;

	export function max(x: any, y: any): number;

	export function isPow2(v: any): boolean;

	export function log2(v: any): number;

	export function log10(v: any): 1 | 0 | 2 | 4 | 3 | 9 | 5 | 6 | 7 | 8;

	export function popCount(v: any): number;

	export function countTrailingZeros(v: any): number;

	export function nextPow2(v: any): any;

	export function prevPow2(v: any): number;

	export function parity(v: any): number;

	export function reverse(v: any): number;

	export function interleave2(x: any, y: any): number;

	export function deinterleave2(v: any, n: any): number;

	export function interleave3(x: any, y: any, z: any): number;

	export function deinterleave3(v: any, n: any): number;

	export function nextCombination(v: any): number;

	export interface RendererOptions {
		/**
		 * canvas标签,注意只有一种情况，canvasRender用于画质缓存canvas时不用，其他时候都必带
		 */
		htmlElement?: HTMLCanvasElement;
		/**
		 * If the render view is transparent
		 * 设置透明，则背景色无效
		 */
		transparent?: boolean;
		/**
		 * sets antialias (only applicable in chrome at the moment)
		 *
		 */
		antialias?: boolean;
		/**
		 * enables drawing buffer preservation,
		 * enable this if you need to call toDataUrl on the webgl context
		 */
		preserveDrawingBuffer?: boolean;
		/**
		 * The background color of the rendered area
		 * (shown if not transparent).0x000000
		 */
		backgroundColor?: number;
		/**
		 * If true will Math.floor() x/y values when rendering,
		 *  stopping pixel interpolation.
		 */
		roundPixels?: boolean;
	}

	export class SystemRenderer extends EventDispatcher {
		/**
		 * 渲染器参数
		 */
		options: RendererOptions;
		/**
		 * 渲染类型
		 *
		 * @member {number}
		 * @default RENDERER_TYPE.UNKNOWN
		 * @see RENDERER_TYPE
		 */
		type: RENDERER_TYPE;
		/**
		 * canvas对象
		 */
		htmlElement: HTMLCanvasElement;
		/**
		 * 是否透明
		 */
		transparent: boolean;
		/**
		 * The value of the preserveDrawingBuffer flag affects whether or not the contents of
		 * the stencil buffer is retained after rendering.
		 */
		preserveDrawingBuffer: boolean;
		/**
		 * 背景色，十六进制数值
		 * @member {number}
		 */
		_backgroundColor: number;
		/**
		 * 背景色，rgba值
		 * @member {number[]}
		 */
		_backgroundColorRgba: number[];
		/**
		 * 背景色，字符串
		 */
		_backgroundColorString: string;
		/**
		 * 临时父级对象，用于更新updateTransform
		 */
		_tempDisplayObjectParent: Container;
		/**
		 * 上一个被渲染的根显示对象
		 */
		_lastObjectRendered: DisplayObject;

		constructor(options: RendererOptions);

		/**
		 * 尺寸重置，暂时根据屏幕分辨率
		 * @param {number} screenWidth
		 * @param {number} screenHeight
		 */
		resize(width: number, height: number): void;

		/**
		 * 核心渲染方法，子级重写
		 */
		render(displayObject: any, renderTexture?: any, transform?: any): void;

		/**
		 * 销毁方法
		 */
		destroy(): void;

		/**
		 * 背景色，十六进制颜色
		 * @member {number}
		 */
		/**
		 * 背景色，十六进制颜色
		 */
		backgroundColor: number;
	}

	export class CanvasGraphicsRenderer {
		renderer: CanvasRenderer;

		/**
		 * @param {CanvasRenderer} renderer - The current renderer.
		 */
		constructor(renderer: CanvasRenderer);

		/**
		 * Renders a Graphics object to a canvas.
		 *
		 * @param {Graphics} graphics - the actual graphics object to render
		 */
		render(graphics: Graphics): void;

		/**
		 * destroy graphics object
		 *
		 */
		destroy(): void;
	}

	export function holePath(graphicsData: GraphicsData, context: CanvasRenderingContext2D, ccw?: boolean): void;

	export function judgeCcw(points: number[]): boolean;

	export class CanvasMaskManager extends HashObject {
		renderer: any;

		/**
		 * @param {CanvasRenderer} renderer - The canvas renderer.
		 */
		constructor(renderer: CanvasRenderer);

		/**
		 * This method adds it to the current stack of masks.
		 *
		 * @param {object} maskData - the maskData that will be pushed
		 */
		pushMask(maskData: Graphics): void;

		/**
		 * Renders a Graphics shape.
		 *
		 * @param {Graphics} graphics - The object to render.
		 */
		renderGraphicsShape(graphics: Graphics): void;

		/**
		 * Restores the current drawing context to the state it was before the mask was applied.
		 *
		 * @param {CanvasRenderer} renderer - The renderer context to use.
		 */
		popMask(renderer: CanvasRenderer): void;

		/**
		 * Destroys this canvas mask manager.
		 *
		 */
		destroy(): void;
	}

	export class CanvasRenderTarget {
		/**
		 * The Canvas object that belongs to this CanvasRenderTarget.
		 */
		canvas: HTMLCanvasElement;
		/**
		 * A CanvasRenderingContext2D object representing a two-dimensional rendering context.
		 */
		context: CanvasRenderingContext2D;

		/**
		 * @param {number} width - the width for the newly created canvas
		 * @param {number} height - the height for the newly created canvas
		 */
		constructor(width: number, height: number);

		/**
		 * Clears the canvas that was created by the CanvasRenderTarget class.
		 *
		 * @private
		 */
		clear(): void;

		/**
		 * Resizes the canvas to the specified width and height.
		 *
		 * @param {number} width - the new width of the canvas
		 * @param {number} height - the new height of the canvas
		 */
		resize(width: number, height: number): void;

		/**
		 * Destroys this canvas.
		 *
		 */
		destroy(): void;

		/**
		 * The width of the canvas buffer in pixels.
		 *
		 * @member {number}
		 */
		width: number;
		/**
		 * The height of the canvas buffer in pixels.
		 *
		 * @member {number}
		 */
		height: number;
	}

	export class BaseRenderTexture extends BaseTexture {
		/**
		 * A map of renderer IDs to webgl renderTargets
		 * 画在他的帧缓存中
		 * @private
		 * @member {object<number, WebGLTexture>}
		 */
		_glRenderTargets: {};
		/**
		 * A reference to the canvas render target (we only need one as this can be shared across renderers)
		 * 画在他的上下文中canvas中
		 * @private
		 * @member {object<number, WebGLTexture>}
		 */
		_canvasRenderTarget: any;
		/**
		 * This will let the renderer know if the texture is valid. If it's not then it cannot be rendered.
		 */
		valid: boolean;

		/**
		 * @param {number} [width=100] - The width of the base render texture
		 * @param {number} [height=100] - The height of the base render texture
		 * @param {number} [scaleMode=SCALE_MODES.LINEAR] - See {@link SCALE_MODES} for possible values
		 */
		constructor(width?: number, height?: number, scaleMode?: number);

		/**
		 * Resizes the BaseRenderTexture.
		 *
		 * @param {number} width - The width to resize to.
		 * @param {number} height - The height to resize to.
		 */
		resize(width: number, height: number): void;

		/**
		 * Destroys this texture
		 *
		 */
		destroy(): void;
	}

	export class RenderTexture extends Texture {
		/**
		 * @param {BaseRenderTexture} baseRenderTexture - The renderer used for this RenderTexture
		 * @param {Rectangle} [frame] - The rectangle frame of the texture to show
		 */
		constructor(baseRenderTexture: BaseRenderTexture, frame?: Rectangle);

		/**
		 * Resizes the RenderTexture.
		 *
		 * @param {number} width - The width to resize to.
		 * @param {number} height - The height to resize to.
		 * @param {boolean} doNotResizeBaseTexture - Should the baseTexture.width and height values be resized as well?
		 */
		resize(width: number, height: number, doNotResizeBaseTexture?: boolean): void;

		/**
		 * A short hand way of creating a render texture.
		 *
		 * @param {number} [width=100] - The width of the render texture
		 * @param {number} [height=100] - The height of the render texture
		 * @param {number} [scaleMode=settings.SCALE_MODE] - See {@link SCALE_MODES} for possible values
		 * @return {RenderTexture} The new render texture
		 */
		static create(width: number, height: number, scaleMode?: number): RenderTexture;
	}

	export class BatchDrawCall {
		/**
		 * 存储基础图片
		 */
		textures: BaseTexture[];
		/**
		 * 图片总数
		 */
		textureCount: number;
		/**
		 * 索引起始
		 */
		start: number;
		/**
		 * 总数
		 */
		size: number;
		/**
		 * 绘制类型
		 */
		type: number;

		constructor();
	}

	export class ObjectRenderer {
		/**
		 * The renderer this manager works for.
		 *
		 * @member {Renderer}
		 */
		renderer: WebglRenderer;

		constructor(renderer: WebglRenderer);

		/**
		 * Generic method called when there is a WebGL context change.
		 *
		 * @param {WebGLRenderingContext} gl new webgl context
		 */
		onContextChange(): void;

		/**
		 * Starts the renderer and sets the shader
		 *
		 */
		start(): void;

		/**
		 * Stops the renderer
		 *
		 */
		stop(): void;

		/**
		 * Stub method for rendering content and emptying the current batch.
		 *
		 */
		flush(): void;

		/**
		 * Renders an object
		 *
		 */
		render(object: DisplayObject): void;

		/**
		 * Generic destroy methods to be overridden by the subclass
		 */
		destroy(): void;
	}

	export function createContext(canvas: HTMLCanvasElement, options?: any): WebGLRenderingContext;

	export function checkMaxIfStatementsInShader(maxIfs: number, gl: WebGLRenderingContext): number;

	export class BatchBuffer {
		/**
		 * 顶点数据,类型化数组
		 * https://developer.mozilla.org/zh-CN/docs/Web/JavaScript/Reference/Global_Objects/ArrayBuffer
		 */
		vertices: ArrayBuffer;
		/**
		 * 顶点的位置信息视图，浮点数
		 * View on the vertices as a Float32Array for positions
		 */
		float32View: Float32Array;
		/**
		 * uv及颜色值的信息视图，整数
		 * View on the vertices as a Uint32Array for uvs
		 */
		uint32View: Uint32Array;
		/**
		 * positions,uvs,colors暂时不用做单独记录
		 */
		positions: any;
		uvs: any;
		colors: any;

		/**
		 * @param {number} size - The size of the buffer in bytes.
		 */
		constructor(size: number);

		/**
		 * Destroys the buffer.
		 *
		 */
		destroy(): void;
	}

	export class GLTexture {
		/**
		 * 当前上下文
		 * The current WebGL rendering context
		 */
		gl: WebGLRenderingContext;
		texture: WebGLTexture;
		/**
		 * If mipmapping was used for this texture, enable and disable with enableMipmap()
		 * 是否对纹理进行存储缩小的各种尺寸纹理，比如原图1024*1024，存储512*512，256*256，128*128等一直到1*1;为了纹理的缩放时处理，是取相邻或线性插值
		 */
		mipmap: boolean;
		/**
		 * Set to true to enable pre-multiplied alpha
		 * 设置纹理预乘透明值，为true，https://blog.csdn.net/mydreamremindme/article/details/50817294
		 */
		premultiplyAlpha: any;
		/**
		 * 纹理宽度
		 */
		width: number;
		/**
		 * 纹理高度
		 */
		height: number;
		/**
		 * {number} the pixel format of the texture. defaults to gl.RGBA
		 * 纹理格式，默认gl.RGBA  还有gl.RGB
		 */
		format: any;
		/**
		 * {number} the gl type of the texture. defaults to gl.UNSIGNED_BYTE
		 * 纹理类型，默认gl.UNSIGNED_BYTE   //https://developer.mozilla.org/zh-CN/docs/Web/API/WebGLRenderingContext/texImage2D
		 */
		type: any;

		constructor(gl: WebGLRenderingContext, width?: number, height?: number, format?: any, type?: any);

		/**
		 * Uploads this texture to the GPU
		 * GPU存储纹理数据
		 * @param source {HTMLImageElement|ImageData|HTMLVideoElement} the source image of the texture
		 */
		upload(source: any): void;

		/**
		 * Use a data source and uploads this texture to the GPU
		 * 数据类型的纹理
		 * @param data {TypedArray} the data to upload to the texture
		 * @param width {number} the new width of the texture
		 * @param height {number} the new height of the texture
		 */
		uploadData: (data: any, width: any, height: any) => void;

		/**
		 * Binds the texture
		 * 绑定纹理，不传location表示不激活额外纹理，绑定的纹理位置与原状态相同
		 * @param  location
		 */
		bind(location?: number): void;

		/**
		 * Unbinds the texture
		 * 解除纹理绑定，解除位置与原状态相同
		 */
		unbind(): void;

		/**
		 * @param linear {Boolean} if we want to use linear filtering or nearest neighbour interpolation
		 * 缩小的纹理像素 按线性插值，还是按钮邻近原则
		 */
		minFilter(linear: boolean): void;

		/**
		 * @param linear {Boolean} if we want to use linear filtering or nearest neighbour interpolation
		 * 放大的纹理像素 按线性插值，还是按钮邻近原则
		 */
		magFilter(linear: boolean): void;

		/**
		 * Enables mipmapping
		 * 生成缩小的纹理集，只能在图片宽高满足2的指数时使用
		 */
		enableMipmap(): void;

		/**
		 * Enables linear filtering
		 * 设置线性
		 */
		enableLinearScaling(): void;

		/**
		 * Enables nearest neighbour interpolation
		 * 设置邻近
		 */
		enableNearestScaling(): void;

		/**
		 * Enables clamping on the texture so WebGL will not repeat it
		 * 如果纹理不满足2的指数时必设，以边缘像素延申
		 */
		enableWrapClamp(): void;

		/**
		 * Enable tiling on the texture
		 * 允许纹理重复，地砖模式
		 */
		enableWrapRepeat(): void;

		/**
		 * 镜像形式重复
		 */
		enableWrapMirrorRepeat(): void;

		/**
		 * Destroys this texture
		 */
		destroy(): void;

		/**
		 * 从图片数据创建纹理
		 * @static
		 * @param gl {WebGLRenderingContext} The current WebGL context
		 * @param source {HTMLImageElement|ImageData} the source image of the texture
		 * @param premultiplyAlpha {Boolean} If we want to use pre-multiplied alpha
		 */
		static fromSource(gl: WebGLRenderingContext, source: HTMLImageElement | ImageData, premultiplyAlpha?: boolean): GLTexture;

		/**
		 * @static
		 * @param gl {WebGLRenderingContext} The current WebGL context
		 * @param data {TypedArray} the data to upload to the texture
		 * @param width {number} the new width of the texture
		 * @param height {number} the new height of the texture
		 */
		static fromData(gl: any, data: any, width: any, height: any): GLTexture;
	}

	export class GLBuffer {
		/**
		 * 当前上下文
		 * @member {WebGLRenderingContext}
		 */
		gl: any;
		/**
		 * The WebGL buffer, created upon instantiation
		 * @member {WebGLBuffer}
		 */
		buffer: any;
		/**
		 * buffer类型
		 * 顶点或索引
		 * @member {gl.ARRAY_BUFFER|gl.ELEMENT_ARRAY_BUFFER}
		 */
		type: any;
		/**
		 * The draw type of the buffer
		 * 绘制类型
		 * gl.STATIC_DRAW: 缓冲区的内容可能经常使用，而不会经常更改
		 * gl.DYNAMIC_DRAW: 缓冲区的内容可能经常被使用，并且经常更改
		 * gl.STREAM_DRAW: 缓冲区的内容可能不会经常使用,且不会经常更改
		 * @member {gl.STATIC_DRAW|gl.DYNAMIC_DRAW|gl.STREAM_DRAW}
		 */
		drawType: any;
		/**
		 * The data in the buffer, as a typed array
		 * 用来表示通用的、固定长度的原始二进制数据缓冲区。ArrayBuffer 不能直接操作，
		 * 而是要通过类型数组对象或 DataView 对象来操作，
		 * 它们会将缓冲区中的数据表示为特定的格式，并通过这些格式来读写缓冲区的内容。
		 * @member {ArrayBuffer| SharedArrayBuffer|ArrayBufferView}
		 */
		data: any;
		/**
		 * 更新ID
		 */
		_updateID: number;

		constructor(gl: any, type?: any, data?: any, drawType?: any);

		/**
		 * 上传数据
		 * Uploads the buffer to the GPU
		 * @param data {ArrayBuffer| SharedArrayBuffer|ArrayBufferView} an array of data to upload
		 * @param offset {Number} if only a subset of the data should be uploaded, this is the amount of data to subtract
		 * @param dontBind {Boolean} whether to bind the buffer before uploading it  是否不绑定buffer
		 */
		upload(data: any, offset?: number, dontBind?: boolean): void;

		/**
		 * Binds the buffer
		 * 状态机接下来使用的buffer
		 *
		 */
		bind(): void;

		/**
		 * Destroys the buffer
		 *
		 */
		destroy: () => void;

		/**
		 * 创建顶点缓存
		 * @param gl
		 * @param data
		 * @param drawType
		 */
		static createVertexBuffer(gl: any, data?: any, drawType?: any): GLBuffer;

		/**
		 * 创建索引缓存
		 * @param gl
		 * @param data
		 * @param drawType
		 */
		static createIndexBuffer(gl: any, data?: any, drawType?: any): GLBuffer;

		static create(gl: any, type: any, data: any, drawType: any): GLBuffer;
	}

	export function setVertexAttribArrays(gl: any, attribs: any, state?: any): void;

	export class VertexArrayObject {
		nativeVaoExtension: any;
		nativeState: any;
		nativeVao: any;
		/**
		 * 当前上下文
		 */
		gl: WebGLRenderingContext;
		/**
		 * An array of attributes
		 * attributes数组
		 */
		attributes: any[];
		/**
		 * 索引buffer
		 *  @member {GLBuffer}
		 */
		indexBuffer: any;
		/**
		 * A boolean flag
		 */
		dirty: boolean;

		constructor(gl: WebGLRenderingContext, state: any);

		/**
		 * Binds the buffer
		 * 绑定数据
		 */
		bind(): this;

		/**
		 * Unbinds the buffer
		 * 解绑数据
		 */
		unbind(): this;

		/**
		 * Uses this vao
		 * 激活vao
		 */
		activate(): this;

		/**
		 * 添加attribute
		 * @param buffer     {gl.GLBuffer}
		 * @param attribute  {*}
		 * @param type       {String}
		 * @param normalized {Boolean}
		 * @param stride     {Number}
		 * @param start      {Number}
		 */
		addAttribute(buffer: any, attribute: any, type?: any, normalized?: any, stride?: any, start?: any): this;

		/**
		 * 添加索引数据
		 * @param buffer   {gl.GLBuffer}
		 */
		addIndex(buffer: any): this;

		/**
		 * Unbinds this vao and disables it
		 * 解绑废弃vao
		 */
		clear(): this;

		/**
		 * 执行绘制
		 * @param type  {Number} gl.TRIANGLES\gl.TRIANGLE_STRIP等
		 * @param size  {Number} 个数
		 * @param start {Number} 偏移
		 */
		draw(type: any, size?: any, start?: any): this;

		/**
		 * Destroy this vao
		 */
		destroy(): void;

		getSize(): number;

		/**
		 * Some devices behave a bit funny when using the newer extensions (im looking at you ipad 2!)
		 * If you find on older devices that things have gone a bit weird then set this to true.
		 */
		/**
		 * Lets the VAO know if you should use the WebGL extension or the native methods.
		 * Some devices behave a bit funny when using the newer extensions (im looking at you ipad 2!)
		 * If you find on older devices that things have gone a bit weird then set this to true.
		 * @static
		 * @property {Boolean} FORCE_NATIVE
		 */
		static FORCE_NATIVE: boolean;
	}

	export class GLFramebuffer {
		gl: any;
		framebuffer: any;
		stencil: any;
		texture: any;
		width: any;
		height: any;

		constructor(gl: any, width: any, height: any);

		/**
		 * Adds a texture to the frame buffer
		 * @param texture {glCore.GLTexture}
		 */
		enableTexture(texture: any): void;

		/**
		 * Initialises the stencil buffer
		 */
		enableStencil(): void;

		/**
		 * Erases the drawing area and fills it with a colour
		 * @param  r {Number} the red value of the clearing colour
		 * @param  g {Number} the green value of the clearing colour
		 * @param  b {Number} the blue value of the clearing colour
		 * @param  a {Number} the alpha value of the clearing colour
		 */
		clear(r: any, g: any, b: any, a: any): void;

		/**
		 * Binds the frame buffer to the WebGL context
		 */
		bind(): void;

		/**
		 * Unbinds the frame buffer to the WebGL context
		 */
		unbind(): void;

		/**
		 * Resizes the drawing area of the buffer to the given width and height
		 * @param  width  {Number} the new width
		 * @param  height {Number} the new height
		 */
		resize(width: any, height: any): void;

		/**
		 * Destroys this buffer
		 */
		destroy(): void;

		/**
		 * Creates a frame buffer with a texture containing the given data
		 * @static
		 * @param gl {WebGLRenderingContext} The current WebGL rendering context
		 * @param width {Number} the width of the drawing area of the frame buffer
		 * @param height {Number} the height of the drawing area of the frame buffer
		 * @param data {ArrayBuffer| SharedArrayBuffer|ArrayBufferView} an array of data
		 */
		static createRGBA(gl: any, width: any, height: any, data?: any): GLFramebuffer;

		/**
		 * Creates a frame buffer with a texture containing the given data
		 * @static
		 * @param gl {WebGLRenderingContext} The current WebGL rendering context
		 * @param width {Number} the width of the drawing area of the frame buffer
		 * @param height {Number} the height of the drawing area of the frame buffer
		 * @param data {ArrayBuffer| SharedArrayBuffer|ArrayBufferView} an array of data
		 */
		static createFloat32(gl: any, width: any, height: any, data: any): GLFramebuffer;
	}

	export function compileProgram(gl: WebGLRenderingContext, vertexSrc: string, fragmentSrc: string, attributeLocations?: any): WebGLProgram;

	export function mapType(gl: any, type: any): any;

	export function mapSize(type: string): number;

	export function extractAttributes(gl: WebGLRenderingContext, program: WebGLProgram): {};

	export function defaultValue(type: string, size: number): false | any[] | Int32Array | 0;

	export function extractUniforms(gl: WebGLRenderingContext, program: WebGLProgram): {};

	export function setPrecision(src: string, precision: string): string;

	export function generateUniformAccessObject(gl: WebGLRenderingContext, uniformData: any): {
		data: {};
	}

	export class GLShader {
		/**
		 * The current WebGL rendering context
		 * @member {WebGLRenderingContext}
		 */
		gl: WebGLRenderingContext;
		/**
		 * The shader program
		 * @member {WebGLProgram}
		 */
		program: WebGLProgram;
		/**
		 *  The attributes of the shader as an object containing the following properties
		 * {
		 * 	type,
		 * 	size,
		 * 	location,
		 * 	pointer
		 * }
		 * @member {Object}
		 */
		attributes: any;
		uniformData: any;
		/**
		 * The uniforms of the shader as an object containing the following properties
		 * {
		 * 	gl,
		 * 	data
		 * }
		 * @member {Object}
		 */
		uniforms: any;

		constructor(gl: WebGLRenderingContext, vertexSrc: string, fragmentSrc: string, precision?: string, attributeLocations?: any);

		/**
		 * Uses this shader
		 * 状态机当前使用的shader
		 * @return {glCore.GLShader} Returns itself.
		 */
		bind(): this;

		/**
		 * Destroys this shader
		 * TODO
		 */
		destroy(): void;
	}

	export function generateMultiTextureShader(gl: any, maxTextures: any): GLShader;

	export class BatchRenderer extends ObjectRenderer {
		vertSize: number;
		vertByteSize: number;
		size: number;
		/**
		 * 当前顶点数量
		 */
		currentSize: number;
		/**
		 * 当前索引数量
		 */
		currentIndexSize: number;
		/**
		 * 顶点数据
		 */
		aBuffers: {};
		/**
		 * 索引数据
		 */
		iBuffers: {};
		/**
		 * The default shaders that is used if a sprite doesn't have a more specific one.
		 * there is a shader for each number of textures that can be rendered.
		 * These shaders will also be generated on the fly as required.
		 */
		shader: GLShader;
		/**
		 * 当前累计的绘制对象数量
		 */
		currentIndex: number;
		groups: BatchDrawCall[];
		/**
		 * 绘制对象数组
		 * 包括图片的和矢量图的
		 */
		elements: any[];
		vaos: VertexArrayObject[];
		vaoMax: number;
		vertexCount: number;
		MAX_TEXTURES: number;
		vertexBuffers: GLBuffer[];
		indexBuffers: GLBuffer[];

		constructor(renderer: WebglRenderer);

		/**
		 * Sets up the renderer context and necessary buffers.
		 */
		onContextChange(): void;

		/**
		 * Called before the renderer starts rendering.
		 *
		 */
		onPreRender(): void;

		/**
		 * Renders the sprite object.
		 * element必须的属性
		 *
		 * _texture  里面有.BaseTexture
		 * vertexData
		 * indices
		 * worldAlpha
		 * _tintRGB
		 * uvs
		 *
		 * the sprite to render when using this spritebatch
		 */
		render(element: any): void;

		/**
		 * 获得索引buffer
		 * @param size
		 */
		getIndexBuffer(size: any): any;

		/**
		 * 获取相应的顶点数据buffer
		 * @param size
		 */
		getAttributeBuffer(size: number): any;

		/**
		 * Renders the content and empties the current batch.
		 *
		 */
		flush(): void;

		packGeometry(element: any, float32View: any, uint32View: any, indexBuffer: any, index: any, indexCount: any): void;

		/**
		 * Starts a new sprite batch.
		 */
		start(): void;

		/**
		 * Stops and flushes the current batch.
		 *
		 */
		stop(): void;

		/**
		 * Destroys the SpriteRenderer.
		 *
		 */
		destroy(): void;
	}

	export class RenderTarget {
		/**
		 * The current WebGL drawing context.
		 *
		 */
		gl: WebGLRenderingContext;
		/**
		 * A frame buffer
		 */
		frameBuffer: GLFramebuffer;
		/**
		 * The texture
		 */
		texture: GLTexture;
		/**
		 * The background colour of this render target, as an array of [r,g,b,a] values
		 *
		 */
		clearColor: number[];
		/**
		 * The size of the object as a rectangle
		 */
		size: Rectangle;
		/**
		 * The projection matrix
		 */
		projectionMatrix: Matrix;
		/**
		 * The object's transform
		 */
		transform: Matrix;
		/**
		 * The frame.
		 *
		 */
		frame: Rectangle;
		defaultFrame: Rectangle;
		destinationFrame: any;
		sourceFrame: any;
		/**
		 * The stencil buffer stores masking data for the render target
		 *
		 */
		stencilBuffer: GLBuffer;
		/**
		 * The data structure for the stencil masks
		 */
		stencilMaskStack: Graphics[];
		/**
		 * The scale mode.
		 *
		 */
		scaleMode: number;
		/**
		 * Whether this object is the root element or not
		 */
		root: boolean;

		/**
		 * @param {WebGLRenderingContext} gl - The current WebGL drawing context
		 * @param {number} [width=0] - the horizontal range of the filter
		 * @param {number} [height=0] - the vertical range of the filter
		 * @param {number} [scaleMode=SCALE_MODES.LINEAR] - See {@link SCALE_MODES} for possible values
		 * @param {boolean} [root=false] - Whether this object is the root element or not
		 */
		constructor(gl: WebGLRenderingContext, width?: number, height?: number, scaleMode?: number, root?: boolean);

		/**
		 * Clears the filter texture.
		 *
		 * @param {number[]} [clearColor=this.clearColor] - Array of [r,g,b,a] to clear the framebuffer
		 */
		clear(clearColor?: number[]): void;

		/**
		 * Binds the stencil buffer.
		 *
		 */
		attachStencilBuffer(): void;

		/**
		 * Sets the frame of the render target.
		 *
		 * @param {Rectangle} destinationFrame - The destination frame.
		 * @param {Rectangle} sourceFrame - The source frame.
		 */
		setFrame(destinationFrame?: Rectangle, sourceFrame?: Rectangle): void;

		/**
		 * Binds the buffers and initialises the viewport.
		 *
		 */
		activate(): void;

		/**
		 * Updates the projection matrix based on a projection frame (which is a rectangle)
		 *
		 * @param {Rectangle} destinationFrame - The destination frame.
		 * @param {Rectangle} sourceFrame - The source frame.
		 */
		calculateProjection(destinationFrame: Rectangle, sourceFrame?: Rectangle): void;

		/**
		 * Resizes the texture to the specified width and height
		 *
		 * @param {number} width - the new width of the texture
		 * @param {number} height - the new height of the texture
		 */
		resize(width: number, height: number): void;

		/**
		 * 销毁方法
		 */
		destroy(): void;
	}

	export class TextureManager {
		boundTextures: any;
		renderer: WebglRenderer;
		emptyTextures: any;
		_nextTextureLocation: number;
		currentLocation: number;
		gl: WebGLRenderingContext;
		/**
		 * Track textures in the renderer so we can no longer listen to them on destruction.
		 */
		_managedTextures: Array<any>;

		/**
		 * @param {WebGLRenderer} renderer - A reference to the current renderer
		 */
		constructor(renderer: WebglRenderer);

		onContextChange(): void;

		/**
		 * Binds the texture. This will return the location of the bound texture.
		 * It may not be the same as the one you pass in. This is due to optimisation that prevents
		 * needless binding of textures. For example if the texture is already bound it will return the
		 * current location of the texture instead of the one provided. To bypass this use force location
		 *
		 * @param {Texture} texture - the new texture
		 * @param {number} location - the suggested texture location
		 * @param {boolean} forceLocation - force the location
		 * @return {number} bound texture location
		 */
		bindTexture(texture: any, location?: number, forceLocation?: boolean): number;

		/**
		 * Gets a texture.
		 *
		 */
		getTexture(): void;

		/**
		 * Updates and/or Creates a WebGL texture for the renderer's context.
		 *
		 * @param {BaseTexture|Texture} texture - the texture to update
		 * @param {number} location - the location the texture will be bound to.
		 * @return {GLTexture} The gl texture.
		 */
		updateTexture(texture: any, location?: number): GLTexture;

		/**
		 * unbinds the texture ...
		 *
		 * @param {Texture} texture - the texture to unbind
		 */
		unbindTexture(texture: any): void;

		/**
		 * Deletes the texture from WebGL
		 *
		 * @param {BaseTexture|Texture} texture - the texture to destroy
		 * @param {boolean} [skipRemove=false] - Whether to skip removing the texture from the TextureManager.
		 */
		destroyTexture(texture: any, skipRemove: any): void;

		/**
		 * Deletes all the textures from WebGL
		 */
		removeAll(): void;

		/**
		 * Destroys this manager and removes all its textures
		 */
		destroy(): void;
	}

	export class TextureGarbageCollector {
		renderer: WebglRenderer;
		count: number;
		checkCount: number;
		maxIdle: number;
		checkCountMax: number;
		mode: number;

		/**
		 * @param {WebGLRenderer} renderer - The renderer this manager works for.
		 */
		constructor(renderer: WebglRenderer);

		/**
		 * Checks to see when the last time a texture was used
		 * if the texture has not been used for a specified amount of time it will be removed from the GPU
		 */
		update(): void;

		/**
		 * Checks to see when the last time a texture was used
		 * if the texture has not been used for a specified amount of time it will be removed from the GPU
		 */
		run(): void;

		/**
		 * Removes all the textures within the specified displayObject and its children from the GPU
		 *
		 * @param {DisplayObject} displayObject - the displayObject to remove the textures from.
		 */
		unload(displayObject: any): void;
	}

	export class WebGLState {
		blendModes: any;
		/**
		 * The current active state
		 *
		 */
		activeState: Uint8Array;
		/**
		 * The default state
		 *
		 */
		defaultState: Uint8Array;
		gl: any;
		maxAttribs: any;
		attribState: {
			tempAttribState: any[];
			attribState: any[];
		};
		nativeVaoExtension: any;

		/**
		 * @param {WebGLRenderingContext} gl - The current WebGL rendering context
		 */
		constructor(gl: WebGLRenderingContext);

		/**
		 * Sets the current state
		 *
		 * @param {*} state - The state to set.
		 */
		setState(state: any): void;

		/**
		 * Enables or disabled blending.
		 *
		 * @param {boolean} value - Turn on or off webgl blending.
		 */
		setBlend(value: any): void;

		/**
		 * Sets the blend mode.
		 *
		 * @param {number} value - The blend mode to set to.
		 */
		setBlendMode(value: any): void;

		/**
		 * Sets whether to enable or disable depth test.
		 *
		 * @param {boolean} value - Turn on or off webgl depth testing.
		 */
		setDepthTest(value: any): void;

		/**
		 * Sets whether to enable or disable cull face.
		 *
		 * @param {boolean} value - Turn on or off webgl cull face.
		 */
		setCullFace(value: any): void;

		/**
		 * Sets the gl front face.
		 *
		 * @param {boolean} value - true is clockwise and false is counter-clockwise
		 */
		setFrontFace(value: any): void;

		/**
		 * Disables all the vaos in use
		 *
		 */
		resetAttributes(): void;

		/**
		 * Resets all the logic and disables the vaos
		 */
		resetToDefault(): void;

		/**
		 *
		 */
		resetTo3D(): void;
	}

	export class BatchManager {
		/**
		 *
		 */
		renderer: WebglRenderer;
		/**
		 * 空
		 */
		emptyRenderer: ObjectRenderer;
		/**
		 * 当前使用的ObjectRenderer
		 * @member {ObjectRenderer}
		 */
		currentRenderer: ObjectRenderer;

		constructor(renderer: WebglRenderer);

		/**
		 * Changes the current renderer to the one given in parameter
		 *
		 * @param {ObjectRenderer} objectRenderer - The object renderer to use.
		 */
		setObjectRenderer(objectRenderer: ObjectRenderer): void;

		/**
		 * This should be called if you wish to do some custom rendering
		 * It will basically render anything that may be batched up such as sprites
		 */
		flush(): void;

		/**
		 * Reset the system to an empty renderer
		 */
		reset(): void;
	}

	export class MaskManager {
		renderer: WebglRenderer;
		scissor: boolean;
		scissorData: any;
		scissorRenderTarget: RenderTarget;
		enableScissor: boolean;
		alphaMaskPool: any[];
		alphaMaskIndex: number;

		/**
		 * - The renderer this manager works for.
		 */
		constructor(renderer: WebglRenderer);

		/**
		 * Applies the Mask and adds it to the current filter stack.
		 *
		 * @param {DisplayObject} target - Display Object to push the mask to
		 * @param {Sprite|Graphics} maskData - The masking data.
		 */
		pushMask(target: any, maskData: any): void;

		/**
		 * Removes the last mask from the mask stack and doesn't return it.
		 *
		 * @param {DisplayObject} target - Display Object to pop the mask from
		 * @param {Sprite|Graphics} maskData - The masking data.
		 */
		popMask(target: any, maskData: any): void;

		/**
		 * Applies the Mask and adds it to the current filter stack.
		 *
		 * @param {Graphics} maskData - The masking data.
		 */
		pushStencilMask(maskData: Graphics): void;

		/**
		 * Removes the last filter from the filter stack and doesn't return it.
		 *
		 */
		popStencilMask(): void;

		/**
		 *
		 * @param {DisplayObject} target - Display Object to push the mask to
		 * @param {Graphics} maskData - The masking data.
		 */
		pushScissorMask(target: any, maskData: Graphics): void;

		/**
		 *
		 *
		 */
		popScissorMask(): void;

		destroy(): void;
	}

	export class StencilManager {
		renderer: WebglRenderer;
		stencilMaskStack: Graphics[];

		/**
		 * @param {WebGLRenderer} renderer - The renderer this manager works for.
		 */
		constructor(renderer: WebglRenderer);

		/**
		 * Changes the mask stack that is used by this manager.
		 *
		 * @param {Graphics[]} stencilMaskStack - The mask stack
		 */
		setMaskStack(stencilMaskStack: Graphics[]): void;

		/**
		 * Applies the Mask and adds it to the current stencil stack. @alvin
		 *
		 * @param {Graphics} graphics - The mask
		 */
		pushStencil(graphics: Graphics): void;

		/**
		 * Removes the last mask from the stencil stack. @alvin
		 */
		popStencil(): void;

		/**
		 * Setup renderer to use the current stencil data.
		 */
		_useCurrent(): void;

		/**
		 * Fill 1s equal to the number of acitve stencil masks.
		 *
		 * @return {number} The bitwise mask.
		 */
		_getBitwiseMask(): number;

		/**
		 * Destroys the mask stack.
		 *
		 */
		destroy(): void;
	}

	export class WebglRenderer extends SystemRenderer {
		/**
		 * 所有插件列表，目前只有batch
		 */
		plugins: {};
		/**
		 * webgl上下文
		 */
		gl: WebGLRenderingContext;
		/**
		 * 渲染器位移标识
		 */
		CONTEXT_UID: number;
		/**
		 * 遮罩管理
		 */
		maskManager: MaskManager;
		/**
		 * 模板管理
		 */
		stencilManager: StencilManager;
		/**
		 * 批处理管理
		 */
		batchManager: BatchManager;
		/**
		 * 纹理管理
		 */
		textureManager: TextureManager;
		/**
		 * 纹理回收器，对于长期未使用的纹理，从gpu移除
		 */
		textureGC: TextureGarbageCollector;
		/**
		 * 状态机，暂时不需要切换，以后有3d再说
		 */
		state: WebGLState;
		/**
		 * 是否渲染到主画布
		 */
		renderingToScreen: boolean;
		/**
		 * 当前着色器对象
		 */
		_activeShader: GLShader;
		/**
		 * 当前vao
		 */
		_activeVao: VertexArrayObject;
		/**
		 * 当前渲染对象，不是帧缓存时就是root
		 */
		_activeRenderTarget: RenderTarget;
		/**
		 * 主画布
		 */
		rootRenderTarget: RenderTarget;

		constructor(options: RendererOptions);

		_initContext(): void;

		render(displayObject: DisplayObject, renderTexture?: any, transform?: any): void;

		/**
		 * Erases the active render target and fills the drawing area with a colour
		 *
		 * @param {number} [clearColor] - The colour
		 */
		clear(clearColor: number[]): void;

		/**
		 * Sets the transform of the active render target to the given matrix
		 *
		 * @param {Matrix} matrix - The transformation matrix
		 */
		setTransform(matrix: Matrix): void;

		/**
		 * Erases the render texture and fills the drawing area with a colour
		 *
		 * @param {RenderTexture} renderTexture - The render texture to clear
		 * @param {number} [clearColor] - The colour
		 * @return {WebGLRenderer} Returns itself.
		 */
		clearRenderTexture(renderTexture: any, clearColor: any): this;

		/**
		 * Changes the current shader to the one given in parameter
		 *
		 * @param {Shader} shader - the new shader
		 * @param {boolean} [autoProject=true] - Whether automatically set the projection matrix
		 * @return {WebGLRenderer} Returns itself.
		 */
		bindShader(shader: GLShader, autoProject?: boolean): WebglRenderer;

		/**
		 * Creates a new VAO from this renderer's context and state.
		 *
		 * @return {VertexArrayObject} The new VAO.
		 */
		createVao(): VertexArrayObject;

		/**
		 * Changes the current Vao to the one given in parameter
		 *
		 * @param {VertexArrayObject} vao - the new Vao
		 * @return {WebGLRenderer} Returns itself.
		 */
		bindVao(vao: VertexArrayObject): WebglRenderer;

		/**
		 * Resets the WebGL state so you can render things however you fancy!
		 *
		 * @return {WebGLRenderer} Returns itself.
		 */
		reset(): WebglRenderer;

		/**
		 * Binds a render texture for rendering
		 *
		 * @param {RenderTexture} renderTexture - The render texture to render
		 * @param {Matrix} transform - The transform to be applied to the render texture
		 * @return {WebGLRenderer} Returns itself.
		 */
		bindRenderTexture(renderTexture: any, transform?: Matrix): WebglRenderer;

		/**
		 * 绑定当前渲染对象
		 * @param {RenderTarget} renderTarget - the new render target
		 * @return {WebglRenderer} Returns itself.
		 */
		bindRenderTarget(renderTarget: RenderTarget): WebglRenderer;

		/**
		 *
		 * @param screenWidth
		 * @param screenHeight
		 */
		resize(screenWidth: any, screenHeight: any): void;

		/**
		 * 渲染器销毁方法
		 */
		destroy(): void;

		/**
		 * webgl上下文恢复时
		 * @private
		 */
		handleContextRestored(): void;

		/**
		 * webgl上下文丢失时
		 * @private
		 * @param {WebGLContextEvent} event - The context lost event.
		 */
		handleContextLost(event: WebGLContextEvent): void;

		/**
		 * 初始化插件
		 * @protected
		 * @param {object} staticMap - The dictionary of statically saved plugins.
		 */
		initPlugins(staticMap: any): void;

		destroyPlugins(): void;

		static __plugins: any;

		/**
		 * Adds a plugin to the renderer.
		 *
		 * @method
		 * @param {string} pluginName - The name of the plugin.
		 * @param {Function} ctor - The constructor function or class for the plugin.
		 */
		static registerPlugin(pluginName: string, ctor: Function): void;
	}

	export class Sprite extends Container {
		/**
		 * 关于贴图的锚点，0到1，默认为texture自己的
		 *
		 */
		private _anchorTexture;
		/**
		 * 色值调色
		 */
		private _tint;
		/**
		 * RGB形式色值，webgl用
		 */
		_tintRGB: number;
		/**
		 * 和_tint比较用，用于canvas调色缓存
		 */
		_cachedTint: number;
		/**
		 * 使用的贴图
		 * @member {Texture}
		 */
		_texture: Texture;
		/**
		 * 初始化为texture的宽度
		 */
		_width: number;
		/**
		 * 初始化为texture的宽度
		 */
		_height: number;
		/**
		 * 标记更新transform
		 */
		_transformID: number;
		/**
		 * 标记更新过texture
		 */
		_textureID: number;
		_transformTrimmedID: number;
		_textureTrimmedID: number;
		/**
		 * 指定渲染用的渲染器插件
		 */
		pluginName: string;
		/**
		 * 顶点索引，下面几个均为webgl用到
		 */
		indices: Uint16Array;
		start: number;
		uvs: Float32Array;
		/**
		 * 顶点数据，长度8
		 */
		vertexData: Float32Array;
		/**
		 * Trimmed的顶点数据，用于裁剪掉过透明像素的texture
		 */
		vertexTrimmedData: Float32Array;

		/**
		 * @param {Texture} texture
		 */
		constructor(texture?: Texture);

		/**
		 * texture更新时触发
		 *
		 * @private
		 */
		_onTextureUpdate(): void;

		/**
		 * 当贴图锚点修改时
		 *
		 * @private
		 */
		_onAnchorUpdate(): void;

		/**
		 * 01——23
		 * |   |
		 * 67——45
		 * calculates worldTransform * vertices, store it in vertexData
		 */
		calculateVertices(): void;

		/**
		 * calculates worldTransform * vertices for a non texture with a trim. store it in vertexTrimmedData
		 * This is used to ensure that the true width and height of a trimmed texture is respected
		 */
		calculateTrimmedVertices(): void;

		/**
		 * 自身webgl绘制方法
		 * @private
		 * @param {WebglRenderer} renderer
		 */
		_renderWebGL(renderer: WebglRenderer): void;

		/**
		 * 自身canvas绘制方法
		 * @private
		 * @param {CanvasRenderer} renderer
		 */
		_renderCanvas(renderer: CanvasRenderer): void;

		/**
		 * 更新自己的bounds，计算全局
		 * @private
		 */
		_calculateBounds(): void;

		/**
		 * 重写父类
		 * @param {Rectangle} rect - The output rectangle.
		 * @return {Rectangle} The bounds.
		 */
		getLocalBounds(rect?: Rectangle): Rectangle;

		/**
		 * 重写碰撞检测方法
		 * @param globalPoint
		 * @param isMouseEvent
		 */
		hitTestPoint(globalPoint: Point, isMouseEvent?: boolean): any;

		/**
		 * 销毁
		 */
		destroy(): void;

		/**
		 * 重写Container父类，废弃先
		 * texture的宽度和缩放乘积
		 * @member {number}
		 */
		width: number;
		/**
		 * texture的高度和缩放乘积
		 * @member {number}
		 */
		height: number;
		/**
		 * 0，0标识左上角，0.5，0.5表示中间，1，1表示右下角
		 * @member {ObservablePoint}
		 */
		anchorTexture: any;
		/**
		 * @member {Texture}
		 */
		texture: Texture;
		tint: number;

		/**
		 * 方便创建sprite
		 * The source can be - frame id, image url, canvas element, base texture
		 * @static
		 * @param {number|string|BaseTexture|HTMLCanvasElement} source
		 * @return {Sprite} The newly created sprite
		 */
		static from(source: any): Sprite;

		/**
		 * TextureCache缓存里取得frameId，通常图集里得名字
		 * @static
		 * @param {string} frameId - The frame Id of the texture in the cache
		 * @return {Sprite} A new Sprite using a texture from the texture cache matching the frameId
		 */
		static fromFrame(frameId: string): Sprite;

		/**
		 * 根据图片路径创建sprite
		 * @static
		 * @param {string} imageId - The image url of the texture
		 * @param {boolean} [crossorigin=(auto)] - if you want to specify the cross-origin parameter
		 * @param {number} [scaleMode=settings.SCALE_MODE] - if you want to specify the scale mode,
		 *  see {@link SCALE_MODES} for possible values
		 * @return {Sprite} A new Sprite using a texture from the texture cache matching the image id
		 */
		static fromImage(imageId: string, crossorigin?: boolean, scaleMode?: number): Sprite;
	}

	export class CanvasSpriteRenderer {
		renderer: CanvasRenderer;

		/**
		 * @param {CanvasRenderer} renderer -The renderer sprite this batch works for.
		 */
		constructor(renderer: CanvasRenderer);

		/**
		 * Renders the sprite object.
		 * 渲染带texture的显示对象，Sprite,Graphic
		 * @param {Sprite} sprite - the sprite to render when using this spritebatch
		 */
		render(sprite: any): void;

		/**
		 * destroy the sprite object.
		 *
		 */
		destroy(): void;
	}

	export class CanvasRenderer extends SystemRenderer {
		/**
		 * 主屏幕渲染上下文
		 */
		rootContext: CanvasRenderingContext2D;
		/**
		 * 当前使用的上下文
		 */
		context: CanvasRenderingContext2D;
		/**
		 * 遮罩管理类
		 */
		maskManager: CanvasMaskManager;
		/**
		 * 考虑是否需要支持
		 */
		smoothProperty: string;
		/**
		 * 插件，暂时只有图形和图片的
		 */
		plugins: {
			sprite: CanvasSpriteRenderer;
			graphics: CanvasGraphicsRenderer;
		};
		renderingToScreen: boolean;

		constructor(options: RendererOptions);

		/**
		 * 渲染方法
		 * @param {DisplayObject} displayObject - 渲染对象
		 * @param {RenderTexture} [renderTexture] -离屏渲染纹理
		 * @param {Matrix} [transform] - 矩阵偏移
		 */
		render(displayObject: DisplayObject, renderTexture?: any, transform?: Matrix): void;

		/**
		 * 清空画布
		 */
		clear(clearColor: string): void;

		/**
		 * 销毁
		 */
		destroy(): void;

		/**
		 * 缩放尺寸
		 * @param {number} screenWidth
		 * @param {number} screenHeight
		 */
		resize(screenWidth: number, screenHeight: number): void;

		destroyPlugins(): void;
	}

	export function DrawAllToCanvas(images: any, callback: any): void;

	export function uid(): number;

	export const backupCanvas: HTMLCanvasElement;

	export function getGradientColor(points: any, colors: any): any;

	export function getCanvasBitmapStyle(image: any): any;

	export function hex2rgb(hex: number, out?: number[] | Float32Array): number[] | Float32Array;

	export function hex2string(hex: any): string;

	export function string2hex(string: string): number;

	export function rgb2hex(rgb: number[]): number;

	export function getRGBA(color: string, alpha: number): string;

	export function inputFeildIosEnable(): void;

	export function decomposeDataUri(dataUri: any): {
		mediaType: any;
		subType: any;
		charset: any;
		encoding: any;
		data: any;
	}

	export function getUrlFileExtension(url: any): any;

	export function sign(n: number): number;

	export function premultiplyTint(tint: number, alpha: number): number;

	export const TextureCache: any;

	export const BaseTextureCache: any;

	export const TextureSheetCache: any;

	export function destroyTextureCache(): void;

	export function clearTextureCache(): void;

	export const GlobalPro: {
		/**
		 * 舞台渲染类型，
		 */
		stageRenderType: RENDERER_TYPE;
		/**
		 * 实际渲染分辨率
		 */
		dpi: number;
		/**
		 * 图集间隙
		 */
		padding: number;
		/**
		 * 时钟起始时间
		 */
		startTime: number;
	}

	export function isWebGLSupported(): boolean;

	export function removeItems(arr: Array<any>, startIdx: number, removeCount: number): void;

	export function mapWebGLBlendModesToPixi(gl: any, array?: any[]): any[];

	export class BaseTexture extends EventDispatcher {
		/**
		 * 贴图回收时用到，是否被使用了
		 */
		touched: number;
		/**
		 * 宽度
		 */
		width: number;
		/**
		 * 高度
		 */
		height: number;
		/**
		 * SCALE_MODES，一般是线性
		 * 用于glTexture
		 */
		scaleMode: number;
		/**
		 * 加载完成会设置为true
		 * 加载失败或没有贴图数据则为false
		 * @readonly
		 * @member {boolean}
		 */
		hasLoaded: boolean;
		/**
		 * 正在加载
		 */
		isLoading: boolean;
		/**
		 * 实际元素，考虑是否set方法调用loadSource();
		 * 图片标签，canvas画布
		 * @readonly
		 * @member {HTMLImageElement|HTMLCanvasElement}
		 */
		source: HTMLImageElement | HTMLCanvasElement;
		/**
		 * image类型 eg. `png`
		 * 暂时不考虑svg
		 * @readonly
		 */
		imageType: string;
		/**
		 * rgb预乘alpha，webgl用到，png图片设置必为true，否则色值会出问题
		 * @default true
		 */
		premultipliedAlpha: boolean;
		/**
		 * 图片路径
		 * @member {string}
		 */
		imageUrl: string;
		/**
		 * 是否尺寸为2的次方，尽可能图集尺寸都为2的次方，gpu处理方便，并且能做mipmap缓存，性能更好
		 */
		isPowerOfTwo: boolean;
		/**
		 * 尺寸是2的次方下才能设置true，用于生成mipmap缓存
		 * @default true
		 */
		mipmap: boolean;
		/**
		 * 非2的次方时要设置CLAMP
		 * WebGL Texture wrap mode
		 * @default WRAP_MODES.CLAMP
		 */
		wrapMode: number;
		/**
		 * A map of renderer IDs to webgl textures
		 * 不同渲染器对应的记录，暂时应该只需要一个
		 * @member {object<number, WebGLTexture>}
		 */
		_glTextures: {};
		/**
		 * 批处理时用到的标志位
		 */
		_enabled: number;
		/**
		 * 是否已被销毁，为true，该纹理不可用
		 * @member {boolean}
		 * @private
		 * @readonly
		 */
		_destroyed: boolean;
		/**
		 * The ids under which this BaseTexture has been added to the base texture cache. This is
		 * automatically set as long as BaseTexture.addToCache is used, but may not be set if a
		 * BaseTexture is added directly to the BaseTextureCache array.
		 * @member {string[]}
		 */
		textureCacheIds: string[];

		/**
		 * @param {HTMLImageElement|HTMLCanvasElement} [source] - the source object of the texture.
		 * @param {number} [scaleMode=settings.SCALE_MODE] - possible values
		 */
		constructor(source?: HTMLImageElement | HTMLCanvasElement, scaleMode?: number);

		/**
		 * Updates the texture on all the webgl renderers, this also assumes the src has changed.
		 *
		 * @fires BaseTexture#update
		 */
		update(): void;

		/**
		 * @protected
		 * @param {HTMLImageElement|HTMLCanvasElement} source - the source object of the texture.
		 */
		loadSource(source: any): void;

		/**
		 * 更新图片类型
		 */
		private _updateImageType;
		/**
		 * 加载完成后执行
		 * @private
		 */
		private _sourceLoaded;

		/**
		 * 销毁 base texture
		 */
		destroy(): void;

		/**
		 * 用于释放gpu，需要时再上传到GPU
		 * Frees the texture from WebGL memory without destroying this texture object.
		 * This means you can still use the texture later which will upload it to GPU
		 * memory again.
		 * @fires BaseTexture#dispose
		 */
		dispose(): void;

		/**
		 * 修改source路径
		 * 原先source必须也是imageElement
		 * 貌似基本不用，新建比较好
		 * @param {string} newSrc - the path of the image
		 */
		updateSourceImage(newSrc: string): void;

		/**
		 * 根据图片路径创建BaseTexture
		 * @static
		 * @param {string} imageUrl 图片路径
		 * @param {boolean} [crossorigin=(auto)] -是否跨域，也可传string
		 * @param {number} [scaleMode]
		 * @return {BaseTexture} The new base texture.
		 */
		static fromImage(imageUrl: string, crossorigin?: any, scaleMode?: number): BaseTexture;

		/**
		 * 根据canvas创建BaseTexture
		 * baseTexture会缓存
		 * @static
		 * @param {HTMLCanvasElement} canvas - canvas标签
		 * @param {number} scaleMode - See {@link SCALE_MODES} for possible values
		 * @param {string} [origin='canvas'] - 类型标志位，用于生成缓存id
		 * @return {BaseTexture} The new base texture.
		 */
		static fromCanvas(canvas: HTMLCanvasElement, scaleMode?: number, origin?: string): BaseTexture;

		/**
		 * @static
		 * @param {string|HTMLImageElement|HTMLCanvasElement|BaseTexture} source - The source to create base texture from.
		 * @param {number} [scaleMode=settings.SCALE_MODE] - See {@link SCALE_MODES} for possible values
		 * @return {BaseTexture} The new base texture.
		 */
		static from(source: any, scaleMode?: number): BaseTexture;

		/**
		 * 加入缓存
		 * @static
		 * @param {BaseTexture} baseTexture
		 * @param {string} id
		 */
		static addToCache(baseTexture: BaseTexture, id: string): void;

		/**
		 * 移除缓存
		 * @static
		 * @param {string|BaseTexture} baseTexture id或者BaseTexture
		 * @return {BaseTexture|null} 移除的BaseTexture或null
		 */
		static removeFromCache(baseTexture: string | BaseTexture): BaseTexture | null;
	}

	export class TextureUvs {
		x0: number;
		y0: number;
		x1: number;
		y1: number;
		x2: number;
		y2: number;
		x3: number;
		y3: number;
		uvsUint32: Uint32Array;
		uvsFloat32: Float32Array;

		/**
		 * 用于记录图片的uv
		 * 00.....10
		 * .      .
		 * .      .
		 * 01.....11
		 */
		constructor();

		/**
		 * Sets the texture Uvs based on the given frame information.
		 *
		 * @private
		 * @param {Rectangle} frame - The frame of the texture
		 * @param {Rectangle} baseFrame - The base frame of the texture
		 * @param {number} rotate - Rotation of frame, see {@link GroupD8}
		 */
		set(frame: Rectangle, baseFrame: Rectangle | BaseTexture, rotate: number): void;
	}

	export class Texture extends EventDispatcher {
		/**
		 * texture没有frame
		 */
		noFrame: boolean;
		/**
		 * BaseTexture，必有
		 */
		baseTexture: BaseTexture;
		/**
		 * This is the area of the BaseTexture image to actually copy to the Canvas / WebGL when rendering,
		 * irrespective of the actual frame size or placement (which can be influenced by trimmed texture atlases)
		 */
		_frame: Rectangle;
		/**
		 * 如果矩形边缘有透明像素被裁减后的缩小的区域
		 * This is the trimmed area of original texture, before it was put in atlas
		 * Please call `_updateUvs()` after you change coordinates of `trim` manually.
		 */
		trim: Rectangle;
		/**
		 * 贴图是否可用，true为可用
		 */
		valid: boolean;
		/**
		 * 对应贴图uv
		 * The WebGL UV data cache.
		 */
		_uvs: TextureUvs;
		/**
		 * 原始尺寸，放入图集前
		 * This is the area of original texture, before it was put in atlas
		 */
		orig: Rectangle;
		/**
		 * 贴图旋转及镜像
		 */
		_rotate: number;
		/**
		 * 贴图的锚点，默认0，0，左上角，范围0到1
		 *
		 * @default {0,0}
		 */
		defaultAnchor: Point;
		/**
		 * 更新的id标志
		 */
		_updateID: number;
		/**
		 * 一般不用，需要时再说
		 * Contains data for uvs. May contain clamp settings and some matrices.
		 * Its a bit heavy, so by default that object is not created.
		 * @member {TextureMatrix}
		 * @default null
		 */
		transform: any;
		/**
		 * The ids under which this Texture has been added to the texture cache. This is
		 * automatically set as long as Texture.addToCache is used, but may not be set if a
		 * Texture is added directly to the TextureCache array.
		 *
		 * @member {string[]}
		 */
		textureCacheIds: string[];
		/**
		 * 空纹理
		 */
		static EMPTY: Texture;
		/**
		 * 白贴图
		 */
		static WHITE: Texture;

		/**
		 * @param {BaseTexture} baseTexture - The base texture source to create the texture from
		 * @param {Rectangle} [frame] - The rectangle frame of the texture to show
		 * @param {Rectangle} [orig] - The area of original texture
		 * @param {Rectangle} [trim] - Trimmed rectangle of original texture
		 * @param {number} [rotate] - indicates how the texture was rotated by texture packer. See {@link GroupD8}
		 * @param {Point} [anchor] - Default anchor point used for sprite placement / rotation
		 * 暂时不需要rotate
		 */
		constructor(baseTexture: BaseTexture | Texture, frame?: Rectangle, orig?: Rectangle, trim?: Rectangle, rotate?: number, anchor?: Point);

		/**
		 * Updates this texture on the gpu.
		 *
		 */
		update(): void;

		/**
		 * base texture 加载完成时执行
		 * @private
		 * @param {BaseTexture} baseTexture - The base texture.
		 */
		onBaseTextureLoaded(e: Event): void;

		/**
		 * base texture 更新时触发
		 * @private
		 * @param {BaseTexture} baseTexture - The base texture.
		 */
		onBaseTextureUpdated(e: Event): void;

		/**
		 * 销毁
		 */
		destroy(): void;

		/**
		 * 克隆
		 * @return {Texture} The new texture
		 */
		clone(): Texture;

		/**
		 *
		 * Updates the internal WebGL UV cache. Use it after you change `frame` or `trim` of the texture.
		 */
		_updateUvs(): void;

		/**
		 * The frame specifies the region of the base texture that this texture uses.
		 * Please call `_updateUvs()` after you change coordinates of `frame` manually.
		 *
		 * @member {Rectangle}
		 */
		frame: Rectangle;
		/**
		 * Indicates whether the texture is rotated inside the atlas
		 * set to 2 to compensate for texture packer rotation
		 * set to 6 to compensate for spine packer rotation
		 * can be used to rotate or mirror sprites
		 * See {@link GroupD8} for explanation
		 *
		 * @member {number}
		 */
		rotate: number;
		/**
		 * 宽高都是贴图真实的宽高，不管trim
		 * The width of the Texture in pixels.
		 *
		 * @member {number}
		 */
		readonly width: number;
		/**
		 * The height of the Texture in pixels.
		 *
		 * @member {number}
		 */
		readonly height: number;

		/**
		 * 不缓存base64，如需要，手动Texture.addToCache(texture,name),
		 * 自己输入名字，不用base字符串作为key
		 * @static
		 * @param {string} imageUrl - 图片路径
		 * @param {boolean} [crossorigin] - 是否跨域
		 * @param {number} [scaleMode=settings.SCALE_MODE] - See {@link SCALE_MODES} for possible values
		 * @return {Texture} The newly created texture
		 */
		static fromImage(imageUrl: string, crossorigin?: boolean, scaleMode?: number): Texture;

		/**
		 * TexturePacker图集中的名字
		 * @static
		 * @param {string} frameId - The frame Id of the texture in the cache
		 * @return {Texture} The newly created texture
		 */
		static fromFrame(frameId: string): Texture;

		/**
		 * Helper function that creates a new Texture based on the given canvas element.
		 * 不会自动缓存，有需要自己addToCache
		 * @static
		 * @param {HTMLCanvasElement} canvas - The canvas element source of the texture
		 * @param {number} [scaleMode=settings.SCALE_MODE] - See {@link SCALE_MODES} for possible values
		 * @param {string} [origin='canvas'] - A string origin of who created the base texture
		 * @return {Texture} The newly created texture
		 */
		static fromCanvas(canvas: HTMLCanvasElement, scaleMode?: number, origin?: string): Texture;

		/**
		 * The source can be - frame id, image url,image element,canvas element, base texture
		 *
		 * @static
		 * @param {number|string|HTMLImageElement|HTMLCanvasElement|BaseTexture}
		 *        source - Source to create texture from
		 * @return {Texture} The newly created texture
		 */
		static from(source: any): Texture;

		/**
		 * 加入全局缓存，TextureCache[name]调用
		 * @static
		 * @param {Texture} texture - The Texture to add to the cache.
		 * @param {string} id - The id that the Texture will be stored against.
		 */
		static addToCache(texture: Texture, id: string): void;

		/**
		 * 从全局缓存中移除
		 * @static
		 * @param {string|Texture} texture - id of a Texture to be removed, or a Texture instance itself
		 * @return {Texture|null} The Texture that was removed
		 */
		static removeFromCache(texture: any): Texture;
	}

	export class FillStyle {
		/**
		 * 十六进制颜色
		 * 默认0xFFFFFF 白色
		 */
		color: number;
		/**
		 * 透明度0到1
		 * 默认1
		 */
		alpha: number;
		/**
		 * 纹理，
		 * 默认Texture.WHITE
		 */
		texture: Texture;
		/**
		 * 纹理应用的矩阵
		 * 默认null
		 */
		matrix: Matrix;
		/**
		 * 是否绘制
		 * 默认false
		 */
		visible: boolean;
		/**
		 * 用于设置alpha为0做点击触发用
		 */
		alphaBlock: boolean;

		constructor();

		clone(): FillStyle;

		/**
		 * 重置
		 */
		reset(): void;

		/**
		 * 销毁
		 */
		destroy(): void;
	}

	export class LineStyle extends FillStyle {
		/**
		 * webgl的画线模式
		 * 默认false
		 */
		native: boolean;
		/**
		 * 线宽度
		 * 默认0
		 */
		width: number;
		/**
		 * 线的对齐方式
		 * 默认LINE_ALIGNMENT.middle  0.5
		 */
		alignment: LINE_ALIGNMENT;

		/**
		 * Clones the object
		 *
		 * @return {LineStyle}
		 */
		clone(): LineStyle;

		/**
		 * 重置
		 */
		reset(): void;
	}

	export class GraphicsData extends HashObject {
		fillStyle: FillStyle;
		lineStyle: LineStyle;
		matrix: Matrix;
		holes: any[];
		/**
		 * The shape object to draw.
		 * @member {Circle|Ellipse|Polygon|Rectangle|RoundedRectangle}
		 */
		shape: any;
		/**
		 * The type of the shape, see the Const.Shapes file for all the existing types,
		 * @member {number}
		 */
		type: number;
		/**
		 * 点的一维数组[x,y,x1,y1,x2,y2]
		 * 存下shape的点数据，因为如果shape不是poly不会特意存下points
		 */
		points: number[];

		/**
		 *
		 */
		constructor(shape: Circle | Rectangle | Ellipse | Polygon | RoundedRectangle, fillStyle?: FillStyle, lineStyle?: LineStyle, matrix?: Matrix);

		/**
		 * Creates a new GraphicsData object with the same values as this one.
		 *
		 * @return {GraphicsData} Cloned GraphicsData object
		 */
		clone(): GraphicsData;

		/**
		 * Destroys the Graphics data.
		 */
		destroy(): void;
	}

	export function earcut(data: any, holeIndices: any, dim: any): any[];

	export const _default: {
		build(graphicsData: GraphicsData): void;
		triangulate(graphicsData: GraphicsData, graphicsGeometry: Graphics): void;
	}

	export const _default: {
		build(graphicsData: GraphicsData): void;
		triangulate(graphicsData: any, graphicsGeometry: Graphics): void;
	}

	export const _default: {
		build(graphicsData: GraphicsData): void;
		triangulate(graphicsData: GraphicsData, graphics: Graphics): void;
	}

	export const _default: {
		build(graphicsData: GraphicsData): void;
		triangulate(graphicsData: GraphicsData, graphicsGeometry: Graphics): void;
	}

	export function bezierCurveControlPoint(points: any): {
		A: any[];
		B: any[];
	}

	export const GRAPHICS_CURVES: {
		adaptive: boolean;
		maxLength: number;
		minSegments: number;
		maxSegments: number;
		segmentsCount: Function;
	}

	export class Graphics extends Container {
		/**
		 * 当前的填充样式
		 * @member {FillStyle}
		 */
		private _fillStyle;
		/**
		 * 当前的画线样式
		 * @member {LineStyle}
		 */
		private _lineStyle;
		/**
		 * @member {FillStyle}
		 * @readonly
		 */
		readonly fill: FillStyle;
		/**
		 * @member {LineStyle}
		 * @readonly
		 */
		readonly line: LineStyle;
		/**
		 * 图形默认色调
		 * 默认白色，不会有任何影响
		 * @member {number}
		 * @default 0xFFFFFF
		 */
		private _tint;
		tint: number;
		/**
		 * 判断是否更新色值
		 */
		private batchTint;
		/**
		 * 当前图形应用的矩阵，一般用不着
		 *
		 * @member {Matrix}
		 */
		private _matrix;
		matrix: Matrix;
		hitTestByPixel: boolean;
		/**
		 * 当前是否是画洞
		 * @member {boolean}
		 * @default false
		 * @protected
		 */
		private _holeMode;
		/**
		 * Current path
		 * 只用于画多边形时用
		 * @member {Polygon}
		 */
		private _currentPath;
		/**
		 * 图形数据，为了一个Graphics里能绘制多个
		 * @member {GraphicsData[]}
		 */
		graphicsData: GraphicsData[];
		/**
		 * 是否用作mask
		 * @member {boolean}
		 */
		isUsedToMask: boolean;
		/**
		 * The bounds' padding used for bounds calculation.
		 * @member {number}
		 */
		boundsPadding: number;
		/**
		 * Used to detect if the graphics object has changed. If this is set to true then the graphics
		 * object will be recalculated.
		 * 通过比对确定是否该刷新
		 * @member {boolean}
		 * @private
		 */
		private dirty;
		/**
		 * 与dirty对比是否重算bounds
		 * Used to detect if we we need to recalculate local bounds
		 * @type {Number}
		 */
		private boundsDirty;
		/**
		 * canvas必用缓存
		 * renderCanvas默认用缓存,也就canvas上使用，如果经常需要重绘，设置为false
		 * webgl上用贴图占用GPU空间太大，用其他方法实现,贴图是白图就一张，用几何方法实现
		 * @name cacheAsBitmap
		 * @member {boolean}
		 * @memberof Graphics#
		 * @default false
		 */
		cacheAsBitmap: boolean;
		/**
		 * 需与dirty一致
		 */
		private cacheDirty;
		private _canvasBuffer;
		_texture: Texture;
		offsetX: number;
		offsetY: number;
		verts: any;
		indices: any;
		private batches;
		private geoBatches;
		private batchDirty;
		private uvs;
		private vertexData;
		private shapeIndex;
		private _transformID;

		constructor();

		/**
		 * 克隆该Graphics的几何绘制，不包括它自身的transform
		 * @return {Graphics} A clone of the graphics object
		 */
		cloneSelf(): Graphics;

		/**
		 * line属于附属属性，暂不写beginStroke，只要没调用过beginFill就是纯stroke
		 * Specifies the line style used for subsequent calls to Graphics methods such as the lineTo()
		 * method or the drawCircle() method.
		 * @param {number} [lineWidth=0] - width of the line to draw, will update the objects stored style
		 * @param {number} [color=0] - color of the line to draw, will update the objects stored style
		 * @param {number} [alpha=1] - alpha of the line to draw, will update the objects stored style
		 * @param {number} [alignment=0.5] - alignment of the line to draw, (0 = inner, 0.5 = middle, 1 = outter)
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		lineStyle(lineWidth?: number, color?: number, alpha?: number, alignment?: number, native?: boolean): Graphics;

		lineTextureStyle(width?: number, texture?: Texture, color?: number, alpha?: number, matrix?: any, alignment?: number, native?: boolean): this;

		private startPoly;
		private finishPoly;

		moveTo(x: number, y: number): Graphics;

		lineTo(x: number, y: number): Graphics;

		/**
		 * 初始化曲线
		 * @param x
		 * @param y
		 */
		private _initCurve;

		/**
		 * Calculate the points for a quadratic bezier curve and then draws it.
		 * Based on: https://stackoverflow.com/questions/785097/how-do-i-implement-a-bezier-curve-in-c
		 *
		 * @param {number} cpX - Control point x
		 * @param {number} cpY - Control point y
		 * @param {number} toX - Destination point x
		 * @param {number} toY - Destination point y
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		quadraticCurveTo(cpX: number, cpY: number, toX: number, toY: number): Graphics;

		/**
		 * Calculate the points for a bezier curve and then draws it.
		 *
		 * @param {number} cpX - Control point x
		 * @param {number} cpY - Control point y
		 * @param {number} cpX2 - Second Control point x
		 * @param {number} cpY2 - Second Control point y
		 * @param {number} toX - Destination point x
		 * @param {number} toY - Destination point y
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		bezierCurveTo(cpX: number, cpY: number, cpX2: number, cpY2: number, toX: number, toY: number): Graphics;

		/**
		 * The arcTo() method creates an arc/curve between two tangents on the canvas.
		 *
		 * "borrowed" from https://code.google.com/p/fxcanvas/ - thanks google!
		 *
		 * @param {number} x1 - The x-coordinate of the beginning of the arc
		 * @param {number} y1 - The y-coordinate of the beginning of the arc
		 * @param {number} x2 - The x-coordinate of the end of the arc
		 * @param {number} y2 - The y-coordinate of the end of the arc
		 * @param {number} radius - The radius of the arc
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		arcTo(x1: number, y1: number, x2: number, y2: number, radius: number): Graphics;

		/**
		 * The arc method creates an arc/curve (used to create circles, or parts of circles).
		 *
		 * @param {number} cx - The x-coordinate of the center of the circle
		 * @param {number} cy - The y-coordinate of the center of the circle
		 * @param {number} radius - The radius of the circle
		 * @param {number} startAngle - The starting angle, in radians (0 is at the 3 o'clock position
		 *  of the arc's circle)
		 * @param {number} endAngle - The ending angle, in radians
		 * @param {boolean} [anticlockwise=false] - Specifies whether the drawing should be
		 *  counter-clockwise or clockwise. False is default, and indicates clockwise, while true
		 *  indicates counter-clockwise.
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		arc(cx: number, cy: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean): Graphics;

		/**
		 * Specifies a simple one-color fill that subsequent calls to other Graphics methods
		 * (such as lineTo() or drawCircle()) use when drawing.
		 *
		 * @param {number} [color=0] - the color of the fill 十六进制颜色0xffffff
		 * @param {number} [alpha=1] - the alpha of the fill
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		beginFill(color?: number | string, alpha?: number): Graphics;

		beginTextureFill(texture?: Texture, color?: number, alpha?: number, matrix?: Matrix): this;

		/**
		 * Applies a fill to the lines and shapes that were added since the last call to the beginFill() method.
		 *
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		endFill(): Graphics;

		/**
		 * Closes the current path.
		 * 只用在多边形的路径里
		 * @return {Graphics} Returns itself.
		 */
		closePath(): Graphics;

		/**
		 * 开始画洞
		 */
		beginHole(): this;

		/**
		 * 结束画洞
		 */
		endHole(): this;

		/**
		 * 画矩形
		 * @param {number} x - The X coord of the top-left of the rectangle
		 * @param {number} y - The Y coord of the top-left of the rectangle
		 * @param {number} width - The width of the rectangle
		 * @param {number} height - The height of the rectangle
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawRect(x: number, y: number, width: number, height: number): Graphics;

		/**
		 * 画圆角矩形
		 * @param {number} x - The X coord of the top-left of the rectangle
		 * @param {number} y - The Y coord of the top-left of the rectangle
		 * @param {number} width - The width of the rectangle
		 * @param {number} height - The height of the rectangle
		 * @param {number} radius - Radius of the rectangle corners
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawRoundedRect(x: number, y: number, width: number, height: number, radius: number): Graphics;

		/**
		 * 画圆形
		 * @param {number} x - The X coordinate of the center of the circle
		 * @param {number} y - The Y coordinate of the center of the circle
		 * @param {number} radius - The radius of the circle
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawCircle(x: number, y: number, radius: number): Graphics;

		/**
		 * 画椭圆
		 * @param {number} x - The X coordinate of the center of the ellipse
		 * @param {number} y - The Y coordinate of the center of the ellipse
		 * @param {number} width - The half width of the ellipse
		 * @param {number} height - The half height of the ellipse
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawEllipse(x: number, y: number, width: number, height: number): Graphics;

		/**
		 * 画多边形
		 * @param {number[]|Point[]|Polygon} path - The path data used to construct the polygon.
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawPolygon(path: number[] | Point[] | Polygon): Graphics;

		/**
		 * 画任意多角形。points为数量
		 * @param {number} x - Center X position of the star
		 * @param {number} y - Center Y position of the star
		 * @param {number} points - The number of points of the star, must be > 1
		 * @param {number} radius - The outer radius of the star
		 * @param {number} [innerRadius] - The inner radius between points, default half `radius`
		 * @param {number} [rotation=0] - The rotation of the star in radians, where 0 is vertical
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		drawStar(x: number, y: number, points: number, radius: number, innerRadius: number, rotation?: number): Graphics;

		/**
		 * Draws the given shape to this Graphics object. Can be any of Circle, Rectangle, Ellipse, Line or Polygon.
		 *
		 * @param {Circle|Ellipse|Polygon|Rectangle|RoundedRectangle} shape - The shape object to draw.
		 * @return {Graphics}
		 */
		private drawShape;

		/**
		 * Clears the graphics that were drawn to this Graphics object, and resets fill and line style settings.
		 *
		 * @return {Graphics} This Graphics object. Good for chaining method calls
		 */
		clear(): Graphics;

		/**
		 * 暂时不用
		 * @returns {boolean} True if only 1 rect.
		 */
		isFastRect(): boolean;

		_renderWebGL(renderer: WebglRenderer): void;

		/**
		 * 根据GraphicsData分出batch，主要根据透明度和颜色
		 * _texture  里面有.BaseTexture
		 * vertexData
		 * indices
		 * graAlpha 图形透明度自身计算需要
		 * worldAlpha
		 * _batchRGB  图形颜色自身计算需要
		 * _tintRGB
		 * uvs
		 */
		private updateBatch;
		/**
		 * 对每个batch计算颜色
		 * @protected
		 */
		private calculateTints;
		/**
		 * If there's a transform update or a change to the shape of the
		 * geometry, recaculate the vertices.
		 * @protected
		 */
		private calculateVertices;
		private transformPoints;

		/**
		 * canvas渲染
		 * @private
		 * @param {CanvasRenderer} renderer - The renderer
		 */
		_renderCanvas(renderer: CanvasRenderer): void;

		/**
		 * _boundsId不知道怎么改，暂时不管，少用
		 * 计算全局bounds，_localBoundsSelf做个全局转换就行
		 * @private
		 */
		_calculateBounds(): void;

		/**
		 * 碰撞检测方法重写
		 * @param {Point} point - the point to test
		 */
		hitTestPoint(point: Point, isMouseEvent?: boolean): DisplayObject;

		/**
		 * 精确检测
		 * 用几何方法
		 * 不像shape里用的像素值
		 * @param point
		 */
		private hitTestPointAccuratly;

		/**
		 * 更新自身包围盒
		 */
		updateLocalBoundsSelf(): void;

		/**
		 * Generates a canvas texture.
		 * 不包括变形的，只根据图形数据产生的原生贴图，经过变形的，考虑，是否新建方法，这个暂时只为canvas缓存
		 * 也不考虑遮罩
		 * @param {number} scaleMode - The scale mode of the texture.
		 * @param {number} resolution - The resolution of the texture.
		 * @return {Texture} The new texture.
		 */
		private generateCanvasTexture;

		/**
		 *
		 */
		destroy(): void;

		/**
		 * 废弃先
		 * @member {number}
		 */
		/**
		 * The height of the sprite, setting this will actually modify the scale to achieve the value set
		 *
		 * @member {number}
		 */
		/**
		 * Process the holes data.
		 *
		 * @param {GraphicsData[]} holes - Holes to render
		 * @protected
		 */
		private proccessHoles;
		/**
		 * Generates the UVs for a shape.
		 * 不支持纹理填充，所以uv都时0
		 * @protected
		 * @param {number[]} verts - Vertices
		 * @param {number[]} uvs - UVs
		 * @param {Texture} texture - Reference to Texture
		 * @param {number} start - Index buffer start index.
		 * @param {number} size - The size/length for index buffer.
		 * @param {Matrix} [matrix] - Optional transform for all points.
		 */
		private addUvs;
		/**
		 * Modify uvs array according to position of texture region
		 * Does not work with rotated or trimmed textures
		 * @param {number[]} uvs array
		 * @param {Texture} texture region
		 * @param {number} start starting index for uvs
		 * @param {number} size how many points to adjust
		 */
		private adjustUvs;
	}

	export class FloatDisplay extends DisplayObject {
		/**
		 * 需要封装起来的html元素的引用。你可以通过这个引用来调用或设置此元素自身的属性方法和事件,甚至是样式
		 * @property htmlElement
		 * @public
		 * @since 1.0.0
		 * @type{HtmlElement}
		 */
		private _htmlElement;
		/**
		 * 是否已经添加了舞台事件
		 * @property _isAdded
		 * @since 1.0.0
		 * @type {boolean}
		 * @private
		 */
		private _isAdded;
		/**
		 * 记录是否需要修改位置矩阵
		 */
		private _transformID;

		/**
		 * 构造函数
		 * @method FloatDisplay
		 * @since 1.0.0
		 * @public
		 * @example
		 *      var floatDisplay = new FloatDisplay();
		 *      floatDisplay.init(document.getElementById('aaa'));
		 *      s.addChild(floatDisplay);
		 *
		 * <p><a href="" target="_blank">测试链接</a></p>
		 *
		 * @example
		 *  //创建悬浮的html元素
		 *  var section = document.createElement('section');
		 *   section.id = "rule";
		 *   section.style.overflowX = "hidden";
		 *   section.style.overflowY = "auto";
		 *   section.style.width = w + "px";
		 *   section.style.height = h + "px";
		 *   section.style.lineHeight = lh + "px";
		 *   section.style.fontFamily = '微软雅黑';
		 *   section.style.fontSize = fs + 'px';
		 *   section.style.color = "#ffffff";
		 *   //创建Floatview 把我们要悬浮的元素封装进去
		 *   var rule = new FloatDisplay();
		 *   stage.addChild(rule);
		 *   rule.x = ox;
		 *   rule.y = oy;
		 *   rule.init(this.section);
		 *   section.innerHTML = DataManager.ins.getData("ajaxElement").data.rule;
		 *
		 */
		constructor();

		htmlElement: any;

		/**
		 * 初始化方法,htmlElement 一定要设置width和height样式,并且一定要用px单位
		 * @method init
		 * @public
		 * @since 1.0.0
		 * @param {HtmlElement} htmlElement 需要封装起来的html元素的引用。你可以通过这个引用来调用或设置此元素自身的属性方法和事件,甚至是样式
		 */
		protected init(htmlElement: any): void;

		/**
		 * @method getStyle
		 * @param {HTMLElement} elem
		 * @param cssName
		 * @return {any}
		 */
		private getStyle;

		/**
		 * @method updateStyle
		 * @public
		 * @since 1.1.4
		 */
		updateStyle(): void;

		destroy(): void;
	}

	export class MouseEvent extends Event {
		/**
		 * 鼠标或者手指按下事件
		 * @property MOUSE_DOWN
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static MOUSE_DOWN: string;
		/**
		 * 鼠标或者手指抬起事件
		 * @property MOUSE_UP
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static MOUSE_UP: string;
		/**
		 * 鼠标或者手指单击
		 * @property CLICK
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static CLICK: string;
		/**
		 * 鼠标或者手指移动事件
		 * @property MOUSE_MOVE
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static MOUSE_MOVE: string;
		/**
		 * 鼠标或者手指移入到显示对象上里触发的事件
		 * @property MOUSE_OVER
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static MOUSE_OVER: string;
		/**
		 * 鼠标或者手指移出显示对象边界触发的事件
		 * @property MOUSE_OUT
		 * @static
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 */
		static MOUSE_OUT: string;
		/**
		 * mouse或touch事件时rootDiv坐标x点
		 * @property clientX
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		clientX: number;
		/**
		 * mouse或touch事件时rootDiv坐标y点
		 * @property clientY
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		clientY: number;
		/**
		 * mouse或touch事件时全局坐标x点
		 * @property stageX
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		stageX: number;
		/**
		 * mouse或touch事件时全局坐标y点
		 * @property stageY
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		stageY: number;
		/**
		 * mouse或touch事件时本地坐标x点
		 * @property localX
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		localX: number;
		/**
		 * mouse或touch事件时本地坐标y点
		 * @property localY
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 */
		localY: number;
		/**
		 * 触发事件的终点对象
		 * @property currentTarget
		 * @public
		 * @since 1.0.0
		 * @type{DisplayObject}
		 * @default null
		 */
		currentTarget: DisplayObject;
		/**
		 * 触摸或者鼠标事件的手指唯一标识
		 * @property identifier
		 * @type {number}
		 * @since 1.1.2
		 * @public
		 */
		identifier: any;

		/**
		 * @method MouseEvent
		 * @public
		 * @since 1.0.0
		 * @param {string} type
		 */
		constructor(type: string);

		/**
		 * 事件后立即更新显示列表状态
		 * @method updateAfterEvent
		 * @since 1.0.9
		 * @public
		 */
		updateAfterEvent(): void;

		destroy(): void;
	}

	export class GDispatcher {
		/**
		 * 事件回调池
		 */
		private static callbackPool;
		/**
		 * 事件作用域池
		 */
		private static thisObjPool;

		/**
		 *
		 * @param name 事件名
		 * @param callback 回调
		 * @param thisObj 作用域
		 */
		static addEvent(name: string, callback: any, thisObj: any): void;

		/**
		 *
		 * @param name 事件名
		 * @param callback 回调
		 * @param thisObj 作用域
		 */
		static removeEvent(name: string, callback: any, thisObj?: any): void;

		/**
		 * 派发事件
		 * @param name 事件名
		 * @param args 任意参数
		 */
		static dispatchEvent(name: string, ...args: any[]): void;
	}

	export const requestAnimationFrame: any;

	export class Stage extends Container {
		/**
		 * 是否阻止ios端双击后页面会往上弹的效果，因为如果阻止了，可能有些html元素出现全选框后无法取消
		 * 所以需要自己灵活设置,默认阻止.
		 * @property iosTouchendPreventDefault
		 * @type {boolean}
		 * @default true
		 * @since 1.0.4
		 * @public
		 */
		iosTouchendPreventDefault: boolean;
		/**
		 * 是否禁止引擎所在的canvas的鼠标事件或触摸事件的默认形为，默认为true是禁止的。
		 * @property isPreventDefaultEvent
		 * @since 1.0.9
		 * @default true
		 * @type {boolean}
		 */
		isPreventDefaultEvent: boolean;
		/**
		 * 整个引擎的最上层的div元素,
		 * 承载canvas的那个div html元素
		 * @property rootDiv
		 * @public
		 * @since 1.0.0
		 * @type {Html Div}
		 * @default null
		 */
		rootDiv: any;
		/**
		 * 当前stage所使用的渲染器
		 * 渲染器有两种,一种是canvas 一种是webGl
		 * @property renderObj
		 * @public
		 * @since 1.0.0
		 * @type {IRender}
		 * @default null
		 */
		renderObj: SystemRenderer;
		/**
		 * 渲染模式值 只读 CANVAS:2, webGl: 1
		 * @property renderType
		 * @readonly
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 0
		 * @readonly
		 */
		renderType: RENDERER_TYPE;

		/**
		 * 直接获取stage的引用，避免总是从Event.ADD_TO_STAGE 事件中去获取stage引用
		 * @property getStage
		 * @param {string} stageName
		 * @return {any}
		 * @since 2.0.0
		 */
		static getStage(stageName?: string): Stage;

		/**
		 * @property _stageList
		 * @static
		 * @type {Object}
		 * @private
		 */
		private static _stageList;
		/**
		 * 是否暂停
		 * @property pause
		 * @static
		 * @type {boolean}
		 * @public
		 * @since 1.0.0
		 * @default false
		 */
		static pause: boolean;
		/**
		 * @property _pause
		 * @type {boolean}
		 * @private
		 * @static
		 */
		private static _pause;
		/**
		 * 舞台在设备里截取后的可见区域,有些时候知道可见区域是非常重要的,因为这样你就可以根据舞台的可见区域做自适应了。
		 * @property viewRect
		 * @public
		 * @readonly
		 * @since 1.0.0
		 * @type {Rectangle}
		 * @default {x:0,y:0,width:0,height:0}
		 * @readonly
		 */
		viewRect: Rectangle;
		/**
		 * 当设备尺寸更新，或者旋转后是否自动更新舞台方向
		 * 端默认不开启
		 * @property autoSteering
		 * @public
		 * @since 1.0.0
		 * @type {boolean}
		 * @default false
		 */
		autoSteering: boolean;
		/**
		 * 当设备尺寸更新，或者旋转后是否自动更新舞台尺寸
		 * @property autoResize
		 * @public
		 * @since 1.0.0
		 * @type {boolean}
		 * @default false
		 */
		autoResize: boolean;
		/**
		 * 舞台的尺寸宽,也就是我们常说的设计尺寸
		 * @property desWidth
		 * @public
		 * @since 1.0.0
		 * @default 320
		 * @type {number}
		 * @readonly
		 */
		desWidth: number;
		/**
		 * 舞台的尺寸高,也就是我们常说的设计尺寸
		 * @property desHeight
		 * @public
		 * @since 1.0.0
		 * @default 240
		 * @type {number}
		 * @readonly
		 */
		desHeight: number;
		/**
		 * 舞台在当前设备中的真实高
		 * @property divHeight
		 * @public
		 * @since 1.0.0
		 * @default 320
		 * @type {number}
		 * @readonly
		 */
		divHeight: number;
		/**
		 * 舞台在当前设备中的真实宽
		 * @property divWidth
		 * @public
		 * @since 1.0.0
		 * @default 240
		 * @readonly
		 * @type {number}
		 */
		divWidth: number;
		/**
		 * 舞台的背景色
		 * 默认就是透明背景
		 * 可能设置一个颜色值改变舞台背景
		 * @property bgColor
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default "";
		 */
		private _bgColor;
		/**
		 * 设置颜色，即改变渲染器颜色
		 */
		bgColor: number;
		/**
		 * 舞台的缩放模式
		 * 默认为空就是无缩放的真实大小
		 * "noBorder" 无边框模式
		 * ”showAll" 显示所有内容
		 * “fixedWidth" 固定宽
		 * ”fixedHeight" 固定高
		 * @property scaleMode
		 * @public
		 * @since 1.0.0
		 * @default "onScale"
		 * @type {string}
		 * @example
		 *      //动态更改stage的对齐方式示例
		 *      //以下代码放到一个舞台的显示对象的构造函数中
		 *      let s=this;
		 *      s.addEventListener(Event.ADD_TO_STAGE,function(e){
		 *          let i=0;
		 *          s.stage.addEventListener(MouseEvent.CLICK,function(e){
		 *              let aList=[StageScaleMode.EXACT_FIT,StageScaleMode.NO_BORDER,StageScaleMode.NO_SCALE,StageScaleMode.SHOW_ALL,StageScaleMode.FIXED_WIDTH,StageScaleMode.FIXED_HEIGHT]
		 *              let state=e.currentTarget;
		 *              state.scaleMode=aList[i];
		 *              state.resize();
		 *              if(i>5){i=0;}
		 *          }
		 *      }
		 *
		 */
		scaleMode: string;
		private _scaleMode;
		/**
		 * 原始为60的刷新速度时的计数器
		 * @property _flush
		 * @private
		 * @since 1.0.0
		 * @default 0
		 * @type {number}
		 */
		private _flush;
		/**
		 * 当前的刷新次数计数器
		 * @property _currentFlush
		 * @private
		 * @since 1.0.0
		 * @default 0
		 * @type {number}
		 */
		private _currentFlush;
		/**
		 * @property _dragDisplay
		 * @private
		 * @type {null}
		 * @private
		 * @static
		 */
		static _dragDisplay: DisplayObject;
		/**
		 * @property _isLoadedVConsole
		 * @type {Array}
		 * @private
		 * @static
		 */
		private static _isLoadedVConsole;
		/**
		 * 上一次鼠标或触碰经过的显示对象列表
		 * @property _lastDpList
		 * @type {Object}
		 * @private
		 */
		private _lastDpList;
		/**
		 * 窗口resize计时器id
		 * @property _rid
		 * @type {number}
		 * @private
		 */
		private _rid;
		/**
		 * dom层记录
		 * @property _floatDisplayList
		 * @type {any[]}
		 * @private
		 */
		private _floatDisplayList;

		/**
		 * 显示对象入口函数
		 * @method Stage
		 * @param {string} rootDivId
		 * @param {number} desW 舞台宽
		 * @param {number} desH 舞台高
		 * @param {number} frameRate 刷新率
		 * @param {string} scaleMode 缩放模式 StageScaleMode
		 * @param {number} renderType 渲染类型2canvas
		 * @param {boolean} transparent 透明否，默认透明true,此时bgColor无效
		 * @param {number} bgColor 背景颜色十六进制
		 * @public
		 * @since 1.0.0
		 */
		constructor(rootDivId?: string, desW?: number, desH?: number, frameRate?: number, scaleMode?: string, renderType?: RENDERER_TYPE, transparent?: boolean, bgColor?: number);

		/**
		 * 主渲染函数
		 * @method render
		 */
		render(): void;

		/**
		 * 这个是鼠标事件的MouseEvent对象池,因为如果用户有监听鼠标事件,如果不建立对象池,那每一秒将会new Fps个数的事件对象,影响性能
		 * @property _ml
		 * @type {Array}
		 * @private
		 */
		private _ml;
		/**
		 * 这个是事件中用到的Point对象池,以提高性能
		 * @property _mp
		 * @type {Array}
		 * @private
		 */
		private _mp;
		/**
		 * 刷新mouse或者touch事件
		 * @method _initMouseEvent
		 * @private
		 */
		private _initMouseEvent;
		/**
		 * @property _mouseDownPoint
		 * @type {Object}
		 * @private
		 */
		private _mouseDownPoint;

		/**
		 * 循环刷新页面的函数
		 * @method flush
		 * @private
		 * @return {void}
		 */
		flush(): void;

		/**
		 * 引擎的刷新率,就是一秒中执行多少次刷新
		 * @method setFrameRate
		 * @param {number} fps 最好是60的倍数如 1 2 3 6 10 12 15 20 30 60
		 * @since 1.0.0
		 * @public
		 */
		setFrameRate(fps: number): void;

		/**
		 * 引擎的刷新率,就是一秒中执行多少次刷新
		 * @method getFrameRate
		 * @since 1.0.0
		 * @public
		 */
		getFrameRate(): number;

		/**
		 * 获取引擎所在的div宽高
		 * @method getRootDivWH
		 * @public
		 * @since 1.0.0
		 * @param {HTMLDivElement} div
		 * @return {{w: number, h: number}}
		 */
		getRootDivWH(div: HTMLDivElement): {
			w: number;
			h: number;
		};

		/**
		 * 当一个stage不再需要使用,或者要从浏览器移除之前,请先停止它,避免内存泄漏
		 * @method kill
		 * @since 1.0.0
		 * @public
		 */
		kill(): void;

		/**
		 * html的鼠标或单点触摸对应的引擎事件类型名
		 * @property _mouseEventTypes
		 * @type {{mousedown: string, mouseup: string, mousemove: string, touchstart: string, touchmove: string, touchend: string}}
		 * @private
		 */
		private _mouseEventTypes;
		/**
		 * 无多指，无拖动
		 * @method onMouseEvent
		 * @param e
		 * @private
		 */
		private onMouseEvent;
		/**
		 * 设置舞台的对齐模式
		 * @method setAlign
		 * @private
		 * @return {void}
		 */
		private setAlign;

		/**
		 * 当舞台尺寸发生改变时,如果stage autoResize 为 true，则此方法会自己调用；
		 * 如果设置stage autoResize 为 false 你需要手动调用此方法以更新界面.
		 * 不管autoResize 的状态是什么，你只要侦听 了stage 的 Event.RESIZE 事件
		 * 都可以接收到舞台变化的通知。
		 * @method resize
		 * @public
		 * @since 1.0.0
		 */
		resize(): void;

		getBounds(): Rectangle;

		/**
		 * 要循环调用 flush 函数对象列表
		 * @method allUpdateObjList
		 * @static
		 * @since 1.0.0
		 * @type {Array}
		 */
		private static allUpdateObjList;

		/**
		 * 刷新所有定时器
		 * @static
		 * @private
		 * @since 1.0.0
		 * @method flushAll
		 */
		static flushAll(): void;

		/**
		 * 添加一个刷新对象，这个对象里一定要有一个 flush 函数。
		 * 因为一但添加，这个对象的 flush 函数会以stage的fps间隔调用
		 * 如，你的stage是30fps 那么你这个对象的 flush 函数1秒会调用30次。
		 * @method addUpdateObj
		 * @param target 要循化调用 flush 函数的对象
		 * @public
		 * @static
		 * @since
		 */
		static addUpdateObj(target: any): void;

		/**
		 * 移除掉已经添加的循环刷新对象
		 * @method removeUpdateObj
		 * @param target
		 * @public
		 * @static
		 * @since 1.0.0
		 */
		static removeUpdateObj(target: any): void;

		destroy(): void;
	}

	export class DisplayObject extends EventDispatcher {
		parent;
		/**
		 * 基础信息
		 */
		transform: Transform;
		/**
		 * 是否可见
		 */
		$visible: boolean;
		/**
		 * 是否可绘制
		 * 自身不绘制，但是参与updateTransform
		 * 在有些时候的遮罩处理
		 */
		renderable: boolean;
		/**
		 * 父级
		 * @member {Container}
		 * @readonly
		 */
		wrapper: any;
		/**
		 * 舞台
		 */
		stage: Stage;
		/**
		 * 名字,预留
		 */
		name: string;
		/**
		 * 全局透明度
		 */
		worldAlpha: number;
		/**
		 * 用于只管局部自己,不包括子级，用于各继承修改自身，hitTest检测自己用
		 */
		protected _localBoundsSelf: Rectangle;
		/**
		 * 用于计算世界bounds，包括自身和子级，一般只在获取宽高时使用，变形后的
		 */
		protected _bounds: Rectangle;
		protected _boundsID: number;
		protected _lastBoundsID: number;
		/**
		 * 遮罩
		 * @member {Graphics}
		 * @private
		 */
		private $mask;
		/**
		 * 透明度
		 * 0到1
		 */
		private $alpha;
		/**
		 * 是否已销毁
		 * @member {boolean}
		 * @readonly
		 */
		destroyed: boolean;
		/**
		 * 是否可响应鼠标事件
		 */
		mouseEnabled: boolean;
		/**
		 * 临时父级,特殊用处
		 */
		tempDisplayObjectParent: DisplayObject;
		/**
		 * 特殊用处
		 * 缓存的DisplayObject的updateTransform
		 */
		displayObjectUpdateTransform: any;
		/**
		 * 特殊用处
		 * 缓存的DisplayObject的hitTestPoint
		 */
		displayObjectHitTestPoint: any;
		protected _width: number;
		protected _height: number;

		constructor();

		visible: boolean;

		attachVisibility(): void;

		/**
		 * @private
		 */
		private readonly _tempDisplayObjectParent;

		/**
		 * 更新矩阵
		 */
		updateTransform(): void;

		static _bp: Point;
		static _p1: Point;
		static _p2: Point;
		static _p3: Point;
		static _p4: Point;
		static temBounds: Rectangle;

		/**
		 * 点击碰撞测试,就是舞台上的一个point是否在显示对象内,在则返回该对象，不在则返回null
		 * 对于那些不是继承container，而直接继承displayObject的不用重写，如Bitmap
		 * @method hitTestPoint
		 * @public
		 * @since 1.0.0
		 * @param {Point} point 需要碰到的坐标点
		 * @param {boolean} isMouseEvent 是否是鼠标事件调用此方法,一般都为true
		 * @return {DisplayObject}
		 */
		hitTestPoint(point: Point, isMouseEvent?: boolean): DisplayObject;

		/**
		 * 递归更新矩阵方法
		 */
		private _recursivePostUpdateTransform;

		/**
		 * 获取全局的bounds，变形后的
		 * @param {boolean} skipUpdate - setting to true will stop the transforms of the scene graph from
		 *  being updated. This means the calculation returned MAY be out of date BUT will give you a
		 *  nice performance boost
		 * @param {Rectangle} rect - Optional rectangle to store the result of the bounds calculation
		 * @return {Rectangle} the rectangular bounding area
		 */
		getBounds(skipUpdate?: boolean, rect?: Rectangle): Rectangle;

		/**
		 * 以父级为世界坐标系的本地包围盒
		 * @param {Rectangle} [rect] - Optional rectangle to store the result of the bounds calculation
		 * @return {Rectangle} the rectangular bounding area
		 */
		getLocalBounds(rect?: Rectangle): Rectangle;

		calculateBounds(): void;

		/**
		 *将全局坐标转换到本地坐标值
		 * @method globalToLocal
		 * @since 1.0.0
		 * @public
		 * @param {Point} point
		 * @param bp
		 * @return {Point}
		 */
		globalToLocal(point: Point, bp?: Point): Point;

		/**
		 *将本地坐标转换到全局坐标值
		 * @method localToGlobal
		 * @public
		 * @since 1.0.0
		 * @param {Point} point
		 * @param bp
		 * @return {Point}
		 */
		localToGlobal(point: Point, bp?: Point): Point;

		/**
		 * 调用些方法会冒泡的将事件向显示列表下方传递，主要用于移除舞台，和添加进舞台事件，修改stage
		 * @method _onDispatchBubbledEvent
		 * @public
		 * @since 1.0.0
		 * @param {string} type
		 * @return {void}
		 */
		_onDispatchBubbledEvent(type: string): void;

		/**
		 * webgl渲染
		 * @param {WebGLRenderer} renderer - The renderer
		 */
		renderWebGL(renderer: any): void;

		/**
		 * canvas渲染
		 * @param {CanvasRenderer} renderer - The renderer
		 */
		renderCanvas(renderer: any): void;

		/**
		 * 设置父级
		 * @param {Container} container - The Container to add this DisplayObject to
		 * @return {Container} The Container that this DisplayObject was added to
		 */
		setParent(container: any): any;

		/**
		 *根据常规属性 设置矩阵属性，弧度制
		 * @param {number} [x=0] - The X position
		 * @param {number} [y=0] - The Y position
		 * @param {number} [scaleX=1] - The X scale value
		 * @param {number} [scaleY=1] - The Y scale value
		 * @param {number} [rotation=0] - The rotation
		 * @param {number} [skewX=0] - The X skew value
		 * @param {number} [skewY=0] - The Y skew value
		 * @param {number} [anchorX=0] - The X anchor value
		 * @param {number} [anchorY=0] - The Y anchor value
		 * @return {DisplayObject} The DisplayObject instance
		 */
		setTransform(x?: number, y?: number, scaleX?: number, scaleY?: number, rotation?: number, skewX?: number, skewY?: number, anchorX?: number, anchorY?: number): DisplayObject;

		/**
		 * 基本销毁方法
		 */
		destroy(): void;

		alpha: number;
		x: number;
		y: number;
		/**
		 * 获取世界矩阵
		 * 手动修改时，this.transform._worldID++,保证子级的worldMatrix会修改，尽量别那么做
		 * @member {Matrix}
		 * @readonly
		 */
		readonly worldMatrix: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/Matrix").Matrix;
		/**
		 * 获取本地矩阵
		 * 手动修改时this.transform._parentID=-1;保证其worldMatrix会更新
		 * @member {Matrix}
		 * @readonly
		 */
		readonly localMatrix: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/Matrix").Matrix;
		/**
		 * 获取位置对象
		 * @member {Point|ObservablePoint}
		 */
		/**
		 * 设置位置对象
		 */
		position: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/ObservablePoint").ObservablePoint;
		/**
		 * 获取缩放对象
		 * @member {Point|ObservablePoint}
		 */
		/**
		 * 设置缩放对象
		 */
		scale: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/ObservablePoint").ObservablePoint;
		scaleX: number;
		scaleY: number;
		/**
		 * 获取锚点对象
		 * @member {Point|ObservablePoint}
		 */
		anchor: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/ObservablePoint").ObservablePoint;
		anchorX: number;
		anchorY: number;
		/**
		 * 获取斜切对象
		 * @member {ObservablePoint}
		 */
		skew: import("../../../../../../../../Users/rockyl/WorkSpaces/VisualEditor/zeroing-engine/src/2d/math/ObservablePoint").ObservablePoint;
		/**
		 * 获取旋转值,顺时针,角度制
		 * @member {number}
		 */
		/**
		 * 角度制
		 */
		rotation: number;
		/**
		 * 自身是否可见,检测所有父级的visible
		 * @member {boolean}
		 * @readonly
		 */
		readonly worldVisible: boolean;
		/**
		 * 获取遮罩
		 * @member {Graphics}
		 */
		mask: Graphics;
		/**
		 * 子类必重写，
		 */
		/**
		 * 子类必重写，如果设置过宽高_width有值且不为0，子类在更新texture时需设置scale
		 */
		width: number;
		height: number;

		/**
		 * 更新方法，帧循环的监听事件放在这
		 */
		update(deltaTime: number): void;
	}

	export function applyAutoAdjust(ctor: Function): void;

	export const ESCAPE_REG_EXP: RegExp;

	export const linkedFlag = "$_linked_$";

	export const nodeScheme = "node://";

	export const linkScheme = "link://";

	export function arrayFind(arr: any, predicate: any): any;

	export function objClone(obj: any): any;

	export function propertyParse(key: any, node: any, properties: any): void;

	export function getDataByPath(scope: any, path: any, throwException?: any): any;

	export function injectProp(target: any, data?: any, callback?: Function, ignoreMethod?: boolean, ignoreNull?: boolean): boolean;

	export function copyProp(target: any, data: any, config?: any): void;

	export function obj2query(obj: any): string;

	export function importCJSCode(code: any, node?: any): any;

	export function importUMDCode(code: any): any;

	export function trimChar(str: string, char: string): string;

	export function joinPath(...segments: string[]): string;

	export function findVariable(name: string, ...contexts: any[]): any;

	export function htmlToPureText(htmlText: any): any;

	export function strLen(str: any): number;

	export function strShort(str: any, limit: any, replace?: string): string;

	export function instantiateScript(node: any, ScriptConfig: any): void;

	export function injectProperties(target: any, source: any): any;

	export function md5(string: any): string;

	export class ObjectPool {
		static MAX: number;
		static createMethods: any;
		static initMethods: any;
		static pools: any;

		static registerPool(name: string, createMethod: Function, initMethod?: Function): void;

		static getObject(name: string, ...params: any[]): any;

		static recycleObject(name: string, obj: any): void;

		static debug(): void;
	}

	export function applyScript(ctor: Function): void;

	export class ScriptBase {
		private _host;
		private _disabled;
		readonly host: Container;
		disabled: boolean;

		mounted(): void;

		destroy(): void;

		update(t: number): void;

		awake(): void;

		sleep(): void;
	}

	export function registerScriptDef(id: any, def: any): void;

	export function registerScripts(scripts: any): void;

	export class DataCenter extends EventDispatcher {
		store: {};
		watchers: any[];

		constructor();

		/**
		 * 注册一个分组
		 * @param name
		 * @param origin
		 */
		registerGroup(name: any, origin?: any): any;

		/**
		 * 反注册分组
		 * @param name
		 */
		unregisterGroup(name: any): void;

		/**
		 * 获取分组
		 * @param name
		 */
		getGroup(name: any): any;

		/**
		 * 根据路径获取数据
		 * @param path
		 * @param groupName
		 * @param throwException
		 */
		getDataByPath(path: any, groupName?: any, throwException?: any): any;

		/**
		 * 根据绑定名获取数据
		 * @param name
		 * @param throwException
		 */
		getDataByName(name: any, throwException?: any): any;

		/**
		 * 填充数据
		 * @param str
		 * @param escapes
		 */
		formatString(str: any, escapes: any): any;

		/**
		 * 数值递增
		 * @param groupName
		 * @param step
		 * @param path
		 * @param dispatch
		 */
		increase(groupName: any, step?: any, path?: any, dispatch?: boolean): void;

		/**
		 * 修改数据
		 * @param groupName
		 * @param data
		 * @param path
		 * @param dispatch
		 */
		mutate(groupName: any, data?: any, path?: any, dispatch?: boolean): void;

		/**
		 * 添加数据侦听
		 * @param name
		 * @param path
		 */
		watch(name: any, path: any): void;

		/**
		 * 根据名字获取侦听者
		 * @param name
		 */
		getWatcher(name: any): any;

		/**
		 * 注册数据绑定配置
		 * @param dataMapping
		 */
		registerDataMapping(dataMapping: any): void;
	}

	export const dataCenter: DataCenter;

	export let env: any;

	export function injectEnv(data: any): void;

	export enum Logs {
		PROCESS = "process"
	}

	export function getLogSwitch(id: any): boolean;

	export class Process {
		private readonly id;
		private _config;
		private _parent;
		private _vm;
		private _sequence;
		private _meta;
		private _target;
		private _originProps;
		private _scope;

		constructor(id: any);

		readonly processConfig: any;
		readonly parent: Process;
		readonly sequence: any;
		readonly scope: any;

		init(context: any): void;

		execute(args: any): Promise<any>;

		_executeMetaScript(type: any, payload: any, meta: any): Promise<{
			type: any;
			payload: any;
		}>;

		_executeSubProcess(type: any, payload: any): Promise<{
			type: any;
			payload: any;
		}>;

		_executeNextProcess(type: any, payload: any): Promise<{
			type: any;
			payload: any;
		}>;

		/**
		 * 获取过程配置
		 * @param id
		 * @return {null}
		 */
		getProcessMeta(id: any): any;

		/**
		 * 获取props
		 */
		getProps(key?: any): any;

		/**
		 * 更新props
		 */
		updateProps(props: any, args: any, originProps: any, propsConfig: any): void;

		resolveLinkedProp(data: any, key: any): any;
	}

	export class VM {
		_processMetaLibs: any;
		_globalContext: any;
		_target: any;
		_id: any;
		_scriptMap: any;
		_pid: number;

		setup(context: any): void;

		executeProcess(sequence: any, id: any, parentProcess: any, args: any): Promise<any>;

		getMeta(id: any): any;

		getScript(hash: any): any;

		readonly globalContext: any;
		readonly id: any;
	}

	export function setProcessMetaLibs(...metaContexts: any[]): void;

	export function setGlobalContext(context: any): void;

	export function setScriptMap(_scriptMap: any): void;

	export function executeBehavior(sequence: any, subEntry: string, target: any, args?: any): void;

	export const globalEvent: EventDispatcher;

	export const DATA_CENTER_EVENT: string;

	export function applyEvents(ctor: Function): void;

	export function afterConstructor(ctor: Function): void;

	export class Container extends DisplayObject {
		percentWidth: number;
		percentHeight: number;
		left: number;
		top: number;
		right: number;
		bottom: number;
		horizonCenter: number;
		verticalCenter: number;
		/**
		 * 为false鼠标事件不再向下传递
		 */
		mouseChildren: boolean;
		/**
		 * 孩子们
		 * @member {DisplayObject[]}
		 * @readonly
		 */
		children: any[];
		/**
		 * 特殊用处
		 * 缓存的Container的updateTransform
		 */
		containerUpdateTransform: any;

		constructor();

		afterConstructor(): void;

		/**
		 * children改变时
		 * @private
		 */
		onChildrenChange(index: any): void;

		attachVisibility(): void;

		/**
		 * 添加child
		 * @param {DisplayObject} child
		 * @return {DisplayObject}
		 */
		addChild(child: DisplayObject): DisplayObject;

		/**
		 * 在相应index处添加child
		 * @param {DisplayObject} child - The child to add
		 * @param {number} index - The index to place the child in
		 * @return {DisplayObject} The child that was added.
		 */
		addChildAt(child: DisplayObject, index: number): DisplayObject;

		/**
		 * 只用于交换索引
		 * @param {DisplayObject} child - First display object to swap
		 * @param {DisplayObject} child2 - Second display object to swap
		 */
		swapChildren(child1: DisplayObject, child2: DisplayObject): boolean;

		/**
		 * 获取child的层级索引index
		 * @param {DisplayObject} child - The DisplayObject instance to identify
		 * @return {number} The index position of the child display object to identify
		 */
		getChildIndex(child: DisplayObject): number;

		/**
		 * 是否含有child
		 * @param child
		 */
		contains(child: DisplayObject): boolean;

		/**
		 * 设置child的层级
		 * @param {DisplayObject} child
		 * @param {number} index
		 */
		setChildIndex(child: DisplayObject, index: number): void;

		/**
		 * Returns the child at the specified index
		 * @param {number} index - The index to get the child at
		 * @return {DisplayObject} The child at the given index, if any.
		 */
		getChildAt(index: number): DisplayObject;

		/**
		 * 根据路径获取子节点
		 * @param path
		 * @param method
		 */
		getChildByPath(path: any, method: string): DisplayObject;

		/**
		 * 根据名称路径获取子节点
		 * @param path
		 */
		getChildByNamePath(path: string): DisplayObject;

		/**
		 * 根据索引路径获取子节点
		 * @param path
		 */
		getChildByIndexPath(path: string): DisplayObject;

		/**
		 * 根据uuid搜索子节点
		 * @param uuid
		 */
		findChildByUUID(uuid: string): any;

		/**
		 * 通过名字获取子级
		 * @param name
		 * @param isOnlyOne
		 * @param isRecursive
		 */
		getChildByName(name: string | RegExp, isOnlyOne?: boolean, isRecursive?: boolean): any;

		/**
		 * 移除child
		 * @param {DisplayObject} child
		 * @return {DisplayObject}
		 */
		removeChild(child: DisplayObject): DisplayObject;

		/**
		 * 在index处移除child
		 * @param {number} index - The index to get the child from
		 * @return {DisplayObject} The child that was removed.
		 */
		removeChildAt(index: number): DisplayObject;

		/**
		 * 通过索引批量移除child
		 * @param {number} [beginIndex=0]
		 * @param {number} [endIndex=this.children.length]
		 * @returns {DisplayObject[]} List of removed children
		 */
		removeChildren(beginIndex?: number, endIndex?: number): DisplayObject[];

		/**
		 * 更新矩阵
		 */
		updateTransform(): void;

		/**
		 * 父类重写
		 * 都是全局的
		 */
		calculateBounds(): void;

		/**
		 * 加"_"的方法基本是为了自己特殊处理
		 */
		protected _calculateBounds(): void;

		/**
		 * 检测点是否在任何child上
		 * 重写父类方法
		 */
		hitTestPoint(globalPoint: Point, isMouseEvent?: boolean): any;

		/**
		 * webgl渲染
		 * @param {WebglRenderer} renderer - The renderer
		 */
		renderWebGL(renderer: WebglRenderer): void;

		/**
		 * 高级渲染方法
		 *
		 * @private
		 * @param {WebGLRenderer} renderer - The renderer
		 */
		private renderAdvancedWebGL;

		/**
		 * 自身渲染方式
		 * @private
		 * @param {WebglRenderer} renderer - The renderer
		 */
		protected _renderWebGL(renderer: WebglRenderer): void;

		/**
		 * canvas渲染方式
		 * @param {CanvasRenderer} renderer - The renderer
		 */
		renderCanvas(renderer: CanvasRenderer): void;

		/**
		 * 自身渲染方法
		 *
		 * @private
		 * @param {CanvasRenderer} renderer - The renderer
		 */
		protected _renderCanvas(renderer: CanvasRenderer): void;

		/**
		 * 更新方法
		 */
		update(deltaTime: number): void;

		/**
		 * 调用此方法对自己及其child触发一次指定事件
		 * @method _onDispatchBubbledEvent
		 * @public
		 * @param {string} type
		 * @since 1.0.0
		 */
		_onDispatchBubbledEvent(type: string): void;

		/**
		 *
		 */
		destroy(): void;

		/**
		 * 一般用于获取宽高并设置
		 * 包括子级的,容器的尽量少用，子类可重写
		 * @member {number}
		 */
		width: number;
		/**
		 * 高度同width
		 * @member {number}
		 */
		height: number;

		clone(withEvents?: boolean, withScripts?: boolean): any;

		/**
		 * @method _getElementsByName
		 * @param {RegExp} rex
		 * @param {Container} root
		 * @param {boolean} isOnlyOne
		 * @param {boolean} isRecursive
		 * @param {Array<DisplayObject>} resultList
		 * @private
		 * @static
		 */
		private static _getElementsByName;
	}

	export class Shape extends Sprite {
		canvas: HTMLCanvasElement;
		context: CanvasRenderingContext2D;
		dirty: boolean;
		offsetX: number;
		offsetY: number;

		constructor();

		/**
		 * 一个数组，每个元素也是一个数组[类型 0是属性,1是方法,名字 执行的属性或方法名,参数]
		 *
		 * @property _command
		 * @private
		 * @since 1.0.0
		 * @type {Array}
		 * @default []
		 */
		private _command;
		/**
		 * @property _isBitmapStroke
		 * @private
		 * @since 1.0.0
		 */
		private _isBitmapStroke;
		/**
		 * @property _isBitmapFill
		 * @private
		 * @since 1.0.0
		 */
		private _isBitmapFill;
		/**
		 * 是否对矢量使用像素碰撞 默认开启
		 * @property hitTestByPixel
		 * @type {boolean}
		 * @default true
		 * @since 1.1.0
		 */
		hitTestByPixel: boolean;

		/**
		 * 画一个带圆角的矩形
		 * @method drawRoundRect
		 * @param {number} x 点x值
		 * @param {number} y 点y值
		 * @param {number} w 宽
		 * @param {number} h 高
		 * @param {number} rTL 左上圆角半径
		 * @param {number} rTR 右上圆角半径
		 * @param {number} rBL 左下圆角半径
		 * @param {number} rBR 右上圆角半径
		 * @public
		 * @since 1.0.0
		 */
		drawRoundRect(x: number, y: number, w: number, h: number, rTL?: number, rTR?: number, rBL?: number, rBR?: number): void;

		/**
		 * 绘画时移动到某一点
		 * @method moveTo
		 * @param {number} x
		 * @param {number} y
		 * @public
		 * @since 1.0.0
		 */
		moveTo(x: number, y: number): void;

		/**
		 * 从上一点画到某一点,如果没有设置上一点，则上一点默认为(0,0)
		 * @method lineTo
		 * @param {number} x
		 * @param {number} y
		 * @public
		 * @since 1.0.0
		 */
		lineTo(x: number, y: number): void;

		/**
		 *
		 * https://www.jianshu.com/p/e8ea5996cd79
		 * @method arcTo
		 * @param {number} x1 圆弧外一点，由它连接两切点
		 * @param {number} y1
		 * @param {number} x2 结束点 没啥用，设啥都是那个切点
		 * @param {number} y2
		 * @param {number} radius 半径
		 * @public
		 * @since 1.0.0
		 */
		arcTo(x1: number, y1: number, x2: number, y2: number, radius: number): void;

		/**
		 * 二次贝赛尔曲线
		 * 从上一点画二次贝赛尔曲线到某一点,如果没有设置上一点，则上一占默认为(0,0)
		 * @method quadraticCurveTo
		 * @param {number} cpX 控制点X
		 * @param {number} cpX 控制点Y
		 * @param {number} x 终点X
		 * @param {number} y 终点Y
		 * @public
		 * @since 1.0.0
		 */
		quadraticCurveTo(cpX: number, cpY: number, x: number, y: number): void;

		/**
		 * 三次贝赛尔曲线
		 * 从上一点画二次贝赛尔曲线到某一点,如果没有设置上一点，则上一占默认为(0,0)
		 * @method bezierCurveTo
		 * @param {number} cp1X 1控制点X
		 * @param {number} cp1Y 1控制点Y
		 * @param {number} cp2X 2控制点X
		 * @param {number} cp2Y 2控制点Y
		 * @param {number} x 终点X
		 * @param {number} y 终点Y
		 * @public
		 * @since 1.0.0
		 */
		bezierCurveTo(cp1X: number, cp1Y: number, cp2X: number, cp2Y: number, x: number, y: number): void;

		/**
		 * 闭合一个绘画路径
		 * @method closePath
		 * @public
		 * @since 1.0.0
		 */
		closePath(): void;

		/**
		 * 画一个矩形
		 * @method drawRect
		 * @param {number} x
		 * @param {number} y
		 * @param {number} w
		 * @param {number} h
		 * @public
		 * @since 1.0.0
		 */
		drawRect(x: number, y: number, w: number, h: number): void;

		/**
		 * 画一个弧形
		 * https://www.w3school.com.cn/tags/canvas_arc.asp
		 *
		 * @method drawArc
		 * @param {number} x 起始点x
		 * @param {number} y 起始点y
		 * @param {number} radius 半径
		 * @param {number} start 开始角度 弧度
		 * @param {number} end 结束角度 弧度
		 * @param {number} counterclockwise False = 顺时针，true = 逆时针
		 * @public
		 * @since 1.0.0
		 */
		arc(x: number, y: number, radius: number, startAngle: number, endAngle: number, counterclockwise?: boolean): void;

		/**
		 * 画一个圆
		 * @method drawCircle
		 * @param {number} x 圆心x
		 * @param {number} y 圆心y
		 * @param {number} radius 半径
		 * @param {boolean} counterclockwise 半径
		 * @public
		 * @since 1.0.0
		 */
		drawCircle(x: number, y: number, radius: number, counterclockwise?: boolean): void;

		/**
		 * 画一个椭圆
		 * @method drawEllipse
		 * @param {number} x
		 * @param {number} y
		 * @param {number} w
		 * @param {number} h
		 * @public
		 * @since 1.0.0
		 */
		drawEllipse(x: number, y: number, w: number, h: number): void;

		/**
		 * 清除掉之前所有绘画的东西
		 * @method clear
		 * @public
		 * @since 1.0.0
		 */
		clear(): void;

		/**
		 * 开始绘画填充,如果想画的东西有颜色填充,一定要从此方法开始
		 * @method beginFill
		 * @param {number} color 颜色值 单色和RGBA格式
		 * @public
		 * @since 1.0.0
		 */
		beginFill(color: number, alpha?: number): void;

		/**
		 * 线性渐变填充 一般给Flash2x用
		 * @method beginLinearGradientFill
		 * @param {Array} points 一组点
		 * @param {Array} colors 一组颜色值
		 * @public
		 * @since 1.0.0
		 */
		beginLinearGradientFill(points: any, colors: any): void;

		/**
		 * 径向渐变填充 一般给Flash2x用
		 * @method beginRadialGradientFill
		 * @param {Array} points 一组点
		 * @param {Array} colors 一组颜色值
		 * @param {Object} matrixDate 如果渐变填充有矩阵变形信息
		 * @public
		 * @since 1.0.0
		 */
		beginRadialGradientFill: (points: any, colors: any) => void;

		/**
		 * 位图填充 一般给Flash2x用
		 * @method beginBitmapFill
		 * @param {Image} image
		 * @param { Array} matrix
		 * @public
		 * @since 1.0.0
		 */
		beginBitmapFill(image: any, matrix: Array<number>): void;

		private _fill;

		/**
		 * 给线条着色
		 * @method beginStroke
		 * @param {number} color  颜色值
		 * @param {number} lineWidth 宽度
		 * @param {number} cap 线头的形状 0 butt 1 round 2 square 默认 butt
		 * @param {number} join 线与线之间的交接处形状 0 miter 1 bevel 2 round  默认miter
		 * @param {number} miter 正数,规定最大斜接长度,如果斜接长度超过 miterLimit 的值，边角会以 lineJoin 的 "bevel" 类型来显示 默认10
		 * @public
		 * @since 1.0.0
		 */
		beginStroke(color: number, lineWidth?: number, cap?: LINE_CAP, join?: LINE_JOIN, miter?: number, alpha?: number): void;

		/**
		 * 画线性渐变的线条 一般给Flash2x用
		 * @method beginLinearGradientStroke
		 * @param {Array} points 一组点
		 * @param {Array} colors 一组颜色值
		 * @param {number} lineWidth
		 * @param {number} cap 线头的形状 0 butt 1 round 2 square 默认 butt
		 * @param {number} join 线与线之间的交接处形状 0 miter 1 bevel 2 round  默认miter
		 * @param {number} miter 正数,规定最大斜接长度,如果斜接长度超过 miterLimit 的值，边角会以 lineJoin 的 "bevel" 类型来显示 默认10
		 * @public
		 * @since 1.0.0
		 */
		beginLinearGradientStroke(points: Array<number>, colors: any, lineWidth?: number, cap?: LINE_CAP, join?: LINE_JOIN, miter?: number): void;

		/**
		 * 画径向渐变的线条 一般给Flash2x用
		 * @method beginRadialGradientStroke
		 * @param {Array} points 一组点
		 * @param {Array} colors 一组颜色值
		 * @param {number} lineWidth
		 * @param {string} cap 线头的形状 butt round square 默认 butt
		 * @param {string} join 线与线之间的交接处形状 bevel round miter 默认miter
		 * @param {number} miter 正数,规定最大斜接长度,如果斜接长度超过 miterLimit 的值，边角会以 lineJoin 的 "bevel" 类型来显示 默认10
		 * @public
		 * @since 1.0.0
		 */
		beginRadialGradientStroke(points: Array<number>, colors: any, lineWidth?: number, cap?: LINE_CAP, join?: LINE_JOIN, miter?: number): void;

		/**
		 * 线条位图填充 一般给Flash2x用
		 * @method beginBitmapStroke
		 * @param {Image} image
		 * @param {Array} matrix
		 * @param {number} lineWidth
		 * @param {string} cap 线头的形状 butt round square 默认 butt
		 * @param {string} join 线与线之间的交接处形状 bevel round miter 默认miter
		 * @param {number} miter 正数,规定最大斜接长度,如果斜接长度超过 miterLimit 的值，边角会以 lineJoin 的 "bevel" 类型来显示 默认10
		 * @public
		 * @since 1.0.0
		 */
		beginBitmapStroke(image: any, matrix: Array<number>, lineWidth?: number, cap?: LINE_CAP, join?: LINE_JOIN, miter?: number): void;

		/**
		 * @method _stroke
		 * @param strokeStyle
		 * @param {number} width
		 * @param {number} cap
		 * @param {number} join
		 * @param {number} miter
		 * @private
		 * @since 1.0.0
		 */
		private _stroke;

		/**
		 * 结束填充
		 * @method endFill
		 * @public
		 * @since 1.0.0
		 */
		endFill(): void;

		/**
		 * 结束画线
		 * @method endStroke
		 * @public
		 * @since 1.0.0
		 */
		endStroke(): void;

		/**
		 * 重写刷新
		 * @method update
		 * @public
		 * @since 1.0.0
		 */
		updateShape(): void;

		/**
		 * @method _drawShape
		 * @param ctx
		 * @private
		 * @return {void}
		 */
		private _drawShape;

		/**
		 * 重写hitTestPoint
		 * @method  hitTestPoint
		 * @param {Point} globalPoint
		 * @param {boolean} isMouseEvent
		 * @return {any}
		 * @public
		 * @since 1.0.0
		 */
		hitTestPoint(globalPoint: Point, isMouseEvent?: boolean): DisplayObject;

		/**
		 * 如果有的话,改变矢量对象的边框或者填充的颜色.
		 * @method changeColor
		 * @param {Object} infoObj
		 * @param {string|any} infoObj.fillColor 填充颜色值，如"#fff" 或者 "rgba(255,255,255,1)"或者是Shape.getGradientColor()方法返回的渐变对象;
		 * @param {string} infoObj.strokeColor 线条颜色值，如"#fff" 或者 "rgba(255,255,255,1)";
		 * @param {number} infoObj.lineWidth 线条的粗细，如"1,2,3...";
		 * @public
		 * @since 1.0.2
		 */
		changeColor(infoObj: any): void;

		/**
		 * 更新texture及baseTexture属性
		 * 不考虑trim，
		 */
		updateTexture(): void;

		_renderCanvas(renderer: any): void;

		_renderWebGL(renderer: any): void;

		destroy(): void;
	}

	export class TextureSheet {
		/**
		 * Reference to ths source texture
		 */
		baseTexture: any;
		/**
		 * A map containing all textures of the sprite sheet.
		 * Can be used to create a {@link Sprite|Sprite}:
		 * ```js
		 * new Sprite(sheet.textures["image.png"]);
		 */
		textures: {};
		/**
		 * A map containing the textures for each animation.
		 * Can be used to create an {@link extras.AnimatedSprite|AnimatedSprite}:
		 * ```js
		 * new extras.AnimatedSprite(sheet.animations["anim_name"])
		 * ```
		 * @member {Object}
		 */
		animations: {};
		/**
		 * Reference to the original JSON data.
		 * @type {Object}
		 */
		data: any;
		/**
		 * Map of TextureSheet frames.
		 * @type {Object}
		 * @private
		 */
		_frames: any;
		/**
		 * Collection of frame names.
		 * @type {string[]}
		 * @private
		 */
		_frameKeys: string[];
		/**
		 * Callback when parse is completed.
		 * @private
		 */
		_callback: Function;

		/**
		 * @param {BaseTexture} baseTexture Reference to the source BaseTexture object.
		 * @param {Object} data - TextureSheet image data.
		 */
		constructor(baseTexture: BaseTexture | Texture, data: any);

		/**
		 * Parser TextureSheet from loaded data. This is done asynchronously
		 * to prevent creating too many Texture within a single process.
		 *
		 * @param {Function} callback - Callback when complete returns
		 *        a map of the Textures for this TextureSheet.
		 */
		parse(callback: Function): void;

		/**
		 * Process a batch of frames
		 *
		 * @private
		 * @param {number} initialFrameIndex - The index of frame to start.
		 */
		_processFrames(initialFrameIndex: number): void;

		/**
		 * Parse animations config
		 *
		 * @private
		 */
		_processAnimations(): void;

		/**
		 * The parse has completed.
		 *
		 * @private
		 */
		_parseComplete(): void;

		/**
		 * Destroy TextureSheet and don't use after this.
		 *
		 * @param {boolean} [destroyBase=false] Whether to destroy the base texture as well
		 */
		destroy(destroyBase?: boolean): void;
	}

	export function getTexture(str: string): any;

	export function httpRequest(url: string, method?: string, params?: any, type?: 'text' | 'json' | 'jsonp', headers?: any): any;

	export function jsonp(url: any, params: any): any;

	export function urlJoin(url: any, query: any): any;

	export function createTextureSheet(baseTexture: BaseTexture, altaData: any): void;

	export class Loader extends EventDispatcher {
		/**
		 * 记录原始数据，json和image，贴图在建立时会被缓存
		 */
		caches: {};
		crossOrigin: boolean;

		/**
		 *
		 */
		constructor();

		/**
		 *
		 * @param url
		 */
		loadSheet(url: string): any;

		loadRaw(url: string, uuid?: string, type?: 'text' | 'json'): any;

		loadJson(url: string, uuid?: string): any;

		loadText(url: string, uuid?: string): any;

		loadImage(url: string, uuid?: string): any;

		/**
		 * 加载纹理
		 * @param url
		 * @param uuid
		 */
		loadTexture(url: string, uuid?: string): any;

		cache(name: string, data: any): void;

		get(name: string): any;
	}

	export const globalLoader: Loader;

	export class TextField extends Sprite {
		canvas: HTMLCanvasElement;
		context: CanvasRenderingContext2D;
		/**
		 * 与其他类不同，用了Boolean，再考虑
		 */
		dirty: boolean;
		offsetX: number;
		offsetY: number;

		constructor();

		/**
		 * @property textAlpha
		 * @since 2.0.0
		 * @public
		 */
		textAlpha: number;
		private _textAlpha;
		/**
		 * 文本的水平对齐方式 left center right
		 * 设置过textHeight才有效，如果是多行的，对排版有作用
		 * @property textAlign
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 * @default left
		 */
		textAlign: TEXT_ALIGN;
		private _textAlign;
		/**
		 * 垂直对齐方式
		 * 设置过textHeight才有效
		 */
		verticalAlign: VERTICAL_ALIGN;
		private _verticalAlign;
		/**
		 * 文本的宽，
		 * @property textWidth
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 0
		 */
		/**
		 * 文本的行高，设置过能进行垂直适配，
		 * 文本超出行高会被裁切
		 * @property textHeight
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 0
		 */
		width: number;
		height: number;
		/**
		 * 行间距
		 * @property lineSpacing
		 * @public
		 * @since 1.0.0
		 * @param {number} value
		 */
		lineSpacing: number;
		private _lineSpacing;
		/**
		 * 文本类型,单行还是多行 single multi
		 * @property lineType
		 * @public
		 * @since 1.0.0
		 * @type {string} 两种 single和multi
		 * @default single
		 */
		lineType: TEXT_lINETYPE;
		private _lineType;
		protected _text: string;
		/**
		 * 文本内容
		 * @property text
		 * @type {string}
		 * @public
		 * @default ""
		 * @since 1.0.0
		 */
		text: string;

		protected _setText(value: any): void;

		protected _textFlow: any;
		protected _pureText: string;
		textFlow: any;
		readonly isPureText: boolean;
		readonly pureText: string;

		protected getStyle(index: any): any;

		/**
		 * 文本的css字体样式
		 * @property font
		 * @public
		 * @since 1.0.0
		 * @type {string}
		 * @default 12px Arial
		 */
		font: string;
		private _font;
		/**
		 * 文本的size
		 * @property size
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 12
		 */
		size: number;
		private _size;
		/**
		 * 文本的填充颜色值
		 * @property fillColor
		 * @type {string}
		 * @public
		 * @since 1.0.0
		 * @default #fff
		 */
		fillColor: any;
		private _fillColor;
		/**
		 * 文本的描边颜色值
		 * @property strokeColor
		 * @type {string}
		 * @public
		 * @since 1.0.0
		 * @default #fff
		 */
		strokeColor: string;
		private _strokeColor;
		/**
		 * 文本描边宽度,为0则不描边
		 * @property stroke
		 * @public
		 * @since
		 * @default 0
		 * @type {number}
		 */
		stroke: number;
		private _stroke;
		/**
		 * 文本是否倾斜
		 * @property italic
		 * @public
		 * @since
		 * @default false
		 * @type {boolean}
		 */
		italic: boolean;
		private _italic;
		/**
		 * 文本是否加粗
		 * @property bold
		 * @public
		 * @since
		 * @default false
		 * @type {boolean}
		 */
		bold: boolean;
		_bold: boolean;
		/**
		 * 设置或获取是否有边框
		 * @property property
		 * @param {boolean} show true或false
		 * @public
		 * @since 1.0.6
		 */
		border: boolean;
		private _border;
		/**
		 * 设置文本在canvas里的渲染样式
		 * @method _prepContext
		 * @param ctx
		 * @private
		 * @since 1.0.0
		 */
		private _prepContext;

		/**
		 * 获取当前文本中单行文字的宽，注意是文字的不是文本框的宽
		 * @method getTextWidth
		 * @param {number} lineIndex 获取的哪一行的高度 默认是第1行
		 * @since 2.0.0
		 * @public
		 * @return {number}
		 */
		getTextWidth(lineIndex?: number): number;

		/**
		 * @property _lines 获取当前文本行数
		 * @type {number}
		 * @public
		 * @readonly
		 * @since 2.0.0
		 */
		readonly lines: number;
		/**
		 * 获取文本宽
		 * @method _getMeasuredWidth
		 * @param text
		 * @return {number}
		 * @private
		 * @since 1.0.0
		 */
		private _getMeasuredWidth;
		private realLines;

		/**
		 * 更新文本，主要重画canvas
		 */
		updateText(): void;

		/**
		 * 更新texture及baseTexture属性
		 * 不考虑trim，
		 */
		updateTexture(): void;

		_renderCanvas(renderer: any): void;

		_renderWebGL(renderer: any): void;

		destroy(): void;
	}

	export class InputText extends FloatDisplay {
		/**
		 * canvas上的视图,暂时不用，如果遇到截图的，可以先调用方法建立TextField，如果超了字符，只取前面的
		 */
		private _view;
		/**
		 * 输入文本的类型.
		 * @property inputType
		 * @public
		 * @since 1.0.0
		 * @type {number} 0 input 1 password 2 mulit
		 * @default 0
		 */
		inputType: number;
		/**
		 * 在手机端是否需要自动收回软键盘，在pc端此参数无效
		 * @property isAutoDownKeyBoard
		 * @type {boolean}
		 * @since 1.0.3
		 * @default true
		 */
		isAutoDownKeyBoard: boolean;
		/**
		 * @property _inputTypeList
		 * @static
		 * @type {string[]}
		 * @private
		 * @since 2.0.0
		 */
		private static _inputTypeList;

		/**
		 * @method InputText
		 * @public
		 * @since 1.0.0
		 * @param {number} inputType 0 input 1 password 2 multiline
		 * @example
		 *      var inputText=new InputText();
		 *      inputText.initInfo('aa',100,100,'#ffffff','left',14,'微软雅黑',false,2);
		 */
		constructor(inputType?: number);

		/**
		 * 初始化输入文本
		 * @method init
		 * @param htmlElement
		 * @public
		 * @return {void}
		 * @since 1.0.0
		 */
		protected init(htmlElement: any): void;

		/**
		 * 被始化输入文件的一些属性
		 * @method initInfo
		 * @public
		 * @since 1.0.0
		 * @param {string} text 默认文字
		 * @param {string}color 文字颜色
		 * @param {string}align 文字的对齐方式
		 * @param {number}size  文字大小
		 * @param {string}font  文字所使用的字体
		 * @param {boolean}showBorder 是否需要显示边框
		 * @param {number}lineSpacing 如果是多行,请设置行高
		 * @param {number}textWidth 文本宽
		 */
		initInfo(text: string, color: string, align: string, size: number, font: string, showBorder: boolean, lineSpacing: number, textWidth: number): void;

		/**
		 * @property lineSpacing
		 * @public
		 * @since 2.0.0
		 * @param {number} value
		 */
		lineSpacing: number;
		/**
		 * 设置文本是否为粗体
		 * @property bold
		 * @param {boolean} bold true或false
		 * @public
		 * @since 1.0.3
		 */
		bold: boolean;
		/**
		 * 设置文本是否倾斜
		 * @property italic
		 * @param {boolean} italic true或false
		 * @public
		 * @since 1.0.3
		 */
		italic: boolean;
		/**
		 * 文本的行高
		 * @property textHeight
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 0
		 */
		textHeight: number;
		/**
		 * 文本的宽
		 * @property textWidth
		 * @public
		 * @since 1.0.0
		 * @type {number}
		 * @default 0
		 */
		textWidth: number;
		/**
		 * 设置文本颜色
		 * @property color
		 * @param {boolean} italic true或false
		 * @public
		 * @since 1.0.3
		 */
		color: string;
		/**
		 * 设置或获取是否有边框
		 * @property property
		 * @param {boolean} show true或false
		 * @public
		 * @since 1.0.3
		 */
		border: boolean;
		/**
		 * 获取或设置输入文本的值
		 * 之前的getText 和setText 已废弃
		 * @property text
		 * @public
		 * @since 1.0.3
		 * @return {string}
		 */
		text: string;
		/**
		 * 输入文本的最大输入字数
		 * @public
		 * @since 1.1.0
		 * @property maxCharacters
		 * @return {number}
		 */
		maxCharacters: number;

		/**
		 * 转换为动态文本，位置上下会有偏差，再说，基本用于截图时，需要截到输入文本时
		 * @param textField
		 */
		convertToTextField(textField?: TextField): TextField;
	}

	export class EditableText extends TextField {
		htmlElement: any;
		private _prompt;
		prompt: string;
		private _promptColor;
		promptColor: string;
		private _textColor;
		textColor: string;

		constructor();
	}

	export class ScrollViewBase extends Container {
		/**
		 * 滚动方向
		 * @property direction
		 * @type {SCROLL_DIRECTION}
		 * @private
		 * @since 1.0.0
		 * @default true
		 */
		protected _direction: SCROLL_DIRECTION;
		/**
		 * 可见区域的宽
		 * @property viewWidth
		 * @type {number}
		 * @private
		 * @since 1.0.0
		 * @default 0
		 */
		/**
		 * 可见区域的高
		 * @property viewHeight
		 * @type {number}
		 * @private
		 * @since 1.0.0
		 * @default 0
		 */
		/**
		 * 整个滚动的最大距离值
		 * @property maxDistance
		 * @type {number}
		 * @public
		 * @since 1.0.0
		 * @default 1040
		 */
		/**
		 * @property 滚动距离
		 * @type {number}
		 * @protected
		 * @default 0
		 * @since 1.0.0
		 */
		protected distance: number;
		/**
		 * 最小鼠标滑动距离
		 * @type {number}
		 */
		private minDis;
		/**
		 * 遮罩对象
		 * @property maskObj
		 * @since 1.0.0
		 * @private
		 * @type {Graphics}
		 */
		private maskObj;
		/**
		 * 真正的容器对象，所有滚动的内容都应该是添加到这个容器中
		 * @property view
		 * @public
		 * @since 1.0.0
		 * @type {Container}
		 */
		viewPort: Container;
		/**
		 * 最后鼠标经过的坐标值
		 * @property lastValue
		 * @private
		 * @since 1.0.0
		 * @type {number}
		 */
		private lastValue;
		/**
		 * 速度
		 * @property speed
		 * @protected
		 * @since 1.0.0
		 * @type {number}
		 */
		protected speed: number;
		/**
		 * 加速度
		 * @property addSpeed
		 * @private
		 * @since 1.0.0
		 * @type {number}
		 */
		private addSpeed;
		/**
		 * 是否是停止滚动状态
		 * @property isStop
		 * @public
		 * @since 1.0.0
		 * @type {boolean}
		 * @default true
		 */
		isStop: boolean;
		/**
		 * 滚动的最大速度，直接影响一次滑动之后最长可以滚多远
		 * @property maxSpeed
		 * @public
		 * @since 1.0.0
		 * @default 100
		 * @type {number}
		 */
		maxSpeed: number;
		/**
		 * 摩擦力,值越大，减速越快
		 * @property fSpeed
		 * @public
		 * @since 1.0.0
		 * @default 20
		 * @type {number}
		 */
		fSpeed: number;
		protected paramXY: string;
		protected paramSize: string;
		private stopTimes;
		private isMouseDownState;
		/**
		 * 是否是通过scrollTo方法在滑动中
		 * @property autoScroll
		 * @since 1.0.2
		 * @type {boolean}
		 * @private
		 * @default false
		 */
		private autoScroll;
		isSpringBack: boolean;

		/**
		 * 构造函数
		 * @method  ScrollPage
		 * @param {number} maxDistance 最大滚动的长度
		 * @param isFull
		 * @example
		 *      var sPage=new ScrollPage(640,s.stage.viewRect.height,4943);
		 *      sPage.isSpringBack = false;//是否回弹
		 *      stage.addChild(sPage);
		 *      sPage.view.addChild(view);
		 *      sPage.y=stage.viewRect.y;
		 *
		 */
		constructor(maxDistance?: number, isFull?: boolean);

		readonly maxDistance: any;

		protected calMaxDistance(): any;

		direction: SCROLL_DIRECTION;

		protected updateDirection(): void;

		/**
		 * 设置可见区域，可见区域的坐标始终在本地坐标中0,0点位置
		 * @method setViewRect
		 * @public
		 * @since 1.1.1
		 */
		updateViewRect(): void;

		cancelScroll(): void;

		private onMouseEvent;

		/**
		 * 滚到指定的坐标位置
		 * @method scrollTo
		 * @param {number} dis 需要去到的位置,初始位置0,最大为maxDistance- s.viewWidth : s.viewHeight
		 * @param {number} time 滚动需要的时间 默认为0 即没有动画效果直接跳到指定页
		 * @since 1.1.1
		 * @public
		 */
		scrollTo(dis: number, time?: number): void;

		destroy(): void;
	}

	export function fieldChanged(onModify: any): (target: any, key: string) => void;

	export const dirtyFieldDetector: (target: any, key: string) => void;

	export const deepDirtyFieldDetector: (target: any, key: string) => void;

	export const dirtyFieldTrigger: (target: any, key: string) => void;

	export const deepDirtyFieldTrigger: (target: any, key: string) => void;

	export abstract class ScrollListItemBase extends Container {
		id: number;
		data: any;
		sli_id: number;

		_initData(id: number, data: any): void;

		abstract updateData(data: any): any;
	}

	export class ScrollListBase extends ScrollViewBase {
		private _items;
		private _itemRow;
		private _itemCount;
		private _isInit;
		data: Array<any>;
		private downL;
		private _cols;
		private _disParam;
		private _lastFirstId;
		private _updateId;
		itemWidth: number;
		itemHeight: number;
		itemCol: number;
		itemClass: any;
		/**
		 * 获取下拉滚动的loadingView对象
		 * @property loadingView
		 * @since 1.0.9
		 * @return {DisplayObject}
		 */
		readonly loadingView: DisplayObject;

		/**
		 * 构造函数
		 * @method ScrollList
		 * @since 1.0.9
		 */
		constructor();

		protected onNextFrame(event: any): void;

		onModify(value: any, key: any): void;

		protected calMaxDistance(): any;

		/**
		 * 更新列表数据
		 * @method updateData
		 * @param {Array} data
		 * @param {boolean} isReset 是否重围数据列表。
		 * @since 1.0.9
		 */
		updateData(data: Array<any>, isReset?: boolean): void;

		private flushData;

		/**
		 * 设置可见区域，可见区域的坐标始终在本地坐标中0,0点位置
		 * @method setViewRect
		 * @public
		 * @since 1.1.1
		 */
		updateViewRect(): void;

		protected _updateViewRect(): void;

		protected createItem(): any;

		/**
		 * 设置加载数据时显示的loading对象
		 * @since 1.0.9
		 * @method setLoading
		 * @param {DisplayObject} downLoading
		 */
		setLoading(downLoading: DisplayObject): void;

		destroy(): void;
	}

	export class Button extends Sprite {
		textureUp: any;
		textureDown: any;
		textureDisable: any;

		constructor(tUp: Texture, tDown?: Texture, tDisable?: Texture);

		/**
		 * @method  _mouseEvent
		 * @param e
		 * @private
		 */
		private _mouseEvent;

		initButton(): void;

		/**
		 * 设置是否为点击状态
		 * @property clicked
		 * @param {boolean} value
		 * @public
		 * @since 2.0.0
		 * @default false
		 */
		clicked: boolean;
		private _clicked;
		textureStatusEnable: boolean;

		destroy(): void;
	}

	export class Tween {
		/**
		 * 不做特殊处理
		 * @constant {number} Tween.NONE
		 * @private
		 */
		private static NONE;
		/**
		 * 循环
		 * @constant {number} Tween.LOOP
		 * @private
		 */
		private static LOOP;
		/**
		 * 倒序
		 * @constant {number} Tween.REVERSE
		 * @private
		 */
		private static REVERSE;
		/**
		 * @private
		 */
		private static _tweens;
		/**
		 * @private
		 */
		private static IGNORE;
		/**
		 * @private
		 */
		private static _plugins;
		/**
		 * @private
		 */
		private static _inited;
		/**
		 * @private
		 */
		private _target;
		/**
		 * @private
		 */
		private _useTicks;
		/**
		 * @private
		 */
		private ignoreGlobalPause;
		/**
		 * @private
		 */
		private loop;
		/**
		 * @private
		 */
		private pluginData;
		/**
		 * @private
		 */
		private _curQueueProps;
		/**
		 * @private
		 */
		private _initQueueProps;
		/**
		 * @private
		 */
		private _steps;
		/**
		 * @private
		 */
		private paused;
		/**
		 * @private
		 */
		private duration;
		/**
		 * @private
		 */
		private _prevPos;
		/**
		 * @private
		 */
		private position;
		/**
		 * @private
		 */
		private _prevPosition;
		/**
		 * @private
		 */
		private _stepPosition;
		/**
		 * @private
		 */
		private passive;

		/**
		 * Activate an object and add a Tween animation to the object
		 * @param target {any} The object to be activated
		 * @param props {any} Parameters, support loop onChange onChangeObj
		 * @param pluginData {any} Write realized
		 * @param override {boolean} Whether to remove the object before adding a tween, the default value false
		 * Not recommended, you can use Tween.removeTweens(target) instead.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 激活一个对象，对其添加 Tween 动画
		 * @param target {any} 要激活 Tween 的对象
		 * @param props {any} 参数，支持loop(循环播放) onChange(变化函数) onChangeObj(变化函数作用域)
		 * @param pluginData {any} 暂未实现
		 * @param override {boolean} 是否移除对象之前添加的tween，默认值false。
		 * 不建议使用，可使用 Tween.removeTweens(target) 代替。
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static get(target: any, props?: {
			loop?: boolean;
			onChange?: Function;
			onChangeObj?: any;
		}, pluginData?: any, override?: boolean): Tween;

		/**
		 * Delete all Tween animations from an object
		 * @param target The object whose Tween to be deleted
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 删除一个对象上的全部 Tween 动画
		 * @param target  需要移除 Tween 的对象
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static removeTweens(target: any): void;

		/**
		 * Pause all Tween animations of a certain object
		 * @param target The object whose Tween to be paused
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 暂停某个对象的所有 Tween
		 * @param target 要暂停 Tween 的对象
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static pauseTweens(target: any): void;

		/**
		 * Resume playing all easing of a certain object
		 * @param target The object whose Tween to be resumed
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 继续播放某个对象的所有缓动
		 * @param target 要继续播放 Tween 的对象
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static resumeTweens(target: any): void;

		/**
		 * @private
		 *
		 * @param delta
		 * @param paused
		 */
		private static tick;

		/**
		 * flush方法，为了能加入总循环
		 * 默认是锁步的
		 * @param delta
		 * @param paused ,暂时不用，全局禁止
		 */
		static flush(): void;

		private static _lastTime;
		/**
		 * @private
		 *
		 * @param tween
		 * @param value
		 */
		private static _register;

		/**
		 * Delete all Tween
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 删除所有 Tween
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static removeAllTweens(): void;

		/**
		 * 创建一个 Tween 对象
		 * @private
		 * @version
		 * @platform Web,Native
		 */
		constructor(target: any, props: any, pluginData: any);

		onChange: Function;
		/**
		 * @private
		 *
		 * @param target
		 * @param props
		 * @param pluginData
		 */
		private initialize;

		/**
		 * @private
		 *
		 * @param value
		 * @param actionsMode
		 * @returns
		 */
		setPosition(value: number, actionsMode?: number): boolean;

		/**
		 * @private
		 *
		 * @param startPos
		 * @param endPos
		 * @param includeStart
		 */
		private _runAction;
		/**
		 * @private
		 *
		 * @param step
		 * @param ratio
		 */
		private _updateTargetProps;

		/**
		 * Whether setting is paused
		 * @param value {boolean} Whether to pause
		 * @returns Tween object itself
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 设置是否暂停
		 * @param value {boolean} 是否暂停
		 * @returns Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		setPaused(value: boolean): Tween;

		/**
		 * @private
		 *
		 * @param props
		 * @returns
		 */
		private _cloneProps;
		/**
		 * @private
		 *
		 * @param o
		 * @returns
		 */
		private _addStep;
		/**
		 * @private
		 *
		 * @param o
		 * @returns
		 */
		private _appendQueueProps;
		/**
		 * @private
		 *
		 * @param o
		 * @returns
		 */
		private _addAction;
		/**
		 * @private
		 *
		 * @param props
		 * @param o
		 */
		private _set;

		/**
		 * Wait the specified milliseconds before the execution of the next animation
		 * @param duration {number} Waiting time, in milliseconds
		 * @param passive {boolean} Whether properties are updated during the waiting time
		 * @returns Tween object itself
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 等待指定毫秒后执行下一个动画
		 * @param duration {number} 要等待的时间，以毫秒为单位
		 * @param passive {boolean} 等待期间属性是否会更新
		 * @returns Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		wait(duration: number, passive?: boolean): Tween;

		/**
		 * Modify the property of the specified object to a specified value
		 * @param props {Object} Property set of an object
		 * @param duration {number} Duration
		 * @param ease {Ease} Easing algorithm
		 * @returns {Tween} Tween object itself
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 将指定对象的属性修改为指定值
		 * @param props {Object} 对象的属性集合
		 * @param duration {number} 持续时间
		 * @param ease {Ease} 缓动算法
		 * @returns {Tween} Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		to(props: any, duration?: number, ease?: Function): Tween;

		/**
		 * Execute callback function
		 * @param callback {Function} Callback method
		 * @param thisObj {any} this action scope of the callback method
		 * @param params {any[]} Parameter of the callback method
		 * @returns {Tween} Tween object itself
		 * @version
		 * @platform Web,Native
		 * @example
		 * <pre>
		 *  Tween.get(display).call(function (a:number, b:string) {
		 *      console.log("a: " + a); // the first parameter passed 233
		 *      console.log("b: " + b); // the second parameter passed “hello”
		 *  }, this, [233, "hello"]);
		 * </pre>
		 * @language en_US
		 */
		/**
		 * 执行回调函数
		 * @param callback {Function} 回调方法
		 * @param thisObj {any} 回调方法this作用域
		 * @param params {any[]} 回调方法参数
		 * @returns {Tween} Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @example
		 * <pre>
		 *  Tween.get(display).call(function (a:number, b:string) {
		 *      console.log("a: " + a); //对应传入的第一个参数 233
		 *      console.log("b: " + b); //对应传入的第二个参数 “hello”
		 *  }, this, [233, "hello"]);
		 * </pre>
		 * @language zh_CN
		 */
		call(callback: Function, thisObj?: any, params?: any[]): Tween;

		/**
		 * Now modify the properties of the specified object to the specified value
		 * @param props {Object} Property set of an object
		 * @param target The object whose Tween to be resumed
		 * @returns {Tween} Tween object itself
		 * @version
		 * @platform Web,Native
		 */
		/**
		 * 立即将指定对象的属性修改为指定值
		 * @param props {Object} 对象的属性集合
		 * @param target 要继续播放 Tween 的对象
		 * @returns {Tween} Tween对象本身
		 * @version
		 * @platform Web,Native
		 */
		set(props: any, target?: any): Tween;

		/**
		 * Execute
		 * @param tween {Tween} The Tween object to be operated. Default: this
		 * @returns {Tween} Tween object itself
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 执行
		 * @param tween {Tween} 需要操作的 Tween 对象，默认this
		 * @returns {Tween} Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		play(tween?: Tween): Tween;

		/**
		 * Pause
		 * @param tween {Tween} The Tween object to be operated. Default: this
		 * @returns {Tween} Tween object itself
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * 暂停
		 * @param tween {Tween} 需要操作的 Tween 对象，默认this
		 * @returns {Tween} Tween对象本身
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		pause(tween?: Tween): Tween;

		/**
		 * @method Tween#tick
		 * @param delta {number}
		 * @private
		 * @version
		 * @platform Web,Native
		 */
		$tick(delta: number): void;
	}

	export class Ease {
		/**
		 * @version
		 * @platform Web,Native
		 */
		constructor();

		/**
		 * get.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static get(amount: number): (t: number) => number;

		/**
		 * get pow in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get pow in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getPowIn(pow: number): (t: number) => number;

		/**
		 * get pow out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get pow out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getPowOut(pow: number): (t: number) => number;

		/**
		 * get pow in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get pow in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getPowInOut(pow: number): (t: number) => number;

		/**
		 * quad in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quad in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quadIn: (t: number) => number;
		/**
		 * quad out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quad out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quadOut: (t: number) => number;
		/**
		 * quad in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quad in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quadInOut: (t: number) => number;
		/**
		 * cubic in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * cubic in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static cubicIn: (t: number) => number;
		/**
		 * cubic out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * cubic out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static cubicOut: (t: number) => number;
		/**
		 * cubic in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * cubic in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static cubicInOut: (t: number) => number;
		/**
		 * quart in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quart in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quartIn: (t: number) => number;
		/**
		 * quart out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quart out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quartOut: (t: number) => number;
		/**
		 * quart in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quart in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quartInOut: (t: number) => number;
		/**
		 * quint in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quint in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quintIn: (t: number) => number;
		/**
		 * quint out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quint out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quintOut: (t: number) => number;
		/**
		 * quint in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * quint in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static quintInOut: (t: number) => number;

		/**
		 * sine in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * sine in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static sineIn(t: number): number;

		/**
		 * sine out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * sine out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static sineOut(t: number): number;

		/**
		 * sine in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * sine in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static sineInOut(t: number): number;

		/**
		 * get back in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get back in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getBackIn(amount: number): (t: number) => number;

		/**
		 * back in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * back in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static backIn: (t: number) => number;

		/**
		 * get back out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get back out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getBackOut(amount: number): (t: any) => number;

		/**
		 * back out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * back out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static backOut: (t: any) => number;

		/**
		 * get back in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get back in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getBackInOut(amount: number): (t: number) => number;

		/**
		 * back in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * back in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static backInOut: (t: number) => number;

		/**
		 * circ in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * circ in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static circIn(t: number): number;

		/**
		 * circ out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * circ out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static circOut(t: number): number;

		/**
		 * circ in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * circ in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static circInOut(t: number): number;

		/**
		 * bounce in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * bounce in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static bounceIn(t: number): number;

		/**
		 * bounce out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * bounce out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static bounceOut(t: number): number;

		/**
		 * bounce in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * bounce in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static bounceInOut(t: number): number;

		/**
		 * get elastic in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get elastic in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getElasticIn(amplitude: number, period: number): (t: number) => number;

		/**
		 * elastic in.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * elastic in。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static elasticIn: (t: number) => number;

		/**
		 * get elastic out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get elastic out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getElasticOut(amplitude: number, period: number): (t: number) => number;

		/**
		 * elastic out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * elastic out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static elasticOut: (t: number) => number;

		/**
		 * get elastic in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * get elastic in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static getElasticInOut(amplitude: number, period: number): (t: number) => number;

		/**
		 * elastic in out.See example.
		 * @version
		 * @platform Web,Native
		 * @language en_US
		 */
		/**
		 * elastic in out。请查看示例
		 * @version
		 * @platform Web,Native
		 * @language zh_CN
		 */
		static elasticInOut: (t: number) => number;
	}

	export class StackContainer extends Container {
		private _mutex;
		private _stack;

		constructor(mutex?: boolean);

		/**
		 * 推入视图
		 * @param view
		 * @param options
		 * @param dispatch
		 */
		push(view: DisplayObject, options?: any, dispatch?: boolean): void;

		/**
		 * 替换顶层视图
		 * @param view
		 * @param options
		 */
		replace(view: DisplayObject, options?: any): void;

		/**
		 * 撤出视图
		 */
		pop(dispatch?: boolean): boolean;

		/**
		 * 撤出全部视图
		 * @param view
		 * @param options
		 */
		popAll(view?: DisplayObject, options?: any): void;
	}

	export function parseSheet(sheet: any): void;

	export function parseAssetConfig(assetConfig: any, uuid?: any): void;

	export function parse(assetConfig: any, data: any): void;

	export function getFont(name: any): any;

	export function loadAssets(config: any, onProgress?: any, onComplete?: any): any;

	export function getAssetByUUID(uuid: any): any;

	export function getAssetByName(name: any): any;

	export class ShapeBase extends Graphics {
		protected __fieldDirty: boolean;
		fillColor: any;
		strokeColor: any;
		strokeWidth: number;

		constructor();

		private onResize;
		private onEnterFrame;

		protected redraw(): void;
	}

	export class Rect extends ShapeBase {
		borderRadius: number;

		protected redraw(): void;
	}

	export class Circle extends ShapeBase {
		protected redraw(): void;
	}

	export {};

	export class Image extends Sprite {
		private _originText;
		private _escapes;
		private _registeredEvents;
		private _source;
		source: string;
		private updateSource;
		private _setSourceDirect;
		private _setSource;
		/**
		 * 把之前侦听的全部移除
		 */
		private unregisterEvents;
		/**
		 * 当数据改变时
		 * @param e
		 */
		private onDataMutated;
	}

	export class Label extends TextField {
		private _originText;
		private _escapes;
		private _registeredEvents;
		private _htmlText;

		/**
		 * 重载文本设置
		 * @param value
		 * @private
		 */
		protected _setText(value: any): void;

		private dealEscape;
		/**
		 * 设置html文本
		 */
		htmlText: any;
		/**
		 * 把之前侦听的全部移除
		 */
		private unregisterEvents;
		/**
		 * 当TEXT改变时
		 * @param e
		 */
		private onTextMutated;
		/**
		 * 当HTML改变时
		 * @param e
		 */
		private onHtmlMutated;
	}

	export class TextInput extends Label {
		private _floatDisplay;
		private _placeholderLabel;
		private _input;
		placeholder: string;
		placeholderColor: any;
		maxLength: number;
		type: string;
		pattern: string;
		private _oldFillColor;
		private _oldStrokeColor;

		constructor();

		onModify(value: any, key: any): void;

		private setup;

		protected _setText(value: any): void;

		private setMaxLength;
		private showPlaceholderLabel;

		setFocus(): void;

		setBlur(): void;

		private onBlur;
		private onInput;
		private onClickStage;
		private onClickStatic;
		private onResize;
		static _bp: Point;

		/**
		 * 检测点是否在任何child上
		 * 重写父类方法
		 */
		hitTestPoint(globalPoint: Point, isMouseEvent?: boolean): this;
	}

	export const proxyMethods: string[];

	export class ScrollView extends ScrollViewBase {
		constructor();
	}

	export class ScrollListItem extends ScrollListItemBase {
		view: Container;

		setView(view: any): void;

		updateData(data: any): void;
	}

	export class ScrollList extends ScrollListBase {
		firstItem: Container;
		cloneEvents: boolean;
		cloneScripts: boolean;

		constructor();

		protected onNextFrame(event: any): void;

		protected createItem(): any;
	}

	export class BitmapText extends Container {
		private _charCache;
		private _fontConfig;
		text: string;
		font: string;
		verticalAlign: VERTICAL_ALIGN;
		letterSpacing: number;

		onModify(value: any, key: any): void;

		protected render(): void;

		private addChar;
	}

	export class HtmlView extends FloatDisplay {
	}

	export function registerNodeType(name: any, def: any): void;

	export function instantiate(config: any): any;

	export class Toast extends Container {
		private _contentSample;
		private _content;
		private _gameStage;

		constructor(gameStage: GameStage);

		show(props?: any): void;

		hide(animation?: boolean, hideDuration?: number): void;

		private removeContentView;
		private getContent;
	}

	export class GameStage extends Container {
		private _sceneContainer;
		private _popupContainer;
		private _toast;
		private _blackLayer;
		private _stage;
		private _dataCenter;
		private _config;
		private _viewCache;

		constructor(stage: Stage);

		/**
		 * 场景容器
		 */
		readonly sceneContainer: StackContainer;
		/**
		 * 弹层容器
		 */
		readonly popupContainer: StackContainer;
		/**
		 * Toast提示
		 */
		readonly toast: Toast;
		/**
		 * 数据中心
		 */
		readonly dataCenter: DataCenter;

		/**
		 * 启动游戏
		 * @param config
		 * @param onAssetsProgress
		 * @param onAssetsComplete
		 */
		launch(config: any, onAssetsProgress?: any, onAssetsComplete?: any, onStart?: any): Promise<void>;

		/**
		 * 开始游戏
		 */
		start(): void;

		/**
		 * 实例化视图
		 * @param name
		 * @param cache 如果开启缓存，就会以单例形式存在
		 */
		instantiateView(name: any, cache?: boolean): any;

		/**
		 * 根据name获取视图配置
		 * @param name
		 */
		getViewConfigByName(name: any): any;

		/**
		 * 设置半透明层是否可见
		 * @param visible
		 */
		setBlackLayerVisible(visible: any): void;

		onPopupContainerChange(e: any): void;
	}

	export function registerCustomModules(customs: any): void;

	export function registerCustomModule(id: any, def: any): void;

	export function registerCustomCodeModule(config: any): void;

	export function registerCustomModuleFromConfig(config: any): void;

	export function addCustomModule(id: any, container: Container, options?: any): any;

	export function resolveCustomAsset(id: any, uuid: any): any;

	export function getProps(id: any): any;

	export function playSound(uuid: any, options?: any, name?: any): any;

	export function stopSound(name: any): void;

	export function mute(muted: boolean): void;

	export function destroySound(name: any): void;

	export const _default: {
		onProgress(done: any, total: any): void;
		onComplete(): void;
	}

	export let gameStage: GameStage;

	export function launch(url: any, loadingDelegate?: any, onStart?: any): any;

	export function launchWithLocalStorage(id: any, loadingDelegate?: any, onStart?: any): any;

	export function launchWithWindowVariable(name: any, loadingDelegate?: any, onStart?: any): any;

	export function launchWithConfig(config: any, loadingDelegate?: any, onStart?: any): any;

	export let queryParams: any;

	export const emojiRegexp: RegExp;

	export {instantiate, registerNodeType, emojiRegexp,};
}
