/**
 * Project Name:tuia-youtui-web<br>
 * File Name:ActivityBO.java<br>
 * Package Name:com.duiba.tuia.youtui.web.bo<br>
 * Date:2017年1月17日下午4:15:57<br>
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.<br>
 */

package com.duiba.tuia.youtui.web.service;

import cn.com.duiba.tuia.activity.center.api.dto.ActivityOptionDto;
import cn.com.duiba.tuia.activity.center.api.dto.ActivitySkinDataDto;
import cn.com.duiba.tuia.activity.center.api.dto.BuoyDto;
import cn.com.duiba.tuia.activity.center.api.dto.ProbDto;
import cn.com.duiba.tuia.activity.center.api.dto.SkinTemplateDataDto;
import cn.com.duiba.tuia.activity.center.api.dto.TuiaActivityDto;
import com.duiba.tuia.youtui.web.exception.ActivityException;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;


/**
 * ClassName: ActivityBO <br/>
 * Function: 活动业务编排. <br/>
 * date: 2017年1月17日 下午4:15:57 <br/>
 *
 * @author leiliang
 * @since JDK 1.6
 */
public interface ActivityService {

    /**
     * 获取活动标签.
     *
     * @param activityId 活动id
     * @param activityType 活动type
     * @return the activity tag
     */
    String getActivityTag(Long activityId, Integer activityType);

    /**
     * 获取活动标签
     * @param activityId
     * @return
     */
    Set<Long> getActivityTag(Long activityId);

    /**
     * 批量获取活动标签
     * @param activityIds
     * @return
     */
    Map<Long,Set<Long>> getActivitysTags(List<Long> activityIds);

    /**
     * 查询活动详情.
     *
     * @param activityId 活动id
     * @return 活动详情
     * @throws ActivityException the activity exception
     */
    TuiaActivityDto getActivityById(Long activityId) throws ActivityException;

    /**
     * 查询活动详情.
     *
     * @param activityIds 活动ids
     * @return 活动详情
     * @throws ActivityException the activity exception
     */
    List<TuiaActivityDto> getBatchActivitys(List<Long> activityIds) throws ActivityException;

    /**
     * 查询活动奖项列表.
     *
     * @param activityId 活动id
     * @param includeStock 是否包含库存值
     * @return 活动的奖项列表
     * @throws ActivityException the activity exception
     */
    List<ActivityOptionDto> selectActivityOptions(Long activityId, boolean includeStock);

    /**
     * 从内存缓存查询活动奖项
     * @param activityId 活动id
     * @return 活动奖项dto list
     */
    List<ActivityOptionDto> selectActivityOptionsFromCache(Long activityId);

    /**
     * 查询活动皮肤.
     *
     * @param activityId the activity id
     * @return the activity skin data dto
     * @throws ActivityException the activity exception
     */
    ActivitySkinDataDto selectActivitySkinByActivityId(long activityId,String login) throws ExecutionException;

    /**
     * 根据模板标识 查询.
     *
     * @param skinType the skin type
     * @return the skin template data dto
     * @throws ActivityException the activity exception
     */
    SkinTemplateDataDto selectSkinTemplateBySkinType(String skinType) throws ActivityException;

    /**
     * 通过id获取奖项详情.
     *
     * @param optionId the option id
     * @return the activity option dto
     * @throws ActivityException the activity exception
     */
    ActivityOptionDto selectActivityOption(long optionId);

    /**
     * 获取一个浮标.
     *
     * @return the activity option dto
     * @throws ActivityException the activity exception
     */
    BuoyDto selectBuoy(Long buoyId) throws ActivityException;

    List<BuoyDto> selectOpenBuoyList() throws ActivityException;

    /**
     * selectProb:(获取流量入口跳转概率). <br/>
     *
     * @param entranceType the entrance type
     * @return the prob dto
     * @throws ActivityException the activity exception
     * @author youhaijun
     * @since JDK 1.6
     */
    ProbDto selectProb(int entranceType) throws ActivityException;


    /**
     * 更新用户返回拦截跳转次数.
     *
     * @param slotId the slot id
     * @param consumerId the consumer id
     * @throws ActivityException the activity exception
     */
    void updateReturnTimes(Long slotId, Long consumerId) throws ActivityException;

    /**
     * 获取返回拦截跳转次数.
     *
     * @param slotId the slot id
     * @param consumerId the consumer id
     * @return 跳转次数，或者0
     * @throws ActivityException the activity exception
     */
    int getReturnTimes(Long slotId, Long consumerId) throws ActivityException;


    /**
     * 更新用户活动参与次数缓存.
     * 缓存key为用户id,value为redis map结构,保存活动id-次数键值对
     *
     * @param activityId 用户已经抽完的活动id
     * @param consumerId the consumer id
     * @throws ActivityException the activity exception
     */
    int updateConsumerActivityDojoinCache(Long activityId, Long consumerId);

    /**
     * 获取用户活动参与次数缓存.
     * 缓存key为用户id,value为redis map结构,保存活动id-次数键值对
     *
     * @param consumerId the consumer id
     * @return 用户已抽完活动id列表
     * @throws ActivityException the activity exception
     */
    int getConsumerActivityDojoinCache(Long activityId, Long consumerId);

    /**
     * @Description: 获取用户活动参与次数缓存Map
     * @Param: [consumerId]
     * @Return: java.util.Map<java.lang.Object,java.lang.Object>
     * @throws:
     * @author: youhaijun
     * @Date: 2017/6/7
     */
    Map<Object, Object> getConsumerActivityDojoinCacheMap(Long consumerId);

    /**
     * @Description: 更新活动参与pv缓存
     * @Param: [activityId] 活动id
     * @Return: void
     * @author: youhaijun
     * @Date: 2017/6/2
     */
    void updateDojoinPvCache(Long activityId);

    /**
     * @Description: 查询活动参与pv缓存值
     * @Param: [activityId]
     * @Return: int
     * @author: youhaijun
     * @Date: 2017/6/2
     */
    int getDojoinPvCache(Long activityId);

    /**
     * @Description: 更新实物奖品奖项每日参与限制数量
     * @Param: [activityId, optionId]
     * @Return: void
     * @throws:
     * @author: youhaijun
     * @Date: 2017/6/2
     */
    void updatePhysicalDailyLimit(Long activityId, Long optionId);

    /**
     * @Description: 获取实物奖品每日参与数量
     * @Param: [activityId, optionId] 活动id和实物奖品奖项id
     * @Return: int
     * @throws:
     * @author: youhaijun
     * @Date: 2017/6/2
     */
    int getPhysicalDailyLimit(Long activityId, Long optionId);

    /**
     * 更新奖项当日已经抽取用户列表缓存
     * @param optionId 奖项id
     * @param deviceId 设备id
     * @param ip 用户ip
     */
    void updatePrizeWinConsumers(Long optionId, String deviceId, String ip);

    /**
     * 更新奖项本周已经抽取用户列表缓存
     * @param optionId 奖项id
     * @param deviceId 设备id
     * @param ip 用户ip
     */
    void updatePrizeWinConsumersWeek(Long optionId, String deviceId, String ip);

    /**
     * @Description: 获取奖品当日已经抽取用户列表
     * @Param: [prizeId, consumerId]
     * @Return: boolean
     * @throws:
     * @author: youhaijun
     * @Date: 2017/6/2
     */
    boolean isHaveWonPrizeDay(Long optionId, String deviceId, String ip);

    /**
     * 查询活动每天参与次数（内存缓存，有延迟,非实时，用于展示用）
     * @param activityId
     * @return 次数
     */
    int getDojoinPvDelay(Long activityId) throws ExecutionException;

    /**
     * 查看活动每天参与次数 内存缓存内容
     * @return Map
     */
    Map<Long, Integer> showCache();
/**
     * @Description: incrAlipayDailyBudget 记录支付宝红包每日中奖金额
     *
     * @Param: [optionId, amount]
     * @Return: int
     * @throws:
     * @author: youhaijun
     * @Date:   2017/7/12
     */
    int incrAlipayDailyBudget(Long optionId, int amount);

    boolean isHaveWonPrizeWeek(Long optionId, String deviceId, String ip);



    /**
     * 更新活动中心广告跳转次数.
     * @Param: location 特殊标识广告落地页
     * @throws ActivityException the activity exception
     */
    void updateAdvertTimes(String  location) throws ActivityException;

    /**
     * 获取活动中心广告跳转次数.
     *
     * @Param: location 特殊标识广告落地页
     * @return 跳转次数，或者0
     * @throws ActivityException the activity exception
     */
    int getAdvertTimes(String  location) throws ActivityException;


    /**
     * 根据广告位获取落地页
     */
    String getLandUrl(Long slotId);

    void indexCount();

    void doJoinCount();

    /**
     * 是否展示下拉推荐
     *
     * @param slotId
     * @param activityId
     * @param doJointTimes
     * @return
     */
    int isPullDown(Long slotId, Long activityId, int doJointTimes);


    /**
     * 根据媒体ID获取落地页域名
     *
     * @param appId
     * @return
     */
    String getLandUrlByAppId(Long appId);


    void refreshCache(String cacheName,String key);
}
