/**
 * Project Name:tuia-youtui-web<br>
 * File Name:ActivityController.java<br>
 * Package Name:com.duiba.tuia.youtui.web.controller<br>
 * Date:2017年1月17日下午12:05:04<br>
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.<br>
 */

package com.duiba.tuia.youtui.web.controller;

import cn.com.duiba.tuia.activity.center.api.constant.LoginType;
import cn.com.duiba.tuia.activity.center.api.dto.BuoyDto;
import cn.com.duiba.tuia.activity.center.api.dto.LayerSkinDto;
import cn.com.duiba.tuia.activity.center.api.remoteservice.RemoteSkinService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duibaboot.ext.autoconfigure.accesslog.AccessLogFilter;
import cn.com.tuia.advert.model.SpmlogReq;
import com.alibaba.fastjson.JSONObject;
import com.dianping.cat.Cat;
import com.duiba.tuia.youtui.web.bo.ActivityBO;
import com.duiba.tuia.youtui.web.bo.JoinActivityBO;
import com.duiba.tuia.youtui.web.constant.Constants;
import com.duiba.tuia.youtui.web.constant.CouponSource;
import com.duiba.tuia.youtui.web.constant.ErrorCode;
import com.duiba.tuia.youtui.web.exception.ActivityException;
import com.duiba.tuia.youtui.web.log.UrlLogService;
import com.duiba.tuia.youtui.web.model.LayerVO;
import com.duiba.tuia.youtui.web.model.RecordDetailVO;
import com.duiba.tuia.youtui.web.model.RecordGeneralVO;
import com.duiba.tuia.youtui.web.model.Result;
import com.duiba.tuia.youtui.web.model.UserAccount;
import com.duiba.tuia.youtui.web.model.req.*;
import com.duiba.tuia.youtui.web.model.rsp.*;
import com.duiba.tuia.youtui.web.service.ActivityService;
import com.duiba.tuia.youtui.web.tool.DomainUtil;
import com.duiba.tuia.youtui.web.tool.EmbedUtil;
import com.duiba.tuia.youtui.web.tool.RequestLocal;
import com.duiba.tuia.youtui.web.tool.RequestTool;
import com.google.common.collect.Maps;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.util.List;
import java.util.Map;

/**
 * ClassName: ActivityController <br/>
 * Function: 活动. <br/>
 * date: 2017年1月17日 下午12:05:04 <br/>
 *
 * @author leiliang
 * @since JDK 1.6
 */
@Controller
@RequestMapping("/activity")
@Api("活动API")
public class ActivityController extends BaseController {

    private static final String ACTIVITY_PAGE_INIT_ERROR = "初始化活动页面失败，请检查活动ID :";

    private static final String PUBLIC_NOTICE_STYLE_PAGE_ERROR = "中奖公示预览报错，ID :";

    /**
     * The join activity bo.
     */
    @Autowired
    private JoinActivityBO joinActivityBO;

    /**
     * The activity bo.
     */
    @Autowired
    private ActivityBO activityBO;

    @Autowired
    private ActivityService activityService;

    @Autowired
    private RemoteSkinService remoteSkinService;

    /**
     * Ajax options.
     *
     * @param req the req
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the result< activity option rsp>
     */
    @ResponseBody
    @RequestMapping(value = "/ajaxOptions", method = {RequestMethod.GET, RequestMethod.POST})
    @ApiOperation(value = "ajaxOptions", httpMethod = "GET", notes = "获取活动奖项信息接口")
    public Result<ActivityOptionRsp> ajaxOptions(@Valid ActivityOptionReq req, BindingResult result,
                                                 HttpServletRequest request, HttpServletResponse response) {
        try {
            paramAuth(result);
            req.setConsumerId(RequestLocal.get().getCid());
            req.setAppId(RequestLocal.get().getAppId());
            return successResult(this.activityBO.getActivityOption(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 获取活动中奖次数接口
     *
     * @param req
     * @param result
     * @return
     */
    @ResponseBody
    @RequestMapping(value = "/getLimitTimes", method = RequestMethod.POST)
    @ApiOperation(value = "getLimitTimes", httpMethod = "POST", notes = "获取活动中奖次数接口")
    public Result<ActivityLimitTimesRsp> getLimitTimes(@Valid ActivityOptionReq req, BindingResult result) {
        try {
            paramAuth(result);
            req.setConsumerId(RequestLocal.get().getCid());
            req.setAppId(RequestLocal.get().getAppId());
            return successResult(activityBO.getLimitTimes(req));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * Do join.
     *
     * @param doJoinActivityReq the do join activity req
     * @param result the result
     * @param request the request
     * @param response the response
     * @return the result< json object>
     */
    @ResponseBody
    @RequestMapping(value = "/doJoin", method = RequestMethod.POST)
    @ApiOperation(value = "doJoin", httpMethod = "POST", notes = "参与活动接口")
    public Result<JSONObject> doJoin(@Valid DoJoinActivityReq doJoinActivityReq, BindingResult result,
                                     HttpServletRequest request, HttpServletResponse response) {
        try {
            // 1.参数校验
            paramAuth(result);
            doJoinActivityReq.setIp(RequestTool.getIpAddr(request));
            doJoinActivityReq.setUserAgent(RequestTool.getUserAgent(request));
            doJoinActivityReq.setOs(RequestTool.getOSNew(request));
            doJoinActivityReq.setAppId(RequestLocal.get().getAppId());
            doJoinActivityReq.setConsumerId(RequestLocal.get().getCid());
            doJoinActivityReq.setUserId(RequestLocal.get().getUserId());
            doJoinActivityReq.setReferer(request.getHeader(Constants.HTTPHEADER.REFERER));

            if (StringUtils.isNotBlank(RequestLocal.get().getSlotIdByAppId(doJoinActivityReq.getAppId()))) {
                doJoinActivityReq.setSlotId(Long.valueOf(RequestLocal.get().getSlotIdByAppId(doJoinActivityReq.getAppId())));
            }
            doJoinActivityReq.setDeviceId(RequestLocal.get().getDeviceId());
            // 福袋取coll参数
            doJoinActivityReq.setParamMap(RequestLocal.get().getCollParams());

            //将网络类型放入logMap中
            if (StringUtils.isNotBlank(doJoinActivityReq.getConnection_type())) {
                doJoinActivityReq.getParamMap().put("connection_type", doJoinActivityReq.getConnection_type());
            }

            // 将流量来源放入logMap中
            if (StringUtils.isNotBlank(doJoinActivityReq.getAppFlowType())) {
                doJoinActivityReq.getParamMap().put("appFlowType", doJoinActivityReq.getAppFlowType());
            }

            // 保存cookie中的活动页来源
            EmbedUtil.getSourceMap(request, doJoinActivityReq.getParamMap(), CouponSource.ACTIVITY.getCode(), String.valueOf(doJoinActivityReq.getActivityId()));

            //活动皮肤ABtest
            String skinType = request.getParameter("skinType");
            if (StringUtils.isNotBlank(skinType)) {
                AccessLogFilter.putExPair("skinType", skinType);
            }
            // styleType
            String styleType = request.getParameter("styleType");
            if (StringUtils.isNotBlank(skinType)) {
                AccessLogFilter.putExPair("styleType", styleType);
            }

            AccessLogFilter.putExPair("use_host",request.getHeader("host"));

            activityService.doJoinCount();

            return successResult(joinActivityBO.doJoin(request, doJoinActivityReq));

        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * get result by ConsumerId and orderId.
     *
     * @param orderId the activity Order Id
     * @return the result< json object>
     */
    @ResponseBody
    @RequestMapping(value = "/result", method = {RequestMethod.POST, RequestMethod.GET})
    @ApiOperation(value = "result", httpMethod = "POST", notes = "参与活动接口")
    public Result<JSONObject> result(@Valid String orderId, @RequestParam(defaultValue = "false") Boolean showVersion, HttpServletRequest request, HttpServletResponse response) {
        try {
            return successResult(joinActivityBO.result(request, orderId, showVersion));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * Gets the skin config.
     *
     * @param req the req
     * @param result the result
     * @param request the request
     * @return the skin config
     */
    @ResponseBody
    @RequestMapping(value = "/getSkinConfig", method = {RequestMethod.GET, RequestMethod.POST})
    @ApiOperation(value = "getHdtoolConfig", httpMethod = "GET", notes = "获取活动配置接口")
    public Result<GetSkinConfigRsp> getSkinConfig(@Valid ActivitySkinReq req, BindingResult result,
                                                  HttpServletRequest request) {

        try {
            // 1.参数校验
            paramAuth(result);
            return successResult(activityBO.getActivitySkinConfig(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * Gets the prize detail.
     *
     * @param req the req
     * @param request the request
     * @return the prize detail
     */
    @ResponseBody
    @RequestMapping(value = "/getPrizeDetail", method = {RequestMethod.GET, RequestMethod.POST})
    @ApiOperation(value = "getPrizeDetail", httpMethod = "POST", notes = "获取奖项详情接口")
    public Result<PrizeDteailRsp> getPrizeDetail(@Valid PrizeReq req, HttpServletRequest request) {

        try {
            UserAccount consumer = RequestLocal.get().getUserAccount();
            if (null == consumer) {
                throw new ActivityException(ErrorCode.E0200001);
            }
            if (null == consumer.getAppId()) {
                throw new ActivityException(ErrorCode.E0000001.geteCode(), "appId为空");
            }

            request.getServerName();

            return successResult(activityBO.getPrizeDetail(req, consumer, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * Gets the record list.
     *
     * @param req the req
     * @return the record list
     */
    @ResponseBody
    @RequestMapping(value = "/getRecordList", method = RequestMethod.GET)
    @ApiOperation(value = "getRecordList", httpMethod = "GET", notes = "获取兑换记录列表接口")
    public Result<PrizeRecordsRsp<RecordGeneralVO>> getRecordList(@ModelAttribute OptionListReq req, HttpServletRequest request) {
        try {
            UserAccount consumer = RequestLocal.get().getUserAccount();
            if (null == consumer) {
                throw new ActivityException(ErrorCode.E0200001);
            }
            if (null == req.getPageSize()) {
                req.setPageSize(PageQueryReq.PAGE_SIZE_50);
            }
            req.setRowStart((req.getCurrentPage() - 1) * req.getPageSize());
            return successResult(activityBO.getRecordList(req, consumer, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * Gets the record.
     *
     * @param orderId the orderId
     * @return the record
     */
    @ResponseBody
    @RequestMapping(value = "/getRecord", method = RequestMethod.GET)
    @ApiOperation(value = "getRecord", httpMethod = "GET", notes = "获取兑换记录详情接口")
    @ApiImplicitParam(name = "orderId", required = true, dataType = "String")
    public Result<RecordDetailVO> getRecord(@RequestParam String orderId, HttpServletRequest request) {
        try {
            UserAccount consumer = RequestLocal.get().getUserAccount();
            if (null == consumer || null == consumer.getUserId()) {
                throw new ActivityException(ErrorCode.E0200002);
            }
            return successResult(activityBO.getRecord(orderId, consumer, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }


    /**
     * 活动工具首页初始化.
     *
     * @param req the req
     * @param request the request
     * @return the model and view
     * @throws ActivityException the activity exception
     * @author
     * @since JDK 1.6
     */
    @RequestMapping(value = "/index", method = RequestMethod.GET)
    ModelAndView index(@ModelAttribute IndexReq req, HttpServletRequest request) throws ActivityException {
        // 1.获取用户信息
        UserAccount consumer = RequestLocal.get().getUserAccount();
        if (null == consumer || null == req.getId()) {
            return new ModelAndView(Constants.MODEL_ERROR);
        }
        try {
            Cat.logMetricForCount("活动中心-活动访问");
            activityService.indexCount();
        } catch (Exception e) {
            logger.error(Constants.CAT_ERROR, e);
        }
        try {
            //1 根据广告位判断是否跳转主页面  芒果tv跳转到iframe
            if (null != req.getSlotId() && req.getSlotId() == 2488 && RequestLocal.get().getAppId() == 31665) {
                return activityBO.iframeIndex(req, consumer, request);
            }
            // 2.普通活动直接进活动首页
            AccessLogFilter.putExPair("use_host",request.getHeader("host"));
            AccessLogFilter.putExPair("agreement",request.getHeader("X-Forwarded-Proto")==null?"http":request.getHeader("X-Forwarded-Proto"));
            //设置地域信息
            req.setIp(getIp(request));
            return activityBO.index(req, consumer, request);
        } catch (Exception e) {
            logger.warn(ACTIVITY_PAGE_INIT_ERROR + req.getId(), e);
            return new ModelAndView(Constants.MODEL_ERROR);
        }
    }

    /**
     * 活动工具首页初始化.
     *
     * @param req the req
     * @param request the request
     * @return the model and view
     * @throws ActivityException the activity exception
     * @author
     * @since JDK 1.6
     */
    @RequestMapping(value = "/normalIndex", method = RequestMethod.GET)
    ModelAndView normaIndex(@ModelAttribute IndexReq req, HttpServletRequest request) throws ActivityException {
        // 1.获取用户信息
        UserAccount consumer = RequestLocal.get().getUserAccount();
        if (null == consumer || null == req.getId()) {
            return new ModelAndView(Constants.MODEL_ERROR);
        }
        try {
            Cat.logMetricForCount("活动中心-活动访问");
        } catch (Exception e) {
            logger.error(Constants.CAT_ERROR, e);
        }
        try {
            req.setIp(getIp(request));
            return activityBO.index(req, consumer, request);
        } catch (Exception e) {
            logger.warn(ACTIVITY_PAGE_INIT_ERROR + req.getId(), e);
            return new ModelAndView(Constants.MODEL_ERROR);
        }
    }

    /**
     * 兑换记录首页初始化.
     *
     * @param vmName the vm name
     * @param request the request
     * @return the model and view
     * @throws ActivityException the activity exception
     * @author
     * @since JDK 1.6
     */
    @RequestMapping(value = "/indexRecord", method = RequestMethod.GET)
    ModelAndView indexRecord(@RequestParam String vmName,Long appId,Long userId, HttpServletRequest request) throws ActivityException {
        try {
            return activityBO.indexRecord(vmName, request);
        } catch (Exception e) {
            logger.warn("初始化兑换记录页面失败，请检查活动模板 :" + vmName, e);
            return new ModelAndView(Constants.MODEL_ERROR);
        }
    }

    /**
     * 曝光日志.
     *
     * @param response the response
     * @return the model and view
     */
    @RequestMapping(value = "/showLog", method = {RequestMethod.GET, RequestMethod.POST})
    public void showLog(@ModelAttribute WebShowClickLogReq req, String callback, HttpServletRequest request,
                        HttpServletResponse response) {
        try {
            // 1.构建曝光日志打印参数
            SpmlogReq log = activityBO.budilSpmlogReq(req, request);
            // 2.打曝光日志
            activityBO.showLog(log);
        } catch (Exception e) {
            logger.warn("曝光失败，请检查参数 :" + req, e);
        }

    }

    /**
     * Redirect.
     *
     * @param response the response
     * @return the model and view
     */
    @RequestMapping(value = "/redirect", method = {RequestMethod.GET, RequestMethod.POST})
    public void redirect(@ModelAttribute WebShowClickLogReq req, HttpServletRequest request,
                         HttpServletResponse response) {

        try {
            // 区分为推啊的活动获取基本信息
            String url = StringEscapeUtils.unescapeHtml3(req.getUrl());

            // 如果广告ID和订单ID为空，说明不是正常广告跳转，不打印点击日志
            // 1.构建曝光日志打印参数
            SpmlogReq log = activityBO.budilSpmlogReq(req, request);
            // 2.记录点击事件
            activityBO.clickLog(log);

            // 获取cid
            String cid = null;
            Long consumerId = req.getConsumerId();
            if (consumerId == null) {
                consumerId = RequestLocal.get().getCid();
            }

            if (consumerId != null) {
                cid = Long.toString(consumerId);
            }

            // url特殊处理
            if (StringUtils.isNotBlank(url)) {
                url = activityBO.urlFilter(request, url, req.getOrderId(), cid);
                response.sendRedirect(url);
            } else {
                UrlLogService.log(req);
            }
        } catch (Exception e) {
            UrlLogService.log(req);
            logger.error("跳转页面失败，请检查", e);
        }


    }

    @ResponseBody
    @RequestMapping(value = "/getReturnPage", method = RequestMethod.GET)
    @ApiOperation(value = "getReturnPage", httpMethod = "GET", notes = "获取返回拦截跳转去向接口")
    @ApiImplicitParam(name = "slotId", required = true, dataType = "Long")
    public Result<UrlRsp> getReturnPage(@ModelAttribute IndexReq req, HttpServletRequest request) {
        try {
            req.setHost(request.getHeader("host"));
            if (null == req.getSlotId() || null == req.getId()) {
                throw new ActivityException(ErrorCode.E0000001);
            }
            req.setReferer(request.getHeader(Constants.HTTPHEADER.REFERER));
            req.setIp(getIp(request));
            return successResult(activityBO.getReturnPage(req));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @ResponseBody
    @RequestMapping(value = "/updateReturnTimes", method = RequestMethod.POST)
    @ApiOperation(value = "updateReturnTimes", httpMethod = "POST", notes = "更新用户返回拦截跳转次数接口")
    @ApiImplicitParam(name = "slotId", required = true, dataType = "Long")
    public Result<Boolean> updateReturnTimes(@RequestParam Long slotId, HttpServletRequest request) {
        try {
            return successResult(activityBO.updateReturnTimes(slotId, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @ResponseBody
    @RequestMapping(value = "/getRecommend", method = RequestMethod.GET)
    @ApiOperation(value = "getRecommend", httpMethod = "GET", notes = "获取推荐位信息接口")
    @ApiImplicitParam(name = "slotId", required = true, dataType = "Long")
    public Result<RecommendRsp> getRecommend(@ModelAttribute RecommendReq req, HttpServletRequest request) {
        try {
            req.setHost(request.getHeader("host"));
            if (req.getLogin() == null) {
                req.setLogin(LoginType.NORMAL.getCode());
            }
            if (req.getDeviceId() == null) {
                req.setDeviceId(RequestLocal.get().getDeviceId());
            }
            if (null == req.getRecommendType()){
                req.setRecommendType(1);
            }
            return successResult(activityBO.getRecommend(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }


    @ResponseBody
    @RequestMapping(value = "/getClosedRecommend", method = RequestMethod.GET)
    @ApiOperation(value = "getClosedRecommend", httpMethod = "GET", notes = "获取没开启推荐弹层的默认弹层接口")
    @ApiImplicitParam(name = "slotId", required = true, dataType = "Long")
    public Result<RecommendRsp> getClosedRecommend(@ModelAttribute RecommendReq req, HttpServletRequest request) {
        try {
            req.setHost(request.getHeader("host"));
            if (req.getLogin() == null) {
                req.setLogin(LoginType.NORMAL.getCode());
            }
            if (req.getDeviceId() == null) {
                req.setDeviceId(RequestLocal.get().getDeviceId());
            }
            if (null == req.getRecommendType()){
                req.setRecommendType(1);
            }
            return successResult(activityBO.getCloseRecommend(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @ResponseBody
    @RequestMapping(value = "/getBuoy", method = RequestMethod.GET)
    @ApiOperation(value = "getBuoy", httpMethod = "GET", notes = "获取浮标接口")
    public Result<BuoyRsp> getBuoy(@ModelAttribute IndexReq req, HttpServletRequest request) {
        try {
            req.setHost(request.getHeader("host"));
            if (null == req.getSlotId()) {
                throw new ActivityException(ErrorCode.E0000001);
            }
            return successResult(activityBO.getBuoy(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @ResponseBody
    @RequestMapping(value = "/getBuoyById", method = RequestMethod.GET)
    @ApiOperation(value = "getBuoyById", httpMethod = "GET", notes = "根据浮标id获取浮标接口")
    public Result<BuoyDto> getBuoyById(@RequestParam Long id, HttpServletRequest request, HttpServletResponse response) {
        try {
            response.setHeader("Access-Control-Allow-Origin", "*");
            return successResult(activityService.selectBuoy(id));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @RequestMapping("/getDojoinTimes")
    @ApiOperation(value = "getDojoinTimes", notes = "获取活动参与次数(每日)")
    @ApiImplicitParam(name = "activityId", required = true, dataType = "Long")
    public Result<Map<String, Object>> getDojoinTimes(Long activityId) {
        try {
            Map<String, Object> rs = Maps.newHashMap();
            rs.put("times", activityService.getDojoinPvDelay(activityId));
            return successResult(rs);
        } catch (Exception e) {
            return failResult(e);
        }
    }

    @ResponseBody
    @RequestMapping("/showDojoinCache")
    public Result<Map<String, Object>> showDojoinCache() {
        Map<String, Object> rs = Maps.newHashMap();
        rs.put("cache", activityService.showCache());
        return successResult(rs);
    }

    @ResponseBody
    @RequestMapping("/getAllSkin")
    public Result<List<LayerSkinDto>> getAllLayerSkin(@ModelAttribute SkinReq req, HttpServletRequest request) {
        return successResult(this.activityBO.getAllLayerSkin(req, request));
    }

    @ResponseBody
    @RequestMapping(value = "/getKefu", method = RequestMethod.GET)
    @ApiOperation(value = "getKefu", httpMethod = "GET", notes = "获取浮标接口")
    public Result<KefuResponse> getKefu(@RequestParam(required = false) Long slotId ,HttpServletRequest request){
        try {
            return successResult(activityBO.getKefu(slotId, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 获取支付宝红包中奖信息
     *
     */
    @ResponseBody
    @RequestMapping("/getAlipayWinInfo")
    @ApiOperation(value = "getAlipayWinInfo", httpMethod = "GET", notes = "获取支付宝红包中奖信息")
    public Result<List<AlipayWinInfoRsp>> getAlipayWinInfo(HttpServletRequest request) {
        try {
            return successResult(activityBO.getAlipayWinInfo());
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * get result by ConsumerId and orderId.
     *
     * @return the result< json object>
     */
    @ResponseBody
    @RequestMapping(value = "/modal", method = {RequestMethod.POST, RequestMethod.GET})
    @ApiOperation(value = "modal", httpMethod = "POST", notes = "活动预览接口")
    public ModelAndView pluginToolsPreview(@Valid Long activityId, HttpServletRequest request, HttpServletResponse response) {

        // 1.获取用户信息
        UserAccount consumer = RequestLocal.get().getUserAccount();
        if (null == consumer) {
            return new ModelAndView(Constants.MODEL_ERROR);
        }

        try {
            return new ModelAndView("preview/coupon")
                    .addObject("host", DomainUtil.getHost(request))
                    .addObject("pageHost", "//" + request.getHeader("host"));
        } catch (Exception e) {
            logger.warn(ACTIVITY_PAGE_INIT_ERROR + activityId, e);
            return new ModelAndView(Constants.MODEL_ERROR);

        }
    }

    /**
     * 获取真实中奖信息
     *
     * @param id 活动id
     * @return
     */
    @ResponseBody
    @RequestMapping("/getRealWinInfo")
    @ApiOperation(value = "getRealWinInfo", httpMethod = "GET", notes = "获取真实中奖信息")
    public Result<List<AlipayWinInfoRsp>> getRealWinInfo(@Valid Long id) {
        try {
            return successResult(activityBO.getRealWinInfo(id));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 中奖公示
     *
     * @param id 活动id
     * @param request
     * @return
     */
    @ResponseBody
    @RequestMapping("/publicNotice")
    @ApiOperation(value = "publicNotice", httpMethod = "GET", notes = "获取真实中奖信息")
    public Result<PublicNoticeRsp> publicNotice(@Valid Long id, HttpServletRequest request) {
        try {
            UserAccount consumer = RequestLocal.get().getUserAccount();
            if (null == consumer) {
                throw new ActivityException(ErrorCode.E0200001);
            }
            return successResult(activityBO.publicNotice(id, consumer, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 中奖公示-预览
     *
     * @param id 中奖公示样式id
     * @return
     */
    @RequestMapping(value = "/publicNotice/preview", method = RequestMethod.GET)
    public ModelAndView publicNoticePriview(@Valid Long id, HttpServletRequest request) {
        try {
            ModelAndView model = new ModelAndView("preview/lotteryPublic");
            //中奖公示样式信息
            model.addObject("id", id);
            model.addObject("host", DomainUtil.getHost(request));
            return model;
        } catch (Exception e) {
            logger.warn(PUBLIC_NOTICE_STYLE_PAGE_ERROR + id, e);
            return new ModelAndView(Constants.MODEL_ERROR);
        }
    }

    /**
     * 中奖公示-预览
     *
     * @param id 中奖公示样式id
     * @return
     */
    @ResponseBody
    @RequestMapping(value = "/publicNotice/previewData")
    @ApiOperation(value = "publicNotice", httpMethod = "GET", notes = "中奖公示-预览")
    public Result<PublicNoticeRsp> publicNoticePriviewData(@Valid Long id) {
        try {
            return successResult(activityBO.publicNoticePriviewData(id));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 下拉推荐位
     *
     * @param req
     * @param request
     * @return
     */
    @ResponseBody
    @RequestMapping(value = "/pullDown", method = {RequestMethod.POST, RequestMethod.GET})
    @ApiOperation(value = "pullDown", httpMethod = "GET", notes = "下拉推荐位")
    public Result<PullDownRsp> pullDown(@ModelAttribute IndexReq req, HttpServletRequest request) {
        try {
            req.setHost(request.getHeader("host"));
            if (req.getLogin() == null) {
                req.setLogin(LoginType.NORMAL.getCode());
            }
            if (req.getDeviceId() == null) {
                req.setDeviceId(RequestLocal.get().getDeviceId());
            }
            return successResult(activityBO.getPullDown(req, request));
        } catch (Exception e) {
            return failResult(e);
        }
    }

    /**
     * 线上测试接口，测试使用，具体找liaokefu
     *
     * @return
     */
    @RequestMapping(value = "/mkf", method = {RequestMethod.GET})
    public ModelAndView mkfTest() {
            return new ModelAndView("mkf/index");
    }


    @ApiOperation(value = "查询弹层皮肤", httpMethod = "GET")
    @ResponseBody
    @RequestMapping(value = "/selectLayerSkin", method = RequestMethod.GET)
    public Result<LayerVO> selectLayerSkin(Long id, HttpServletResponse response) {
        try {

            DubboResult<LayerSkinDto> dr = this.remoteSkinService.selectLayerSkin(id);
            return successResult(new LayerVO(dr.getResult()));
        } catch (Exception e) {
            return failResult(e);
        }
    }
}
