/** @format */
type Optional<T> = { [P in keyof T]?: T[P] }

function request(option) {
  if (String(option) !== '[object Object]') return undefined
  option.method = option.method ? option.method.toUpperCase() : 'GET'
  option.data = option.data || {}
  var formData = []
  for (var key in option.data) {
    formData.push(''.concat(key, '=', option.data[key]))
  }
  option.data = formData.join('&')

  if (option.method === 'GET') {
    option.url += location.search.length === 0 ? ''.concat('?', option.data) : ''.concat('&', option.data)
  }

  var xhr = new XMLHttpRequest()
  xhr.responseType = option.responseType || 'json'
  xhr.onreadystatechange = function () {
    if (xhr.readyState === 4) {
      if (xhr.status === 200) {
        if (option.success && typeof option.success === 'function') {
          option.success(xhr.response)
        }
      } else {
        if (option.error && typeof option.error === 'function') {
          option.error()
        }
      }
    }
  }
  xhr.open(option.method, option.url, true)
  if (option.method === 'POST') {
    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded')
  }
  xhr.send(option.method === 'POST' ? option.data : null)
}

function getIPCByDomain(domain: string) {
  return new Promise((resolve, reject) => {
    request({
      url: `http://activity.tuiatest.cn/land/getIcpInfo?domain=${encodeURIComponent(domain)}`,
      method: 'GET',
      success(res) {
        console.log('当前域名的IPC备案为', res)
        resolve(res)
      },
      error(err) {
        reject(err)
      }
    })
  })
}

type IPCFooterOptions = {
  domain?: string
  dom?: HTMLElement
  footerStyles?: Optional<CSSStyleDeclaration>
  boardStyles?: Optional<CSSStyleDeclaration>
}

function addStyles(ele: HTMLElement, styles: Optional<CSSStyleDeclaration> = {}) {
  Object.entries(styles).forEach(([key, val]) => {
    ele.style[key] = val
  })
}

const defaultFooterStyles: Optional<CSSStyleDeclaration> = {
  width: '100vw',
  position: 'fixed',
  left: '0',
  right: '0',
  bottom: '0',
  display: 'flex',
  justifyContent: 'center',
  alignItems: 'center'
}
const defaultBoardStyles: Optional<CSSStyleDeclaration> = {
  display: 'inline-block',
  textDecoration: 'none',
  color: '#FFF',
  backgroundColor: 'rgba(0,0,0,.4)',
  margin: '4px auto',
  height: '22px',
  lineHeight: '22px',
  padding: '0 20px',
  borderRadius: '4px',
  fontSize: '12px'
}

/**
 * 挂载
 * @param {IPCFooterOptions} opts
 */
async function mountIPCFooter(opts: IPCFooterOptions = {}) {
  let IPCNumberString = ''
  try {
    IPCNumberString = (await getIPCByDomain(opts.domain || location.host)) as string
  } catch (err) {
    console.log('备案号获取失败')
  }

  const IPCFooter = document.createElement('div')
  const IPCBoard = document.createElement('a')
  const targetDom = opts.dom || document.querySelector('body')
  IPCFooter.id = '__IPC_footer__'
  IPCBoard.id = '__IPC_board__'
  addStyles(IPCFooter, { ...defaultFooterStyles, ...(opts.footerStyles || {}) })
  addStyles(IPCBoard, { ...defaultBoardStyles, ...(opts.footerStyles || {}) })
  IPCBoard.innerText = (IPCNumberString as string) || 'IPC备-'
  IPCBoard.href = 'https://beian.miit.gov.cn/#/Integrated/index'
  IPCFooter.appendChild(IPCBoard)
  targetDom.appendChild(IPCFooter)
}

/**
 * 卸载
 */
function unmount() {
  let target = document.getElementById('__IPC_footer__')
  if (target) {
    let parent = target.parentElement
    parent.removeChild(target)
  }
}

/**
 * 重新挂载
 * @param {IPCFooterOptions} opts
 */
function remount(opts: IPCFooterOptions = {}) {
  let target = document.getElementById('__IPC_footer__')
  if (target) {
    unmount()
    mountIPCFooter(opts)
  } else {
    setTimeout(() => {
      unmount()
      mountIPCFooter(opts)
    }, 800)
  }
}

export default { getIPCByDomain, mountIPCFooter, unmount, remount }
